/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService;

import java.util.EventListener;
import java.util.List;

import jdbcacsess2.sqlService.parse.SqlExecuteSentencies.SqlExecuteSentence;

/**
 * SQL実行結果の通知するリスナーです。
 * 
 * @author sima
 * 
 */
public interface SqlExecutedListener extends EventListener {

	/**
	 * SQL実行が受け付けた時に呼ばれます。このイベントだけは、呼出と同じスレッドで呼ばれ、SQL実行スレッドからではありません。
	 * executeの実行毎に１回だけ呼ばれます。
	 * 
	 * @param sqlExecuteTask
	 *            SQL実行スレッドに中断／再開等の制御を指示する為の実行タスク
	 */
	public void taskAccept(SqlExecuteTask sqlExecuteTask);

	/**
	 * １SQL分の実行が開始した時に呼ばれます。
	 * 
	 * @param sqlExecuteSentence
	 *            １SQL分の解析結果
	 */
	public void executBegin(SqlExecuteSentence sqlExecuteSentence);

	/**
	 * 検索結果のColumnAttributeResultを通知します。
	 * 
	 * @param header
	 *            検索結果のカラム属性
	 */
	public void resultHeader(List<ColumnAttributeResult> header);

	/**
	 * １行分の検索結果を通知します。
	 * 
	 * @param seq
	 *            検索結果行に対する１からの通し番号
	 * @param detail
	 *            検索結果リスト
	 */
	public void resultDetail(int seq, List<Result> detail);

	/**
	 *　結果取得件数が大量になった為、結果の取得が中断した事を通知します。
	 * 
	 * @param seqNo
	 *            中断した処理結果行の通し番号
	 */
	public void statusContinue(int seqNo);

	/**
	 * 正常に終了した事を通知します。
	 * 
	 * @param rowCnt
	 *            処理行数
	 */
	public void executNormalFinish(int rowCnt);

	/**
	 * 異常終了した事を通知します。
	 * 
	 * @param t
	 *            例外
	 */
	public void executeException(Throwable t);

	/**
	 * 受け付けたSQL全てが終了した事を通知します。
	 */
	public void executeAllEnd();
}
