/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jdbcacsess2.sqlService.history;

import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;

import jdbcacsess2.main.Config;
import jdbcacsess2.main.Jdbcacsess2;
import jdbcacsess2.main.ShowDialog;
import net.java.ao.EntityManager;
import net.java.ao.Query;

/**
 * @author sima
 *
 */
public class HistryTableModel extends AbstractTableModel {

	private static final long serialVersionUID = -7447463865535694333L;

	public static final int FILTERCLOUMN = 6;

	private final EntityManager entityManager;

	private final List<ExecHistory> historyList = new ArrayList<ExecHistory>();
	private final List<String> startDate = new ArrayList<String>();

	private final SimpleDateFormat timeFormat;
	private final SimpleDateFormat dateFormat;

	public HistryTableModel(Config config) {
		dateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS");
		timeFormat = new SimpleDateFormat("m:ss.SSS");
		timeFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
		entityManager = config.getEntityManager();

		initialize();
	}

	synchronized private void initialize() {
		new Thread() {
			@Override
			public void run() {
				try {
					for (ExecHistory o : entityManager.find(ExecHistory.class,
					                                        Query.select().order(ExecHistory.STARTDATE))) {
						historyList.add(o);
						startDate.add(dateFormat.format(o.getStartDate()));
					}

					SwingUtilities.invokeLater(new Runnable() {
						@Override
						public void run() {
							fireTableDataChanged();
						}
					});
					Jdbcacsess2.logger.info("history load end");
				} catch (SQLException e) {
					ShowDialog.errorMessage(e);
				}
			}
		}.start();

	}
	synchronized public int add(ExecHistory execHistory) {
		historyList.add(execHistory);
		startDate.add(dateFormat.format(execHistory.getStartDate()));
		return historyList.size() - 1;
	}

	@Override
	public int getRowCount() {
		return historyList.size();
	}

	private final String header[] = { "start", "elaps", "command", "status", "cnt", "text", "connect" };

	@Override
	public String getColumnName(int column) {
		return header[column];
	}

	@Override
	public int getColumnCount() {
		return header.length;
	}

	@Override
	public Object getValueAt(int rowIndex, int columnIndex) {
		ExecHistory eh = historyList.get(rowIndex);
		switch (columnIndex) {
		case 0:
			return startDate.get(rowIndex);
		case 1:
			if (eh.getEndDate() == null) {
				return null;
			}
			return timeFormat.format(new Date(eh.getEndDate().getTime() - eh.getStartDate().getTime()));
		case 2:
			return eh.getSqlCommand();
		case 3:
			return eh.getResultStatus();
		case 4:
			return eh.getResultCnt();
		case 5:
			return eh.getSqlText();
		case 6:
			return eh.getConnectName();
		}
		return null;
	}

}
