/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService.parse;

import java.util.ArrayList;
import java.util.regex.Pattern;

import jdbcacsess2.main.Jdbcacsess2;

/**
 * SQL文を解析し、最初の命令句/入力変数/更新可能SELECT文であるか？などを抽出します。
 * 
 * 画面から入力されたSQLが複数文であった場合、１SQL文毎に分解されます。 SQL文の区切りは{@link SqlSentenceParse}
 * で行われるので、通知されたSQL文の番号を使用して、それぞれのSQL文を区別します。
 * 
 * @author sima
 * 
 */
public class SqlExecuteSentencies {

	/**
	 * コンストラクタ、SQL文のparseを行う。
	 * 
	 * @param sqlSentence
	 *            分析したいSQL文
	 * @param sentenceSeparator
	 *            複数SQL文を区切り正規表現文字列
	 */
	public SqlExecuteSentencies(String sqlSentence, String sentenceSeparator) {


		final SqlSentenceParse sqlSentenceParse = new SqlSentenceParse();

		sqlSentenceParse.addSqlSentenceListener(new SqlExecuteSentence());

		sqlSentenceParse.registSeparatePattern(Pattern.compile(sentenceSeparator));
		sqlSentenceParse.registSeparateListener(new SqlSentenceSparateListener() {
			@Override
			public void rangeSeparate(String matchKeyword,
			                          int beginPosition,
			                          int endPosition,
			                          Pattern pattern) {
				sqlSentenceParse.removeAllSqlSentenceListener();
				sqlSentenceParse.addSqlSentenceListener(new SqlExecuteSentence());
			}
		});

		// 解析開始
		sqlSentenceParse.parse(sqlSentence);

		Jdbcacsess2.logger.info(sqlExecuteSentenceList.toString());
	}

	private int seq = 0;
	final private ArrayList<SqlExecuteSentence> sqlExecuteSentenceList = new ArrayList<SqlExecuteSentence>();

	/**
	 * SQL文単位に分解された最終結果
	 * 
	 * @return　SQL文解析結果のリスト
	 */
	public ArrayList<SqlExecuteSentence> getSqlExecuteSentenceList() {
		return sqlExecuteSentenceList;
	}

	/**
	 * 分解された１SQL文の解析結果
	 * 
	 * @author sima
	 * 
	 */
	public class SqlExecuteSentence implements SqlSentenceListener {

		private String sqlSentence;
		private int sentenceCount;

		/**
		 * SQLの連番
		 * 
		 * @return sentenceCount
		 */
		public int getSentenceCount() {
			return sentenceCount;
		}

		/**
		 * SQL文
		 * 
		 * @return sqlSentence
		 */
		public String getSqlSentence() {
			return sqlSentence;
		}

		private final SqlInputParameter sqlInputParameter = new SqlInputParameter();

		/**
		 * 入力パラメータ
		 * 
		 * @return sqlInputParameter
		 */
		public SqlInputParameter getSqlInputParameter() {
			return sqlInputParameter;
		}

		private String sqlCommand;

		/**
		 * SQL文の最初の命令句
		 * 
		 * @return "SELECT","INSERT"."DELETE","CALL","CREATE","ALTER"等の文字列。
		 *         常に大文字で返却されます 。
		 */
		public String getSqlCommand() {
			return sqlCommand;
		}

		private boolean editable = false;

		/**
		 * 更新可能SELECT文であるか
		 * 
		 * @return editable
		 */
		public boolean isEditable() {
			return editable;
		}

		private int phraseCnt = 0;
		private boolean checkEnd = false;
		private boolean select = false;
		private boolean from = false;

		@Override
		public void started(String sqlSentence) {
			// finished で上書きされるの意味はない
			this.sqlSentence = sqlSentence;
		}

		@Override
		public void rangeInput(String phrase, int begin, int end) {
			sqlInputParameter.addInputItemName(phrase);
		}

		@Override
		public void rangePhrase(String phrase, int begin, int end) {
			if (checkEnd) {
				return;
			}
			String upperPhrase = phrase.toUpperCase();

			phraseCnt++;
			if (phraseCnt == 1 && upperPhrase.equals("SELECT")) {
				select = true;
				return;
			}
			if (!select) {
				checkEnd = true;
				editable = false;
				return;
			}
			if (upperPhrase.equals("SELECT") || upperPhrase.equals("GROUP")) {
				checkEnd = true;
				editable = false;
				return;
			}

			if (upperPhrase.equals("FROM")) {
				// FROM句を見つけたので、とりあえず更新可能とする
				editable = true;

				from = true;
				return;
			}
			if (upperPhrase.equals("WHERE")) {
				from = false;
				return;
			}

		}

		@Override
		public void rangeSymbol(String symbol, int begin, int end) {
			if (checkEnd) {
				return;
			}
			if (from && symbol.equals(",")) {
				// FROM句中にカンマを見つけたので、更新不可が確定した
				checkEnd = true;
				editable = false;
				return;
			}
		}

		@Override
		public void finished(String sentence, String command) {
			sqlSentence = sentence;
			sqlCommand = command;
			if (!sqlCommand.equals("")) {
				sentenceCount = ++seq;
				if (sqlCommand.equals("CREATE") || sqlCommand.equals("ALTER")) {
					sqlInputParameter.clear();
				}
				sqlExecuteSentenceList.add(this);
			}
		}

		@Override
		public void rangeComment(String comment, int begin, int end) {
		}

		@Override
		public void rangeConstant(String constant, int begin, int end) {
		}

		@Override
		public void rangeDelimiter(String delimiter, int begin, int end) {
		}

		@Override
		public String toString() {
			return "command:" + sqlCommand + " editable:" + editable + " inputItems:"
					+ sqlInputParameter.getInputItemNames() + "["
					+ sqlSentence + "]";
		}

	}


}
