/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 * 
 * Copyright (C) 2002 SUZUKI Jun
 * 
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 * 
 * 
 * $Id: Array.cpp,v 1.13 2004/03/24 11:53:44 randy Exp $
 */

#include <stdio.h>

#include "soopy.h"

/*
 * class SpArray
 */

SpBaseArray::SpBaseArray(uintvec& v)
  : dimension(v)
{
    depth = dimension.size();
}

// primitives
SpValue& SpBaseArray::getDepth(SpValue& self)
{
    SpArray* array = self.asArray();
    //    static SpValue result;
    //    result.setInt(array->depth);
    //    return result;
    return SpIntResult(array->depth);
}

SpValue& SpBaseArray::getDimension(SpValue& self)
{
    SpValue temp;
    SpArray* array = self.asArray();
    uintvec::iterator it;
    it = array->dimension.begin();
    temp.setInt(*it);
    SpCons* head = new SpCons(temp);
    SpCons* ptr = head;
    it++;
    for(; it != array->dimension.end(); it++){
        temp.setInt(*it);
        ptr->append(temp);
        SpValue v;
        v = ptr->nextList();
        ptr = (SpCons*)v.asList();
    }
    //    static SpValue result;
    //    result.setNewObject(head);
    //    return result;
    return SpObjectResult(head);
}

// init Message Handler
void SpBaseArray::init()
{
    SpValue SymDepth(new SpSymbol("depth"));
    SpValue PrimDepth(new SpPrim1(getDepth));
    ArrayMsgHandler.append(SymDepth, PrimDepth);

    SpValue SymDimension(new SpSymbol("dimension"));
    SpValue PrimDimension(new SpPrim1(getDimension));
    ArrayMsgHandler.append(SymDimension, PrimDimension);
}


/*
 * class SpObjectArray
 */

SpObjectArray::SpObjectArray(SpValue& t, uintvec& v)
  : SpArray(v), typ(t)
{
//    depth = dimension.size();
    int allsize = 1;
    uintvec::iterator it;
    it = dimension.begin();
    for(; it != dimension.end(); it++){
        allsize *= *it;
    }
    pvec = new SpValueVector(allsize);
}

SpValue& SpObjectArray::toString()
{
    SpString* str;
    char buf[128];

    str = new SpString();
    *str = "Array:";
    if(!typ.isNil()){
        *str += typ;
    }
    *str += " (";

    uintvec::iterator it;
    it = dimension.begin();
#ifdef HAVE_ITOA
    *str += itoa(*it, buf, 10);
#else
    sprintf(buf, "%d", *it);
    *str += buf;
#endif
    it++;
    for(; it != dimension.end(); it++){
        *str += ",";
#ifdef HAVE_ITOA
        *str += itoa(*it, buf, 10);
#else
    sprintf(buf, "%d", *it);
    *str += buf;
#endif
    }

    *str += ")";
    //    static SpValue s;
    //    s.setNewObject(str);
    //    return s;
    return SpObjectResult(str);
}

SpValue& SpObjectArray::operator[](SpValue& tuple)
{
    if(tuple.isTuple()){
        SpTuple* t = dynamic_cast<SpTuple*>(tuple.getObject());
        if((SpInt)depth != t->length()){
            throw SpException("array depth mismatch");
        }
        uintvec::iterator uit;
        SpValueVector::iterator tit;
        uit = dimension.begin();
        tit = t->begin();
        unsigned int offset = 0;
        unsigned int times = 1;
        for(; uit != dimension.end(); uit++, tit++){
            SpValue v = tit->eval();
            if(!v.isInt()){
                throw SpException("not int (access array)");
            }
            unsigned int index = v.getInt();
            if((*uit - 1) < index){
                throw SpException("too large index (access array)");
            }
            offset += times * index;
            times *= *uit;
        }
        return (*pvec)[offset];

    }else{ // not tuple
        SpValue v = tuple.eval();
        if(!v.isInt()){
            throw SpException("not int (access array)");
        }
        if(depth != 1){
            throw SpException("depth not match (access array)");
        }
        unsigned int index = v.getInt();
        if((dimension[0] - 1) < index){
            throw SpException("over range (access array)");
        }
        return (*pvec)[index];
    }
}

void SpObjectArray::set(SpValue& tuple, SpValue& value)
{
    (*this)[tuple] = value;
}

SpValue& SpObjectArray::onMessage(SpValue& rec, SpValue& msg)
{
    if(ArrayMsgHandler.hasMessage(msg)){
        return ArrayMsgHandler(rec, msg);
    }
    SpValue val = msg.eval();
    if(val.isInt()){
        return operator[](val);
    }
    if(val.isTuple()){
        return operator[](val);
    }

    // Error
    static char buf[256];
    strncpy(buf, rec.toCStringWithEncoder(), 255);
    buf[255] = '\0';
    static char buf2[256];
    strncpy(buf2, msg.toCStringWithEncoder(), 255);
    buf2[255] = '\0';
    throw SpNoMethodException("no such a feature", buf, buf2);
}


/*
 * class SpByteArray
 */

SpByteArray::SpByteArray(uintvec& v)
  : SpBaseArray(v)
{
    int allsize = 1;
    uintvec::iterator it;
    it = dimension.begin();
    for(; it != dimension.end(); it++){
        allsize *= *it;
    }
//    buf = new char(allsize);
    buf = (unsigned char*)malloc(allsize);
    if(buf == NULL){
        throw SpException("out of memory");
    }
}

SpValue& SpByteArray::toString()
{
    SpString* str;
    char buf[128];

    str = new SpString();
    *str = "Array:byte (";

    uintvec::iterator it;
    it = dimension.begin();
#ifdef HAVE_ITOA
    *str += itoa(*it, buf, 10);
#else
    sprintf(buf, "%d", *it);
    *str += buf;
#endif
    it++;
    for(; it != dimension.end(); it++){
        *str += ",";
#ifdef HAVE_ITOA
        *str += itoa(*it, buf, 10);
#else
    sprintf(buf, "%d", *it);
    *str += buf;
#endif
    }

    *str += ")";
    //    static SpValue s;
    //    s.setNewObject(str);
    //    return s;
    return SpObjectResult(str);
}

unsigned char& SpByteArray::operator[](SpValue& tuple)
{
    if(tuple.isTuple()){
        SpTuple* t = dynamic_cast<SpTuple*>(tuple.getObject());
        if((SpInt)depth != t->length()){
            throw SpException("array depth mismatch");
        }
        uintvec::iterator uit;
        SpValueVector::iterator tit;
        uit = dimension.begin();
        tit = t->begin();
        unsigned int offset = 0;
        unsigned int times = 1;
        for(; uit != dimension.end(); uit++, tit++){
            SpValue v = tit->eval();
            if(!v.isInt()){
                throw SpException("not int (access array)");
            }
            unsigned int index = v.getInt();
            if((*uit - 1) < index){
                throw SpException("too large index (access array)");
            }
            offset += times * index;
            times *= *uit;
        }
        return buf[offset];

    }else{ // not tuple
        SpValue v = tuple.eval();
        if(!v.isInt()){
            throw SpException("not int (access array)");
        }
        if(depth != 1){
            throw SpException("depth not match (access array)");
        }
        unsigned int index = v.getInt();
        if((dimension[0] - 1) < index){
            throw SpException("over range (access array)");
        }
        return buf[index];
    }
}

SpValue& SpByteArray::getFeatureType()
{
    return SymByte;
}

SpValue& SpByteArray::onMessage(SpValue& rec, SpValue& msg)
{
  //    static SpValue result;

    if(ArrayMsgHandler.hasMessage(msg)){
        return ArrayMsgHandler(rec, msg);
    }
    SpValue val = msg.eval();
    if(val.isInt()){
      //        result.setInt(operator[](val));
      //        return result;
      return SpIntResult(operator[](val));
    }
    if(val.isTuple()){
      //        result.setInt(operator[](val));
      //        return result;
      return SpIntResult(operator[](val));
    }

    // Error
    static char buf[256];
    strncpy(buf, rec.toCStringWithEncoder(), 255);
    buf[255] = '\0';
    static char buf2[256];
    strncpy(buf2, msg.toCStringWithEncoder(), 255);
    buf2[255] = '\0';
    throw SpNoMethodException("no such a feature", buf, buf2);
}


/*
 * class MakeArray
 */

SpValue& MakeArray::eval()
{
    uintvec intVec; // int vector
    SpTuple* tuple = vec.asTuple();
    SpValueVector::iterator it;
    it = tuple->begin();
    for(; it != tuple->end(); it++){
        SpValue v = it->eval();;
        if(!v.isInt()){ throw SpException("not int (make array)"); }
        intVec.push_back(v.getInt());
    }

    //    static SpValue result;
    if(typ == SymByte){
      //        result.setNewObject(new SpByteArray(intVec));
      return SpObjectResult(new SpByteArray(intVec));
    }else{
      //        result.setNewObject(new SpObjectArray(typ, intVec));
      return SpObjectResult(new SpObjectArray(typ, intVec));
    }
    //    return result;
}

SpValue& MakeArray::toString()
{
    SpString* str;
    //char buf[128];

    str = new SpString();
    *str = "MakeArray:";
    if(!typ.isNil()){
        *str += typ;
    }
    *str += " (";

    SpTuple* tuple = vec.asTuple();
    SpValueVector::iterator it;
    it = tuple->begin();
    *str += *it;
    it++;
    for(; it != tuple->end(); it++){
        *str += ",";
        *str += *it;
    }

    *str += ")";
    //    static SpValue s;
    //    s.setNewObject(str);
    //    return s;
    return SpObjectResult(str);
}

