/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 * 
 * Copyright (C) 2002 SUZUKI Jun
 * 
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 * 
 * 
 * $Id: Object.cpp,v 1.127 2004/05/23 13:15:53 randy Exp $
 */


#include <stdio.h>
#include "soopy.h"

char Filename[SP_NAME_MAX];
unsigned int lineno = 1;

SpValue NilObject;
#ifndef USE_PTHREAD
SpValue gResult;
#endif  /* USE_PTHREAD */
SpValue TrueObject((bool)true);
SpValue FalseObject((bool)false);
SpValue NullString;
SpValue ObjectZero((int)0);
SpValue ObjectOne((int)1);
SpValue ObjectTwo((int)2);

StreamWriter* strmout;
SjisWriteEncoder* sjisout;
EucJPWriteEncoder* eucjpout;
RawWriteEncoder* write_raw_encoder;

StreamReader* strmin;
SjisReadDecoder* sjisin;
EucJPReadDecoder* eucjpin;
JapaneseReadDecoder* japanin;
RawReadEncoder* read_raw_encoder;
WriteEncoder* spout;
ReadEncoder* spin;

MsgHandler ObjectMsgHandler;
MsgHandler NilMsgHandler(&ObjectMsgHandler);
MsgHandler BoolMsgHandler(&ObjectMsgHandler);
MsgHandler CharMsgHandler(&ObjectMsgHandler);
MsgHandler IntMsgHandler(&ObjectMsgHandler);
MsgHandler RealMsgHandler(&ObjectMsgHandler);
MsgHandler StringMsgHandler(&ObjectMsgHandler);
MsgHandler SymbolMsgHandler(&ObjectMsgHandler);
MsgHandler ListMsgHandler(&ObjectMsgHandler);
MsgHandler TupleMsgHandler(&ObjectMsgHandler);
MsgHandler NameSpaceMsgHandler(&ObjectMsgHandler);
MsgHandler ArrayMsgHandler(&ObjectMsgHandler);
MsgHandler FileInMsgHandler(&ObjectMsgHandler);
MsgHandler FileOutMsgHandler(&ObjectMsgHandler);
MsgHandler DirMsgHandler(&ObjectMsgHandler);
MsgHandler FileMsgHandler(&ObjectMsgHandler);
#ifdef USE_SOCKET
MsgHandler SocketMsgHandler(&ObjectMsgHandler);
MsgHandler ServerSocketMsgHandler(&ObjectMsgHandler);
#endif
MsgHandler DateMsgHandler(&ObjectMsgHandler);
MsgHandler TimeMsgHandler(&ObjectMsgHandler);
MsgHandler DateTimeMsgHandler(&ObjectMsgHandler);
#ifdef USE_DATABASE
MsgHandler DatabaseMsgHandler(&ObjectMsgHandler);
#endif

SpValue SymIsInt;
SpValue SymIsChar;
SpValue SymIsBool;
SpValue SymIsReal;
SpValue SymIsList;
SpValue SymIsNil;
SpValue SymIsTuple;
SpValue SymIsString;
SpValue SymIsSymbol;
SpValue SymHead;
SpValue SymTail;
SpValue SymLength;
SpValue SymVar;
SpValue SymRequire;
SpValue SymDo;
SpValue SymEnsure;
SpValue SymRescue;
SpValue SymFrom;
SpValue SymStep;
SpValue SymWhile;
SpValue SymUntil;
SpValue SymVariant;
SpValue SymInvariant;
SpValue SymSoopy;
SpValue SymEnv;
SpValue SymClose;
SpValue SymRename;
SpValue SymArg;
SpValue SymName;
SpValue SymToInt;
SpValue SymToString;
SpValue SymNth;
SpValue SymEach;
SpValue SymFirst;
SpValue SymSecond;
SpValue SymThird;
SpValue SymFourth;
SpValue SymFifth;
SpValue SymEucjpOut;
SpValue SymEucjpIn;
SpValue SymSjisOut;
SpValue SymSjisIn;
SpValue SymJapaneseIn;
// types for type check
SpValue SymBool;
SpValue SymInt;
SpValue SymByte;
SpValue SymWord;
SpValue SymChar;
SpValue SymReal;
SpValue SymString;
SpValue SymSymbol;
SpValue SymFunction;
SpValue SymTuple;
SpValue SymList;
SpValue SymArray;
SpValue SymNameSpace;
SpValue SymReader;
SpValue SymWriter;
SpValue SymEncoderIn;
SpValue SymEncoderOut;
SpValue SymDecoderIn;
SpValue SymDecoderOut;
SpValue SymDir;
SpValue SymMember;
SpValue SymOpen;
SpValue SymOpenIn;
SpValue SymOpenOut;
SpValue SymOpenAppend;
SpValue SymSelf;
SpValue SymWriteChar;
SpValue SymWrite;
SpValue SymWriteLine;
SpValue SymReadLine;
SpValue SymFlush;
SpValue SymClosed;
SpValue SymIsEOF;
SpValue SymUpper;
SpValue SymLower;
SpValue SymAppend;
SpValue SymKeys;
SpValue SymDelete;
SpValue SymSplit;
SpValue SymSplit1;
SpValue SymSort;
SpValue SymStdin;
SpValue SymStdout;
SpValue SymMap;
SpValue SymFoldl;
SpValue SymFoldr;
SpValue SymFilter;
SpValue SymReplace;
SpValue SymReplace1;
SpValue SymFind;
SpValue SymYear;
SpValue SymMonth;
SpValue SymDay;
SpValue SymSucc;
SpValue SymPred;
SpValue SymHour;
SpValue SymMinute;
SpValue SymDate;
SpValue SymTime;
SpValue SymAtime;
SpValue SymCtime;
SpValue SymMtime;
#ifdef USE_DATABASE
SpValue SymDatabase;
#endif
SpValue SymGet;
SpValue SymSet;
SpValue SymCurrentThread;

SpSymbol* symbolBool;
SpSymbol* symbolInt;
SpSymbol* symbolByte;
SpSymbol* symbolWord;
SpSymbol* symbolChar;
SpSymbol* symbolReal;
SpSymbol* symbolString;
SpSymbol* symbolSymbol;
SpSymbol* symbolFunction;
SpSymbol* symbolTuple;
SpSymbol* symbolList;
SpSymbol* symbolArray;
SpSymbol* symbolNameSpace;
SpSymbol* symbolReader;
SpSymbol* symbolWriter;
SpSymbol* symbolEncoderIn;
SpSymbol* symbolEncoderOut;
SpSymbol* symbolDecoderIn;
SpSymbol* symbolDecoderOut;

SpValue PrimNil;
SpValue PrimNil2;
SpValue PrimNil3;
SpValue PrimTrue;
SpValue PrimFalse;
SpValue PrimZero;

// type check
void CheckType(SpValue& featureType, SpValue& value)
{
    static char buf[256];
    static char buf2[256];

    if(featureType.isNil()){  // any type is OK.
        return;
    }
    if(!featureType.isSymbol()){
        throw SpException("not symbol (type)");
    }
    SpSymbol* sym = featureType.asSymbol();
    switch(value.getType()){
    case TypeBool:
        if(*sym == *symbolBool) return;
        break;
    case TypeInt:
        if(*sym == *symbolInt) return;
        if(*sym == *symbolByte) return;
        if(*sym == *symbolWord) return;
        break;
    case TypeChar:
        if(*sym == *symbolChar) return;
        break;
    case TypeReal:
        if(*sym == *symbolReal) return;
        break;
    case TypeString:
        if(*sym == *symbolString) return;
        break;
    case TypeSymbol:
        if(*sym == *symbolSymbol) return;
        break;
    case TypeList:
        if(*sym == *symbolList) return;
        break;
    case TypeTuple:
        if(*sym == *symbolTuple) return;
        break;
    case TypeNameSpace:
        if(*sym == *symbolNameSpace) return;
        break;
    case TypeExpr:
        throw SpSystemError(__FILE__, __LINE__);
    case TypeArray:
        if(*sym == *symbolArray) return;
        break;
    case TypeDType:
        SpDType* dt = value.asDType();
        if(featureType == dt->symbol) return;
        if(featureType == dt->con->datatype->symbol) return;
        strncpy(buf, featureType.toCStringWithEncoder(), 255);
        buf[255] = '\0';
        strncpy(buf2, value.typeString(), 255);
        buf2[255] = '\0';
        throw SpTypeException("type error", buf, buf2);
        break;
/*
    case TypeObject:
        break;
*/
    }
    strncpy(buf, featureType.toCStringWithEncoder(), 255);
    buf[255] = '\0';
    strncpy(buf2, value.typeString(), 255);
    buf2[255] = '\0';
    throw SpTypeException("type error", buf, buf2);
}

// primitives
SpValue& returnNil(SpValue&)
{
    return NilObject;
}

SpValue& returnNil2(SpValue&, SpValue&)
{
    return NilObject;
}

SpValue& returnNil3(SpValue&, SpValue&, SpValue&)
{
    return NilObject;
}

SpValue& returnTrue(SpValue&)
{
    return TrueObject;
}

SpValue& returnFalse(SpValue&)
{
    return FalseObject;
}

SpValue& returnZero(SpValue&)
{
    return ObjectOne;
}

static void setSymbols()
{
    SymIsInt.setNewObject(new SpSymbol("isint?"));
    SymIsChar.setNewObject(new SpSymbol("ischar?"));
    SymIsBool.setNewObject(new SpSymbol("isbool?"));
    SymIsReal.setNewObject(new SpSymbol("isreal?"));
    SymIsList.setNewObject(new SpSymbol("islist?"));
    SymIsNil.setNewObject(new SpSymbol("isnil?"));
    SymIsTuple.setNewObject(new SpSymbol("istuple?"));
    SymIsString.setNewObject(new SpSymbol("isstring?"));
    SymIsSymbol.setNewObject(new SpSymbol("issymbol?"));
    SymHead.setNewObject(new SpSymbol("head"));
    SymTail.setNewObject(new SpSymbol("tail"));
    SymLength.setNewObject(new SpSymbol("length"));
    SymVar.setNewObject(new SpSymbol("var"));
    SymRequire.setNewObject(new SpSymbol("require"));
    SymDo.setNewObject(new SpSymbol("do"));
    SymEnsure.setNewObject(new SpSymbol("ensure"));
    SymRescue.setNewObject(new SpSymbol("rescue"));
    SymWhile.setNewObject(new SpSymbol("while"));
    SymUntil.setNewObject(new SpSymbol("until"));
    SymVariant.setNewObject(new SpSymbol("variant"));
    SymInvariant.setNewObject(new SpSymbol("invariant"));
    SymFrom.setNewObject(new SpSymbol("from"));
    SymStep.setNewObject(new SpSymbol("step"));
    SymSoopy.setNewObject(new SpSymbol("soopy"));
    SymEnv.setNewObject(new SpSymbol("env"));
    SymClose.setNewObject(new SpSymbol("close"));
    SymRename.setNewObject(new SpSymbol("rename"));
    SymArg.setNewObject(new SpSymbol("arg"));
    SymName.setNewObject(new SpSymbol("name"));
    SymToInt.setNewObject(new SpSymbol("toint"));
    SymToString.setNewObject(new SpSymbol("tostring"));
    SymNth.setNewObject(new SpSymbol("nth"));
    SymEach.setNewObject(new SpSymbol("each"));
    SymFirst.setNewObject(new SpSymbol("first"));
    SymSecond.setNewObject(new SpSymbol("second"));
    SymThird.setNewObject(new SpSymbol("third"));
    SymFourth.setNewObject(new SpSymbol("fourth"));
    SymFifth.setNewObject(new SpSymbol("fifth"));
    SymEucjpOut.setNewObject(new SpSymbol("eucjpout"));
    SymEucjpIn.setNewObject(new SpSymbol("eucjpin"));
    SymSjisOut.setNewObject(new SpSymbol("sjisout"));
    SymSjisIn.setNewObject(new SpSymbol("sjisin"));
    SymDir.setNewObject(new SpSymbol("dir"));
    SymJapaneseIn.setNewObject(new SpSymbol("japanesein"));
    SymYear.setNewObject(new SpSymbol("year"));
    SymMonth.setNewObject(new SpSymbol("month"));
    SymDay.setNewObject(new SpSymbol("day"));
    SymSucc.setNewObject(new SpSymbol("succ"));
    SymPred.setNewObject(new SpSymbol("pred"));
    SymHour.setNewObject(new SpSymbol("hour"));
    SymMinute.setNewObject(new SpSymbol("minute"));
    SymDate.setNewObject(new SpSymbol("date"));
    SymTime.setNewObject(new SpSymbol("time"));
    SymAtime.setNewObject(new SpSymbol("atime"));
    SymCtime.setNewObject(new SpSymbol("ctime"));
    SymMtime.setNewObject(new SpSymbol("mtime"));
#ifdef USE_DATABASE
    SymDatabase.setNewObject(new SpSymbol("database"));
#endif


    symbolBool = new SpSymbol("bool");
    symbolInt  = new SpSymbol("int");
    symbolByte = new SpSymbol("byte");
    symbolWord = new SpSymbol("word");
    symbolChar = new SpSymbol("char");
    symbolReal = new SpSymbol("real");
    symbolString   = new SpSymbol("string");
    symbolFunction = new SpSymbol("function");
    symbolTuple = new SpSymbol("tuple");
    symbolList  = new SpSymbol("list");
    symbolArray = new SpSymbol("array");
    symbolNameSpace = new SpSymbol("namespace");
    symbolReader    = new SpSymbol("reader");
    symbolWriter    = new SpSymbol("writer");
    symbolEncoderIn = new SpSymbol("encoderin");
    symbolEncoderOut= new SpSymbol("encoderout");
    symbolDecoderIn = new SpSymbol("decoderin");
    symbolDecoderOut= new SpSymbol("decoderout");

    SymBool.setNewObject(symbolBool);
    SymInt.setNewObject(symbolInt);
    SymByte.setNewObject(symbolByte);
    SymWord.setNewObject(symbolWord);
    SymChar.setNewObject(symbolChar);
    SymReal.setNewObject(symbolReal);
    SymString.setNewObject(symbolString);
    SymFunction.setNewObject(symbolFunction);
    SymTuple.setNewObject(symbolTuple);
    SymList.setNewObject(symbolList);
    SymArray.setNewObject(symbolArray);
    SymNameSpace.setNewObject(symbolNameSpace);
    SymReader.setNewObject(symbolReader);
    SymWriter.setNewObject(symbolWriter);
    SymEncoderIn.setNewObject(symbolEncoderIn);
    SymEncoderOut.setNewObject(symbolEncoderOut);
    SymDecoderIn.setNewObject(symbolDecoderIn);
    SymDecoderOut.setNewObject(symbolDecoderOut);

    SymMember.setNewObject(new SpSymbol("member?"));
    SymOpen.setNewObject(new SpSymbol("open"));
    SymOpenIn.setNewObject(new SpSymbol("openin"));
    SymOpenOut.setNewObject(new SpSymbol("openout"));
    SymOpenAppend.setNewObject(new SpSymbol("openappend"));
    SymSelf.setNewObject(new SpSymbol("self"));
    SymWriteChar.setNewObject(new SpSymbol("writechar"));
    SymWrite.setNewObject(new SpSymbol("write"));
    SymWriteLine.setNewObject(new SpSymbol("writeline"));
    SymReadLine.setNewObject(new SpSymbol("readline"));
    SymFlush.setNewObject(new SpSymbol("flush"));
    SymClosed.setNewObject(new SpSymbol("closed?"));
    //SymEof.setNewObject(new SpSymbol("eof"));
    SymIsEOF.setNewObject(new SpSymbol("eof?"));
    SymUpper.setNewObject(new SpSymbol("upper"));
    SymLower.setNewObject(new SpSymbol("lower"));
    SymAppend.setNewObject(new SpSymbol("append"));
    SymKeys.setNewObject(new SpSymbol("keys"));
    SymDelete.setNewObject(new SpSymbol("delete"));
    SymSplit.setNewObject(new SpSymbol("split"));
    SymSplit1.setNewObject(new SpSymbol("split1"));
    SymSort.setNewObject(new SpSymbol("sort"));
    SymMap.setNewObject(new SpSymbol("map"));
    SymFoldl.setNewObject(new SpSymbol("foldl"));
    SymFoldr.setNewObject(new SpSymbol("foldr"));
    SymFilter.setNewObject(new SpSymbol("filter"));
    SymReplace.setNewObject(new SpSymbol("replace"));
    SymReplace1.setNewObject(new SpSymbol("replace1"));
    SymFind.setNewObject(new SpSymbol("find"));

    SymStdin.setNewObject(new SpSymbol("stdin"));
    SymStdout.setNewObject(new SpSymbol("stdout"));
    SymGet.setNewObject(new SpSymbol("get"));
    SymSet.setNewObject(new SpSymbol("set"));
    SymCurrentThread.setNewObject(new SpSymbol("current_thread"));
}

void setPrims()
{
    PrimNil.setNewObject(new SpPrim1(&returnNil));
    PrimNil2.setNewObject(new SpPrim2(&returnNil2));
    PrimNil3.setNewObject(new SpPrim3(&returnNil3));
    PrimTrue.setNewObject(new SpPrim1(&returnTrue));
    PrimFalse.setNewObject(new SpPrim1(&returnFalse));
    PrimZero.setNewObject(new SpPrim1(&returnZero));
}

void InitCommandLine(int argc, char** argv)
{
    SpValue val;
    if(argc > 0){
        SpValue temp(new SpString(*argv));
        SpCons* head = new SpCons(temp);
        argc--; argv++;
        for(;argc > 0; argc--, argv++){
            SpValue v(new SpString(*argv));
            head->append(v);
        }
        val.setNewObject(head);
    }
    SpValue SymCLine(new SpSymbol("commandline"));
    PSoopyNameSpace->internConst(SymCLine, val);
}

void InitEnv(char* env[])
{
    for(; *env != NULL; env++){
        char* ptr = *env;
        char* p = strchr(ptr, '=');
        if(p != NULL){
            *p = '\0';
            SpValue keystr(new SpString(ptr));
            SpValue valstr(new SpString(p+1));
            PEnvNameSpace->internConst(keystr, valstr);
            *p = '=';
        }
    }
}

// init SOOPY_LIB

#ifndef PATH_DELIMIT
#define PATH_DELIMIT ':'
#endif /* PATH_DELIMIT */

char* LibPaths[MaxLibPath] = {
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
  };

char LibPathBuffer[MaxLibPathBuffer];

int NumberOfLibPath = 0;

void InitLib()
{
    char* env = getenv("SOOPY_LIB");
    if(env != NULL){
        int len = strlen(env);
        if(len >= (MaxLibPathBuffer-1)){
            warning("too long env 'SOOPY_LIB'");
            len = MaxLibPathBuffer-2;
        }
        memcpy(LibPathBuffer, env, len);
        LibPathBuffer[len] = '\0';

        // f~^ŕ
        char* ptr = LibPathBuffer;
        for(NumberOfLibPath=0; true; NumberOfLibPath++){
            if(NumberOfLibPath >= MaxLibPath){
                warning("too much lib_path");
                NumberOfLibPath = MaxLibPath - 1;
                break;
            }
            LibPaths[NumberOfLibPath] = ptr;
            ptr = strchr(ptr, PATH_DELIMIT);
            if(ptr == NULL){
                NumberOfLibPath++;
                break;
            }
            *ptr = '\0';
            ptr++;
        }
    }
}

void SpInit(int argc, char* argv[], char* env[])
{
    extern void initNil();
    extern void initBool();
    extern void initChar();
    extern void initInt();
    extern void initReal();
    extern void initPrim();
    extern void initSjisFile();
    extern void initEucJPFile();
    extern void initISO2022JPFile();
    extern void initJapaneseFile();
    extern void initByteFile();
    extern void initWordFile();
    extern void initRawFile();
    extern void initCRLF2LF();
    extern void initCRLF2CR();
    extern void initLF2CRLF();
    extern void initCR2CRLF();
    extern void initCR2LF();
    extern void initLF2CR();
    extern void initBase64();
    extern void initURL();

    // init Filename
    strcpy(Filename, "STDIN");

  // init stream
    strmout = new StreamWriter(&cout);
    sjisout = new SjisWriteEncoder(strmout);
    eucjpout = new EucJPWriteEncoder(strmout);
    write_raw_encoder = new RawWriteEncoder(strmout);
    static SpValue StrmOut(strmout);
    static SpValue SjisOut(sjisout);
    static SpValue EucJPOut(eucjpout);
    static SpValue RawOut(write_raw_encoder);

    strmin = new StreamReader(&cin);
    sjisin = new SjisReadDecoder(strmin);
    eucjpin = new EucJPReadDecoder(strmin);
    japanin = new JapaneseReadDecoder(strmin);
    read_raw_encoder= new RawReadEncoder(strmin);
    static SpValue StrmIn(strmin);
    static SpValue SjisIn(sjisin);
    static SpValue EucJPIn(eucjpin);
    static SpValue JapaneseIn(japanin);
    static SpValue RawIn(read_raw_encoder);

#ifdef EUC_JP
    spout = eucjpout;
//    spin = eucjpin;
#else
    spout = sjisout;
//    spin = sjisin;
#endif
    spin = japanin;

    static SpValue Stdin(spin);
    static SpValue Stdout(spout);

    NullString.setNewObject(new SpString(""));

    setSymbols();
    setPrims();

    initNil();
    initBool();
    initChar();
    initInt();
    initReal();
    SpObject::init();
    SpString::init();
    SpSymbol::init();
    SpList::init();
    SpTuple::init();
#ifdef THREAD
    SpThread::init();
#endif /* THREAD */
    SpNameSpace::init();
#ifdef THREAD
    SpThread::init2();
    SpMutex::init();
#endif /* THREAD */
    //SpFileIn::init();
    Reader::init();
    //SpFileOut::init();
    Writer::init();
    SpArray::init();
    SpDate::init();
    SpTime::init();
    SpDateTime::init();
    SpDir::init();
    SpFile::init();
#ifdef USE_DATABASE
    SpDatabase::init();
#endif
    initPrim();
    initSjisFile();
    initEucJPFile();
    initISO2022JPFile();
    initJapaneseFile();
    initByteFile();
    initWordFile();
    initRawFile();
    initCRLF2LF();
    initCRLF2CR();
    initLF2CRLF();
    initCR2CRLF();
    initCR2LF();
    initLF2CR();
    initBase64();
    initURL();
#ifdef USE_SOCKET
    Socket::init();
#endif

    PMainNameSpace->internConst(SymStdin, Stdin);
    PMainNameSpace->internConst(SymStdout, Stdout);

    InitCommandLine(argc, argv);
    InitLib();
    InitEnv(env);

}

void SpFinalize()
{
#ifdef THREAD
  SpThread::finalize();
#endif
}


//
// class SpObject
//
const char* SpObject::toCString(WriteEncoder& encoder)
{
    //return toString().toCStringWithEncoder(encoder);
    SpValue temp;
    temp = toString();
    return temp.toCStringWithEncoder(encoder);
}

const char* SpObject::toCStringWithEncoder(WriteEncoder& encoder)
{
    //return toString().toCStringWithEncoder(encoder);
    SpValue temp;
    temp = toString();
    return temp.toCStringWithEncoder(encoder);
}

bool SpObject::operator<(SpObject& obj)
{
  //    return typeid(*this).before(typeid(obj));
  return this < &obj;
}

SpValue& SpObject::prim_toString(SpValue& self)
{
    //    static SpValue result;
    //    result = self.toString();
    //    return result;
    return SpValueResult(self.toString());
}


/*
 * Message Handler
 */

// init Message Handler
void SpObject::init()
{
    ObjectMsgHandler.append(SymIsNil, PrimFalse);
    ObjectMsgHandler.append(SymIsInt, PrimFalse);
    ObjectMsgHandler.append(SymIsChar, PrimFalse);
    ObjectMsgHandler.append(SymIsBool, PrimFalse);
    ObjectMsgHandler.append(SymIsReal, PrimFalse);
    ObjectMsgHandler.append(SymIsList, PrimFalse);
    ObjectMsgHandler.append(SymIsTuple, PrimFalse);
    ObjectMsgHandler.append(SymIsString, PrimFalse);
    ObjectMsgHandler.append(SymIsSymbol, PrimFalse);
    SpValue PrimToString(new SpPrim1(prim_toString));
    ObjectMsgHandler.append(SymToString, PrimToString);
}

SpValue& SpObject::onMessage(SpValue& rec, SpValue& msg)
{
    return SpValueResult(ObjectMsgHandler(rec, msg));
}

SpValue& SpObject::uminus(SpValue&)
{
    throw SpException("not defined operator unary -");
}

SpValue& SpObject::plus(SpValue&, SpValue&)
{
    throw SpException("not defined operator +");
}

SpValue& SpObject::minus(SpValue&, SpValue&)
{
    throw SpException("not defined operator -");
}

SpValue& SpObject::times(SpValue&, SpValue&)
{
    throw SpException("not defined operator *");
}

SpValue& SpObject::div(SpValue&, SpValue&)
{
    throw SpException("not defined operator /");
}

SpValue& SpObject::mod(SpValue&, SpValue&)
{
    throw SpException("not defined operator %");
}

SpValue& SpObject::boolNot(SpValue&)
{
    throw SpException("not defined operator !");
}

SpValue& SpObject::boolAnd(SpValue&, SpValue&)
{
    throw SpException("not defined operator &&");
}

SpValue& SpObject::boolOr(SpValue&, SpValue&)
{
    throw SpException("not defined operator ||");
}

SpValue& SpObject::eq(SpValue&, SpValue&)
{
    throw SpException("not defined operator ==");
}

SpValue& SpObject::ne(SpValue& e1, SpValue& e2)
{
    if(eq(e1,e2).isTrue()){
        return FalseObject;
    }
    return TrueObject;
}

SpValue& SpObject::gt(SpValue&, SpValue&)
{
    throw SpException("not defined operator >");
}

SpValue& SpObject::ge(SpValue& e1, SpValue& e2)
{
    if(eq(e1,e2).isTrue()){
        return TrueObject;
    }
    if(gt(e1,e2).isTrue()){
        return TrueObject;
    }
    return FalseObject;
}

SpValue& SpObject::lt(SpValue&, SpValue&)
{
    throw SpException("not defined operator <");
}

SpValue& SpObject::le(SpValue& e1, SpValue& e2)
{
    if(eq(e1,e2).isTrue()){
        return TrueObject;
    }
    if(lt(e1,e2).isTrue()){
        return TrueObject;
    }
    return FalseObject;
}

//
// class SpValue
//

// copy constructor
SpValue::SpValue(const SpValue& val)
{
    typ = val.typ;
    switch(typ){
    case TypeNil:
        pObject = NULL;
        break;
    case TypeBool:
        Bool = val.Bool;
        break;
    case TypeInt:
        Int = val.Int;
        break;
    case TypeChar:
        Char = val.Char;
        break;
    case TypeReal:
        Real = val.Real;
        break;
    case TypeObject:
        RetainObject(val.pObject);
        break;
    }
}

//SpValue::SpValue(char c, Reader& reader)
SpValue::SpValue(char c, ReadEncoder& reader)
{
    typ = TypeChar;
    Char = reader.fromChar(c);
}

const char* SpValue::typeString()
{
    switch(typ){
    case TypeNil:
        return "nil";
    case TypeBool:
        return "bool";
    case TypeInt:
        return "int";
    case TypeChar:
        return "char";
    case TypeReal:
        return "real";
    case TypeObject:
        return pObject->typeString();
    }
    throw SpException("illegal type -- system error");
}

SpObject* SpValue::getObject(){
    if(isObject()){
        if(pObject != NULL){
            return pObject;
        }else{
#ifdef TEST
            cout << "no objcet" << endl;
#endif
            throw SpException("no object");
        }
    }else{
#ifdef TEST
        cout << "NOT objcet" << endl;
#endif
        throw SpException("not object");
    }
}

SpInt SpValue::asInt(){
    if(typ != TypeInt){
        throw SpException("not int (asInt)");
    }
    return Int;
}

SpTuple* SpValue::asTuple()
{
    if(!isTuple()){
        throw SpException("not tuple");
    }
    SpTuple* ptr = dynamic_cast<SpTuple*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpNameSpace* SpValue::asNameSpace()
{
    if(!isNameSpace()){
        throw SpException("not namespace");
    }
    SpNameSpace* ptr = dynamic_cast<SpNameSpace*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpDataType* SpValue::asDataType()
{
    if(!isDataType()){
        throw SpException("not datatype");
    }
    SpDataType* ptr = dynamic_cast<SpDataType*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpConstructor* SpValue::asConstructor()
{
    if(!isConstructor()){
        throw SpException("not constructor");
    }
    SpConstructor* ptr = dynamic_cast<SpConstructor*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpArg* SpValue::asArg()
{
    if(!isArg()){
        throw SpException("not arg");
    }
    SpArg* ptr = dynamic_cast<SpArg*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpDType* SpValue::asDType()
{
    if(!isDType()){
        throw SpException("not dtype");
    }
    SpDType* ptr = dynamic_cast<SpDType*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

NSKey* SpValue::asNSKey()
{
    if(!isNSKey()){
        throw SpException("not nskey");
    }
    NSKey* ptr = dynamic_cast<NSKey*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpList* SpValue::asList()
{
    if(!isList()){
        throw SpException("not list");
    }
    SpList* ptr = dynamic_cast<SpList*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpString* SpValue::asString()
{
    if(!isString()){
        throw SpException("not string");
    }
    SpString* ptr = dynamic_cast<SpString*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

Reader* SpValue::asReader()
{
    if(!isReader()){
        throw SpException("not reader");
    }
    Reader* ptr = dynamic_cast<Reader*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

Writer* SpValue::asWriter()
{
    if(!isWriter()){
        throw SpException("not writer");
    }
    Writer* ptr = dynamic_cast<Writer*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpFunc* SpValue::asFunc()
{
    if(!isFunc()){
        throw SpException("not function");
    }
    SpFunc* ptr = dynamic_cast<SpFunc*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpArray* SpValue::asArray()
{
    if(!isArray()){
        throw SpException("not array");
    }
    SpArray* ptr = dynamic_cast<SpArray*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpByteArray* SpValue::asByteArray()
{
    if(!isByteArray()){
        throw SpException("not byte array");
    }
    SpByteArray* ptr = dynamic_cast<SpByteArray*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpSymbol* SpValue::asSymbol()
{
    if(!isSymbol()){
        throw SpException("not Symbol");
    }
    SpSymbol* ptr = dynamic_cast<SpSymbol*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpDir* SpValue::asDir()
{
    if(!isDir()){
        throw SpException("not dir");
    }
    SpDir* ptr = dynamic_cast<SpDir*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpAssign* SpValue::asAssign()
{
    if(!isAssign()){
        throw SpException("not assignment");
    }
    SpAssign* ptr = dynamic_cast<SpAssign*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

#ifdef USE_SOCKET
Socket* SpValue::asSocket()
{
    if(!isSocket()){
        throw SpException("not socket");
    }
    Socket* ptr = dynamic_cast<Socket*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

ServerSocket* SpValue::asServerSocket()
{
    if(!isServerSocket()){
        throw SpException("not server socket");
    }
    ServerSocket* ptr = dynamic_cast<ServerSocket*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}
#endif

SpDate* SpValue::asDate()
{
    if(!isDate()){
        throw SpException("not date");
    }
    SpDate* ptr = dynamic_cast<SpDate*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpTime* SpValue::asTime()
{
    if(!isTime()){
        throw SpException("not time");
    }
    SpTime* ptr = dynamic_cast<SpTime*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

SpDateTime* SpValue::asDateTime()
{
    if(!isDateTime()){
        throw SpException("not datetime");
    }
    SpDateTime* ptr = dynamic_cast<SpDateTime*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}

#ifdef USE_DATABASE
SpDatabase* SpValue::asDatabase()
{
    if(!isDatabase()){
        throw SpException("not database");
    }
    SpDatabase* ptr = dynamic_cast<SpDatabase*>(pObject);
    if(ptr == NULL){ throw SpSystemError(__FILE__, __LINE__); }
    return ptr;
}
#endif

SpValue& SpValue::operator=(const SpValue& val)
{
    //    if(this != &val){
    if(*this != (SpValue&)val){
        clearObject();
        typ = val.typ;
        switch(typ){
        case TypeNil:
            pObject = NULL;
            break;
        case TypeBool:
            Bool = val.Bool;
            break;
        case TypeInt:
            Int = val.Int;
            break;
        case TypeChar:
            Char = val.Char;
            break;
        case TypeReal:
            Real = val.Real;
            break;
        case TypeObject:
            RetainObject(val.pObject);
            break;
        }
    }
    return *this;
}

bool SpValue::operator==(SpValue& val)
{
    if(this == &val){
        return true;
    }else{
        if(typ != val.typ){
            return false;
        }else{
            switch(typ){
            case TypeNil:
                return true;
            case TypeBool:
                return Bool == val.Bool;
            case TypeInt:
                return Int == val.Int;
            case TypeChar:
                return Char == val.Char;
            case TypeReal:
                return Real == val.Real;
            case TypeObject:
                return *(getObject()) == *(val.getObject());
            }
        }
    }
    return false;
}

bool SpValue::operator<(SpValue& val)
{
    if(this == &val){
        return false;
    }else{
        if(typ != val.typ){
            return typ < val.typ;
        }else{
            switch(typ){
            case TypeNil:
                return false;
            case TypeBool:
                return Bool < val.Bool;
            case TypeInt:
                return Int < val.Int;
            case TypeChar:
                return Char < val.Char;
            case TypeReal:
                return Real < val.Real;
            case TypeObject:
                return *(getObject()) < *(val.getObject());
            }
        }
    }
    return false;
}

const char* SpValue::toCString()
{
    static char buf[1024];

    switch(typ){
    case TypeNil:
        return "nil";
    case TypeBool:
        if(Bool){
            return "true";
        }else{
            return "false";
        }
    case TypeInt:
        sprintf(buf, "%d", Int);
        break;
    case TypeChar:
        return spCharToCString(Char, *write_raw_encoder);
    case TypeReal:
        sprintf(buf, "%G", Real);
        break;
    case TypeObject:
        return getObject()->toCString();
    defaut:
        throw SpException("System error: illegal object (toCString)");
    }
    return buf;
}

const char* SpValue::toCStringWithEncoder(WriteEncoder& encoder)
{
    static char buf[1024];

    switch(typ){
    case TypeNil:
        return "nil";
    case TypeBool:
        if(Bool){
            return "true";
        }else{
            return "false";
        }
    case TypeInt:
        sprintf(buf, "%d", Int);
        break;
    case TypeChar:
        return spCharToCString(Char, encoder);
    case TypeReal:
        sprintf(buf, "%G", Real);
        break;
    case TypeObject:
        return getObject()->toCStringWithEncoder(encoder);
    defaut:
        throw SpException("System error: illegal object (toCStringWithEncoder)");
    }
    return buf;
}

SpValue& SpValue::toString()
{
    //    static SpValue val;
    SpValue val;
    if(this->isString()){
        //        val = *this;
        return *this;
    }
    //    val.setNewObject(new SpString(toCStringWithEncoder()));
    //    return val;
    return SpObjectResult(new SpString(toCStringWithEncoder()));
}

SpType SpValue::getType()
{
    if(typ == TypeObject){
        return getObject()->getType();
    }
    return typ;
}

SpValue& SpValue::eval()
{
    if(typ == TypeObject){
        SpObject* obj = getObject();
        if(obj->canEval()){
            //            return obj->eval();
            SpValue result = obj->eval();
            return SpValueResult(result);
        }else{
            return *this;
            //            return SpValueResult(*this);
        }
    }
    return *this;
}

SpValue& SpValue::onMessage(SpValue& msg)
{
    SpValue result;
    if(typ == TypeObject){
        SpObject* obj = getObject();
        SpValue result = obj->onMessage(*this, msg);
        return SpValueResult(result);
    }
    if(!msg.isSymbol()){
        throw SpIllegalMessageException(msg);
    }
    switch(typ){
    case TypeNil:
        result = NilMsgHandler(*this, msg);
	break;
    case TypeBool:
        result = BoolMsgHandler(*this, msg);
	break;
    case TypeInt:
        result = IntMsgHandler(*this, msg);
	break;
    case TypeChar:
        result = CharMsgHandler(*this, msg);
	break;
    case TypeReal:
        result = RealMsgHandler(*this, msg);
	break;
    default:
        throw SpException("system error: send message to not implemented type");
    }
    return SpValueResult(result);
}

/*
 * +,-,*,/,etc
 */
SpValue& SpValue::uminus()
{
    //    static SpValue v;
    SpValue v;

    switch(typ){
    case TypeNil:
    case TypeBool:
    case TypeChar:
        throw SpException("not defined operator unary -");
    case TypeInt:
        v.typ = TypeInt;
        v.Int = -Int;
        break;
    case TypeReal:
        v.typ = TypeReal;
        v.Real = -Real;
        break;
    case TypeObject:
        v = getObject()->uminus(*this);
    }
    //    return v;
    return SpValueResult(v);
}

// +
SpValue& SpValue::plus(SpValue& e)
{
    //    static SpValue v;
    SpValue v;

    switch(typ){
    case TypeNil:
    case TypeBool:
    case TypeChar:
        throw SpException("not defined operator +");
    case TypeInt:
        if(e.typ == TypeInt){
            v.typ = TypeInt;
            v.Int = Int + e.Int;
        }else if(e.typ == TypeReal){
            v.typ = TypeReal;
            v.Real = Int + e.Real;
        }else{
            throw SpException("type mismatch in +");
        }
        break;
    case TypeReal:
        if(e.typ == TypeReal){
            v.typ = TypeReal;
            v.Real = Real + e.Real;
        }else if(e.typ == TypeInt){
            v.typ = TypeReal;
            v.Real = Real + e.Int;
        }else{
            throw SpException("type mismatch in +");
        }
        break;
    case TypeObject:
        v = getObject()->plus(*this, e);
    }
    //    return v;
    return SpValueResult(v);
}

// -
SpValue& SpValue::minus(SpValue& e)
{
    //    static SpValue v;
    SpValue v;

    switch(typ){
    case TypeNil:
    case TypeBool:
    case TypeChar:
        throw SpException("not defined operator -");
    case TypeInt:
        if(e.typ == TypeInt){
            v.typ = TypeInt;
            v.Int = Int - e.Int;
        }else if(e.typ == TypeReal){
            v.typ = TypeReal;
            v.Real = Int - e.Real;
        }else{
            throw SpException("type mismatch in -");
        }
        break;
    case TypeReal:
        if(e.typ == TypeReal){
            v.typ = TypeReal;
            v.Real = Real - e.Real;
        }else if(e.typ == TypeInt){
            v.typ = TypeReal;
            v.Real = Real - e.Int;
        }else{
            throw SpException("type mismatch in -");
        }
        break;
    case TypeObject:
        v = getObject()->minus(*this, e);
    }
    //    return v;
    return SpValueResult(v);
}

// *
SpValue& SpValue::times(SpValue& e)
{
    //    static SpValue v;
    SpValue v;

    switch(typ){
    case TypeNil:
    case TypeBool:
    case TypeChar:
        throw SpException("not defined operator *");
    case TypeInt:
        if(e.typ == TypeInt){
            v.typ = TypeInt;
            v.Int = Int * e.Int;
        }else if(e.typ == TypeReal){
            v.typ = TypeReal;
            v.Real = Int * e.Real;
        }else{
            throw SpException("type mismatch in *");
        }
        break;
    case TypeReal:
        if(e.typ == TypeReal){
            v.typ = TypeReal;
            v.Real = Real * e.Real;
        }else if(e.typ == TypeInt){
            v.typ = TypeReal;
            v.Real = Real * e.Int;
        }else{
            throw SpException("type mismatch in *");
        }
        break;
    case TypeObject:
        v = getObject()->times(*this, e);
    }
    //    return v;
    return SpValueResult(v);
}

// /
SpValue& SpValue::div(SpValue& e)
{
    //    static SpValue v;
    SpValue v;

    switch(typ){
    case TypeNil:
    case TypeBool:
    case TypeChar:
        throw SpException("not defined operator /");
    case TypeInt:
        if(e.typ == TypeInt){
            v.typ = TypeInt;
            v.Int = Int / e.Int;
        }else if(e.typ == TypeReal){
            v.typ = TypeReal;
            v.Real = Int / e.Real;
        }else{
            throw SpException("type mismatch in /");
        }
        break;
    case TypeReal:
        if(e.typ == TypeReal){
            v.typ = TypeReal;
            v.Real = Real / e.Real;
        }else if(e.typ == TypeInt){
            v.typ = TypeReal;
            v.Real = Real / e.Int;
        }else{
            throw SpException("type mismatch in /");
        }
        break;
    case TypeObject:
        v = getObject()->div(*this, e);
    }
    //    return v;
    return SpValueResult(v);
}

// %
SpValue& SpValue::mod(SpValue& e)
{
    //    static SpValue v;
    SpValue v;

    switch(typ){
    case TypeNil:
    case TypeBool:
    case TypeChar:
    case TypeReal:
        throw SpException("not defined operator %");
    case TypeInt:
        if(e.typ == TypeInt){
            v.typ = TypeInt;
            v.Int = Int % e.Int;
        }else{
            throw SpException("type mismatch in %");
        }
        break;
    case TypeObject:
        v = getObject()->mod(*this, e);
    }
    //    return v;
    return SpValueResult(v);
}

// !
SpValue& SpValue::boolNot()
{
    //    static SpValue v;
    SpValue v;

    v.typ = TypeBool;
    switch(typ){
    case TypeNil:
    case TypeChar:
    case TypeInt:
    case TypeReal:
        throw SpException("not defined operator !");
    case TypeBool:
        v.Bool = !Bool;
        break;
    case TypeObject:
        v = getObject()->boolNot(*this);
    }
    //    return v;
    return SpValueResult(v);
}

// &&
SpValue& SpValue::boolAnd(SpValue& e)
{
    //    static SpValue v;
    SpValue v;

    v.typ = TypeBool;
    switch(typ){
    case TypeNil:
    case TypeChar:
    case TypeInt:
    case TypeReal:
        throw SpException("not defined operator &&");
    case TypeBool:
        if(e.typ == TypeBool){
            v.Bool = Bool && e.Bool;
        }else{
            throw SpException("type mismatch in &&");
        }
        break;
    case TypeObject:
        v = getObject()->boolAnd(*this, e);
    }
    //    return v;
    return SpValueResult(v);
}

// ||
SpValue& SpValue::boolOr(SpValue& e)
{
    //    static SpValue v;
    SpValue v;

    v.typ = TypeBool;
    switch(typ){
    case TypeNil:
    case TypeChar:
    case TypeInt:
    case TypeReal:
        throw SpException("not defined operator ||");
    case TypeBool:
        if(e.typ == TypeBool){
            v.Bool = Bool || e.Bool;
        }else{
            throw SpException("type mismatch in ||");
        }
        break;
    case TypeObject:
        v = getObject()->boolOr(*this, e);
    }
    //    return v;
    return SpValueResult(v);
}

// ==
SpValue& SpValue::eq(SpValue& e)
{
    //    static SpValue v;
    SpValue v;

    v.typ = TypeBool;
    v.Bool = false;
    switch(typ){
    case TypeNil:
        if(e.typ == TypeNil){
            v.Bool = true;
        }
        break;
    case TypeBool:
        if(e.typ == TypeBool){
            v.Bool = Bool == e.Bool;
        }
        break;
    case TypeChar:
        if(e.typ == TypeChar){
            v.Bool = Char == e.Char;
        }
        break;
    case TypeInt:
        if(e.typ == TypeInt){
            v.Bool = Int == e.Int;
        }else if(e.typ == TypeReal){
            v.Bool = Int == e.Real;
        }else{
            throw SpException("type mismatch in ==");
        }
        break;
    case TypeReal:
        if(e.typ == TypeReal){
            v.Bool = Real == e.Real;
        }else if(e.typ == TypeInt){
            v.Bool = Real == e.Int;
        }else{
            throw SpException("type mismatch in ==");
        }
        break;
    case TypeObject:
        v = getObject()->eq(*this, e);
    }
    //    return v;
    return SpValueResult(v);
}

// !=
SpValue& SpValue::ne(SpValue& e)
{
    //    static SpValue v;
    SpValue v;
    v = eq(e);
    v.Bool = !v.Bool;
    //    return v;
    return SpValueResult(v);
}

// >
SpValue& SpValue::gt(SpValue& e)
{
    //    static SpValue v;
    SpValue v;

    v.typ = TypeBool;
    switch(typ){
    case TypeNil:
    case TypeBool:
    case TypeChar:
        throw SpException("not defined operator >");
    case TypeInt:
        if(e.typ == TypeInt){
            v.Bool = Int > e.Int;
        }else if(e.typ == TypeReal){
            v.Bool = Int > e.Real;
        }else{
            throw SpException("type mismatch in >");
        }
        break;
    case TypeReal:
        if(e.typ == TypeReal){
            v.Bool = Real > e.Real;
        }else if(e.typ == TypeInt){
            v.Bool = Real > e.Int;
        }else{
            throw SpException("type mismatch in >");
        }
        break;
    case TypeObject:
        v = getObject()->gt(*this, e);
    }
    //    return v;
    return SpValueResult(v);
}

// >=
SpValue& SpValue::ge(SpValue& e)
{
    //    static SpValue v;
    SpValue v;

    v.typ = TypeBool;
    switch(typ){
    case TypeNil:
    case TypeBool:
    case TypeChar:
        throw SpException("not defined operator >=");
    case TypeInt:
        if(e.typ == TypeInt){
            v.Bool = Int >= e.Int;
        }else if(e.typ == TypeReal){
            v.Bool = Int >= e.Real;
        }else{
            throw SpException("type mismatch in >=");
        }
        break;
    case TypeReal:
        if(e.typ == TypeReal){
            v.Bool = Real >= e.Real;
        }else if(e.typ == TypeInt){
            v.Bool = Real >= e.Int;
        }else{
            throw SpException("type mismatch in >=");
        }
        break;
    case TypeObject:
        v = getObject()->ge(*this, e);
    }
    //    return v;
    return SpValueResult(v);
}

// <
SpValue& SpValue::lt(SpValue& e)
{
    //    static SpValue v;
    SpValue v;

    v.typ = TypeBool;
    switch(typ){
    case TypeNil:
    case TypeBool:
    case TypeChar:
        throw SpException("not defined operator <");
    case TypeInt:
        if(e.typ == TypeInt){
            v.Bool = Int < e.Int;
        }else if(e.typ == TypeReal){
            v.Bool = Int < e.Real;
        }else{
            throw SpException("type mismatch in <");
        }
        break;
    case TypeReal:
        if(e.typ == TypeReal){
            v.Bool = Real < e.Real;
        }else if(e.typ == TypeInt){
            v.Bool = Real < e.Int;
        }else{
            throw SpException("type mismatch in <");
        }
        break;
    case TypeObject:
        v = getObject()->lt(*this, e);
    }
    //    return v;
    return SpValueResult(v);
}

// <=
SpValue& SpValue::le(SpValue& e)
{
    //    static SpValue v;
    SpValue v;

    v.typ = TypeBool;
    switch(typ){
    case TypeNil:
    case TypeBool:
    case TypeChar:
        throw SpException("not defined operator <=");
    case TypeInt:
        if(e.typ == TypeInt){
            v.Bool = Int <= e.Int;
        }else if(e.typ == TypeReal){
            v.Bool = Int <= e.Real;
        }else{
            throw SpException("type mismatch in <=");
        }
        break;
    case TypeReal:
        if(e.typ == TypeReal){
            v.Bool = Real <= e.Real;
        }else if(e.typ == TypeInt){
            v.Bool = Real <= e.Int;
        }else{
            throw SpException("type mismatch in <=");
        }
        break;
    case TypeObject:
        v = getObject()->le(*this, e);
    }
    //    return v;
    return SpValueResult(v);
}

// ::
SpValue& SpValue::wcolon(SpValue& e2)
{
    if(!(e2.isList() || e2.isNil())){
        throw SpException("tail is not list in ::");
    }
    SpCons* cons = new SpCons(*this, e2);
    //    static SpValue v;
    //    v.setNewObject(cons);
    //    return v;
    SpValue v;
    v.setNewObject(cons);
    //    return SpObjectResult(cons);
    return SpValueResult(v);
}

// @
SpValue& SpValue::at(SpValue& e2)
{
    //    static SpValue result;
    SpValue result;

    if(isNil()){
        if(e2.isNil()){
            return NilObject;
        }else{
            return SpValueResult(e2);
        }
    }
    if(e2.isNil()){
        result = *this;
        //        return result;
        return SpValueResult(result);
    }

    if(!e2.isList()){
        throw SpException("tail is not list in @");
    }
    if(isCons()){
      SpValue temp;
      SpCons* ptr = dynamic_cast<SpCons*>(getObject());
      //SpCons* head = new SpCons(ptr->value());
      temp = ptr->value();
      SpCons* head = new SpCons(temp);
      SpCons* ptr2 = head;
      SpValue next = ptr->nextList();
      while(!next.isNil()){
        SpCons* ptr = dynamic_cast<SpCons*>(next.getObject());
        //SpCons* cons = new SpCons(ptr->value());
	temp = ptr->value();
        SpCons* cons = new SpCons(temp);
        SpValue v(cons);
        ptr2->setNext(v);
        ptr2 = cons;
        next = ptr->nextList();
      }

      ptr2->setNext(e2);
      result.setNewObject(head);

    }else
    if(isList()){
      SpAppendedList* a = new SpAppendedList(*this, e2, getCurrentNS());
      result.setNewObject(a);

    }else{
      throw SpException("head is not list in @");
    }
    //    return result;
    return SpValueResult(result);
}


/*
 * pattern match
 */

bool SpValue::match(SpValue& val, SpNameSpace* ns)
{
    switch(typ){
    case TypeNil:
        if(val.typ == TypeNil){
            return true;
        }
        break;
    case TypeBool:
        if(val.typ == TypeBool){
            return Bool == val.Bool;
        }
        break;
    case TypeChar:
        if(val.typ == TypeChar){
            return Char == val.Char;
        }
        break;
    case TypeInt:
        if(val.typ == TypeInt){
            return Int == val.Int;
        }
        break;
    case TypeReal:
        if(val.typ == TypeReal){
            return Real == val.Real;
        }
        break;
    case TypeObject:
        return getObject()->match(*this, val, ns);
    }
    return false;
}

