/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 * 
 * Copyright (C) 2002 SUZUKI Jun
 * 
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 * 
 * 
 * $Id: Thread.cpp,v 1.9 2004/04/14 14:01:00 randy Exp $
 */

#include "soopy.h"

//
// Soopy Thread Class
//

#ifdef USE_PTHREAD

pthread_key_t pthread_key;
std::vector<SpValue> SpThread::all_threads;

SpThread::~SpThread()
{
  std::vector<SpValue>::iterator it;
  for(it = all_threads.begin(); it != all_threads.end(); it++){
    SpThread* th = (SpThread*)it->getObject();
    if(th == this){
      all_threads.erase(it);
      break;
    }
  }
}

void SpThread::removeContext(ThreadContext* con)
{
  delete con;
}

void SpThread::run()
{
  // create thread
  if(pthread_create(&thread, NULL, (void*(*)(void*))start, this) != 0){
    throw SpException("can't create thread");
  }
}

void SpThread::start(SpThread* th)
{
  // create thread local data
  ThreadContext* context = new ThreadContext();
  pthread_setspecific(pthread_key, context);
  pushCurrentNS(th->assoc);

  // set this thread to 'current_thread'
  SpValue curThread;
  curThread.setObject(th);
  SpNameSpace* ns = th->assoc.asNameSpace();
  ns->internConst(SymCurrentThread, curThread);

  // thread main
  SpValue func = th->body;
  SpValue taker;
  SpFunc* f;
  if(func.isNameSpace()){
    SpNameSpace* ns = func.asNameSpace();
    f = SpUsrFunc::fromNameSpace(ns);
    taker.setObject(f);
  }else if(func.isFunc()){
    f = func.asFunc();
  }else{
    throw SpException("not function (thread.start)");
  }
  SpValue temp;
  temp.setObject(th);
  try{
    (*f)(temp);
  }catch(SpException& e){
    cout << "Uncaught exception in thread: " << e.what() << endl;
  }catch(const exception& e){
    cout << "Exception in thread: " << e.what() << endl;
  }
}

SpValue& SpThread::toString()
{
    SpString* str;
    char buf[32];

    sprintf(buf, "%X", &thread);
    str = new SpString();
    *str = "<thread: ";
    *str += buf;
    *str += ">";
    return SpObjectResult(str);
}

//
// primitives
//
SpValue& SpThread::prim_start(SpValue& body)
{
  SpValue result;
  SpValue ns = getCurrentNS();
  SpNameSpace* n = ns.asNameSpace();
  n->setParent(getPrevNS());
  SpThread* thread = new SpThread(ns, body);
  result.setNewObject(thread);
  all_threads.push_back(result);
  thread->run();
  return SpValueResult(result);
}

//
// instance primitives
//
SpValue& SpThread::prim_join(SpValue& th)
{
  SpThread* ptr = dynamic_cast<SpThread*>(th.getObject());
  if(ptr == NULL){
    throw SpException("not thread (join)");
  }
  if(pthread_join(ptr->thread, NULL) != 0){
    throw SpException("can't join thread");
  }

  return TrueObject;
}

SpValue& SpThread::prim_quit(SpValue& thread)
{
  SpThread* ptr = dynamic_cast<SpThread*>(thread.getObject());
  if(ptr == NULL){
    throw SpException("not thread (quit)");
  }
  pthread_exit(NULL);

  return TrueObject;
}

// init
void SpThread::init()
{
  // create key
  pthread_key_create(&pthread_key, (void(*)(void*))SpThread::removeContext);
  // make ThreadContext for main thread
  pthread_setspecific(pthread_key, new ThreadContext());
}

void SpThread::finalize()
{
  pthread_key_delete(pthread_key);
}

#endif /* USE_PTHREAD */

#ifdef USE_W32THREAD

std::vector<SpValue> SpThread::all_threads;
DWORD tls_key;

SpThread::~SpThread()
{
  std::vector<SpValue>::iterator it;
  for(it = all_threads.begin(); it != all_threads.end(); it++){
    SpThread* th = (SpThread*)it->getObject();
    if(th == this){
      all_threads.erase(it);
      break;
    }
  }
  CloseHandle(hThread);
}

void SpThread::removeContext(ThreadContext* con)
{
  delete con;
}

void SpThread::run()
{
  // create thread
  hThread = (HANDLE)_beginthreadex(NULL, 0, (unsigned(__stdcall*)(void*))start, this, 0, &id);
  if(hThread == 0){
    throw SpException("can't create thread");
  }
}

unsigned __stdcall SpThread::start(SpThread* th)
{
  // create thread local data
  ThreadContext* context = new ThreadContext();
  if(TlsSetValue(tls_key, context) == 0){
    throw SpException("can't set tls value");
  }
  pushCurrentNS(th->assoc);

  // set this thread to 'current_thread'
  SpValue curThread;
  curThread.setObject(th);
  SpNameSpace* ns = th->assoc.asNameSpace();
  ns->internConst(SymCurrentThread, curThread);

  // thread main
  SpValue func = th->body;
  SpValue taker;
  SpFunc* f;
  if(func.isNameSpace()){
    SpNameSpace* ns = func.asNameSpace();
    f = SpUsrFunc::fromNameSpace(ns);
    taker.setObject(f);
  }else if(func.isFunc()){
    f = func.asFunc();
  }else{
    throw SpException("not function (thread.start)");
  }
  SpValue temp;
  temp.setObject(th);
  (*f)(temp);

  return 0;
}

SpValue& SpThread::toString()
{
    SpString* str;
    char buf[32];

    sprintf(buf, "%X", &hThread);
    str = new SpString();
    *str = "<thread: ";
    *str += buf;
    *str += ">";
    return SpObjectResult(str);
}

//
// primitives (Thread)
//
SpValue& SpThread::prim_start(SpValue& body)
{
  SpValue result;
  SpValue ns = getCurrentNS();
  SpNameSpace* n = ns.asNameSpace();
  n->setParent(getPrevNS());
  SpThread* thread = new SpThread(ns, body);
  result.setNewObject(thread);
  all_threads.push_back(result);
  thread->run();
  return SpValueResult(result);
}

//
// instance primitives
//
SpValue& SpThread::prim_join(SpValue& thread)
{
  SpThread* ptr = dynamic_cast<SpThread*>(thread.getObject());
  if(ptr == NULL){
    throw SpException("not thread (join)");
  }
  WaitForSingleObject(ptr->hThread, INFINITE);

  return TrueObject;
}

SpValue& SpThread::prim_quit(SpValue& thread)
{
  SpThread* ptr = dynamic_cast<SpThread*>(thread.getObject());
  if(ptr == NULL){
    throw SpException("not thread (quit)");
  }
  _endthreadex(0);

  return TrueObject;
}


// init

void SpThread::init()
{
  // create key
  tls_key = TlsAlloc();
  if(tls_key == 0xffffffff){
    throw SpException("can't get TlsKey");
  }
  // make ThreadContext for main thread
  if(TlsSetValue(tls_key, new ThreadContext()) == 0){
    TlsFree(tls_key);
    throw SpException("can't set tls value");
  }
}

void SpThread::finalize()
{
  TlsFree(tls_key);
}

#endif /* USE_W32THREAD */


#ifdef THREAD

MsgHandler ThreadMsgHandler;

// onMessage
SpValue& SpThread::onMessage(SpValue& rec, SpValue& msg)
{
    return ThreadMsgHandler(rec, msg);
}

//
// init
//


SpNameSpace* PThreadNameSpace;
SpValue      ThreadNS;

void SpThread::init2()
{
  // set namespace to symbol 'thread'.
  //  SpValue parentNS(PMainNameSpace);
  //  PThreadNameSpace = new SpNameSpace(parentNS);
  PThreadNameSpace = new SpNameSpace();
  ThreadNS.setNewObject(PThreadNameSpace);
  SpValue SymThread(new SpSymbol("thread"));
  PMainNameSpace->internConst(SymThread, ThreadNS);

  SpValue SymStart(new SpSymbol("start"));
  SpValue PrimStart(new SpPrim1(prim_start));
  PThreadNameSpace->internFunc(SymStart, PrimStart);

  // init instance method
  SpValue SymJoin(new SpSymbol("join"));
  SpValue PrimJoin(new SpPrim1(prim_join));
  ThreadMsgHandler.append(SymJoin, PrimJoin);

  SpValue SymQuit(new SpSymbol("quit"));
  SpValue PrimQuit(new SpPrim1(prim_quit));
  ThreadMsgHandler.append(SymQuit, PrimQuit);
}

#endif /* THREAD */

