/*
 * Programming Language SOOPY
 *   (Simple Object Oriented Programming sYstem)
 *
 * Copyright (C) 2004 SUZUKI Jun
 *
 * URL: http://sourceforge.jp/projects/soopy/
 * License: GPL(GNU General Public License)
 *
 *
 * $Id: TableView.mm,v 1.12 2004/05/16 00:10:44 randy Exp $
 */

#include "soopyg.h"

static SpValue GetRow;
static SpValue SetRow;
static SpValue GetCol;
static SpValue SetCol;
static SpValue GetTable;


@implementation SpNSTableViewDataSource

-(id)initWithRows:(unsigned int)r Columns:(unsigned int)c View:(NSView*)v
{
  rows = r;
  cols = c;
  view = v;
  return [self initWithCapacity: (rows * cols)];
}

-(int)numberOfRowsInTableView:(NSTableView*)aTableView
{
  return rows;
}

-(id)tableView:(NSTableView*)aTableView objectValueForTableColumn:(NSTableColumn*)column row:(int)rowIndex
{
  int col = [[column identifier] intValue];
  if(col < 0){
    return NULL;
  }
  int index = (rowIndex * cols) + col;
  return [ary objectAtIndex:index];
}

-(void)tableView:(NSTableView*)aTableView setObjectValue:(id)anObject forTableColumn:(NSTableColumn*)column row:(int)rowIndex
{
  int col = [[column identifier] intValue];
  if(col < 0){
    return;
  }
  int index = (rowIndex * cols) + [aTableView selectedColumn];
  [ary replaceObjectAtIndex:index withObject:anObject];
  [aTableView display];
}

-(id)objectAtRow:(int)row atColumn:(int)col
{
  int index = (row * cols) + col;
  return [ary objectAtIndex:index];
}

-(void)setObject:(id)anObject atRow:(int)row atColumn:(int)col
{
  int index = (row * cols) + col;
  [ary replaceObjectAtIndex:index withObject:anObject];
  [view display];
}

@end

/*
 * class NSTableViewForTableView
 */
@implementation NSTableViewForTableView

-(id)objectAtRow:(int)row atColumn:(int)col
{
  return [[self dataSource] objectAtRow:row atColumn:col];
}

-(void)setObject:(id)anObject atRow:(int)row atColumn:(int)col
{
  [[self dataSource] setObject:anObject atRow:row atColumn:col];
}

@end

/*
 * class SpNSTableView
 */
@implementation SpNSTableView

-(id)initWithNameSpace:(SpNameSpace*)assoc parent:(NSObject*)parent comp:(SpComponent*)spc
{
    ns = assoc;
    spcomp = spc;
    // set window origin & size
    SpValue vtop = assoc->lookup(SymTop);
    SpValue vleft = assoc->lookup(SymLeft);
    SpValue vwidth = assoc->lookup(SymWidth);
    SpValue vheight = assoc->lookup(SymHeight);

    NSRect rect = [(NSView*)[parent contentView] frame];
    float left = (float)vleft.getInt();
    float width = (float)vwidth.getInt();
    float height = (float)vheight.getInt();
    float top = rect.size.height - (float)vtop.getInt() - height;
    NSRect frame = NSMakeRect(left, top, width, height);
    float scrollerWidth = [NSScroller scrollerWidth] + 2;
    rect = NSMakeRect(left, top, width-scrollerWidth, height-scrollerWidth);
    [self initWithFrame: frame];

    // set cols & rows
    SpValue vcols  = assoc->lookup(SymCols);
    SpValue vrows  = assoc->lookup(SymRows);
    SpInt cols     = vcols.asInt();
    SpInt rows     = vrows.asInt();
    SpValue header = assoc->lookup(SymHeader);
    if(!header.isList()){
      header = NilObject;
    }

    //
    // create TableView
    //
    //    NSTableViewForTableView* tableView;
    
    // Create NSTableView
    tableView = [[NSTableViewForTableView alloc] initWithFrame:rect];
    if(header.isNil()){
      [tableView setHeaderView: nil];
    }
    // set rows & cols
    for(int i=0; i < cols; i++){
        NSTableColumn* column = [[NSTableColumn alloc] initWithIdentifier:[NSNumber numberWithInt: i]];
	[column setWidth:80.0];
	if(!header.isNil()){
	  SpList* list = header.asList();
	  [column setHeaderCell: [[NSTableHeaderCell alloc]
				   initTextCell: [NSString stringWithCString: list->value().toCString()]]];  // set header
	  header = list->nextList();
	}
	[tableView addTableColumn:column];
	[column release];
    }
    SpNSTableViewDataSource* source = [[SpNSTableViewDataSource alloc] initWithRows:rows Columns:cols View:tableView];
    [tableView setDataSource:source];
    [tableView setTarget: (id)self];
    [tableView setAction:@selector(change:)];

    [self setBorderType:NSBezelBorder];
    [self setHasVerticalScroller:YES];
    [self setHasHorizontalScroller:NO];
    //    [self setAutoresizingMask:NSViewWidthSizable | NSViewHeightSizable];

    [self setDocumentView:tableView];
    [tableView release];

    return self;
}

-(SpNameSpace*)getNameSpace
{
  return ns;
}

-(SpComponent*)getSpComponent
{
  return spcomp;
}

-(void)change:(NSObject*)sender
{
    SpValue onChange  = ns->lookup(SymOnChange);
    if(onChange.isFunc()){
        SpFunc* f = onChange.asFunc();
        (*f)(NilObject);
    }
}

-(int)count
{
  return [tableView count];
}

-(int)selectedColumn
{
  return [tableView selectedRow];
}

-(void)selectColumn:(int)col byExtendingSelection:(BOOL)flag
{
  [tableView selectColumn: col byExtendingSelection: flag];
}

-(int)selectedRow
{
  return [tableView selectedRow];
}

-(void)selectRow:(int)col byExtendingSelection:(BOOL)flag
{
  [tableView selectRow: col byExtendingSelection: flag];
}

-(id)objectAtIndex:(int)index
{
  return [tableView objectAtIndex:index];
}

-(id)objectAtRow:(int)row atColumn:(int)col
{
  return [tableView objectAtRow:row atColumn:col];
}

-(void)setObject:(id)anObject atRow:(int)row atColumn:(int)col
{
  [tableView setObject:anObject atRow:row atColumn:col];
}

@end

/*
 * class SpTableView
 */

SpValue& SpTableView::make()
{
    SpTableView* table = new SpTableView(getCurrentNS());
    return SpObjectResult(table);
}

// Caption
SpValue& SpTableView::getCaption(NSObject* w)
{
    throw SpCaptionException("can't get tableview.caption");
}

void SpTableView::setCaption(NSObject* w, const char* str)
{
    throw SpCaptionException("can't set tableview.caption");
}

void SpTableView::creation(SpValue& parentNS, NSObject* parent)
{
    SpNameSpace* assoc = ns.asNameSpace();

    // Create NSScrollView(SpNSTableView)
    SpNSTableView* tableview = [[SpNSTableView alloc] initWithNameSpace:assoc parent:parent comp:this];
    [[parent contentView] addSubview: (NSView*)tableview];
    [tableview setAutoresizingMask: NSViewMinYMargin];
    widget = (NSObject*)tableview;

    // Table
    assoc->internPrimProperty(SymTable, NilObject, GetTable, NilObject);

    // set self
    SpValue self(this);
    assoc->internConst(SymSelf, self);

    // set align
    SpValue valign  = assoc->lookup(SymAlign);
    setAlign((Align)valign.asInt());

    initProperties(assoc); // SpComponent::initProperties
}

// primitives
SpValue& SpTableView::primGetRow()
{
    SpValue n = getCurrentNS();
    SpNameSpace* ns = n.asNameSpace();
    SpValue self = ns->lookup(SymSelf);
    SpComponent* comp = self.asComponent();
    SpNSTableView* tview = (SpNSTableView*)comp->widget;

    SpValue result;
    result.setInt([tview selectedRow]);
    return SpValueResult(result);
}

SpValue& SpTableView::primSetRow(SpValue& value)
{
    if(!value.isInt()){
        throw SpException("not int(tableview.row)");
    }
    SpInt i = value.asInt();
    SpValue n = getCurrentNS();
    SpNameSpace* ns = n.asNameSpace();
    SpValue self = ns->lookup(SymSelf);
    SpComponent* comp = self.asComponent();
    SpNSTableView* tview = (SpNSTableView*)comp->widget;
    [tview selectRow: i byExtendingSelection: NO];

    return TrueObject;
}

SpValue& SpTableView::primGetCol()
{
    SpValue n = getCurrentNS();
    SpNameSpace* ns = n.asNameSpace();
    SpValue self = ns->lookup(SymSelf);
    SpComponent* comp = self.asComponent();
    SpNSTableView* tview = (SpNSTableView*)comp->widget;

    SpValue result;
    result.setInt([tview selectedColumn]);
    return SpValueResult(result);
}

SpValue& SpTableView::primSetCol(SpValue& value)
{
    if(!value.isInt()){
        throw SpException("not int(tableview.row)");
    }
    SpInt i = value.asInt();
    SpValue n = getCurrentNS();
    SpNameSpace* ns = n.asNameSpace();
    SpValue self = ns->lookup(SymSelf);
    SpComponent* comp = self.asComponent();
    SpNSTableView* tview = (SpNSTableView*)comp->widget;
    [tview selectColumn: i byExtendingSelection: NO];

    return TrueObject;
}

SpValue& SpTableView::primGetTable()
{
    SpValue n = getCurrentNS();
    SpNameSpace* ns = n.asNameSpace();
    SpValue self = ns->lookup(SymSelf);
    SpComponent* comp = self.asComponent();
    SpNSTableView* tview = (SpNSTableView*)comp->widget;

    return SpObjectResult(new TableOfTableView(tview));
}

// init

void SpTableView::initProperties(SpNameSpace* pNS)
{
    SpComponent::initProperties(pNS);

    // Row
    pNS->internPrimProperty(SymRow, NilObject, GetRow, SetRow);
    // Column
    pNS->internPrimProperty(SymCol, NilObject, GetCol, SetCol);
/*
    // Table
    pNS->internPrimProperty(SymTable, NilObject, GetTable, NilObject);
*/
}

void SpTableView::init()
{
    // set namespace to symbol 'std_tableview'.
    PTableViewNameSpace = new SpNameSpace(PComponentNameSpace->getMap());
    TableViewNS.setNewObject(PTableViewNameSpace);
    SpValue SymStdTableView(new SpSymbol("std_tableview"));
    PMainNameSpace->internConst(SymStdTableView, TableViewNS);

    // set default top, left, width, height
    SpValue I;
    I.setInt(80);
    PListBoxNameSpace->internConst(SymWidth, I);
    I.setInt(97);
    PListBoxNameSpace->internConst(SymHeight, I);

    GetRow.setNewObject(new SpPrim0(primGetRow));
    SetRow.setNewObject(new SpPrim1(primSetRow));
    GetCol.setNewObject(new SpPrim0(primGetCol));
    SetCol.setNewObject(new SpPrim1(primSetCol));
    GetTable.setNewObject(new SpPrim0(primGetTable));

    SpValue PrimMake(new SpPrim0(make));
    PTableViewNameSpace->internFunc(SymMake, PrimMake);
}

/*
 * class TableOfTableView
 */

SpValue& TableOfTableView::toString()
{
    return SpObjectResult(new SpString("<table of tableview>"));
}

SpValue& TableOfTableView::operator[](SpValue& tuple)
{
    if(!tuple.isTuple()){
        throw SpException("few arguments (access table)");
    }
    SpTuple* t = dynamic_cast<SpTuple*>(tuple.getObject());
    if(t->length() != 2){
        throw SpException("too much arguments (access table)");
    }
    SpValueVector::iterator tit;
    tit = t->begin();
    SpValue i = tit->eval();
    if(!i.isInt()){
        throw SpException("not int (access table)");
    }
    tit++;
    SpValue j = tit->eval();
    if(!j.isInt()){
        throw SpException("not int (access table)");
    }
    int col = i.getInt();
    int row = j.getInt();

    return SpObjectResult(new SpString([[view objectAtRow:row atColumn:col] cString]));
}

void TableOfTableView::set(SpValue& tuple, SpValue& value)
{
    if(!tuple.isTuple()){
        throw SpException("few arguments (access table)");
    }
    SpTuple* t = dynamic_cast<SpTuple*>(tuple.getObject());
    if(t->length() != 2){
        throw SpException("too much arguments (access table)");
    }
    SpValueVector::iterator tit;
    tit = t->begin();
    SpValue i = tit->eval();
    if(!i.isInt()){
        throw SpException("not int (access table)");
    }
    tit++;
    SpValue j = tit->eval();
    if(!j.isInt()){
        throw SpException("not int (access table)");
    }
    int col = i.getInt();
    int row = j.getInt();
    [view setObject:[NSString stringWithCString:value.toString().toCString()]
	  atRow:row
	  atColumn:col];
}

SpValue& TableOfTableView::onMessage(SpValue& rec, SpValue& msg)
{
//    if(TableMsgHandler.hasMessage(msg)){
//        return TableMsgHandler(rec, msg);
//    }
    SpValue val = msg.eval();
    if(val.isTuple()){
        return operator[](val);
    }

    // Error
    static char buf[256];
    strncpy(buf, rec.toCStringWithEncoder(), 255);
    buf[255] = '\0';
    static char buf2[256];
    strncpy(buf2, msg.toCStringWithEncoder(), 255);
    buf2[255] = '\0';
    throw SpNoMethodException("no such a feature", buf, buf2);
}

