﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Collections.ObjectModel;
using System.Speech.Synthesis;
using System.Globalization; 
using System.Text.RegularExpressions;
using System.IO;

namespace SpeechCast
{
    public partial class FormMain : Form
    {
        public FormMain()
        {
            InitializeComponent();

            foreach (InstalledVoice voice in synthesizer.GetInstalledVoices())
            {
                toolStripComboBoxSelectVoice.Items.Add(voice.VoiceInfo.Name);
            }

            synthesizer.SpeakProgress += new EventHandler<SpeakProgressEventArgs>(synthesizer_SpeakProgress);
            synthesizer.SpeakCompleted += new EventHandler<SpeakCompletedEventArgs>(synthesizer_SpeakCompleted);

            FormCaption.Instance = new FormCaption();

            for (int vol = 100; vol >= 0; vol -= 10)
            {
                toolStripComboBoxVolume.Items.Add(vol);
            }
        }



        void synthesizer_SpeakCompleted(object sender, SpeakCompletedEventArgs e)
        {
            speakingCompletedTime = System.DateTime.Now;
            FormCaption.Instance.CaptionText = speakingText;
            isSpeaking = false;
            if (currentResNumber <= 1000)
            {
                currentResNumber++;
            }
        }

        System.DateTime speakingCompletedTime;
        System.DateTime gettingWebTime;


        void synthesizer_SpeakProgress(object sender, SpeakProgressEventArgs e)
        {
            int index = e.CharacterPosition + e.CharacterCount;

            if (replacementIndices != null)
            {
                if (index > 0)
                {
                    index--;
                }

                if (index < replacementIndices.Count)
                {
                    index = replacementIndices[index] + 1;
                }
                else
                {
                    index = speakingText.Length;
                }
            }

            string speackStr = speakingText.Substring(0, index);

            FormCaption.Instance.CaptionText = speackStr;
            //AddLog(speackStr);
        }

        private SpeechSynthesizer synthesizer = new SpeechSynthesizer();

        private void toolStripComboBoxSelectVoice_SelectedIndexChanged(object sender, EventArgs e)
        {
            int idx = toolStripComboBoxSelectVoice.SelectedIndex;

            if (idx >= 0)
            {
                string voiceName = toolStripComboBoxSelectVoice.Items[idx].ToString();
                synthesizer.SelectVoice(voiceName);
                UserConfig.VoiceName = voiceName;
            }
        }

        //private void button1_Click(object sender, EventArgs e)
        //{
         //   synthesizer.Speak(textBoxTest.Text);
        //}

        private void toolStripButtonEnter_Click(object sender, EventArgs e)
        {
            GetFromURL();
        }

        private void toolStripTextBoxURL_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyData == Keys.Enter)
            {
                GetFromURL();
            }
        }

        private string rawURL = null;

        private void GetFromURL()
        {
            try
            {
                GetFromURL(false);
            }
            catch (Exception ex)
            {
                AddLog("Exception: {0}", ex.Message);
            }
            autoUpdate = false;
        }

        Regex jbbsRegex = new System.Text.RegularExpressions.Regex(@"(http://jbbs.livedoor.jp/bbs/)read.cgi(/\w+/(\d+)/(\d+)/)");
        Regex yyRegex = new System.Text.RegularExpressions.Regex(@"(http://yy.+\.(kg|com))/.+/read.cgi(/\w+)/(\d+)/");
        string encodingName = "";

        int datSize = 0;

        private Cursor pushedCursor = null;

        public void PushAndSetWaitCursor()
        {
            pushedCursor = this.Cursor;
            this.Cursor = Cursors.WaitCursor;
        }

        public void PopCursor()
        {
            Cursor = pushedCursor;
        }

        private void GetFromURL(bool next)
        {
            System.Net.WebClient webClient = new System.Net.WebClient();

            string url = null;
            bool clearItems = true;

            if (next && rawURL != null)
            {
                switch (Response.Style)
                {
                    case Response.BBSStyle.jbbs:
                        url = string.Format("{0}{1}-", rawURL, responses.Count + 1);
                        clearItems = false;
                        break;
                    case Response.BBSStyle.yykakiko:
                        url = rawURL;
                        clearItems = true;
                        //string dataTimeStr = string.Format("If-Modified-Since: {0}", gettingWebTime.ToUniversalTime());
                        //webClient.Headers.Add(dataTimeStr);
                        //string rangeStr = string.Format("Range: bytes={0}-", datSize);
                        //webClient.Headers.Add(rangeStr);
                        break;
                }
            }
            else
            {
                Match m = jbbsRegex.Match(toolStripTextBoxURL.Text);
                if (m.Success)
                {
                    rawURL = m.Groups[1].Value + "rawmode.cgi" + m.Groups[2];
                    AddLog("jbbs rawmode: {0}", rawURL);
                    Response.Style = Response.BBSStyle.jbbs;
                    encodingName = "EUC-JP";
                }
                else
                {
                    m = yyRegex.Match(toolStripTextBoxURL.Text);
                    if (m.Success)
                    {
                        rawURL = m.Groups[1].Value + m.Groups[3].Value + "/dat/" + m.Groups[4].Value + ".dat";
                        Response.Style = Response.BBSStyle.jbbs;
                    }
                    Response.Style = Response.BBSStyle.yykakiko;

                    AddLog("yykakiko dat mode: {0}", rawURL);

                    encodingName = "Shift_JIS";
                    
                }


                if (rawURL == null)
                {
                    MessageBox.Show("サポートしていないＵＲＬです");
                    return;
                }
                url = rawURL;
            }

            listViewResponses.BeginUpdate();
            PushAndSetWaitCursor();
            try
            {
                if (clearItems)
                {
                    datSize = 0;
                    responses.Clear();
                    listViewResponses.Items.Clear();
                }

                try
                {
                    using (MemoryStream memStream = new MemoryStream())
                    {
                        byte[] buf = new byte[1024];
                        using (Stream data = webClient.OpenRead(url))
                        {
                            while (true)
                            {
                                int size = data.Read(buf, 0, buf.Length);

                                if (size == 0)
                                {
                                    break;
                                }
                                memStream.Write(buf, 0, size);
                            }

                        }

                        gettingWebTime = System.DateTime.Now;

                        memStream.Seek(0, SeekOrigin.Begin);
                        using (StreamReader reader = new StreamReader(memStream, Encoding.GetEncoding(encodingName)))
                        {
                            while (true)
                            {
                                string s = reader.ReadLine();
                                if (s == null)
                                {
                                    break;
                                }
                                //AddLog("{0}",s);

                                Response res = new Response();

                                res.Number = responses.Count + 1; //先に入れておかないとHTMLが正しく作成されない
                                if (res.SetRawText(s))
                                {
                                    responses.Add(res);
                                    listViewResponses.Items.Add(res.CreateListViewItem());
                                }
                            }
                            //AddLog("Stream Size = {0}bytes", memStream.Length);

                            datSize += (int)memStream.Length;
                            //AddLog("dat Size = {0}bytes", datSize);

                        }
                    }
                }
                catch (System.Net.WebException we)
                {
                    AddLog("WebException: status={0}", we.Status.ToString());
                    if (we.Response != null && we.Response.Headers != null)
                    {
                        for (int j = 0; j < we.Response.Headers.Count; j++)
                        {
                            AddLog("WebException: header {0}={1}", we.Response.Headers.Keys[j], we.Response.Headers[j]);

                        }
                    }
                }

                string[] htmlStrings = new string[responses.Count];

                int i = 0;
                foreach (Response res in responses)
                {
                    htmlStrings[i] = res.Html;
                    i++;
                }

                string html = string.Format("<html><body>{0}</body></html>"
                                    , string.Concat(htmlStrings)
                                    );
                if (responses.Count != 0)
                {
                    this.Text = string.Format("SpeechCast - {0}", responses[0].ThreadTitle);
                }

                try
                {
                    webBrowser.DocumentText = html;
                }
                catch (Exception ex)
                {
                    AddLog("webBrowser Exception: {0}", ex.Message);
                }
            }
            finally
            {
                listViewResponses.EndUpdate();
                PopCursor();
            }
        }

        private List<Response> responses = new List<Response>();

        private void AddLog(string format, params object[] args)
        {
            string str = string.Format(format, args);

            AddLog(str);
        }

        private void AddLog(string str)
        {
            textBoxLog.AppendText(str + "\r\n");
        }

        private void listViewResponses_Click(object sender, EventArgs e)
        {
            Response res = GetSelectedResonse();

            if (res != null)
            {
                webBrowser.Document.Window.ScrollTo(0, res.ScrollY);
            }
        }

        private void webBrowser_DocumentCompleted(object sender, WebBrowserDocumentCompletedEventArgs e)
        {
            Regex regexRes = new Regex(@"res(\d+)");

            int scrollY = -1;
            foreach (HtmlElement elem in webBrowser.Document.GetElementsByTagName("a"))
            {
                Match mm = regexRes.Match(elem.Name);

                if (mm.Success)
                {
                    int no = System.Convert.ToInt32(mm.Groups[1].Value);

                    if (no >= 1 && no <= responses.Count)
                    {
                        responses[no - 1].ScrollY = elem.OffsetRectangle.Y;
                        scrollY = elem.OffsetRectangle.Y;
                    }
                }
            }

            if (autoUpdate && scrollY >= 0)
            {
                webBrowser.Document.Window.ScrollTo(0, scrollY);
            }
        }

        private int currentResNumber_ = 0;

        private int currentResNumber
        {
            get
            {
                return this.currentResNumber_;
            }
            set
            {
                this.currentResNumber_ = value;
                this.toolStripStatusLabelResNumber.Text = string.Format("レス番号{0}", value);
            }
        }

        private bool autoUpdate
        {
            get
            {
                return toolStripButtonAutoUpdate.Checked;
            }

            set
            {
                if (toolStripButtonAutoUpdate.Checked != value)
                {

                    toolStripButtonAutoUpdate.Checked = value;

                    if (value)
                    {
                        Response res = GetSelectedResonse();

                        if (res != null)
                        {
                            currentResNumber = res.Number;
                        }
                        else
                        {
                            currentResNumber = responses.Count + 1;
                        }

                        StartSpeaking();
                    }
                }
            }
        }

        private Response GetSelectedResonse()
        {
            if (listViewResponses.SelectedItems.Count != 0)
            {
                Response res = listViewResponses.SelectedItems[0].Tag as Response;

                return res;
            }
            return null;
        }

        private bool isSpeaking = false;
        private string speakingText = "";
        private void StartSpeaking()
        {
            isSpeaking = true;

            synthesizer.Rate = UserConfig.SpeakingRate;
            if (currentResNumber >= 1 && currentResNumber <= responses.Count)
            {
                Response res = responses[currentResNumber - 1];

                string text = res.Text;

                if (UserConfig.IncludesNGWord(res.RawText))
                {
                    text = "(NG Word)";
                }
                else
                {
                    if (text.Length > UserConfig.MaxSpeakingCharacterCount)
                    {
                        text = text.Substring(0, UserConfig.MaxSpeakingCharacterCount) + "\n(長文のため以下省略)";
                    }

                    //Regex rx1 = new Regex("w+", RegexOptions.IgnoreCase);
                    //Regex rx2 = new Regex("ｗ+", RegexOptions.IgnoreCase);

                    //text = rx1.Replace(text, "w");
                    //text = rx2.Replace(text, "w");
                }

                if (UserConfig.SpeaksResNumber)
                {
                    text = string.Format("レス{0}\n{1}", res.Number, text);
                }


                speakingText = text;

                //AddLog("Speack={0}", speakingText);

                pronounciationText = UserConfig.ReplacePronouncationWords(text, out replacementIndices);

                //AddLog("Pronoun={0}", pronounciationText);

                StringBuilder sb = new StringBuilder();
                for (int i = 0; i < pronounciationText.Length; i++)
                {
                    sb.AppendFormat("{0:d2} ", replacementIndices[i]);
                }
                //AddLog("Indices={0}", sb.ToString());

                FormCaption.Instance.CaptionText = "";

                synthesizer.SpeakAsync(pronounciationText);
                listViewResponses.Items[currentResNumber - 1].Selected = true;
                webBrowser.Document.Window.ScrollTo(0, res.ScrollY);
            }
            else if (currentResNumber > 1000)
            {
                speakingText = string.Format("レス1000を超えました。\nこれ以上は表示できません。\n次スレを立ててください。");

                replacementIndices = null;
                synthesizer.SpeakAsync(speakingText);
            }
            else
            {
                speakingCompletedTime = System.DateTime.Now;
                isSpeaking = false;
            }
        }

        private List<int> replacementIndices;
        private string pronounciationText;

        private void toolStripButtonCaption_Click(object sender, EventArgs e)
        {
            FormCaption.Instance.Visible = !FormCaption.Instance.Visible;

            toolStripButtonCaption.Checked = FormCaption.Instance.Visible;
        }

        private void toolStripButtonBorder_Click(object sender, EventArgs e)
        {
            FormCaption.Instance.BorderVisible = !FormCaption.Instance.BorderVisible;

            toolStripButtonBorder.Checked = FormCaption.Instance.BorderVisible;
        }

        private void listViewResponses_DoubleClick(object sender, EventArgs e)
        {
            Response res = GetSelectedResonse();

            if (res != null)
            {
                currentResNumber = res.Number;

                if (synthesizer.State == SynthesizerState.Ready)
                {
                    StartSpeaking();
                }
            }
        }

        private void toolStripButtonAutoUpdate_Click(object sender, EventArgs e)
        {
            autoUpdate = !autoUpdate;
        }


        private void timer_Tick(object sender, EventArgs e)
        {
            if (isSpeaking == false)
            {
                TimeSpan diff = System.DateTime.Now - speakingCompletedTime;
                if (diff.TotalMilliseconds >= UserConfig.SpeakingInvervalMillsec)
                {
                    if (currentResNumber <= responses.Count && autoUpdate)
                    {
                        StartSpeaking();
                    }
                    else
                    {
                        FormCaption.Instance.CaptionText = "";                        
                    }
                }

                TimeSpan diffWeb = System.DateTime.Now - gettingWebTime;

                if (currentResNumber > responses.Count && autoUpdate && diffWeb.TotalMilliseconds >= UserConfig.AutoGettingWebInvervalMillsec)
                {
                    if (currentResNumber <= 1000)
                    {
                        GetFromURL(true);
                    }
                    else
                    {
                        if (diff.TotalMilliseconds >= UserConfig.SpeakingInvervalMillsec)
                        {
                            StartSpeaking();
                        }
                    }
                }
            }
        }

        public static UserConfig UserConfig;

        private void FormMain_Load(object sender, EventArgs e)
        {
            string userConfigPath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData), @"SpeachCast\Application.cfg");

            if (File.Exists(userConfigPath))
            {
                try
                {
                    UserConfig = UserConfig.Deserialize(userConfigPath);

                    UserConfig.SetRectToForm(UserConfig.FormMainRect, this);
                    UserConfig.SetRectToForm(UserConfig.FormCaptionRect, FormCaption.Instance);
                    toolStripTextBoxURL.Text = UserConfig.URL;
                }
                catch (Exception ex)
                {
                    AddLog("Deserialze error :{0}", ex.Message);
                    UserConfig = new UserConfig(userConfigPath);
                }
            }
            else
            {
                UserConfig = new UserConfig(userConfigPath);
            }


            int selectedIndex = 0;
            int idx = toolStripComboBoxSelectVoice.Items.IndexOf(UserConfig.VoiceName);
            if (idx >= 0)
            {
                selectedIndex = idx;
            }

            if (selectedIndex < toolStripComboBoxSelectVoice.Items.Count)
            {
                toolStripComboBoxSelectVoice.SelectedIndex = selectedIndex;
            }

            selectedIndex = 0;
            idx = toolStripComboBoxVolume.Items.IndexOf(UserConfig.SpeakingVolume);
            if (idx >= 0)
            {
                selectedIndex = idx;
            }

            if (selectedIndex < toolStripComboBoxVolume.Items.Count)
            {
                toolStripComboBoxVolume.SelectedIndex = selectedIndex;
            }

        }

        private void FormMain_FormClosing(object sender, FormClosingEventArgs e)
        {

            UserConfig.SetFormToRect(ref UserConfig.FormMainRect, this);
            UserConfig.SetFormToRect(ref UserConfig.FormCaptionRect, FormCaption.Instance);
            UserConfig.URL = toolStripTextBoxURL.Text;
            UserConfig.Serialize();
        }

        private void toolStripButtonSettings_Click(object sender, EventArgs e)
        {
            FormSettings form = new FormSettings();


            form.SetUserConfig(UserConfig);

            try
            {
                form.NumericUpDownMaxSpeakingCharacterCount.Value = UserConfig.MaxSpeakingCharacterCount;
            }
            catch //(Exception ex)
            {
            }
            form.CaptionFont = UserConfig.CaptionFont;
            try
            {
                form.NumericUpDownAutoGettingWebInverval.Value = UserConfig.AutoGettingWebInvervalMillsec;
            }
            catch //(Exception ex)
            {
            }

            if (form.ShowDialog() == DialogResult.OK)
            {
                form.GetUserConfig(UserConfig);
                FormCaption.Instance.Refresh();
            }
        }

        private void toolStripComboBoxVolume_SelectedIndexChanged(object sender, EventArgs e)
        {
            int vol = (int)toolStripComboBoxVolume.SelectedItem;

            UserConfig.SpeakingVolume = vol;
            synthesizer.Volume = vol;
            
        }
    }
}
