﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;

namespace FDK
{
	public static class Utilities
	{
		/// <summary>
		/// Dispose して、null を代入する。
		/// </summary>
		/// <param name="obj">IDisposable を実装するオブジェクト。</param>
		public static void 解放する<T>( ref T obj ) where T : IDisposable
		{
			( obj as IDisposable )?.Dispose();
			obj = default( T );
		}

		/// <summary>
		/// ターゲットに対して描画処理を実行する。
		/// </summary>
		/// <remarks>
		/// 描画処理は、ターゲットの BeginDraw() と EndDraw() の間で行われることを保証する。
		/// 描画処理中に例外が発生しても EndDraw() の呼び出しは保証する。
		/// </remarks>
		/// <param name="target">レンダーターゲット。</param>
		/// <param name="描画処理">BeginDraw() と EndDraw() の間で行う処理。</param>
		public static void D2DBatchDraw( SharpDX.Direct2D1.RenderTarget target, Action 描画処理 )
		{
			target.BeginDraw();
			try
			{
				描画処理();
			}
			finally
			{
				// SharpDX の EndDraw() では HRESULT を捨ててしまうので、デバイスロストのチェックは、ここではできない。
				target.EndDraw();
			}
		}
		/// <summary>
		/// 深度から射影行列（定数）を計算して返す。
		/// Direct2D 用。
		/// </summary>
		/// <param name="depth">深度。</param>
		/// <returns>深度から計算された射影行列。</returns>
		public static SharpDX.Matrix D2DPerspectiveProjection( float depth )
		{
			var mat = SharpDX.Matrix.Identity;
			mat.M34 = ( 0 != depth ) ? -( 1.0f / depth ) : 0.0f;
			return mat;
		}

		public static int 最大公約数を返す( int m, int n )
		{
			if( ( 0 >= m ) || ( 0 >= n ) )
				throw new FDKException( "引数に0以下の数は指定できません。" );

			// ユーグリッドの互除法
			int r;
			while( ( r = m % n ) != 0 )
			{
				m = n;
				n = r;
			}

			return n;
		}
		public static int 最小公倍数を返す( int m, int n )
		{
			if( ( 0 >= m ) || ( 0 >= n ) )
				throw new FDKException( "引数に0以下の数は指定できません。" );

			return ( m * n / Utilities.最大公約数を返す( m, n ) );
		}

		/// <summary>
		/// このメソッドの 呼び出し元のメソッド名 を返す。デバッグログ用。
		/// </summary>
		public static string 現在のメソッド名
		{
			get
			{
				// 1つ前のスタックフレームを取得。
				var prevFrame = new System.Diagnostics.StackFrame( skipFrames: 1, fNeedFileInfo: false );

				var クラス名 = prevFrame.GetMethod().ReflectedType.ToString();
				var メソッド名 = prevFrame.GetMethod().Name;

				return $"{クラス名}.{メソッド名}()";
			}
		}

		/// <summary>
		/// 画像ファイルからシェーダリソースビューを作成して返す。
		/// </summary>
		/// <remarks>
		/// （参考: http://qiita.com/oguna/items/c516e09ee57d931892b6 ）
		/// </remarks>
		public static SharpDX.Direct3D11.ShaderResourceView CreateShaderResourceViewFromFile(
			SharpDX.Direct3D11.Device d3dDevice, string 画像ファイルパス, out SharpDX.Size2F ビューのサイズ, out SharpDX.Direct3D11.Texture2D テクスチャ )
		{
			ビューのサイズ = new SharpDX.Size2F( 0, 0 );
			var srv = (SharpDX.Direct3D11.ShaderResourceView) null;
			using( var image = new System.Drawing.Bitmap( 画像ファイルパス ) )
			{
				var 画像の矩形 = new System.Drawing.Rectangle( 0, 0, image.Width, image.Height );
				using( var bitmap = image.Clone( 画像の矩形, System.Drawing.Imaging.PixelFormat.Format32bppArgb ) )
				{
					var ロック領域 = bitmap.LockBits( 画像の矩形, System.Drawing.Imaging.ImageLockMode.ReadOnly, bitmap.PixelFormat );
					var dataBox = new[] { new SharpDX.DataBox( ロック領域.Scan0, bitmap.Width * 4, bitmap.Height ) };
					var textureDesc = new SharpDX.Direct3D11.Texture2DDescription() {
						ArraySize = 1,
						BindFlags = SharpDX.Direct3D11.BindFlags.ShaderResource,
						CpuAccessFlags = SharpDX.Direct3D11.CpuAccessFlags.None,
						Format = SharpDX.DXGI.Format.B8G8R8A8_UNorm,
						Height = bitmap.Height,
						Width = bitmap.Width,
						MipLevels = 1,
						OptionFlags = SharpDX.Direct3D11.ResourceOptionFlags.None,
						SampleDescription = new SharpDX.DXGI.SampleDescription( 1, 0 ),
						Usage = SharpDX.Direct3D11.ResourceUsage.Default
					};
					var texture = new SharpDX.Direct3D11.Texture2D( d3dDevice, textureDesc, dataBox );
					bitmap.UnlockBits( ロック領域 );
					srv = new SharpDX.Direct3D11.ShaderResourceView( d3dDevice, texture );

					// 戻り値２：テクスチャ
					テクスチャ = texture;
				}
				// 戻り値１：ビューのサイズ
				ビューのサイズ = new SharpDX.Size2F( 画像の矩形.Width, 画像の矩形.Height );
			}
			// 戻り値３：シェーダリソースビュー
			return srv;
		}

		/// <summary>
		/// 空のテクスチャとそのシェーダーリソースビューを作成し、返す。
		/// </summary>
		public static SharpDX.Direct3D11.ShaderResourceView CreateShaderResourceView(
			SharpDX.Direct3D11.Device d3dDevice, SharpDX.Size2 サイズdpx, out SharpDX.Direct3D11.Texture2D テクスチャ )
		{
			var textureDesc = new SharpDX.Direct3D11.Texture2DDescription() {
				ArraySize = 1,
				BindFlags = SharpDX.Direct3D11.BindFlags.ShaderResource,
				CpuAccessFlags = SharpDX.Direct3D11.CpuAccessFlags.None,
				Format = SharpDX.DXGI.Format.B8G8R8A8_UNorm,
				Height = サイズdpx.Height,
				Width = サイズdpx.Width,
				MipLevels = 1,
				OptionFlags = SharpDX.Direct3D11.ResourceOptionFlags.None,
				SampleDescription = new SharpDX.DXGI.SampleDescription( 1, 0 ),
				Usage = SharpDX.Direct3D11.ResourceUsage.Default
			};
			var texture = new SharpDX.Direct3D11.Texture2D( d3dDevice, textureDesc );

			// 戻り値１：テクスチャ
			テクスチャ = texture;

			// 戻り値２：シェーダリソースビュー
			return new SharpDX.Direct3D11.ShaderResourceView( d3dDevice, texture );
		}
	}
}
