﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Threading.Tasks;
using FDK.同期;

namespace FDK.メディア
{
	/// <remarks>
	/// 活性化後の最初の進行描画時に、再生が開始される。
	/// </remarks>
	public unsafe class 動画 : Activity
	{
		public enum 再生状態ID { 未再生, 再生中, 再生終了済み }
		public FDK.同期.RWLock<再生状態ID> 再生状態 { get; } = new RWLock<再生状態ID>( 再生状態ID.未再生 );
		public bool 加算合成 { get; set; } = false;
		public float 不透明度0to1 { get; set; } = 1.0f;
		public bool ループ再生する { get; set; } = false;

		public 動画( string 動画ファイルパス )
		{
			this.動画ファイルパス = FDK.フォルダ.絶対パスに含まれるフォルダ変数を展開して返す( 動画ファイルパス );
		}
		protected override void Onデバイス依存リソースの作成( デバイスリソース dr )
		{
			// todo: デバイスがリサイズされると動画もリセットされてしまう。活性化とコードを分離すること（課題）。

			this.再生状態.Value = 再生状態ID.未再生;
			this.前フレームの時刻 = FDK.カウンタ.QPCTimer.未使用;
			this.ループした際の先頭時刻 = 0;

			string 変数付きファイルパス = フォルダ.絶対パスに含まれるフォルダ変数を展開して返す( this.動画ファイルパス );
			var result = SharpDX.Result.Ok;
			try
			{
				this.動画の生成に成功した = false;

				// 動画ファイルパスを基に、SourceReaderEx, MediaType, フレームサイズ, 描画先WICビットマップを準備する。

				#region " 動画ファイルパスの有効性を確認する。"
				//-----------------
				if( string.IsNullOrEmpty( this.動画ファイルパス ) )
				{
					Log.ERROR( $"動画ファイルパスが null または空文字列です。[{変数付きファイルパス}]" );
					return;
				}
				if( false == System.IO.File.Exists( this.動画ファイルパス ) )
				{
					Log.ERROR( $"動画ファイルが存在しません。[{変数付きファイルパス}]" );
					return;
				}
				//-----------------
				#endregion
				#region " 動画のファイルパス（URI扱い）から SourceReaderEx を作成する。"
				//-----------------
				try
				{
					using( var 属性 = new SharpDX.MediaFoundation.MediaAttributes() )
					{
						// DXVAに対応しているGPUの場合にはそれをデコードに利用するよう指定する。
						属性.Set<SharpDX.ComObject>( SharpDX.MediaFoundation.SourceReaderAttributeKeys.D3DManager, dr.DXGIDeviceManager );

						// 追加のビデオプロセッシングを有効にする。
						属性.Set<bool>( SharpDX.MediaFoundation.SourceReaderAttributeKeys.EnableAdvancedVideoProcessing, true );  // bool だったり

						// 追加のビデオプロセッシングを有効にしたら、こちらは無効に。
						属性.Set<int>( SharpDX.MediaFoundation.SinkWriterAttributeKeys.ReadwriteDisableConverters, 0 );           // int だったり

						// 属性を使って、SourceReaderEx を生成。
						using( var sourceReader = new SharpDX.MediaFoundation.SourceReader( this.動画ファイルパス, 属性 ) )
						{
							this.SourceReaderEx = sourceReader.QueryInterface<SharpDX.MediaFoundation.SourceReaderEx>();
						}
					}
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"SourceReaderEx の作成に失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					result = e.ResultCode;
					return;
				}
				//-----------------
				#endregion
				#region " 最初のビデオストリームを選択し、その他のすべてのストリームを非選択にする。"
				//-----------------
				try
				{
					this.SourceReaderEx.SetStreamSelection( SharpDX.MediaFoundation.SourceReaderIndex.AllStreams, false );
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"すべてのストリームの選択解除に失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					return;
				}
				try
				{
					this.SourceReaderEx.SetStreamSelection( SharpDX.MediaFoundation.SourceReaderIndex.FirstVideoStream, true );
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"最初のビデオストリームの選択に失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					result = e.ResultCode;
					return;
				}
				//-----------------
				#endregion
				#region " ARGB32 フォーマットに合わせたメディアタイプを作成し、SourceReaderEx に登録してデコーダを準備する。"
				//-----------------
				try
				{
					using( var mediaType = new SharpDX.MediaFoundation.MediaType() )
					{
						// フォーマットを部分メディアタイプの属性に設定。
						mediaType.Set<Guid>( SharpDX.MediaFoundation.MediaTypeAttributeKeys.MajorType, SharpDX.MediaFoundation.MediaTypeGuids.Video );
						mediaType.Set<Guid>( SharpDX.MediaFoundation.MediaTypeAttributeKeys.Subtype, SharpDX.MediaFoundation.VideoFormatGuids.Argb32 ); // ARGB32 フォーマットで固定。

						// 部分メディアタイプを SourceReaderEx にセットする。SourceReaderEx は、必要なデコーダをロードするだろう。
						this.SourceReaderEx.SetCurrentMediaType( SharpDX.MediaFoundation.SourceReaderIndex.FirstVideoStream, mediaType );
					}
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"MediaType (Video, ARGB32) の設定または必要なデコーダの読み込みに失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					result = e.ResultCode;
					return;
				}
				//-----------------
				#endregion
				#region " ビデオストリームが選択されていることを再度保証する。"
				//-----------------
				try
				{
					this.SourceReaderEx.SetStreamSelection( SharpDX.MediaFoundation.SourceReaderIndex.FirstVideoStream, true );
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"最初のビデオストリームの選択に失敗しました（MediaType 設定後）。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					result = e.ResultCode;
					return;
				}
				//-----------------
				#endregion
				#region " デコーダの読み込みにより完成した完全メディアタイプを取得する。"
				//-----------------
				try
				{
					this.MediaType = this.SourceReaderEx.GetCurrentMediaType( SharpDX.MediaFoundation.SourceReaderIndex.FirstVideoStream );
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"完全メディアタイプの取得に失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					result = e.ResultCode;
					return;
				}
				//-----------------
				#endregion
				#region " フレームサイズを取得する。動画の途中でのサイズ変更には対応しない。"
				//-----------------
				try
				{
					var packedFrameSize = this.MediaType.Get<long>( SharpDX.MediaFoundation.MediaTypeAttributeKeys.FrameSize );
					this.サイズdpx.Width = (int) ( ( packedFrameSize >> 32 ) & 0xFFFFFFFF );
					this.サイズdpx.Height = (int) ( ( packedFrameSize ) & 0xFFFFFFFF );
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"フレームサイズの取得に失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					result = e.ResultCode;
					return;
				}
				//-----------------
				#endregion
				#region " 描画先WICビットマップを作成する。"
				//-----------------
				try
				{
					this.WICビットマップ = new SharpDX.WIC.Bitmap(
						dr.WicImagingFactory2,
						this.サイズdpx.Width,
						this.サイズdpx.Height,
						SharpDX.WIC.PixelFormat.Format32bppBGR,
						SharpDX.WIC.BitmapCreateCacheOption.CacheOnLoad );
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"描画先 WIC ビットマップの作成に失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					result = e.ResultCode;
					return;
				}
				//-----------------
				#endregion

				this.動画の生成に成功した = true;
				Log.Info( $"{Utilities.現在のメソッド名}: 動画を生成しました。[{変数付きファイルパス}]" );

				#region " デコードの前処理を行う。"
				//----------------
				this.デコードキャッシング();
				Log.Info( $"{Utilities.現在のメソッド名}: 動画のセットアップを行いました。[{変数付きファイルパス}]" );
				//----------------
				#endregion

				// 準備完了。以降は、サンプルを読み込みたいときに、適宜 ReadSample() する。
				// コールバックを使っていないので、IMFSourceReader::ReadSample() はサンプルが用意できるまでブロックすることに注意。
			}
			finally
			{
				if( result.Failure )
					this.全リソースを解放する();

				// 失敗しても、一応活性化は成功とする。（進行描画はされない。）
			}
		}
		protected override void Onデバイス依存リソースの解放( デバイスリソース dr )
		{
			// todo: デバイスがリサイズされると動画もリセットされてしまう。非活性化とコードを分離すること（課題）。

			// デコード中なら、タスクが終了するまで待つ。
			while( ( this.再生状態.Value == 再生状態ID.再生中 ) && ( this.デコーダ状態.Value == デコーダ状態ID.デコード中 ) )
				System.Threading.Thread.Sleep( 500 );

			this.全リソースを解放する();
			Log.Info( $"{Utilities.現在のメソッド名}: 動画を解放しました。[{フォルダ.絶対パスに含まれるフォルダ変数を展開して返す( this.動画ファイルパス )}]" );
		}
		public void 進行描画する( デバイスリソース dr, SharpDX.RectangleF 描画先矩形dpx, float 不透明度0to1 = 1.0f, bool ループ再生する = false )
		{
			// Direct2D の行列は、設計単位じゃなく物理単位で計算するので注意。
			var 変換行列2Dpx = SharpDX.Matrix3x2.Identity
				* SharpDX.Matrix3x2.Scaling( 描画先矩形dpx.Width / this.サイズdpx.Width, 描画先矩形dpx.Height / this.サイズdpx.Height )  // スケーリング。
				* dr.行列を単位変換するDPXtoPX( SharpDX.Matrix3x2.Translation( 描画先矩形dpx.Left, 描画先矩形dpx.Top ) );  // 平行移動（物理単位）、

			// 描画する。
			this.進行描画する( dr, 変換行列2Dpx );
		}
		public void 進行描画する( デバイスリソース dr, SharpDX.Matrix3x2 変換行列 )
		{
			Debug.Assert( this.活性化している );

			// 活性化でエラーが発生している場合は、何もしない（何も描画しない）。
			if( false == this.動画の生成に成功した )
				return;

			// 以下、再生状態とデコーダ状態に応じて処理分岐。
			switch( this.再生状態.Value )
			{
				case 再生状態ID.未再生:
					// (A) 未再生なら、デコードタスクを開始する。
					this.再生状態.Value = 再生状態ID.再生中;
					this.デコードタスクを開始する();
					break;

				case 再生状態ID.再生終了済み:
					// (B) 再生が終了している場合は、何もしない（何も描画しない）。
					break;

				case 再生状態ID.再生中:
					// (C) 再生中なら、D2Dビットマップを描画する。
					switch( this.デコーダ状態.Value )
					{
						case デコーダ状態ID.デコード中:
							// (C-a) タスクが次のサンプルフレームをデコード中なら、現在のD2Dビットマップ（直前のフレーム画像が入ってる）をもう一度描画する。
							this.D2Dビットマップを描画する( dr, 変換行列 );
							break;

						case デコーダ状態ID.完了:
							// (C-b) タスクの次のサンプルフレームのデコード処理が完了しているなら、サンプルをD2Dビットマップに転送し、それを描画し、デコードタスクを再び起動する。
							this.サンプルをD2Dビットマップに転送する( dr );
							this.D2Dビットマップを描画する( dr, 変換行列 );
							this.デコードタスクを開始する();
							break;

						case デコーダ状態ID.スキップ:
							// (C-c) タスクの次のサンプルフレームをスキップしたなら、現在のD2Dビットマップ（直前のフレーム画像が入ってる）をもう一度描画し、デコードタスクを再び起動する。
							this.D2Dビットマップを描画する( dr, 変換行列 );
							this.デコードタスクを開始する();
							break;
					}
					break;
			}
		}

		private enum デコーダ状態ID { 完了, デコード中, スキップ }
		private readonly FDK.同期.RWLock<デコーダ状態ID> デコーダ状態 = new RWLock<デコーダ状態ID>( デコーダ状態ID.完了 );
		private string 動画ファイルパス = "";
		private bool 動画の生成に成功した = false;    // 活性化時に成功の成否を設定。
		private SharpDX.Result LastSharpDXResult = SharpDX.Result.Ok;
		private SharpDX.MediaFoundation.Sample Sample = null;
		private SharpDX.MediaFoundation.SourceReaderEx SourceReaderEx = null;
		private SharpDX.WIC.Bitmap WICビットマップ = null;    // MediaFoundation は WICBitmap に出力する。
		private SharpDX.Direct2D1.Bitmap D2Dビットマップ = null;  // WICBitmap をこれに転送して D2D に Drawする。
		private SharpDX.MediaFoundation.MediaType MediaType = null;
		private SharpDX.Size2 サイズdpx = new SharpDX.Size2( 1, 1 );  // CopyMemory()で使うので、Size2F じゃなく Size2 を使う。
		private FDK.カウンタ.QPCTimer タイマ = new FDK.カウンタ.QPCTimer();
		private long ループした際の先頭時刻 = 0;
		private long 前フレームの時刻 = FDK.カウンタ.QPCTimer.未使用;  // 未使用 = 先頭のフレームである

		private unsafe void サンプルをD2Dビットマップに転送する( デバイスリソース dr )
		{
			var buffer = (SharpDX.MediaFoundation.MediaBuffer) null;
			var buffer2d2 = (SharpDX.MediaFoundation.Buffer2D2) null;
			try
			{
				#region " サンプルからサンプルバッファ (MediaBuffer) を取得する。"
				//-----------------
				try
				{
					buffer = this.Sample.ConvertToContiguousBuffer();
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"サンプルバッファの取得に失敗しました。(0x{e.HResult:x8})" );
					this.LastSharpDXResult = e.ResultCode;
					return;
				}
				//-----------------
				#endregion
				#region " サンプルバッファを Buffer2D2 にキャストする。"
				//-----------------
				try
				{
					buffer2d2 = buffer.QueryInterface<SharpDX.MediaFoundation.Buffer2D2>();
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"サンプルバッファから Buffer2D2 へのキャストに失敗しました。(0x{e.HResult:x8})" );
					this.LastSharpDXResult = e.ResultCode;
					return;
				}
				//-----------------
				#endregion
				#region " サンプルバッファをロックする。"
				//-----------------
				byte[] scanLine0_bp = new byte[ 8 ];   // 生ポインタが格納される。32bitなら[0～3]、64bitなら[0～7]が有効。（CPUではなく.NETに依存）
				int pitch = 0;
				byte[] bufferStart_bp = new byte[ 8 ];  // 生ポインタが格納される。詳細は同上だが、ここでは受け取るだけで何もしない。
				int bufferLength;
				try
				{
					buffer2d2.Lock2DSize(
						SharpDX.MediaFoundation.Buffer2DLockFlags.Read,
						scanLine0_bp,
						out pitch,
						bufferStart_bp,
						out bufferLength );
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"サンプルバッファのロックに失敗しました。(0x{e.HResult:x8})" );
					this.LastSharpDXResult = e.ResultCode;
					return;
				}
				//-----------------
				#endregion
				try
				{
					#region " サンプルバッファのネイティブ先頭アドレスを取得する。"
					//-----------------
					byte* scanLine0 = null;
					try
					{
						scanLine0 = (byte*) this.生ポインタを格納したbyte配列からポインタを取得して返す( scanLine0_bp );
					}
					catch( SharpDX.SharpDXException e )
					{
						Log.ERROR( $"サンプルバッファのアドレスの取得に失敗しました。(0x{e.HResult:x8})" );
						this.LastSharpDXResult = e.ResultCode;
						return;
					}
					//-----------------
					#endregion
					#region " サンプルバッファから描画先WICビットマップへ、ARGB32 を G8B8R8X8 に変換しながらコピーする。"
					//-----------------
					if( null == this.D2Dビットマップ )
					{
						// (A) 初回は WICBitmap 経由で D2Dビットマップを新規生成する。
						#region " サンプルバッファから描画先WICビットマップへ、ARGB32 を G8B8R8X8 に変換しながらコピーする。"
						//-----------------
						try
						{
							// 描画先WICビットマップをロックする。
							using( var bitmapLock = this.WICビットマップ.Lock(
								new SharpDX.Rectangle( 0, 0, this.WICビットマップ.Size.Width, this.WICビットマップ.Size.Height ),
								SharpDX.WIC.BitmapLockFlags.Write ) )
							{
								// サンプルバッファからWICビットマップへ、ARGB32 を G8B8R8X8 に変換しながらコピー。
								int bitmapStride = bitmapLock.Stride;
								byte* src = scanLine0;
								byte* dest = (byte*) bitmapLock.Data.DataPointer.ToPointer();

								for( int y = 0; y < this.サイズdpx.Height; y++ )
								{
									// ARGB32 to G8B8R8X8 ではデータ変換が不要なので、一行を一気にコピー。
									動画.CopyMemory( dest, src, this.サイズdpx.Width * 4 );   // ARGB=4バイト。
									src += pitch;
									dest += bitmapStride;   // bitmapStride は byte 単位
								}
							}
						}
						catch( SharpDX.SharpDXException e )
						{
							Log.ERROR( $"WICビットマップの Lock に失敗しました。(0x{e.HResult:x8})" );
							this.LastSharpDXResult = e.ResultCode;
							return;
						}
						catch( Exception e )
						{
							Log.ERROR( $"サンプルバッファから WIC ビットマップへのデータの転送に失敗しました。(0x{e.HResult:x8})" );
							this.LastSharpDXResult = SharpDX.Result.GetResultFromException( e );
							return;
						}
						//-----------------
						#endregion
						#region " 描画先WICビットマップからD2Dビットマップを生成する。 "
						//-----------------
						try
						{
							this.D2Dビットマップ?.Dispose();
							this.D2Dビットマップ = SharpDX.Direct2D1.Bitmap.FromWicBitmap(
								dr.D2DContext1,
								this.WICビットマップ );
						}
						catch( SharpDX.SharpDXException e )
						{
							Log.ERROR( $"D2Dビットマップの作成に失敗しました。(0x{e.HResult:x8})" );
							this.LastSharpDXResult = e.ResultCode;
							return;
						}
						//-----------------
						#endregion
					}
					else
					{
						// (B) ２回目以降は直接 D2Dビットマップへコピーする。
						this.D2Dビットマップ.CopyFromMemory( new IntPtr( scanLine0 ), pitch );
					}
					//-----------------
					#endregion
				}
				finally
				{
					#region " サンプルバッファのロックを解除する。"
					//-----------------
					buffer2d2.Unlock2D();
					//-----------------
					#endregion
				}
			}
			finally
			{
				FDK.Utilities.解放する( ref buffer2d2 );
				FDK.Utilities.解放する( ref buffer );
				if( this.LastSharpDXResult.Failure )
				{
					// 描画に失敗したら、全リソースを解放して再生終了。
					this.再生状態.Value = 再生状態ID.再生終了済み;
					this.全リソースを解放する();
				}
			}
		}
		private void D2Dビットマップを描画する( デバイスリソース dr, SharpDX.Matrix3x2 変換行列2Dpx )
		{
			if( null == this.D2Dビットマップ )
				return;

			Utilities.D2DBatchDraw( dr.D2DContext1, () => {

				// 変換行列とブレンドモードを設定する。
				dr.D2DContext1.Transform = 変換行列2Dpx;
				dr.D2DContext1.PrimitiveBlend =
					( this.加算合成 ) ? SharpDX.Direct2D1.PrimitiveBlend.Add : SharpDX.Direct2D1.PrimitiveBlend.SourceOver;

				// D2Dビットマップを描画する。
				dr.D2DContext1.DrawBitmap( this.D2Dビットマップ, this.不透明度0to1, SharpDX.Direct2D1.InterpolationMode.Linear );
			} );
		}
		private void デコードタスクを開始する()
		{
			// 再生中なら、デコードタスクをタスクとして起動する（タスクの完了を待たず、このメソッドはすぐ抜ける）。
			if( this.再生状態.Value == 再生状態ID.再生中 )
			{
				Task.Run( () => {
					this.デコードタスク( this.ループ再生する );
				} );
			}
		}
		private void デコードタスク( bool ループ再生する )
		{
			//  サンプル（フレーム）を１つ取得するタスク。

			this.デコーダ状態.Value = デコーダ状態ID.デコード中;
			this.LastSharpDXResult = SharpDX.Result.Ok;
			try
			{
				var streamFlags = SharpDX.MediaFoundation.SourceReaderFlags.None;
				long サンプルの時刻 = 0;

				#region " SourceReader から次のサンプル（フレーム）を１つ取得する。"
				//-----------------
				{
					int actualStreamIndex = 0;
					try
					{
						this.Sample?.Dispose();
						this.Sample = this.SourceReaderEx.ReadSample(
							SharpDX.MediaFoundation.SourceReaderIndex.FirstVideoStream,
							SharpDX.MediaFoundation.SourceReaderControlFlags.None,
							out actualStreamIndex,
							out streamFlags,
							out サンプルの時刻 );
					}
					catch( SharpDX.SharpDXException e )
					{
						Log.ERROR( $"SourceReaderEx.ReadSample() に失敗しました。(0x{e.HResult:x8})" );
						this.LastSharpDXResult = e.ResultCode;
						return;
					}
				}
				//-----------------
				#endregion
				#region " 取得結果フラグに応じて、必要な処理があれば行なう。"
				//---------------------------------------------------
				if( streamFlags.HasFlag( SharpDX.MediaFoundation.SourceReaderFlags.Endofstream ) )  // BOX化コストとか気にしない
				{
					#region " ストリーム終了 "
					//----------------
					if( ループ再生する )   // メンバじゃなくパラメータのほう。
					{
						// (A) ループ再生する場合
						FDK.Log.Info( "動画をループ再生します。" );
						this.ループした際の先頭時刻 = this.前フレームの時刻;   // このフラグがセットされるときは、ReadSample() で返されるサンプル時刻は 0 になってるので使えない。
						this.SourceReaderEx.SetCurrentPosition( 0 );  // ストリーム再生位置を先頭へ。
						this.デコードタスク( ループ再生する ); // 再帰で先頭サンプルを取得して終了。
						return;
					}
					else
					{
						// (B) ループ再生しない場合
						FDK.Log.Info( "動画の再生を終了します。" );
						this.再生状態.Value = 再生状態ID.再生終了済み;
						this.LastSharpDXResult = SharpDX.Result.Ok; // SharpDX のエラーによる終了じゃないよと。
						return;
					}
					//----------------
					#endregion
				}
				else if( streamFlags.HasFlag( SharpDX.MediaFoundation.SourceReaderFlags.Newstream ) )
				{
					// 未対応。
				}
				else if( streamFlags.HasFlag( SharpDX.MediaFoundation.SourceReaderFlags.Nativemediatypechanged ) )
				{
					// 未対応。
				}
				else if( streamFlags.HasFlag( SharpDX.MediaFoundation.SourceReaderFlags.Currentmediatypechanged ) )
				{
					// 未対応。
				}
				else if( streamFlags.HasFlag( SharpDX.MediaFoundation.SourceReaderFlags.StreamTick ) )
				{
					// 未対応。
				}
				else if( streamFlags.HasFlag( SharpDX.MediaFoundation.SourceReaderFlags.AllEffectsremoved ) )
				{
					// 未対応。
				}
				else if( streamFlags.HasFlag( SharpDX.MediaFoundation.SourceReaderFlags.Error ) )
				{
					#region " エラー "
					//----------------
					FDK.Log.Info( "エラーが発生したので、動画の再生を終了します。" );
					this.再生状態.Value = 再生状態ID.再生終了済み;
					this.LastSharpDXResult = SharpDX.Result.Fail;
					return;
					//----------------
					#endregion
				}
				//---------------------------------------------------
				#endregion
				#region " 前フレームの時刻を、先頭時刻＋サンプルの時刻 に設定する。"
				//-----------------
				if( FDK.カウンタ.QPCTimer.未使用 == 前フレームの時刻 )    // 最初のフレームである場合（ループ時は除く）
				{
					// 最初のフレームの取得には時間がかかることが多いので、最初のフレームを取得した場合のみ、タイマをサンプルの時刻にリセットする。
					this.タイマ.リセットする( サンプルの時刻 );
				}

				サンプルの時刻 += this.ループした際の先頭時刻;
				this.前フレームの時刻 = サンプルの時刻;
				//-----------------
				#endregion
				#region " サンプルの表示時刻までの時刻調整を行う。"
				//-----------------
				var 現在時刻 = this.タイマ.現在のリアルタイムカウント100ns単位;
				if( 現在時刻 > サンプルの時刻 )
				{
					// (A) サンプルの時刻が過去である → このサンプルとタスクは捨てる（さっさと次のサンプルへ移る）。
					this.デコーダ状態.Value = デコーダ状態ID.スキップ;
					return;
				}
				else
				{
					// (B) サンプルの時刻が未来である → その時刻を過ぎるまで Sleep する。
					while( 現在時刻 < サンプルの時刻 )
					{
						System.Threading.Thread.Sleep( 2 );      // 500fps の動画まで対応（理論上（汗
						現在時刻 = タイマ.現在のリアルタイムカウント100ns単位;
					}
				}
				//-----------------
				#endregion
			}
			finally
			{
				#region " 失敗した、または EndOfStream に達した場合は、Sample を無効化する。"
				//----------------
				if( this.LastSharpDXResult.Failure )
				{
					FDK.Log.Info( "SharpDX でエラーが発生したので、動画の再生を終了します。" );
					this.再生状態.Value = 再生状態ID.再生終了済み;
					FDK.Utilities.解放する( ref this.Sample );
				}
				//----------------
				#endregion

				this.デコーダ状態.Value = デコーダ状態ID.完了;
			}
		}
		private void デコードキャッシング()
		{
			// 最初のほうのサンプルのデコードには 100～200 ms ほどかかってしまうので、あらかじめ少しデコードしてメモリにキャッシュさせることでこれを緩和する。

			this.デコーダ状態.Value = デコーダ状態ID.デコード中;
			try
			{
				// 先頭から複数のフレームを読み込む。
				for( int i = 0; i < 60; i++ )   // 60フレームもあれば、2つめのキーフレームくらいには届くだろう……
				{
					#region " SourceReader から次のサンプル（フレーム）を１つ取得しては解放する。"
					//-----------------
					var sample = (SharpDX.MediaFoundation.Sample) null;
					int actualStreamIndex = 0;
					var streamFlags = SharpDX.MediaFoundation.SourceReaderFlags.None;
					long サンプルの時刻 = 0;
					try
					{
						sample = this.SourceReaderEx.ReadSample(
							SharpDX.MediaFoundation.SourceReaderIndex.FirstVideoStream,
							SharpDX.MediaFoundation.SourceReaderControlFlags.None,
							out actualStreamIndex,
							out streamFlags,
							out サンプルの時刻 );
					}
					catch( SharpDX.SharpDXException e )
					{
						Log.ERROR( $"SourceReaderEx.ReadSample() に失敗しました。(0x{e.HResult:x8})" );
						this.LastSharpDXResult = e.ResultCode;
						return;
					}
					finally
					{
						// すぐ解放。
						sample?.Dispose();
					}
					//-----------------
					#endregion
				}
				// SourceReader を先頭へリセット。
				this.SourceReaderEx.SetCurrentPosition( 0 );
			}
			finally
			{
				this.デコーダ状態.Value = デコーダ状態ID.完了;
			}
		}
		private void 全リソースを解放する()
		{
			FDK.Utilities.解放する( ref this.D2Dビットマップ );
			FDK.Utilities.解放する( ref this.WICビットマップ );
			FDK.Utilities.解放する( ref this.Sample );
			FDK.Utilities.解放する( ref this.MediaType );
			FDK.Utilities.解放する( ref this.SourceReaderEx );
		}
		private unsafe void* 生ポインタを格納したbyte配列からポインタを取得して返す( byte[] 生ポインタ )
		{
			if( ( 4 == IntPtr.Size ) && System.BitConverter.IsLittleEndian )
			{
				// (A) 32bit, リトルエンディアン
				int 生アドレス32bit = 0;
				for( int i = 0; i < 4; i++ )
					生アドレス32bit += ( (int) 生ポインタ[ i ] ) << ( i * 8 );
				return new IntPtr( 生アドレス32bit ).ToPointer();
			}
			else if( ( 8 == IntPtr.Size ) && System.BitConverter.IsLittleEndian )
			{
				// (B) 64bit, リトルエンディアン
				long 生アドレス64bit = 0;
				for( int i = 0; i < 8; i++ )
					生アドレス64bit += ( (int) 生ポインタ[ i ] ) << ( i * 8 );
				return new IntPtr( 生アドレス64bit ).ToPointer();
			}
			else if( ( 4 == IntPtr.Size ) && ( false == System.BitConverter.IsLittleEndian ) )
			{
				// (C) 32bit, ビッグエンディアン
				int 生アドレス32bit = 0;
				for( int i = 0; i < 4; i++ )
					生アドレス32bit += ( (int) 生ポインタ[ 4 - i ] ) << ( i * 8 );
				return new IntPtr( 生アドレス32bit ).ToPointer();
			}
			else if( ( 8 == IntPtr.Size ) && ( false == System.BitConverter.IsLittleEndian ) )
			{
				// (D) 64bit, ビッグエンディアン
				long 生アドレス64bit = 0;
				for( int i = 0; i < 8; i++ )
					生アドレス64bit += ( (int) 生ポインタ[ 8 - i ] ) << ( i * 8 );
				return new IntPtr( 生アドレス64bit ).ToPointer();
			}

			throw new SharpDX.SharpDXException( SharpDX.Result.NotImplemented, "この .NET アーキテクチャには対応していません。" );
		}

		#region " Win32 API "
		//-----------------
		[System.Runtime.InteropServices.DllImport( "kernel32.dll", SetLastError = true )]
		private static extern unsafe void CopyMemory( void* dst, void* src, int size );
		//-----------------
		#endregion
	}
}
