﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using FDK;

namespace SST.IO
{
	/// <summary>
	///		SSTで使用するフォルダの定義。
	/// </summary>
	class Folder : FDK.Folder
	{
		/// <summary>
		///		アプリ（exe）のあるフォルダ。
		/// </summary>
		public static string Exe
		{
			get
				=> Path.GetDirectoryName( Assembly.GetExecutingAssembly().Location ) + @"\";
		}

		/// <summary>
		///		静的な（アプリを通じて不変の）リソースファイルが置かれる場所のルートフォルダ。
		/// </summary>
		public static string System
		{
			get
				=> Path.Combine( Folder.Exe, @"System\" );
		}

		/// <summary>
		///		共通データが保存される場所のルートフォルダ。
		/// </summary>
		public static string AppData
		{
			get
				=> Path.Combine( Environment.GetFolderPath( Environment.SpecialFolder.ApplicationData, Environment.SpecialFolderOption.Create ), @"StrokeStyleT\" );
		}

		/// <summary>
		///		ユーザの情報（ユーザフォルダなど）が格納されるルートフォルダ。
		/// </summary>
		public static string UserRoot
		{
			get
				=> Path.Combine( Folder.AppData, @"Users\" );
		}


		/// <summary>
		///		フォルダ変数を初期化する。
		/// </summary>
		public static void 初期化する()
		{
			Folder.フォルダ変数を追加または更新する( "Exe", Folder.Exe );
			Folder.フォルダ変数を追加または更新する( "System", Folder.System );
			Folder.フォルダ変数を追加または更新する( "AppData", Folder.AppData );
			Folder.フォルダ変数を追加または更新する( "UserRoot", Folder.UserRoot );

			if( false == Directory.Exists( Folder.AppData ) )
				Directory.CreateDirectory( Folder.AppData );
		}

		/// <summary>
		///		ユーザフォルダ（SSTユーザごとに用意されたデータの保存フォルダ）を返す。
		/// </summary>
		/// <remarks>
		///		SSTユーザ名に null または空文字列を指定すると、ユーザルートフォルダ（<see cref="UserRoot"/>）が返される。
		///		フォルダが存在するか否かは判断しないので、必要があれば随時作成すること。
		/// </remarks>
		public static string User( string SSTユーザ名 )
		{
			if( SSTユーザ名.Nullまたは空である() )
			{
				return Folder.UserRoot;
			}
			else
			{
				var 調整後のSSTユーザ名 = ファイルやパスで使用可能な文字列に調整して返す( SSTユーザ名 );
				return Path.Combine( Folder.UserRoot, $@"{調整後のSSTユーザ名}\" );
			}
		}

		/// <remarks>
		///		ユーザ名をそのままユーザフォルダとして転用する仕様だが、
		///		ユーザ名にファイルやパスとして使用できない文字が含まれている場合を想定し、それらをすべてエスケープする。
		/// </remarks>
		public static string ファイルやパスで使用可能な文字列に調整して返す( string 文字列 )
		{
			char[] ファイル名に使えない文字 = Path.GetInvalidFileNameChars();
			文字列 = string.Concat( 文字列.Select( c => ファイル名に使えない文字.Contains( c ) ? '_' : c ) );

			char[] パスに使えない文字 = Path.GetInvalidPathChars();
			文字列 = string.Concat( 文字列.Select( c => パスに使えない文字.Contains( c ) ? '_' : c ) );

			return 文字列;
		}
	}
}
