﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.IO;
using System.Windows.Forms;
using System.Security;

namespace StrokeStyleT
{
	static class Folder
	{
		/// <summary>
		/// <para>システムフォルダ。StrokeStyleT.exe が格納されているフォルダを意味する。読み込み専用。</para>
		/// <para>（例："c:\Program Files\StrokeStyleT" ）</para>
		/// </summary>
		public static string stgシステムフォルダ
		{
			get;
			private set;
		}

		/// <summary>
		/// <para>全ユーザで共通のデータを格納するフォルダ。読み書き両用。</para>
		/// <para>（例："C:\ProgramData\StrokeStyleT"）</para>
		/// </summary>
		public static string stgユーザ共通フォルダ
		{
			get;
			private set;
		}

		/// <summary>
		/// <para>ユーザフォルダ。読み書き両用。ログイン時に設定され、ログアウト時に null に設定される。</para>
		/// <para>ユーザ共通フォルダ ＋ SSTユーザ名。</para>
		/// <para>（例："C:\PrograData\StrokeStyleT\SSTFViewer" ）</para>
		/// </summary>
		public static string stgユーザ個別フォルダ
		{
			get;
			set;	// CUsers から更新される。
		}

		/// <summary>
		/// <para>ユーザのテーマフォルダ。ログインしている場合は Config.Theme を、していない場合は default を使用する。</para>
		/// </summary>
		public static string stgテーマフォルダ
		{
			get
			{
				string stgテーマ名 = ( Global.User.Config != null ) ? Global.User.Config.ThemeName : @"default";	
				return Path.Combine( Folder.stgシステムフォルダ, @"Theme\", stgテーマ名 );
			}
		}

		/// <summary>
		/// <para>ログファイルを出力するフォルダ。全ユーザ共通。</para>
		/// <para>（例："C:\Users\Kenji\Documents）</para>
		/// </summary>
		public static string stgログ出力フォルダ
		{
			get;
			private set;
		}


		static Folder()
		{
			// システムフォルダ;
			//	 (a) Debug 時 ……… システムフォルダはカレントフォルダ（VSで設定した作業フォルダ）にする。
			//	 (b) Release 時 …… システムフォルダは exe の存在するフォルダにする。（必ずしもカレントフォルダとは一致しないため。）
#if DEBUG
			Folder.stgシステムフォルダ = Environment.CurrentDirectory;
#else
			Folder.stgシステムフォルダ =  Path.GetDirectoryName( Application.ExecutablePath );
#endif

			// ユーザ共通フォルダ

			Folder.stgユーザ共通フォルダ = Path.Combine( Environment.GetFolderPath( Environment.SpecialFolder.CommonApplicationData ), @"StrokeStyleT" );
			Folder.tフォルダがなければ作成する( Folder.stgユーザ共通フォルダ );


			// ユーザ個別フォルダ

			Folder.stgユーザ個別フォルダ = null;	// ログインするまで未定。


			// ログ出力フォルダ

			Folder.stgログ出力フォルダ = Path.Combine( Environment.GetFolderPath( Environment.SpecialFolder.MyDocuments ), @"StrokeStyleT" );
			Folder.tフォルダがなければ作成する( Folder.stgログ出力フォルダ );
		}

		/// <summary>
		/// <para>テーマフォルダ内のファイルの絶対パスを返す。</para>
		/// </summary>
		public static string stgテーマファイル( string stgファイル名 )
		{
			string path;

			// (1) ログインしていれば、最初にユーザ指定のテーマ名でファイルの存在を確認する。存在すればそれを返す。

			if( Global.User.Config != null )
			{
				path = Path.Combine( Folder.stgテーマフォルダ, stgファイル名 );

				if( File.Exists( path ) )
					return path;
			}

			// (2) ログインしていない、またはユーザ指定のテーマ内にファイルが存在しなかった場合は default テーマで返す。

			return Path.Combine( Folder.stgシステムフォルダ, @"Theme\default", stgファイル名 );	// こちらは存在確認なし。
		}

		/// <summary>
		/// <para>指定されたパスを絶対パスに変換して返す。</para>
		/// <para>・path が相対パス指定であれば、str相対パス時のルートからの相対パスと見なす。</para>
		/// <para>・path が空文字列 or null であれば空文字列を返す。</para>
		/// <para>・path に問題があれば例外を発出する。</para>
		/// </summary>
		public static string tパスの正当性を確認のうえ絶対パスに変換して返す( string path, string str相対パス時のルート = null )
		{
			// path が 空文字列 or null の場合は空文字列を返す。

			if( string.IsNullOrEmpty( path ) )
				return "";

	
			// path が相対パス指定であれば、引数 str相対パス時のルートからの相対パスと見なす。

			try
			{
				if( !Path.IsPathRooted( path ) )
				{
					if( string.IsNullOrEmpty( str相対パス時のルート ) )
						throw new NullReferenceException( string.Format( "相対パスですがルートパスが指定されていません。[{0}]", Folder.tファイルパスをマクロ付きパスに変換する( path ) ) );
					
					path = Path.Combine( str相対パス時のルート, path );
				}
			}
			catch( ArgumentException e )
			{
				throw new ArgumentException( string.Format( "パス文字列に無効な文字が含まれています。[{0}]", Folder.tファイルパスをマクロ付きパスに変換する( path ) ), e );
			}


			// path 文字列がパスとして正しいかを個別確認する。

			try
			{
				path = Path.GetFullPath( path );	// path は既に絶対パスになっているが、Path.GetFullPath() を利用して、パス文字列の正当性確認を行う。
			}
			catch( ArgumentException e )
			{
				throw new ArgumentException( string.Format( "パス文字列に無効な文字が含まれています。[{0}]", path ), e );
			}
			catch( SecurityException e )
			{
				throw new SecurityException( string.Format( "パスにはアクセス権限がありません。[{0}]", path ), e );
			}
			catch( NotSupportedException e )
			{
				throw new NotSupportedException( string.Format( "パス文字列にサポートされていない文字が含まれています。[{0}]", path ), e );
			}
			catch( PathTooLongException e )
			{
				throw new PathTooLongException( string.Format( "パス文字列が長すぎます。[{0}]", path ), e );
			}

			return path;
		}


		// 以下は、パスの一部をマクロで隠蔽するために使う。絶対パスをそのままコンフィグファイルやログファイルに出すと色々まずいことがあるので。

		public static string tファイルパス内のマクロを展開する( string path )
		{
			Folder.tファイルパス内のマクロを展開する( ref path );
			return path;
		}
		public static void tファイルパス内のマクロを展開する( ref string path )
		{
			// システムフォルダ
			path = path.Replace( @"$(SystemFolder)", Folder.stgシステムフォルダ );
			
			// ユーザ共通フォルダ
			path = path.Replace( @"$(CommonUserFolder)", Folder.stgユーザ共通フォルダ );

			// ユーザ個別フォルダ
			if( !string.IsNullOrEmpty( Folder.stgユーザ個別フォルダ ) )
				path = path.Replace( @"$(UserFolder)", Folder.stgユーザ個別フォルダ );
		}

		public static string tファイルパスをマクロ付きパスに変換する( string path )
		{
			Folder.tファイルパスをマクロ付きパスに変換する( ref path );
			return path;
		}
		public static void tファイルパスをマクロ付きパスに変換する( ref string path )
		{
			// システムフォルダ
			path = path.Replace( Folder.stgシステムフォルダ, @"$(SystemFolder)" );

			// ユーザ共通フォルダ
			path = path.Replace( Folder.stgユーザ共通フォルダ, @"$(CommonUserFolder)" );

			// ユーザ個別フォルダ
			if( !string.IsNullOrEmpty( Folder.stgユーザ個別フォルダ ) )
				path = path.Replace( Folder.stgユーザ個別フォルダ, @"$(UserFolder)" );
		}


		static void tフォルダがなければ作成する( string path )
		{
			if( !Directory.Exists( path ) )
				Directory.CreateDirectory( path );
		}
	}
}
