﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;
using System.Reflection;
using FDK;

namespace StrokeStyleT
{
	/// <summary>
	/// <para>このクラスは、StrokeStyleT 本体と SSTFEditor の両方で使用する。</para>
	/// </summary>
	public class Cスコア : IDisposable
	{
		// 定数プロパティ

		public const double db初期BPM = 120.0;
		public const double db初期小節解像度 = 480.0;
		public const double dbBPM初期値固定での1小節4拍の時間ms = ( 60.0 * 1000 ) / ( Cスコア.db初期BPM / 4.0 );

		/// <summary>
		/// <para>1ms あたりの pixel 数。</para>
		/// <para>BPM 150 のとき、1小節が 234 pixel になるように調整。</para>
		/// <para>→ 60秒で150拍のとき、1小節(4拍)が 234 pixel。</para>
		/// <para>→ 60秒の間に、150[拍]÷4[拍]＝37.5[小節]。</para>
		/// <para>→ 60秒の間に、37.5[小節]×234[pixel/小節]＝ 8775[pixel]。</para>
		/// <para>→ 1ms の間に、8775[pixel]÷60000[ms]＝0.14625[pixel/ms]。割り切れて良かった。</para>
		/// </summary>
		public const double db基準譜面速度pxms = 0.14625 * 1.25;	// "* 1.25" は「x1.0はもう少し速くてもいいんではないか？」という感覚的な調整分。

		public static readonly Dictionary<ESSTFレーン, List<Eチップ>> dicSSTFレーンチップ対応表 =
			new Dictionary<ESSTFレーン, List<Eチップ>>() {
				{ ESSTFレーン.Bass, new List<Eチップ>() { Eチップ.Bass } },
				{ ESSTFレーン.BPM, new List<Eチップ>() { Eチップ.BPM } },
				{ ESSTFレーン.China, new List<Eチップ>() { Eチップ.China } },
				{ ESSTFレーン.HiHat, new List<Eチップ>() { Eチップ.HiHat_Close, Eチップ.HiHat_Foot, Eチップ.HiHat_HalfOpen, Eチップ.HiHat_Open } },
				{ ESSTFレーン.LeftCrash, new List<Eチップ>() { Eチップ.LeftCrash } },
				{ ESSTFレーン.Ride, new List<Eチップ>() { Eチップ.Ride, Eチップ.Ride_Cup } },
				{ ESSTFレーン.RightCrash, new List<Eチップ>() { Eチップ.RightCrash } },
				{ ESSTFレーン.Snare, new List<Eチップ>() { Eチップ.Snare, Eチップ.Snare_ClosedRim, Eチップ.Snare_Ghost, Eチップ.Snare_OpenRim } },
				{ ESSTFレーン.Song, new List<Eチップ>() { Eチップ.背景動画 } },
				{ ESSTFレーン.Splash, new List<Eチップ>() { Eチップ.Splash } },
				{ ESSTFレーン.Tom1, new List<Eチップ>() { Eチップ.Tom1, Eチップ.Tom1_Rim } },
				{ ESSTFレーン.Tom2, new List<Eチップ>() { Eチップ.Tom2, Eチップ.Tom2_Rim } },
				{ ESSTFレーン.Tom3, new List<Eチップ>() { Eチップ.Tom3, Eチップ.Tom3_Rim } },
			};


		// プロパティ；読み込み時または編集時に設定される

		public class CHeader
		{
			public string str曲名 = "(no title)";
			public string str説明文 = "";
		}
		public string str背景動画ファイル名 = "";
		public CHeader Header = new CHeader();
		public List<Cチップ> listチップ
		{
			get;
			protected set;
		}
		public List<double> list小節長倍率
		{
			get;
			protected set;
		}
		public int n最大小節番号
		{
			get
			{
				int n最大小節番号 = 0;

				foreach( Cチップ chip in this.listチップ )
				{
					if( chip.n小節番号 > n最大小節番号 )
						n最大小節番号 = chip.n小節番号;
				}

				return n最大小節番号;
			}
		}
		public Dictionary<int, string> dicメモ = new Dictionary<int, string>();	// SSTFEditorで使用。


		// メソッド

		public Cスコア()
		{
			this.listチップ = new List<Cチップ>();
			this.list小節長倍率 = new List<double>();
		}
		public Cスコア( string str曲データファイル名 )
			: this()
		{
			this.t曲データファイルを読み込む( str曲データファイル名 );
		}

		/// <summary>
		/// <para>指定された曲データファイルを読み込む。</para>
		/// <para>失敗すれば何らかの例外を発出する。</para>
		/// </summary>
		public void t曲データファイルを読み込む( string str曲データファイル名 )
		{
			#region [ 初期化 ]
			//-----------------
			this.list小節長倍率 = new List<double>();
			this.dicメモ = new Dictionary<int, string>();
			//-----------------
			#endregion

			#region [ 曲データファイルの読み込み ]
			//-----------------
			var sr = new StreamReader( str曲データファイル名, Encoding.UTF8 );

			try
			{
				int n行番号 = 0;
				int n現在の小節番号 = 0;
				int n現在の小節解像度 = 384;
				Eチップ e現在のチップ = Eチップ.Unknown;

				while( !sr.EndOfStream )
				{
					// 1行ずつ読み込む。

					n行番号++;

					string str行 = sr.ReadLine();


					// 前処理。

					#region [ 改行は ';' に、TABは空白文字にそれぞれ変換し、先頭末尾の空白を削除する。]
					//-----------------
					str行 = str行.Replace( Environment.NewLine, ";" );
					str行 = str行.Replace( '\t', ' ' );
					str行 = str行.Trim();
					//-----------------
					#endregion

					#region [ 行中の '#' 以降はコメントとして除外する。また、コメントだけの行はスキップする。]
					//-----------------
					int n区切り位置 = 0;

					for( n区切り位置 = 0; n区切り位置 < str行.Length; n区切り位置++ )
					{
						if( str行[ n区切り位置 ] == '#' )
							break;
					}

					if( n区切り位置 == 0 )
						continue;		// コメントだけの行はスキップ。

					if( n区切り位置 < str行.Length )
					{
						str行 = str行.Substring( 0, n区切り位置 - 1 );
						str行 = str行.Trim();
					}
					//-----------------
					#endregion

					#region [ 空行ならこの行はスキップする。]
					//-----------------
					if( string.IsNullOrEmpty( str行 ) )
						continue;
					//-----------------
					#endregion


					// ヘッダコマンド処理。

					#region [ ヘッダコマンドの処理を行う。]
					//-----------------
					if( str行.StartsWith( "Title", StringComparison.OrdinalIgnoreCase ) )
					{
						#region [ Title コマンド ]
						//-----------------
						string[] items = str行.Split( '=' );

						if( items.Length != 2 )
						{
							Trace.TraceError( string.Format( "Title の書式が不正です。スキップします。[{0}行目]", n行番号 ) );
							continue;
						}

						this.Header.str曲名 = items[ 1 ].Trim();
						//-----------------
						#endregion

						continue;
					}
					if( str行.StartsWith( "Description", StringComparison.OrdinalIgnoreCase ) )
					{
						#region [ Description コマンド ]
						//-----------------
						string[] items = str行.Split( '=' );

						if( items.Length != 2 )
						{
							Trace.TraceError( string.Format( "Description の書式が不正です。スキップします。[{0}行目]", n行番号 ) );
							continue;
						}
						this.Header.str説明文 = items[ 1 ].Trim().Replace( @"\n", Environment.NewLine );	// "\n" は改行に復号！
						//-----------------
						#endregion

						continue;
					}
					//-----------------
					#endregion


					// メモ（小節単位）処理。

					#region [ メモ（小節単位）の処理を行う。]
					//-----------------
					if( str行.StartsWith( "PartMemo", StringComparison.OrdinalIgnoreCase ) )
					{
						#region [ '=' 以前を除去する。]
						//-----------------
						int n等号位置 = str行.IndexOf( '=' );
						if( n等号位置 <= 0 )
						{
							Trace.TraceError( string.Format( "PartMemo の書式が不正です。スキップします。[{0}]行目]", n行番号 ) );
							continue;
						}
						str行 = str行.Substring( n等号位置 + 1 ).Trim();
						if( string.IsNullOrEmpty( str行 ) )
						{
							Trace.TraceError( string.Format( "PartMemo の書式が不正です。スキップします。[{0}]行目]", n行番号 ) );
							continue;
						}
						//-----------------
						#endregion

						#region [ カンマ位置を取得。]
						//-----------------
						int nカンマ位置 = str行.IndexOf( ',' );
						if( nカンマ位置 <= 0 )
						{
							Trace.TraceError( string.Format( "PartMemo の書式が不正です。スキップします。[{0}]行目]", n行番号 ) );
							continue;
						}
						//-----------------
						#endregion

						#region [ 小節番号を取得。]
						//-----------------
						string str小節番号 = str行.Substring( 0, nカンマ位置 );
						int n小節番号;
						if( !int.TryParse( str小節番号, out n小節番号 ) || n小節番号 < 0 )
						{
							Trace.TraceError( string.Format( "PartMemo の小節番号が不正です。スキップします。[{0}]行目]", n行番号 ) );
							continue;
						}
						//-----------------
						#endregion

						#region [ メモを取得。]
						//-----------------
						string strメモ = str行.Substring( nカンマ位置 + 1 );
						strメモ = strメモ.Replace( @"\n", Environment.NewLine );	// "\n" は改行に復号！
						//-----------------
						#endregion

						#region [ メモが空文字列でないなら dicメモ に登録すると同時に、チップとしても追加する。]
						//-----------------
						if( !string.IsNullOrEmpty( strメモ ) )
						{
							this.dicメモ.Add( n小節番号, strメモ );

							this.listチップ.Add(
								new Cチップ() {
									eチップ = Eチップ.小節メモ,
									n小節番号 = n小節番号,
									n小節内位置 = 0,
									n小節解像度 = 1,
								} );
						}
						//-----------------
						#endregion

						continue;
					}
					//-----------------
					#endregion


					// 上記行頭コマンド以外は、チップ記述行だと見なす。

					#region [ チップ記述コマンドの処理を行う。]
					//-----------------

					// 行を区切り文字でトークンに分割。
					string[] tokens = str行.Split( new char[] { ';', ':' } );

					// すべてのトークンについて……
					foreach( string token in tokens )
					{
						// トークンを分割。

						#region [ トークンを区切り文字 '=' で strコマンド と strパラメータ に分割し、それぞれの先頭末尾の空白を削除する。]
						//-----------------
						string[] items = token.Split( '=' );

						if( items.Length != 2 )
						{
							if( token.Trim().Length == 0 )	// 空文字列（行末など）は不正じゃない。
								continue;

							Trace.TraceError( string.Format( "コマンドとパラメータの記述書式が不正です。このコマンドをスキップします。[{0}行目]", n行番号 ) );
							continue;
						}

						string strコマンド = items[ 0 ].Trim();
						string strパラメータ = items[ 1 ].Trim();
						//-----------------
						#endregion


						// コマンド別に処理。

						if( strコマンド.Equals( "Part", StringComparison.OrdinalIgnoreCase ) )
						{
							#region [ Part（小節番号指定）コマンド ]
							//-----------------

							#region [ 小節番号を取得・設定。]
							//-----------------
							string str小節番号 = this.t指定された文字列の先頭から数字文字列を取り出す( ref strパラメータ );
							if( string.IsNullOrEmpty( str小節番号 ) )
							{
								Trace.TraceError( string.Format( "Part（小節番号）コマンドに小節番号の記述がありません。このコマンドをスキップします。[{0}行目]", n行番号 ) );
								continue;
							}

							int n小節番号 = 0;
							if( !int.TryParse( str小節番号, out n小節番号 ) )
							{
								Trace.TraceError( string.Format( "Part（小節番号）コマンドの小節番号が不正です。このコマンドをスキップします。[{0}行目]", n行番号 ) );
								continue;
							}
							if( n小節番号 < 0 )
							{
								Trace.TraceError( string.Format( "Part（小節番号）コマンドの小節番号が負数です。このコマンドをスキップします。[{0}行目]", n行番号 ) );
								continue;
							}

							n現在の小節番号 = n小節番号;
							//-----------------
							#endregion

							#region [ Part 属性があれば取得する。]
							//-----------------

							while( strパラメータ.Length > 0 )
							{
								// 属性ID を取得。

								char c属性ID = char.ToLower( strパラメータ[ 0 ] );


								// Part 属性があれば取得する。

								if( c属性ID == 's' )
								{
									#region [ 小節長倍率(>0) → list小節長倍率 ]
									//-----------------

									strパラメータ = strパラメータ.Substring( 1 ).Trim();

									string str小節長倍率 = this.t指定された文字列の先頭から数字文字列を取り出す( ref strパラメータ );
									strパラメータ = strパラメータ.Trim();

									if( string.IsNullOrEmpty( str小節長倍率 ) )
									{
										Trace.TraceError( string.Format( "Part（小節番号）コマンドに小節長倍率の記述がありません。この属性をスキップします。[{0}行目]", n行番号 ) );
										continue;
									}

									double db小節長倍率 = 1.0;

									if( !double.TryParse( str小節長倍率, out db小節長倍率 ) )
									{
										Trace.TraceError( string.Format( "Part（小節番号）コマンドの小節長倍率が不正です。この属性をスキップします。[{0}行目]", n行番号 ) );
										continue;
									}

									if( db小節長倍率 <= 0.0 )
									{
										Trace.TraceError( string.Format( "Part（小節番号）コマンドの小節長倍率が 0.0 または負数です。この属性をスキップします。[{0}行目]", n行番号 ) );
										continue;
									}


									// 小節長倍率辞書に追加 or 上書き更新。

									this.t小節長倍率を設定する( n現在の小節番号, db小節長倍率 );

									//-----------------
									#endregion

									continue;
								}
							}

							//-----------------
							#endregion

							//-----------------
							#endregion

							continue;
						}
						if( strコマンド.Equals( "Lane", StringComparison.OrdinalIgnoreCase ) )
						{
							#region [ Lane（レーン指定）コマンド（チップ種別の仮決め）]
							//-----------------
							if( strパラメータ.Equals( "LeftCrash", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.LeftCrash;

							else if( strパラメータ.Equals( "Ride", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.Ride;

							else if( strパラメータ.Equals( "China", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.China;

							else if( strパラメータ.Equals( "Splash", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.Splash;

							else if( strパラメータ.Equals( "HiHat", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.HiHat_Close;

							else if( strパラメータ.Equals( "Snare", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.Snare;

							else if( strパラメータ.Equals( "Bass", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.Bass;

							else if( strパラメータ.Equals( "Tom1", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.Tom1;

							else if( strパラメータ.Equals( "Tom2", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.Tom2;

							else if( strパラメータ.Equals( "Tom3", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.Tom3;

							else if( strパラメータ.Equals( "RightCrash", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.RightCrash;

							else if( strパラメータ.Equals( "BPM", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.BPM;

							else if( strパラメータ.Equals( "Song", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = Eチップ.背景動画;
							else
								Trace.TraceError( string.Format( "Lane（レーン指定）コマンドのパラメータ記述 '{1}' が不正です。このコマンドをスキップします。[{0}行目]", n行番号, strパラメータ ) );
							//-----------------
							#endregion

							continue;
						}
						if( strコマンド.Equals( "Resolution", StringComparison.OrdinalIgnoreCase ) )
						{
							#region [ Resolution（小節解像度指定）コマンド ]
							//-----------------
							int n解像度 = 1;

							if( !int.TryParse( strパラメータ, out n解像度 ) )
							{
								Trace.TraceError( string.Format( "Resolution（小節解像度指定）コマンドの解像度が不正です。このコマンドをスキップします。[{0}行目]", n行番号 ) );
								continue;
							}

							if( n解像度 < 1 )
							{
								Trace.TraceError( string.Format( "Resolution（小節解像度指定）コマンドの解像度は 1 以上でなければなりません。このコマンドをスキップします。[{0}行目]", n行番号 ) );
								continue;
							}

							n現在の小節解像度 = n解像度;
							//-----------------
							#endregion

							continue;
						}
						if( strコマンド.Equals( "Chips", StringComparison.OrdinalIgnoreCase ) )
						{
							#region [ Chips（チップ指定）コマンド ]
							//-----------------

							// パラメータを区切り文字 ',' でチップトークンに分割。

							string[] chipTokens = strパラメータ.Split( ',' );


							// すべてのチップトークンについて……

							for( int i = 0; i < chipTokens.Length; i++ )
							{
								chipTokens[ i ].Trim();

								#region [ 空文字はスキップ。]
								//-----------------
								if( chipTokens[ i ].Length == 0 )
									continue;
								//-----------------
								#endregion
								#region [ チップを生成。]
								//-----------------
								var chip = new Cチップ() {
									n小節番号 = n現在の小節番号,
									eチップ = e現在のチップ,
									n小節解像度 = n現在の小節解像度,
									n音量 = 4,
								};
								if( chip.eチップ == Eチップ.China ) chip.strチップ内文字列 = "C N";
								if( chip.eチップ == Eチップ.Splash ) chip.strチップ内文字列 = "S P";
								//-----------------
								#endregion

								#region [ チップ位置の取得。]
								//-----------------
								int nチップ位置 = 0;

								string str位置番号 = this.t指定された文字列の先頭から数字文字列を取り出す( ref chipTokens[ i ] );
								chipTokens[ i ].Trim();

								#region [ 文法チェック。]
								//-----------------
								if( string.IsNullOrEmpty( str位置番号 ) )
								{
									Trace.TraceError( string.Format( "チップの位置指定の記述がありません。このチップをスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1 ) );
									continue;
								}
								//-----------------
								#endregion
								#region [ 位置を取得。]
								//-----------------
								if( !int.TryParse( str位置番号, out nチップ位置 ) )
								{
									Trace.TraceError( string.Format( "チップの位置指定の記述が不正です。このチップをスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1 ) );
									continue;
								}
								//-----------------
								#endregion
								#region [ 値域チェック。]
								//-----------------
								if( nチップ位置 < 0 ||
									nチップ位置 >= n現在の小節解像度 )
								{
									Trace.TraceError( string.Format( "チップの位置が負数であるか解像度(Resolution)以上の値になっています。このチップをスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1 ) );
									continue;
								}
								//-----------------
								#endregion

								chip.n小節内位置 = nチップ位置;
								//-----------------
								#endregion
								#region [ 共通属性・レーン別属性があれば取得する。]
								//-----------------
								while( chipTokens[ i ].Length > 0 )
								{
									// 属性ID を取得。

									char c属性ID = char.ToLower( chipTokens[ i ][ 0 ] );


									// 共通属性があれば取得。

									if( c属性ID == 'v' )
									{
										#region [ 音量(1～4) ]
										//-----------------
										chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();

										string str音量 = this.t指定された文字列の先頭から数字文字列を取り出す( ref chipTokens[ i ] );
										chipTokens[ i ].Trim();

										int nチップ音量 = 0;

										#region [ 文法チェック。]
										//-----------------
										if( string.IsNullOrEmpty( str音量 ) )
										{
											Trace.TraceError( string.Format( "チップの音量指定の記述がありません。この属性をスキップします。[{0}行目l {1}個目のチップ]", n行番号, i + 1 ) );
											continue;
										}
										//-----------------
										#endregion
										#region [ チップ音量の取得。]
										//-----------------
										if( !int.TryParse( str音量, out nチップ音量 ) )
										{
											Trace.TraceError( string.Format( "チップの音量指定の記述が不正です。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1 ) );
											continue;
										}
										//-----------------
										#endregion
										#region [ 値域チェック。]
										//-----------------
										if( nチップ音量 < 1 || nチップ音量 > 4 )
										{
											Trace.TraceError( string.Format( "チップの音量が適正範囲（1～4）を超えています。このチップをスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1 ) );
											continue;
										}
										//-----------------
										#endregion

										chip.n音量 = nチップ音量;
										//-----------------
										#endregion

										continue;
									}


									// レーン別属性があれば取得。

									switch( e現在のチップ )
									{
										#region [ case LeftCymbal ]
										//-----------------
										case Eチップ.LeftCrash:

											#region [ 未知の属性 ]
											//-----------------
											chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
											Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
											//-----------------
											#endregion

											continue;

										//-----------------
										#endregion
										#region [ case Ride ]
										//-----------------
										case Eチップ.Ride:
										case Eチップ.Ride_Cup:

											if( c属性ID == 'c' )
											{
												#region [ Ride.カップ ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.eチップ = Eチップ.Ride_Cup;
												//-----------------
												#endregion
											}
											else
											{
												#region [ 未知の属性 ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region [ case China ]
										//-----------------
										case Eチップ.China:

											#region [ 未知の属性 ]
											//-----------------
											chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
											Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
											//-----------------
											#endregion

											continue;

										//-----------------
										#endregion
										#region [ case Splash ]
										//-----------------
										case Eチップ.Splash:

											#region [ 未知の属性 ]
											//-----------------
											chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
											Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
											//-----------------
											#endregion

											continue;

										//-----------------
										#endregion
										#region [ case HiHat ]
										//-----------------
										case Eチップ.HiHat_Close:
										case Eチップ.HiHat_HalfOpen:
										case Eチップ.HiHat_Open:
										case Eチップ.HiHat_Foot:

											if( c属性ID == 'o' )
											{
												#region [ HiHat.オープン ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.eチップ = Eチップ.HiHat_Open;
												//-----------------
												#endregion
											}
											else if( c属性ID == 'h' )
											{
												#region [ HiHat.ハーフオープン ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.eチップ = Eチップ.HiHat_HalfOpen;
												//-----------------
												#endregion
											}
											else if( c属性ID == 'c' )
											{
												#region [ HiHat.クローズ ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.eチップ = Eチップ.HiHat_Close;
												//-----------------
												#endregion
											}
											else if( c属性ID == 'f' )
											{
												#region [ HiHat.フットスプラッシュ ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.eチップ = Eチップ.HiHat_Foot;
												//-----------------
												#endregion
											}
											else
											{
												#region [ 未知の属性 ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region [ case Snare ]
										//-----------------
										case Eチップ.Snare:
										case Eチップ.Snare_ClosedRim:
										case Eチップ.Snare_OpenRim:
										case Eチップ.Snare_Ghost:

											if( c属性ID == 'o' )
											{
												#region [ Snare.オープンリム ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.eチップ = Eチップ.Snare_OpenRim;
												//-----------------
												#endregion
											}
											else if( c属性ID == 'c' )
											{
												#region [ Snare.クローズドリム ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.eチップ = Eチップ.Snare_ClosedRim;
												//-----------------
												#endregion
											}
											else if( c属性ID == 'g' )
											{
												#region [ Snare.ゴースト ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.eチップ = Eチップ.Snare_Ghost;
												//-----------------
												#endregion
											}
											else
											{
												#region [ 未知の属性 ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region [ case Bass ]
										//-----------------
										case Eチップ.Bass:

											#region [ 未知の属性 ]
											//-----------------
											chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
											Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
											//-----------------
											#endregion

											continue;

										//-----------------
										#endregion
										#region [ case Tom1 ]
										//-----------------
										case Eチップ.Tom1:
										case Eチップ.Tom1_Rim:

											if( c属性ID == 'r' )
											{
												#region [ Tom1.リム ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.eチップ = Eチップ.Tom1_Rim;
												//-----------------
												#endregion
											}
											else
											{
												#region [ 未知の属性 ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region [ case Tom2 ]
										//-----------------
										case Eチップ.Tom2:
										case Eチップ.Tom2_Rim:

											if( c属性ID == 'r' )
											{
												#region [ Tom2.リム ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.eチップ = Eチップ.Tom2_Rim;
												//-----------------
												#endregion
											}
											else
											{
												#region [ 未知の属性 ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region [ case Tom3 ]
										//-----------------
										case Eチップ.Tom3:
										case Eチップ.Tom3_Rim:

											if( c属性ID == 'r' )
											{
												#region [ Tom3.リム ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.eチップ = Eチップ.Tom3_Rim;
												//-----------------
												#endregion
											}
											else
											{
												#region [ 未知の属性 ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region [ case RightCymbal ]
										//-----------------
										case Eチップ.RightCrash:

											#region [ 未知の属性 ]
											//-----------------
											chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
											Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
											//-----------------
											#endregion

											continue;

										//-----------------
										#endregion
										#region [ case BPM ]
										//-----------------
										case Eチップ.BPM:

											if( c属性ID == 'b' )
											{
												#region [ BPM値 ]
												//-----------------

												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();

												string strBPM = this.t指定された文字列の先頭から数字文字列を取り出す( ref chipTokens[ i ] );
												chipTokens[ i ].Trim();

												if( string.IsNullOrEmpty( strBPM ) )
												{
													Trace.TraceError( string.Format( "BPM数値の記述がありません。この属性をスキップします。[{0}行目l {1}個目のチップ]", n行番号, i + 1 ) );
													continue;
												}

												double dbBPM = 0.0;

												if( !double.TryParse( strBPM, out dbBPM ) || dbBPM <= 0.0 )
												{
													Trace.TraceError( string.Format( "BPM数値の記述が不正です。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1 ) );
													continue;
												}

												chip.dbBPM = dbBPM;
												chip.strチップ内文字列 = dbBPM.ToString( "###.##" );
												//-----------------
												#endregion
											}
											else
											{
												#region [ 未知の属性 ]
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region [ case Song ]
										//-----------------
										case Eチップ.背景動画:

											#region [ 未知の属性 ]
											//-----------------
											chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
											Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
											//-----------------
											#endregion

											continue;

										//-----------------
										#endregion
									}

									#region [ 未知の属性 ]
									//-----------------
									chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
									Trace.TraceError( string.Format( "未対応の属性「{2}」が指定されています。この属性をスキップします。[{0}行目; {1}個目のチップ]", n行番号, i + 1, c属性ID ) );
									//-----------------
									#endregion
								}
								//-----------------
								#endregion


								// チップをリストに追加。

								this.listチップ.Add( chip );
							}

							//-----------------
							#endregion

							continue;
						}

						Trace.TraceError( string.Format( "不正なコマンド「{0}」が存在します。[{1}行目]", strコマンド, n行番号 ) );
					}
					//-----------------
					#endregion
				}
			}
			finally
			{
				sr.Close();
			}
			//-----------------
			#endregion

			#region [ 拍線の追加。小節線を先に追加すると小節が１つ増えるので、先に拍線から追加する。]
			//-----------------

			// 「this.n最大小節番号」はチップ数に依存して変化するので、for 文には組み込まないこと。
			int n最大小節番号 = this.n最大小節番号;

			for( int i = 0; i <= n最大小節番号; i++ )
			{
				double db小節長倍率 = this.t小節長倍率を取得する( i );
				for( int n = 1; n * 0.25 < db小節長倍率; n++ )
				{
					this.listチップ.Add(
						new Cチップ() {
							n小節番号 = i,
							eチップ = Eチップ.拍線,
							n小節内位置 = (int) ( ( n * 0.25 ) * 100 ),
							n小節解像度 = (int) ( db小節長倍率 * 100 ),
						} );
				}
			}
			//-----------------
			#endregion

			#region [ 小節線の追加。]
			//-----------------
			n最大小節番号 = this.n最大小節番号;

			for( int i = 0; i <= n最大小節番号 + 1; i++ )
			{
				this.listチップ.Add(
					new Cチップ() {
						n小節番号 = i,
						eチップ = Eチップ.小節線,
						n小節内位置 = 0,
						n小節解像度 = 1,
					} );
			}
			//-----------------
			#endregion

			this.listチップ.Sort();

			#region [ 全チップの発声/描画時刻と譜面内位置の計算 ]
			//-----------------

			// 1. BPMチップを無視し(初期BPMで固定)、dic小節長倍率, Cチップ.n小節解像度, Cチップ.n小節内位置 から両者を計算する。
			//    以下、listチップが小節番号順にソートされているという前提で。

			double dbチップが存在する小節の先頭時刻ms = 0.0;
			int n現在の小節の番号 = 0;

			foreach( Cチップ chip in this.listチップ )
			{
				#region [ チップの小節番号が現在の小節番号よりも大きい場合、チップが存在する小節に至るまで、「dbチップが存在する小節の先頭時刻ms」を更新する。]
				//-----------------
				while( n現在の小節の番号 < chip.n小節番号 )
				{
					double db現在の小節の小節長倍率 = this.t小節長倍率を取得する( n現在の小節の番号 );
					dbチップが存在する小節の先頭時刻ms += dbBPM初期値固定での1小節4拍の時間ms * db現在の小節の小節長倍率;

					n現在の小節の番号++;	// n現在の小節番号 が chip.n小節番号 に追いつくまでループする。
				}
				//-----------------
				#endregion

				#region [ チップの発声/描画時刻を求める。]
				//-----------------
				double dbチップが存在する小節の小節長倍率 = this.t小節長倍率を取得する( n現在の小節の番号 );

				chip.n発声時刻ms =
					chip.n描画時刻ms =
						(long) ( dbチップが存在する小節の先頭時刻ms + ( dbBPM初期値固定での1小節4拍の時間ms * dbチップが存在する小節の小節長倍率 * chip.n小節内位置 ) / chip.n小節解像度 );
				//-----------------
				#endregion
			}


			// 2. BPMチップを考慮しながら調整する。（譜面内位置grid はBPMの影響を受けないので無視）

			double db現在のBPM = Cスコア.db初期BPM;
			int nチップ数 = this.listチップ.Count;

			for( int i = 0; i < nチップ数; i++ )
			{
				// BPM チップ以外は無視。

				var BPMチップ = this.listチップ[ i ];
				if( BPMチップ.eチップ != Eチップ.BPM )
					continue;


				// BPMチップより後続の全チップの n発声/描画時刻ms を、新旧BPMの比率（加速率）で修正する。

				double db加速率 = BPMチップ.dbBPM / db現在のBPM;	// BPMチップ.dbBPM > 0.0 であることは読み込み時に保証済み。
				for( int j = i + 1; j < nチップ数; j++ )
				{
					this.listチップ[ j ].n発声時刻ms =
						this.listチップ[ j ].n描画時刻ms =
							(long) ( BPMチップ.n発声時刻ms + ( ( this.listチップ[ j ].n発声時刻ms - BPMチップ.n発声時刻ms ) / db加速率 ) );
				}

				db現在のBPM = BPMチップ.dbBPM;
			}
			//-----------------
			#endregion
		}
		internal void t曲データファイルを読み込む・ヘッダだけ( string str曲データファイル名 )
		{
			this.list小節長倍率 = new List<double>();

			#region [ 曲データファイルの読み込み ]
			//-----------------
			var sr = new StreamReader( str曲データファイル名, Encoding.UTF8 );
			try
			{
				int n行番号 = 0;

				while( !sr.EndOfStream )
				{
					// 1行ずつ読み込む。

					n行番号++;

					string str行 = sr.ReadLine();


					// 前処理。

					#region [ 文字列に前処理を行う。]
					//-----------------

					// 改行は ';' に、TABは空白文字にそれぞれ変換し、先頭末尾の空白を削除する。

					str行 = str行.Replace( Environment.NewLine, ";" );
					str行 = str行.Replace( '\t', ' ' );
					str行 = str行.Trim();


					// 行中の '#' 以降はコメントとして除外する。また、コメントだけの行はスキップする。

					int n区切り位置 = 0;

					for( n区切り位置 = 0; n区切り位置 < str行.Length; n区切り位置++ )
					{
						if( str行[ n区切り位置 ] == '#' )
							break;
					}

					if( n区切り位置 == 0 )
						continue;		// コメントだけの行はスキップ。

					if( n区切り位置 < str行.Length )
					{
						str行 = str行.Substring( 0, n区切り位置 - 1 );
						str行 = str行.Trim();
					}


					// 空行ならこの行はスキップする。

					if( string.IsNullOrEmpty( str行 ) )
						continue;

					//-----------------
					#endregion


					// ヘッダコマンド処理。

					#region [ ヘッダコマンドの処理を行う。]
					//-----------------

					if( str行.StartsWith( "Title", StringComparison.OrdinalIgnoreCase ) )
					{
						#region [ Title コマンド ]
						//-----------------
						string[] items = str行.Split( '=' );

						if( items.Length != 2 )
						{
							Trace.TraceError( string.Format( "Title の書式が不正です。スキップします。[{0}行目]", n行番号 ) );
							continue;
						}

						this.Header.str曲名 = items[ 1 ].Trim();
						//-----------------
						#endregion

						continue;
					}
					if( str行.StartsWith( "Description", StringComparison.OrdinalIgnoreCase ) )
					{
						#region [ Description コマンド ]
						//-----------------
						string[] items = str行.Split( '=' );

						if( items.Length != 2 )
						{
							Trace.TraceError( string.Format( "Description の書式が不正です。スキップします。[{0}行目]", n行番号 ) );
							continue;
						}

						this.Header.str説明文 = items[ 1 ].Trim().Replace( @"\n", Environment.NewLine );	// "\n" は改行に復号！
						//-----------------
						#endregion

						continue;
					}
					//-----------------
					#endregion


					// 上記行頭コマンド以外は無視。
				}
			}
			finally
			{
				sr.Close();
			}
			//-----------------
			#endregion
		}
		/// <summary>
		/// <para>現在の Cスコア の内容をデータファイル（*.sstf）に書き出す。</para>
		/// <para>小節線、拍線、Unknown チップは出力しない。</para>
		/// <para>失敗すれば何らかの例外を発出する。</para>
		/// </summary>
		public void t曲データファイルを書き出す( string str曲データファイル名, string strヘッダ行 )
		{
			var sw = new StreamWriter( str曲データファイル名, false, Encoding.UTF8 );
			try
			{
				// ヘッダ行の出力

				sw.WriteLine( string.Format( "{0}", strヘッダ行 ) );	// strヘッダ行に"{...}"が入ってても大丈夫なようにstring.Format()で囲む。
				sw.WriteLine( "" );


				// ヘッダコマンド行の出力

				sw.WriteLine( "Title={0}", ( string.IsNullOrEmpty( this.Header.str曲名 ) ) ? "(no title)" : this.Header.str曲名 );
				if( !string.IsNullOrEmpty( this.Header.str説明文 ) )
					sw.WriteLine( "Description=" + this.Header.str説明文.Replace( Environment.NewLine, @"\n" ) );	// 改行は "\n" に置換！
				sw.WriteLine( "" );


				// 全チップの出力

				#region [ 全チップの最終小節番号を取得。]
				//-----------------
				int n最終小節番号 = 0;
				foreach( var cc in this.listチップ )
				{
					if( cc.n小節番号 > n最終小節番号 )
						n最終小節番号 = cc.n小節番号;
				}
				//-----------------
				#endregion

				for( int n小節番号 = 0; n小節番号 <= n最終小節番号; n小節番号++ )
				{
					#region [ dicレーン別チップリストの初期化。]
					//-----------------
					Dictionary<ESSTFレーン, List<Cチップ>> dicレーン別チップリスト = new Dictionary<ESSTFレーン, List<Cチップ>>();

					foreach( var lane in Enum.GetValues( typeof( ESSTFレーン ) ) )
						dicレーン別チップリスト[ (ESSTFレーン) lane ] = new List<Cチップ>();
					//-----------------
					#endregion
					#region [ dicレーン別チップリストの構築； n小節番号 の小節に存在するチップのみをレーン別に振り分けて格納する。]
					//-----------------
					foreach( var cc in this.listチップ )
					{
						if( cc.n小節番号 > n小節番号 )
							break;		// listチップは昇順に並んでいるので、これ以上検索しても無駄。

						if( cc.eチップ == Eチップ.小節線 ||
							cc.eチップ == Eチップ.拍線 ||
							cc.eチップ == Eチップ.小節メモ ||
							cc.eチップ == Eチップ.Unknown )
						{
							continue;	// これらは出力しないので無視。
						}

						if( cc.n小節番号 == n小節番号 )
						{
							var lane = ESSTFレーン.Bass;	// 対応するレーンがなかったら Bass でも返しておく。

							foreach( var kvp in dicSSTFレーンチップ対応表 )
							{
								if( kvp.Value.Contains( cc.eチップ ) )
								{
									lane = kvp.Key;
									break;
								}
							}

							dicレーン別チップリスト[ lane ].Add( cc );
						}
					}
					//-----------------
					#endregion

					#region [ Part行 出力。]
					//-----------------
					sw.Write( "Part = {0}", n小節番号.ToString() );

					if( this.list小節長倍率[ n小節番号 ] != 1.0 )
						sw.Write( "s" + this.list小節長倍率[ n小節番号 ].ToString() );

					sw.WriteLine( ";" );
					//-----------------
					#endregion
					#region [ Lane, Resolution, Chip 行 出力。]
					//-----------------
					foreach( var laneObj in Enum.GetValues( typeof( ESSTFレーン ) ) )
					{
						var lane = (ESSTFレーン) laneObj;

						if( dicレーン別チップリスト[ lane ].Count > 0 )
						{
							sw.Write( "Lane={0}; ", lane.ToString() );

							#region [ int n新しい解像度 を求める。]
							//-----------------
							int n新しい解像度 = 1;
							foreach( var cc in dicレーン別チップリスト[ lane ] )
								n新しい解像度 = this.t最小公倍数を返す( n新しい解像度, cc.n小節解像度 );
							//-----------------
							#endregion
							#region [ dicレーン別チップリスト[ lane ] 要素の n小節解像度 と n小節内位置 を n新しい解像度 に合わせて修正する。 ]
							//-----------------
							foreach( var cc in dicレーン別チップリスト[ lane ] )
							{
								int n倍率 = n新しい解像度 / cc.n小節解像度;		// n新しい解像度は n小節解像度 の最小公倍数なので常に割り切れる

								cc.n小節解像度 *= n倍率;
								cc.n小節内位置 *= n倍率;
							}
							//-----------------
							#endregion

							sw.Write( "Resolution = {0}; ", n新しい解像度 );
							sw.Write( "Chips = " );

							for( int i = 0; i < dicレーン別チップリスト[ lane ].Count; i++ )
							{
								Cチップ cc = dicレーン別チップリスト[ lane ][ i ];

								// 位置を出力

								sw.Write( cc.n小節内位置.ToString() );

								// 属性を出力（あれば）

								#region [ (1) 共通属性 ]
								//-----------------
								if( cc.n音量 < 4 )
									sw.Write( "v" + cc.n音量.ToString() );
								//-----------------
								#endregion
								#region [ (2) 専用属性 ]
								//-----------------
								switch( cc.eチップ )
								{
									case Eチップ.Ride_Cup:
										sw.Write( 'c' );
										break;

									case Eチップ.HiHat_Open:
										sw.Write( 'o' );
										break;

									case Eチップ.HiHat_HalfOpen:
										sw.Write( 'h' );
										break;

									case Eチップ.HiHat_Foot:
										sw.Write( 'f' );
										break;

									case Eチップ.Snare_OpenRim:
										sw.Write( 'o' );
										break;

									case Eチップ.Snare_ClosedRim:
										sw.Write( 'c' );
										break;

									case Eチップ.Snare_Ghost:
										sw.Write( 'g' );
										break;

									case Eチップ.Tom1_Rim:
										sw.Write( 'r' );
										break;

									case Eチップ.Tom2_Rim:
										sw.Write( 'r' );
										break;

									case Eチップ.Tom3_Rim:
										sw.Write( 'r' );
										break;

									case Eチップ.BPM:
										sw.Write( "b" + cc.dbBPM.ToString() );
										break;
								}
								//-----------------
								#endregion


								// 区切り文字 または 終端文字 を出力

								sw.Write( ( i == dicレーン別チップリスト[ lane ].Count - 1 ) ? ";" : "," );
							}

							sw.WriteLine( "" );	// 改行

						}
					}
					//-----------------
					#endregion

					sw.WriteLine( "" );	// 次の Part 前に１行開ける
				}


				// メモ（小節単位）の出力

				#region [ dicメモ を小節番号で昇順にソートし直した dic昇順メモ を作成する。]
				//-----------------
				var dic昇順メモ = new Dictionary<int, string>();
				int n最大小節番号 = this.n最大小節番号;

				for( int i = 0; i <= n最大小節番号; i++ )
				{
					if( this.dicメモ.ContainsKey( i ) )
						dic昇順メモ.Add( i, this.dicメモ[ i ] );
				}
				//-----------------
				#endregion
				#region [ dic昇順メモを出力する。]
				//-----------------
				foreach( var kvp in dic昇順メモ )
				{
					int n小節番号 = kvp.Key;
					string strメモ = kvp.Value.Replace( Environment.NewLine, @"\n" );	// 改行は "\n" に置換！

					sw.WriteLine( "PartMemo={0},{1}", n小節番号, strメモ );
				}
				sw.WriteLine( "" );
				//-----------------
				#endregion
			}
			finally
			{
				sw.Close();
			}
		}
		/// <summary>
		/// <para>Config.Speed を考慮し、指定された時間[ms]の間に流れるピクセル数を算出して返す。</para>
		/// </summary>
		internal int n指定された時間msに対応する符号付きピクセル数を返す( long n指定時間ms )
		{
			return (int) ( n指定時間ms * Cスコア.db基準譜面速度pxms * Global.User.Config.ScrollSpeed );
		}
		public double t小節長倍率を取得する( int n小節番号 )
		{
			// list小節長倍率 が短ければ増設する。

			if( n小節番号 >= this.list小節長倍率.Count )
			{
				int n不足数 = n小節番号 - this.list小節長倍率.Count + 1;
				for( int i = 0; i < n不足数; i++ )
					this.list小節長倍率.Add( 1.0 );
			}


			// 倍率を返す。

			return this.list小節長倍率[ n小節番号 ];
		}
		public void t小節長倍率を設定する( int n小節番号, double db倍率 )
		{
			// list小節長倍率 が短ければ増設する。

			if( n小節番号 >= this.list小節長倍率.Count )
			{
				int n不足数 = n小節番号 - this.list小節長倍率.Count + 1;
				for( int i = 0; i < n不足数; i++ )
					this.list小節長倍率.Add( 1.0 );
			}

			// 倍率を登録する。

			this.list小節長倍率[ n小節番号 ] = db倍率;
		}

		#region [ IDisposable 実装 …… 特になし ]
		//-----------------
		public void Dispose()
		{
		}
		//-----------------
		#endregion

		// <summary>
		// <para>取出文字列の先頭にある数字（小数点も有効）の連続した部分を取り出して、戻り値として返す。</para>
		// <para>また、取出文字列から取り出した数字文字列部分を除去した文字列を再度格納する。</para>
		// </summary>
		protected string t指定された文字列の先頭から数字文字列を取り出す( ref string str取出文字列 )
		{
			int n桁数 = 0;

			while( ( n桁数 < str取出文字列.Length ) && ( char.IsDigit( str取出文字列[ n桁数 ] ) || str取出文字列[ n桁数 ] == '.' ) )
				n桁数++;

			if( n桁数 == 0 )
				return "";

			string str数字文字列 = str取出文字列.Substring( 0, n桁数 );
			str取出文字列 = ( n桁数 == str取出文字列.Length ) ? "" : str取出文字列.Substring( n桁数 );

			return str数字文字列;
		}

		protected int t最大公約数を返す( int m, int n )
		{
			if( m <= 0 || n <= 0 )
				throw new ArgumentException( "引数に0以下の数は指定できません。" );

			int r;

			// ユーグリッドの互除法
			while( ( r = m % n ) != 0 )
			{
				m = n;
				n = r;
			}

			return n;
		}
		protected int t最小公倍数を返す( int m, int n )
		{
			if( m <= 0 || n <= 0 )
				throw new ArgumentException( "引数に0以下の数は指定できません。" );

			return ( m * n / this.t最大公約数を返す( m, n ) );
		}
	}
}
