/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Data/TimeAndSalesSerializer.cs#10 $
 * $DateTime: 2008/03/13 13:20:43 $
 * 
 * Time & Sales̃VACY
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;

using TimeAndSales=Bellagio.Values.TimeAndSales;
using TickData=Bellagio.Values.TickData;
using TickItaRelation=Bellagio.Values.TickItaRelation;
using BTime=Bellagio.Values.BTime;
using Bellagio.Data;

#if UNITTEST
using Bellagio.Environment;
using NUnit.Framework;
#endif

namespace Bellagio.Data {
    /*
     * l@Ff[^TCYɂ
     * @PڂAEliEoBӂ̊i[12oCgB
     * @PTick񐔂́Ao敨10000A؎͂1000`3000ƂƂB
     * @ƁA12oCg * 1000 * 2000ƂĂ 24MB łP̑STickData͊i[łBłߑ匩ς肾A邱Ƃ͂Ȃ̂ŏ\IłB
     * @łƂ̍炢͍B
     */

    /* wb_̍\
     *   [magic number] [version] [date] [brand count]
     *   { [code] [data offset](Xg[[) } * brand_count
     *   { [0] [data offset](I[L) }
     *   
     * ̂ƂTime&SalesЂÂ
     */
    public abstract class BinaryTimeAndSalesIO {
        public const int MAGIC_NUMBER = 0x0000DA0A;
        public const int TICKDATA_SIZE = 16;

        protected int _version;
        public struct OffsetInfo {
            public int code;
            public int offset;
        }
        protected int _rawDate;
        protected OffsetInfo[] _offsets;

        public int Version {
            get {
                return _version;
            }
        }
        public int RawDate {
            get {
                return _rawDate;
            }
        }
        internal static TimeAndSales GetTimeAndSales(Stock stock) {
            return BellagioRoot.IntraDayTradeProvider.Lookup(stock).TimeAndSales;
        }
    }

    public class BinaryTimeAndSalesReader : BinaryTimeAndSalesIO {
        private Stream _strm;
        public BinaryTimeAndSalesReader(Stream strm) {
            _strm = strm;
        }
        public OffsetInfo[] LoadAndImport() {
            BinaryReader br = new BinaryReader(_strm);
            //Header
            if(br.ReadInt32()!=MAGIC_NUMBER) FormatError();
            _version = br.ReadInt32();
            if(_version!=1) FormatError();
            _rawDate = br.ReadInt32();
            int count = br.ReadInt32();
            if(count<0 || count>10000) FormatError(); //PƂƂ͂Ȃ
            _offsets = new OffsetInfo[count+1];
            for(int i=0; i<_offsets.Length; i++) {
                _offsets[i].code = br.ReadInt32();
                _offsets[i].offset = br.ReadInt32();
            }
            
            //BODY
            for(int i=0; i<_offsets.Length-1; i++) {
                Stock stock = BellagioRoot.GlobalStockCollection.FindExact(_offsets[i].code.ToString()).Primary;
                TimeAndSales ts = GetTimeAndSales(stock);
                ts.Clear();
                int ts_count = (_offsets[i+1].offset - _offsets[i].offset) / TICKDATA_SIZE;
                if(ts_count<0 || ts_count>0x10000) FormatError();
                for(int j=0; j<ts_count; j++) {
                    int time = br.ReadInt32(); int price = br.ReadInt32(); int vol = br.ReadInt32(); TickItaRelation rel = (TickItaRelation)br.ReadInt32();
                    ts.Add(time, price, vol, rel);
                }
            }


            return _offsets;
        }

        private void FormatError() {
            throw new IOException(); //NOTE 蔲
        }

    }

    public class BinaryTimeAndSalesWriter : BinaryTimeAndSalesIO {
        private Stream _strm;
        public BinaryTimeAndSalesWriter(Stream strm) {
            _strm = strm;
            _version = 1;
        }
        public void Write(int raw_date, ListedStockCollection collection) {
            _rawDate = raw_date;
            _offsets = new OffsetInfo[collection.Count+1]; //I[
            long stream_offset = _strm.Position;
            int header_size = 4 * (4 + 2 * _offsets.Length);
            _strm.Seek(header_size, SeekOrigin.Current);
            BinaryWriter bw = new BinaryWriter(_strm);

            //Body
            int offset = header_size;
            int stock_index = 0;
            foreach(Stock stock in collection) {
                _offsets[stock_index].code = Int32.Parse(stock.Profile.Code);
                _offsets[stock_index].offset = offset;
                TimeAndSales ts =GetTimeAndSales(stock);
                int ts_count = ts.Count;
                for(int j=0; j<ts_count; j++) {
                    TickData td = ts.TickAt(j);
                    bw.Write(td.Time);
                    bw.Write(td.Price);
                    bw.Write(td.Volume);
                    bw.Write((int)td.TickItaRelation);
                }

                stock_index++;
                offset += TICKDATA_SIZE * ts.Count;
            }
            _offsets[stock_index].code = 0;
            _offsets[stock_index].offset = offset; //Xgf[^
            bw.Flush();

            //Header
            _strm.Seek(stream_offset, SeekOrigin.Begin);
            bw = new BinaryWriter(_strm);
            bw.Write(MAGIC_NUMBER);
            bw.Write(_version);
            bw.Write(_rawDate);
            bw.Write(collection.Count);
            for(int i=0; i<_offsets.Length; i++) {
                bw.Write(_offsets[i].code);
                bw.Write(_offsets[i].offset);
            }
            bw.Flush();
            Debug.Assert(_strm.Position==(long)header_size);
        }
    }

    //Ƃƕʂ̏ꏊɂR[hȂ̂ŏLƐ܂
    public class TextTimeAndSalesReader {
        public TimeAndSales Read(TextReader reader) {
            TimeAndSales result =new TimeAndSales();

            //ǂ񂾂ƃ\[ĝňUʃRNV
            List<TickData> r = new List<TickData>();
            string line = reader.ReadLine();
            while(line!=null) {
                r.Add(ParseTickData(line));
                line = reader.ReadLine();
            }

            r.Sort(delegate(TickData t1, TickData t2) { return t1.Time - t2.Time; });
            ModifyTime(r);
            foreach(TickData td in r) result.Add(td);

            return result;
        }

        //Time Price VolumeX^C̍sp[X
        private TickData ParseTickData(string line) {
            string[] x = line.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
            if(x.Length < 3) throw new IOException("TickData format error(line=["+line+"])");
            return new TickData(BTime.ParseToInt(x[0]), Int32.Parse(x[1]), Int32.Parse(x[2]), TickItaRelation.Unknown);
        }

        //PʂœɂȂĂƂAb𓙊Ԋuɂ΂炷
        private void ModifyTime(List<TickData> r) {
            int start_index = 0;
            while(start_index < r.Count) {
                int end_index = start_index+1;
                int time = r[start_index].Time;
                while(end_index < r.Count && r[end_index].Time==time) end_index++;

                //oKvȂ
                if(end_index > start_index+1) {
                    int len = end_index - start_index; //܂ɑƂ60ɂȂȂ悤
                    for(int i = 1; i < len; i++) {
                        r[start_index+i].SetTime(r[start_index+i].Time + (60 * i / len));
                    }
                }

                start_index = end_index;
            }

        }
    }

#if UNITTEST
    [TestFixture]
    public class TimeAndSalesSerializerTests {
        private TimeAndSales CreateSampleTimeAndSales1() {
            TimeAndSales ts = new TimeAndSales();
            ts.Add(900, 1000, 10, TickItaRelation.Ask);
            return ts;
        }
        private TimeAndSales CreateSampleTimeAndSales2() {
            TimeAndSales ts = new TimeAndSales();
            ts.Add(900, 101, 11, TickItaRelation.Ask);
            ts.Add(901, 102, 12, TickItaRelation.Ask);
            ts.Add(902, 103, 13, TickItaRelation.Bid);
            return ts;
        }

        private TimeAndSales GetTimeAndSales(Stock stock) {
            return BellagioRoot.IntraDayTradeProvider.Lookup(stock).TimeAndSales;
        }

        private void Init(int stock_count) {
            BellagioEnvironmentParam p = new BellagioEnvironmentParam();
            p.SetupForUnitTest();
            p.UNITTEST_STOCK_COUNT = stock_count;
            BellagioRoot.Init(p);
            BellagioRoot.BootForTest();
        }

        private ListListedStockCollection ProfileToCollection(StockProfileCollection r) {
            ListListedStockCollection c = new ListListedStockCollection();
            foreach(AbstractStockProfile sp in r)
                c.Add(sp.Primary);
            return c;
        }


        private void AssertTimeAndSalesEquals(TimeAndSales e, TimeAndSales a) {
            Assert.AreEqual(e.Count, a.Count);
            for(int i=0; i<e.Count; i++) {
                TickData tde = e.TickAt(i);
                TickData tda = a.TickAt(i);
                Assert.AreEqual(tde.Time, tda.Time);
                Assert.AreEqual(tde.Price, tda.Price);
                Assert.AreEqual(tde.Volume, tda.Volume);
                Assert.AreEqual(tde.TickItaRelation, tda.TickItaRelation);
            }
        }


        [Test]
        public void TextBasic1() {
            StringReader reader = new StringReader("9:00:00 100  1000 ");
            TimeAndSales ts = new TextTimeAndSalesReader().Read(reader);
            reader.Close();

            Assert.AreEqual(1, ts.Count);
            TickData td = ts.TickAt(0);
            Assert.AreEqual("09:00:00", td.TimeAsString);
            Assert.AreEqual(100, td.Price);
            Assert.AreEqual(1000, td.Volume);
        }
        [Test]
        public void TextBasic2() {
            StringReader reader = new StringReader("9:00:00 100  1000 \n9:01:00 100 2000");
            TimeAndSales ts = new TextTimeAndSalesReader().Read(reader);
            reader.Close();

            Assert.AreEqual(2, ts.Count);
            Assert.AreEqual("09:00:00", ts.TickAt(0).TimeAsString);
            Assert.AreEqual("09:01:00", ts.TickAt(1).TimeAsString);
        }

        [Test]
        public void TextModifyTime() {
            StringReader reader = new StringReader("9:00 100  1000 \n9:00 100 2000\n9:00 100 3000");
            TimeAndSales ts = new TextTimeAndSalesReader().Read(reader);
            reader.Close();

            Assert.AreEqual(3, ts.Count);

            TickData td = ts.TickAt(0);
            Assert.AreEqual("09:00:00", td.TimeAsString);

            td = ts.TickAt(1);
            Assert.AreEqual("09:00:20", td.TimeAsString);

            td = ts.TickAt(2);
            Assert.AreEqual("09:00:40", td.TimeAsString);
        }
    }
#endif

}
