/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Evaluators/EvalTests.cs#6 $
 * $DateTime: 2007/10/25 12:56:12 $
 * 
 * Evaluator̃eXgP[X
 */
#if UNITTEST
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;

//using Bellagio.Data;
using Bellagio.Script;
using Bellagio.Values;

namespace Bellagio.Evaluators {
    //֐Ăяo܂܂ȂAlZƂ̊{Z
    [TestFixture]
    public class PrimitiveEvalTests : IParameterProvider {

        [Test]
        public void Literal1() {
            StandAloneEvaluator ev = CreateEvaluator("10");
            Assert.IsTrue(ev.Content is LiteralEvaluator);
            Assert.IsTrue(ev.BT==BT.Int);

            BV value = ev.Eval(new EvalContext());
            Assert.IsTrue(value is BInt);
            Assert.AreEqual(10, ((BInt)value).Value);
        }
        [Test]
        public void Literal2() {
            StandAloneEvaluator ev = CreateEvaluator("1.0");
            Assert.IsTrue(ev.Content is LiteralEvaluator);
            Assert.IsTrue(ev.BT==BT.Double);

            BV value = ev.Eval(new EvalContext());
            Assert.IsTrue(value is BDouble);
            Assert.AreEqual(1.0, ((BDouble)value).Value);
        }
        [Test]
        public void Literal3() {
            StandAloneEvaluator ev = CreateEvaluator("true");
            Assert.IsTrue(ev.Content is LiteralEvaluator);
            Assert.IsTrue(ev.BT==BT.Bool);

            BV value = ev.Eval(new EvalContext());
            Assert.IsTrue(value is BBoolean);
            Assert.IsTrue(((BBoolean)value).Value);
        }
        [Test]
        public void Literal4() {
            StandAloneEvaluator ev = CreateEvaluator("@t900");
            Assert.IsTrue(ev.Content is LiteralEvaluator);
            Assert.IsTrue(ev.BT==BT.Time);

            BV value = ev.Eval(new EvalContext());
            Assert.IsTrue(value is BTime);
            Assert.AreEqual(900, ((BTime)value).AsInt());
        }
        [Test]
        public void Literal5() {
            StandAloneEvaluator ev = CreateEvaluator("@d1231");
            Assert.IsTrue(ev.Content is LiteralEvaluator);
            Assert.IsTrue(ev.BT==BT.Date);

            BV value = ev.Eval(new EvalContext());
            Assert.IsTrue(value is BDate);
            Assert.AreEqual(1231, ((BDate)value).AsInt());
        }
        [Test]
        public void Param1() {
            StandAloneEvaluator ev = CreateEvaluator("$i10");
            Assert.IsTrue(ev.Content is ParameterEvaluator);
            Assert.IsTrue(ev.BT==BT.Int);

            BV value = ev.Eval(new EvalContext());
            Assert.IsTrue(value is BInt);
            Assert.AreEqual(10, ((BInt)value).Value);
        }
        [Test]
        public void Param2() {
            StandAloneEvaluator ev = CreateEvaluator("$d10");
            Assert.IsTrue(ev.Content is ParameterEvaluator);
            Assert.IsTrue(ev.BT==BT.Double);

            BV value = ev.Eval(new EvalContext());
            Assert.IsTrue(value is BDouble);
            Assert.AreEqual(10.0, ((BDouble)value).Value);
        }

        [Test]
        public void Unary() {
            StandAloneEvaluator ev1 = CreateEvaluator("-(3+5)");
            BV value = ev1.Eval(new EvalContext());
            Assert.IsTrue(value is BInt);
            Assert.AreEqual(-8, ((BInt)value).Value);

            StandAloneEvaluator ev2 = CreateEvaluator("!(0 < 1)");
            value = ev2.Eval(new EvalContext());
            Assert.IsTrue(value is BBoolean);
            Assert.IsFalse(((BBoolean)value).Value);
        }

        [Test]
        public void BinOp1() {
            StandAloneEvaluator ev1 = CreateEvaluator("3+5");
            BV value = ev1.Eval(new EvalContext());
            Assert.IsTrue(value is BInt);
            Assert.AreEqual(8, ((BInt)value).Value);

            StandAloneEvaluator ev2 = CreateEvaluator("3-5");
            BInt v = (BInt)ev2.Eval(new EvalContext());
            Assert.AreEqual(-2, v.Value);

            StandAloneEvaluator ev3 = CreateEvaluator("3*5");
            v = (BInt)ev3.Eval(new EvalContext());
            Assert.AreEqual(15, v.Value);
            
            StandAloneEvaluator ev4 = CreateEvaluator("3/5");
            v = (BInt)ev4.Eval(new EvalContext());
            Assert.AreEqual(0, v.Value);

            StandAloneEvaluator ev5 = CreateEvaluator("3%5");
            v = (BInt)ev5.Eval(new EvalContext());
            Assert.AreEqual(3, v.Value);
        }
        [Test]
        public void BinOp2() {
            StandAloneEvaluator ev1 = CreateEvaluator("5.0 * 2.0");
            BV value = ev1.Eval(new EvalContext());
            Assert.IsTrue(value is BDouble);
            Assert.AreEqual(10.0, ((BDouble)value).Value);

            StandAloneEvaluator ev2 = CreateEvaluator("5.0 + 2.0");
            BDouble v = (BDouble)ev2.Eval(new EvalContext());
            Assert.AreEqual(7.0, v.Value);

            StandAloneEvaluator ev3 = CreateEvaluator("5.0 - 2.0");
            v = (BDouble)ev3.Eval(new EvalContext());
            Assert.AreEqual(3.0, v.Value);

            StandAloneEvaluator ev4 = CreateEvaluator("5.0 / 2.0");
            v = (BDouble)ev4.Eval(new EvalContext());
            Assert.AreEqual(2.5, v.Value);
        }
        [Test]
        public void BinOp3() {
            StandAloneEvaluator ev1 = CreateEvaluator("10 == 8");
            BBoolean value = (BBoolean)ev1.Eval(new EvalContext());
            Assert.IsFalse(value.Value);

            StandAloneEvaluator ev2 = CreateEvaluator("10 != 8");
            value = (BBoolean)ev2.Eval(new EvalContext());
            Assert.IsTrue(value.Value);

            StandAloneEvaluator ev3 = CreateEvaluator("10 > 8");
            value = (BBoolean)ev3.Eval(new EvalContext());
            Assert.IsTrue(value.Value);

            StandAloneEvaluator ev4 = CreateEvaluator("10 < 8");
            value = (BBoolean)ev4.Eval(new EvalContext());
            Assert.IsFalse(value.Value);

            StandAloneEvaluator ev5 = CreateEvaluator("10 >= 8");
            value = (BBoolean)ev5.Eval(new EvalContext());
            Assert.IsTrue(value.Value);

            StandAloneEvaluator ev6 = CreateEvaluator("10 <= 8");
            value = (BBoolean)ev6.Eval(new EvalContext());
            Assert.IsFalse(value.Value);
        }
        [Test]
        public void BinOp4() {
            StandAloneEvaluator ev1 = CreateEvaluator("10.0 == 8.0");
            BBoolean value = (BBoolean)ev1.Eval(new EvalContext());
            Assert.IsFalse(value.Value);

            StandAloneEvaluator ev2 = CreateEvaluator("10.0 != 8.0");
            value = (BBoolean)ev2.Eval(new EvalContext());
            Assert.IsTrue(value.Value);

            StandAloneEvaluator ev3 = CreateEvaluator("10.0 > 8.0");
            value = (BBoolean)ev3.Eval(new EvalContext());
            Assert.IsTrue(value.Value);

            StandAloneEvaluator ev4 = CreateEvaluator("10.0 < 8.0");
            value = (BBoolean)ev4.Eval(new EvalContext());
            Assert.IsFalse(value.Value);

            StandAloneEvaluator ev5 = CreateEvaluator("10.0 >= 8.0");
            value = (BBoolean)ev5.Eval(new EvalContext());
            Assert.IsTrue(value.Value);

            StandAloneEvaluator ev6 = CreateEvaluator("10.0 <= 8.0");
            value = (BBoolean)ev6.Eval(new EvalContext());
            Assert.IsFalse(value.Value);
        }
        [Test]
        public void BinOp5() {
            StandAloneEvaluator ev1 = CreateEvaluator("1 < 2 && 3 < 4");
            BBoolean value = (BBoolean)ev1.Eval(new EvalContext());
            Assert.IsTrue(value.Value);

            StandAloneEvaluator ev2 = CreateEvaluator("1 < 2 || 3 > 4");
            value = (BBoolean)ev2.Eval(new EvalContext());
            Assert.IsTrue(value.Value);
        }
        [Test]
        public void BinOp6() { //int->doubleϊ
            StandAloneEvaluator s1 = CreateEvaluator("1 < 2.0");
            BinaryOpEvaluator ev1 = s1.Content as BinaryOpEvaluator;
            Assert.IsTrue(ev1.LeftEvaluator is TypeConversionEvaluator<BInt,BDouble>);
            Assert.IsFalse(ev1.RightEvaluator is TypeConversionEvaluator<BInt, BDouble>);
            BBoolean value = (BBoolean)s1.Eval(new EvalContext());
            Assert.IsTrue(value.Value);

            StandAloneEvaluator s2 = CreateEvaluator("1.0 + 2");
            BinaryOpEvaluator ev2 = s2.Content as BinaryOpEvaluator;
            Assert.IsFalse(ev2.LeftEvaluator is TypeConversionEvaluator<BInt, BDouble>);
            Assert.IsTrue(ev2.RightEvaluator is TypeConversionEvaluator<BInt, BDouble>);
            BDouble value2 = (BDouble)s2.Eval(new EvalContext());
            Assert.AreEqual(3.0, value2.Value);
        }

        [Test]
        public void Condition() {
            StandAloneEvaluator ev = CreateEvaluator("1<2 ? 3 : 4");
            BV value = ev.Eval(new EvalContext());
            Assert.IsTrue(value is BInt);
            Assert.AreEqual(3, ((BInt)value).Value);

            ev = CreateEvaluator("1>2 ? 3 : 4");
            value = ev.Eval(new EvalContext());
            Assert.IsTrue(value is BInt);
            Assert.AreEqual(4, ((BInt)value).Value);
        }

        [Test]
        public void Complex1() {
            StandAloneEvaluator ev = CreateEvaluator("1>2 ? 3 : ($i10 - 1)");
            BV value = ev.Eval(new EvalContext());
            Assert.IsTrue(value is BInt);
            Assert.AreEqual(9, ((BInt)value).Value);

            //Eval̂тBV쐬ȂƂmF ȉComplexV[Yl
            int count = BV.instanceCount;
            ev.Eval(new EvalContext());
            Assert.AreEqual(count, BV.instanceCount);
        }
        [Test]
        public void Complex2() {
            StandAloneEvaluator ev = CreateEvaluator("(1+2*3+4*5)*10-70");
            BV value = ev.Eval(new EvalContext());
            Assert.IsTrue(value is BInt);
            Assert.AreEqual(200, ((BInt)value).Value);

            int count = BV.instanceCount;
            ev.Eval(new EvalContext());
            Assert.AreEqual(count, BV.instanceCount);
        }
        [Test]
        public void Complex3() {
            StandAloneEvaluator ev = CreateEvaluator("2-3-4"); //((2-3)-4)ɂȂ悤
            BV value = ev.Eval(new EvalContext());
            Assert.IsTrue(value is BInt);
            Assert.AreEqual(-5, ((BInt)value).Value);

            int count = BV.instanceCount;
            ev.Eval(new EvalContext());
            Assert.AreEqual(count, BV.instanceCount);
        }
        [Test]
        public void Inverse() {
            StandAloneEvaluator ev1 = CreateEvaluator("-(3+5)");
            BV value = ev1.Eval(new EvalContext());
            Assert.IsTrue(value is BInt);
            Assert.AreEqual(-8, ((BInt)value).Value);

            StandAloneEvaluator ev2 = CreateEvaluator("-(1.5/2.0)");
            value = ev2.Eval(new EvalContext());
            Assert.IsTrue(value is BDouble);
            Assert.AreEqual(-0.75, ((BDouble)value).Value);
        }

        private StandAloneEvaluator CreateEvaluator(string expression) {
            Expression expr = Expression.ParseExpression(expression);
            EvaluatorBuildContext ctx = new EvaluatorBuildContext(this);
            ctx.Initialize(null, LocalVariable.EmptyArray);
            EvaluatorBuilder bld = new EvaluatorBuilder(ctx);
            Evaluator ev = bld.Build(expr);
            return new StandAloneEvaluator(ev, ctx, BV.EmptyArray);
        }

        private class TransientParameter : INamedValue {
            private BV _value;
            private string _name;

            public TransientParameter(string name, BV value) {
                _name = name;
                _value = value;
            }

            public BT BT {
                get { return _value.BT; }
            }

            public BV GetValue(EvalContext ctx) {
                return _value;
            }

            public string Name {
                get { return _name; }
            }
        }




        INamedValue IParameterProvider.FindParameter(string name) {
            if(name=="i10")
                return new TransientParameter(name, new BInt(10));
            else if(name=="d10")
                return new TransientParameter(name, new BDouble(10));
            else
                return null;
        }

    }

}

#endif
