﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.Xml;
using System.IO;

using Bellagio.Environment;
using Bellagio.Values;
using Bellagio.Data;

using Travis.Collections;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.ExternalData {


    public class ExternalDataObjectRef : BV {
        private object _value;
        private ExternalDataObjectType _type;

        public ExternalDataObjectRef(ExternalDataObjectType type, object value) {
            _type = type;
            _value = value;
        }

        public object Value {
            get {
                return _value;
            }
            set {
                _value = value;
            }
        }

        public override BT BT {
            get { return _type; }
        }

        public override bool IsNil {
            get {
                return _value==null;
            }
        }
        public override void Format(BVFormatter formatter) {
            formatter.AppendDirect("external");
        }
        public override void Let(BV value) {
            _value = ((ExternalDataObjectRef)value)._value;
        }
    }

    public class ExternalDataObjectType : BT {
        public ExternalDataObjectType(string name)
            : base(name) {
            _nilInstance = new ExternalDataObjectRef(this, null);
        }
        public override BV CreateInstance() {
            return new ExternalDataObjectRef(this, null);
        }
    }

    public abstract class ExternalDataHostBase : IExternalDataHost {
        protected IExternalDataProvider _provider;
        protected ExternalDataObjectType _elementType; //関数登録で使う

        public ExternalDataHostBase(IExternalDataProvider provider) {
            _provider = provider;
            _elementType = new ExternalDataObjectType(provider.FunctionPostfix);
        }

        public abstract void Initialize();

        //状態
        public abstract int ElementCount { get; }
        public abstract BDate LastDataDate { get; }
        public string Description {
            get {
                return _provider.Description;
            }
        }
        //全体を書きだし
        public abstract void WriteToXmlDocument(TextWriter writer);

        public abstract void ReadFromXmlDocument(TextReader reader);

        public abstract void DeleteAll();
        public abstract void DeleteAfter(int date);

        public AbstractStockProfile FindStock(string code) {
            return BellagioRoot.GlobalStockCollection.FindExact(code);
        }
        public abstract void Sort();



        //アプリとして実行するときの環境
        public void LoadFromAppFile() {
            string filename = GetFileName();
            if(File.Exists(filename)) {
                using(TextReader r = new StreamReader(filename)) {
                    ReadFromXmlDocument(r);
                }
            }
        }
        public void SaveToAppFile() {
            using(TextWriter w = new StreamWriter(GetFileName())) {
                WriteToXmlDocument(w);
            }
        }

        private string GetFileName() {
            return BellagioRoot.PathInfo.DataHomeDir + _provider.ProviderId + ".xml";
        }

        //外部コマンド実行の動作
        public void ResetAll(int today) {
            DeleteAll();
            DailyUpdate(0, today);
        }
        public void NotifyLoadStatus(string msg) {
            //ロード中状況の通知　Pluginで共通してハンドルする
            ExternalDataPlugin.Instance.NotifyLoadStatus(this, msg);
        }
        public void DailyUpdate(int first_date, int last_date) {
            if(first_date==0) { //月単位で指定させる
                DateTime dt = DateTime.Now.AddMonths(-1 * _provider.AvailableMonthCount);
                first_date = BDate.DateTimeToInt(dt);
            }
            Debug.WriteLine(String.Format("DL {0} {1}-{2}", _provider.ProviderId, first_date, last_date));
            _provider.ImportDataUntil(first_date, last_date);
            Sort();
        }

        //関数登録実装
        void IExternalDataHost.RegisterElementFunction(string name, ExternalData_DoubleDelegate dg) {
            BellagioRoot.Functions.User.DefineInternalFunction(name, _elementType, BT.Double, new BT[0],
                delegate(BV target, EvalContext extra, BV[] args, BV result) {
                    ExternalDataObjectRef r = (ExternalDataObjectRef)target;
                    double raw = dg(r.Value);
                    if(Double.IsNaN(raw)) return ExecResult.Nil;

                    ((BDouble)result).Value = raw;
                    return ExecResult.OK;
                });
        }
        void IExternalDataHost.RegisterElementFunction(string name, ExternalData_BoolDelegate dg) {
            BellagioRoot.Functions.User.DefineInternalFunction(name, _elementType, BT.Bool, new BT[0],
                delegate(BV target, EvalContext extra, BV[] args, BV result) {
                    ExternalDataObjectRef r = (ExternalDataObjectRef)target;
                    bool raw = dg(r.Value);
                    ((BBoolean)result).Value = raw;
                    return ExecResult.OK;
                });
        }

        protected static XmlReader CreateXmlReader(TextReader reader) {
            XmlReaderSettings settings = new XmlReaderSettings();
            settings.IgnoreWhitespace = true;
            settings.IgnoreComments = true;
            return XmlReader.Create(reader, settings);
        }
    }

    public class DateBasedExternalDataHost : ExternalDataHostBase, IDateBasedExternalDataHost {
        public class Node {
            public BDate date;
            public object value;
            public Node(BDate d, object v) {
                date = d; value = v;
            }
        }
        private static Comparison<Node> nodeComparison = (Node n1, Node n2) => n1.date.AsInt().CompareTo(n2.date.AsInt());

        private List<Node> _data;
        private static BinarySearch<Node, int>.Comparator _dateFinder = 
            (Node node, int date) => node.date.AsInt().CompareTo(date);

        public DateBasedExternalDataHost(IExternalDataProvider provider) : base(provider) {
            _data = new List<Node>();
        }

        public override int ElementCount {
            get {
                return _data.Count;
            }
        }
        public override BDate LastDataDate {
            get {
                return _data.Count==0? null : _data[_data.Count-1].date;
            }
        }
        public Node GetNodeAt(int index) {
            return _data[index];
        }

        //もろもろ関数登録他
        public override void Initialize() {
            BellagioRoot.Functions.RegisterExternalType(_elementType);
            //検索系関数を登録
            BellagioRoot.Functions.User.DefineInternalFunction("get"+_provider.FunctionPostfix, null, _elementType, new BT[] { BT.Int }, new BInternalExecution(GetElementFunction));

            _provider.InitializeProvider(this);
        }
        private ExecResult GetElementFunction(BV target, EvalContext extra, BV[] args, BV result) {
            int index = BinarySearch<Node, int>.FindUpperBound(_data, ((BInt)args[0]).Value, _dateFinder);
            if(index < 0) return ExecResult.Nil;

            ExternalDataObjectRef r = (ExternalDataObjectRef)result;
            r.Value = _data[index].value;
            return ExecResult.OK;
        }


        public void AppendElement(int date, object value) {
            Debug.Assert(value.GetType()==_provider.ElementType);
            _data.Add(new Node(new BDate(date), value));
        }

        public override void DeleteAll() {
            _data.Clear();
        }

        public override void DeleteAfter(int date) {
            int index = BinarySearch<Node, int>.FindLowerBound(_data, date, _dateFinder);
            if(index >= 0)
                _data.RemoveRange(index, _data.Count-index);
        }
        public override void Sort() {
            _data.Sort(nodeComparison);
        }

        public override void WriteToXmlDocument(TextWriter writer) {
            string prid = _provider.ProviderId;
            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Indent = true;
            XmlWriter xw = XmlWriter.Create(writer, settings);

            xw.WriteStartDocument();
            xw.WriteStartElement("tactico-externalData");
            xw.WriteAttributeString("providerID", prid);
            foreach(Node node in _data) {
                xw.WriteStartElement(prid);
                xw.WriteAttributeString("date", node.date.AsInt().ToString());
                _provider.SerializeElement(xw, node.value);
                xw.WriteEndElement();
            }
            xw.WriteEndElement();
            xw.WriteEndDocument();
            xw.Close();
        }
        public override void ReadFromXmlDocument(TextReader reader) {
            string prid = _provider.ProviderId;

            XmlReader xr = CreateXmlReader(reader);

            xr.ReadStartElement("tactico-externalData");
            while(xr.NodeType==XmlNodeType.Element) {
                xr.MoveToAttribute("date");
                BDate date = new BDate(Int32.Parse(xr.Value));
                _data.Add(new Node(date, _provider.DeserializeElement(xr)));

                xr.Read();
                
            }
            if(xr.NodeType!=XmlNodeType.None)
                xr.ReadEndElement();
            xr.Close();
        }
    }

    public class DateAndStockBasedExternalDataHost : ExternalDataHostBase, IDateAndStockBasedExternalDataHost {

        public class Node {
            public BDate date;
            public Stock stock;
            public object value;
            public Node(BDate d, Stock s, object v) {
                date = d; stock = s;  value = v;
            }
        }
        private static Comparison<Node> nodeComparison = (Node n1, Node n2) => n1.date.AsInt().CompareTo(n2.date.AsInt());

        //指定期間内のオブジェクトを配列として返す、みたいなのがあるので日付順の配列として持つのがよさそうである
        private List<Node> _data;
        private static BinarySearch<Node, int>.Comparator _dateFinder = 
            (Node node, int date) => node.date.AsInt().CompareTo(date);

        public DateAndStockBasedExternalDataHost(IExternalDataProvider provider)
            : base(provider) {
            _data = new List<Node>();
        }

        public override int ElementCount {
            get {
                return _data.Count;
            }
        }
        public override BDate LastDataDate {
            get {
                return _data.Count==0? null : _data[_data.Count-1].date;
            }
        }
        public Node GetNodeAt(int index) {
            return _data[index];
        }

        //もろもろ関数登録他
        public override void Initialize() {
            BellagioRoot.Functions.RegisterExternalType(_elementType);
            //検索系関数を登録
            //第一引数の日付「まで」の第二引数期間の日数で、コンテキストのStockと一致するものを返す
            BellagioRoot.Functions.User.DefineInternalFunction("findRange"+_provider.FunctionPostfix, null, _elementType.ArrayType(), new BT[] { BT.Int, BT.Int }, new BInternalExecution(FindRangeFunction));

            _provider.InitializeProvider(this);
        }
        private ExecResult FindRangeFunction(BV target, EvalContext extra, BV[] args, BV result) {
            int enddate = ((BInt)args[0]).Value;
            int endindex = BinarySearch<Node, int>.FindUpperBound(_data, enddate, _dateFinder);
            if(endindex < 0) return ExecResult.Nil;

            //営業日ベースの計算ではない, 注意
            DateTime startdate = BDate.ToDateTime(enddate).AddDays(-(((BInt)args[1]).Value - 1));
            int startindex = BinarySearch<Node, int>.FindLowerBound(_data, BDate.DateTimeToInt(startdate), _dateFinder);
            if(startindex < 0 || startindex>endindex) return ExecResult.Nil;

            Debug.Assert(startindex <= endindex); //少なくとも長さ１の配列 (startindex/endindexはinclusive)
            BRegularArray arr = (BRegularArray)result;
            List<Node> result_objects = new List<Node>(); //Stockを見てのチェックが入る
            for(int i=startindex; i<=endindex;  i++) {
                Node node = _data[i];
                if(extra.CurrentStock==null || extra.CurrentStock.Stock==node.stock)  //CurrentStockがnullなら常に結果はカウントされる
                    result_objects.Add(node);
            }

            arr.Extend(result_objects.Count);
            for(int i=0; i<arr.Count; i++) {
                ExternalDataObjectRef r = (ExternalDataObjectRef)arr[i];
                r.Value = result_objects[i].value;
            }
            return ExecResult.OK;
        }

        public void AppendElement(int date, Stock stock, object value) {
            Debug.Assert(value.GetType()==_provider.ElementType);
            _data.Add(new Node(new BDate(date), stock, value));
        }


        public override void DeleteAll() {
            _data.Clear();
        }

        public override void DeleteAfter(int date) {
            int index = BinarySearch<Node, int>.FindLowerBound(_data, date, _dateFinder);
            if(index >= 0)
                _data.RemoveRange(index, _data.Count-index);
        }

        public override void Sort() {
            _data.Sort(nodeComparison);
        }

        public override void WriteToXmlDocument(TextWriter writer) {
            string prid = _provider.ProviderId;
            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Indent = true;
            settings.Encoding = Encoding.Default;
            XmlWriter xw = XmlWriter.Create(writer, settings);

            xw.WriteStartDocument();
            xw.WriteStartElement("tactico-externalData");
            xw.WriteAttributeString("providerID", prid);
            foreach(Node node in _data) {
                xw.WriteStartElement(prid);
                xw.WriteAttributeString("date", node.date.AsInt().ToString());
                xw.WriteAttributeString("stock", node.stock.Profile.Code);
                _provider.SerializeElement(xw, node.value);
                xw.WriteEndElement();
            }
            xw.WriteEndElement();
            xw.WriteEndDocument();
            xw.Close();
        }
        public override void ReadFromXmlDocument(TextReader reader) {
            string prid = _provider.ProviderId;
            
            XmlReader xr = CreateXmlReader(reader);

            xr.ReadStartElement("tactico-externalData");
            while(xr.NodeType==XmlNodeType.Element) {
                xr.MoveToAttribute("date");
                BDate date = new BDate(Int32.Parse(xr.Value));
                xr.MoveToAttribute("stock");
                AbstractStockProfile prof = BellagioRoot.GlobalStockCollection.FindExact(xr.Value);
                object value = _provider.DeserializeElement(xr);
                if(prof!=null) 
                    _data.Add(new Node(date, prof.Primary, value));

                xr.Read();
                
            }
            if(xr.NodeType!=XmlNodeType.None)
                xr.ReadEndElement();
            xr.Close();
        }


    }


#if UNITTEST
    [TestFixture]
    public class ExternalDataHostAndProviderTest {
        private DateBasedExternalDataHost _dateHost;
        private DateBasedTestProvider _dateProvider;
        private DateAndStockBasedExternalDataHost _stockHost;
        private DateAndStockBasedTestProvider _stockProvider;

        [TestFixtureSetUp]
        public void Setup() {
            BellagioEnvironmentParam p = new BellagioEnvironmentParam();
            p.SetupForUnitTest();
            p.UNITTEST_STOCK_COUNT = 3;

            BellagioRoot.Init(p);
            BellagioRoot.BootForTest();

            _dateProvider = new DateBasedTestProvider();
            _dateHost = new DateBasedExternalDataHost(_dateProvider);
            _dateHost.Initialize();

            _stockProvider = new DateAndStockBasedTestProvider();
            _stockHost = new DateAndStockBasedExternalDataHost(_stockProvider);
            _stockHost.Initialize();
        }
        private EvaluatorAndContext CreateEvaluator(string expression) {
            return EvaluatorAndContext.CreateEvaluator(expression);
        }

        [Test]
        public void DateBasedSerialize() { //基本のシリアライズ
            _dateHost.DeleteAll();
            _dateHost.AppendElement(20080601, new DateBasedTestProvider.Element(100, "abc"));
            _dateHost.AppendElement(20080604, new DateBasedTestProvider.Element(200, "def"));

            StringWriter wr = new StringWriter();
            _dateHost.WriteToXmlDocument(wr);

            string result = wr.ToString();

            _dateHost.DeleteAll();

            _dateHost.ReadFromXmlDocument(new StringReader(result));
            Assert.AreEqual(2, _dateHost.ElementCount);


            DateBasedExternalDataHost.Node n1 = _dateHost.GetNodeAt(0);
            DateBasedTestProvider.Element e1 = (DateBasedTestProvider.Element)n1.value;
            Assert.AreEqual(20080601, n1.date.AsInt());
            Assert.AreEqual(100, e1.intVal);
            Assert.AreEqual("abc", e1.strVal);
            DateBasedExternalDataHost.Node n2 = _dateHost.GetNodeAt(1);
            DateBasedTestProvider.Element e2 = (DateBasedTestProvider.Element)n2.value;
            Assert.AreEqual(20080604, n2.date.AsInt());
            Assert.AreEqual(200, e2.intVal);
            Assert.AreEqual("def", e2.strVal);
        }

        [Test]
        public void DateBasedFunction() { //関数登録系
            _dateHost.DeleteAll();
            _dateHost.AppendElement(20080601, new DateBasedTestProvider.Element(100, "abc"));
            _dateHost.AppendElement(20080604, new DateBasedTestProvider.Element(200, "def"));

            EvaluatorAndContext ev1 = CreateEvaluator("getTestProvider1(20080604)");
            BV r1 = (BV)ev1.Eval();
            Assert.IsTrue(!r1.IsNil);
            Assert.IsTrue(r1 is ExternalDataObjectRef);

            EvaluatorAndContext ev2 = CreateEvaluator("getTestProvider1(20080604).intVal()");
            BV r2 = ev2.Eval();
            Assert.AreEqual(200, ((BDouble)r2).Value);
            EvaluatorAndContext ev3 = CreateEvaluator("getTestProvider1(20080601).startsWithA()");
            BV r3 = ev3.Eval();
            Assert.IsTrue(((BBoolean)r3).Value);
            EvaluatorAndContext ev4 = CreateEvaluator("getTestProvider1(20080610).startsWithA()");
            BV r4 = ev4.Eval();
            Assert.IsTrue(r4.IsNil);
        }

        [Test]
        public void StockBasedSerialize() {
            _stockHost.DeleteAll();
            //銘柄があることを確認
            AbstractStockProfile p1 = BellagioRoot.GlobalStockCollection.FindExact("1");
            Assert.IsNotNull(p1);
            Stock s1 = p1.Primary;
            Stock s2 = BellagioRoot.GlobalStockCollection.FindExact("2").Primary;

            _stockHost.AppendElement(20080601, s2, new DateAndStockBasedTestProvider.Element(40));
            _stockHost.AppendElement(20080601, s1, new DateAndStockBasedTestProvider.Element(50));

            StringWriter wr = new StringWriter();
            _stockHost.WriteToXmlDocument(wr);

            string result = wr.ToString();
            _stockHost.DeleteAll();

            _stockHost.ReadFromXmlDocument(new StringReader(result));
            Assert.AreEqual(2, _stockHost.ElementCount);

            DateAndStockBasedExternalDataHost.Node n1 = _stockHost.GetNodeAt(0);
            DateAndStockBasedTestProvider.Element e1 = (DateAndStockBasedTestProvider.Element)n1.value;
            Assert.AreEqual(20080601, n1.date.AsInt());
            Assert.AreEqual(s2, n1.stock);
            Assert.AreEqual(40, e1.intVal);

            DateAndStockBasedExternalDataHost.Node n2 = _stockHost.GetNodeAt(1);
            DateAndStockBasedTestProvider.Element e2 = (DateAndStockBasedTestProvider.Element)n2.value;
            Assert.AreEqual(20080601, n2.date.AsInt());
            Assert.AreEqual(s1, n2.stock);
            Assert.AreEqual(50, e2.intVal);
        }

        [Test]
        public void StockBasedFunction() {
            Stock s1 = BellagioRoot.GlobalStockCollection.FindExact("1").Primary;
            Stock s2 = BellagioRoot.GlobalStockCollection.FindExact("2").Primary;

            _stockHost.DeleteAll();
            _stockHost.AppendElement(20080601, s1, new DateAndStockBasedTestProvider.Element(10));
            _stockHost.AppendElement(20080601, s2, new DateAndStockBasedTestProvider.Element(20));
            _stockHost.AppendElement(20080602, s1, new DateAndStockBasedTestProvider.Element(30));
            _stockHost.AppendElement(20080604, s2, new DateAndStockBasedTestProvider.Element(40));
            DateAndStockBasedTestProvider.Element last_elem = new DateAndStockBasedTestProvider.Element(50);
            _stockHost.AppendElement(20080605, s1, last_elem);

            EvaluatorAndContext ev1 = CreateEvaluator("findRangeTestProvider2(20080605, 1)");
            BV r1 = ev1.Eval();
            Assert.IsTrue(!r1.IsNil);
            Assert.IsTrue(r1 is BRegularArray);
            BRegularArray ar1 = (BRegularArray)r1;
            Assert.AreEqual(1, ar1.Count);
            ExternalDataObjectRef elem1 = (ExternalDataObjectRef)ar1[0];
            Assert.IsTrue(Object.ReferenceEquals(elem1.Value, last_elem));

            EvaluatorAndContext ev2 = CreateEvaluator("findRangeTestProvider2(20080605, 5)");
            ev2.contextStock = s1;
            BV r2 = ev2.Eval();
            Assert.AreEqual(3, ((BRegularArray)r2).Count);

            ev2.contextStock = s2;
            r2 = ev2.Eval();
            Assert.AreEqual(2, ((BRegularArray)r2).Count);
        }
    }


    public class DateBasedTestProvider : IExternalDataProvider {
        public class Element {
            public int intVal;
            public string strVal;
            public Element(int i, string s) {
                intVal = i;
                strVal = s;
            }
            public Element() { }
        }

        public Type ElementType {
            get { return typeof(Element); }
        }

        public ExternalDataKeyType KeyType {
            get { return ExternalDataKeyType.Date; }
        }
        public string Description {
            get {
                return "test";
            }
        }
        public string ProviderId {
            get { return "testProvider1"; }
        }
        public string FunctionPostfix {
            get { return "TestProvider1"; }
        }
        public int AvailableMonthCount {
            get {
                return 1;
            }
        }

        public void InitializeProvider(IExternalDataHost host) {
            host.RegisterElementFunction("intVal", (object value) => (double)((Element)value).intVal);
            host.RegisterElementFunction("startsWithA", (object value) => ((Element)value).strVal[0]=='a');
        }

        public void SerializeElement(XmlWriter writer, object value) {
            Element e = (Element)value;
            writer.WriteAttributeString("intVal", e.intVal.ToString());
            writer.WriteAttributeString("strVal", e.strVal);
        }

        public object DeserializeElement(XmlReader reader) {
            Element e = new Element();
            reader.MoveToAttribute("intVal");
            e.intVal = Int32.Parse(reader.Value);
            reader.MoveToAttribute("strVal");
            e.strVal = reader.Value;

            reader.MoveToElement();
            return e;
        }
        public void ImportDataUntil(int startdate, int enddate) {
        }
    }

    public class DateAndStockBasedTestProvider : IExternalDataProvider {
        public class Element {
            public int intVal;
            public Element(int i) {
                intVal = i;
            }
            public Element() { }
        }

        public Type ElementType {
            get { return typeof(Element); }
        }

        public int AvailableMonthCount {
            get {
                return 1;
            }
        }
        public ExternalDataKeyType KeyType {
            get { return ExternalDataKeyType.DateAndStock; }
        }
        public string Description {
            get {
                return "test";
            }
        }

        public string ProviderId {
            get { return "testProvider2"; }
        }

        public string FunctionPostfix {
            get { return "TestProvider2"; }
        }

        public void InitializeProvider(IExternalDataHost host) {
            host.RegisterElementFunction("intVal", (object value) => (double)((Element)value).intVal);
        }

        public void SerializeElement(XmlWriter writer, object value) {
            Element e = (Element)value;
            //もう片方とは異なりElementを使う形にする
            writer.WriteStartElement("E");
            writer.WriteValue(e.intVal);
            writer.WriteEndElement();
        }

        public object DeserializeElement(XmlReader reader) {
            Element e = new Element();
            reader.MoveToElement();

            reader.ReadStartElement();
            reader.Read(); //StartElement -> Text
            e.intVal = Int32.Parse(reader.Value);
            reader.Read(); //Text -> EndElement(E)
            reader.Read(); //EndElement -> next

            return e;
        }
        public void ImportDataUntil(int startdate, int enddate) {
        }

    }

#endif
}
