/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Ordering/OrderListControl.cs#14 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * X[YXN[AJX^`XgRg[BI[i[h[ListControlł撣ΎƂ͂ł邪O
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;

using Bellagio.Values;
using Bellagio.Forms;
using Bellagio.Drawing;
using Bellagio.Common;
using Bellagio.Environment;
using Bellagio.Data;
using Poderosa;

using Travis;
#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Ordering {
    public class TradeOrderListDrawingSettings : DrawingSettingsBase {
        public DSFont largeLabelFont;    //傫̃tHg
        public DSFont labelFont;         //ʏtHg
        public DSFont numberFont;        //ltHg

        public DSPenStyle frameColor;    //g̐F
        public DSBrush finalStatusColor; //ŏI(SAς݂Ȃ)̔wiF
        public DSPenStyle delimiterColor;//tB[h؂̐F
        public DSBrush labelColor;       //W̃eLXgF
        public DSBrush reserveColor;     //ۗ̃eLXgF
        public DSBrush buyColor;         //t̐F
        public DSBrush sellColor;        //t̐F
        public DSBrush genbutsuColor;    //̐F
        public DSBrush shinnyoColor;     //Mp̐F
        public DSBrush stockColor;       //̐F
        public DSBrush numberColor;      //l̐F

        public DSInt itemHeight;         //Ps̍
        public DSInt frameMargin;        //gƃeLXg̊Ԃ̗]
        public DSInt numberFontYOffset;  //lɈقȂtHggƂ̌ḧʒu킹

        public TradeOrderListDrawingSettings(string prefID) : base(prefID){
        }

        protected override void CreateElements() {

            itemHeight = DefinePositiveInt("itemHeight", 40);
            frameMargin = DefineInt("frameMargin", 4);

            largeLabelFont = DefineFont("largeLabelFont", DSFont.GenericFontName, 10);
            labelFont = DefineFont("labelFont", "lr SVbN", 9);
            numberFont = DefineFont("numberFont", DSFont.GenericFontName, 11);
            numberFontYOffset = DefineInt("numberFontYOffset", -1);

            frameColor = DefinePenStyle("frameColor", Color.DarkOliveGreen, BPenStyle.DashStyle.Dot, 1);
            finalStatusColor = DefineBrush("finalStatusColor", Color.FromArgb(32, 32, 32));
            delimiterColor = DefinePenStyle("delimiterColor", Color.DarkOliveGreen, BPenStyle.DashStyle.Dot, 1);
            labelColor = DefineBrush("labelColor", Color.White);
            buyColor = DefineBrush("buyColor", Color.OrangeRed);
            sellColor = DefineBrush("sellColor", Color.LightBlue);
            genbutsuColor = DefineBrush("genbutsuColor", Color.FromArgb(0xe0, 0xff, 0xff));
            shinnyoColor = DefineBrush("shinnyoColor", Color.FromArgb(0xfe, 0xff, 0xe0));
            stockColor = DefineBrush("stockColor", Color.Honeydew);
            numberColor = DefineBrush("numberColor", Color.LightCyan);
            reserveColor = DefineBrush("reserveColor", Color.Orange);

        }
    }

    //ꗗ̃Rg[@{̂ƃXN[o[S
    public class OrderListControl : BellagioListControl {
        private OrderListForm _parentForm;
        private ToolTip _toolTip;

        private bool _testMode;
        private TradeOrderList _items;
        private TradeOrderListDrawingSettings _settings;
        private IOrderItemFormatter _formatter;

        private ItemPosition _posIndex;
        private ItemPosition _posBuyOrSell;
        private ItemPosition _posStock;
        private ItemPosition _posMarket;
        private ItemPosition _posKouza;
        private ItemPosition _posVolume;
        private ItemPosition _posTypeAndCondition;
        private ItemPosition _posTrigger;
        private ItemPosition _posTime;
        private ItemPosition _posTradeAmount;
        private ItemPosition _posStatus;
        private ItemPosition _posID;
        private ItemPosition _posSendReserved;
        private ItemPosition _posOrder;

        private IconSetInfo _iconSetInfoForPreparing;
        private IconSetInfo _iconSetInfoForOrdered;

        //}EX̓ɑΉ铮̂߂̈ʒu ȂƂ-1
        private int _hoveringItemIndex;
        private int _hoveringIconIndex;
        private bool _hoveringMultipleTrade;

        //肪Ƃ̕\pc[`bv
        private ToolTip _multipleTradeInfoToolTip;
        private static readonly Size MULTIPLETRADE_ICON_SIZE = new Size(16, 16);

        public OrderListControl() {
            Init(null);
        }
        internal OrderListControl(OrderListForm f) {
            Init(f);
        }
        private void Init(OrderListForm f) {
            _parentForm = f;
            this.BackColor = Color.Black;
            _iconSetInfoForPreparing = CreateIconSetForPreparingOrderItem();
            _iconSetInfoForOrdered = CreateIconSetForOrderedOrderItem();
            _hoveringItemIndex = -1;
            _hoveringItemIndex = -1;

            _multipleTradeInfoToolTip = new ToolTip();
            _multipleTradeInfoToolTip.InitialDelay = 1;

            ReloadPreference();
            _toolTip = new ToolTip();
            CreateItemPositions();
        }

        public TradeOrderListDrawingSettings Settings {
            get {
                return _settings;
            }
            set {
                _settings = value;
            }
        }
        public bool TestMode {
            get {
                return _testMode;
            }
            set {
                _testMode = value;
            }
        }
        public IOrderItemFormatter ItemFormatter {
            get {
                return _formatter;
            }
            set {
                _formatter = value;
            }
        }

        internal void SetContent(TradeOrderList value) {
            _items = value;
        }
        //\̃eXgpɃ_CNgɒl
        public void SetContentForTest(TradeOrderList value) {
            _items = value;
        }
        public override void ReloadPreference() {
            OrderingPreferences p = BellagioRoot.FixedPreferences.Ordering;
            _parentForm.Opacity = p.HalfTransparentView? 0.8 : 1.0;
            //͒cʕ\̌̂ōĕ`Kv
            if(this.Visible) Invalidate();
        }

        protected override void OnResize(EventArgs e) {
            base.OnResize(e);
            AdjustPositions(this.ContentWidth);
        }

        protected override void ResetInternalState() {
            _hoveringItemIndex = -1;
            _hoveringIconIndex = -1;
        }

        protected override BellagioDocumentStatus GetDocumentStatus() {
            return _testMode? BellagioDocumentStatus.Complete : _parentForm.Document.DocumentStatus;
        }

        protected override int ItemCount {
            get {
                return _items==null? 0 : _items.Count+1;
            }
        }

        protected override int ItemHeight {
            get {
                return _settings.itemHeight.V;
            }
        }

        protected override void DrawItem(int itemIndex, Graphics g, int y) {
            try {
                bool explanatory = itemIndex==0;
                TradeOrderItem item = explanatory? null : _items[itemIndex-1];
                int top = y + _settings.frameMargin.V;
                int bottom = y + _settings.itemHeight.V - _settings.frameMargin.V;
                int middle = (top + bottom) / 2; //s̒SBQƂĕ`悷邱Ƃ͂悭
                int twoline_y1 = (middle + top) / 2;
                int twoline_y2 = (middle + bottom) / 2; //Qs\̂Ƃ̊eC̒SYW
                int delim_top = top + 3;
                int delim_bottom = bottom - 3;
                int margin = 3;
                int x; //x,y̓[JϐƂđpByɂĂ͈j󂵂Ă̂Œ

                //g
                Pen delimiter_pen = _settings.delimiterColor.P;
                Pen frame_pen = _settings.frameColor.P;
                Rectangle frame = new Rectangle(2, y+2, base.ContentWidth-4, _settings.itemHeight.V-4);
                if(!explanatory && OrderingUtil.IsFinalStatus(item.OrderStatus, item.TransactionStatus))
                    RoundRectUtil.Fill(g, _settings.finalStatusColor.B, frame, 4);
                RoundRectUtil.Draw(g, frame_pen, frame, 4);

                //ԍ
                if(_posIndex.width > 0) {
                    if(explanatory)
                        DrawStringUtil.MiddleCenter(g, "", _settings.labelFont.F, _settings.labelColor.B, _posIndex.center, middle);
                    else
                        DrawStringUtil.MiddleCenter(g, FormatShortcutKey(itemIndex-1), _settings.labelFont.F, _settings.labelColor.B, _posIndex.center, middle);
                }

                //
                if(_posBuyOrSell.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posBuyOrSell.x, delim_top, delim_bottom);
                    if(explanatory)
                        DrawStringUtil.MiddleCenter(g, "", _settings.largeLabelFont.F, _settings.labelColor.B, _posBuyOrSell.center, middle);
                    else {
                        Brush br_bs = item.IsBuying? _settings.buyColor.B : _settings.sellColor.B;
                        if(item.DealType==OrderDealType.Derivatives) {
                            if(item.SpecialClose!=SpecialOrderClose.None) {
                                DrawStringUtil.MiddleCenter(g, _formatter.FormatTradeDescription(item.DealType, item.LongShort, item.NewOrClose), _settings.largeLabelFont.F, br_bs, _posBuyOrSell.center, twoline_y1);
                                DrawStringUtil.MiddleCenter(g, "SQ", _settings.largeLabelFont.F, br_bs, _posBuyOrSell.center, twoline_y2);
                            }
                            else
                                DrawStringUtil.MiddleCenter(g, _formatter.FormatTradeDescription(item.DealType, item.LongShort, item.NewOrClose), _settings.largeLabelFont.F, br_bs, _posBuyOrSell.center, middle);
                        }
                        else {
                            Brush br_cat;
                            if(item.DealType==OrderDealType.Genbutsu) br_cat = _settings.genbutsuColor.B;
                            else br_cat = _settings.shinnyoColor.B; 

                            DrawStringUtil.MiddleCenter(g, _formatter.FormatOrderDealTypeCategory(item.DealType), _settings.labelFont.F, br_cat, _posBuyOrSell.center, twoline_y1);
                            if(item.SpecialClose!=SpecialOrderClose.None) {
                                string sc = item.SpecialClose==SpecialOrderClose.Hakabu? "P" : item.SpecialClose==SpecialOrderClose.Genhiki? "" : "n";
                                DrawStringUtil.MiddleCenter(g, sc, _settings.largeLabelFont.F, br_bs, _posBuyOrSell.center, twoline_y2);
                            }
                            else
                                DrawStringUtil.MiddleCenter(g, _formatter.FormatTradeDescription(item.DealType, item.LongShort, item.NewOrClose), _settings.largeLabelFont.F, br_bs, _posBuyOrSell.center, twoline_y2);
                        }
                    }
                }

                //R[hƖO
                if(_posStock.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posStock.x, delim_top, delim_bottom);
                    if(explanatory)
                        DrawStringUtil.MiddleLeftWithClip(g, "", _settings.labelFont.F, _settings.labelColor.B, _posStock.x, middle, _posStock.width);
                    else {
                        DrawStringUtil.MiddleLeftWithClip(g, item.Stock.Profile.Code, _settings.labelFont.F, _settings.labelColor.B, _posStock.x, twoline_y1, _posStock.width);
                        string name = item.Stock.Profile.Name;
                        Font f = name.Length>5? _settings.labelFont.F : _settings.largeLabelFont.F;
                        DrawStringUtil.MiddleLeftWithClip(g, name, f, _settings.labelColor.B, _posStock.x, twoline_y2, _posStock.width);
                    }
                }

                //s/Mp敪
                if(_posMarket.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posMarket.x, delim_top, delim_bottom);
                    if(explanatory) {
                        DrawStringUtil.MiddleCenter(g, "s", _settings.labelFont.F, _settings.labelColor.B, _posMarket.center, twoline_y1);
                        DrawStringUtil.MiddleCenter(g, "Mp敪", _settings.labelFont.F, _settings.labelColor.B, _posMarket.center, twoline_y2);
                    }
                    else if(item.DealType==OrderDealType.Genbutsu) { //ł͕KvȂQsɂ킽
                        string[] m = item.MarketAsText;
                        if(m!=null) {
                            if(m.Length==1)
                                DrawStringUtil.MiddleCenter(g, m[0], _settings.labelFont.F, _settings.stockColor.B, _posMarket.center, twoline_y1);
                            else if(m.Length==2) {
                                DrawStringUtil.MiddleCenter(g, m[0], _settings.labelFont.F, _settings.labelColor.B, _posMarket.center, twoline_y1);
                                DrawStringUtil.MiddleCenter(g, m[1], _settings.labelFont.F, _settings.labelColor.B, _posMarket.center, twoline_y2);
                            }
                        }
                    }
                    else if(item.DealType==OrderDealType.Derivatives) {
                        DrawStringUtil.MiddleCenter(g, _formatter.FormatStockExchange(item.Stock.Market), _settings.labelFont.F, _settings.stockColor.B, _posMarket.center, twoline_y1);
                        DrawStringUtil.MiddleCenter(g, "--", _settings.labelFont.F, _settings.stockColor.B, _posMarket.center, twoline_y2);
                    }
                    else { //Mp
                        if(item.SpecialClose!=SpecialOrderClose.None) {
                            DrawStringUtil.MiddleCenter(g, "--", _settings.labelFont.F, _settings.stockColor.B, _posMarket.center, twoline_y1);
                            DrawStringUtil.MiddleCenter(g, _formatter.FormatOrderDealType(item.DealType), _settings.labelFont.F, _settings.stockColor.B, _posMarket.center, twoline_y2);
                        }
                        else {
                            DrawStringUtil.MiddleCenter(g, _formatter.FormatStockExchange(item.Stock.Market), _settings.labelFont.F, _settings.stockColor.B, _posMarket.center, twoline_y1);
                            DrawStringUtil.MiddleCenter(g, _formatter.FormatOrderDealType(item.DealType), _settings.labelFont.F, _settings.stockColor.B, _posMarket.center, twoline_y2);
                        }
                    }
                }

                //
                if(_posKouza.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posKouza.x, delim_top, delim_bottom);
                    if(explanatory) {
                        DrawStringUtil.MiddleCenter(g, "", _settings.labelFont.F, _settings.labelColor.B, _posKouza.center, middle);
                    }
                    else {
                        string kouza = item.Stock.Profile is DerivativeStockProfile? "--" : _formatter.FormatOrderKouza(item.Kouza);
                        DrawStringUtil.MiddleCenter(g, kouza, _settings.labelFont.F, _settings.stockColor.B, _posKouza.center, middle);
                    }
                }

                //ʁ@iʁAi萔
                if(_posVolume.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posVolume.x, delim_top, delim_bottom);
                    x = _posVolume.right - margin;
                    y = (explanatory || item.IsPartialTraded)? twoline_y1 : middle;
                    if(explanatory)
                        DrawStringUtil.MiddleRight(g, "", _settings.labelFont.F, _settings.labelColor.B, x, y);
                    else {
                        DrawStringUtil.MiddleRight(g, item.Stock.Profile.VolumeUnitString, _settings.labelFont.F, _settings.labelColor.B, ref x, y);
                        DrawStringUtil.MiddleRight(g, item.Volume.ToString("N0"), _settings.numberFont.F, _settings.numberColor.B, ref x, y + _settings.numberFontYOffset.V);
                    }

                    x = _posVolume.right - margin;
                    PartialTradeDisplay pd = BellagioRoot.FixedPreferences.Ordering.PartialTradeDisplay;
                    if(explanatory)
                        DrawStringUtil.MiddleRight(g, pd==PartialTradeDisplay.Traded? "(萔)" : "(萔)", _settings.labelFont.F, _settings.labelColor.B, x, twoline_y2);
                    else if(item.IsPartialTraded) {
                        DrawStringUtil.MiddleRight(g, item.Stock.Profile.VolumeUnitString+")", _settings.labelFont.F, _settings.labelColor.B, ref x, twoline_y2);
                        int partial_vol = pd==PartialTradeDisplay.Traded? item.TradedVolume : item.Volume - item.TradedVolume;
                        DrawStringUtil.MiddleRight(g, partial_vol.ToString("N0"), _settings.numberFont.F, _settings.numberColor.B, ref x, twoline_y2 + _settings.numberFontYOffset.V);
                        DrawStringUtil.MiddleRight(g, "(", _settings.labelFont.F, _settings.labelColor.B, ref x, twoline_y2);
                    }
                }

                ///ϖ
                if(_posTypeAndCondition.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posTypeAndCondition.x, delim_top, delim_bottom);
                    x = _posTypeAndCondition.right - margin;

                    if(explanatory) {
                        DrawStringUtil.MiddleRight(g, "", _settings.labelFont.F, _settings.labelColor.B, x, twoline_y1);
                        DrawStringUtil.MiddleRight(g, "ϖP", _settings.labelFont.F, _settings.labelColor.B, x, twoline_y2);
                    }
                    else {
                        if(item.SpecialClose==SpecialOrderClose.None) {
                            if(OrderingUtil.IsSashineRequired(item.Condition)) {
                                DrawStringUtil.MiddleRight(g, "~", _settings.labelFont.F, _settings.labelColor.B, ref x, twoline_y1);
                                DrawStringUtil.MiddleRight(g, FormatPrice(item.AdjustPriceI(item.Sashine)), _settings.numberFont.F, _settings.numberColor.B, ref x, twoline_y1 + _settings.numberFontYOffset.V);
                            }
                            if(item.Condition!=OrderCondition.Sashine) //̎wlł͕\Ȃ
                                DrawStringUtil.MiddleRight(g, _formatter.FormatOrderCondition(item.Condition), _settings.labelFont.F, _settings.labelColor.B, ref x, twoline_y1);
                        }

                        double price = item.ElementCount==0? 0 : item.AverageTradePrice;
                        if(price > 0) {
                            x = _posTypeAndCondition.right - margin;
                            //ACR
                            if(item.ElementCount > 1) {
                                Point imgpos = new Point(x - MULTIPLETRADE_ICON_SIZE.Width, twoline_y2 - MULTIPLETRADE_ICON_SIZE.Height/2);
                                bool hover = new Rectangle(imgpos, MULTIPLETRADE_ICON_SIZE)
                                    .Contains(this.PointToClient(Control.MousePosition));
                                g.DrawImage(BellagioRoot.CommonIcons.SmallCommandButtons.GetBitmap(hover? 2 : 1), imgpos);
                                x -= 20;
                            }

                            DrawStringUtil.MiddleRight(g, "~", _settings.labelFont.F, _settings.labelColor.B, ref x, twoline_y2);
                            DrawStringUtil.MiddleRight(g, FormatPriceAP(price, item.IsBuying), _settings.numberFont.F, _settings.numberColor.B, ref x, twoline_y2 + _settings.numberFontYOffset.V);
                        }
                    }
                }

                //^Cv
                if(_posTrigger.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posTrigger.x, delim_top, delim_bottom);
                    if(explanatory) {
                        DrawStringUtil.MiddleRight(g, "^Cv", _settings.labelFont.F, _settings.labelColor.B, _posTrigger.right, twoline_y1);
                        DrawStringUtil.MiddleRight(g, "twl", _settings.labelFont.F, _settings.labelColor.B, _posTrigger.right, twoline_y2);
                    }
                    else {
                        OrderTriggerType trigger = item.TriggerType;
                        if(trigger==OrderTriggerType.None)
                            DrawStringUtil.MiddleCenter(g, "ʏ", _settings.labelFont.F, _settings.labelColor.B, _posTrigger.center, middle);
                        else if(trigger==OrderTriggerType.Unsupported)
                            DrawStringUtil.MiddleCenter(g, "--", _settings.labelFont.F, _settings.labelColor.B, _posTrigger.center, middle);
                        else {
                            Debug.Assert(trigger==OrderTriggerType.UpperBreak || trigger==OrderTriggerType.LowerBreak);
                            DrawStringUtil.MiddleRight(g, "twl", _settings.labelFont.F, _settings.labelColor.B, _posTrigger.right, twoline_y1);
                            x = _posTrigger.right;
                            DrawStringUtil.MiddleRight(g, trigger==OrderTriggerType.UpperBreak? "~ȏ" : "~ȉ", _settings.labelFont.F, _settings.labelColor.B, ref x, twoline_y2);
                            DrawStringUtil.MiddleRight(g, FormatPrice(item.AdjustPriceI(item.TriggerPrice)), _settings.numberFont.F, _settings.numberColor.B, ref x, twoline_y2 + _settings.numberFontYOffset.V);

                        }
                    }
                }


                //E莞
                if(_posTime.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posTime.x, delim_top, delim_bottom);
                    if(explanatory) {
                        DrawStringUtil.MiddleCenter(g, "", _settings.labelFont.F, _settings.labelColor.B, _posTime.center, twoline_y1);
                        DrawStringUtil.MiddleCenter(g, "L", _settings.labelFont.F, _settings.labelColor.B, _posTime.center, twoline_y2);
                    }
                    else {
                        if(item.OrderTime!=null)
                            DrawStringUtil.MiddleLeftWithClip(g, FormatOrderDateTime(item.OrderDate, item.OrderTime), _settings.labelFont.F, _settings.labelColor.B, _posTime.x, twoline_y1, _posTime.width);

                        //
                        string kigen_string;
                        if(item.LimitDateValue==null)
                            kigen_string = item.LimitDate==OrderLimitDate.Today? "" : "T";
                        else
                            kigen_string = BDate.FormatMMDD(item.LimitDateValue) + "܂";
                        DrawStringUtil.MiddleLeftWithClip(g, kigen_string, _settings.labelFont.F, _settings.labelColor.B, _posTime.x, twoline_y2, _posTime.width);
                        /* 莞\R[h̎c[
                        if(item.ElementCount==0)
                            DrawingUtil.DrawStringVCenterLeft(g, "--:--", _settings.labelFont.F, _settings.labelC.B, _posTime.x, twoline_y2);
                        else if(item.ElementCount==1)
                            DrawingUtil.DrawStringVCenterLeft(g, FormatOrderDateTime(item.LastTradeItem.Date, item.LastTradeItem.Time), _settings.labelFont.F, _settings.labelC.B, _posTime.x, twoline_y2);
                        else
                            DrawingUtil.DrawStringVCenterLeft(g, "()", _settings.labelFont.F, _settings.labelC.B, _posTime.x, twoline_y2);
                        */
                    }
                }


                //
                if(_posTradeAmount.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posTradeAmount.x, delim_top, delim_bottom);

                    x = _posTradeAmount.right - margin;
                    if(explanatory)
                        DrawStringUtil.MiddleRight(g, "", _settings.labelFont.F, _settings.labelColor.B, x, middle);
                    else if(item.LastTradeItem!=null) {
                        DrawStringUtil.MiddleRight(g, "~", _settings.labelFont.F, _settings.labelColor.B, ref x, middle);
                        DrawStringUtil.MiddleRight(g, item.TotalTradeAmount.ToString("N0"), _settings.numberFont.F, _settings.numberColor.B, ref x, middle + _settings.numberFontYOffset.V);
                    }
                }

                //
                if(_posStatus.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posStatus.x, delim_top, delim_bottom);

                    if(explanatory)
                        DrawStringUtil.MiddleCenter(g, "", _settings.labelFont.F, _settings.labelColor.B, _posStatus.center, middle);
                    else {
                        //sRƂ͂gĂQs\
                        string detailed_message = item.OrderStatusMessage;
                        y = String.IsNullOrEmpty(detailed_message)? middle : twoline_y1;
                        Brush br = item.OrderStatus==OrderStatus.Preparing? _settings.reserveColor.B : _settings.labelColor.B;
                        DrawStringUtil.MiddleCenter(g, _formatter.FormatOrderStatus(item.OrderStatus, item.TransactionStatus), _settings.labelFont.F, br, _posStatus.center, y);
                        if(!String.IsNullOrEmpty(detailed_message))
                            DrawStringUtil.MiddleCenter(g, detailed_message, _settings.labelFont.F, _settings.labelColor.B, _posStatus.center, twoline_y2);
                    }
                }

                //ID
                if(_posID.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posID.x, delim_top, delim_bottom);
                    if(explanatory)
                        DrawStringUtil.MiddleCenter(g, "ԍ", _settings.labelFont.F, _settings.labelColor.B, _posID.center, middle);
                    else
                        DrawStringUtil.MiddleCenter(g, item.UniqueStringEx, _settings.labelFont.F, _settings.labelColor.B, _posID.center, middle); //\p
                }

                //ۗ̔
                if(_posSendReserved.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posSendReserved.x, delim_top, delim_bottom);
                    if(explanatory)
                        DrawStringUtil.MiddleCenter(g, "", _settings.labelFont.F, _settings.labelColor.B, _posSendReserved.center, middle);
                    else if(item.OrderStatus==OrderStatus.Preparing)
                        _iconSetInfoForPreparing.Draw(g, _posSendReserved.center, middle);
                }

                //E{^
                if(_posOrder.width > 0) {
                    DrawDelimiter(g, delimiter_pen, _posOrder.x, delim_top, delim_bottom);
                    if(explanatory)
                        DrawStringUtil.MiddleCenter(g, "ύX/", _settings.labelFont.F, _settings.labelColor.B, _posOrder.center, middle);
                    else if(item.OrderRestriction!=OrderRestriction.Locked) { //PTS񂾒͕ύXE͂łȂ
                        OrderStatus status = item.OrderStatus;
                        if(OrderingUtil.CanCancellableStatus(status, item.TransactionStatus))
                            _iconSetInfoForOrdered.Draw(g, _posOrder.center, middle);
                    }
                }

            }
            catch(Exception ex) {
                Debug.WriteLine(ex.Message);
                Debug.WriteLine(ex.StackTrace);
            }
        }

        protected override void OnMouseMove(MouseEventArgs e) {
            base.OnMouseMove(e);
            ItemAndPoint pt = base.GetPointInItem(e.X, e.Y);
            int new_item_index = pt.itemIndex;
            int new_icon_index = -1; 
            IconSetInfo iconSet = null;
            bool new_multiple_trade = false;

            if(new_item_index > 0) {
                TradeOrderItem item = _items[new_item_index-1];
                if(item.OrderStatus==OrderStatus.Preparing && _posSendReserved.width>0) {
                    iconSet = _iconSetInfoForPreparing;
                    new_icon_index = iconSet.FindHoveringIndex(pt.x - _posSendReserved.center, pt.y - this.ItemHeight/2);
                }
                if(new_icon_index==-1 && OrderingUtil.CanCancellableStatus(item.OrderStatus, item.TransactionStatus) && item.OrderRestriction!=OrderRestriction.Locked && _posOrder.width>0) {
                    iconSet = _iconSetInfoForOrdered;
                    new_icon_index = iconSet.FindHoveringIndex(pt.x - _posOrder.center, pt.y - this.ItemHeight/2);
                }
                if(new_icon_index==-1 && item.ElementCount > 1 && _posTypeAndCondition.width > 0) {
                    Point imgpos = new Point(_posTypeAndCondition.right - MULTIPLETRADE_ICON_SIZE.Width - 4, this.ItemHeight/2 - MULTIPLETRADE_ICON_SIZE.Height/2);
                    new_multiple_trade = new Rectangle(imgpos, MULTIPLETRADE_ICON_SIZE).Contains(pt.x, pt.y);
                }
            }
            //Debug.WriteLine(String.Format("ei={0} icon={1}", new_hovering_index, new_icon_index));

            bool icon_change = _hoveringItemIndex!=new_item_index || _hoveringIconIndex!=new_icon_index;
            bool multitrade_change = _hoveringItemIndex!=new_item_index || _hoveringMultipleTrade!=new_multiple_trade;

            //ԕω
            if(icon_change) {
                this.Cursor = new_icon_index!=-1? Cursors.Hand : Cursors.Default;
                _hoveringItemIndex = new_item_index;
                _hoveringIconIndex = new_icon_index;
                
                //Tooltip
                if(new_icon_index < 0) //͈͊O܂͖}
                    _toolTip.SetToolTip(this, null);
                else
                    _toolTip.SetToolTip(this, iconSet.GetAt(new_icon_index).toolTipText);

            }

            if(multitrade_change) {
                if(new_multiple_trade) {
                    TradeOrderItem item = _items[new_item_index-1];
                    if(item.ElementCount > BellagioRoot.FixedPreferences.Ordering.MaxTradeElementForToolTip) { //Ƃ̏u
                        DetailedTradeList dtl = new DetailedTradeList();
                        dtl.Popup(this, item, this.PointToScreen(e.Location));
                    }
                    else {
                        _multipleTradeInfoToolTip.Active = true;
                        _multipleTradeInfoToolTip.Show(FormatMultipleTradeInfoText(item), this, e.Location, 3600000); //3600bȂ當Ȃ͂
                    }
                }
                else
                    _multipleTradeInfoToolTip.Active = false;

                this.Invalidate();
                _hoveringItemIndex = new_item_index;
                _hoveringMultipleTrade = new_multiple_trade;
            }
        }
        private string FormatMultipleTradeInfoText(TradeOrderItem item) {
            StringBuilder bld = new StringBuilder();
            string vu = item.Stock.Profile.VolumeUnitString;
            foreach(TradeOrderItem.Element e in item.Trades) {
                if(bld.Length > 0) bld.Append("\r\n");
                bld.Append(BTime.FormatHHMM(e.Time)).Append(" ");
                bld.Append(FormatPrice(item.AdjustPriceI(e.Price))).Append("~").Append(" ");
                bld.Append(e.Volume.ToString("N0")).Append(vu);
            }
            return bld.ToString();
        }

        protected override void OnMouseUp(MouseEventArgs e) {
            base.OnMouseUp(e);
            if(e.Button==MouseButtons.Left) {

                ItemAndPoint pt = base.GetPointInItem(e.X, e.Y);
                if(pt.itemIndex > 0) {
                    TradeOrderItem item = _items[pt.itemIndex-1];

#if DEBUG //ʒm\
                    if(Control.ModifierKeys==Keys.Control && item.ElementCount>0) {
                        OrderingPlugin.Instance.Commands.OpenTradeNotification(BellagioPlugin.Instance.WindowManager.ActiveWindow, item, item.LastTradeItem);
                        return;
                    }
#endif

                    //̂Ƃ͂ACR\Ȃ
                    if(item.OrderRestriction==OrderRestriction.Locked) return;

                    OrderStatus status = item.OrderStatus;
                    int icon_index;
                    //TODO IconPositionɃANV֘At悤Ƀf[^\ς
                    if(status==OrderStatus.Preparing) {
                        icon_index = _iconSetInfoForPreparing.FindHoveringIndex(pt.x - _posSendReserved.center, pt.y - this.ItemHeight/2);
                        if(icon_index==0) {
                            SendReservedOrder(item);
                            return;
                        }
                    }

                    if(OrderingUtil.CanCancellableStatus(status, item.TransactionStatus)) {
                        icon_index = _iconSetInfoForOrdered.FindHoveringIndex(pt.x - _posOrder.center, pt.y - this.ItemHeight/2);

                        if(icon_index>=0 && item.PTSOrder!=PTSOrder.None) {
                            BUtil.ShowWarningMessageBox("Ԏ͉̒TCgpB");
                            return;
                        }
                        if(icon_index>=0 && item.Stock.Profile.IsTransient) {
                            BUtil.ShowWarningMessageBox("I͉̒TCgpB");
                            return;
                        }

                        if(icon_index==0)
                            ShowChangeOrderForm(item);
                        else if(icon_index==1)
                            CancelOrder(item);
                    }

                }
            }
            else if(e.Button==MouseButtons.Right) {
                ShowVisibleConfigContextMenu();
            }

        }
        protected override void OnVisibleConfigChanged() {
            BellagioRoot.FixedPreferences.Ordering.OrderListHiddenColumns = FormatHiddenColumns();
            AdjustPositions(this.ContentWidth);
            _parentForm.AdjustWidthInfo();
            Invalidate();
        }

        public override void DoActionByNumericIndex(int index) {
            if(index>=0 && index<_items.Count) {
                TradeOrderItem item = _items[index];
                if(OrderingUtil.CanCancellableStatus(item.OrderStatus, item.TransactionStatus)) ShowChangeOrderForm(item);
            }
        }
        private void ShowChangeOrderForm(TradeOrderItem item) {
            OrderFormParameter p;
            if(item.OrderStatus==OrderStatus.Preparing) {
                if(item.NewOrClose==OrderNewClose.New)
                    p = OrderFormParameter.ForNewPosition(item);
                else
                    p = OrderFormParameter.ForClosePosition(item);
            }
            else {
                if(item.OrderRestriction==OrderRestriction.Locked || item.OrderRestriction==OrderRestriction.CancelOnly) {
                    BUtil.ShowWarningMessageBox("͕̒ύXł܂B");
                    return;
                }
                p = OrderFormParameter.ForChangeOrder(item);
            }

            OrderingPlugin.Instance.Commands.OpenOrderFormCommand(CastCommandTarget(), p);
        }
        private void CancelOrder(TradeOrderItem item) {
            try {
                if(item.OrderStatus==OrderStatus.Preparing) {
                    BellagioRoot.OrderDataProvider.RemoveReservedOrder(item);
                    if(item.Stock.Profile.IsDerivative)
                        BellagioRoot.OrderDataProvider.DerivativeOrder.CallEventDelegates();
                    else
                        BellagioRoot.OrderDataProvider.StockOrder.CallEventDelegates();
                }
                else {
                    string pinCode = null;
                    if(item.OrderRestriction==OrderRestriction.Locked) {
                        BUtil.ShowWarningMessageBox("͎̒ł܂B");
                        return;
                    }

                    if(BellagioRoot.FixedPreferences.Ordering.ShowOrderConfirmDialog || !BellagioRoot.FixedPreferences.PinCodeVerified) {
                        if(!BellagioRoot.FixedPreferences.Ordering.ShowOrderConfirmDialog && !BellagioRoot.FixedPreferences.PinCodeVerified)
                            BUtil.ShowWarningMessageBox("Ïؔԍ͂̂ߊmFʂ\܂B");
                        OrderConfirmDialog dlg = new OrderConfirmDialog("mF", item, true, "");
                        if(dlg.ShowDialog(_parentForm)==DialogResult.OK) {
                            pinCode = dlg.PinCode;
                            BellagioRoot.FixedPreferences.Ordering.ShowOrderConfirmDialog = dlg.ShowConfirmDialogChecked;
                        }
                        else
                            return;
                    }

                    if(pinCode==null && !GetPinCode(out pinCode)) return;

                    BellagioRoot.OrderDataProvider.SendCancelOrder(item, pinCode);
                    item.OrderStatus = OrderStatus.Cancelled;
                    item.TransactionStatus = OrderTransactionStatus.Lock;

                }
            }
            catch(IllegalOrderException ex) {
                BUtil.ShowWarningMessageBox(ex.Message);
            }
            catch(BellagioException ex) {
                RuntimeUtil.ReportException(ex);
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }

            this.Cursor = Cursors.Default;
            //G[ǂɂ炸ĕ`
            this.Invalidate();
        }
        //ۗ̔
        private void SendReservedOrder(TradeOrderItem item) {
            try {
                Debug.Assert(item.OrderStatus==OrderStatus.Preparing);
                string pinCode = null;
                if(BellagioRoot.FixedPreferences.Ordering.ShowOrderConfirmDialog || !BellagioRoot.FixedPreferences.PinCodeVerified) {
                    if(!BellagioRoot.FixedPreferences.Ordering.ShowOrderConfirmDialog && !BellagioRoot.FixedPreferences.PinCodeVerified)
                        BUtil.ShowWarningMessageBox("Ïؔԍ͂̂ߊmFʂ\܂B");
                    OrderConfirmDialog dlg = new OrderConfirmDialog("mF", item, true, "");
                    DialogResult r = dlg.ShowDialog(this);
                    pinCode = dlg.PinCode;
                    dlg.Dispose();
                    if(r!=DialogResult.OK) return;
                }

                if(pinCode==null && !GetPinCode(out pinCode)) return;

                this.Cursor = Cursors.WaitCursor;
                if(item.NewOrClose==OrderNewClose.Close && item.DealType!=OrderDealType.Genbutsu)
                    BellagioRoot.OrderDataProvider.SendSpecifiedElementCloseOrder(item, pinCode);
                else
                    BellagioRoot.OrderDataProvider.SendNewOrCloseOrder(item, pinCode);
                BellagioRoot.OrderDataProvider.RemoveReservedOrder(item); //MɐReservedOrder͏
            }
            catch(IllegalOrderException ex) {
                BUtil.ShowWarningMessageBox(ex.Message);
            }
            catch(BellagioException ex) {
                RuntimeUtil.ReportException(ex);
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
            this.Cursor = Cursors.Default;
            this.Invalidate();
        }

        //Ï؂̊l
        private bool GetPinCode(out string pinCode) {
            if(BellagioRoot.FixedPreferences.PinCodeVerified) {
                pinCode = BellagioRoot.FixedPreferences.SavedPinCode;
                return true;
            }

            pinCode = null;
            return false;
            
        }

        private void CreateItemPositions() {

            _posIndex = new ItemPosition("ԍ", 32, true, true);
            _posStock = new ItemPosition("", 120, true, true);
            _posStatus = new ItemPosition("", 80, true, true);
            _posOrder = new ItemPosition("ύX/", 72, true, true);
            _posSendReserved = new ItemPosition("", 36, true, true);
            _posBuyOrSell = new ItemPosition("", 60, true, true);  //܂ł̍v400MinimumSizeɊ֌W

            _posVolume = new ItemPosition("", 108, true, false);
            _posTypeAndCondition = new ItemPosition("/ϖP", 124, true, false);
            _posTrigger = new ItemPosition("^Cv/twl", 112, true, false);
            _posMarket = new ItemPosition("s/Mp敪", 60, true, false);
            _posKouza = new ItemPosition("", 52, true, false);
            _posTime = new ItemPosition("/L", 80, true, false);
            _posTradeAmount = new ItemPosition("", 150, false, false);
            _posID = new ItemPosition("ԍ", 48, false, false);

            //͕\
            _itemPositions = new ItemPosition[] {
                _posIndex, _posBuyOrSell,  _posStock, _posMarket, _posKouza, _posVolume,
                _posTypeAndCondition,  _posTrigger,  _posTime,  _posTradeAmount, 
                _posStatus, _posID, _posSendReserved, _posOrder
            };
            AdjustItemPositionVisibility(BellagioRoot.FixedPreferences.Ordering.OrderListHiddenColumns);
        }


        private void AdjustPositions(int width) {
            ItemPositionAlloc alloc = new ItemPositionAlloc();
            alloc.remain = width;
            SetFixedPositionWidth(_posIndex, ref alloc);
            SetFixedPositionWidth(_posStock, ref alloc);
            SetFixedPositionWidth(_posStatus, ref alloc);
            SetFixedPositionWidth(_posOrder, ref alloc);
            SetFixedPositionWidth(_posSendReserved, ref alloc);
            SetFixedPositionWidth(_posBuyOrSell, ref alloc);

            SetFixedPositionWidth(_posVolume, ref alloc);
            SetFixedPositionWidth(_posTypeAndCondition, ref alloc);
            SetFixedPositionWidth(_posTrigger, ref alloc); 
            SetFixedPositionWidth(_posMarket, ref alloc);
            SetFixedPositionWidth(_posKouza, ref alloc);
            SetFixedPositionWidth(_posTime, ref alloc);
            SetFixedPositionWidth(_posTradeAmount, ref alloc);
            SetFixedPositionWidth(_posID, ref alloc);
            if(alloc.remain > 0) AdjustRemainingWidth2(_posStock, _posTradeAmount, alloc.remain);

            //X𖄂߂
            _itemPositions[0].x = 0;
            for(int i=1; i<_itemPositions.Length; i++)
                _itemPositions[i].x = _itemPositions[i-1].right;

            //_omittedItemExists = !(_posID.width>0 && _posBuyOrSell.width>0 && _posMarketOrKouza.width>0 && _posStock.width>0 && _posTypeAndCondition.width>0 && _posVolume.width>0 && _posTradePrice.width>0 && _posStatus.width>0 && _posTime.width>0 && _posOrder.width>0);
            _omittedItemExists = alloc.omitted_flag;
        }

        private static IconSetInfo CreateIconSetForPreparingOrderItem() {
            IconList il = OrderingPlugin.OrderingIcons;
            int w = il.IconSize.Width / 2;
            IconSetInfo r = new IconSetInfo(il, new IconPosition[] { new IconPosition(9, -w, "") });
            return r;
        }
        private static IconSetInfo CreateIconSetForOrderedOrderItem() {
            IconList il = OrderingPlugin.OrderingIcons;
            int w = il.IconSize.Width / 2;
            IconSetInfo r = new IconSetInfo(il, new IconPosition[] { new IconPosition(10, w * -2 - 1, "ύX"), new IconPosition(11, 1, "") });
            return r;
        }


        //͐؂グA͐؂̂āB_ȉ͕KvȂƂ̂ݕ\
        private static string FormatPriceAP(double value, bool buying) {
            double x = value * 100;
            x = buying? Math.Ceiling(x) : Math.Floor(x);

            double ip = Math.Truncate(value); //
            int dp100 = (int)Math.Round(x - ip*100.0); //100{

            if(dp100==0)
                return ip.ToString("N0");
            else
                return String.Format("{0:N0}.{1:D2}", ip, dp100);
            
        }
        private static string FormatPrice(int value) {
            return value.ToString("N0");
        }

        private static string FormatOrderDateTime(BDate date, BTime time) {
            string date_string = null;
            if(date!=null && date.AsInt()!=BellagioRoot.DataSourceHost.GetCurrentOrderingDate()) { //݂̓tƈႤƂ̂ݓt\
                date_string = BDate.FormatMMDD(date);
            }

            string time_string = BTime.FormatHHMM(time);
            return date_string==null? time_string : date_string + " " + time_string;

        }
#if UNITTEST
        [TestFixture]
        public class Formatting {
            [Test]
            public void PriceAP() {
                Assert.AreEqual("100", FormatPriceAP(100.0, true));
                Assert.AreEqual("10,000", FormatPriceAP(10000.0, true));
                Assert.AreEqual("100.01", FormatPriceAP(100.005, true));
                Assert.AreEqual("100", FormatPriceAP(100.005, false));
                Assert.AreEqual("1,000.39", FormatPriceAP(1000.39, true));
                Assert.AreEqual("10,000.40", FormatPriceAP(10000.40, true));
            }
        }
#endif
    }
}
