﻿/*
 * Copyright (c) 2008  Lagarto Technology, Inc.
 * 
 * $Id$
 * $DateTime$
 * 
 */
#if UNITTEST
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.Windows.Forms;
using System.Reflection;
using System.ComponentModel;

using NUnit.Framework;

namespace Travis.ObjectPeepHole {
    //WinFormsの型縦断
    public enum EventHackFlags {
        Ancestor=1, //基底クラスに同名イベント。keyFieldは無効
        StaticKey=2,
        InstanceField=3
    }
    public class EventHack {
        public Type control;
        public EventInfo eventInfo;
        public FieldInfo keyField;
        public EventHackFlags flags;

        public EventHack(Type c, EventInfo e, FieldInfo f, EventHackFlags fl) {
            control = c;
            eventInfo = e;
            keyField = f;
            flags = fl;
        }
        public EventHack(Type c, EventInfo e, EventHackFlags fl) {
            control = c;
            eventInfo = e;
            keyField = null;
            flags = fl;
        }
    }

    public static class WinFormsEventResolver {

        private static List<Type> _importedTypes = new List<Type>();
        private static List<EventHack> _data = new List<EventHack>();

        private static List<Type> controlStyleTypes = CreateTypeList(
            typeof(Control),
            typeof(ButtonBase),
            typeof(Button),
            typeof(GroupBox),
            typeof(LinkLabel),
            typeof(MenuStrip),
            typeof(Panel),
            typeof(PropertyGrid),
            typeof(ProgressBar),
            typeof(SplitterPanel),
            typeof(ThreadExceptionDialog),
            typeof(StatusStrip),
            typeof(TabPage),
            typeof(TableLayoutPanel),
            typeof(ToolStrip),
            typeof(ToolStripContainer),
            typeof(ToolStripContentPanel),
            typeof(ToolStripPanel),
            typeof(ToolStripDropDown),
            typeof(UpDownBase)
            );

        private static List<Type> capitalStyleTypes = CreateTypeList(
            typeof(CheckBox),
            typeof(ComboBox),
            typeof(DataGrid),
            typeof(DataGridView),
            typeof(DateTimePicker),
            typeof(Form),
            typeof(Label),
            typeof(ListBox),
            typeof(ListControl),
            typeof(ListView),
            typeof(MaskedTextBox),
            typeof(PictureBox),
            typeof(PrintPreviewControl),
            typeof(PrintPreviewDialog),
            typeof(RadioButton),
            typeof(RichTextBox),
            typeof(ScrollBar),
            typeof(VScrollBar),
            typeof(HScrollBar),
            typeof(ScrollableControl),
            typeof(SplitContainer),
            typeof(Splitter),
            typeof(StatusBar),
            typeof(TabControl),
            typeof(TextBox),
            typeof(TextBoxBase),
            typeof(TrackBar),
            typeof(UserControl)
            );

        private static List<Type> normalStyleTypes = CreateTypeList(
            typeof(BindingNavigator),
            typeof(CheckedListBox),
            typeof(ContainerControl),
            typeof(DomainUpDown),
            typeof(MonthCalendar),
            typeof(NumericUpDown),
            typeof(ToolBar),
            typeof(TreeView),
            typeof(WebBrowser)
            );


        private static List<Type> unsupportedStyleTypes = CreateTypeList(
            typeof(AxHost),
            typeof(WebBrowserBase),
            GetNestedPrivateType(typeof(UpDownBase), "UpDownButtons"),
            typeof(System.Windows.Forms.Design.ComponentEditorForm),
            typeof(System.Windows.Forms.Design.ComponentEditorPage),
            typeof(System.Windows.Forms.Design.EventsTab),
            typeof(System.Windows.Forms.Design.PropertyTab)
            );


        public static void Init() {
            _importedTypes.Clear();
            _data.Clear();
        }
        public static EventHack Find(Type controltype, EventInfo eventInfo) {
            foreach(EventHack h in _data) {
                if(h.control==controltype && h.eventInfo.Name==eventInfo.Name) return h;
            }
            return null;
        }
        public static EventHack GetOrCreate(Type controltype, EventInfo eventInfo) {
            Debug.Assert(controltype!=null);
            Debug.Assert(eventInfo!=null);
            if(!_importedTypes.Contains(controltype)) AddTypeWithBases(controltype);

            EventHack h = Find(eventInfo.DeclaringType, eventInfo);
            if(h==null) 
                throw new ArgumentException(String.Format("{0} event is not found in {1}", eventInfo.Name, controltype.FullName));

            if(h.flags==EventHackFlags.Ancestor)
                return GetOrCreate(controltype.BaseType, controltype.BaseType.GetEvent(eventInfo.Name));

            return h;
        }
        private static void AddTypeWithBases(Type target) {
            Type b = target.BaseType;
            if(!_importedTypes.Contains(b) && b!=typeof(Component))
                AddTypeWithBases(b);
            AddType(target);
        }

        public static bool IsSupported(Type target) {
            return target==typeof(Control) || (target.IsSubclassOf(typeof(Control)) && !unsupportedStyleTypes.Contains(target));
        }
        public static void AddType(Type target) {
            if(target==typeof(Control) || target.IsSubclassOf(typeof(Control))) {
                EventInfo[] events = target.GetEvents(BindingFlags.Public|BindingFlags.DeclaredOnly|BindingFlags.Instance);

                if(controlStyleTypes.Contains(target))
                    ProcessControlStyleType(target, events);
                else if(unsupportedStyleTypes.Contains(target))
                    return;
                else if(capitalStyleTypes.Contains(target))
                    ProcessCapitalStyleType(target, events);
                else if(normalStyleTypes.Contains(target))
                    ProcessNormalStyleType(target, events);
                else { //unknown issue
                    foreach(EventInfo e in events) {
                        if(!HasAncestorEvent(target, e.Name))
                            Debug.WriteLine(String.Format("uncategorized class-{0}!{1}", target.Name, e.Name));
                    }
                }

                _importedTypes.Add(target);
            }
        }
        public static void AddTypes(Type[] controltypes) {
            foreach(Type t in controltypes)
                AddType(t);
        }

        //同名イベントが基底クラスにあるかを調べる
        private static bool HasAncestorEvent(Type target, string name) {
            target = target.BaseType;
            while(target!=typeof(object)) {
                if(target.GetEvent(name, BindingFlags.Public|BindingFlags.DeclaredOnly|BindingFlags.Instance)!=null)
                    return true;
                target = target.BaseType;
            }
            return false;
        }

        private static List<Type> CreateTypeList(params Type[] types) {
            List<Type> h = new List<Type>();
            foreach(Type t in types) h.Add(t);
            return h;
        }

        private static void ProcessControlStyleType(Type target, EventInfo[] events) {
            //Controlのように、Click -> EventClickのような変換でstaticフィールドをとるタイプ
            foreach(EventInfo e in events) {
                EventHack h = ResolveControlStyleType(target, e);
                if(h!=null)
                    _data.Add(h);
                else
                    Debug.WriteLine(String.Format("Unresolved event: {0} of {1}", e.Name, target.Name));
            }
        }
        private static EventHack ResolveControlStyleType(Type target, EventInfo e) {
            string name = e.Name;

            string[] fieldname;
            if(name.EndsWith("Changed"))
                fieldname = new string[] { "Event"+name.Substring(0, name.Length-7), "Event"+name }; //Changedは消えるものとそうでないのと。どっちかでとれればよいとする
            else
                fieldname = new string[] { "Event"+name };

            FieldInfo fi = null;
            for(int i=0; i<fieldname.Length; i++) {
                fi = target.GetField(fieldname[i], BindingFlags.Static|BindingFlags.NonPublic|BindingFlags.DeclaredOnly);
                if(fi!=null) break;
            }

            if(fi==null || fi.FieldType!=typeof(object)) {
                if(target!=typeof(Control) && HasAncestorEvent(target, name)) return new EventHack(target, e, EventHackFlags.Ancestor); //基底で同名があるなら救済
                EventHack h = ResolveNormalStyleType(target, e); //複合するクラスもある
                if(h!=null) return h;
            }

            return new EventHack(target, e, fi, EventHackFlags.StaticKey);
        }
        private static void ProcessCapitalStyleType(Type target, EventInfo[] events) {
            //CheckBoxのように、Click -> EVENT_CLICKのような変換でstaticフィールドをとるタイプ
            foreach(EventInfo e in events) {
                EventHack h = ResolveCapitalStyleType(target, e);
                if(h!=null)
                    _data.Add(h);
                else
                    Debug.WriteLine(String.Format("Unresolved event: {0} of {1}", e.Name, target.Name));
            }
        }
        private static EventHack ResolveCapitalStyleType(Type target, EventInfo e) {
            string name = e.Name;

            List<string> fieldname = new List<string>();
            string upper = name.ToUpper();
            fieldname.Add("EVENT_"+upper);
            if(target==typeof(DataGridView))
                fieldname.Add("EVENT_DATAGRIDVIEW"+upper);
            else if(name.EndsWith("Changed"))
                fieldname.Add("EVENT_"+upper.Substring(0, name.Length-7));

            //例外的なやつもいくつかあった
            if(target==typeof(DataGrid) && name=="ShowParentDetailsButtonClick")
                fieldname.Add("EVENT_DOWNBUTTONCLICK");
            if(target==typeof(Form)) {
                if(name=="MdiChildActivate")
                    fieldname.Add("EVENT_MDI_CHILD_ACTIVATE");
                else if(name=="InputLanguageChanged")
                    fieldname.Add("EVENT_INPUTLANGCHANGE");
                else if(name=="InputLanguageChanging")
                    fieldname.Add("EVENT_INPUTLANGCHANGEREQUEST");
            }
            if(target==typeof(MaskedTextBox) && name=="TypeValidationCompleted")
                fieldname.Add("EVENT_VALIDATIONCOMPLETED");
            if(target==typeof(PictureBox)) {
                if(name=="LoadCompleted")
                    fieldname.Add("loadCompletedKey");
                else if(name=="LoadProgressChanged")
                    fieldname.Add("loadProgressChangedKey");
            }
            if(target==typeof(RichTextBox)) {
                if(name=="ContentsResized")
                    fieldname.Add("EVENT_REQUESTRESIZE");
                else if(name=="LinkClicked")
                    fieldname.Add("EVENT_LINKACTIVATE");
                else if(name=="SelectionChanged")
                    fieldname.Add("EVENT_SELCHANGE");
            }
            if(target==typeof(SplitContainer) || target==typeof(Splitter)) {
                if(name=="SplitterMoving")
                    fieldname.Add("EVENT_MOVING");
                else if(name=="SplitterMoved")
                    fieldname.Add("EVENT_MOVED");
            }
            if(target==typeof(StatusBar)) {
                if(name=="DrawItem")
                    fieldname.Add("EVENT_SBDRAWITEM");
            }


            FieldInfo fi = null;
            for(int i=0; i<fieldname.Count; i++) {
                fi = target.GetField(fieldname[i], BindingFlags.Static|BindingFlags.NonPublic|BindingFlags.DeclaredOnly);
                if(fi!=null) break;
            }

            if(fi==null || fi.FieldType!=typeof(object)) {
                if(target!=typeof(Control) && HasAncestorEvent(target, name)) return new EventHack(target, e, EventHackFlags.Ancestor); //基底で同名があるなら救済

                EventHack h = ResolveNormalStyleType(target, e); //複合するクラスもある
                if(h!=null) return h;
            }

            return new EventHack(target, e, fi, EventHackFlags.StaticKey);
        }
        private static void ProcessNormalStyleType(Type target, EventInfo[] events) {
            //ふつうにpublic event ... と書いたときに準じるスタイル
            foreach(EventInfo e in events) {
                EventHack h = ResolveNormalStyleType(target, e);
                if(h!=null)
                    _data.Add(h);
                else
                    Debug.WriteLine(String.Format("Unresolved event: {0} of {1}", e.Name, target.Name));
            }
        }
        private static EventHack ResolveNormalStyleType(Type target, EventInfo e) {
            //イベントと同名で先頭を小文字にしたやつか、"on"をつけた名前を持ったDelegateの場合
            char[] name = e.Name.ToCharArray();
            name[0] = Char.ToLower(name[0]);
            string[] fieldname = new string[] { new string(name), "on"+e.Name, e.Name };

            FieldInfo fi = null;
            for(int i=0; i<fieldname.Length; i++) {
                fi = target.GetField(fieldname[i], BindingFlags.Instance|BindingFlags.NonPublic|BindingFlags.DeclaredOnly);
                if(fi!=null) break;
            }


            if(fi==null || !fi.FieldType.IsSubclassOf(typeof(Delegate))) {
                if(target!=typeof(Control) && HasAncestorEvent(target, e.Name)) return new EventHack(target, e, EventHackFlags.Ancestor); //基底で同名があるなら救済
                Debug.WriteLine(String.Format("Unknown Event(Normal) {0} in {1}", e.Name, target.Name));
                return null;
            }

            return new EventHack(target, e, fi, EventHackFlags.InstanceField);
        }

        private static Type GetNestedPrivateType(Type c, string name) {
            Type t = c.GetNestedType(name, BindingFlags.NonPublic|BindingFlags.Public);
            Debug.Assert(t!=null);
            return t;
        }
    }

    [TestFixture]
    public class EventHackTests {
        //ターゲットフィールドがあることをテスト。実際にイベントハンドラをくっつけるところまではやらない
        [Test]
        public void ExistenseTest() {
            Type[] all = typeof(Control).Assembly.GetTypes();
            foreach(Type target in all) {
                if(WinFormsEventResolver.IsSupported(target)) {
                    WinFormsEventResolver.Init();
                    WinFormsEventResolver.AddType(target);
                    foreach(EventInfo ev in target.GetEvents(BindingFlags.Public|BindingFlags.DeclaredOnly|BindingFlags.Instance)) {
                        EventHack h = WinFormsEventResolver.Find(target, ev);
                        string name = String.Format("{0} - {1}", ev.Name, target.FullName);
                        CheckEventHack(target, h, name);
                    }
                }
            }
        }
        private void CheckEventHack(Type target, EventHack h, string name) {
            Assert.IsNotNull(h, name);
            if(h.flags==EventHackFlags.Ancestor) return;

            if(h.flags==EventHackFlags.StaticKey) {
                Assert.IsNotNull(h.keyField, name);
                Assert.IsTrue(h.keyField.IsStatic && h.keyField.FieldType==typeof(object), name);
            }
            else {
                Assert.IsNotNull(h.keyField, name);
                Assert.IsTrue(!h.keyField.IsStatic && h.keyField.FieldType.IsSubclassOf(typeof(Delegate)), name);
            }
        }
    }

}
#endif
