/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Data/RealtimeItaProvider.cs#9 $
 * $DateTime: 2008/04/17 18:20:55 $
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;
using System.Threading;

using Poderosa;
using Poderosa.Util.Collections;
using Bellagio.Environment;

namespace Bellagio.Data {

    public interface IItaDataSource : IPrimaryStreamDataBase {
        AsyncOpResult OpenItaData(Stock stock, ref string message);
        void CloseItaData(Stock stock);
    }
    public interface IItaDataHost : IPrimaryStreamDataHostBase {

        void OnConnectFailed(Stock stock, string message);

        //case1 KvȉOnPartialItaDataĂŁAEndItaUpdateŃLbN
        void OnPartialItaData(Stock stock, AskBid askbid, int index, bool visible, int price, int volume, SashineRemark remark);
        void EndItaUpdate(Stock stock);

        //case2 
        void SetItaAll(Stock stock, Ita ask, Ita bid);
    }


    public class RealtimeItaProvider : StockDataProviderBaseT<RealtimeIta, RealtimeItaInitializer>, IItaDataHost {
        private IItaDataSource _dataSource;

        protected override RealtimeIta CreateValue(Stock stock) {
            return new RealtimeIta(stock, MarketUtil.ItaSize(stock));
        }

        protected override RealtimeItaInitializer CreateInitializer(RealtimeIta value) {
            return new RealtimeItaInitializer(value);
        }

        protected override AsyncOpResult OpenValue(RealtimeItaInitializer initializer, ref string message) {
            EnsureDataSource();
            AsyncOpResult r = _dataSource.OpenItaData(initializer.Stock, ref message);
            return r;
        }
        protected override void CloseValue(Stock stock, RealtimeIta value) {
            _dataSource.CloseItaData(stock);
        }

        void IItaDataHost.OnConnectFailed(Stock stock, string message) {
            base.FailLoading(stock, message);
        }


        void IItaDataHost.OnPartialItaData(Stock stock, AskBid askbid, int index, bool visible, int price, int volume, SashineRemark remark) {
            StockTag tag = base.GetTagOrNull(stock);
            Debug.Assert(tag!=null);

            RealtimeIta ita = tag.value;
            Debug.Assert(ita!=null);
            if(index >= ita.AskWork.Size) return;

            Sashine s = askbid==AskBid.Ask? ita.AskWork.SashineAt(index) : ita.BidWork.SashineAt(index);
            s.Visible = visible;
            s.Price = price;
            s.Volume = volume;
            s.Remark = remark;
        }
        void IItaDataHost.EndItaUpdate(Stock stock) {
            StockTag tag = base.GetTagOrNull(stock);
            Debug.Assert(tag!=null);
            if(tag.value.DataStatus==PrimaryDataStatus.Preparing) {
                tag.initializer.CompleteInitialization();
                base.CompleteLoading(stock);
            }
            else
                tag.value.FireDataUpdateEvent();
        }
        void IItaDataHost.SetItaAll(Stock stock, Ita ask, Ita bid) {
            StockTag tag = base.GetTagOrNull(stock);
            Debug.Assert(tag!=null);

            RealtimeIta ita = tag.value;
            Debug.Assert(ita!=null);
            int size = ita.AskWork.Size;
            for(int i=0; i<size; i++) {
                Sashine s = ask.SashineAt(i);
                ita.AskWork.SetDirect(i, s.Price, s.Volume, s.OrderCount, s.Remark, s.Visible);
                s = bid.SashineAt(i);
                ita.BidWork.SetDirect(i, s.Price, s.Volume, s.OrderCount, s.Remark, s.Visible);
            }
            ((IItaDataHost)this).EndItaUpdate(stock);
        }


        IAdaptable IAdaptable.GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }

        private void EnsureDataSource() {
            if(_dataSource!=null) return;
            _dataSource = BellagioRoot.DataSourceHost.ExternalEnvironment.ItaDataSource;
            _dataSource.Prepare(this);
        }
#if UNITTEST
        public void SetItaDataSource(IItaDataSource ds) {
            _dataSource = ds;
            if(ds!=null) ds.Prepare(this);
        }
#endif
    }
}
