/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Forms/CodeInputBox.cs#16 $
 * $DateTime: 2008/02/22 14:40:25 $
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Windows.Forms;
using System.Diagnostics;
using System.Runtime.InteropServices;

//P̃eXĝƂɂtrue
#if !CODEINTPUTBOX_TEST
using Bellagio.Data;
#endif

namespace Bellagio.Forms {

    public class CodeInputBox : RichTextBox {
        private Font _codeFont;
        private Font _nameFont;
        private Color _codeColor;
        private Color _nameColor;
        private string _prevText;
        private string _lastCorrectText; //ESCƂɃZbgl
        private ICodeDropDownDataProvider _dataProvider;
        private StringBuilder _adjustTextBuffer;
        private ICodeDropDownItem _currentItem;
        private static CodeDropDownWindow _popupWindow; //͑ŜłP̃CX^X

        public event EventHandler StockChanged;

        //R[ĥ݂͂Ă邩AÔ݂AR[h{OB
        //ꂪωƂɃtHgƐF̒N
        private enum TextStatus {
            Invalid,
            CodeOnly,
            NameOnly,
            Mixed
        }
        private TextStatus _textStatus;

        public CodeInputBox() {
            this.Multiline = false;
            _codeColor = Color.LimeGreen;
            this.ForeColor = _codeColor;
            _nameColor = SystemColors.Window;
            _textStatus = TextStatus.Invalid;
            _prevText = "";
            _adjustTextBuffer = new StringBuilder();
        }

        public override Font Font {
            get {
                return base.Font;
            }
            set {
                _codeFont = value;
                _nameFont = value;
                base.Font = value;
            }
        }

        public Font FontEx {
            get {
                return base.Font;
            }
            set {
                //tHg͓ɂȂƂӂƂƂɕ\ʒuɂČh邢
                //TODO N[΂Ō邢
                _codeFont = (Font)value.Clone();
                _nameFont = (Font)value.Clone();
            }
        }

        public Font CodeFont {
            get {
                return _codeFont;
            }
        }
        public Font NameFont {
            get {
                return _nameFont;
            }
        }
        public Color CodeColor {
            get {
                return _codeColor;
            }
            set {
                _codeColor = value;
            }
        }
        public Color NameColor {
            get {
                return _nameColor;
            }
            set {
                _nameColor = value;
            }
        }
        public string LastCorrectText {
            get {
                return _lastCorrectText;
            }
            set {
                _lastCorrectText = value;
            }
        }

        //ӂDisabledɂƐFiD
        public bool EnabledEx {
            get {
                return !this.ReadOnly;
            }
            set {
                this.ReadOnly = !value;
                this.BackColor = value? Color.Black : Color.FromArgb(64, 64, 64); //̐ݒς
            }
        }

        public ICodeDropDownItem CurrentItem {
            get {
                return _currentItem;
            }
        }
        //̃f[^񋟂C^tF[XB͐ݒ肵Ȃƃ_
        public ICodeDropDownDataProvider DataProvider {
            get {
                return _dataProvider;
            }
            set {
                _dataProvider = value;
            }
        }

        public string GetCodeText() {
            int name_start_pos;
            TextStatus st = TextToStatus(this.Text, out name_start_pos);
            if(st==TextStatus.CodeOnly)
                return this.Text;
            else if(st==TextStatus.Mixed)
                return this.Text.Substring(0, name_start_pos).Trim();
            else
                return "";
        }

        private bool _gotFocusGuard;
        protected override void OnGotFocus(EventArgs e) {
            base.OnGotFocus(e);
            if(_gotFocusGuard) return;

            this.SelectAll(); //IȂŃtH[JX^Ƃ
        }
        protected override void OnDoubleClick(EventArgs e) {
            base.OnDoubleClick(e);
            this.SelectAll();
        }

        protected override void Dispose(bool disposing) {
            base.Dispose(disposing);
            if(disposing) {
                _codeFont.Dispose();
                _nameFont.Dispose();
            }
        }

        private bool _textChangeGruard;
        protected override void OnTextChanged(EventArgs e) {
            base.OnTextChanged(e);
            if(_textChangeGruard) return;

            _textChangeGruard = true;
            bool changed = AdjustText();
            AdjustStyle(changed);

            //|bvAbṽ`FbN
            if(_popupWindow==null || !_popupWindow.Visible) {
                string t = this.Text;
                if(_textStatus==TextStatus.CodeOnly && _prevText.Length>=1 && t.Length>_prevText.Length) //R[ĥƂ͂Q͕KvB
                    StartPopup();
                else if(_textStatus==TextStatus.NameOnly && t.Length>=2) //ÔP
                    StartPopup();
                else  { //ȊÔƂ͊Sv̂邩
                    string p = TextToCode(_prevText);
                    string c = TextToCode(t);
                    if(c.Length > 0 && p!=c &&_prevText.Length-1!=t.Length) { //R[hAOƕωƂBobNXy[Xō폜ɂƂ̂ƂlAP󋵂͏
                        ICodeDropDownItem item = _dataProvider.Find(c); //Aɑ݂Ȃ玩m
                        if(item!=null)
                            ComplementText(item);
                    }
                }
            }

            _prevText = this.Text;
            _textChangeGruard = false;
        }

        //FςeLXgZbgȂ
        public void CommitStyle() {
            AdjustStyle(false);
            this.Invalidate();
        }

        protected override bool ProcessDialogKey(Keys keyData) {
            if(keyData==Keys.Escape && _lastCorrectText!=null) {
                this.SetTextExternal(_lastCorrectText);
                return true;
            }
            else if(keyData==Keys.F2) {
                this.SelectAll();
                return true;
            }
            else
                return base.ProcessDialogKey(keyData);
        }

        //OR[heLXgZbgꍇ
        public void SetTextExternal(string text) {
            if(this.IsDisposed) return;
            if(this.Text==text) return;

            _textChangeGruard = true;
            _textStatus = TextStatus.Invalid;
            this.Text = text;
            _prevText = text;
            AdjustStyle(false);
            this.SelectionStart = text.Length;
            this.SelectionLength = 0;
            _currentItem = null;
            _textChangeGruard = false;
        }

        public void SetFocusWithoutSelection() {
            _gotFocusGuard = true;
            this.Focus();
            _gotFocusGuard = false;
        }

        public void SelectAllEx() {
            this.Focus();
        }

        //eLXgBSpĂ悤
        private bool AdjustText() {
            string t = this.Text;
            int ss = this.SelectionStart;
            _adjustTextBuffer.Remove(0, _adjustTextBuffer.Length);

            bool changed = false;
            for(int i=0; i<t.Length; i++) {
                char ch = t[i];
                if(i<5 && CheckAcceptableZenkaku(ref ch)) { //ӓIAwł͖OɐlĂ邱ƂȂ̂ŕsvȕϊ
                    changed = true;
                    _adjustTextBuffer.Append(ch);
                }
                else
                    _adjustTextBuffer.Append(ch);
            }

            if(changed) {
                this.Text = _adjustTextBuffer.ToString();
                this.SelectionStart = ss;
            }

            return changed;
        }

        private void AdjustStyle(bool force_apply_color) {
            int name_start_pos;
            TextStatus st = TextToStatus(this.Text, out name_start_pos);
            //Debug.WriteLine("Text=" + this.Text + " St="+st.ToString() + " ns="+name_start_pos);
            if(!force_apply_color && _textStatus==st) return; //Ȃ牽Ȃ

            int ss = this.SelectionStart;
            if(st==TextStatus.CodeOnly) {
                this.Font = (Font)_codeFont.Clone();
                this.ForeColor = _codeColor;

                this.SelectAll();
                this.SelectionColor = _codeColor;
            }
            else if(st==TextStatus.NameOnly) {
                this.Font = (Font)_nameFont.Clone();
                this.ForeColor = _nameColor;

                this.SelectAll();
                this.SelectionColor = _nameColor;
            }
            else if(st==TextStatus.Mixed) {
                //this.Font = _nameFont;
                //this.ForeColor = _nameColor;

                this.SelectionStart = 0;
                this.SelectionLength = name_start_pos;
                this.SelectionColor = _codeColor;

                this.SelectionStart = name_start_pos;
                this.SelectionLength = this.Text.Length-name_start_pos;
                this.SelectionColor = _nameColor;
            }

            this.SelectionStart = ss;
            this.SelectionLength = 0;
            _textStatus = st;
        }

        private TextStatus TextToStatus(string src, out int name_start_pos) {
            name_start_pos = -1;

            int i = 0;
            while(i < src.Length) {
                char ch = src[i];
                if(!IsCodeChar(ch)) break; //ŏ̃R[h\ȂT
                i++;
            }

            
            if(i==src.Length)
                return TextStatus.CodeOnly; //src.Length==0̂Ƃ܂ނƂɒ
            else if(i==0)
                return TextStatus.NameOnly;
            else {
                name_start_pos = i;
                return TextStatus.Mixed;
            }
        }
        private string TextToCode(string src) {
            int ns;
            TextStatus st = TextToStatus(src, out ns);
            if(st==TextStatus.CodeOnly)
                return src;
            else if(st==TextStatus.Mixed)
                return src.Substring(0, ns);
            else
                return "";
        }

        private void StartPopup() {
            _popupWindow = new CodeDropDownWindow();
            _popupWindow.Popup(this, _textStatus==TextStatus.CodeOnly? CodeDropDownMethod.Code : CodeDropDownMethod.Substring, this.Text);
            //_popupWindow.Dispose();
        }

        public void NotifyEndPopup() {
            this.Focus();
            //Lbg𖖔
            this.SelectionStart = this.Text.Length;
            this.SelectionLength = 0;
        }
        public void ComplementText(ICodeDropDownItem item) {
            string text = item.Format();
            _lastCorrectText = text;
            SetTextExternal(text);
            //Lbg擪
            this.SelectionStart = 0;

            _currentItem = item;
            if(this.StockChanged!=null) StockChanged(this, new EventArgs());
        }
        public void PerformBackspace() {
            string t = this.Text;
            if(t.Length>0) {
                int ss = this.SelectionStart;
                if(ss==t.Length) ss--;
                this.Text = t.Substring(0, t.Length-1);
                this.SelectionStart = ss;
            }
        }

        //󂯓\ȑSpł΁AΉ锼pɏCtrueԂBłȂfalseԂB
        private static bool CheckAcceptableZenkaku(ref char ch) {
            if('O'<=ch && ch<='X') {
                ch = (char)('0' + (ch - 'O'));
                return true;
            }
            if('`'<=ch && ch<='y') {
                ch = (char)('A' + (ch - '`'));
                return true;
            }
            if(''<=ch && ch<='') {
                ch = (char)('a' + (ch - ''));
                return true;
            }

            return false;
        }

        private static bool IsCodeChar(char ch) {
            return ('0'<=ch && ch<='9') || ('A'<=ch && ch<='Z') || ('a'<=ch && ch<='z');
        }
    }

    public enum CodeDropDownMethod {
        Code, //R[h͂ɂ
        Substring //ɂ
    }
    public interface ICodeDropDownDataProvider {
        void FillCandidate(CodeDropDownMethod method, string hint, ICollection<ICodeDropDownItem> collection);
        ICodeDropDownItem Find(string hint);
    }
    public interface ICodeDropDownItem {
        string Code { get; }
        string Name { get; }
        string Format(); 
    }

    public class CodeDropDownWindow : ToolStripDropDown {
        public enum ComplementStatus {
            Hidden,
            Selecting,
            Cancelled,
            Complement,
            Exiting
        }

        private ComplementStatus _status;
        private CodeDropDownMethod _method;
        private CodeListBox _listBox;
        private CodeInputBox _parent;
        private string _hint;
        private StringBuilder _appendedText; //|bvAbvɓ͂ꂽ
        private List<ICodeDropDownItem> _items;

        public CodeDropDownWindow() {
            _appendedText = new StringBuilder();
            _items = new List<ICodeDropDownItem>();

            _listBox = new CodeListBox(this);

            ToolStripControlHost tch = new ToolStripControlHost(_listBox);
            this.Items.Add(tch);
            tch.Padding = Padding.Empty;
            this.Padding = Padding.Empty;
        }

        public CodeInputBox CurrentInputBox {
            get {
                return _parent;
            }
        }

        public bool Popup(CodeInputBox inputbox, CodeDropDownMethod method, string hint) {
            this.BackColor = inputbox.BackColor;
            _listBox.BackColor = this.BackColor;
            this.ForeColor = inputbox.ForeColor;
            _parent = inputbox;
            _method = method;
            _hint = hint.ToUpper();

            _items.Clear();
            ICodeDropDownDataProvider dp = inputbox.DataProvider;
            Debug.Assert(dp!=null);
            dp.FillCandidate(_method, hint, _items);
            if(_items.Count==0)
                return false; //ȂȂ疳
            else if(_items.Count==1) {
                _parent.ComplementText(_items[0]); //␔PȂ瑦
                return false;
            }

            InitListBox();
            
            _appendedText.Remove(0, _appendedText.Length);
            //TopLeveltrueƁAȂʂ̃CEBhEIɃANeBuɂȂƂۂB̂ƂActivateCxgȂ̂B
            //Ȃ̂łDropDowntH[̒ڂ̎qƂ邪ÂtH[܂菬ƂɉBĂ܂QB
            this.TopLevel = false;
            inputbox.TopLevelControl.Controls.Add(this);
            this.Show(inputbox.TopLevelControl.PointToClient(inputbox.PointToScreen(new Point(0, inputbox.Height))));
            _listBox.Focus();
            _status = ComplementStatus.Selecting;
            return true;
        }

        private void InitListBox() {
            this.SuspendLayout();
            Graphics g = this.CreateGraphics();
            SizeF cf1 = g.MeasureString("0", _parent.CodeFont);
            SizeF cf2 = g.MeasureString("00", _parent.CodeFont);
            g.Dispose();

            _listBox.SetAppearance((int)(cf1.Height+4), cf2.Width-cf1.Width);
            _listBox.Font = _parent.NameFont;
            _listBox.HorizontalExtent = _parent.Width;
            _listBox.MaximumSize = new Size(_parent.Width, 200); //ĂƗ]Ă܂Ƃ͂ȂȂ

            CollectItem(_hint);
            this.ResumeLayout(true);
        }
        private void CollectItem(string input) {
            _listBox.BeginUpdate();
            _listBox.Items.Clear();
            int selected_index_candidate = -1;
            foreach(ICodeDropDownItem item in _items) {
                bool hit = _method==CodeDropDownMethod.Code? item.Code.StartsWith(input) : item.Name.IndexOf(input)!=-1;
                if(hit) {
                    if(selected_index_candidate==-1) selected_index_candidate = _listBox.Items.Count;
                    _listBox.Items.Add(item);
                }
            }
            _listBox.SelectedIndex = selected_index_candidate;
            _listBox.EndUpdate();
        }

        //ListBox̃R[obNn
        public void Cancel() {
            if(_status==ComplementStatus.Exiting) return; //ExitClose()LostFocusȂǂoRčċAĂꍇ
            _status = ComplementStatus.Cancelled;
            Exit();
        }
        public void DoChar(char ch) {
            if(ch==0x08) { //baskspace
                if(_appendedText.Length>0) _appendedText.Remove(_appendedText.Length-1, 1);
                if(_appendedText.Length==0)
                    Cancel();
                else {
                    string head = _hint + _appendedText.ToString();
                    CollectItem(head);
                }
                _parent.PerformBackspace();
            }
            else if(0x20<=(int)ch) { //printableȂ
                _appendedText.Append(ch);
                string head = _hint + _appendedText.ToString();
                CollectItem(head);
                
                int count = _listBox.Items.Count;
                if(count==1)
                    Complement(); //₪PɂȂ猈
                else if(count==0)
                    Cancel();
                else
                    _parent.AppendText(new string(ch, 1));
            }

        }
        public void Complement() {
            _status = ComplementStatus.Complement;
            ICodeDropDownItem item = _listBox.Items[_listBox.SelectedIndex] as ICodeDropDownItem;
            _parent.ComplementText(item);
            Exit();
        }

        private void Exit() {
            ComplementStatus s = _status;
            _status = ComplementStatus.Exiting;
            this.Parent = null;
            this.Dispose();
            _parent.NotifyEndPopup();
            _status = s;
        }

        public class CodeListBox : ListBox {

            private CodeDropDownWindow _parent;
            private float _codeTextPitch; //R[h̕

            public CodeListBox(CodeDropDownWindow parent) {
                _parent = parent;
                this.IntegralHeight = false;
                this.DrawMode = DrawMode.OwnerDrawFixed;
            }
            public void SetAppearance(int item_height, float code_text_pitch) {
                this.ItemHeight = item_height;
                _codeTextPitch = code_text_pitch;
            }

            protected override bool ProcessCmdKey(ref Message msg, Keys keyData) {
                Keys key = keyData & Keys.KeyCode;

                switch(key) {
                    case Keys.Escape:
                        _parent.Cancel();
                        return base.ProcessCmdKey(ref msg, keyData);
                    case Keys.Tab:
                    case Keys.Enter:
                        if(this.SelectedIndex!=-1)
                            _parent.Complement();
                        return true;
                    case Keys.Left:
                    case Keys.Right:
                        //_parent.DoSpecialKey(Keys.None, key); //E͓nKvȂ
                        return true;
                    case Keys.Up:
                    case Keys.Down:
                    case Keys.PageUp:
                    case Keys.PageDown:
                        return base.ProcessCmdKey(ref msg, keyData); //Xg{bNXɏ
                    default:
                        return base.ProcessCmdKey(ref msg, keyData);
                }
            }

            protected override void OnDrawItem(DrawItemEventArgs e) {
                base.OnDrawItem(e);
                if(e.Index==-1) return;

                CodeInputBox c = _parent.CurrentInputBox;
                ICodeDropDownItem item = this.Items[e.Index] as ICodeDropDownItem;

                e.DrawBackground();
                bool selected = (e.State & DrawItemState.Selected)!=DrawItemState.None;

                Brush code_br = new SolidBrush(selected? SystemColors.HighlightText : c.CodeColor);
                Brush name_br = new SolidBrush(selected? SystemColors.HighlightText : c.NameColor);
                e.Graphics.DrawString(item.Code, c.CodeFont, code_br, e.Bounds);
                e.Graphics.DrawString(item.Name, c.NameFont, name_br, new Point(e.Bounds.Left+(int)(_codeTextPitch*(item.Code.Length+1)), e.Bounds.Top));
                code_br.Dispose();
                name_br.Dispose();
            }
            protected override void OnKeyPress(KeyPressEventArgs e) {
                _parent.DoChar(e.KeyChar);
                e.Handled = true;
                base.OnKeyPress(e);
            }

            protected override void OnDoubleClick(EventArgs e) {
                base.OnDoubleClick(e);
                if(this.SelectedIndex!=-1) _parent.Complement();
            }

            protected override void OnLostFocus(EventArgs e) {
                base.OnLostFocus(e);
                _parent.Cancel();
            }
        }
    }

#if !CODEINTPUTBOX_TEST
    public class CodeDropDownItemImpl : ICodeDropDownItem {
        private AbstractStockProfile _stock;
        public CodeDropDownItemImpl(AbstractStockProfile stock) {
            _stock = stock;
        }
        public string Code {
            get {
                return _stock.Code;
            }
        }

        public string Name {
            get {
                return _stock.Name;
            }
        }
        public string Format() {
            return CodeDropDownDataProvider.DefaultFormat(_stock);
        }
    }
    public class CodeDropDownDataProvider : ICodeDropDownDataProvider {
        private bool _forDaily; //n̑IړIǂ

        public bool ForDaily {
            get {
                return _forDaily;
            }
            set {
                _forDaily = value;
            }
        }

        public void FillCandidate(CodeDropDownMethod method, string hint, ICollection<ICodeDropDownItem> collection) {
            if(method==CodeDropDownMethod.Code) {
                hint = hint.ToUpper();
                FillByCodeOfHead(hint, collection);
            }
            else
                FillBySubstring(hint, collection);
        }
        private void FillByCodeOfHead(string head, ICollection<ICodeDropDownItem> collection) {
            StockProfileCollection sc = BellagioRoot.GlobalStockCollection;
            int begin = sc.SearchDictionary(head);
            if(begin==-1) return;

            char[] t = head.ToCharArray();
            t[t.Length-1]++; //Ō̈ꕶς
            int end = sc.SearchDictionary(new string(t));
            if(end==-1) end = sc.Count;

            for(int i=begin; i<end; i++) {
                AbstractStockProfile sp = sc.GetAt(i);
                if(IsCodeDropDownCandidate(sp))
                    collection.Add(new CodeDropDownItemImpl(sp));
            }
        }
        private void FillBySubstring(string hint, ICollection<ICodeDropDownItem> collection) {
            foreach(AbstractStockProfile s in BellagioRoot.GlobalStockCollection)
                if(IsCodeDropDownCandidate(s) && s.Name.IndexOf(hint)!=-1) collection.Add(new CodeDropDownItemImpl(s));
        }

        private bool IsCodeDropDownCandidate(AbstractStockProfile sp) {
            if((sp.Flags & StockProfileFlags.Hidden)!=StockProfileFlags.None) return false;

            if(sp is MarketIndex)
                return _forDaily; //w͓nł̂ݑIł
            else if(sp is BasicStockProfile) { //ʏ́Ap~Ȃnł̂ݑIł
                if(sp.Primary==null) return false; //{肦ȂA}X^̃oO

                if((sp.Primary.StockFlags & StockFlags.Obsolete)!=StockFlags.None)
                    return _forDaily;
                else
                    return true;
            }
            else {
                Debug.Assert(sp is DerivativeStockProfile);
                DerivativeStockProfile dp = sp as DerivativeStockProfile;
                return dp.ConcatinatedFuture? _forDaily : true; //SԘÂ͓݁AȊO͑S\BCujOZbVw肪ʂ̂Hiddenŏ̂OK
            }
        }

        public ICodeDropDownItem Find(string hint) {
            AbstractStockProfile sp = BellagioRoot.GlobalStockCollection.FindExact(hint);
            return sp!=null && IsCodeDropDownCandidate(sp)? new CodeDropDownItemImpl(sp) : null;
        }

        public static string DefaultFormat(AbstractStockProfile stock) {
            return String.Format("{0} {1}", stock.Code, stock.Name);
        }
    }
#endif

}
