﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Runtime.InteropServices;

#if UNITTEST
using NUnit.Framework;
#endif

// COPYRIGHT (C) 2001, Tomas Restrepo (tomasr@mvps.org)
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// 上記作者のソースに必要な修正を施した

namespace Travis.IO {

    public enum MapAccess {
        FileMapCopy=0x0001,
        FileMapWrite=0x0002,
        FileMapRead=0x0004,
        FileMapAllAccess=0x001f,
    }
    /// <summary>
    ///   Specifies page protection for the mapped file
    ///   These correspond to the PAGE_XXX set of flags
    ///   passed to CreateFileMapping()
    /// </summary>
    [Flags]
    public enum MapProtection {
        PageNone=0x00000000,
        // protection
        PageReadOnly=0x00000002,
        PageReadWrite=0x00000004,
        PageWriteCopy=0x00000008,
        // attributes
        SecImage=0x01000000,
        SecReserve=0x04000000,
        SecCommit=0x08000000,
        SecNoCache=0x10000000,
    }
    public class FileMapIOException : IOException {
        //
        // properties
        //
        private int m_win32Error = 0;
        public int Win32ErrorCode {
            get { return m_win32Error; }
        }
        public override string Message {
            get {
                if(Win32ErrorCode != 0)
                    return base.Message + " (" + Win32ErrorCode + ")";
                else
                    return base.Message;
            }
        }

        // construction
        public FileMapIOException(int error)
            : base() {
            m_win32Error = error;
        }
        public FileMapIOException(string message)
            : base(message) {
        }

    } // class FileMapIOException



    public class MemoryMappedFile : IDisposable {
        //! handle to MemoryMappedFile object
        private IntPtr _hMap = IntPtr.Zero;
        private IntPtr _mappedAddress;

        private const int GENERIC_READ  = unchecked((int)0x80000000);
        private const int GENERIC_WRITE = unchecked((int)0x40000000);
        private const int OPEN_ALWAYS   = 4;
        private static readonly IntPtr INVALID_HANDLE_VALUE = new IntPtr(-1);
        private static readonly IntPtr NULL_HANDLE = IntPtr.Zero;

        public bool IsOpen {
            get { return (_hMap != NULL_HANDLE); }
        }
        public void Dispose() {
            if(_mappedAddress!=IntPtr.Zero)
                Unmap();
            if(IsOpen)
                Win32MapApis.CloseHandle(_hMap);
            _hMap = NULL_HANDLE;
        }

        /// <summary>
        ///   Map a view of the file mapping object
        ///   This returns a stream, giving you easy access to the memory,
        ///   as you can use StreamReaders and StreamWriters on top of it
        /// </summary>
        /// <param name="access">desired access to the view</param>
        /// <param name="offset">offset of the file mapping object to 
        ///            start view at</param>
        /// <param name="size">size of the view</param>
        public IntPtr MapView(MapAccess access, long offset, int size) {
            if(!IsOpen)
                throw new ObjectDisposedException("MMF already closed");

            IntPtr baseAddress = IntPtr.Zero;
            baseAddress = Win32MapApis.MapViewOfFile(
                              _hMap, (int)access,
                              (int)((offset >> 32) & 0xFFFFFFFF),
                              (int)(offset & 0xFFFFFFFF), size
                           );

            if(baseAddress == IntPtr.Zero)
                throw new FileMapIOException(Marshal.GetHRForLastWin32Error());

            _mappedAddress = baseAddress;
            return baseAddress;

        }
        public void Unmap() {
            if(_mappedAddress==IntPtr.Zero)
                throw new FileMapIOException("MMF not mapped to address");
            Win32MapApis.UnmapViewOfFile(_mappedAddress);
            _mappedAddress = IntPtr.Zero;
        }

        /// <summary>
        ///   Create a named map object 
        /// </summary>
        /// <param name="fileName">name of backing file, or null 
        ///            for a pagefile-backed map</param>
        /// <param name="protection">desired access to the mapping 
        ///            object</param>
        /// <param name="maxSize">maximum size of filemap object, or 0 
        ///            for size of file</param>
        /// <param name="name">name of file mapping object</param>
        /// <returns>The memory mapped file instance</returns>
        public static MemoryMappedFile
           Create(string fileName, MapProtection protection,
                             long maxSize, String name) {
            MemoryMappedFile map = new MemoryMappedFile();

            // open file first
            IntPtr hFile = INVALID_HANDLE_VALUE;

            if(fileName != null) {
                // determine file access needed
                // we'll always need generic read access
                int desiredAccess = GENERIC_READ;
                if((protection == MapProtection.PageReadWrite) ||
                  (protection == MapProtection.PageWriteCopy)) {
                    desiredAccess |= GENERIC_WRITE;
                }

                // open or create the file
                // if it doesn't exist, it gets created
                hFile = Win32MapApis.CreateFile(
                            fileName, desiredAccess, 0,
                            IntPtr.Zero, OPEN_ALWAYS, 0, IntPtr.Zero
                         );
                if(hFile == INVALID_HANDLE_VALUE)
                    throw new FileMapIOException(Marshal.GetHRForLastWin32Error());

                // FIX: Is support neede for zero-length files!?!
            }

            map._hMap = Win32MapApis.CreateFileMapping(
                        hFile, IntPtr.Zero, (int)protection,
                        (int)((maxSize >> 32) & 0xFFFFFFFF),
                        (int)(maxSize & 0xFFFFFFFF), name
                     );

            // close file handle, we don't need it
            if(hFile != INVALID_HANDLE_VALUE) Win32MapApis.CloseHandle(hFile);
            if(map._hMap == NULL_HANDLE)
                throw new FileMapIOException(Marshal.GetHRForLastWin32Error());

            return map;
        }

        /// <summary>
        ///   Open an existing named File Mapping object
        /// </summary>
        /// <param name="access">desired access to the map</param>
        /// <param name="name">name of object</param>
        /// <returns>The memory mapped file instance</returns>
        public static MemoryMappedFile Open(MapAccess access, String name) {
            MemoryMappedFile map = new MemoryMappedFile();

            map._hMap = Win32MapApis.OpenFileMapping((int)access, false, name);
            if(map._hMap == NULL_HANDLE)
                throw new FileMapIOException(Marshal.GetHRForLastWin32Error());

            return map;
        }

    }

    internal class Win32MapApis {
        [DllImport("kernel32", SetLastError=true, CharSet=CharSet.Auto)]
        public static extern IntPtr CreateFile(
           String lpFileName, int dwDesiredAccess, int dwShareMode,
           IntPtr lpSecurityAttributes, int dwCreationDisposition,
           int dwFlagsAndAttributes, IntPtr hTemplateFile);

        [DllImport("kernel32", SetLastError=true, CharSet=CharSet.Auto)]
        public static extern IntPtr CreateFileMapping(
           IntPtr hFile, IntPtr lpAttributes, int flProtect,
           int dwMaximumSizeLow, int dwMaximumSizeHigh,
           String lpName);

        [DllImport("kernel32", SetLastError=true)]
        public static extern bool FlushViewOfFile(
           IntPtr lpBaseAddress, int dwNumBytesToFlush);

        [DllImport("kernel32", SetLastError=true)]
        public static extern IntPtr MapViewOfFile(
           IntPtr hFileMappingObject, int dwDesiredAccess, int dwFileOffsetHigh,
           int dwFileOffsetLow, int dwNumBytesToMap);

        [DllImport("kernel32", SetLastError=true, CharSet=CharSet.Auto)]
        public static extern IntPtr OpenFileMapping(
           int dwDesiredAccess, bool bInheritHandle, String lpName);

        [DllImport("kernel32", SetLastError=true)]
        public static extern bool UnmapViewOfFile(IntPtr lpBaseAddress);

        [DllImport("kernel32", SetLastError=true)]
        public static extern bool CloseHandle(IntPtr handle);

    } // class Win32MapApis

#if UNITTEST
    [TestFixture]
    public class MemoryMappedFileTests {
        private string _fileName1;

        [TestFixtureTearDown]
        public void TearDown() {
            if(File.Exists(_fileName1)) File.Delete(_fileName1);
        }

        [Test]
        public void MapFile1() {
            _fileName1 = Path.GetTempFileName();
            FileStream strm = new FileStream(_fileName1, FileMode.Create, FileAccess.Write);
            Random rnd = new Random();
            byte[] data = new byte[256];
            rnd.NextBytes(data);

            strm.Write(data, 0, data.Length);
            strm.Close();

            //これをMemoryMappedFIleで開き、中身がdataと同一であることをチェック
            MemoryMappedFile mmf = MemoryMappedFile.Create(_fileName1, MapProtection.PageReadOnly, 0, "TravisUnitTest1");
            IntPtr ptr = mmf.MapView(MapAccess.FileMapRead, 0, 256);

            for(int i=0; i<data.Length; i++) {
                byte rawdata;
                unsafe {
                    rawdata = *(((byte*)ptr) + i);
                }
                Assert.AreEqual(data[i], rawdata);
            }
            mmf.Dispose();

        }
    }

#endif

}
