/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Data/QuoteUtil.cs#11 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * TEւ̕ϊ
 */
using System;
using System.Collections.Generic;
using System.Diagnostics;

using Bellagio.Values;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Data {
    //T̎n܂NX@Tf[^Ƃ͂Ƒ
    public class WeeklyDateIndex {
		private int _index;
		private int _firstDate; //BDate`̓t
		private bool _startOfQuarter;

		public int FirstDate {
			get {
				return _firstDate;
			}
		}
        public int LastDate {
            get {
                return WeeklyDateMap.GetAt(_index+1).FirstDate; //TODO Ō̏TƂ܂Iɂ͐SzȂĂ
            }
        }
		public int Index {
			get {
				return _index;
			}
		}
		public bool StartOfQuarter {
			get {
				return _startOfQuarter;
			}
		}

		public WeeklyDateIndex(int index, int firstdate, bool quarter) {
			_index = index;
			_firstDate = firstdate;
			_startOfQuarter = quarter;
		}
    }

    //Tj̃RNV
    internal static class WeeklyDateMap {
        private static WeeklyDateIndex[] _data;

        private static void Construct() {
            DateTime dt = new DateTime(1987, 12, 27); //͓j
            List<WeeklyDateIndex> t = new List<WeeklyDateIndex>(2000); //2000TԂΊԈႢȂ
            DateTime lim = DateTime.Now;
            lim = lim.AddDays(30);
            int i = 0;
            while(dt < lim) {
                int dti = BDate.DateTimeToInt(dt);
                int m = (dti % 10000) / 100;
                int d = dti % 100;
                t.Add(new WeeklyDateIndex(i++, dti, d<=7 && (m==1 || m==4 || m==7 || m==10)));
                dt = dt.AddDays(7);
            }

            _data = t.ToArray();
        }

        public static WeeklyDateIndex FindByDate(int date) {
            if(_data==null) Construct();
            if(date<_data[1].FirstDate)
                return _data[0]; //[͂悭P[XȂ̂
            else if(date>=_data[_data.Length-1].FirstDate)
                return _data[_data.Length-1];
            else
                return FindByDate(date, 0, _data.Length);
        }
        private static WeeklyDateIndex FindByDate(int date, int start, int end) {
            int m = (start+end)/2;
            if(_data[m].FirstDate > date)
                return FindByDate(date, start, m);
            else if(_data[m+1].FirstDate <= date)
                return FindByDate(date, m+1, end);
            else
                return _data[m];
        }

        public static WeeklyDateIndex GetAt(int index) {
            if(_data==null) Construct();
            Debug.Assert(index>=0 && index<_data.Length);
            return _data[index];
        }
        public static int Length {
            get {
                if(_data==null) Construct();
                return _data.Length;
            }
        }

    }

    //T, ɕϊ
    internal static class QuoteConverter {
        //T̓t́ȀT̍ŏ̓tB͌jAjނƈقȂ
        public static ConcreteQuote DailyToWeekly(Quote daily) {
            Debug.Assert(daily.Unit==Quote.QuoteUnit.Daily);
            Debug.Assert(daily.Scale==1);

            WeeklyDateIndex weekly_begin = WeeklyDateMap.FindByDate(daily.CandleAt(0).Time);
			WeeklyDateIndex weekly_end   = WeeklyDateMap.FindByDate(daily.LastCandle.Time);
            
            int size = weekly_end.Index - weekly_begin.Index + 1;
            List<Candle> body = new List<Candle>(size);
            int date_index = 0;
            for(int i=weekly_begin.Index; i<=weekly_end.Index; i++) {
                int start_of_week = WeeklyDateMap.GetAt(i).FirstDate;
                int end_of_week = WeeklyDateMap.GetAt(i+1).FirstDate;
                Candle s = daily.CandleAt(date_index);
                if(s.Time < start_of_week || s.Time>=end_of_week) { //̑
                    int p = body.Count==0? 0 : body[body.Count-1].Close;
                    body.Add(new Candle(null, -1, start_of_week, p, p, p, p, 0, CandleFlags.None));
                    continue;
                }

                Candle week = new Candle(s);
                //̏TW߂
                date_index++;
                s = daily.CandleAtOrNull(date_index);
                while(s!=null && s.Time < end_of_week) {
                    week.High = Math.Max(week.High, s.High);
                    week.Low = Math.Min(week.Low, s.Low);
                    week.Close = s.Close;
                    week.Volume += s.Volume;
                    date_index++;
                    s = daily.CandleAtOrNull(date_index);
                }

                body.Add(week);
            }
            Debug.Assert(body.Count==size);

            return new ConcreteQuote(Quote.QuoteUnit.Weekly, 1, body);
        }

        public static ConcreteQuote DailyToMonthly(Quote daily) {
            Debug.Assert(daily.Unit==Quote.QuoteUnit.Daily);
            Debug.Assert(daily.Scale==1);

            BDate monthly_begin = new BDate(ToFirstDayOfMonth(daily.CandleAt(0).Time));
            BDate monthly_end = new BDate(ToFirstDayOfMonth(daily.LastCandle.Time));

            int date_index = 0;
            BDate i = new BDate(monthly_begin);
            List<Candle> body = new List<Candle>();
            while(i.AsInt()<=monthly_end.AsInt()) {
                BDate end_of_month = NextMonth(i);
                Candle s = daily.CandleAt(date_index);
                if(s.Time < i.AsInt() || s.Time>=end_of_month.AsInt()) { //̑
                    int p = body.Count==0? 0 : body[body.Count-1].Close;
                    body.Add(new Candle(null, -1, i.AsInt(), p, p, p, p, 0, CandleFlags.None));
                    i = end_of_month;
                    continue;
                }

                Candle month = new Candle(s);

                //̌W߂
                date_index++;
                s = daily.CandleAtOrNull(date_index);
                while(s!=null && s.Time < end_of_month.AsInt()) {
                    month.High = Math.Max(month.High, s.High);
                    month.Low = Math.Min(month.Low, s.Low);
                    month.Close = s.Close;
                    month.Volume += s.Volume;
                    date_index++;
                    s = daily.CandleAtOrNull(date_index);
                }

                i = end_of_month;
                body.Add(month);
            }

            return new ConcreteQuote(Quote.QuoteUnit.Monthly, 1, body);
        }

        private static int ToFirstDayOfMonth(int src) {
            src -= src % 100;
            return src + 1;
        }
        private static BDate NextMonth(BDate src) {
            int y = src.Year;
            int m = src.Month + 1;
            if(m==13) {
                m = 1;
                y++;
            }
            return new BDate(y, m, 1);
        }

        //܂Ƃ߂BPRAȂǁB
        public static ConcreteQuote Shrink(Quote src, int scale) {
            ConcreteQuote result = new ConcreteQuote(src.Unit, src.Scale * scale);
            if(src.Count > 0) {
                int len = src.Count/scale;
                for(int i=0; i<len; i++) {
                    int j = i * scale;
                    int next = j+scale;
                    Candle y = new Candle(src.CandleAt(j++));
                    while(j<next)
                        y.Update(src.CandleAt(j++));

                    result.Add(y);
                }
            }
            return result;
        }

        //񔽉fς݂猳̃f[^Ԃ
        public static ConcreteQuote InverseReflectSplit(Quote src, SplitInfo[] splits) {
            return ReflectSplitInternal(src, splits, true);
        }
        //𔽉f̂Ԃ
        public static ConcreteQuote ReflectSplit(Quote src, SplitInfo[] splits) {
            return ReflectSplitInternal(src, splits, false);
        }
        //̔f
        private static ConcreteQuote ReflectSplitInternal(Quote src, SplitInfo[] splits, bool inverse) {
            Debug.Assert(src.Unit==Quote.QuoteUnit.Daily);
            ConcreteQuote result = new ConcreteQuote(src.Unit);
            result.SetCapacity(src.Count);
            result.SetSize(src.Count);

            int si_index = splits.Length-1;
            int date = src.LastCandle.Time;
            while(si_index >= 0 && splits[si_index].Date > date) si_index--; //͖̕

            //tU߂
            double ratio = 1;
            int next_target_date = si_index>=0? splits[si_index].Date : 0; //ratioׂt
            for(int index = src.Count-1; index >= 0; index--) {
                Candle c = src.CandleAt(index);
                if(c.Time < next_target_date) {
                    if(inverse)
                        ratio /= splits[si_index].Ratio;
                    else
                        ratio *= splits[si_index].Ratio;
                    si_index--;
                    next_target_date = si_index>=0? splits[si_index].Date : 0;
                }

                Candle d = new Candle(result, result.Count, c.Time, Adj(c.Open, ratio), Adj(c.High, ratio), Adj(c.Low, ratio), Adj(c.Close, ratio), Adj(c.Volume, 1.0/ratio), CandleFlags.None);
                d.CreditLong = Adj(c.CreditLong, 1.0/ratio);
                d.CreditShort = Adj(c.CreditShort, 1.0/ratio);
                result.SetDirect(index, d);
                
            }

            return result;
        }
        private static int Adj(int src, double ratio) {
            return (int)(src / ratio);
        }

    }
#if UNITTEST
    [TestFixture]
    public class QuoteConverterTests {
        [Test]
        public void Split() {
            ConcreteQuote q = new ConcreteQuote(Quote.QuoteUnit.Daily);
            q.Add(801, 300, 100);
            q.Add(802, 300, 100);
            q.Add(803, 100, 300);
            q.Add(804, 100, 300);
            q.Add(805,  50, 600);
            q.Add(806,  50, 600);

            //t݂̑Ȃ邱ƂmF
            //o^
            ConcreteQuote s = QuoteConverter.ReflectSplit(q, new SplitInfo[] { new SplitInfo(803, 3.0), new SplitInfo(805, 2.0), new SplitInfo(810, 2.0) });
            Assert.AreEqual(6, s.Count);
            for(int i=0; i<s.Count; i++) {
                Candle c = s.CandleAt(i);
                Assert.AreEqual(801+i, c.Time);
                Assert.AreEqual(50, c.Open);
                Assert.AreEqual(50, c.Close);
                Assert.AreEqual(600, c.Volume);
            }
        }
    }
#endif

}
