/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/FigureBoard/FigureBoardSession.cs#17 $
 * $DateTime: 2008/05/14 13:05:12 $
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;

using Poderosa;
using Poderosa.UI;
using Poderosa.Forms;
using Poderosa.Preferences;
using Poderosa.Commands;
using Poderosa.Plugins;
using Poderosa.Sessions;

using Travis.ORT;
using Sansa.Runtime;

using Bellagio.Environment;
using Bellagio.Data;
using Bellagio.Values;
using Bellagio.Drawing;
using Bellagio.Forms;
using Bellagio.Evaluators;

namespace Bellagio.FigureBoard {

    //ZbV\zp[^
    public class FigureBoardSessionCreateParam {
        private FigureBoardSchema _schema;
        public FigureBoardSessionCreateParam(FigureBoardSchema schema) {
            _schema = schema;
        }

        public FigureBoardSchema Schema {
            get {
                return _schema;
            }
        }
    }

    public class FigureBoardSessionSettings {
        public SmartColor backColor;
        public SmartColor timeColor;
        public int itemMargin; //ACeԂ̊Ԋu ㉺EE
        public Size itemSize; //_ɊÂĒBɍB

        public FigureBoardSessionSettings() {
            backColor.Set(BellagioRoot.FixedPreferences.DefaultBackColor);
            timeColor.Set(Color.LightGray);
            itemMargin = 8;
            itemSize = new Size(150, 0);
        }
    }

    public class FigureBoardItem {
        private FigureBoardDocument _document;

        private int _index;
        private string _label;
        private IEvaluator _evaluator; //͕K{
        private double _lastValue;
        private double _currentValue;
        private IEvaluator _subEvaluator; //IvVi
        private double _currentSubValue;

        internal FigureBoardItem(FigureBoardDocument doc, int index, string label, IEvaluator evaluator, IEvaluator subEvaluator) {
            _document = doc;
            _index = index;
            _label = label;
            _evaluator = evaluator;
            _subEvaluator = subEvaluator;
            _lastValue = Double.NaN;
            _currentValue = Double.NaN;
            _currentSubValue = Double.NaN;
        }

        internal IEvaluator Evaluator {
            get {
                return _evaluator;
            }
        }
        internal IEvaluator SubEvaluator {
            get {
                return _subEvaluator;
            }
        }

        public bool ValueChanging {
            get {
                return _lastValue!=_currentValue;
            }
        }
        public double CurrentValue {
            get {
                return _currentValue;
            }
        }
        public double CurrentSubValue {
            get {
                return _currentSubValue;
            }
        }

        internal void SetCurrentValue(BV value, BV sub) {
            _lastValue = _currentValue;
            _currentValue = BVToDouble(value);
            _currentSubValue = BVToDouble(sub);
        }

        private static double BVToDouble(BV value) {
            if(value==null || value.IsNil) return Double.NaN;
            BDouble d = value as BDouble;
            if(d!=null) return d.Value;
            BInt i = value as BInt;
            if(i!=null) return i.Value;

            return Double.NaN;
        }

        public string Label {
            get {
                return _label;
            }
        }

    }

    public class FigureBoardDocument : IPeriodicalDataSubscriber {

        //쐬ς݂̃J
        private FigureBoardSession _session;
        private EvalContext _evalContext;
        private BTime _lastUpdateTime;
        private List<Stock> _subscribingIndices;
        private RealtimeSubscriber _realtimeSubscriber;
        private List<FigureBoardItem> _items;
        private int _interval;
       
        public FigureBoardDocument(FigureBoardSession sess) {
            _session = sess;
            _items = new List<FigureBoardItem>();
            _evalContext = new EvalContext();
            _subscribingIndices = new List<Stock>();
            _lastUpdateTime = new BTime(0);
        }
        public void AddItem(FigureBoardItem item) {
            _items.Add(item);
        }
        public int ItemCount {
            get {
                return _items.Count;
            }
        }
        public EvalContext EvalContext {
            get {
                return _evalContext;
            }
        }
        public FigureBoardItem ItemAt(int index) {
            return _items[index];
        }
        public int Interval {
            get {
                return _interval;
            }
            set {
                _interval = value;
            }
        }

        public BellagioDocumentStatus DocumentStatus {
            get {
                //IndexɂĂ̓TuXNCũRv[gƂTO͂Ȃ
                return _realtimeSubscriber.IsEmpty? BellagioDocumentStatus.Complete : _realtimeSubscriber.DocumentStatus;
            }
        }

        public void CompleteCreation() {
            EvalContext ec = new EvalContext();
            EvalSpecial_CodeCollector codes = new EvalSpecial_CodeCollector();
            ec.EvalSpecial_CodeCollector = codes;
            //ꃂ[hŕ]Jn
            foreach(FigureBoardItem item in _items) {
                item.Evaluator.Eval(ec);
                if(item.SubEvaluator!=null) item.SubEvaluator.Eval(ec);
            }

            List<Stock> targets = new List<Stock>();
            foreach(string code in codes.ResultStocks) {
                AbstractStockProfile stock = BellagioRoot.GlobalStockCollection.FindExact(code);
                if(stock==null || (!(stock is BasicStockProfile) && !(stock is DerivativeStockProfile)))
                    throw new BellagioException(String.Format("R[h {0} ͊/敨̃R[hł͂܂", code)); //TODO ꂪ`Ăʒu񂪕IɂłƂ悢

                targets.Add(stock.Primary);
            }
            foreach(string code in codes.ResultIndex) {
                MarketIndex stock = BellagioRoot.GlobalStockCollection.FindExact(code) as MarketIndex;
                if(stock==null)
                    throw new BellagioException(String.Format("R[h {0} ͎w̃R[hł͂܂", code)); 

                //_subscribingIndices.Add(stock.Primary);
                targets.Add(stock.Primary);
            }

            _realtimeSubscriber = new RealtimeSubscriber(this, targets.ToArray());
            //StartSubscribe();
        }

        public void StartSubscribe() {
            if(!_realtimeSubscriber.IsEmpty)
                BellagioRoot.DataSubscriberManager.AddRealtimeSubscriber(_realtimeSubscriber, _realtimeSubscriber.CreateSubscribeRequest() );
            BellagioRoot.DataSubscriberManager.AddPeriodicalSubscriber(this);
        }

        public BTime LastUpdateTime {
            get {
                return _lastUpdateTime;
            }
        }

        int IPeriodicalDataSubscriber.IntervalSec {
            get {
                return _interval;
            }
        }

        DataThreadToMainThread IDataSubscriber.NotifyDelegate {
            get {
                return _session.ViewContent.UpdateNotifyDelegate;
            }
        }

        void IPeriodicalDataSubscriber.PeriodicalProcess(DataProcessArg args) {
            UpdateValues(args.LocalTime);
        }

        public void UpdateValues() {
            UpdateValues(BellagioRoot.TimeManager.Current);
        }

        //ōśA擾ĂR[hQ̊m(̂)Aɑ΂XVvMB
        //̂ƃf[^Ar[ɕ`˗
        public void UpdateValues(BTime current) {
            _lastUpdateTime.LetInt(current.AsInt());
            BV sub ;
            foreach(FigureBoardItem item in _items) {
                sub = null;
                if(item.SubEvaluator!=null) {
                    BV.Let(ref sub, item.SubEvaluator.Eval(_evalContext));
                }
                BV d = (BV)item.Evaluator.Eval(_evalContext); //CEvalBVg܂킵ASetCurrentValuedoubleôOK
                item.SetCurrentValue(d, sub);
            }
        }

        //Unsubscribe
        public void Terminate() {
            if(!_realtimeSubscriber.IsEmpty)
                BellagioRoot.DataSubscriberManager.RemoveSubscriber(_realtimeSubscriber);
            BellagioRoot.DataSubscriberManager.RemoveSubscriber(this);
        }

        //A^Cf[^̃TuXNCu
        //ʃNXłȂƁAFigureBoard̂PeriodicalSubscriberȂ̂ƗłȂ
        private class RealtimeSubscriber : BellagioDocumentBase, IRealTimeDataSubscriber {
            private Stock[] _stocks;
            private FigureBoardDocument _document;

            public RealtimeSubscriber(FigureBoardDocument doc, Stock[] stocks) {
                _stocks = stocks;
                _documentStatus = BellagioDocumentStatus.Loading;
                _notifyDelegate = null;
                _document = doc;
            }
            public bool IsEmpty {
                get {
                    return _stocks.Length==0;
                }
            }

            public int ThinningTime {
                get {
                    return 500;
                }
            }

            public void RealTimeProcess(DataProcessArg arg) {
                //ȂOK
            }

            public SubscribeRequest CreateSubscribeRequest() {
                return new SubscribeRequest(_stocks, SubscribeDataKind.Ticks);
            }

            public void SubscribeComplete() {
                _documentStatus = BellagioDocumentStatus.Complete;
                //ʕ`摣ߒʒm
                DataThreadToMainThread m = ((IDataSubscriber)_document).NotifyDelegate;
                m.Invoke();
            }

            public void SubscribeFailed(string reason) {
                base.DefaultSubscribeFailed(reason);
            }

        }

    }

    public class FigureBoardContent : IDisposable {
        private FigureBoardSession _session;
        private FigureBoardSessionSettings _settings;
        private FigureBoardItemDrawing _drawing;
        private DataThreadToMainThread _updateNotifyDelegate;
        //FigureBoardłDrawingSNXƈ̉Ă̂łDataTimerĂ܂
        private DoubleTimer _dataAnimationTimer;

        private bool _renderedAdjusted;
        private Form _container;

        //CAEg
        private PointF _firstItemPos; //̃ACe̍W
        private int _itemCountX;

        public FigureBoardContent(FigureBoardSession sess) {
            _session = sess;
            _settings = new FigureBoardSessionSettings();
            _drawing = new FigureBoardItemDrawing();
            _updateNotifyDelegate = delegate() { StartAnimation(); };
            _renderedAdjusted = false;

            IPlatformBridge platform = BellagioPlugin.Instance.SansaPlatform;
            BellagioPlugin.Instance.SansaPlatform.AssureStart();

            _dataAnimationTimer = new DoubleTimer(platform,
                new DTimerHandler(delegate() { _dataAnimationTimer.reset(0, 1, platform.getRootTimer().secondToTick(0.5)); }, new DTimerHandler.D(InvalidateAll)));
        }

        public void Attach(Form parent) {
            _container = parent;
            //̂ݕ]
            if(!_renderedAdjusted) {
                Graphics g = _container.CreateGraphics();
                _drawing.Settings.CreateByGraphics(g);
                _settings.itemSize.Height = (int)(_drawing.Settings.labelHeight + _drawing.Settings.priceHeight);
                g.Dispose();

                _renderedAdjusted = true;
            }

            LayoutItems(parent.ClientRectangle);
        }


        public void OnPaint(PaintEventArgs args) {
            try {
                Graphics g = args.Graphics;
                Rectangle clip = args.ClipRectangle;
                FigureBoardDocument doc = _session.Document;

                //ݎ
                /*
                Rectangle time_area = new Rectangle((int)_firstItemPos.X, 4, 100, 16);
                if(clip.IntersectsWith(time_area) && doc.LastUpdateTime.AsInt()!=0) {
                    Brush br = new SolidBrush(_settings.timeColor);
                    g.DrawString(String.Format("{0} XV", BTime.FormatHHMM(doc.LastUpdateTime)), _container.Font, br, time_area.Location);
                    br.Dispose();
                }
                */
                if(PaintIncompleteCases(args)) return;

                float progress = _dataAnimationTimer.IsActive()? (float)_dataAnimationTimer.value() : 1F; //^C}[̐isɂ킹
                for(int i=0; i<doc.ItemCount; i++) {
                    float x = _firstItemPos.X + (_settings.itemSize.Width+_settings.itemMargin) * (i % _itemCountX);
                    float y = _firstItemPos.Y + (_settings.itemSize.Height+_settings.itemMargin) * (i / _itemCountX);
                    if(clip.IntersectsWith(new Rectangle((int)x, (int)y, _settings.itemSize.Width, _settings.itemSize.Height))) {
                        FigureBoardItem item = doc.ItemAt(i);
                        _drawing.Draw(g, item, new PointF(x, y), _settings.itemSize.Width, item.ValueChanging? progress : 1F);
                    }
                }
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }
        public void OnResize() {
            LayoutItems(_container.ClientRectangle);
            _container.Invalidate();
        }

        public void Dispose() {
            
            _dataAnimationTimer.Abort();
           
        }

        public DataThreadToMainThread UpdateNotifyDelegate {
            get {
                return _updateNotifyDelegate;
            }
        }
        public void StartAnimation() {
            //Debug.WriteLine("StartAnim " + _dataAnimationTimer.IsActive());
            if(BellagioRoot.FixedPreferences.Chart.EnableFigureBoardAnimation && !_dataAnimationTimer.IsActive())
                _dataAnimationTimer.Initialize();
            else
                InvalidateAll();
        }

        //f[^^C}[AXV̂Ƃ̎s
        private void InvalidateAll() {
            if(_container!=null) _container.Invalidate();
        }

        protected static WaitingMark _waitingMark = new WaitingMark(); //Ŝŋ
        private bool PaintIncompleteCases(PaintEventArgs args) {
            Graphics g = args.Graphics;
            switch(_session.Document.DocumentStatus) {
                case BellagioDocumentStatus.Complete:
                    return false;
                case BellagioDocumentStatus.Loading: {
                        int s = _waitingMark.Radius * 2;
                        if(args.ClipRectangle.IntersectsWith(DrawingUtil.CenterPointToRect(DrawingUtil.RectCenterPoint(_container.ClientRectangle), s, s))) {
                            using(HighQualityDrawingSupport dr = new HighQualityDrawingSupport(g)) {
                                _waitingMark.Draw(g, DrawingUtil.RectCenterPoint(_container.ClientRectangle), BellagioPlugin.Instance.SansaPlatform.getRootTimer().getTick());
                            }
                        }
                        return true;
                    }
                default:
                    return true;
            }
        }


        private void LayoutItems(Rectangle client_rect) {
            Size size_with_margin = new Size(_settings.itemSize.Width + _settings.itemMargin, _settings.itemSize.Height + _settings.itemMargin);
            _itemCountX = Math.Min(_session.Document.ItemCount, Math.Max(1, (client_rect.Width-8) / size_with_margin.Width)); //ɕׂb}[WvZBŒP

            //ɌvZ
            _firstItemPos.X = (client_rect.Width  - _settings.itemSize.Width * _itemCountX - _settings.itemMargin * (_itemCountX-1)) / 2;
            _firstItemPos.Y = 4;
        }
    }



    //FigureBoard́APXL[}ƂɂPZbV݂̂B
    //ȑOPoderosaSessionAƗtH[ɂȂ悤ɉ
    public class FigureBoardSession {
        private string _caption;
        private FigureBoardContent _content;
        private FigureBoardDocument _document;
        private Rectangle _formRect;
        private Form _form;

        public FigureBoardSession(FigureBoardSessionCreateParam p) {
            _content = new FigureBoardContent(this);
            _document = new FigureBoardDocument(this);
            FigureBoardSchema schema = p.Schema;

            Create(schema, p);

            ShowForm();
            InternalStart();
        }

        public FigureBoardDocument Document {
            get {
                return _document;
            }
        }
        public FigureBoardContent ViewContent {
            get {
                return _content;
            }
        }

        private void InternalStart() {
            _document.StartSubscribe();
        }

        private void InternalTerminate() {
            BellagioRoot.DataSubscriberManager.RemoveSubscriber(_document);
            _document.Terminate();
        }

        //xÎĊJ
        public void Resume() {
            if(_form==null) {
                ShowForm();
                InternalStart();
            }
        }

        //쐬
        private void Create(FigureBoardSchema schema, FigureBoardSessionCreateParam param) {
            _caption = schema.description.Value;
            EvaluatorBuildContext bc = new EvaluatorBuildContext(schema);

            foreach(FigureBoardItemSchema item_schema in schema.item) {
                FigureBoardItem item = new FigureBoardItem(
                    _document,
                    _document.ItemCount,
                    item_schema.name.ParseMandatoryString(),
                    EvaluatorBuilderFromText.Build(item_schema.expression, bc),
                    CreateSubEvaluator(item_schema, bc)
                    );

                _document.AddItem(item);
            }

            _document.Interval = schema.interval.ParseOptionalInt(60);
            _document.CompleteCreation();
        }

        private IEvaluator CreateSubEvaluator(FigureBoardItemSchema item_schema, EvaluatorBuildContext bc) {
            string sub_expr = item_schema.subExpression.ParseOptionalString("");
            if(sub_expr.Length>0)
                return EvaluatorBuilderFromText.Build(item_schema.subExpression, bc);
            else
                return null;
        }

        //tH[̍쐬ƕ\
        private void ShowForm() {
            _form = new BellagioDefaultForm();
            FormUtil.AdjustStyleForModelessSizableDialog(_form, BellagioPlugin.Instance.MainWindow);
            if(_formRect.IsEmpty)
                _form.ClientSize = new Size(200, 250);
            else
                _form.DesktopBounds = _formRect;

            _form.Text = _caption;
            _form.BackColor = BellagioRoot.FixedPreferences.DefaultBackColor;
            _form.Paint += delegate(object sender, PaintEventArgs a) { _content.OnPaint(a); };
            _form.Resize += delegate(object sender, EventArgs a) { _content.OnResize(); };
            _form.FormClosed += new FormClosedEventHandler(OnClosed);
            _content.Attach(_form);

            _form.Show();
        }
        private void OnClosed(object sender, FormClosedEventArgs a) {
            try {
                InternalTerminate(); //Dispose͍ĊJɂłȂł
                _formRect = _form==null? Rectangle.Empty : _form.DesktopBounds; //ςnull̃P[X
                _form = null;
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }

        //PXL[}PZbVɂ邽߂̃}bv
        private static Dictionary<FigureBoardSchema, FigureBoardSession> _sessionMap = new Dictionary<FigureBoardSchema, FigureBoardSession>();


        public static IPoderosaCommand CreateOpenOrActivateSessionCommand(ExtensionKitItem item) {
            Debug.Assert(item is FigureBoardSchema);
            return new OpenOrActivateSessionCommand((FigureBoardSchema)item);
        }
        private class OpenOrActivateSessionCommand : PoderosaCommandImpl {
            private FigureBoardSchema _schema;
            public OpenOrActivateSessionCommand(FigureBoardSchema s) {
                _schema = s;
            }
            public override CommandResult InternalExecute(ICommandTarget target, params IAdaptable[] args) {
                FigureBoardSessionCreateParam cp = new FigureBoardSessionCreateParam(_schema);
                FigureBoardSession s;
                if(_sessionMap.TryGetValue(_schema, out s))
                    s.Resume();
                else {
                    s = new FigureBoardSession(cp);
                    _sessionMap.Add(_schema, s);
                }
                return CommandResult.Succeeded;
            }
        }
    }

}
