/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Ordering/OrderAndPositionListForm.cs#10 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * ƉA|WVƉ̃tH[@ʕ̂łP̃t@Cɂ܂Ƃ߂Ă܂
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;

using Poderosa.Forms;
using Travis.LocationOperation;
using Bellagio.Forms;
using Bellagio.Data;
using Bellagio.Environment;
using Poderosa;

namespace Bellagio.Ordering {
    public abstract class OrderAndPositionListFormBase : Form {
        protected BellagioListControl _list; //C̃XgRg[@͔̂hNXɂ
        protected OrderAndPositionPage _page;
        private RectOp _rectOp;
        private Label _widthTooShortLabel;
        protected CheckBox _omitWarningCheckBox;
        private RibbonMenuButton _stockButton;
        private RibbonMenuButton _derivativeButton;
        private Button _refreshButton;

        protected Label _displayStyleLabel;
        protected ComboBox _displayStyleBox;
        protected Label _sortLabel;
        protected ComboBox _sortBox;

        public OrderAndPositionListFormBase() {
            this.Icon = BellagioRoot.CommonIcons.AppIcon16;
            this.ShowInTaskbar = false;
            this.FormBorderStyle = FormBorderStyle.Sizable;
            this.StartPosition = FormStartPosition.Manual;
            this.MaximumSize = new Size(10000, 10000); //ȂʂǂȂɂłĂ낤
            //this.Opacity = 0.8;

            _widthTooShortLabel = FormUtil.CreateLabel("TCY\傫Ȃ߈ꕔڂ̕\ȗĂ܂");
            _widthTooShortLabel.Visible = false;
            _widthTooShortLabel.ForeColor = Color.Red;
            _omitWarningCheckBox = new CheckBox();
            _omitWarningCheckBox.Visible = false;

            _stockButton = FormUtil.CreateRibbonTextButton("(S)", new MouseEventHandler(OnStockButtonClick));
            _stockButton.Checked = true;
            _derivativeButton = FormUtil.CreateRibbonTextButton("敨(F)", new MouseEventHandler(OnDerivativeButtonClick));
            _derivativeButton.Checked = false;
            _refreshButton = FormUtil.CreateImageAndTextButton(BellagioRoot.CommonIcons.SmallCommandButtons.GetBitmap(0), "XV(&R)", new EventHandler(OnRefreshButtonClick));

            CreateContentControls();

            _rectOp = new RectangleOperatorBuilder(4)
                .SplitTop(48)
                    .SplitFixed(SplitDir.Right, 96).PlaceControl(ContentAlignment.MiddleLeft, _refreshButton, 88, 28).CloseBranch()
                    
                    .SplitFixed(SplitDir.Right, 156)
                        .SplitFixed(SplitDir.Top, 24)
                            .PlaceControl(ContentAlignment.MiddleLeft, _displayStyleLabel, 85, 19).RightControl(_displayStyleBox, 65, 19)
                            .CloseBranch()
                        .PlaceControl(ContentAlignment.MiddleLeft, _sortLabel, 85, 19).RightControl(_sortBox, 65, 19)
                    .CloseBranch()
                    .SplitFixed(SplitDir.Right, 156)
                        .PlaceControl(ContentAlignment.MiddleLeft, _omitWarningCheckBox, 156, 19)
                    .CloseBranch()

                    .SplitFixed(SplitDir.Top, 24).ProbeControl(_widthTooShortLabel).CloseBranch()

                    .PlaceControl(ContentAlignment.TopLeft, _stockButton, 80, 19)
                    .RightControl(_derivativeButton, 80, 19)
                .CloseBranch()
                .ProbeControl(_list)
                .Finish();

            this.SuspendLayout();
            this.Controls.AddRange(new Control[] { _list, _widthTooShortLabel, _omitWarningCheckBox, _refreshButton, _stockButton, _derivativeButton, _displayStyleBox, _displayStyleLabel, _sortLabel, _sortBox });
            this.ClientSize = new Size(_list.Width, _list.Height+48);
            this.ResumeLayout();
        }

        public BellagioListControl MainListControl {
            get {
                return _list;
            }
        }

        protected override void OnResize(EventArgs e) {
            base.OnResize(e);
            _rectOp.Rect(this.ClientRectangle);
            AdjustWidthInfo();
            _list.Invalidate();
        }
        public void AdjustWidthInfo() {
            _widthTooShortLabel.Visible = _list.OmittedItemExists;
        }

        public OrderAndPositionPage CurrentPage {
            get {
                return _page;
            }
        }

        //C̃XgTCYݒ肵č쐬BDisplayStyleLabel/Box
        protected abstract void CreateContentControls();

        protected abstract void RefreshInternalData();
        protected abstract void AttachNotifier();
        protected abstract void DetachNotifier();

        private void OnOK(object sender, EventArgs args) {
            this.Visible = false;
            this.DialogResult = DialogResult.None;
        }
        //\[Xj͂Ȃ悤OnFormClosingɂHideBJƂ邩Ȃ
        protected override void OnFormClosing(FormClosingEventArgs e) {
            base.OnFormClosing(e);
            if(!BellagioPlugin.Instance.PoderosaApplication.IsExitingApplication) {
                e.Cancel = true;
                InternalHide();
            }
        }
        private void InternalHide() {
            this.Hide();
            IPoderosaMainWindow w = BellagioPlugin.Instance.ActivePoderosaWindow;
            if(w!=null) w.AsForm().Activate();
        }

        //L[{[hnh
        protected override bool ProcessDialogKey(Keys keyData) {
            //1...9A...Z͊Y|WVɑ΂ANV
            if(Keys.D1<=keyData && keyData<=Keys.D9) {
                _list.DoActionByNumericIndex(keyData - Keys.D1);
                return true;
            }
            else if(Keys.NumPad1<=keyData && keyData<=Keys.NumPad9) {
                _list.DoActionByNumericIndex(keyData - Keys.NumPad1);
                return true;
            }
            else if(Keys.A <= keyData && keyData <= Keys.Z) {
                _list.DoActionByNumericIndex(keyData - Keys.A + 9);
                return true;
            }
            else if(keyData==(Keys.Alt|Keys.S))
                OnStockButtonClick(null, null);
            else if(keyData==(Keys.Alt|Keys.F))
                OnDerivativeButtonClick(null, null);
            else if(keyData==Keys.F5)
                OnRefreshButtonClick(_refreshButton, null);

            //Escapeŕ@[hX_CAOł͎蓮łȂƓȂۂ
            else if(keyData==Keys.Escape)
                InternalHide();

            return base.ProcessDialogKey(keyData);
        }

        private void OnStockButtonClick(object sender, MouseEventArgs args) {
            if(_page!=OrderAndPositionPage.Stocks)
                SetPage(OrderAndPositionPage.Stocks);
        }
        private void OnDerivativeButtonClick(object sender, MouseEventArgs args) {
            if(_page!=OrderAndPositionPage.Derivatives)
                SetPage(OrderAndPositionPage.Derivatives);
        }
        private long _lastRefreshTime;
        private void OnRefreshButtonClick(object sender, EventArgs args) {
            long time = DateTime.Now.Ticks / 10000; //msP
            if(time < _lastRefreshTime+1000) return; //ߐڍXV͖

            _refreshButton.Enabled = false;
            RefreshInternalData();
            _list.ResetContent();
        }

        protected void SetPage(OrderAndPositionPage page) {
            //ڑĂȂȂRemoveႤ
            DetachNotifier();

            _refreshButton.Enabled = false;
            _stockButton.Checked = page==OrderAndPositionPage.Stocks;
            _derivativeButton.Checked = page==OrderAndPositionPage.Derivatives;
            _page = page;
            AttachNotifier();
            RefreshInternalData();
            _list.ResetContent();
        }
        protected void OnReceivedResponse() {
            _lastRefreshTime = DateTime.Now.Ticks / 10000;
            _refreshButton.Enabled = true;
        }


    }

    //Ɖy[W̎ށB܂͊Ɛ敨̂QB
    public enum OrderAndPositionPage {
        Stocks, Derivatives
    }

    //Ɖ̕\IvV
    public enum PositionListDisplayStyle {
        Detailed,
        CombineSameStock //͂܂Ƃ߂
    }

    public class PositionListForm : OrderAndPositionListFormBase {
        private PositionListDrawingSettings _drawingSettings;
        private PositionListDisplayStyle _style;
        private DataThreadToMainThread _stockUpdateHandler;
        private DataThreadToMainThread _derivativeUpdateHandler;
        private Comparison<TradePositionItem>[] _comparisons;

        public PositionListForm() {
            this.Text = "|WVꗗ";
            _stockUpdateHandler = new DataThreadToMainThread(OnStockUpdate);
            _derivativeUpdateHandler = new DataThreadToMainThread(OnDerivativeUpdate);
            this.MinimumSize = new Size(600, 64);
        }
        public OrderDataElement<TradePositionList> Document {
            get {
                return _page==OrderAndPositionPage.Stocks? BellagioRoot.OrderDataProvider.StockPosition : BellagioRoot.OrderDataProvider.DerivativePosition;
            }
        }
        public PositionListDisplayStyle DisplayStyle {
            get {
                return _style;
            }
        }
        protected override void CreateContentControls() {
            _drawingSettings = BellagioRoot.FixedPreferences.Ordering.PositionListDrawingSettings;
            _displayStyleLabel = FormUtil.CreateLabel("ʕʕ\(&C)");
            _displayStyleBox = FormUtil.CreateComboBox("", "ꊇ");
            _displayStyleBox.SelectedIndexChanged += new EventHandler(OnStyleChange);
            _style = PositionListDisplayStyle.Detailed;

            _sortLabel = FormUtil.CreateLabel("(&O)");
            _sortBox = FormUtil.CreateComboBox("", "ʏ", "v");
            _sortBox.SelectedIndex = RuntimeUtil.AdjustIntRange(BellagioRoot.FixedPreferences.Ordering.PositionListSortOption, 0, _sortBox.Items.Count-1);

            _sortBox.SelectedIndexChanged += new EventHandler(OnStyleChange);
            _comparisons = new Comparison<TradePositionItem>[3] { new Comparison<TradePositionItem>(CodeComparison), new Comparison<TradePositionItem>(TypeComparison), new Comparison<TradePositionItem>(ProfitComparison) };

            _omitWarningCheckBox.Text = "ꊇ\̌xȗ";
            _omitWarningCheckBox.Checked = !BellagioRoot.FixedPreferences.Ordering.ShowMultipleCreditCloseWarning;
            _omitWarningCheckBox.Visible = false;
            _omitWarningCheckBox.CheckedChanged += delegate(object sender, EventArgs args) {
                BellagioRoot.FixedPreferences.Ordering.ShowMultipleCreditCloseWarning = !_omitWarningCheckBox.Checked;
            };

            PositionListControl c = new PositionListControl(this);
            c.ItemFormatter = BellagioRoot.OrderDataProvider.OrderItemFormatter;
            c.Settings = _drawingSettings;
            c.ClientSize = new Size(720, 300);

            _list = c;
        }
        protected override void RefreshInternalData() {
            if(_page==OrderAndPositionPage.Stocks)
                BellagioRoot.OrderDataProvider.StockPosition.Refresh();
            else
                BellagioRoot.OrderDataProvider.DerivativePosition.Refresh();
        }
        protected override void OnLoad(EventArgs e) {
            base.OnLoad(e);
            SetPage(OrderAndPositionPage.Stocks);
        }
        protected override void AttachNotifier() {
            BellagioRoot.OrderDataProvider.StockPosition.AddListener(_stockUpdateHandler);
            BellagioRoot.OrderDataProvider.DerivativePosition.AddListener(_derivativeUpdateHandler);
        }
        protected override void DetachNotifier() {
            BellagioRoot.OrderDataProvider.StockPosition.RemoveListener(_stockUpdateHandler);
            BellagioRoot.OrderDataProvider.DerivativePosition.RemoveListener(_derivativeUpdateHandler);
        }
        private void OnStockUpdate() {
            if(_page!=OrderAndPositionPage.Stocks) return;
            RefreshBody(BellagioRoot.OrderDataProvider.StockPosition);
        }
        private void OnDerivativeUpdate() {
            if(_page!=OrderAndPositionPage.Derivatives) return;
            RefreshBody(BellagioRoot.OrderDataProvider.DerivativePosition);
        }
        private void RefreshBody(OrderDataElement<TradePositionList> elem) {
            try {
                Debug.Assert(BUtil.IsExecutingInMainThread);
                SetContent(elem.Value);
                base.OnReceivedResponse();
            }
            catch(Exception ex) {
                Debug.WriteLine(ex.Message);
                BUtil.ShowWarningMessageBox(ex.Message);
            }
        }
        //gZbgBnull̂Ƃ
        private void SetContent(TradePositionList result) {
            if(result!=null) {
                if(_style==PositionListDisplayStyle.CombineSameStock) //[ĥƂ͒
                    result = CombineSameStock(result);
                result.Sort(_comparisons[_sortBox.SelectedIndex]);
            }

            ((PositionListControl)_list).SetContent(result);
            _list.ResetContent();
        }
        private static TradePositionList CombineSameStock(TradePositionList src) {
            TradePositionList r = new TradePositionList();
            foreach(TradePositionItem e in src) {
                TradePositionItem target = r.FindByPositionType(e.Stock, e.DealType, e.LongShort, e.OrderKouza);
                if(target==null || e.OrderRestriction==OrderRestriction.CancelOnly) //CancelOnly͓P[XȂ̂ŏɓƗ
                    r.AddItem(e.Clone()); //݂Ȃ΃Rs[ǉ
                else {
                    Debug.Assert(e.ElementCount==1);
                    Debug.Assert(target.ElementCount>=1);
                    target.AddPositionElement(e.LastTradeItem.Clone()); //݂Elementǉ
                    //OrderRestriction͍Kv ЂƂłʏ킪Βʏ
                    if(e.OrderRestriction==OrderRestriction.None)
                        target.OrderRestriction = OrderRestriction.None;
                }
            }
            return r;
        }
        private void OnStyleChange(object sender, EventArgs args) {
            _style = (PositionListDisplayStyle)_displayStyleBox.SelectedIndex;
            BellagioRoot.FixedPreferences.Ordering.PositionListSortOption = _sortBox.SelectedIndex;
            _omitWarningCheckBox.Visible = _style==PositionListDisplayStyle.CombineSameStock;
            OrderDataElement<TradePositionList> v = _page==OrderAndPositionPage.Stocks?
                BellagioRoot.OrderDataProvider.StockPosition : BellagioRoot.OrderDataProvider.DerivativePosition;
            SetContent(v.Value);
        }
        private static int CodeComparison(TradePositionItem i1, TradePositionItem i2) {
            int d = CodeComparisonPrimitive(i1, i2);
            if(d!=0) return d;

            d = TypeComparisonPrimitive(i1, i2);
            if(d!=0) return d;

            d = ProfitComparisonPrimitive(i1, i2);
            if(d!=0) return d;

            return FinalComparisonPrimitive(i1, i2);
        }
        private static int ProfitComparison(TradePositionItem i1, TradePositionItem i2) {
            int d = ProfitComparisonPrimitive(i1, i2);
            if(d!=0) return d;

            d = TypeComparisonPrimitive(i1, i2);
            if(d!=0) return d;

            d = CodeComparisonPrimitive(i1, i2);
            if(d!=0) return d;

            return FinalComparisonPrimitive(i1, i2);
        }
        private static int TypeComparison(TradePositionItem i1, TradePositionItem i2) {
            int d = TypeComparisonPrimitive(i1, i2);
            if(d!=0) return d;

            d = CodeComparisonPrimitive(i1, i2);
            if(d!=0) return d;

            d = ProfitComparisonPrimitive(i1, i2);
            if(d!=0) return d;

            return FinalComparisonPrimitive(i1, i2);
        }
        private static int CodeComparisonPrimitive(TradePositionItem i1, TradePositionItem i2) {
            return i1.Stock.Profile.Code.CompareTo(i2.Stock.Profile.Code);
        }
        private static int TypeComparisonPrimitive(TradePositionItem i1, TradePositionItem i2) {
            return TypeMarkForComp(i1).CompareTo(TypeMarkForComp(i2));
        }
        private static int ProfitComparisonPrimitive(TradePositionItem i1, TradePositionItem i2) {
            return i2.TotalProfit.CompareTo(i1.TotalProfit); //v͍~
        }
        private static int FinalComparisonPrimitive(TradePositionItem i1, TradePositionItem i2) {
            string u1 = i1.GetPositionAt(0).UniqueID;
            string u2 = i2.GetPositionAt(0).UniqueID;
            if(u1==u2) //ł͂Ƃ
                return 0;
            else
                return u1.CompareTo(u2); 
        }
        private static int TypeMarkForComp(TradePositionItem item) {
            //AMpAMpA敨
            if(item.DealType==OrderDealType.Genbutsu)
                return 0;
            else if(OrderingUtil.IsShinnyo(item.DealType))
                return item.LongShort==LongShort.Long? 1 : 2;
            else
                return item.LongShort==LongShort.Long? 3 : 4;
        }
    }

    public enum OrderListDisplayStyle {
        All,
        NotCompleted, //̂
    }

    public class OrderListForm : OrderAndPositionListFormBase {
        private TradeOrderListDrawingSettings _drawingSettings;
        private DataThreadToMainThread _stockUpdateHandler;
        private DataThreadToMainThread _derivativeUpdateHandler;
        private OrderListDisplayStyle _style;
        private Comparison<TradeOrderItem>[] _comparisons;

        public OrderListForm() {
            this.Text = "Ɖ";
            _stockUpdateHandler = new DataThreadToMainThread(OnStockUpdate);
            _derivativeUpdateHandler = new DataThreadToMainThread(OnDerivativeUpdate);
            this.MinimumSize = new Size(600, 64);
        }
        public OrderDataElement<TradeOrderList> Document {
            get {
                return _page==OrderAndPositionPage.Stocks? BellagioRoot.OrderDataProvider.StockOrder : BellagioRoot.OrderDataProvider.DerivativeOrder;
            }
        }
        protected override void CreateContentControls() {
            _drawingSettings = BellagioRoot.FixedPreferences.Ordering.OrderListDrawingSettings;

            _displayStyleLabel = FormUtil.CreateLabel("\(&C)");
            _displayStyleBox = FormUtil.CreateComboBox("ׂ", "̂");
            _displayStyleBox.SelectedIndexChanged += new EventHandler(OnStyleChange);
            _style = OrderListDisplayStyle.All;

            _sortLabel = FormUtil.CreateLabel("(&O)");
            _sortBox = FormUtil.CreateComboBox("", "", "ԏ", "");
            _sortBox.SelectedIndex = RuntimeUtil.AdjustIntRange(BellagioRoot.FixedPreferences.Ordering.OrderListSortOption, 0, _sortBox.Items.Count-1);
            _sortBox.SelectedIndexChanged += new EventHandler(OnStyleChange);
            _comparisons = new Comparison<TradeOrderItem>[4] {
                new Comparison<TradeOrderItem>(OrderTimeComparison),
                new Comparison<TradeOrderItem>(CodeComparison),
                new Comparison<TradeOrderItem>(StatusComparison),
                new Comparison<TradeOrderItem>(TradeComparison)};

            OrderListControl c = new OrderListControl(this);
            c.ItemFormatter = BellagioRoot.OrderDataProvider.OrderItemFormatter;
            c.Settings = _drawingSettings;
            c.ClientSize = new Size(720, 300);

            _list = c;
        }
        protected override void RefreshInternalData() {
            if(_page==OrderAndPositionPage.Stocks)
                BellagioRoot.OrderDataProvider.StockOrder.Refresh();
            else
                BellagioRoot.OrderDataProvider.DerivativeOrder.Refresh();
        }
        protected override void OnLoad(EventArgs e) {
            base.OnLoad(e);
            SetPage(OrderAndPositionPage.Stocks);
        }
        protected override void AttachNotifier() {
            BellagioRoot.OrderDataProvider.StockOrder.AddListener(_stockUpdateHandler);
            BellagioRoot.OrderDataProvider.DerivativeOrder.AddListener(_derivativeUpdateHandler);
        }
        protected override void DetachNotifier() {
            BellagioRoot.OrderDataProvider.StockOrder.RemoveListener(_stockUpdateHandler);
            BellagioRoot.OrderDataProvider.DerivativeOrder.RemoveListener(_derivativeUpdateHandler);
        }
        private void OnStockUpdate() {
            if(_page!=OrderAndPositionPage.Stocks) return;
            RefreshBody(BellagioRoot.OrderDataProvider.StockOrder);
        }
        private void OnDerivativeUpdate() {
            if(_page!=OrderAndPositionPage.Derivatives) return;
            RefreshBody(BellagioRoot.OrderDataProvider.DerivativeOrder);
        }
        private void RefreshBody(OrderDataElement<TradeOrderList> elem) { 
            try {
                Debug.Assert(BUtil.IsExecutingInMainThread);
                SetContent(elem.Value);
                base.OnReceivedResponse();
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }
        //gZbgBnull̂Ƃ
        private void SetContent(TradeOrderList result) {
            //ۗ̂Ƃ̂ŏɐVCX^X
            if(result!=null) {
                TradeOrderList list = new TradeOrderList();
                TradeOrderList reserved = _page==OrderAndPositionPage.Stocks? BellagioRoot.OrderDataProvider.ReservedStockOrderList : BellagioRoot.OrderDataProvider.ReservedDerivativeOrderList;
                foreach(TradeOrderItem item in reserved) list.AddItem(item); //ۗRemoveΏۂɂȂ̂ŃN[Ă͂Ȃ

                foreach(TradeOrderItem item in result)
                    if(CheckFilterCondition(item)) list.AddItem(item.Clone());

                list.Sort(_comparisons[_sortBox.SelectedIndex]);

                result = list; //ꂪŏI
#if false
                //̓mF
                TradeOrderItem t = new TradeOrderItem(BellagioRoot.GlobalStockCollection.FindExact("5405").Primary);
                t.DealType = OrderDealType.Genbutsu;
                t.Condition = OrderCondition.Nariyuki;
                t.Volume = 10000;
                t.AddTradeItem(3600*9+60*10, 580, 3000);
                t.AddTradeItem(3600*9+60*12, 581, 2000);
                t.AddTradeItem(3600*9+60*14, 582, 5000);
                result.AddItem(t);
#endif
            }

            ((OrderListControl)_list).SetContent(result);
            _list.ResetContent();
        }
        //\Ƀ}b`Ătrue
        private bool CheckFilterCondition(TradeOrderItem item) {
            if(_style==OrderListDisplayStyle.All)
                return true;
            else {
                return !OrderingUtil.IsFinalStatus(item.OrderStatus, item.TransactionStatus);
            }
        }

        private void OnStyleChange(object sender, EventArgs args) {
            _style = (OrderListDisplayStyle)_displayStyleBox.SelectedIndex;
            BellagioRoot.FixedPreferences.Ordering.OrderListSortOption = _sortBox.SelectedIndex;

            OrderDataElement<TradeOrderList> v = _page==OrderAndPositionPage.Stocks?
                BellagioRoot.OrderDataProvider.StockOrder : BellagioRoot.OrderDataProvider.DerivativeOrder;
            SetContent(v.Value);
        }

        private static int OrderTimeComparison(TradeOrderItem i1, TradeOrderItem i2) {
            int d1 = i1.OrderDateAsInt;
            int d2 = i2.OrderDateAsInt;
            if(d1==0) d1 = BellagioRoot.DataSourceHost.ExternalEnvironment.CurrentDate;
            if(d2==0) d2 = BellagioRoot.DataSourceHost.ExternalEnvironment.CurrentDate;

            int d = d2.CompareTo(d1);
            if(d!=0)
                return d;
            else
                return i2.OrderTime.AsInt().CompareTo(i1.OrderTime.AsInt());
        }
        private static int CodeComparison(TradeOrderItem i1, TradeOrderItem i2) {
            int d = i1.Stock.Profile.Code.CompareTo(i2.Stock.Profile.Code);
            if(d!=0) return d;

            //R[hȂ玞
            return OrderTimeComparison(i1, i2);
        }
        private static int StatusComparison(TradeOrderItem i1, TradeOrderItem i2) {
            int d = i1.OrderStatus.CompareTo(i2.OrderStatus);
            if(d!=0) return d;

            return OrderTimeComparison(i1, i2);
        }
        private static int TradeComparison(TradeOrderItem i1, TradeOrderItem i2) {
            int d = TradeTypeValueForComp(i1).CompareTo(TradeTypeValueForComp(i2));
            if(d!=0) return d;

            return OrderTimeComparison(i1, i2);
        }
        private static int TradeTypeValueForComp(TradeOrderItem item) {
            //rp̐lB/AMpVK/AMpϔ/A敨VK/A敨ϔ/̏ɂȂ悤
            bool close = item.NewOrClose==OrderNewClose.Close;
            bool long_ = item.LongShort==LongShort.Long;
            if(item.DealType==OrderDealType.Genbutsu)
                return close? 1 : 0;
            else  {
                int b = OrderingUtil.IsShinnyo(item.DealType)? 2 : 6;
                return b + (close? (long_? 3 : 2) : (long_? 0 : 1));
            }
        }
    }
}
