/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Screening/AutoTradingExecution.cs#3 $
 * $DateTime: 2008/05/14 13:05:12 $
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;

using Bellagio.Data;
using Bellagio.Values;
using Bellagio.Evaluators;
using Bellagio.Environment;

namespace Bellagio.Screening {
    internal class AutoTradingQuery {
        public enum MaxDayCountActionT {
            Ignore, //DayCount`FbNȂ
            Timeout, //^CAEgƂĖ
            ProfitOrLoss, //vEƂĉZ
        }

        public class Condition {
            public ConditionType type;
            public StandAloneEvaluator condition;
            public StandAloneEvaluator sashine; //wl܂͋twlBgȂƂnull
        }
        public class Column { //ǉJ
            public string label;
            public StandAloneEvaluator evaluator;
            public NumberFormatter formatter;
            public Column(string l, StandAloneEvaluator e, NumberFormatter f) {
                label = l;
                evaluator = e;
                formatter = f;
            }
        }
        private AutoTradingSchema _source;
        private string _title;
        private LongShort _longShort;
        private StandAloneEvaluator _filter;
        private Condition _entry;
        private Condition _exit;
        private Condition _losscut;
        private Column[] _entryColumns;
        private Column[] _exitColumns;
        private AbstractStockProfile _targetStock; //null̂Ƃ͑S
        private int _originDate; //؊JntAIt
        private int _lastDate;
        private int _limitSignalCount;
        private MaxDayCountActionT _maxDayCountAction;
        private int _maxDayCount; //PVOǐ؂pő
        private ScreeningFilterType _staticFilter;

        public AutoTradingSchema SourceSchema {
            get {
                return _source;
            }
        }

        public string Title {
            get {
                return _title;
            }
        }
        public LongShort LongShort {
            get {
                return _longShort;
            }
        }
        public Condition Entry {
            get {
                return _entry;
            }
        }
        public Condition Exit {
            get {
                return _exit;
            }
        }
        public Condition Losscut {
            get {
                return _losscut;
            }
        }
        public StandAloneEvaluator Filter {
            get {
                return _filter;
            }
        }
        public int OriginDate {
            get {
                return _originDate;
            }
        }
        public int LastDate {
            get {
                return _lastDate;
            }
        }
        public int MaxDayCount {
            get {
                return _maxDayCount;
            }
        }
        public MaxDayCountActionT MaxDayCountAction {
            get {
                return _maxDayCountAction;
            }
        }
        public int LimitSignalCount {
            get {
                return _limitSignalCount;
            }
        }
        public AbstractStockProfile TargetStockProfile {
            get {
                return _targetStock;
            }
        }
        public ScreeningFilterType StaticFilter {
            get {
                return _staticFilter;
            }
        }
        public Column[] EntryAdditionalColumns {
            get {
                return _entryColumns;
            }
        }
        public Column[] ExitAdditionalColumns {
            get {
                return _exitColumns;
            }
        }

        //ƁA͈ؔ́BPSԁASZԁAԂ̓tȂ
        public void ConstructWithSchema(AutoTradingSchema autotrading) {
            AutoTradingEvalContext.RegisterFunctions(); //ŏɃp[XƂɕKv

            _source = autotrading;

            StringWithParameterParser string_parser = new StringWithParameterParser(autotrading);
            NumberFormatter defaultformatter = new NumberFormatter("");

            _title = string_parser.Parse(autotrading.title);
            _longShort = AutoTradingSchema.ParseLongShort(autotrading.longshort);
            _filter = autotrading.filter.IsOmitted? null : EvaluatorBuilderFromText.BuildStandAlone(autotrading.filter, autotrading, QuoteType.instance, LocalVariable.EmptyArray, BV.EmptyArray);

            if(autotrading.entry==null) throw new FormatException("entryvfKvł");
            _entry = ParseCondition(autotrading, autotrading.entry);
            if(autotrading.losscut==null) throw new FormatException("losscutvfKvł");
            _losscut = ParseCondition(autotrading, autotrading.losscut);
            if(autotrading.exit==null) throw new FormatException("exitvfKvł");
            _exit = ParseCondition(autotrading, autotrading.exit);

            if(autotrading.maxDayCount.IsOmitted)
                _maxDayCountAction = MaxDayCountActionT.Ignore;
            else {
                _maxDayCount = autotrading.maxDayCount.ParseMandatoryInt();
                _maxDayCountAction = autotrading.maxDayCountAction.ParseOptionalString("")=="timeout"? MaxDayCountActionT.Timeout : MaxDayCountActionT.ProfitOrLoss;
            }

            _entryColumns = new Column[autotrading.entryColumn.Count];
            for(int i=0; i<_entryColumns.Length; i++)
                _entryColumns[i] = ParseColumn(autotrading, autotrading.entryColumn[i]);
            _exitColumns = new Column[autotrading.exitColumn.Count];
            for(int i=0; i<_exitColumns.Length; i++)
                _exitColumns[i] = ParseColumn(autotrading, autotrading.exitColumn[i]);

        }
        private Column ParseColumn(AutoTradingSchema autotrading, AutoTradingAdditionalColumn col) {
            EvaluatorBuildContext bc = new EvaluatorBuildContext(autotrading);
            bc.Initialize(QuoteType.instance, LocalVariable.EmptyArray);
            StringWithParameterParser string_parser = new StringWithParameterParser(autotrading);
            string label = string_parser.Parse(col.label);

            IEvaluator ev = EvaluatorBuilderFromText.Build(col.expression, bc);
            if(ev.BT!=BT.Double && ev.BT!=BT.Int) throw new FormatException(String.Format("{0} ^͐lłKv܂B", col.expression));
            NumberFormatter formatter = NumberFormatter.Parse(col.format, null);
            if(formatter==null) formatter = new NumberFormatter("");

            return new Column(label, new StandAloneEvaluator(ev, bc, BV.EmptyArray), formatter);
        }

        public void SetParameters(int originDate, int lastDate, int limitSignalCount, ScreeningFilterType staticFilter, AbstractStockProfile target) {
            _targetStock = target;
            _originDate = originDate;
            _lastDate = lastDate;
            _limitSignalCount = limitSignalCount;
            _staticFilter = staticFilter;
        }

        private Condition ParseCondition(AutoTradingSchema at, AutoTradingCondition cs) {

            Condition c = new Condition();
            c.type = AutoTradingSchema.ParseConditionType(cs.type);
            c.condition = EvaluatorBuilderFromText.BuildStandAlone(cs.condition, at, QuoteType.instance, LocalVariable.EmptyArray, BV.EmptyArray);
            BT t = c.condition.BT;
            if(t!=BT.Bool) throw new FormatException(String.Format("{0}bool^̎Kv܂", cs.condition.ParseMandatoryString()));
            if(c.type==ConditionType.Sashine || c.type==ConditionType.Gyakusashine) {
                c.sashine = EvaluatorBuilderFromText.BuildStandAlone(cs.sashine, at, QuoteType.instance, LocalVariable.EmptyArray, BV.EmptyArray);
                t = c.sashine.BT;
                if(t!=BT.Int && t!=BT.Double) throw new FormatException(String.Format("{0}͐l^̎Kv܂", cs.sashine.ParseMandatoryString()));
            }

            return c;
        }

    }

    internal class AutoTradingExecution : ScreeningExecutorBase {
        private AutoTradingQuery _query; //\[gΏ
        private AutoTradingResult _result;

        public AutoTradingExecution(AutoTradingQuery query)
            : base() {
            _query = query;
        }
        public AutoTradingQuery Query {
            get {
                return _query;
            }
        }
        public AutoTradingResult Result {
            get {
                return _result;
            }
        }

        protected override void BeforeExecute(int count) {
            base.BeforeExecute(count);
            _result = new AutoTradingResult(_query);
            _singleTargetStock = _query.TargetStockProfile;
            _firstFilter = _query.TargetStockProfile==null? ScreeningFirstFilter.NoFilter : ScreeningFirstFilter.SingleStock;
        }
        protected override void AfterExecute(List<AbstractStockProfile> insufficients, List<AbstractStockProfile> filtered, List<AbstractStockProfile> error) {
            _result.SetExcludedStocks(insufficients, filtered, error);
            _result.Sort();
            _result.CalcResults();
        }

        protected override ExecuteBrandResult ExecuteBrand(AbstractStockProfile prof_) {
            if(_result.Count >= _query.LimitSignalCount) {
                _result.HasTooManyResults = true;
                return ExecuteBrandResult.Stop;
            }

            BasicStockProfile prof = prof_ as BasicStockProfile;
            if(prof==null) return ExecuteBrandResult.Ignored; //hȂǂ͖@XN[jOƈႢAp~͖Ȃ

            if(_query.TargetStockProfile!=null) {
                if(prof!=_query.TargetStockProfile) return ExecuteBrandResult.Ignored; //1ΏۂȂȂ烋[v񂷂͖̂ʂǂ
            }
            else {
                if(ExcludesByStaticFilter(prof, _query.StaticFilter)) return ExecuteBrandResult.Ignored;
            }

            int origin_date = _query.OriginDate;
            ConcreteQuote quote = ScreeningPlugin.Instance.DailyDataCache.Request(prof);
            if(quote==null || quote.Count==0 || quote.LastCandle.Time < origin_date) { //Ō̓OȂ̂͂ŏO
                return ExecuteBrandResult.InsufficientData;
            }

            Debug.Assert(quote.Unit==Quote.QuoteUnit.Daily);

            //origin_date̓t͂ł邾擾悤ɂ΂
            while(!HasSufficientData(quote,origin_date)) {
                if(!ScreeningPlugin.Instance.DailyDataCache.ExtendDataIfPossible(prof, ref quote)) break;
            }

            if(_query.Filter!=null) {
                EvalContext context = new EvalContext(quote);
                BV filter = _query.Filter.Eval(context);
                if(filter.IsNil || ((BBoolean)filter).Value==false)
                    return ExecuteBrandResult.Filtered;
            }

            int index = quote.FindNearestFuture(origin_date);
            
            int last_index = quote.FindNearestFuture(_query.LastDate);
            if(last_index==-1) last_index = quote.Count-1; //tȂ΍Ō܂

            //؃[v
            SignalResult current = null; //݌ؒ̃VOiBԂȂnull
            int last_raised_index = 0;

            //TODO ̒ExtendK؂ɂƂ̂ĂȂ
            while(index < last_index) {
                int today = quote.CandleAt(index).Time;

                if(current==null) { //VKVOi
                    bool buying = _query.LongShort==LongShort.Long;
                    EvalContext context = new EvalContext(new SubseqQuote(quote, 0, index+1));
                    context.CurrentStock = new StockRef(prof.Primary);
                    BBoolean v = _query.Entry.condition.Eval(context) as BBoolean;
                    if(v!=null && !v.IsNil && v.Value) {
                        int date = 0, price = 0;
                        if(CalcTradePrice(quote, index, _query.Entry, context, buying, out date, out price)) {
                            current = new SignalResult(_query, prof, date, price, context);
                            last_raised_index = index;
                        }
                    }
                }
                else { //VOi
                    if(_query.MaxDayCountAction!=AutoTradingQuery.MaxDayCountActionT.Ignore && index==last_raised_index + _query.MaxDayCount) { //P[XP@^CAEg
                        if(_query.MaxDayCountAction==AutoTradingQuery.MaxDayCountActionT.Timeout)
                            current.CloseAndInvalidate(SignalResultType.Timeout, today);
                        else { //nlőޏo
                            Candle tomorrow = quote.CandleAtOrNull(index+1);
                            if(tomorrow==null)
                                current.CloseAndInvalidate(SignalResultType.Timeout, today);
                            else {
                                bool win = _query.LongShort==LongShort.Short ^ tomorrow.Open > current.SignalPrice;
                                current.Close(win? SignalResultType.Exit : SignalResultType.Losscut, today, tomorrow.Open, index - last_raised_index, null); //^CAEgN̑ޏo͒ǉJ̌vZȂ
                            }
                        }
                        _result.AddResult(current);
                        current = null;
                    }
                    else { //^CAEgȊO
                        bool buying = _query.LongShort==LongShort.Short; //̌ςȂ̂

                        AutoTradingEvalContext context = new AutoTradingEvalContext(new SubseqQuote(quote, 0, index+1), current.SignalPrice, index - last_raised_index); //wtŎ擾;
                        context.CurrentStock = new StockRef(prof.Primary);

                        BBoolean v = _query.Exit.condition.Eval(context) as BBoolean;
                        bool exit = v!=null && !v.IsNil && v.Value;
                        v = _query.Losscut.condition.Eval(context) as BBoolean;
                        bool losscut = v!=null && !v.IsNil && v.Value; //܂̏

                        //wlɈ|ȂΖȂ̂łɃ`FbN
                        int exit_date = 0, exit_price = 0;
                        int losscut_date = 0, losscut_price = 0;
                        if(exit) exit = CalcTradePrice(quote, index, _query.Exit, context, buying, out exit_date, out exit_price);
                        if(losscut) losscut = CalcTradePrice(quote, index, _query.Losscut, context, buying, out losscut_date, out losscut_price);

                        //łAЕ̓PAЕłΓD
                        if(exit && losscut) {
                            if(IsTodayType(_query.Exit.type) && !IsTodayType(_query.Losscut.type)) losscut = false;
                            else if(!IsTodayType(_query.Exit.type) && IsTodayType(_query.Losscut.type)) exit = false;
                        }

                        if(exit && losscut) //P[XQ@d
                            current.CloseAndInvalidate(SignalResultType.Duplicated, today);
                        else if(exit) //P[XR@vm
                            current.Close(SignalResultType.Exit, today, exit_price, index - last_raised_index, context);
                        else if(losscut) //P[XS@XJbg
                            current.Close(SignalResultType.Losscut, today, losscut_price, index - last_raised_index, context);

                        //ƂfalseɂȂČpꍇ
                        if(current.ResultType!=SignalResultType.None) {
                            _result.AddResult(current);
                            current = null;
                        }

                    }
                }

                index++;
            }

            //`FbNIĎc͕ʓr
            if(current!=null) {
                current.CloseAndInvalidate(SignalResultType.EndOfData, quote.LastCandle.Time);
                _result.AddResult(current);
            }

            return ExecuteBrandResult.Succeeded;
        }
        private bool HasSufficientData(ConcreteQuote q, int origin_date) {
            //origin_date̓tŎwWvZłȂƂȂ̂Œ߂ɂƂB
            //ȂAꂪmۂłȂƎsȂ̂ł͂ȂAꂪ邩x̒̓Ƃ܂ł΂AƂ菇
            Candle c = q.CandleAtOrNull(100); 
            return c!=null && c.Time<=origin_date;
        }

        //ɏ]āAtƒlivZBwlɈȂŖ肵ȂƂfalse
        //t̓VOiEg[ĥǂ炩ݒɏ]ĕԂ
        private bool CalcTradePrice(Quote quote, int index, AutoTradingQuery.Condition condition, EvalContext sashine_context, bool buying, out int date, out int price) {
            Candle c;
            date = 0;
            price = 0;
            int signal_date = quote.CandleAt(index).Time;
            AutoTradingDateDisplay disp = ScreeningPlugin.Instance.Preferences.AutoTradingDateDisplay;

            if(condition.type==ConditionType.TodayClose) {
                c = quote.CandleAt(index);
                date = c.Time; //VOiƈꏏ
                price = c.Close;
                return true;
            }
            else if(condition.type==ConditionType.TomorrowOpen) {
                c = quote.CandleAtOrNull(index+1);
                if(c==null) return false;
                date = disp==AutoTradingDateDisplay.Condition? signal_date : c.Time;
                price = c.Open;
                return true;
            }
            else {
                c = quote.CandleAtOrNull(index+1);
                if(c==null) return false;
                BV sashine_ = condition.sashine.Eval(sashine_context); //wl𓾂B
                if(sashine_.IsNil) return false;
                int sashine = AsInt(sashine_);
                if(SashineHit(c, condition.type, sashine, buying, out price)) {
                    date = disp==AutoTradingDateDisplay.Condition? signal_date : c.Time;
                    return true;
                }
                else
                    return false;
            }
        }

        //wlEtwlŃqbgǂmF
        private bool SashineHit(Candle c, ConditionType type, int sashine, bool buying, out int price) {
            //twĺAw肵l܂ŕsȕɐi񂾂甭AƂ`݂̂z肵ĂB
            //twl̒l s ƂƂ̔̋(sȏŔ)́A
            //  s > l : 肵Ȃ@@l >= s > nl : s Ŗ  nl >= s : nlŖ
            //蒍(sȉŔ)ł
            //  s < l : 肵Ȃ@@l <= s < nl : s Ŗ  nl <= s : nlŖ
            
            //łɒʏwlƁA
            // s ł̔wl@twl̔Ɠ
            // s ł̔wl@twl̔Ɠ

            Debug.Assert(type==ConditionType.Gyakusashine || type==ConditionType.Sashine);
            if(type==ConditionType.Sashine ^ buying) {
                if(sashine > c.High) {
                    price = 0;
                    return false;
                }
                else if(sashine > c.Open) {
                    price = sashine;
                    return true;
                }
                else {
                    price = c.Open;
                    return true;
                }
            }
            else {
                if(sashine < c.Low) {
                    price = 0;
                    return false;
                }
                else if(sashine < c.Open) {
                    price = sashine;
                    return true;
                }
                else {
                    price = c.Open;
                    return true;
                }
            }
        }

        private static int AsInt(BV v) {
            BDouble bd = v as BDouble;
            if(bd!=null) return (int)bd.Value;

            BInt bi = v as BInt;
            if(bi!=null) return bi.Value;

            return 0;
        }
        private static bool IsTodayType(ConditionType ct) {
            return ct==ConditionType.TodayClose;
        }
    }

    //P̌ʁ@ǂ̏ďI
    internal enum SignalResultType {
        None, //
        Exit, //vo
        Losscut, //XJbg
        Duplicated, //߃m[JEg
        Timeout, //ǂȂ܂܋Ko
        EndOfData, //`FbNɃf[^I[
    }
    internal class SignalResult : ScreeningResultItemBase {
        private AutoTradingQuery _query;
        private SignalResultType _result;
        private int _signalDate;
        private int _signalPrice;
        private int _exitDate;
        private int _exitPrice;
        private int _dateLength;
        private BV[] _entryValues;
        private BV[] _exitValues; //gnull̂Ƃ(^CAEg)

        public SignalResult(AutoTradingQuery query, AbstractStockProfile stock, int date, int price, EvalContext context) : base(stock) {
            _query = query;
            _signalDate = date;
            _signalPrice = price;
            _entryValues = new BV[query.EntryAdditionalColumns.Length];
            _result = SignalResultType.None;
            for(int i=0; i<_entryValues.Length; i++) {
                BV v = null;
                BV.Let(ref v, query.EntryAdditionalColumns[i].evaluator.Eval(context));
                _entryValues[i] = v;
            }
            //exit͔z񂾂
            _exitValues = new BV[query.ExitAdditionalColumns.Length];
        }
        //l̊m肵N[Y
        public void Close(SignalResultType result, int date, int price, int dateLength, EvalContext context) {
            Debug.Assert(result==SignalResultType.Exit || result==SignalResultType.Losscut);
            _result = result;
            _exitDate = date;
            _exitPrice = price;
            _dateLength = dateLength;
            if(context!=null) {
                for(int i=0; i<_exitValues.Length; i++) {
                    BV v = null;
                    BV.Let(ref v, _query.ExitAdditionalColumns[i].evaluator.Eval(context));
                    _exitValues[i] = v;
                }
            }
        }
        //ȃP[Xŕ
        public void CloseAndInvalidate(SignalResultType result, int date) {
            Debug.Assert(result==SignalResultType.Duplicated || result==SignalResultType.Timeout || result==SignalResultType.EndOfData);
            _result = result;
            _exitDate = date;
        }

        public SignalResultType ResultType {
            get {
                return _result;
            }
        }
        public int SignalDate {
            get {
                return _signalDate;
            }
        }
        public int SignalPrice {
            get {
                return _signalPrice;
            }
        }
        public int ExitDate {
            get {
                return _exitDate;
            }
        }
        public int ExitPrice {
            get {
                return _exitPrice;
            }
        }
        public int DateLength {
            get {
                return _dateLength;
            }
        }
        public BV[] EntryAdditionalValues {
            get {
                return _entryValues;
            }
        }
        public BV[] ExitAdditionalValues {
            get {
                return _exitValues;
            }
        }

        //ȌʂłtrueԂėvperformanceɁAxȂfalseԂēemessageɓ
        public bool CheckResult(ref double performance, ref string message) {
            if(_result==SignalResultType.Exit || _result==SignalResultType.Losscut) {
                if(_signalPrice==_exitPrice) {
                    message = "l";
                    return false;
                }
                else if(_result==SignalResultType.Exit) {
                    if(_query.LongShort==LongShort.Long ^ _signalPrice<_exitPrice) {
                        message = "vmő";
                        return false;
                    }
                    else {
                        performance = ((double)(_exitPrice - _signalPrice)) / _signalPrice;
                        if(_query.LongShort==LongShort.Short) performance = -performance;
                        Debug.Assert(performance > 0);
                        return true;
                    }
                }
                else { //if(_result==SignalResultType.Losscut) {
                    if(_query.LongShort==LongShort.Long ^ _signalPrice>_exitPrice) {
                        message = "XJbgŗv";
                        return false;
                    }
                    else {
                        performance = ((double)(_exitPrice - _signalPrice)) / _signalPrice;
                        if(_query.LongShort==LongShort.Short) performance = -performance;
                        Debug.Assert(performance < 0);
                        return true;
                    }
                }
            }
            else if(_result==SignalResultType.Duplicated) {
                message = "vmEXJbgɔ";
                return false;
            }
            else if(_result==SignalResultType.Timeout) {
                message = "KԌo";
                return false;
            }
            else {
                Debug.Assert(_result==SignalResultType.EndOfData);
                message = "ʔOɃf[^I[";
                return false;
            }
                    
        }

        //\[gp֐
        public static Comparison<SignalResult> dateComparison = delegate(SignalResult r1, SignalResult r2) {
            return r1.SignalDate - r2.SignalDate;
        };
        public static Comparison<SignalResult> stockComparison = delegate(SignalResult r1, SignalResult r2) {
            return r1.StockProfile.Code.CompareTo(r2.StockProfile.Code);
        };
        public static Comparison<SignalResult> performanceComparison = delegate(SignalResult r1, SignalResult r2) {
            double p1 = 0, p2 = 0;
            string m1 = "", m2 = "";
            if(r1.CheckResult(ref p1, ref m1) && r2.CheckResult(ref p2, ref m2))
                return p2.CompareTo(p1); //ptH[}X͍~(т擪)
            else
                return r1.ResultType.CompareTo(r2.ResultType); //ʏ̌ʂłȂΎޏ̂
        };
    }

    internal class AutoTradingResult : ScreeningResultBase<SignalResult> {
        private AutoTradingQuery _query;
        private bool _overflow;
        private Comparison<SignalResult> _comparison;

        public AutoTradingResult(AutoTradingQuery q) {
            _query = q;
        }
        protected override Comparison<SignalResult> GetSortComparison() {
            return _comparison;

        }
        public AutoTradingQuery Query {
            get {
                return _query;
            }
        }
        public bool HasTooManyResults {
            get {
                return _overflow;
            }
            set {
                _overflow = value;
            }
        }
        public Comparison<SignalResult> Comparison {
            get {
                return _comparison;
            }
            set {
                _comparison = value;
            }
        }

        //ȉAv
        //vpeBɂ܂łȂ̂public
        public double _avgTradeLength; //ωԃg[h
        public int _exitCount;  //s\̂winCount,loseCount𑫂ĂResultCountɂ͈vȂ
        public int _losscutCount;
        public int _invalidCount; //
        public int _totalCount;
        public double _avgExit; //ςƕW΍
        public double _sdevExit;
        public double _avgExitTradeLength;
        public double _avgLosscut;
        public double _sdevLosscut;
        public double _avgLosscutTradeLength;
        public double _avgPerformance;
        public double _totalPerformance;
        public SignalResult _maxExit;
        public SignalResult _maxLosscut;

        public void CalcResults() {
            Phase1();
            Phase2();
        }
        private void Phase1() {
            int total_length = 0, total_exit_length = 0, total_losscut_length = 0;
            double sum_exit = 0, sum_losscut = 0;
            double max_performance = Double.MinValue, min_performance = Double.MaxValue;
            string msg = "";
            for(int i=0; i<this.Count; i++) {
                SignalResult sr = this[i];
                double p = 0;
                if(sr.CheckResult(ref p, ref msg)) {
                    if(sr.ResultType==SignalResultType.Exit) {
                        total_length += sr.DateLength;
                        total_exit_length += sr.DateLength;
                        _totalCount++;
                        _exitCount++;
                        sum_exit += p;
                        _totalPerformance += p;
                        if(p > max_performance) {
                            max_performance = p;
                            _maxExit = sr;
                        }
                    }
                    else if(sr.ResultType==SignalResultType.Losscut) {
                        total_length += sr.DateLength;
                        total_losscut_length += sr.DateLength;
                        _totalCount++;
                        _losscutCount++;
                        sum_losscut += p;
                        _totalPerformance += p;
                        if(p < min_performance) {
                            min_performance = p;
                            _maxLosscut = sr;
                        }
                    }
                }
                else
                    _invalidCount++;
            }

            if(_totalCount   !=0) _avgTradeLength        = (double)total_length / _totalCount;
            if(_exitCount    !=0) _avgExit               = sum_exit / _exitCount;
            if(_losscutCount !=0) _avgLosscut            = sum_losscut / _losscutCount;
            if(_exitCount    !=0) _avgExitTradeLength    = total_exit_length / _exitCount;
            if(_losscutCount !=0) _avgLosscutTradeLength = total_losscut_length / _losscutCount;
            if(_totalCount   !=0) _avgPerformance        = _totalPerformance / _totalCount;
        }
        private void Phase2() {
            double exit_sd_sum = 0, losscut_sd_sum = 0, total_sd_sum = 0;
            string msg = "";
            for(int i=0; i<this.Count; i++) {
                SignalResult sr = this[i];
                double p = 0;
                if(sr.CheckResult(ref p, ref msg)) {
                    if(sr.ResultType==SignalResultType.Exit) {
                        double d = p - _avgExit;
                        exit_sd_sum += d*d;
                        d = p - _avgPerformance;
                        total_sd_sum += d*d;
                    }
                    else if(sr.ResultType==SignalResultType.Losscut) {
                        double d = p - _avgLosscut;
                        losscut_sd_sum += d*d;
                        d = p - _avgPerformance;
                        total_sd_sum += d*d;
                    }
                }
            }

            if(_exitCount!=0) _sdevExit    = Math.Sqrt(exit_sd_sum / _exitCount);
            if(_losscutCount!=0) _sdevLosscut = Math.Sqrt(losscut_sd_sum / _losscutCount);
        }

    }

    internal class AutoTradingEvalContext : EvalContext {
        private int _entryPrice;
        private int _dateCount;
        private static bool _functionRegistered;

        public AutoTradingEvalContext(Quote quote, int entryPrice, int dateCount)
            : base(quote) {

            _entryPrice = entryPrice;
            _dateCount = dateCount ;
        }
        public static void RegisterFunctions() {
            if(!_functionRegistered) {
                //VOiؒ̂ݎg֐̓o^
                FunctionLibrary lib = BellagioRoot.Functions.BuiltIn;
                lib.DefineInternalFunction("entry", null, BT.Int, BuiltInFunctionLibrary.noArg, new BInternalExecution(GetEntryPrice));
                lib.DefineInternalFunction("daycount", null, BT.Int, BuiltInFunctionLibrary.noArg, new BInternalExecution(GetDayCount));

                _functionRegistered = true;
            }
        }

        private static ExecResult GetEntryPrice(BV target, EvalContext extra, BV[] args, BV result) {
            AutoTradingEvalContext nc = extra as AutoTradingEvalContext;
            if(nc==null)
                return ExecResult.Nil;
            else
                ((BInt)result).Value = nc._entryPrice;
            return ExecResult.OK;
        }
        private static ExecResult GetDayCount(BV target, EvalContext extra, BV[] args, BV result) {
            AutoTradingEvalContext nc = extra as AutoTradingEvalContext;
            if(nc==null)
                return ExecResult.Nil;
            else
                ((BInt)result).Value = nc._dateCount;
            return ExecResult.OK;
        }
    }

}
