/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Values/Quote.cs#10 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * S{lAȂǂ̃IuWFNg
 */
using System;
using System.Diagnostics;

using System.Collections;
using System.Collections.Generic;
using System.Text;

using Bellagio.Evaluators;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Values {


    /*
    󋵂ł̑ɂ
    yz
    @Ô͎݃ZbgŁANilłB
    @Ƃ́AôȂ́AliOliAoO̕ʂ̑łB
    yz
    @ôȂƂ͍ŏICzACzlȂƂ͑ȌIlp
    */

    //
    public abstract class Quote : BArray {
        //̎
        public enum QuoteUnit {
            Minutely, Daily, Weekly, Monthly
        }

        public Quote() : base(QuoteType.instance) {
        }
        public override BT BT {
            get {
                return QuoteType.instance;
            }
        }
        public override void Format(BVFormatter formatter) {
            formatter.Quote(this);
        }
        public override BV this[int index] {
            get {
                return CandleAt(index);
            }
        }
        public override void LetAt(int index, BV value) {
            Debug.Assert(false, "unimplemented");
        }
        public abstract QuoteUnit Unit {
            get;
        }
        public abstract int Scale {
            get;
        }
        public abstract int GetLatestIndexWithEvalContext(EvalContext env); //ŏȊindexԂBConcreteQuotełenvɈˑEtKv邱Ƃɒ

        //QT ł͔zꔭň邱Ƃ...
        public int FindExactly(int time) {
            Debug.Assert(!this.IsNil);
            if(this.Count==0) return -1;

            return InternalFind(0, this.Count, time, true);
        }
        public int FindNearestFuture(int time) { //timeȏɂȂtōŏ̂̂Ԃ
            Debug.Assert(!this.IsNil);
            if(this.Count==0) return -1;

            int r = InternalFind(0, this.Count, time, false);
            return r==this.Count? -1 : r;
        }
        private int InternalFind(int start, int end, int time, bool correspond_required) {
            Debug.Assert(start < end);
            if(start+1==end) { //Pɍiꂽ
                int t = CandleAt(start).Time;
                if(correspond_required)
                    return t==time? start : -1; //vOK
                else
                    return t>=time? start : start+1;
            }
            else {
                int mid = (start + end) / 2;
                Debug.Assert(start < mid && mid < end);
                int t = CandleAt(mid).Time;
                if(t > time)
                    return InternalFind(start, mid, time, correspond_required);
                else
                    return InternalFind(mid, end, time, correspond_required);
            }
        }
        public abstract Candle CandleAt(int index);

        public Candle LastCandle {
            get {
                int c = this.Count;
                return c==0? null : CandleAt(c-1);
            }
        }
        public Candle CandleAtOrNull(int index) {
            if(index >=0 && index < this.Count)
                return CandleAt(index);
            else
                return null;
        }

    }


    //ۂɃf[^Bǉł̂͂B폜͕s
    public class ConcreteQuote : Quote {
        private List<Candle> _data;
        private QuoteUnit _unit;
        private int _scale; //ꂪƂ1,3,5ȂPARATAɂȂ

        public ConcreteQuote(QuoteUnit unit, int scale) {
            _data = new List<Candle>();
            _unit = unit;
            _scale = scale;
        }
        public ConcreteQuote(QuoteUnit unit, int scale, List<Candle> body) {
            _data = body;
            _unit = unit;
            _scale = scale;
            for(int i=0; i<body.Count; i++) {
                body[i].SetQuoteAndIndex(this, i);
            }
        }
        public ConcreteQuote(QuoteUnit unit)
            : this(unit, 1) {
        }
        public override int Count {
            get {
                return _data.Count;
            }
        }
        public override bool IsNil {
            get {
                return false;
            }
        }
        public override void Let(BV value) {
            Quote q = (Quote)value;
            Copy(q, 0, this, q.Count);
        }
        public override QuoteUnit Unit {
            get {
                return _unit;
            }
        }
        public override int Scale {
            get {
                return _scale;
            }
        }
        public void SetCapacity(int capacity) {
            _data.Capacity = capacity;
        }

        public Candle Add(int time, int open, int high, int low, int close, int volume, int creditlong, int creditshort) {
            Candle y = new Candle();
            y.SetQuoteAndIndex(this, _data.Count);
            y.Time = time;
            y.Open = open;
            y.High = high;
            y.Low = low;
            y.Close = close;
            y.Volume = volume;
            y.CreditLong = creditlong;
            y.CreditShort = creditshort;
            return Add(y);
        }
        public Candle Add(int time, int price, int volume) {
            return Add(time, price, price, price, price, volume, 0, 0);
        }
        public Candle Add(Candle y) {
            //Debug.Assert(_data.Count==0 || _data[_data.Count-1].Time < y.Time); //Ԃ͑BAssertłȂIɗÔ
            y.SetQuoteAndIndex(this, _data.Count);
            _data.Add(y);
            return y;
        }
        public void AddAllX(Quote q) {
            for(int i=0; i<q.Count; i++)
                _data.Add(q.CandleAt(i));
        }
        public override Candle CandleAt(int index) {
            Debug.Assert(index >= 0);
            Debug.Assert(index < _data.Count, String.Format("_data.Count={0} index={1}", _data.Count, index));
            return _data[index];
        }
        public override int GetLatestIndexWithEvalContext(EvalContext env) {
            //{ɂł̂͂Ƌ^₾
            switch(_unit) {
                case QuoteUnit.Minutely:
                    return env.CurrentTime==null? _data.Count-1 : FindExactly(env.CurrentTime.AsInt());
                default:
                    return env.CurrentDate==null? _data.Count-1 : FindExactly(env.CurrentDate.AsInt());
            }
        }


        //ȉjIȂ̂ňɒ
        public void Clear() {
            _data.Clear();
        }
        //dest͒lengthɂȂ
        public static void Copy(Quote src, int startIndex, ConcreteQuote dest, int length) {
            dest.Clear();
            for(int i=0; i<length; i++)
                dest._data.Add(src.CandleAt(startIndex + i));
        }
        public void SetDirect(int index, Candle y) {
            _data[index] = y;
        }
        public void SetSize(int size) {
            //nullŖ߂B܂jIȂ̂ň͂ɒ
            while(_data.Count < size)
                _data.Add(null);
        }
    }

    //ʂ̑̏I_̂ݕύX́BςĂEvaluatorȂǂł͕Kv
    public class SubseqQuote : Quote {
        private Quote _basePrices;
        private int _startPos;
        private int _length;

        //private static int _ddInstanceCount;
        //private int _ddId;

        public SubseqQuote(Quote q, int startPos, int length) {
            Set(q, startPos, length);
            //_ddId = _ddInstanceCount++;
        }
        public override bool IsNil {
            get {
                return _basePrices==null;
            }
        }
        public void Set(Quote q) {
            Set(q, 0, q.Count);
        }
        public void Set(Quote q, int startPos, int length) {
            SubseqQuote sq = q as SubseqQuote;

            //݂݂ //TODO SubArrayłl̂Ƃł
            while(sq!=null) {
                startPos += sq._startPos;
                q = sq._basePrices;
                sq = q as SubseqQuote;
            }

#if DEBUG
            //ŐڑzĂƂЂǂƂɂȂ̂Ń`FbN
            while(sq!=null) {
                Debug.Assert(!Object.ReferenceEquals(sq, this));
                sq = sq._basePrices as SubseqQuote;
            }
#endif
            _basePrices = q;
            _startPos = startPos;
            _length = length;
        }
        public void SetLength(int len) {
            Debug.Assert(_startPos+len <= _basePrices.Count);
            _length = len;
        }
        public override int GetLatestIndexWithEvalContext(EvalContext env) {
            return _length-1;
        }
        public override int Count {
            get {
                return _length;
            }
        }
        public override void Let(BV value) {
            Quote q = (Quote)value;
            Set(q, 0, q.Count);
        }
        public override Candle CandleAt(int index) {
            Debug.Assert(index < _length);
            Candle y =  _basePrices.CandleAt(_startPos + index);
            return y;
        }
        public override Quote.QuoteUnit Unit {
            get {
                return _basePrices.Unit;
            }
        }
        public override int Scale {
            get {
                return _basePrices.Scale;
            }
        }
        public Quote BaseQuote {
            get {
                return _basePrices;
            }
        }
        public int StartPosition {
            get {
                return _startPos;
            }
        }
    }
    //
    public class QuoteType : BArrayObjectT {
        private QuoteType()
            : base("Quote", CandleType.instance) {
            _nilInstance = new SubseqQuote(null, 0, 0).MakeNil();
        }
        public override BV CreateInstance() {
            return new SubseqQuote(null, 0, 0); //őK{B܂肢Ƃ͂Ȃ
        }

        public static QuoteType instance;

        static QuoteType() {
            instance = new QuoteType();
        }

        public override void RegisterFunctionsTo(FunctionLibrary lib, SystemFunctionLibrary sys) {
            BT isQuotes = instance;

            //
            lib.DefineInternalFunction("lastCandle", isQuotes, CandleType.instance, FunctionLibrary.noArg, new BInternalExecution(LastCandle));
            lib.DefineInternalFunction("candleAt", isQuotes, CandleType.instance, FunctionLibrary.oneIntArg, new BInternalExecution(CandleAt));

            //TuXg
            lib.DefineInternalFunction("sublist", isQuotes, this, FunctionLibrary.oneIntArg, new BInternalExecution(SubList));
            lib.DefineInternalFunction("sublist", isQuotes, this, FunctionLibrary.twoIntArgs, new BInternalExecution(SubList));
            lib.DefineInternalFunction("last", isQuotes, this, FunctionLibrary.oneIntArg, new BInternalExecution(Last));
            lib.DefineInternalFunction("shift", isQuotes, this, FunctionLibrary.oneIntArg, new BInternalExecution(Shift));
            lib.DefineInternalFunction("shiftToTime", isQuotes, this, FunctionLibrary.oneIntArg, new BInternalExecution(ShiftToTime));

            //S{l{o{zBł͌ʂdelegateɂȂ邱Ƃɒ
            new Element(lib, "open", Candle.OpenDelegate, new BInternalExecution(PROpen));
            new Element(lib, "high", Candle.HighDelegate, new BInternalExecution(PRHigh));
            new Element(lib, "low", Candle.LowDelegate, new BInternalExecution(PRLow));
            new Element(lib, "close", Candle.CloseDelegate, new BInternalExecution(PRClose));
            new Element(lib, "volume", Candle.VolumeDelegate, new BInternalExecution(PRVolume));
            new Element(lib, "tradeAmount", Candle.TradeAmountDelegate, new BInternalExecution(PRTradeAmount));
            new Element(lib, "creditlong", Candle.CreditLongDelegate, new BInternalExecution(PRCreditLong));
            new Element(lib, "creditshort", Candle.CreditShortDelegate, new BInternalExecution(PRCreditShort));

            //悭g̃V[gJbg
            lib.DefineInternalFunction("ma", isQuotes, BT.Double, FunctionLibrary.oneIntArg, new BInternalExecution(PRMA)); // = close(x).avg

            //ema/wmaW
            lib.DefineInternalFunction("ema", isQuotes, BT.Double, FunctionLibrary.twoIntArgs, new BInternalExecution(PREMA));
            lib.DefineInternalFunction("wma", isQuotes, BT.Double, FunctionLibrary.oneIntArg , new BInternalExecution(PRWMA));

            //each֐ OmegaChartR
            sys.Register("each", new EachFunction());
        }
        private static ExecResult LastCandle(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            if(quotes.Count==0)
                return ExecResult.Nil;
            else {
                ((Candle)result).Let(quotes.LastCandle);
                return ExecResult.OK;
            }

        }
        private static ExecResult CandleAt(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            int arg = ((BInt)args[0]).Value;
            Candle c = quotes.CandleAtOrNull(quotes.Count-1 - arg);
            if(c==null)
                return ExecResult.Nil;
            else {
                ((Candle)result).Let(c);
                Debug.Assert(((Candle)result).BaseQuote == c.BaseQuote);
                return ExecResult.OK;
            }

        }
        //PEQp (start{Olen{)
        private static ExecResult SubList(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            int current = quotes.GetLatestIndexWithEvalContext(extra);

            int start = current - ((BInt)args[0]).Value + 1;
            int len = args.Length==2? ((BInt)args[1]).Value : ((BInt)args[0]).Value;
            if(start < 0 || len <= 0 || start+len-1 > current)
                return ExecResult.Nil;
            else {
                ((SubseqQuote)result).Set(quotes, start, len);
            }
            return ExecResult.OK;

        }
        //Ō n {
        private static ExecResult Last(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            int current = quotes.GetLatestIndexWithEvalContext(extra);
            int len = ((BInt)args[0]).Value;
            if(current - len + 1 < 0 || len<=0)
                return ExecResult.Nil;
            else {
                ((SubseqQuote)result).Set(quotes, current-len+1, len);
            }
            return ExecResult.OK;
        }
        //K؂藎Ƃ
        private static ExecResult Shift(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            int shift = ((BInt)args[0]).Value;
            if(shift >= quotes.Count)
                return ExecResult.Nil;
            else if(shift < 0) {
                SubseqQuote sq = quotes as SubseqQuote;
                if(sq!=null && sq.StartPosition+sq.Count-shift <= sq.BaseQuote.Count) //Base̒ŋtVtgłȂ
                    ((SubseqQuote)result).Set(sq.BaseQuote, sq.StartPosition, sq.Count-shift);
                else
                    return ExecResult.Nil;
            }
            else
                ((SubseqQuote)result).Set(quotes, 0, quotes.Count-shift); //shift==0łĂresultquote͓IuWFNgɂȂĂ͂ȂȂ̂SetKv
            return ExecResult.OK;
        }
        //K؂藎Ƃ̂Ԏw
        private static ExecResult ShiftToTime(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            int time = ((BInt)args[0]).Value;
            int index = quotes.FindExactly(time);
            if(index==-1)
                return ExecResult.Nil;
            else {
                ((SubseqQuote)result).Set(quotes, 0, index+1); //shift==0łĂresultquote͓IuWFNgɂȂĂ͂ȂȂ̂SetKv
                return ExecResult.OK;
            }
        }
        private static ExecResult PROpen(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            if(quotes.Count==0)
                return ExecResult.Nil;
            else
                ((BDouble)result).Value = quotes.LastCandle.Open;
            return ExecResult.OK;
        }
        private static ExecResult PRClose(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            if(quotes.Count==0)
                return ExecResult.Nil;
            else {
                ((BDouble)result).Value = quotes.LastCandle.Close;
            }
            return ExecResult.OK;
        }
        private static ExecResult PRHigh(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            if(quotes.Count==0)
                return ExecResult.Nil;
            else 
                ((BDouble)result).Value = quotes.LastCandle.High;
            return ExecResult.OK;
        }
        private static ExecResult PRLow(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            if(quotes.Count==0)
                return ExecResult.Nil;
            else
                ((BDouble)result).Value = quotes.LastCandle.Low;
            return ExecResult.OK;
        }
        private static ExecResult PRVolume(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            if(quotes.Count==0)
                return ExecResult.Nil;
            else 
                ((BDouble)result).Value = quotes.LastCandle.Volume;
            return ExecResult.OK;
        }
        private static ExecResult PRCreditLong(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            if(quotes.Count==0)
                return ExecResult.Nil;
            else
                ((BDouble)result).Value = quotes.LastCandle.CreditLong;
            return ExecResult.OK;
        }
        private static ExecResult PRCreditShort(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            if(quotes.Count==0)
                return ExecResult.Nil;
            else
                ((BDouble)result).Value = quotes.LastCandle.CreditShort;
            return ExecResult.OK;
        }
        private static ExecResult PRTradeAmount(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            if(quotes.Count==0)
                return ExecResult.Nil;
            else {
                Candle c = quotes.LastCandle;
                ((BDouble)result).Value = ((double)c.Volume) * c.Close ;
            }
            return ExecResult.OK;
        }
        private static ExecResult PRMA(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            int len = ((BInt)args[0]).Value;
            int count = quotes.Count;
            if(len<=0 || count < len)
                return ExecResult.Nil;
            else {
                int v = 0;
                for(int i=0; i<len; i++) {
                    Candle c = quotes.CandleAt(count-1-i);
                    if(c.IsNil) {
                        return ExecResult.Nil;
                    }
                    v += c.Close;
                }
                ((BDouble)result).Value = ((double)v) / len;
            }
            return ExecResult.OK;
        }

        //EMA ͂QŁAŏAQԖڂ钷
        private static ExecResult PREMA(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            int len = ((BInt)args[0]).Value;
            int smooth = ((BInt)args[1]).Value;
            int count = quotes.Count;
            if(len<=0 || count < len+smooth)
                return ExecResult.Nil;
            else {
                //܂l
                int v = 0;
                for(int i=0; i<len; i++) {
                    Candle c = quotes.CandleAt(count-1-smooth-i);
                    if(c.IsNil) {
                        return ExecResult.Nil;
                    }
                    v += c.Close;
                }
                double avg = ((double)v) / len;

                double alpha = 2.0 / (len+1); //萔
                for(int i=0; i<smooth; i++) {
                    Candle c = quotes.CandleAt(count-smooth+i);
                    if(c.IsNil) {
                        return ExecResult.Nil;
                    }
                    avg = alpha * c.Close + (1-alpha) * avg;
                }
                ((BDouble)result).Value = avg;
            }

            return ExecResult.OK;
        }
        //dړ
        private static ExecResult PRWMA(BV target, EvalContext extra, BV[] args, BV result) {
            Quote quotes = ToQuote(target);
            int len = ((BInt)args[0]).Value;
            int count = quotes.Count;
            if(len<=0 || count < len)
                return ExecResult.Nil;
            else {
                long v = 0;
                int d = 0;
                for(int i=0; i<len; i++) {
                    Candle c = quotes.CandleAt(count-1-i);
                    if(c.IsNil) {
                        return ExecResult.Nil;
                    }
                    v += c.Close * (len-i);
                    d += (len-i);
                }
                ((BDouble)result).Value = ((double)v) / d;
            }
            return ExecResult.OK;
        }

        private static bool IsQuote(BT value) {
            return value==instance;
        }
        private static Quote ToQuote(BV value) {
            Debug.Assert(IsQuote(value.BT));
            return (Quote)value;
        }
        private class Element {
            private string _name;
            private Candle.ElementDelegate _ed;
            public Element(FunctionLibrary lib, string name, Candle.ElementDelegate ed, BInternalExecution noArgFunction) {
                _name= name;
                _ed = ed;

                BT isQuote = instance;
                BT da = BT.Double.ArrayType();
                //S{lɂĂ͈̌ŁAOF̊ԑŚAPFŐV{̑̒iŝƂniljAQF{On܂邙{ƂȂ
                if(noArgFunction!=null)
                    lib.DefineInternalFunction(_name, isQuote, BT.Double, FunctionLibrary.noArg, noArgFunction);
                lib.DefineInternalFunction(_name, isQuote, da, FunctionLibrary.oneIntArg, new BInternalExecution(TailArray));
                lib.DefineInternalFunction(_name, isQuote, da, FunctionLibrary.twoIntArgs, new BInternalExecution(RangeArray));
            }
            private ExecResult TailArray(BV target, EvalContext extra, BV[] args, BV result) {
                Quote quotes = ToQuote(target);
                int start = ((BInt)args[0]).Value;
                return ElementArray(quotes, extra, start, start, result);
            }
            private ExecResult RangeArray(BV target, EvalContext extra, BV[] args, BV result) {
                Quote quotes = ToQuote(target);
                int start = ((BInt)args[0]).Value;
                int len = ((BInt)args[1]).Value;
                return ElementArray(quotes, extra, start, len, result);
            }
            private ExecResult ElementArray(Quote quotes, EvalContext extra, int start_arg, int len, BV result) {
                int current = quotes.GetLatestIndexWithEvalContext(extra);
                int start = current - start_arg + 1;
                if(start < 0 || len <= 0 || start+len-1 > current)
                    return ExecResult.Nil;
                else {
                    //FastenArrayɌŒ\Ǝv
                    BRegularArray array = (BRegularArray)result;
                    array.Extend(len);
                    for(int i=0; i<len; i++) {
                        Candle c = quotes.CandleAt(start + i);
                        if(c.IsNil) { //NilȎl{l̍Ă͑Ŝnil
                            return ExecResult.Nil;
                        }
                        ((BDouble)array[i]).Value = _ed(c);
                    }
                }
                return ExecResult.OK;
            }
        }

        //Each֐
        private class EachFunction : IBSystemFunction {
            public BSystemFunctionArg CreateArgEvaluators(EvaluatorBuilder builder, Bellagio.Script.InvokeExpression expr) {
                if(expr.ArgCount==2 && builder.Context.ContextType==QuoteType.instance) {
                    Evaluator[] arg_evals = new Evaluator[2];
                    arg_evals[0] = expr[0].Apply<Evaluator>(builder); //
                    if(arg_evals[0].BT==BT.Int) {
                        //context type̓RRƓQuote
                        Evaluator second_arg_eval =expr[1].Apply<Evaluator>(builder);
                        BFunctionT bft = second_arg_eval.BT as BFunctionT;
                        if(bft!=null) {
                            if(bft.ArgTypes.Length==1 && bft.ArgTypes[0]==QuoteType.instance) {
                                arg_evals[1] = second_arg_eval;
                                return new BSystemFunctionArg(bft.ReturnType.ArrayType(), arg_evals);
                            }
                        }
                    }
                }
                throw new BellagioException("eachQuoteɑ΂(int, (Quote=>T))̌`ŌĂяoȂ΂܂");
            }

            public ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result) {
                SubseqQuote pr = (SubseqQuote)context.ContextValue;
                int orig_len = pr.Count;
                BRegularArray arr = (BRegularArray)result;

                BInt len = (BInt)evaluators[0].Eval(context);
                arr.Extend(len.Value);

                BT elemtype = ((BFunctionT)evaluators[1].BT).ReturnType;
                BMethodValue func = (BMethodValue)evaluators[1].Eval(context);

                //snil
                if(arr.Count > pr.Count) {
                    return ExecResult.Nil;
                }

                BV[] args = new BV[1] { pr };

                for(int i=0; i<arr.Count; i++) {
                    pr.SetLength(orig_len - (arr.Count-1-i));
                    BV r = arr[i];
                    ExecResult er = func.Execute(null, context, args, r);
                    if(er==ExecResult.Nil) {
                        return ExecResult.Nil;
                    }
                    arr.LetAt(i, r);
                }

                pr.SetLength(orig_len); //ɖ߂
                return ExecResult.OK;
            }
        }
    }


#if UNITTEST
    [TestFixture]
    public class QuoteTests {
        [Test]
        public void Quote1() {
            ConcreteQuote cp = new ConcreteQuote(Quote.QuoteUnit.Minutely, 5);
            Candle y1 = cp.Add(900, 1000, 1000);
            Candle y2 = cp.Add(905, 1010, 2000);
            Candle y3 = cp.Add(910, 1010, 1020, 1000, 1030, 3000, 0, 0);

            //͂̃eXgpȂ̂łƃC`L
            Assert.AreEqual(905,  y2.Time);
            Assert.AreEqual(1010, y2.Open);
            Assert.AreEqual(1010, y2.High);
            Assert.AreEqual(1010, y2.Low);
            Assert.AreEqual(1010, y2.Close);
            Assert.AreEqual(2000, y2.Volume);

            Assert.AreEqual(910,  y3.Time);
            Assert.AreEqual(1010, y3.Open);
            Assert.AreEqual(1020, y3.High);
            Assert.AreEqual(1000, y3.Low);
            Assert.AreEqual(1030, y3.Close);
            Assert.AreEqual(3000, y3.Volume);

            Assert.AreEqual(3, cp.Count);
            SubseqQuote sub = new SubseqQuote(cp, 1, 1);
            Assert.AreEqual(1, sub.Count);
            Assert.AreEqual(cp.Unit, sub.Unit);
            Assert.AreEqual(cp.Scale, sub.Scale);

            y2 = sub.CandleAt(0); //y2ƓeɂȂ͂
            Assert.AreEqual(905, y2.Time);
            Assert.AreEqual(1010, y2.Open);
            Assert.AreEqual(1010, y2.High);
            Assert.AreEqual(1010, y2.Low);
            Assert.AreEqual(1010, y2.Close);
            Assert.AreEqual(2000, y2.Volume);
        }

        [Test]
        public void Quote2() {
            ConcreteQuote cp = new ConcreteQuote(Quote.QuoteUnit.Minutely, 5);
            //ł̃T[`̃eXg
            for(int i=0; i<100; i++)
                cp.Add(i*10, 0, 0);

            //0990܂10݂łĂ͂
            Assert.AreEqual(50, cp.FindExactly(500));
            Assert.AreEqual(-1, cp.FindExactly(501)); //vȂƂ̓_
            Assert.AreEqual(-1, cp.FindExactly(-10));
            Assert.AreEqual( 0, cp.FindExactly(0));
            Assert.AreEqual(99, cp.FindExactly(990));
            Assert.AreEqual(-1, cp.FindExactly(1000));
        }
    }
#endif
}
