/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Drawing/BellagioPen.cs#9 $
 * $DateTime: 2007/12/28 19:23:38 $
 * 
 * ỹX^Cstruct
 */

using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.Drawing;

using Bellagio.Common;
using Travis.ORT;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Drawing {
    
    public struct BPenStyle {
        public enum DashStyle {
            Solid = 0, Dot = 1
        }
        //FȂǂ̐ݒƂ͓ƗɁAvisible̐ݒVACYł悤ɂĂPreference̕ۑŕ֗
        public Color color; 
        public DashStyle dash;
        public int width;
        public bool visible;
        private Pen _pen; //.NET Pen

        public BPenStyle(Color color_, DashStyle dash_, int width_) {
            color = color_;
            dash = dash_;
            width = width_;
            visible = true;
            _pen = null;
        }
        private Pen CreateDotNetPen() {
            Pen p = new Pen(color);
            if(dash==DashStyle.Dot) p.DashStyle = System.Drawing.Drawing2D.DashStyle.Dot;
            p.Width = width;
            return p;
        }
        public Pen Pen {
            get {
                if(_pen==null) _pen = CreateDotNetPen();
                return _pen;
            }
        }
        public void Clear() {
            if(_pen!=null) {
                _pen.Dispose();
                _pen = null;
            }
        }
        public PolyPen CreatePolyPen() {
            return new PolyPen.Win32Pen(ref this);
        }

        //B
        public BPenStyle Bolder() {
            if(dash==DashStyle.Dot)
                return new BPenStyle(color, DashStyle.Solid, width+1);
            else
                return new BPenStyle(color, DashStyle.Solid, width+2);
        }


        //PenStyle
        public static BPenStyle Parse(ORText text) {
            //Debug.Assert(text!=null);
            if(text.IsOmitted)
                throw new BellagioException(ORText.FormatMissingElementMessage(text));

            string[] elems = text.Value.Split(' ');
            BPenStyle bs = new BPenStyle();
            bs.width =1;
            bs.visible = true;
            foreach(string e in elems) {
                if(e=="dot")
                    bs.dash = BPenStyle.DashStyle.Dot;
                else if(e=="!")
                    bs.visible = false;
                else if(e.Length==1 && '0'<=e[0] && e[0]<='9')
                    bs.width = text.ParseIntInORText(e);
                else
                    bs.color = text.ParseColorInORText(e);
            }
            return bs;
        }
        public static BPenStyle Parse(string text) {
            string[] elems = text.Split(' ');
            BPenStyle bs = new BPenStyle();
            bs.width =1;
            bs.visible = true;
            Color c = Color.Empty;
            foreach(string e in elems) {
                if(e=="dot")
                    bs.dash = BPenStyle.DashStyle.Dot;
                else if(e=="!")
                    bs.visible = false;
                else if('0'<=e[0] && e[0]<='9')
                    Int32.TryParse(e, out bs.width);
                else {
                    if(DrawingUtil.TryParseColor(e, out c)) bs.color = c;
                }
            }
            return bs;
        }
        public static bool TryParse(String text, out BPenStyle result) {
            result = Parse(text);
            return !result.color.IsEmpty;
        }

        public override string ToString() {
            StringBuilder bld = new StringBuilder();
            if(!visible) bld.Append("! ");
            if(dash==DashStyle.Dot) bld.Append("dot ");
            if(width!=1) bld.Append(width.ToString()).Append(" ");

            bld.Append(DrawingUtil.ColorToString(color));
            return bld.ToString();
        }
    }

    //.NETłWin32łŐ؂ւeՂȃy
    public abstract class PolyPen { //polymorphic̗
        public abstract void Dispose();
        public abstract void DrawLine(Graphics g, int x1, int y1, int x2, int y2);

        public class DotNetPen : PolyPen {
            private Pen _pen;
            public DotNetPen(ref BPenStyle ps) {
                _pen = ps.Pen;
            }
            public override void Dispose() {
                _pen.Dispose();
            }

            public override void DrawLine(Graphics g, int x1, int y1, int x2, int y2) {
                g.DrawLine(_pen, x1, y1, x2, y2);
            }
        }

        public class Win32Pen : PolyPen {
            private IntPtr _pen;
            private bool _manualDotMode;

            //_̕\ɂ
            //_ł̃CWP[^̕\ł́AfPS_DOTw̃yŕ`悵ĂE\NP{ׂɓhĂ܂_ɌȂ
            //Ȃ̂ŁA_̂Ƃ̂ݎw肳ꂽ̈̒_Œ


            public Win32Pen(ref BPenStyle ps) {
                _manualDotMode = ps.dash==BPenStyle.DashStyle.Dot && ps.width > 1;
                _pen = Win32.CreatePen(ps.dash==BPenStyle.DashStyle.Dot && ps.width==1? Win32.PS_DOT : Win32.PS_SOLID, ps.width, Win32.ToCOLORREF(ps.color));
            }

            public override void Dispose() {
                Win32.DeleteObject(_pen);
            }

            public override void DrawLine(Graphics g, int x1, int y1, int x2, int y2) {
                IntPtr hdc = g.GetHdc();
                Win32.SetBkColor(hdc, 0); //TODO ݍŒ
                Win32.SelectObject(hdc, _pen);
                Win32.MoveToEx(hdc, x1, y1, 0);
                if(_manualDotMode)
                    Win32.LineTo(hdc, (x1+x2)/2, (y1+y2)/2);
                else
                    Win32.LineTo(hdc, x2, y2);
                g.ReleaseHdc(hdc);
            }
        }
    }

#if UNITTEST
    [TestFixture]
    public class PenStyleTests {
        [Test]
        public void ConvString() {
            BPenStyle ps = BPenStyle.Parse("Blue");
            Assert.AreEqual(1, ps.width);
            Assert.AreEqual(Color.Blue, ps.color);
            Assert.AreEqual(BPenStyle.DashStyle.Solid, ps.dash);
            Assert.AreEqual("Blue", ps.ToString());

            ps = BPenStyle.Parse("2 dot #123456");
            Assert.AreEqual(2, ps.width);
            Assert.AreEqual(Color.FromArgb(0x12,0x34,0x56), ps.color);
            Assert.AreEqual(BPenStyle.DashStyle.Dot, ps.dash);
            Assert.AreEqual("dot 2 #123456", ps.ToString());

            ps = BPenStyle.Parse("1 dot pink");
            Assert.AreEqual(1, ps.width);
            Assert.AreEqual(Color.Pink, ps.color);
            Assert.AreEqual(BPenStyle.DashStyle.Dot, ps.dash);
            Assert.AreEqual("dot Pink", ps.ToString());
        }
    }
#endif
}
