/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Environment/DynamicPreferences.cs#5 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * [Uݒ\ȍڂ̂A\gLbg߂
 */

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Drawing;
using System.Diagnostics;

using Poderosa;
using Poderosa.Preferences;
using Travis.Collections;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Environment {
    //̃m[hƃACeTree\zAExtensiblePreferenceB
    public interface IDynamicPreferenceNode {
        string PreferenceID { get; }
        IDynamicPreferenceNode ParentNode { get; } //ŏ(ExtensionKit)łnull
    }

    public interface IDynamicPreferenceItem : IDynamicPreferenceNode {
        string DefaultValue { get; }
        string Value { get; set; }
    }
    public class DynamicPreferenceItemImpl : IDynamicPreferenceItem {
        private string _id;
        private IDynamicPreferenceNode _parent;
        private string _defaultValue;

        public DynamicPreferenceItemImpl(IDynamicPreferenceNode parent, string id, string defaultValue) {
            _parent = parent;
            _id = id;
            _defaultValue = defaultValue;
        }


        public string DefaultValue {
            get {
                return _defaultValue;
            }
        }
        public string Value {
            get {
                return BellagioRoot.ExtensionKitPreference.InternalGetValue(this);
            }
            set {
                BellagioRoot.ExtensionKitPreference.InternalSetValue(this, value);
            }
        }


        public string PreferenceID {
            get {
                return _id;
            }
        }

        public IDynamicPreferenceNode ParentNode {
            get {
                return _parent;
            }
        }
    }


    //p[^уCWP[^̕\ݒɊւ镔
    //IBellagioPreferenceItem̓o^󂯕tAStructuredTextƂ̃}bsOS
    //ꂪIPreferenceLooseNodeContentɂȂĂ̂ŁAč쐬͂₱BZbgƂ͓CX^X̂܂ClearĂԂق悢
    public class DynamicPreference : PreferenceSupplierImpl, IPreferenceLooseNodeContent {
        private string _folderName;
        private StructuredText _text; //ۑ̂ƂɂȂ
        private bool _loaded;

        //o^ς݂IExtensiblePreferenceNode/Item̊֌Wc[ŕێB͋Nߒō쐬ÂƂ͂ȂB
        //Save/Load͂Nodec[go[XStructuredTextƂ̃}bsOsĂ
        public class Node {
            public IDynamicPreferenceNode value;
            public SingleLinkedList<Node> childNodes;
            public SingleLinkedList<IDynamicPreferenceItem> items;

            public Node(IDynamicPreferenceNode v) {
                value = v;
                items = new SingleLinkedList<IDynamicPreferenceItem>();
                //childNodes͒x쐬
            }
            public void AddChildNode(Node v) {
                if(childNodes==null) childNodes = new SingleLinkedList<Node>();
                childNodes.Add(v);
            }
            public void RemoveItem(IDynamicPreferenceItem item) {
                bool r = items.Remove(item);
                Debug.Assert(r);
            }
            public void RemoveNode(Node node) {
                Debug.Assert(childNodes!=null);
                bool r = childNodes.Remove(node);
                Debug.Assert(r);
            }
            public bool IsEmpty {
                get {
                    return (childNodes==null || childNodes.Count==0) && items.Count==0;
                }
            }
        }

        private Node _rootNode;
        private Dictionary<IDynamicPreferenceNode, Node> _nodes; //r̒l [gPqT邱Ƃł邪Aꔭňق֗
        private Dictionary<IDynamicPreferenceItem, string> _changedValues; //[̒l L[Ȃ΃ftHgl

        public DynamicPreference(string prefID) : base(prefID) {
            _folderName = prefID;
            Clear();
        }
        public StructuredText UnderlyingData {
            get {
                return _text;
            }
        }

        public override void InitializePreference(IPreferenceBuilder builder, IPreferenceFolder folder) {
            builder.DefineLooseNode(folder, this, "content");
        }
        public void Clear() {
            _changedValues = new Dictionary<IDynamicPreferenceItem, string>();
            _nodes = new Dictionary<IDynamicPreferenceNode, Node>();
            _rootNode = new Node(null);
        }



        public void RegisterItem(IDynamicPreferenceItem item) {
            Debug.Assert(!_changedValues.ContainsKey(item)); //dNG
            Node n;
            if(!_nodes.TryGetValue(item.ParentNode, out n)) {
                n = RegisterNode(item.ParentNode);
            }

            n.items.Add(item);
            
        }
        
        private Node RegisterNode(IDynamicPreferenceNode key) {
            Debug.Assert(key!=null);
            Debug.Assert(!_nodes.ContainsKey(key));
            Node n = new Node(key);

            if(key.ParentNode==null) { //ParentNodenull̂Ƃ̓[gɂ
                _rootNode.AddChildNode(n);
            }
            else {
                Node p;
                if(!_nodes.TryGetValue(key.ParentNode, out p)) {
                    p = RegisterNode(key.ParentNode);
                }
                p.AddChildNode(n);
            }

            _nodes.Add(key, n);
            return n;
        }
        public void UnregisterItem(IDynamicPreferenceItem item) {
            _changedValues.Remove(item); //changedValues͂ĂȂĂ悵

            Node n;
            if(_nodes.TryGetValue(item.ParentNode, out n)) {
                n.RemoveItem(item);
                if(n.IsEmpty) UnregisterNode(n);
            }
        }
        public void UnregisterNode(IDynamicPreferenceNode node) {
            Node n;
            if(_nodes.TryGetValue(node, out n)) {
                UnregisterNode(n);
            }
        }
        private void UnregisterNode(Node n) {
            bool r = _nodes.Remove(n.value);
            Debug.Assert(r); //͍폜K{

            IDynamicPreferenceNode ipn = n.value.ParentNode;
            Node p = ipn==null? _rootNode : _nodes[ipn];
            p.RemoveNode(n);
            if(p.IsEmpty && p!=_rootNode) UnregisterNode(p); //recursive
        }

        //tID̎擾
        public IDynamicPreferenceItem Find(string full_id) {
            if(!_loaded) LoadMain();

            string[] elements = full_id.Split('.');
            return FindInternal(_rootNode, 0, elements);
        }
        private IDynamicPreferenceItem FindInternal(Node current, int index, string[] elements) {
            string name = elements[index++];

            if(current.childNodes!=null) {
                foreach(Node cn in current.childNodes) {
                    if(cn.value.PreferenceID==name) {
                        if(index < elements.Length)
                            return FindInternal(cn, index, elements);
                        else
                            return null; //\[X̕񂪐s
                    }
                }
            }

            if(index == elements.Length && current.items!=null) { //I[̂݃`FbN
                foreach(IDynamicPreferenceItem item in current.items) {
                    if(item.PreferenceID==name) return item;
                }
            }
            return null;
        }


        //l̎擾Eݒ
        //ExtensiblePreferenceItemĂԁBOڂƂƂ͏Ȃ͂
        public string InternalGetValue(IDynamicPreferenceItem p) {
            if(!_loaded) LoadMain();

            string t;
            if(_changedValues.TryGetValue(p, out t))
                return t;
            else
                return p.DefaultValue;  //GgȂ΃ftHg
        }
        public void InternalSetValue(IDynamicPreferenceItem p, string value) {
            if(!_loaded) LoadMain();

            if(p.DefaultValue==value) //ftHglƓƂ͒lNA
                _changedValues.Remove(p);
            else
                _changedValues[p] = value; //ɃL[邩ǂ͖킸
        }


        //ڂ̓o^@NߒłЂƂƂĂ

        public IPreferenceLooseNodeContent Clone() {
            //TODO {͂ł͂񂪁ÃN[̕KvȕҏW`oꂷ܂ł͖ɂ͂ȂȂ
            return null;
        }

        public void Reset() {
            if(_text!=null) _text.Clear();
            _changedValues.Clear();
        }

        public void LoadFrom(StructuredText node) {
            _text = node;
            _changedValues.Clear();
            _loaded = false;
        }

        public void SaveTo(StructuredText text) {
            Debug.Assert(text.ChildCount==0); //ˁH
            //if(!_modified) return; //Assertʉ߂悤Ƃꂶ܂

            //ftHgƈقȂĂ̂ݓo^ _rootNodeItem𒼐ړo^邱Ƃ͂Ȃ̂ɒ
            if(_rootNode.childNodes!=null) {
                foreach(Node n in _rootNode.childNodes) SaveNode(text, n);
            }
        }
        private void SaveNode(StructuredText text, Node node) {
            StructuredText k = new StructuredText(node.value.PreferenceID);
            if(node.childNodes!=null) {
                foreach(Node n in node.childNodes) SaveNode(k, n);
            }
            foreach(IDynamicPreferenceItem item in node.items) {
                //Gg݂AftHgƈقȂƂ̂݋L^
                string v;
                if(_changedValues.TryGetValue(item, out v) && v!=item.DefaultValue) k.Set(item.PreferenceID, v);
            }

            //܂łăf[^
            if(k.ChildCount > 0) text.AddChild(k);
        }


        private void LoadMain() {
            _changedValues.Clear();
            if(_text!=null)
                LoadNode(_text, _rootNode);
            _loaded = true;
        }
        private void LoadNode(StructuredText text, Node node) {
            foreach(object e_ in text.Children) {
                if(e_ is StructuredText) { //node
                    if(node.childNodes!=null) {
                        StructuredText e = (StructuredText)e_;
                        Node n = node.childNodes.Find(delegate(Node x) { return x.value.PreferenceID==e.Name; });
                        if(n!=null) LoadNode(e, n); //recursive
                    }
                }
                else { //value
                    StructuredText.Entry e = (StructuredText.Entry)e_;
                    IDynamicPreferenceItem item = node.items.Find(delegate(IDynamicPreferenceItem x) { return x.PreferenceID==e.name; });
                    if(item!=null) _changedValues[item] =  e.value;
                }
            }
        }

        //eXgł͒𒼂ł悤
#if UNITTEST
        public Node GetNode(IDynamicPreferenceNode node) {
            Node n;
            if(_nodes.TryGetValue(node, out n))
                return n;
            else
                return null;
           
        }
#endif
        public Node GetRootNode() {
            return _rootNode;
        }

    }

    internal static class PreferenceUtil {
        public static string FormatFullID(IDynamicPreferenceNode item) {
            StringBuilder bld = new StringBuilder();
            FormatFullID(item, bld);
            return bld.ToString();
        }
        public static void FormatFullID(IDynamicPreferenceNode node, StringBuilder buf) {
            IDynamicPreferenceNode parent = node.ParentNode;
            if(parent!=null) {
                FormatFullID(parent, buf);
                buf.Append(".");
            }
            buf.Append(node.PreferenceID);
        }
    }


#if UNITTEST
    [TestFixture]
    public class BellagioPreferenceTests {

        private class TestNode : IDynamicPreferenceNode {
            private string _id;
            private TestNode _parent;
            public TestNode(string id, TestNode node) {
                _id = id;
                _parent = node;
            }

            public string PreferenceID {
                get { return _id; }
            }
            public string FullID {
                get {
                    return _parent==null? _id  : _parent.FullID + "." + _id;
                }
            }

            public IDynamicPreferenceNode ParentNode {
                get { return _parent; }
            }
        }
        private class TestItem : TestNode, IDynamicPreferenceItem {
            private string _defaultValue;
            public TestItem(string id, TestNode parent, string defaultv)
                : base(id, parent) {
                _defaultValue = defaultv;
            }
            public string DefaultValue {
                get { return _defaultValue; }
            }

            //̓eXgɂ͌Ă΂ȂBExtensiblePreferenceւ̎QƂ܂킷̂͂
            public string Value {
                get {
                    Debug.Assert(false);
                    return _defaultValue;
                }
                set {
                    Debug.Assert(false);
                }
            }
        }

        private TestItem _i1;
        private TestItem _i2;
        private TestItem _i3;
        private TestItem _i4;


        private DynamicPreference CreateTestPref() {
            DynamicPreference pr = new DynamicPreference("test");
            TestNode p1 = new TestNode("N1", null);
            TestNode p2 = new TestNode("N2", null);
            _i1 = new TestItem("I1", p1, "1");
            _i2 = new TestItem("I2", p1, "2");
            _i3 = new TestItem("I3", p2, "3");
            _i4 = new TestItem("I4", p2, "4");
            pr.RegisterItem(_i1);
            pr.RegisterItem(_i2);
            pr.RegisterItem(_i3);
            pr.RegisterItem(_i4);
            return pr;
        }

        [Test]
        public void LoadTest() {
            DynamicPreference pr = CreateTestPref();
            StructuredText text = new StructuredText("z"); //ō
            pr.LoadFrom(text);

            Assert.AreEqual("1", pr.InternalGetValue(_i1));
            Assert.AreEqual("2", pr.InternalGetValue(_i2));
            Assert.AreEqual("3", pr.InternalGetValue(_i3));
            Assert.AreEqual("4", pr.InternalGetValue(_i4));

            text = new TextStructuredTextReader(new StringReader("z {\nN1 {\nI1=3\n}\nN2 {\n I4=10\n}\n}\n")).Read();
            pr.LoadFrom(text);

            Assert.AreEqual("3", pr.InternalGetValue(_i1));
            Assert.AreEqual("2", pr.InternalGetValue(_i2));
            Assert.AreEqual("3", pr.InternalGetValue(_i3));
            Assert.AreEqual("10", pr.InternalGetValue(_i4));
        }

        [Test]
        public void UnregisterTest() {
            DynamicPreference pr = CreateTestPref();
            DynamicPreference.Node root = pr.GetRootNode();
            DynamicPreference.Node p1 = pr.GetNode(_i1.ParentNode);
            Assert.AreEqual(2, p1.items.Count);
            Assert.AreEqual(2, root.childNodes.Count);

            pr.UnregisterItem(_i1);
            Assert.AreEqual(1, p1.items.Count);
            Assert.IsNotNull(pr.GetNode(_i1.ParentNode));

            pr.UnregisterItem(_i2);
            Assert.AreEqual(0, p1.items.Count);
            Assert.IsTrue(p1.IsEmpty);
            Assert.IsNull(pr.GetNode(_i1.ParentNode)); //eȂȂ̂NodeĂ͂
            Assert.AreEqual(1, root.childNodes.Count); //root͂PɌĂ͂
        }

        [Test]
        public void SaveTest() {
            DynamicPreference pr = CreateTestPref();
            pr.InternalSetValue(_i1, "1"); //ftHgƂȂ
            pr.InternalSetValue(_i2, "10");
            //I3,I4͕ύXȂBȂ̂Ńm[hȂ͂

            Assert.AreEqual("1", pr.InternalGetValue(_i1));
            Assert.AreEqual("10", pr.InternalGetValue(_i2));

            StructuredText text = new StructuredText("z");
            pr.SaveTo(text);
            StringWriter wr = new StringWriter();
            new TextStructuredTextWriter(wr).Write(text);
            wr.Close();
            Assert.AreEqual("z {\r\n  N1 {\r\n    I2=10\r\n  }\r\n}\r\n", wr.ToString());
        }
    }
#endif
}
