﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Diagnostics;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Travis {
    public static class ColorUtil {
        //あとでDrawingUtilからも持ってくる

        //HSVから色を作る。
        //H: 0から6。定数HueR等を参照。関数内でmod 6するので引数は範囲から外れてもOK
        //S: 0から1。0だと常に黒。
        //V: 0から1。1で原色。HSBとは違う
        public static Color FromHSV(float h, float s, float v) {
            if(h<0 || h>=6.0)
                h = h - (float)(Math.Floor(h / 6.0)*6.0);

            Debug.Assert(0.0<=h && h< 6.0);
            Debug.Assert(0.0<=s && s<=1.0);
            Debug.Assert(0.0<=v && v<=1.0);

            //変換式はWikipediaのHSVの項でみつけたのを使った
            
            double i = Math.Truncate(h);
            float f = h - (float)i;
            float p = v * (1-s);
            float q = v * (1-f*s);
            float t = v * (1- (1-f)*s);

            switch((int)i) {
                case 0:
                    return FromFRGB(v, t, p);
                case 1:
                    return FromFRGB(q, v, p);
                case 2:
                    return FromFRGB(p, v, t);
                case 3:
                    return FromFRGB(p, q, v);
                case 4:
                    return FromFRGB(t, p, v);
                default: //case 5:
                    return FromFRGB(v, p, q);
            }
        }
        public const float HueR = 0.0F;
        public const float HueG = 2.0F;
        public const float HueB = 4.0F;

        private delegate int F2I(float f);
        public static Color FromFRGB(float r, float g, float b) {
            F2I f_to_i = (float f) => {
                int i = (int)(f * 255.0);
                Debug.Assert(i>=0);
                if(i >= 256) i = 255;
                return i;
            };
            return Color.FromArgb(f_to_i(r), f_to_i(g), f_to_i(b));
        }


    }
#if UNITTEST
    [TestFixture]
    public class ColorUtilTest {
        [Test]
        public void FromHSV() {
            Color r = ColorUtil.FromHSV(ColorUtil.HueR, 1.0F, 1.0F);
            Assert.AreEqual(Color.Red.ToArgb(), r.ToArgb());
            Color g = ColorUtil.FromHSV(ColorUtil.HueG, 1.0F, 1.0F);
            Assert.AreEqual(Color.Lime.ToArgb(), g.ToArgb());
            Color b = ColorUtil.FromHSV(ColorUtil.HueB, 1.0F, 1.0F);
            Assert.AreEqual(Color.Blue.ToArgb(), b.ToArgb());

            Color x = ColorUtil.FromHSV(5.0F, 0.5F, 0.5F);
            Assert.AreEqual(x.R, x.B);
        }
    }
#endif
}
