/*
 * linuxrc.c
 *
 * linuxrc program for initrd.
 *
 * Copyright (C) 2005  NTT DATA Corporation
 *
 * Version: 1.0 2005/11/11
 *
 * This is a linuxrc used for mounting root as loopback.
 * Usally, the linuxrc is a nash script, which can't stop when
 * any error occurs and continues until kernel panic.
 * This program stops if any error occurs.
 *
 * You can use diet to reduce the size of executable file.
 * ( [diet] gcc -Wall -O3 --static -s -o linuxrc.exe linuxrc.c )
 *
 */
#define _FILE_OFFSET_BITS 64
#define _LARGEFILE_SOURCE
#define _LARGEFILE64_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mount.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#define dev_t unsigned short int
#include <linux/loop.h>
#include <linux/unistd.h>
#define MNT_DETACH 0x00000002
#include <sys/wait.h>
int pivot_root(const char *new_root, const char *put_old);
#if !defined(memset)
void *memset(void *s, int c, size_t n);
#endif
#if !defined(strncpy)
char *strncpy(char *dest, const char *src, size_t n);
#endif
#if !defined(memcmp)
int memcmp(const void *s1, const void *s2, size_t n);
#endif
#if !defined(strncmp)
int strncmp(const char *s1, const char *s2, size_t n);
#endif

static int mount_flag = 0;

static void WaitKey(void) {
	int c;
	fprintf(stderr, "Press 'Enter' to continue, or 'Ctrl'-'Alt'-'Delete' to reboot.\n");
	while (1) {
		if ((c = getchar()) != EOF) {
			if (c == '\n') break;
		} else {
			sleep(10);
		}
	}
}

static int losetup(const char *dev, const char *file) {
	struct loop_info loopinfo;
	int dev_fd, file_fd, err = 0;
	int mode = O_RDONLY;
	if (mount_flag == 0) {
		if ((file_fd = open(file, O_RDWR | O_LARGEFILE)) == EOF) {
			if (errno != EROFS) return errno;
			mount_flag = MS_RDONLY;
		} else {
			mode = O_RDWR;
			goto ok;
		}
	}
	if ((file_fd = open(file, O_RDONLY | O_LARGEFILE)) == EOF) return errno;
 ok: ;
	if ((dev_fd = open(dev, mode | O_LARGEFILE)) == EOF) {
		err = errno;
		close(file_fd);
		return err;
	}
	memset(&loopinfo, 0, sizeof(loopinfo));
	strncpy(loopinfo.lo_name, file, sizeof(loopinfo.lo_name) - 1);
	if (ioctl(dev_fd, LOOP_SET_FD, file_fd) == EOF) {
		err = errno;
		goto out;
	}
	if (ioctl(dev_fd, LOOP_SET_STATUS, &loopinfo) == EOF) {
		err = errno;
		ioctl(dev_fd, LOOP_CLR_FD, 0);
		goto out;
	}
 out: ;
	close(dev_fd);
	close(file_fd);
	return err;
}

static const char *GetFSType(const char *filename) {
	static unsigned char buffer[65536 + 1024];
	const char *fstype = NULL;
	int fd = open(filename, O_RDONLY | O_LARGEFILE);
	if (fd == EOF) return NULL;
	memset(buffer, 0, sizeof(buffer));
	read(fd, (char *) buffer, sizeof(buffer));
	close(fd);
	if (buffer[0x438 + 1] == 0xEF && buffer[0x438] == 0x53) {
		if ((buffer[0x45C] & 4) == 4) fstype = "ext3"; else fstype = "ext2";
	} else if (memcmp(buffer + 32769, "CD001", 5) == 0 || memcmp(buffer + 37633, "CD001", 5) == 0) {
		fstype = "iso9660";
	} else if (buffer[0x1FE] == 0x55 && buffer[0x1FF] == 0xAA && buffer[0] == 0xEB && buffer[1] == 0x3C && memcmp(buffer + 0x36, "FAT", 3) == 0) {
		fstype = "vfat";
	} else if (buffer[0x1FE] == 0x55 && buffer[0x1FF] == 0xAA && memcmp(buffer + 0x52, "FAT32", 5) == 0) {
		fstype = "vfat";
	} else if (memcmp(buffer + 65588, "ReIsErFs", 8) == 0 || memcmp(buffer + 65588, "ReIsEr2Fs", 9) == 0) {
		fstype = "reiserfs";
	} else if (memcmp(buffer, "XFSB", 4) == 0) {
		fstype = "xfs";
	} else if (memcmp(buffer + 32768, "JFS1", 4) == 0) {
		fstype = "jfs";
	}
	return fstype;
}


#define MAX_ARG 50
static char *args[MAX_ARG];

static void trim(char *buffer) {
	unsigned char *sp = buffer, *dp = buffer;
	int first = 1;
	while (*sp && (*sp <= ' ' || *sp >= 127)) sp++;
	while (*sp) {
		if (!first) *dp++ = ' ';
		first = 0;
		while (*sp > ' ' && *sp < 127) *dp++ = *sp++;
		while (*sp && (*sp <= ' ' || *sp >= 127)) sp++;
	}
	*dp = '\0';
}

static void tokenize(char *buffer) {
	int i;
	memset(args, 0, sizeof(args));
	args[0] = buffer;
	for (i = 1; i < MAX_ARG; i++) {
		args[i] = strchr(args[i - 1] + 1, ' ');
		if (!args[i]) break;
		*args[i]++ = '\0';
	}
}

int main(int argc, char *argv[]) {
	FILE *fp;
	unsigned int root = 0;
	int err;
	const char *container_fstype = NULL;
	const char *image_fstype = NULL;
	const char *rootimg_path = "/loopfs/rootimg";
	mkdir("/sysroot", 0755);
	mkdir("/proc", 0755);
	mkdir("/loopfs", 0755);
	mkdir("/lib", 0755);
	mkdir("/etc", 0755);
	mkdir("/sbin", 0755);
	mkdir("/dev", 0755);
	mkdir("/bin", 0755);
	if (mount("none", "/proc", "proc", 0, NULL) == EOF) {
		fprintf(stderr, "'mount -t proc none /proc' faild : %s\n", strerror(errno));
		WaitKey();
	}
	if ((fp = fopen("/linuxrc.org", "r")) == NULL) {
		fprintf(stderr, "Can't read /linuxrc.org , continuing.\n");
	} else {
		char buffer[1024];
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp)) {
			trim(buffer);
			tokenize(buffer);
			if (!args[0] || !args[1]) continue;
			if (strcmp(args[0], "insmod") == 0) {
				pid_t pid = fork();
				if (pid == 0) {
					execvp(args[0], args);
					exit(1);
				} else if (pid != -1) {
					printf("insmod %s\n", args[1]);
					waitpid(pid, NULL, 0);
				}
			} else if (strcmp(args[0], "sleep") == 0) {
				sleep(atoi(args[1]));
			}
		}
		fclose(fp);
	}
	if ((fp = fopen("/proc/cmdline", "r")) != NULL) {
		char buffer[1024];
		memset(buffer, 0, sizeof(buffer));
		if (fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
			int i;
			trim(buffer);
			tokenize(buffer);
			for (i = 0; i < MAX_ARG; i++) {
				if (!args[i]) break;
				//if (strcmp(args[i], "readonly") == 0) mount_flag = MS_RDONLY;
				if (strncmp(args[i], "CONTAINER=", 10) == 0) {
					static char buffer[128];
					memset(buffer, 0, sizeof(buffer));
					strncpy(buffer, args[i] + 10, sizeof(buffer) - 1);
					container_fstype = buffer;
					if (strcmp(buffer, "iso9660") == 0) mount_flag = MS_RDONLY;
				} else if (strncmp(args[i], "IMAGE=", 6) == 0) {
					static char buffer[128];
					memset(buffer, 0, sizeof(buffer));
					strncpy(buffer, args[i] + 6, sizeof(buffer) - 1);
					image_fstype = buffer;
				} else if (strncmp(args[i], "ROOTIMG=", 8) == 0) {
					static char buffer[1024];
					memset(buffer, 0, sizeof(buffer));
					snprintf(buffer, sizeof(buffer) - 1, "/loopfs/%s", args[i] + 8);
					rootimg_path = buffer;
				} else if (strncmp(args[i], "root=", 5) == 0) {
					struct stat buf;
					if (stat(args[i] + 5, &buf) == 0 && S_ISBLK(buf.st_mode)) root = buf.st_rdev;
				}
			}
		}
		fclose(fp);
	}
	if (root == 0) {
		if ((fp = fopen("/proc/sys/kernel/real-root-dev", "r")) == NULL || fscanf(fp, "%u", &root) != 1) {
			fprintf(stderr, "'cat /proc/sys/kernel/real-root-dev' failed : %s\n", strerror(errno));
			WaitKey();
		}
		fclose(fp);
	}
	unlink("/dev/root");
	if (mknod("/dev/root", S_IFBLK | 0600, (dev_t) root) == EOF) {
		fprintf(stderr, "'mkrootdev /dev/root' faild : rootdev=%X: %s\n", root, strerror(errno));
		WaitKey();
	}
	{
		int trial;
		for (trial = 0; trial <= 50; trial++) {
			int fd = open("/dev/root", O_RDONLY);
			struct timeval tv;
			if (fd != EOF) {
				close(fd);
				fprintf(stderr, "/dev/root ready.\n");
				break;
			}
			if (errno != ENXIO && errno != ENODEV) {
				fprintf(stderr, "open = %d: %s\n", errno, strerror(errno));
				break;
			}
			tv.tv_sec = 0; tv.tv_usec = 100 * 1000;
			select(0, NULL, NULL, NULL, &tv);
		}
	}
	if (!container_fstype) container_fstype = GetFSType("/dev/root");
	if (!container_fstype) {
		fprintf(stderr, "Can't get container's fstype. Try passing CONTAINER= option to kernel.\n");
		WaitKey();
	}
	{
		int trial;
		for (trial = 0; trial <= 50; trial++) {
			errno = 0;
			if (mount("/dev/root", "/loopfs", container_fstype, mount_flag, NULL) == 0) break;
			if (errno == ENXIO) {
				struct timeval tv;
				tv.tv_sec = 0; tv.tv_usec = 100 * 1000;
				select(0, NULL, NULL, NULL, &tv);
			} else if (errno == EROFS) {
				mount_flag = MS_RDONLY;
			} else {
				break;
			}
		}
		if (errno) {
			fprintf(stderr, "'mount -t %s /dev/root /loopfs' failed : rootdev=%X: %s(%d)\n", container_fstype, root, strerror(errno), errno);
			WaitKey();
		}
	}
	mknod("/dev/loop0", S_IFBLK | 0600, (dev_t) 0x700);
	if ((err = losetup("/dev/loop0", rootimg_path)) > 0) {
		fprintf(stderr, "'losetup /dev/loop0 %s' failed : %s\n", rootimg_path, strerror(err));
		WaitKey();
	}
	if (!image_fstype) image_fstype = GetFSType("/dev/loop0");
	if (!image_fstype) {
		fprintf(stderr, "Can't get image's fstype. Try passing IMAGE= option to kernel.\n");
		WaitKey();
	}
	
	if (mount("/dev/loop0", "/sysroot", image_fstype, mount_flag, NULL) == EOF) {
		fprintf(stderr, "'mount -t %s /dev/loop0 /sysroot' failed : %s\n", image_fstype, strerror(errno));
		WaitKey();
	}
	{
		if (access("/sysroot/.linuxrc", X_OK) == 0) {
			pid_t pid = fork();
			if (pid == 0) {
				if (chroot("/sysroot/") || chdir("/")) exit(1);
				argv[0] = "/.linuxrc";
				execv(argv[0], argv);
				exit(2);
			} else if (pid != -1) {
				fprintf(stderr, "Executing /.linuxrc\n");
				waitpid(pid, NULL, 0);
			}
		}
	}
	if ((fp = fopen("/proc/sys/kernel/real-root-dev", "w")) == NULL || fprintf(fp, "0x100") != 5) {
		fprintf(stderr, "'echo 0x100 > /proc/sys/kernel/real-root-dev' failed : %s\n", strerror(errno));
		WaitKey();
	}
	fclose(fp);
	if (pivot_root("/sysroot", "/sysroot/initrd") == EOF) {
		fprintf(stderr, "'pivot_root /sysroot /sysroot/initrd' failed : %s\n", strerror(errno));
		WaitKey();
	}
	if (umount("/initrd/proc") == EOF) {
		fprintf(stderr, "'umount /initrd/proc' failed : %s\n", strerror(errno));
		WaitKey();
	}
	if (umount2("/initrd/loopfs", MNT_DETACH) == EOF) {
		fprintf(stderr, "'unmount -l /initrd/loopfs' failed : %s\n", strerror(errno));
		WaitKey();
	}
	if (umount2("/initrd", MNT_DETACH) == EOF) {
		fprintf(stderr, "'umount -l /initrd' failed : %s\n", strerror(errno));
		WaitKey();
	}
	return 0;
}
