/*
 * makesymlink.cpp
 *
 * Create symbolic links for the given pathnames.
 *
 * Copyright (C) 2005  NTT DATA Corporation
 *
 * Version: 1.0 2005/11/11
 *
 * This program creates minimal symbolic links.
 * Parent directories are created with permission 0755.
 *
 */
#include <iostream>
#include <sstream>
#include <vector>
#include <algorithm>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
using namespace std;

static string decode(const string filename) {
	ostringstream buf;
	int index = 0;
	char c, d, e;
	while ((c = filename[index++]) != '\0') {
		if (c == '\\') {
			switch ((c = filename[index++])) {
			case '\\':      /* "\\" */
				buf << c;
				continue;
			case '0':       /* "\ooo" */
			case '1':
			case '2':
			case '3':
				if ((d = filename[index++]) >= '0' && d <= '7' && (e = filename[index++]) >= '0' && e <= '7') {
					const unsigned char f =
						(((unsigned char) (c - '0')) << 6) +
						(((unsigned char) (d - '0')) << 3) +
						(((unsigned char) (e - '0')));
					if (f && (f <= ' ' || f >= 127)) {
						buf << f;
						continue; /* pattern is not \000 */
					}
				}
			}
			return "";
		} else if (c <= ' ' || c >= 127) {
			return "";
		}
		buf << c;
	}
	return buf.str();
}

int main(int argc, char *argv[]) {
	vector <string> dir_list;
	vector <string> symlink_list;
	vector <string> symlink_data_list;
	string line;
	if (argc > 1 && chroot(argv[1]) || chdir("/")) {
		cerr << argv[0] << " [chroot_dir] [ < symlinked_filelist_file ]" << endl;
		return 0;
	}
	// Read the output of dumpsymlink.
	while (getline(cin, line)) {
		istringstream tokenizer(line);
		string filename, symlink_data;
		if (!(tokenizer >> filename) || !(tokenizer >> symlink_data)) {
			cerr << "Invalid input. " << line << endl;
			continue;
		}
		filename = decode(filename);
		symlink_data = decode(symlink_data);
		symlink_list.push_back(filename);
		symlink_data_list.push_back(symlink_data);
	}
	// Create directories which symbolic links are created in.
	for (int i = 0; i < (int) symlink_list.size(); i++) {
		string filename = symlink_list[i];
		for (int i = 1; i < (int) filename.size(); i++) {
			if (filename[i] == '/') {
				if (mkdir(filename.substr(0, i).c_str(), 0755) == 0) dir_list.push_back(filename.substr(0, i));
			}
		}
	}
	sort(dir_list.begin(), dir_list.end());
	// Create symbolic links, regardless of existence for dereferenced files.
	for (int i = 0; i < (int) symlink_list.size(); i++) {
		string filename = symlink_list[i];
		string symlink_data = symlink_data_list[i];
		if (symlink(symlink_data.c_str(), filename.c_str()) && errno != EEXIST) {
			cerr << "Can't symlink " << filename << " to " << symlink_data << endl;
		}
	}
	// Remove empty directories and dead symbolic links created by this program.
	while (1) {
		int need_restart = 0;
		// Remove directory if empty.
		for (int i = dir_list.size() - 1; i >= 0; i--) {
			if (rmdir(dir_list[i].c_str()) == 0) {
				need_restart = 1;
				dir_list.erase(dir_list.begin() + i);
			}
		}
		// Remove symbolic link if dead.
		for (int i = symlink_list.size() - 1; i >= 0; i--) {
			struct stat buf;
			if (stat(symlink_list[i].c_str(), &buf)) {
				if (unlink(symlink_list[i].c_str())) cerr << "Can't remove " << symlink_list[i] << endl;
				else need_restart = 1;
				symlink_list.erase(symlink_list.begin() + i);
				symlink_data_list.erase(symlink_data_list.begin() + i);
			}
		}
		if (!need_restart) break;
	}
	// Show what was done by this program.
	for (int i = 0; i < (int) dir_list.size(); i++) {
		cout << "mkdir " << dir_list[i] << endl;
	}
	for (int i = 0; i < (int) symlink_list.size(); i++) {
		cout << "ln -s " << symlink_data_list[i] << " " << symlink_list[i] << endl;
	}
	return 0;
}
