package jp.osoite.tomu.xml.jaxb.util.builder;

import java.net.URL;
import jp.osoite.tomu.xml.jaxb.object.ChiefAxis;
import jp.osoite.tomu.xml.jaxb.object.Location;
import jp.osoite.tomu.xml.jaxb.object.SubAxis;
import jp.osoite.tomu.xml.jaxb.object.TomuMessage;
import jp.osoite.tomu.xml.jaxb.util.TomuMessageType;
import jp.osoite.tomu.xml.jaxb.wrapper.Axis;
import jp.osoite.tomu.xml.jaxb.wrapper.Axis.Direction;

/**
 * APPS_RESPONSEメッセージの汎用Builderです．
 * @author shima
 */
public class AppsResponseBuilder {

    private JAXBConstructor cnst;

    /**
     * コンストラクタ
     * @param dest 送信先ID
     * @param src 送信元ID
     * @param msgId メッセージID
     * @param msgSeq メッセージシーケンス番号
     */
    public AppsResponseBuilder(String dest, String src, long msgId, long msgSeq) {
        cnst = new JAXBConstructor(TomuMessageType.APPS_RESPONSE);
        cnst.setHeader(dest, src, msgId);
        cnst.setSequence(msgSeq);
    }

    /**
     * APPS_RESPONSEメッセージを構築します．
     * @return TomuMessageオブジェクト
     */
    public TomuMessage build() {
        try {
            return cnst.build();
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 温度センサデータを追加します．
     * @param sensorId センサID
     * @param val 温度
     * @param sensedTime センシング時刻
     */
    public void addTemperatureSensor(long sensorId, double val, long sensedTime) {
        cnst.addTemperatureData(val, sensedTime, sensorId);
    }

    /**
     * 人流センサデータを追加します．
     * @param sensorId センサID
     * @param sensedTime センシング時刻
     * @param measurementTime センシング時間間隔
     * @param residence 滞留人数
     * @param axis 人流
     */
    public void addPedestrianSensor(long sensorId, long sensedTime, long measurementTime, int residence, Axis axis) {
        ChiefAxis cAxis = JAXBConstructor.getChief(
                axis.getCompass(Direction.PLUS),
                axis.getFlowValue(Direction.PLUS),
                axis.getAverageSpeed(Direction.PLUS),
                axis.getCompass(Direction.MINUS),
                axis.getFlowValue(Direction.MINUS),
                axis.getAverageSpeed(Direction.MINUS));
        cnst.addStreetPedestrianData(measurementTime, residence, cAxis, sensedTime, sensorId);
    }

    /**
     * 交差路を考慮した人流センサデータを追加します．
     * @param sensorId センサID
     * @param sensedTime センシング時刻
     * @param measurementTime センシング時間間隔
     * @param residence 滞留人数
     * @param mainAxis 人流
     * @param subAxis 人流
     */
    public void addInterPedestrianSensor(long sensorId, long sensedTime, long measurementTime, int residence, Axis mainAxis, Axis subAxis) {
        ChiefAxis cAxis = JAXBConstructor.getChief(
                mainAxis.getCompass(Direction.PLUS),
                mainAxis.getFlowValue(Direction.PLUS),
                mainAxis.getAverageSpeed(Direction.PLUS),
                mainAxis.getCompass(Direction.MINUS),
                mainAxis.getFlowValue(Direction.MINUS),
                mainAxis.getAverageSpeed(Direction.MINUS));
        SubAxis sAxis = JAXBConstructor.getSub(
                subAxis.getCompass(Direction.PLUS),
                subAxis.getFlowValue(Direction.PLUS),
                subAxis.getAverageSpeed(Direction.PLUS),
                subAxis.getCompass(Direction.MINUS),
                subAxis.getFlowValue(Direction.MINUS),
                subAxis.getAverageSpeed(Direction.MINUS));
        cnst.addInterPedestrianData(measurementTime, residence, cAxis, sAxis, sensedTime, sensorId);
    }



    /**
     * CO2センサデータを追加します．
     * @param sensorId センサID
     * @param val 温度
     * @param sensedTime センシング時刻
     */
    public void addCO2Sensor(long sensorId, double val, long sensedTime) {
        cnst.addCo2Data(val, sensedTime, sensorId);
    }

    /**
     * 湿度センサデータを追加します．
     * @param sensorId センサID
     * @param val 温度
     * @param sensedTime センシング時刻
     */
    public void addHumiditySensor(long sensorId, double val, long sensedTime) {
        cnst.addHumidityData(val, sensedTime, sensorId);
    }

    /**
     * 画像センサデータを追加します．
     * @param sensorName センサ名
     * @param sensedTime センシング時刻
     * @param compass 方位(360度)
     * @param lat 緯度
     * @param lon 経度
     * @param imageUrl 画像のURL
     * @param thumbnail サムネイル画像のURL
     */
    public void addImageSensor(String sensorName, long sensedTime, int compass, double lat, double lon, URL image, URL thumbnail) {
        Location loc = JAXBConstructor.getLocation(lon, lat, 0);
        cnst.addImageData(sensorName, compass, loc, image.toString(), thumbnail.toString(), sensedTime);
    }
}
