/**
 * @file main.c
 * @brief 関数定義ファイル
 * @date 2010/08/08
 * @author takemasa
 *
 */

/**
 * @mainpage UART送信プログラム
 *
 * UART0による割り込みのデモ。
 *
 * 実行すると、UART0から2400bpsでデータを受信し、エコーバックする。
 * また、1文字受信するたびにLEDを点滅させる。受信の監視は割り込みを使う。
 *
 * さらに、TIMER0から1秒ごとの割り込みをうけ、別のLEDを点滅させる。
 *
 * システムの発振器はデフォルトの状態で使うため、4MHz。UART0には
 * 4MHzが供給される。
 *
 */

#define UART_DLBA		0x80
#define UART_8BIT		0x03
#define UART_RDR		0x01
#define UART_RBR_INT	0x01
#define UART_SPEED	57600
#define UART_DLL		(int)(25000000/16/UART_SPEED)

#define PCTIM0		0x1			// PCOMPレジスタ中のTIMER0ビット位置
#define TIMER0_PCLKSEL_MASK		0x000000C0		// PCLKSEL0レジスタ中のビットマスク
#define TIMER0_PCLKSEL_DIV4		0				// 1/4設定

#define TIMER_RESET		0x2
#define TIMER_ENABLE		0x1
#define TIMER_COUNTER_MODE	0
#define TIMER_MR0_INTERRUPT		1
#define TIMER_MR0_RESET			2
#define TIMER_MR0_STOP			4
#define CLEAR_MR0_INTERRUPT		1


#define PINSEL_UART0_TX	0x10
#define PINSEL_UART0_RX	0x40

#define UART_LED	0x10000000		// GPIO1 bit28 (LED8)
#define TIMER_LED	0x20000000		// GPIO1 bit29 (LED7)

#define SYSTEM_CLOCK 40000000	// 4MHz

void init_uart0(void);
void init_timer0(void);
void init_LED(void);
void enableFlashAccelerator(unsigned int cclkMHz);

#include "vector.h"
#include "LPC17xx.h"
#include "pll.h"
/**
 * @brief main関数
 * @return 正常終了なら0。組み込みでは意味のないことが多い。
 * @details
 * 初期化直後に呼ばれるアプリケーションのエントリ。
 *
 * UART0にピンを割り当て、2400bpsに設定し、割り込み可能にする。
 *
 */

int main(void)
{
		// 100MHz動作用にフラッシュを設定
	enableFlashAccelerator(100);

		// PLLの設定 (2 * 50 * 4MHz)/1/4 = 100MHz
	initPLL0(
			eIrc, 	// pllClockSource_type 	clkSrc,
			0,		// unsigned int 			isMainOsc20MHzMore,
			1,		// unsigned int			N,
			50,		// unsigned int			M,
			4		// unsigned int			cpuClkDiv
		);

		// 割り込みベクトルとNVICの初期化
	init_interrupt();

		// LEDピンの初期化
	init_LED();

		// UART0を初期化し、受信割り込み待ちにする。
	init_uart0();

		// TIMER0を初期化し、周期割り込み待ちを発生させる。
	init_timer0();

		// 割り込み待ち
	while(1)
		;

	return 0;
}

/**
 * @brief ステータスLED初期化関数
 * @details
 * GPIO1_28にピンを与え、出力として設定する。
 */
void init_LED(void)
{
		// GPIO1にピンを割り当てる
	LPC_PINCON->PINSEL3	= 0;

		// LEDを消灯状態にする
	LPC_GPIO1->FIOSET = UART_LED | TIMER_LED;

		// LEDに対応するピンを出力に設定
	LPC_GPIO1->FIODIR = UART_LED | TIMER_LED;

}
/**
 * @brief UART0初期化関数
 * @details
 * UART0にピンを与え、2400baud, 8bits, non-parityに初期化する。
 * また、受信割り込みを有効にする。
 *
 * LPC1768の
 * UART0ピンの割り当ては以下の通り。
 * - P0[2] : TXD0
 * - P0[3] : RXD0
 *
 * P0[2]ピンの機能は、PINSEL0レジスタのbit5:4に設定した値によって
 * 変更することができる。同様にP0[3]ピンの機能は、PINSEL0レジスタの
 * bit7:6に設定した値によって変更することができる。いずれも値が二進で
 * 0b01のときにUART0に機能が割り当てられる。
 *
 */
void init_uart0(void)
{
		// bit5:4=01, bit7:6=01, ピンをUART0に割り当てる。
	LPC_PINCON->PINSEL0 = PINSEL_UART0_TX | PINSEL_UART0_RX;

		// DLAB bit ( bit7 )を1にして、Divisor Latchにアクセス可能にする。
	LPC_UART0->LCR = UART_DLBA;

		/*
		 * LPC1768は、リセット時に内蔵の4MHz発振器で動作する。このクロックは
		 * 4分周されてUART0に入力されている。今回はこの1MHzクロックを使う。
		 *
		 * UART0は、入力クロックを16分周し、さらにDLM:DLLに入力された値で
		 * 分周してボーレートとして使う。今回ボーレートとして2400baudを使う。
		 * @code
		 * 1000000[Hz] / 16 / 2400 = 26.04
		 * @endcode
		 * DLLに26を設定すれば誤差0.2%以下であり、UARTの要求仕様である5％以下を
		 * 十分満たす。
		 */
	LPC_UART0->DLM =  0;
	LPC_UART0->DLL = UART_DLL;		// 2400bps

		// DLAB bit ( bit7 )を0にして、Divisor Latchにアクセス不能にする。
		// 8bit, 1stop bit, non parity
	LPC_UART0->LCR = UART_8BIT;

		// UART0の受信割り込みをイネーブルにする。
	LPC_UART0->IER = UART_RBR_INT;
}

/**
 * @brief UART0割り込みハンドラ
 *
 * @details
 * UART0が1文字受信するたびに呼ばれるハンドラ。受信データをエコーバックし、
 * LEDの状態を反転する。
 */
void uart0_rx_handler(void)
{
		// 受信レジスタから1文字取り出して送信する。受信レジスタから読むことで、
		// 割り込みはクリアされる。
	LPC_UART0->THR = LPC_UART0->RBR;

		// ledstate変数の状態にあわせてLEDを制御する
	LPC_GPIO1->FIOPIN ^= UART_LED;
}

/**
 * @brief TIMER0初期化関数
 * @details
 * TIMER0を有効にし、1秒ごとの割り込みを発生するタイマーとして設定し
 * 動作を開始させる。
 *
 */
void init_timer0(void)
{
		// TIMER0に電源を供給する。TIMER0はリセット後から電源を供給されているので
		// この設定は不要だが、デモとして見せるためにオンにしている。
	LPC_SC->PCONP |= PCTIM0;

		// ペリフェラルクロックの分収比を1/4にする。TIMER0はリセット後から1/4になっているの
		// この設定は不要だが、デモとして見せるために強制している。
	LPC_SC->PCLKSEL0 &= ~ TIMER0_PCLKSEL_MASK;	// フィールドのクリア
	LPC_SC->PCLKSEL0 |=   TIMER0_PCLKSEL_DIV4;	// 値の設定

	LPC_TIM0->TCR = TIMER_RESET;					//　リセットし、停止
	LPC_TIM0->CTCR = TIMER_COUNTER_MODE;			// プリスケーラのクロックを数える
	LPC_TIM0->PR =  25000-1;							// プリスケーラ出力は1mS
	LPC_TIM0->MR0 = 999;								// マッチレジスタは1秒ごとにマッチする。
	LPC_TIM0->MCR = TIMER_MR0_INTERRUPT | TIMER_MR0_RESET; // マッチするごとに割り込みし、リセット
	LPC_TIM0->CCR = 0;								// キャプチャ・ディセーブル
	LPC_TIM0->EMR = 0;								// 外部マッチ・ディセーブル
	LPC_TIM0->TCR = TIMER_ENABLE;					// タイマースタート
}


/**
 * @brief タイマー0割り込みハンドラ
 *
 * @details
 * timer0が割り込みを起こすたびに呼ばれるハンドラ。LEDを反転する。
 */
void timer0_handler(void)
{
		// タイマー割り込みをクリアする
	LPC_TIM0->IR = CLEAR_MR0_INTERRUPT;
		// ledstate変数の状態にあわせてLEDを制御する
	LPC_GPIO1->FIOPIN ^= TIMER_LED;

}

void enableFlashAccelerator(unsigned int cclkMHz)
{
	if ( cclkMHz <= 20)
		LPC_SC->FLASHCFG  = 0;
	else if ( cclkMHz <= 40)
		LPC_SC->FLASHCFG  = 1<<12;
	else if ( cclkMHz <= 60)
		LPC_SC->FLASHCFG  = 2<<12;
	else if ( cclkMHz <= 80)
		LPC_SC->FLASHCFG  = 3<<12;
	else if ( cclkMHz <= 100)
		LPC_SC->FLASHCFG  = 4<<12;
	else
		LPC_SC->FLASHCFG  = 5<<12;
}
