/*
 *  TOPPERS/JSP Kernel
 *      Toyohashi Open Platform for Embedded Real-Time Systems/
 *      Just Standard Profile Kernel
 *
 *  Copyright (C) 2000-2003 by Embedded and Real-Time Systems Laboratory
 *                              Toyohashi Univ. of Technology, JAPAN
 *
 *  TOPPERS/JSP for Blackfin
 *
 *  Copyright (C) 2004,2006,2006 by Takemasa Nakamura
 *  Copyright (C) 2004 by Ujinosuke
 *  Copyright (C) 2010-2013 by Kaneko System Co., Ltd.
 *
 *  上記著作権者は，以下の (1)～(4) の条件か，Free Software Foundation
 *  によって公表されている GNU General Public License の Version 2 に記
 *  述されている条件を満たす場合に限り，本ソフトウェア（本ソフトウェア
 *  を改変したものを含む．以下同じ）を使用・複製・改変・再配布（以下，
 *  利用と呼ぶ）することを無償で許諾する．
 *  (1) 本ソフトウェアをソースコードの形で利用する場合には，上記の著作
 *      権表示，この利用条件および下記の無保証規定が，そのままの形でソー
 *      スコード中に含まれていること．
 *  (2) 本ソフトウェアを，ライブラリ形式など，他のソフトウェア開発に使
 *      用できる形で再配布する場合には，再配布に伴うドキュメント（利用
 *      者マニュアルなど）に，上記の著作権表示，この利用条件および下記
 *      の無保証規定を掲載すること．
 *  (3) 本ソフトウェアを，機器に組み込むなど，他のソフトウェア開発に使
 *      用できない形で再配布する場合には，次のいずれかの条件を満たすこ
 *      と．
 *    (a) 再配布に伴うドキュメント（利用者マニュアルなど）に，上記の著
 *        作権表示，この利用条件および下記の無保証規定を掲載すること．
 *    (b) 再配布の形態を，別に定める方法によって，TOPPERSプロジェクトに
 *        報告すること．
 *  (4) 本ソフトウェアの利用により直接的または間接的に生じるいかなる損
 *      害からも，上記著作権者およびTOPPERSプロジェクトを免責すること．
 *
 *  本ソフトウェアは，無保証で提供されているものである．上記著作権者お
 *  よびTOPPERSプロジェクトは，本ソフトウェアに関して，その適用可能性も
 *  含めて，いかなる保証も行わない．また，本ソフトウェアの利用により直
 *  接的または間接的に生じたいかなる損害に関しても，その責任を負わない．
 *
 *
 */

/*
 *  ターゲットシステム依存モジュール（ADSP-BF609用）
 */

#include "jsp_kernel.h"
#include <sil.h>

#include <cdefBF609.h>
#include <sysreg.h>
#include <builtins.h>

#define REG_CGU0_DIV_VAL \
	((CSELVAL   << BITP_CGU_DIV_CSEL)   | \
	 (S0SELVAL  << BITP_CGU_DIV_S0SEL)  | \
	 (SYSSELVAL << BITP_CGU_DIV_SYSSEL) | \
	 (S1SELVAL  << BITP_CGU_DIV_S1SEL)  | \
	 (DSELVAL   << BITP_CGU_DIV_DSEL)   | \
	 (OSELVAL   << BITP_CGU_DIV_OSEL))

#define REG_CGU0_CTL_VAL \
	((MSELVAL << BITP_CGU_CTL_MSEL) | (DFVAL << BITP_CGU_CTL_DF))

#if SEC_ENABLE_LOCKING == 1
#	define SEC_LOCK_REG(reg)      (*(reg) |=  (1u << 31))
#	define SEC_UNLOCK_REG(reg)    (*(reg) &= ~(1u << 31))
#else /* SEC_ENABLE_LOCKING == 1 */
#	define SEC_LOCK_REG(reg)
#	define SEC_UNLOCK_REG(reg)
#endif /* SEC_ENABLE_LOCKING == 1 */

#ifndef ik_hardware
#define ik_ivg11 11
#endif

/* 割り込みマスク */
static volatile unsigned int sec_imask[IMS_MASK_SIZE];


/*
 *  ターゲットシステム依存の初期化
 */
void
sys_initialize()
{
    /*
     * スプリアス割り込みハンドラの設定
     *
     * cpu_initialize()が行うダミーの割り込みハンドラの設定を上書きする。
     * アプリケーションが割り込みハンドラを設定すると、以下の設定も上書き
     * される。
     */
    int i;
	unsigned int* p_reg;
	unsigned int core_id;
	
    for ( i=0; i<(DEVICE_INTERRUPT_COUNT+CORE_EVENT_COUNT); i++ )
        dev_vector[i] = &spurious_int_handler;

    exc_vector = &spurious_exc_handler;

    /*
     *  PLLの設定
     *
     */
    /*
     *  PLLの値はboard_config.hにて定義。
	 *  FORCE_PLL_INITIALIZEはsys_config.hで必要に応じて宣言する。
     */
#ifndef FORCE_PLL_INITIALIZE
     /* PLLが初期値のままであり、かつ、DDR2-SDRAMが利用中でなければPLLを初期化する */
     if ( ( *pPLL_CTL == 0x00001000 ) && ( !(*pREG_DMC_STAT & BITM_DMC_STAT_MEMINITDONE ) ) )
#endif
     {
		if((*pREG_CGU0_CTL & (BITM_CGU_CTL_MSEL | BITM_CGU_CTL_DF)) != REG_CGU0_CTL_VAL)
		{
			*pREG_CGU0_DIV = REG_CGU0_DIV_VAL;
			*pREG_CGU0_CTL = REG_CGU0_CTL_VAL;
			while ((*pREG_CGU0_STAT & (BITM_CGU_STAT_CLKSALGN | BITM_CGU_STAT_PLLBP)) ||
				  !(*pREG_CGU0_STAT & BITM_CGU_STAT_PLOCK))
			{
				continue;
			}
		}

        /* PLLの分周器に値を設定する */
		*pREG_CGU0_DIV = REG_CGU0_DIV_VAL | BITM_CGU_DIV_UPDT;
		while(*pREG_CGU0_STAT & BITM_CGU_STAT_CLKSALGN)
			continue;

		if((*pREG_DMC0_STAT & BITM_DMC_STAT_MEMINITDONE) != 0u)
		{
			*pREG_DMC0_CTL &= ~BITP_DMC_CTL_SRREQ;
			asm volatile("ssync;\n");
			while((*pREG_DMC0_STAT & BITM_DMC_STAT_SRACK) != 0u)
				continue;
		}
    }

    /*
     *  SECの初期化
     */
	core_id = *pDSPID & BITM_DSPID_COREID;
	if(0 == core_id)
	{
		/* Core-A */
		/* Reset SCI */
	    SEC_UNLOCK_REG(pREG_SEC0_CCTL0);
	    *pREG_SEC0_CCTL0 |= ENUM_SEC_CCTL_RESET;
	    SEC_LOCK_REG(pREG_SEC0_CCTL0);
		/* Clear NMI */
		*pREG_SEC0_CSTAT0 |= ENUM_SEC_CSTAT_NMI;

		/* Reset SEC */
	    SEC_UNLOCK_REG(pREG_SEC0_GCTL);
	    *pREG_SEC0_GCTL = ENUM_SEC_GCTL_RESET;
	    SEC_LOCK_REG(pREG_SEC0_GCTL);

		/* Reset Fault controller */
	    SEC_UNLOCK_REG(pREG_SEC0_FCTL);
	    *pREG_SEC0_FCTL = ENUM_SEC_FCTL_RESET;
	    SEC_LOCK_REG(pREG_SEC0_FCTL);

		/* Enable SEC */
	    SEC_UNLOCK_REG(pREG_SEC0_GCTL);
	    *pREG_SEC0_GCTL |= ENUM_SEC_GCTL_EN;
	    SEC_LOCK_REG(pREG_SEC0_GCTL);

        /* Enable NMI & SCI */
	    SEC_UNLOCK_REG(pREG_SEC0_CCTL0);
//	    *pREG_SEC0_CCTL0 |= ENUM_SEC_CCTL_NMI_EN | ENUM_SEC_CCTL_EN;
	    *pREG_SEC0_CCTL0 |= ENUM_SEC_CCTL_EN;
	    SEC_LOCK_REG(pREG_SEC0_CCTL0);

		/* Prioirty level depth is 16 (TOPPERS/JSP) */
	    SEC_UNLOCK_REG(pSEC0_CPLVL0);
	    *pREG_SEC0_CPLVL0 = 4;
	    SEC_LOCK_REG(pSEC0_CPLVL0);

		/* Set default priorty level (7) */
	    for(i = 0; i < (DEVICE_INTERRUPT_COUNT * (REG_SEC0_SCTL1 - REG_SEC0_SCTL0)); i += (REG_SEC0_SCTL1 - REG_SEC0_SCTL0))
		{
			p_reg = (unsigned int*)(REG_SEC0_SCTL0 + i);
		    SEC_UNLOCK_REG(p_reg);
		    *p_reg = 7 << BITP_SEC_SCTL_PRIO;
		    SEC_LOCK_REG(p_reg);
		}
	}
	else
	{
		/* Core-B */
		/* Reset SCI */
	    SEC_UNLOCK_REG(pREG_SEC0_CCTL1);
	    *pREG_SEC0_CCTL1 |= ENUM_SEC_CCTL_RESET;
	    SEC_LOCK_REG(pREG_SEC0_CCTL1);
		/* Clear NMI */
		*pREG_SEC0_CSTAT1 |= ENUM_SEC_CSTAT_NMI;

        /* Enable NMI & SCI */
	    SEC_UNLOCK_REG(pREG_SEC0_CCTL1);
	    *pREG_SEC0_CCTL1 |= ENUM_SEC_CCTL_NMI_EN | ENUM_SEC_CCTL_EN;
	    SEC_LOCK_REG(pREG_SEC0_CCTL1);

		/* Prioirty level depth is 16 (TOPPERS/JSP) */
	    SEC_UNLOCK_REG(pSEC0_CPLVL1);
	    *pREG_SEC0_CPLVL1 = 4;
	    SEC_LOCK_REG(pSEC0_CPLVL1);
	}
	 asm volatile("ssync;\n");

    /*
     *  UART分周比の設定
     *
     *  Logtaskが動作する前にsys_putc()を使うための設定を行う。以下の設定は
     *  serial関連のタスクが起動したときに上書きされる。
     */
#if LOGTASK_PORTID == 1
	*pREG_UART0_CTL = 0;
	 asm volatile("ssync;\n");
	*pREG_UART0_CLK = UART0_DIVISOR;
    /* モード設定, パリティ無し 8bit data, 1 stop bit */
	*pREG_UART0_CTL = ENUM_UART_CTL_PARITY_DIS | ENUM_UART_CTL_NO_EXTRA_STB |
		ENUM_UART_CTL_WL8BITS | ENUM_UART_CTL_UART_MODE | ENUM_UART_CTL_CLK_EN;
    /* 割込み禁止 */
	*pREG_UART0_IMSK_CLR = 0xFFFFFFFFu;
	*pREG_UART0_STAT = *pREG_UART0_STAT;

	*pREG_PORTD_FER_SET = BITM_PORT_FER_SET_PX8 | BITM_PORT_FER_SET_PX7;
	*pREG_PORTD_MUX = (*pREG_PORTD_MUX & ~(BITM_PORT_MUX_MUX7 | BITM_PORT_MUX_MUX8)) |
		(1 << BITP_PORT_MUX_MUX7) | (1 << BITP_PORT_MUX_MUX8);

#elif LOGTASK_PORTID == 2
	*pREG_UART1_CTL = 0;
	 asm volatile("ssync;\n");
	*pREG_UART1_CLK = UART1_DIVISOR;
    /* モード設定, パリティ無し 8bit data, 1 stop bit */
	*pREG_UART1_CTL = ENUM_UART_CTL_PARITY_DIS | ENUM_UART_CTL_NO_EXTRA_STB |
		ENUM_UART_CTL_WL8BITS | ENUM_UART_CTL_UART_MODE | ENUM_UART_CTL_CLK_EN;
    /* 割込み禁止 */
	*pREG_UART1_IMSK_CLR = 0xFFFFFFFFu;
	*pREG_UART0_STAT = *pREG_UART0_STAT;

	*pREG_PORTG_FER = BITM_PORT_FER_SET_PX14 | BITM_PORT_FER_SET_PX15;
	*pREG_PORTG_MUX = (*pREG_PORTG_MUX & ~(BITM_PORT_MUX_MUX14 | BITM_PORT_MUX_MUX15));
#else
#	error LOGTASK_PORTID is invalid value
#endif

}

/*
 * BF6xxは割り込みコントローラがSICからSECに変わったため、priority_maskを作る必要がなくなった.
 * この関数はuITRONのイニシャライザで使用することを想定しており、特に割り込みから保護していない。
 */
void make_priority_mask( void )
{
	int lc;
	for(lc = 0; lc < IMS_MASK_SIZE; lc++)
	{
		sec_imask[lc] = 0u;
	}
}

/*
 * 割り込みの許可。ADSP-BF54xは効率的な割り込み処理と安全な割り込み禁止を両立する
 * 手段を持たないため、禁止関数は置いていない。
 *
 * 不便ではあるが、プログラマに注意を促すためにそうしている。
 */
ER ena_int( INTNO intno )
{
	unsigned int ims_group;
	unsigned int ims_index;

    SIL_PRE_LOC;

    if ( intno >= DEVICE_INTERRUPT_COUNT )
	{
        return E_PAR;
	}
    else
	{
        SIL_LOC_INT();
		ims_group = intno >> 5;
		ims_index = intno & 0x1Fu;
		sec_imask[ims_group] |= (1u << ims_index);

		/* SEC0_SCTLxのSENとIENをEnableに設定する */
		SEC_UNLOCK_REG(pREG_SEC0_SCTL0[intno]);
		*((unsigned int*)(REG_SEC0_SCTL0 + (REG_SEC0_SCTL1 - REG_SEC0_SCTL0) * intno)) |= 
			ENUM_SEC_SCTL_SRC_EN | ENUM_SEC_SCTL_INT_EN;
		SEC_LOCK_REG(pREG_SEC0_SCTL0[intno]);
		asm volatile("ssync;\n");
        SIL_UNL_INT();
        return 0;
    }
}


/*
 * 割り込みマスクの取得。ADSP-BF54xは効率的な割り込み処理と安全な割り込み禁止を両立する
 * 手段を持たないため、禁止関数は置いていない。
 *
 * 不便ではあるが、プログラマに注意を促すためにそうしている。
 */


extern ER get_ims( IMS * p_ims )
{
    SIL_PRE_LOC;

    SIL_LOC_INT();
    p_ims->imask[0] = sec_imask[0];
    p_ims->imask[1] = sec_imask[1];
    p_ims->imask[2] = sec_imask[2];
    p_ims->imask[3] = sec_imask[3];
    p_ims->imask[4] = sec_imask[4];
    SIL_UNL_INT();
    return 0;
}
int a;

/*
 *  割り込みをデバイスに割り当てる。
 *
 *  この間数は割り込み発生時に共通割り込みハンドラの一部としてアセンブリ言語から割り込み禁止状態で
 *  呼ばれる。実割り込みハンドラを割り込み可能にするため、asm文を使って途中で割り込み可能にしている。
 *  割り込み禁止状態で呼ぶのは割り込み源の特定を安全におこなうためである。
 *
 */
void device_dispatcher( unsigned int priority, unsigned int imask )
{
	unsigned int cec_sid;
	unsigned int ims_group;
	unsigned int ims_index;
	unsigned int candidates;

	/* CECのイベント発生源を特定 */
	if(priority == ik_ivg11)
	{
		cec_sid = *pCEC_SID;
		ims_group = cec_sid >> 5;
		ims_index = cec_sid & 0x1F;
		candidates = sec_imask[ims_group] & (1u << ims_index);

		/* 割り込み禁止解除 */
		*pCEC_SID = cec_sid;
	    asm volatile("sti %0;": : "d"(imask) );

//		if(candidates != 0u)
		    dev_vector[cec_sid]();

		*pREG_SEC0_END = cec_sid;
		asm volatile("ssync;\n");
	}
	else
	{
	    asm volatile("sti %0;": : "d"(imask) );
        if ( priority == ik_hardware_err )
            dev_vector[INHNO_HW_ERROR]();
        else if ( priority == ik_timer )
            dev_vector[INHNO_TIMER]();
        else
            dev_vector[INHNO_RAISE]();      /* ソフトウェア割り込み */
	}
}



/*
 *  ターゲットシステムの終了。TOPPERS/JSPは対話型ROMモニタに戻ることを想定しているが、
 *  このボードにROMはない。
 */
void
sys_exit()
{
    while(1)
        continue;
}

/*
 *  ターゲットシステムの文字出力。割り込みが無効な状態でポーリングによる出力を行う。
 */
void
sys_putc(char c)
{
    if ( c == 0x0A )        /* もし LF ならば */
        sys_putc( 0x0D );   /* CRを一文字送信 */

#if LOGTASK_PORTID == 1
	while((*pREG_UART0_STAT & ENUM_UART_STAT_THR_EMPTY) == 0U)
		asm volatile("ssync;\n");
	*pREG_UART0_THR = c;

#elif LOGTASK_PORTID == 2
	while((*pREG_UART1_STAT & ENUM_UART_STAT_THR_EMPTY) == 0U)
		asm volatile("ssync;\n");
	*pREG_UART1_THR = c;

#else
#	error LOGTASK_PORTID is invalid value
#endif
}



