/*
 *  TOPPERS/JSP Kernel
 *      Toyohashi Open Platform for Embedded Real-Time Systems/
 *      Just Standard Profile Kernel
 *
 *  Copyright (C) 2000-2003 by Embedded and Real-Time Systems Laboratory
 *                              Toyohashi Univ. of Technology, JAPAN
 *
 *  TOPPERS/JSP for Blackfin
 *
 *  Copyright (C) 2004,2006,2009 by Takemasa Nakamura
 *  Copyright (C) 2004 by Ujinosuke
 *  Copyright (C) 2010-2013 by Kaneko System Co., Ltd.
 *
 *  上記著作権者は，以下の (1)～(4) の条件か，Free Software Foundation
 *  によって公表されている GNU General Public License の Version 2 に記
 *  述されている条件を満たす場合に限り，本ソフトウェア（本ソフトウェア
 *  を改変したものを含む．以下同じ）を使用・複製・改変・再配布（以下，
 *  利用と呼ぶ）することを無償で許諾する．
 *  (1) 本ソフトウェアをソースコードの形で利用する場合には，上記の著作
 *      権表示，この利用条件および下記の無保証規定が，そのままの形でソー
 *      スコード中に含まれていること．
 *  (2) 本ソフトウェアを，ライブラリ形式など，他のソフトウェア開発に使
 *      用できる形で再配布する場合には，再配布に伴うドキュメント（利用
 *      者マニュアルなど）に，上記の著作権表示，この利用条件および下記
 *      の無保証規定を掲載すること．
 *  (3) 本ソフトウェアを，機器に組み込むなど，他のソフトウェア開発に使
 *      用できない形で再配布する場合には，次のいずれかの条件を満たすこ
 *      と．
 *    (a) 再配布に伴うドキュメント（利用者マニュアルなど）に，上記の著
 *        作権表示，この利用条件および下記の無保証規定を掲載すること．
 *    (b) 再配布の形態を，別に定める方法によって，TOPPERSプロジェクトに
 *        報告すること．
 *  (4) 本ソフトウェアの利用により直接的または間接的に生じるいかなる損
 *      害からも，上記著作権者およびTOPPERSプロジェクトを免責すること．
 *
 *  本ソフトウェアは，無保証で提供されているものである．上記著作権者お
 *  よびTOPPERSプロジェクトは，本ソフトウェアに関して，その適用可能性も
 *  含めて，いかなる保証も行わない．また，本ソフトウェアの利用により直
 *  接的または間接的に生じたいかなる損害に関しても，その責任を負わない．
 *
 *
 */


#ifndef _CHIP_CONFIG_H_
#define _CHIP_CONFIG_H_


/*
 *  ターゲットシステム依存モジュール（ADSP-BF609用）
 *
 *  このインクルードファイルは，t_config.h のみからインクルードされる．
 *  他のファイルから直接インクルードしてはならない．
*/

/*
 *  ターゲットシステムのハードウェア資源の定義
 *
 *  DEVICE_INTERRUPT_COUNTはSIC_IMASKのビット数を与える。
 *  ADSP-BF609のSECが管理する割り込み数は139
 */

#define DEVICE_INTERRUPT_COUNT	(139)

/*
 * システムが管理するCEC(コアイベントコントローラ)の数
 * HW_ERROR
 * CORE_TIMER
 * SOFTWARE RAISE
 */
#define CORE_EVENT_COUNT		(3)

/*
 *  PLLクロック周波数
 */
#if DFVAL == 0
#	define PLLCLOCK  (CLKIN*MSELVAL)
#else
#	define PLLCLOCK  (CLKIN*MSELVAL/2)
#endif
#define SYS0CLOCK  (PLLCLOCK / SYSSELVAL / S0SELVAL)
#define SYS1CLOCK  (PLLCLOCK / SYSSELVAL / S1SELVAL)
#define SYSCLOCK   SYS0CLOCK
#define CORECLOCK  (PLLCLOCK / CSELVAL)



/*
 *  TICの割込みハンドラのベクタ番号
 */
#ifdef USE_TIC_CORE
#define INHNO_TIMER INHNO_CORE_TIMER
#else
#define INHNO_TIMER INHNO_TIMER0_TMR7
#endif


/*
 *  微少時間待ちのための定義
 *  sil_dly_nseの呼び出しオーバーヘッドとループボディーサイクルを
 *  nSで表す。関数はcpu_support.asmに定義してある。
 *  SIL_DLY_TIM1は24サイクル。32nS( 750MHz ), 40nS( 600MHz )
 *  SIL_DLY_TIM2は12サイクル。16nS( 750MHz ), 20nS( 600MHz )
 */


#define SIL_DLY_TIM1    (24*1000000000/CORECLOCK)
#define SIL_DLY_TIM2    (12*1000000000/CORECLOCK)

/*************************************************************************
 * uart_bf6xx.c用構成マクロ
 */

/* シリアルの割込みハンドラのベクタ番号 */
#define INHNO_SIO0   INHNO_UART0_STAT
#define INHNO_SIO1   INHNO_UART1_STAT

/* UARTレジスタにsil_xxb_iop()を使ってアクセスするときにはUART_IOP_ACCESSを 宣言する。*/
#define UART_IOP_ACCESS

/* UARTレジスタのベースアドレス */
#define UART0_ADDRESS 0xFFC02000    /* UART0_REVIDのアドレス */
#define UART1_ADDRESS 0xFFC02400    /* UART1_REVIDのアドレス */

/* UARTのボーレートジェネレータに設定すべき値． */
#define UART0_DIVISOR   (SYSCLOCK/16/UART0_BAUD_RATE)
#define UART1_DIVISOR   (SYSCLOCK/16/UART1_BAUD_RATE)

/*
 * uart_bf6xx.c用構成マクロ終わり
 *************************************************************************/



/*
 *  割り込みベクトル番号
 *
 *  この定義はコンフィグレーションファイルでDEF_INHの引数として使う
 *  SIC_ISRにおけるビット番号である。
 *
 */

#define INHNO_SEC0_ERR					0           /* Error */
#define INHNO_CGU0_EVT					1           /* Event */
#define INHNO_WDOG0_EXP					2           /* Expiration */
#define INHNO_WDOG1_EXP					3           /* Expiration */
#define INHNO_L2CTL0_ECC_ERR			4           /* ECC Error */
#define INHNO_L2CTL0_ECC_WARNING		5           /* ECC Warning */
#define INHNO_C0_DBL_FAULT				6           /* Core 0 Double Fault */
#define INHNO_C1_DBL_FAULT				7           /* Core 1 Double Fault */
#define INHNO_C0_HW_ERR					8           /* Core 0 Hardware Error */
#define INHNO_C1_HW_ERR					9           /* Core 1 Hardware Error */
#define INHNO_C0_NMI_L1_PARITY_ERR		10          /* Core 0 Unhandled NMI or L1 Memory Parity Error */
#define INHNO_C1_NMI_L1_PARITY_ERR		11          /* Core 1 Unhandled NMI or L1 Memory Parity Error */
#define INHNO_TIMER0_TMR0				12          /* Timer 0 */
#define INHNO_TIMER0_TMR1				13          /* Timer 1 */
#define INHNO_TIMER0_TMR2				14          /* Timer 2 */
#define INHNO_TIMER0_TMR3				15          /* Timer 3 */
#define INHNO_TIMER0_TMR4				16          /* Timer 4 */
#define INHNO_TIMER0_TMR5				17          /* Timer 5 */
#define INHNO_TIMER0_TMR6				18          /* Timer 6 */
#define INHNO_TIMER0_TMR7				19          /* Timer 7 */
#define INHNO_TIMER0_STAT				20          /* Status */
#define INHNO_PINT0_BLOCK				21          /* Pin Interrupt Block */
#define INHNO_PINT1_BLOCK				22          /* Pin Interrupt Block */
#define INHNO_PINT2_BLOCK				23          /* Pin Interrupt Block */
#define INHNO_PINT3_BLOCK				24          /* Pin Interrupt Block */
#define INHNO_PINT4_BLOCK				25          /* Pin Interrupt Block */
#define INHNO_PINT5_BLOCK				26          /* Pin Interrupt Block */
#define INHNO_CNT0_STAT					27          /* Status */
#define INHNO_PWM0_SYNC					28          /* PWMTMR Group */
#define INHNO_PWM0_TRIP					29          /* Trip */
#define INHNO_PWM1_SYNC					30          /* PWMTMR Group */
#define INHNO_PWM1_TRIP					31          /* Trip */
#define INHNO_TWI0_DATA					32          /* Data Interrupt */
#define INHNO_TWI1_DATA					33          /* Data Interrupt */
#define INHNO_SOFT0						34          /* Software-driven Interrupt 0 */
#define INHNO_SOFT1						35          /* Software-driven Interrupt 1 */
#define INHNO_SOFT2						36          /* Software-driven Interrupt 2 */
#define INHNO_SOFT3						37          /* Software-driven Interrupt 3 */
#define INHNO_ACM0_EVT_MISS				38          /* Event Miss */
#define INHNO_ACM0_EVT_COMPLETE			39          /* Event Complete */
#define INHNO_CAN0_RX					40          /* Receive */
#define INHNO_CAN0_TX					41          /* Transmit */
#define INHNO_CAN0_STAT					42          /* Status */
#define INHNO_SPORT0_A_DMA				43          /* Channel A DMA */
#define INHNO_SPORT0_A_STAT				44          /* Channel A Status */
#define INHNO_SPORT0_B_DMA				45          /* Channel B DMA */
#define INHNO_SPORT0_B_STAT				46          /* Channel B Status */
#define INHNO_SPORT1_A_DMA				47          /* Channel A DMA */
#define INHNO_SPORT1_A_STAT				48          /* Channel A Status */
#define INHNO_SPORT1_B_DMA				49          /* Channel B DMA */
#define INHNO_SPORT1_B_STAT				50          /* Channel B Status */
#define INHNO_SPORT2_A_DMA				51          /* Channel A DMA */
#define INHNO_SPORT2_A_STAT				52          /* Channel A Status */
#define INHNO_SPORT2_B_DMA				53          /* Channel B DMA */
#define INHNO_SPORT2_B_STAT				54          /* Channel B Status */
#define INHNO_SPI0_TXDMA				55          /* TX DMA Channel */
#define INHNO_SPI0_RXDMA				56          /* RX DMA Channel */
#define INHNO_SPI0_STAT					57          /* Status */
#define INHNO_SPI1_TXDMA				58          /* TX DMA Channel */
#define INHNO_SPI1_RXDMA				59          /* RX DMA Channel */
#define INHNO_SPI1_STAT					60          /* Status */
#define INHNO_RSI0_DMA					61          /* DMA Channel */
#define INHNO_RSI0_INT0					62          /* Interrupt 0 */
#define INHNO_RSI0_INT1					63          /* Interrupt 1 */
#define INHNO_SDU0_DMA					64          /* DMA */
/* reserved */
/* reserved */
/* reserved */
#define INHNO_EMAC0_STAT				68          /* Status */
/* reserved */
#define INHNO_EMAC1_STAT				70          /* Status */
/* reserved */
#define INHNO_LP0_DMA					72          /* DMA Channel */
#define INHNO_LP0_STAT					73          /* Status */
#define INHNO_LP1_DMA					74          /* DMA Channel */
#define INHNO_LP1_STAT					75          /* Status */
#define INHNO_LP2_DMA					76          /* DMA Channel */
#define INHNO_LP2_STAT					77          /* Status */
#define INHNO_LP3_DMA					78          /* DMA Channel */
#define INHNO_LP3_STAT					79          /* Status */
#define INHNO_UART0_TXDMA				80          /* Transmit DMA */
#define INHNO_UART0_RXDMA				81          /* Receive DMA */
#define INHNO_UART0_STAT				82          /* Status */
#define INHNO_UART1_TXDMA				83          /* Transmit DMA */
#define INHNO_UART1_RXDMA				84          /* Receive DMA */
#define INHNO_UART1_STAT				85          /* Status */
#define INHNO_MDMA0_SRC					86          /* Memory DMA Stream 0 Source / CRC0 Input Channel */
#define INHNO_MDMA0_DST					87          /* Memory DMA Stream 0 Destination / CRC0 Output Channel */
#define INHNO_CRC0_DCNTEXP				88          /* Datacount expiration */
#define INHNO_CRC0_ERR					89          /* Error */
#define INHNO_MDMA1_SRC					90          /* Memory DMA Stream 1 Source / CRC1 Input Channel */
#define INHNO_MDMA1_DST					91          /* Memory DMA Stream 1 Destination / CRC1 Output Channel */
#define INHNO_CRC1_DCNTEXP				92          /* Datacount expiration */
#define INHNO_CRC1_ERR					93          /* Error */
#define INHNO_MDMA2_SRC					94          /* Memory DMA Stream 2 Source Channel */
#define INHNO_MDMA2_DST					95          /* Memory DMA Stream 2 Destination Channel */
#define INHNO_MDMA3_SRC					96          /* Memory DMA Stream 3 Source Channel */
#define INHNO_MDMA3_DST					97          /* Memory DMA Stream 3 Destination Channel */
#define INHNO_EPPI0_CH0_DMA				98          /* Channel 0 DMA */
#define INHNO_EPPI0_CH1_DMA				99          /* Channel 1 DMA */
#define INHNO_EPPI0_STAT				100			/* Status */
#define INHNO_EPPI2_CH0_DMA				101         /* Channel 0 DMA */
#define INHNO_EPPI2_CH1_DMA				102         /* Channel 1 DMA */
#define INHNO_EPPI2_STAT				103         /* Status */
#define INHNO_EPPI1_CH0_DMA				104         /* Channel 0 DMA */
#define INHNO_EPPI1_CH1_DMA				105         /* Channel 1 DMA */
#define INHNO_EPPI1_STAT				106         /* Status */
#define INHNO_PIXC0_CH0_DMA				107         /* Channel 0 DMA */
#define INHNO_PIXC0_CH1_DMA				108         /* Channel 1 DMA */
#define INHNO_PIXC0_CH2_DMA				109         /* Channel 2 DMA */
#define INHNO_PIXC0_STAT				110         /* Status */
#define INHNO_PVP0_CPDOB_DMA			111         /* Camera Pipe Data Out B DMA Channel */
#define INHNO_PVP0_CPDOC_DMA			112         /* Camera Pipe Data Out C DMA Channel */
#define INHNO_PVP0_CPSTAT_DMA			113         /* Camera Pipe Status Out DMA Channel */
#define INHNO_PVP0_CPCI_DMA				114         /* Camera Pipe Control In DMA Channel */
#define INHNO_PVP0_STAT0				115         /* Status 0 */
#define INHNO_PVP0_MPDO_DMA				116         /* Memory Pipe Data Out DMA Channel */
#define INHNO_PVP0_MPDI_DMA				117         /* Memory Pipe Data In DMA Channel */
#define INHNO_PVP0_MPSTAT_DMA			118         /* Memory Pipe Status Out DMA Channel */
#define INHNO_PVP0_MPCI_DMA				119         /* Memory Pipe Control In DMA Channel */
#define INHNO_PVP0_CPDOA_DMA			120         /* Camera Pipe Data Out A DMA Channel */
#define INHNO_PVP0_STAT1				121         /* Status 1 */
#define INHNO_USB0_STAT					122         /* Status/FIFO Data Ready */
#define INHNO_USB0_DATA					123         /* DMA Status/Transfer Complete */
#define INHNO_TRU0_INT0					124         /* Interrupt 0 */
#define INHNO_TRU0_INT1					125         /* Interrupt 1 */
#define INHNO_TRU0_INT2					126         /* Interrupt 2 */
#define INHNO_TRU0_INT3					127         /* Interrupt 3 */
#define INHNO_DMAC_ERR					128         /* DMA Controller Error */
#define INHNO_CGU0_ERR					129         /* Error */
/* reserved */
#define INHNO_DPM0_EVT					131         /* Event */
/* reserved */
#define INHNO_SWU0_EVT					133         /* Event */
#define INHNO_SWU1_EVT					134         /* Event */
#define INHNO_SWU2_EVT					135         /* Event */
#define INHNO_SWU3_EVT					136         /* Event */
#define INHNO_SWU4_EVT					137         /* Event */
#define INHNO_SWU5_EVT					138         /* Event */
#define INHNO_SWU6_EVT					139         /* Event */
/* コアイベント */
#define INHNO_HW_ERROR      			140
#define INHNO_CORE_TIMER   				141
#define INHNO_RAISE	         			142

/*
 *  SIC_ISRの割り込みのベクタ番号
 *
 *  この定義はena_int, dis_intの引数として使う。
 */
#define INTNO_SEC0_ERR					0           /* Error */
#define INTNO_CGU0_EVT					1           /* Event */
#define INTNO_WDOG0_EXP					2           /* Expiration */
#define INTNO_WDOG1_EXP					3           /* Expiration */
#define INTNO_L2CTL0_ECC_ERR			4           /* ECC Error */
#define INTNO_L2CTL0_ECC_WARNING		5           /* ECC Warning */
#define INTNO_C0_DBL_FAULT				6           /* Core 0 Double Fault */
#define INTNO_C1_DBL_FAULT				7           /* Core 1 Double Fault */
#define INTNO_C0_HW_ERR					8           /* Core 0 Hardware Error */
#define INTNO_C1_HW_ERR					9           /* Core 1 Hardware Error */
#define INTNO_C0_NMI_L1_PARITY_ERR		10          /* Core 0 Unhandled NMI or L1 Memory Parity Error */
#define INTNO_C1_NMI_L1_PARITY_ERR		11          /* Core 1 Unhandled NMI or L1 Memory Parity Error */
#define INTNO_TIMER0_TMR0				12          /* Timer 0 */
#define INTNO_TIMER0_TMR1				13          /* Timer 1 */
#define INTNO_TIMER0_TMR2				14          /* Timer 2 */
#define INTNO_TIMER0_TMR3				15          /* Timer 3 */
#define INTNO_TIMER0_TMR4				16          /* Timer 4 */
#define INTNO_TIMER0_TMR5				17          /* Timer 5 */
#define INTNO_TIMER0_TMR6				18          /* Timer 6 */
#define INTNO_TIMER0_TMR7				19          /* Timer 7 */
#define INTNO_TIMER0_STAT				20          /* Status */
#define INTNO_PINT0_BLOCK				21          /* Pin Interrupt Block */
#define INTNO_PINT1_BLOCK				22          /* Pin Interrupt Block */
#define INTNO_PINT2_BLOCK				23          /* Pin Interrupt Block */
#define INTNO_PINT3_BLOCK				24          /* Pin Interrupt Block */
#define INTNO_PINT4_BLOCK				25          /* Pin Interrupt Block */
#define INTNO_PINT5_BLOCK				26          /* Pin Interrupt Block */
#define INTNO_CNT0_STAT					27          /* Status */
#define INTNO_PWM0_SYNC					28          /* PWMTMR Group */
#define INTNO_PWM0_TRIP					29          /* Trip */
#define INTNO_PWM1_SYNC					30          /* PWMTMR Group */
#define INTNO_PWM1_TRIP					31          /* Trip */
#define INTNO_TWI0_DATA					32          /* Data Interrupt */
#define INTNO_TWI1_DATA					33          /* Data Interrupt */
#define INTNO_SOFT0						34          /* Software-driven Interrupt 0 */
#define INTNO_SOFT1						35          /* Software-driven Interrupt 1 */
#define INTNO_SOFT2						36          /* Software-driven Interrupt 2 */
#define INTNO_SOFT3						37          /* Software-driven Interrupt 3 */
#define INTNO_ACM0_EVT_MISS				38          /* Event Miss */
#define INTNO_ACM0_EVT_COMPLETE			39          /* Event Complete */
#define INTNO_CAN0_RX					40          /* Receive */
#define INTNO_CAN0_TX					41          /* Transmit */
#define INTNO_CAN0_STAT					42          /* Status */
#define INTNO_SPORT0_A_DMA				43          /* Channel A DMA */
#define INTNO_SPORT0_A_STAT				44          /* Channel A Status */
#define INTNO_SPORT0_B_DMA				45          /* Channel B DMA */
#define INTNO_SPORT0_B_STAT				46          /* Channel B Status */
#define INTNO_SPORT1_A_DMA				47          /* Channel A DMA */
#define INTNO_SPORT1_A_STAT				48          /* Channel A Status */
#define INTNO_SPORT1_B_DMA				49          /* Channel B DMA */
#define INTNO_SPORT1_B_STAT				50          /* Channel B Status */
#define INTNO_SPORT2_A_DMA				51          /* Channel A DMA */
#define INTNO_SPORT2_A_STAT				52          /* Channel A Status */
#define INTNO_SPORT2_B_DMA				53          /* Channel B DMA */
#define INTNO_SPORT2_B_STAT				54          /* Channel B Status */
#define INTNO_SPI0_TXDMA				55          /* TX DMA Channel */
#define INTNO_SPI0_RXDMA				56          /* RX DMA Channel */
#define INTNO_SPI0_STAT					57          /* Status */
#define INTNO_SPI1_TXDMA				58          /* TX DMA Channel */
#define INTNO_SPI1_RXDMA				59          /* RX DMA Channel */
#define INTNO_SPI1_STAT					60          /* Status */
#define INTNO_RSI0_DMA					61          /* DMA Channel */
#define INTNO_RSI0_INT0					62          /* Interrupt 0 */
#define INTNO_RSI0_INT1					63          /* Interrupt 1 */
#define INTNO_SDU0_DMA					64          /* DMA */
/* reserved */
/* reserved */
/* reserved */
#define INTNO_EMAC0_STAT				68          /* Status */
/* reserved */
#define INTNO_EMAC1_STAT				70          /* Status */
/* reserved */
#define INTNO_LP0_DMA					72          /* DMA Channel */
#define INTNO_LP0_STAT					73          /* Status */
#define INTNO_LP1_DMA					74          /* DMA Channel */
#define INTNO_LP1_STAT					75          /* Status */
#define INTNO_LP2_DMA					76          /* DMA Channel */
#define INTNO_LP2_STAT					77          /* Status */
#define INTNO_LP3_DMA					78          /* DMA Channel */
#define INTNO_LP3_STAT					79          /* Status */
#define INTNO_UART0_TXDMA				80          /* Transmit DMA */
#define INTNO_UART0_RXDMA				81          /* Receive DMA */
#define INTNO_UART0_STAT				82          /* Status */
#define INTNO_UART1_TXDMA				83          /* Transmit DMA */
#define INTNO_UART1_RXDMA				84          /* Receive DMA */
#define INTNO_UART1_STAT				85          /* Status */
#define INTNO_MDMA0_SRC					86          /* Memory DMA Stream 0 Source / CRC0 Input Channel */
#define INTNO_MDMA0_DST					87          /* Memory DMA Stream 0 Destination / CRC0 Output Channel */
#define INTNO_CRC0_DCNTEXP				88          /* Datacount expiration */
#define INTNO_CRC0_ERR					89          /* Error */
#define INTNO_MDMA1_SRC					90          /* Memory DMA Stream 1 Source / CRC1 Input Channel */
#define INTNO_MDMA1_DST					91          /* Memory DMA Stream 1 Destination / CRC1 Output Channel */
#define INTNO_CRC1_DCNTEXP				92          /* Datacount expiration */
#define INTNO_CRC1_ERR					93          /* Error */
#define INTNO_MDMA2_SRC					94          /* Memory DMA Stream 2 Source Channel */
#define INTNO_MDMA2_DST					95          /* Memory DMA Stream 2 Destination Channel */
#define INTNO_MDMA3_SRC					96          /* Memory DMA Stream 3 Source Channel */
#define INTNO_MDMA3_DST					97          /* Memory DMA Stream 3 Destination Channel */
#define INTNO_EPPI0_CH0_DMA				98          /* Channel 0 DMA */
#define INTNO_EPPI0_CH1_DMA				99          /* Channel 1 DMA */
#define INTNO_EPPI0_STAT				100			/* Status */
#define INTNO_EPPI2_CH0_DMA				101         /* Channel 0 DMA */
#define INTNO_EPPI2_CH1_DMA				102         /* Channel 1 DMA */
#define INTNO_EPPI2_STAT				103         /* Status */
#define INTNO_EPPI1_CH0_DMA				104         /* Channel 0 DMA */
#define INTNO_EPPI1_CH1_DMA				105         /* Channel 1 DMA */
#define INTNO_EPPI1_STAT				106         /* Status */
#define INTNO_PIXC0_CH0_DMA				107         /* Channel 0 DMA */
#define INTNO_PIXC0_CH1_DMA				108         /* Channel 1 DMA */
#define INTNO_PIXC0_CH2_DMA				109         /* Channel 2 DMA */
#define INTNO_PIXC0_STAT				110         /* Status */
#define INTNO_PVP0_CPDOB_DMA			111         /* Camera Pipe Data Out B DMA Channel */
#define INTNO_PVP0_CPDOC_DMA			112         /* Camera Pipe Data Out C DMA Channel */
#define INTNO_PVP0_CPSTAT_DMA			113         /* Camera Pipe Status Out DMA Channel */
#define INTNO_PVP0_CPCI_DMA				114         /* Camera Pipe Control In DMA Channel */
#define INTNO_PVP0_STAT0				115         /* Status 0 */
#define INTNO_PVP0_MPDO_DMA				116         /* Memory Pipe Data Out DMA Channel */
#define INTNO_PVP0_MPDI_DMA				117         /* Memory Pipe Data In DMA Channel */
#define INTNO_PVP0_MPSTAT_DMA			118         /* Memory Pipe Status Out DMA Channel */
#define INTNO_PVP0_MPCI_DMA				119         /* Memory Pipe Control In DMA Channel */
#define INTNO_PVP0_CPDOA_DMA			120         /* Camera Pipe Data Out A DMA Channel */
#define INTNO_PVP0_STAT1				121         /* Status 1 */
#define INTNO_USB0_STAT					122         /* Status/FIFO Data Ready */
#define INTNO_USB0_DATA					123         /* DMA Status/Transfer Complete */
#define INTNO_TRU0_INT0					124         /* Interrupt 0 */
#define INTNO_TRU0_INT1					125         /* Interrupt 1 */
#define INTNO_TRU0_INT2					126         /* Interrupt 2 */
#define INTNO_TRU0_INT3					127         /* Interrupt 3 */
#define INTNO_DMAC_ERR					128         /* DMA Controller Error */
#define INTNO_CGU0_ERR					129         /* Error */
/* reserved */
#define INTNO_DPM0_EVT					131         /* Event */
/* reserved */
#define INTNO_SWU0_EVT					133         /* Event */
#define INTNO_SWU1_EVT					134         /* Event */
#define INTNO_SWU2_EVT					135         /* Event */
#define INTNO_SWU3_EVT					136         /* Event */
#define INTNO_SWU4_EVT					137         /* Event */
#define INTNO_SWU5_EVT					138         /* Event */
#define INTNO_SWU6_EVT					139         /* Event */
/* コアイベント */
#define INTNO_HW_ERROR      			140
#define INTNO_CORE_TIMER   				141
#define INTNO_RAISE	         			142



/*
 * IMSで使用する配列数
 */
#define IMS_MASK_SIZE					((DEVICE_INTERRUPT_COUNT + CORE_EVENT_COUNT + 31) / 32)

#ifndef _MACRO_ONLY

/*
* SIC_IARxを解析して、イベント順位ごとに割り当てられている割り込み
* のビットマップを作る。SIC_IARxのフィールド値は優先順位-7なので
* その分補正する。
*/
void make_priority_mask( void );

/*
 *  ターゲットシステム依存の初期化
 */
extern void sys_initialize(void);

/*
 *  ターゲットシステムの終了
 *
 *  システムを終了する時に使う．ROMモニタ／GDB STUB呼出しで実現する．
 */
extern void sys_exit(void);

/*
 *  ターゲットシステムの文字出力
 *
 *  システムの低レベルの文字出力ルーチン．ROMモニタ／GDB STUB呼出しで実
 *  現する．
 */
extern void sys_putc(char c);

/*
* SIC_IARxを解析して、イベント順位ごとに割り当てられている割り込み
* のビットマップを作る。SIC_IARxのフィールド値は優先順位-7なので
* その分補正する。
*/
void make_priority_mask( void );

/*
 * 割り込みの許可、禁止
 *
 */
typedef unsigned int INTNO;
extern ER ena_int( INTNO intno );
extern ER dis_int( INTNO intno );

/*
 * 割り込みマスクの操作
 *
 *
 */
typedef struct
{
    unsigned int imask[IMS_MASK_SIZE];
} IMS;
extern ER chg_ims( IMS ims );
extern ER get_ims( IMS * p_ims );

/*
 *  割り込みをデバイスに割り当てる
 */
extern void device_dispatcher( unsigned int priority, unsigned int imask );

/**
 * スプリアス・イベント・ハンドラ
 */
void spurious_exc_handler(VP p_excinf);
void spurious_int_handler();

#endif /* _MACRO_ONLY */
#endif /* _SYS_CONFIG_H_ */
