/*
 * @file  protocol_module_url.cpp
 * @brief protocol module of any protocol.
 * @brief this module never keep session persistence.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2009  NTT COMWARE Corporation.
 *
 * 2013-10-07 modified for v3.1.0(Kenji Takeda)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/
#include <boost/xpressive/xpressive.hpp>
#include <vector>
#include <string>
#include <list>
#include <algorithm>
#include <iostream>
#include <boost/asio/ip/tcp.hpp>
#include <boost/format.hpp>
#include <boost/xpressive/xpressive.hpp>
#include "protocol_module_url.h"
#include "utility.h"

#include <boost/lambda/lambda.hpp>
#include <boost/lambda/if.hpp>

namespace l7vs
{
#ifdef	DEBUG
std::string	eventtag_to_string( protocol_module_base::EVENT_TAG in ){
	return
	  in == protocol_module_base::INITIALIZE				?	"INITIALIZE"
	: in == protocol_module_base::ACCEPT					?	"ACCEPT"
	: in == protocol_module_base::CLIENT_RECV				?	"CLIENT_RECV"
	: in == protocol_module_base::REALSERVER_SELECT			?	"REALSERVER_SELECT"
	: in == protocol_module_base::REALSERVER_CONNECT		?	"REALSERVER_CONNECT"
	: in == protocol_module_base::REALSERVER_SEND			?	"REALSERVER_SEND"
	: in == protocol_module_base::SORRYSERVER_SELECT		?	"SORRYSERVER_SELECT"
	: in == protocol_module_base::SORRYSERVER_CONNECT		?	"SORRYSERVER_CONNECT"
	: in == protocol_module_base::SORRYSERVER_SEND			?	"SORRYSERVER_SEND"
	: in == protocol_module_base::REALSERVER_RECV			?	"REALSERVER_RECV"
	: in == protocol_module_base::SORRYSERVER_RECV			?	"SORRYSERVER_RECV"
	: in == protocol_module_base::CLIENT_SELECT				?	"CLIENT_SELECT"
	: in == protocol_module_base::CLIENT_CONNECTION_CHECK	?	"CLIENT_CONNECTION_CHECK"
	: in == protocol_module_base::CLIENT_SEND				?	"CLIENT_SEND"
	: in == protocol_module_base::CLIENT_RESPONSE_SEND		?	"CLIENT_RESPONSE_SEND"
	: in == protocol_module_base::REALSERVER_DISCONNECT		?	"REALSERVER_DISCONNECT"
	: in == protocol_module_base::SORRYSERVER_DISCONNECT	?	"SORRYSERVER_DISCONNECT"
	: in == protocol_module_base::CLIENT_DISCONNECT			?	"CLIENT_DISCONNECT"
	: in == protocol_module_base::REALSERVER_CLOSE			?	"REALSERVER_CLOSE"
	: in == protocol_module_base::FINALIZE					?	"FINALIZE"
	: in == protocol_module_base::STOP						?	"STOP"
	: 															"NOT_FOUND"
	;
}

std::string	session_thread_data_to_string( const boost::shared_ptr< protocol_module_url::session_thread_data_url > in_ptr ){
	boost::format	fmt(	"DATA : Thread ID[%d]\n"
							"		Thread_division[%d]\n"
							"		Pair Thread ID[%d]\n"
							"		end_flag[%d]\n"
							"		accept_end_flag[%d]\n"
							"		sorry_flag[%d]\n"
							"		sorryserver_switch_flag[%d]\n"
							"		realserver_switch_flag[%d]\n"
							"		target_endpoint[%s:%d]\n"
							"		client_endpoint[%s:%d]\n"
							"		last_status[%d]\n"
						);
	fmt % in_ptr->thread_id
		% in_ptr->thread_division
		% in_ptr->pair_thread_id
		% in_ptr->end_flag
		% in_ptr->accept_end_flag
		% in_ptr->sorry_flag
		% in_ptr->sorryserver_switch_flag
		% in_ptr->realserver_switch_flag
		% in_ptr->target_endpoint.address().to_string() % in_ptr->target_endpoint.port()
		% in_ptr->client_endpoint_tcp.address().to_string() % in_ptr->client_endpoint_tcp.port()
		% eventtag_to_string( in_ptr->last_status );

	return fmt.str();
}


#endif
const std::string protocol_module_url::MODULE_NAME = "url";
const int protocol_module_url::THREAD_DIVISION_UP_STREAM = 0;
const int protocol_module_url::THREAD_DIVISION_DOWN_STREAM = 1;

const int protocol_module_url::END_FLAG_OFF = 0;
const int protocol_module_url::END_FLAG_ON = 1;

const int protocol_module_url::ACCEPT_END_FLAG_OFF = 0;
const int protocol_module_url::ACCEPT_END_FLAG_ON = 1;

const int protocol_module_url::SORRY_FLAG_ON = 1;
const int protocol_module_url::SORRY_FLAG_OFF = 0;

const int protocol_module_url::SORRYSERVER_SWITCH_FLAG_OFF = 0;
const int protocol_module_url::SORRYSERVER_SWITCH_FLAG_ON = 1;

const int protocol_module_url::REALSERVER_SWITCH_FLAG_OFF = 0;
const int protocol_module_url::REALSERVER_SWITCH_FLAG_ON = 1;

const int protocol_module_url::EDIT_DIVISION_NO_EDIT = 0;
const int protocol_module_url::EDIT_DIVISION_EDIT = 1;

const int protocol_module_url::FORWARDED_FOR_OFF = 0;
const int protocol_module_url::FORWARDED_FOR_ON = 1;

const int protocol_module_url::COLLECT_STATS_OFF = 0;
const int protocol_module_url::COLLECT_STATS_ON = 1;

//! constructor
protocol_module_url::protocol_module_url() :
        http_protocol_module_base(MODULE_NAME), forwarded_for(FORWARDED_FOR_OFF)
{
        sorry_uri.assign('\0');
        sorry_uri[0] = '/';
}
//! destructor
protocol_module_url::~protocol_module_url()
{
}
//! tcp protocol support check
//! @return tcp support is true
//! @return tcp not-support is false
bool protocol_module_url::is_tcp()
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100000, "in/out_function : bool protocol_module_url::is_tcp() : "
                            "return_value = true.", __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        return true;
}

//! udp protocol support check
//! @return udp support is true
//! @return udp not-support is false
bool protocol_module_url::is_udp()
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100001, "in/out_function : bool protocol_module_url::is_udp() : "
                            "return_value = false.", __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        return false;
}

//! replication interval interrupt
//! timer thread call this function. from virtualservice.
void protocol_module_url::replication_interrupt()
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100002, "in/out_function : void protocol_module_url::"
                            "replication_interrupt().", __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
}
//! initialize function. called from module control. module loaded call
//! @param[in]    realserver list iterator begin function object type
//!    @param[in]    realserver list iterator end function object type
//! @param[in]    realserver list iterator next function object type
//! @param[in]    realserver list mutex lock function object type.
//! @param[in]    realserver list mutex unlock function object type
void protocol_module_url::initialize(rs_list_itr_func_type    inlist_begin,
                rs_list_itr_func_type    inlist_end,
                rs_list_itr_next_func_type    inlist_next,
                boost::function< void(void) >    inlist_lock,
                boost::function< void(void) >    inlist_unlock)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100003, "in_function : void protocol_module_url::initialize("
                            "rs_list_itr_func_type inlist_begin, rs_list_itr_func_type inlist_end, "
                            "rs_list_itr_next_func_type inlist_next, boost::function< void( void ) > "
                            "inlist_lock, boost::function< void( void ) > inlist_unlock).", __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        //RealServer list begin function
        rs_list_begin = inlist_begin;
        //RealServer list end function
        rs_list_end = inlist_end;
        //RealServer list next function
        rs_list_next = inlist_next;
        //RealServer list lock function
        rs_list_lock = inlist_lock;
        //RealServer list unlock function
        rs_list_unlock = inlist_unlock;

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100004, "out_function : void protocol_module_url::initialize("
                            "rs_list_itr_func_type inlist_begin, rs_list_itr_func_type inlist_end, "
                            "rs_list_itr_next_func_type inlist_next, boost::function< void( void ) > "
                            "inlist_lock, boost::function< void( void ) > inlist_unlock).", __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
}

//! finalize called from module control. module unloaded call.
void protocol_module_url::finalize()
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100005, "in_function : void protocol_module_url::finalize().", __FILE__,
                            __LINE__);
        }
        /*------DEBUG LOG END------*/

        //RealServer list functions initialization
        //RealServer list begin function
        rs_list_begin.clear();
        //RealServer list end function
        rs_list_end.clear();
        //RealServer list next function
        rs_list_next.clear();
        //RealServer list lock function
        rs_list_lock.clear();
        //RealServer list unlock function
        rs_list_unlock.clear();

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100006, "function : void protocol_module_url::finalize() : "
                            "rs_list_begin.clear(), rs_list_end.clear(), rs_list_next.clear(), rs_list_lock.clear(), rs_list_unlock.clear() end.", __FILE__,
                            __LINE__);
        }
        /*------DEBUG LOG END------*/

        //Replication functions initialization
        //component memory allocate function
        replication_pay_memory.clear();
        //component memory lock function
        replication_area_lock.clear();
        //component memory unlock function
        replication_area_unlock.clear();

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100007, "function : void protocol_module_url::finalize() : "
                            "replication_pay_memory.clear(), replication_area_lock.clear(), replication_area_unlock.clear() end.", __FILE__,
                            __LINE__);
        }
        /*------DEBUG LOG END------*/

        //ScheduleModule's functions initialization
        schedule_tcp.clear();

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100008, "function : void protocol_module_url::finalize() : "
                            "schedule_tcp.clear(), schedule_udp.clear() end.", __FILE__,
                            __LINE__);
        }
        /*------DEBUG LOG END------*/

        //Module's option initialization
        //forwarded_for
        forwarded_for = FORWARDED_FOR_OFF;
        //sorry-uri
        sorry_uri.assign('\0');
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100009, "function : void protocol_module_url::finalize() : "
                            "forwarded_for = 0, sorry_uri.assign('\\0') end.", __FILE__,
                            __LINE__);
        }
        /*------DEBUG LOG END------*/
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100010, "out_function : void protocol_module_url::finalize().", __FILE__,
                            __LINE__);
        }
        /*------DEBUG LOG END------*/

        //logger functions initialization
        //log level getting function
        getloglevel.clear();
        //logger(Fatal)
        putLogFatal.clear();
        //logger(Error)
        putLogError.clear();
        //logger(Warn)
        putLogWarn.clear();
        //logger(Info)
        putLogInfo.clear();
        //logger(Debug)
        putLogDebug.clear();
}

//! sorry support check
//! @return true sorry mode is supported.
//! @return false sorry mode is unsupported.
bool protocol_module_url::is_use_sorry()
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100011, "in/out_function : bool protocol_module_url::is_use_sorry() : "
                            "return_value = true.", __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        return true;
}

//! realserver list update event
void protocol_module_url::handle_rslist_update()
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100012, "in/out_function : void protocol_module_url::handle_rslist_update().", __FILE__,
                            __LINE__);
        }
        /*------DEBUG LOG END------*/
}

//! module parameter check.used by l7vsadm
//! @param[in]    module parameter string list
//! @return    result.flag true is parameter is no problem.
//! @return result.flag false is parameter is problem.
protocol_module_base::check_message_result protocol_module_url::check_parameter(const std::vector <
                std::string > & args)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::check_message_result "
                                        "protocol_module_url::check_parameter("
                                        "const std::vector<std::string>& args) : args = %s.");
                std::string argsdump;
                for (std::vector<std::string>::const_iterator it = args.begin(); it != args.end(); ++it) {
                        argsdump += *it;
                        argsdump += " ";
                }
                formatter % argsdump;
                putLogDebug(100015, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        using namespace boost::xpressive;
	using namespace boost::lambda;
        //set check result flag true
        check_message_result check_result;
        check_result.flag = true;
        bool forward_checked = false;
        bool sorryuri_checked = false;
	bool status_checked = false;
	// For check '-P/--pattern-match'&'-RS/--end-point' is correct 
	bool pattern_checked = false;

        // cf RFC 2396 (A. Collected BNF for URI)
        sregex    sorry_uri_regex
        =    +('/' >>
               *(
                 alpha | digit |
                 (set = '-', '_', '.', '!', '~', '*', '\'', '(', ')') |
                 '%' >> repeat<2>(xdigit) |
                 (set = ':', '@', '&', '=', '+', '$', ',')
               )
               >>
               *(';' >>
                 *(
                   alpha | digit |
                   (set = '-', '_', '.', '!', '~', '*', '\'', '(', ')') | // mark
                   '%' >> repeat<2>(xdigit) | // escaped
                   (set = ':', '@', '&', '=', '+', '$', ',')
                 ) // pchar
               ) // param
             ) // segment
             >>
             !('?' >>
               *(
                 (set = ';', '/', '?', ':', '@', '&', '=', '+', '$', ',') | //reserved
                 alpha | digit |
                 (set = '-', '_', '.', '!', '~', '*', '\'', '(', ')') | // mark
                 '%' >> repeat<2>(xdigit) // escaped
               ) // uric
             ) // query
             >>
             !('#' >>
               *(
                 (set = ';', '/', '?', ':', '@', '&', '=', '+', '$', ',') | //reserved
                 alpha | digit |
                 (set = '-', '_', '.', '!', '~', '*', '\'', '(', ')') | // mark
                 '%' >> repeat<2>(xdigit) // escaped
               ) // uric
             ); // fragment

	typedef std::vector<std::string>::const_iterator vec_str_it;
	typedef boost::asio::ip::tcp tcp_type;	

	try {
	        vec_str_it it = args.begin();
	        vec_str_it it_end = args.end();
		
		pattern_endpointlist_map ptn_eplist_map_for_check;
		pattern_endpointlist_map_it ptn_eplist_map_it;
		pattern_endpointlist_map_it ptn_eplist_map_it_end;

		boost::lambda::placeholder1_type X;

	        //loop option strings
		while( it != it_end ){

                        //option string = "-F"
                        if (*it == "-F" || *it == "--forwarded-for") {
                                //set forward flag  ON
                                forward_checked = true;
                        }
                        //option string = "-S"
                        else if (*it == "-S" || *it == "--sorry-uri") {
                                //set sorryURI flag OFF
                                if (!sorryuri_checked) {
                                        //next item exist
                                        if (++it != it_end) {
                                                if (!it->empty() && (it->substr(0, 1) == "-" || it->substr(0, 2) == "--")) {
                                                        //set check result flag false
                                                        check_result.flag = false;
                                                        //set check result message
                                                        check_result.message = "You have to set option value '-S/--sorry-uri'.";
                                                        putLogError(100000, check_result.message, __FILE__, __LINE__);
                                                        //loop break;
                                                        break;
                                                }
                                                //next option string's length > 127
                                                if (it->size() > MAX_OPTION_SIZE - 1) {
                                                        std::ostringstream ostr;
                                                        ostr << "'-S/--sorry-uri' option value '" << *it << "' is too long.";

                                                        //set check result flag false
                                                        check_result.flag = false;
                                                        //set check result message
                                                        check_result.message = ostr.str();

                                                        putLogError(100001, check_result.message, __FILE__, __LINE__);
                                                        //loop break;
                                                        break;
                                                }
                                                //next option string's length <= 127
                                                else {
                                                        //regex check
                                                        if (regex_match(*it, sorry_uri_regex)) {
                                                                //check OK
                                                                //set sorryURI flag ON
                                                                sorryuri_checked = true;
                                                        }
                                                        //check NG
                                                        else {
                                                                std::ostringstream ostr;
                                                                ostr << "'-S/--sorry-uri' option value '" << *it << "' is not a valid URI.";

                                                                //set check result flag false
                                                                check_result.flag = false;
                                                                //set check result message
                                                                check_result.message = ostr.str();
                                                                putLogError(100002, check_result.message, __FILE__, __LINE__);
                                                                //loop break
                                                                break;
                                                        }
                                                }
                                        }
                                        //next item is not exist
                                        else {
                                                //set check flag false
                                                check_result.flag = false;
                                                //set check result message
                                                check_result.message = "You have to set option value '-S/--sorry-uri'.";
                                                putLogError(100003, check_result.message, __FILE__,
                                                            __LINE__);
                                                //loop break
                                                break;
                                        }
                                }
                                //sorryURI flag = ON
                                else {
                                        //set check result flag false
                                        check_result.flag = false;
                                        //set check result message
                                        check_result.message = "Cannot set multiple option '-S/--sorry-uri'.";
                                        putLogError(100004, check_result.message, __FILE__,
                                                    __LINE__);
                                        //loop break
                                        break;
                                }
                        }
                        //option string = "-c/--statistic"
                        else if (*it == "-c" || *it == "--statistic") {
                                //statistic flag is OFF
                                if (!status_checked) {
                                        //next item exist
                                        if(++it != it_end) {
                                            //collect statistic flag must be 0 or 1
                                            if(*it == "0" || *it == "1"){
                                                        //check OK
                                                        //set statistic flag ON
                                                        status_checked = true;                                            }
                                            else {
                                                        std::ostringstream ostr;
                                                        ostr << "'-c/--statistic' option value '" << *it << "' is not a valid value.";

                                                        //set check result flag false
                                                        check_result.flag = false;
                                                        //set check result message
                                                        check_result.message = ostr.str();
                                                        putLogError(100128, check_result.message, __FILE__, __LINE__);
                                                        //loop break
                                                        break;
                                                }
                                        }
                                        //next item is not exist
                                        else {
                                                //set check flag false
                                                check_result.flag = false;
                                                //set check result message
                                                check_result.message = "You have to set option value '-c/--statistic'.";
                                                putLogError(100129, check_result.message, __FILE__,
                                                            __LINE__);
                                                //loop break
                                                break;
                                        }
                                }
                                //statistic flag is ON
                                else {
                                        //set check result flag false
                                        check_result.flag = false;
                                        //set check result message
                                        check_result.message = "Cannot set multiple option '-c/--statistic'.";
                                        putLogError(100130, check_result.message, __FILE__,
                                                    __LINE__);
                                        //loop break
                                        break;
                                }
                        }
			// URL pattern set 
			else if ( *it == "-P" || *it == "--pattern-match" ) {

				if( ++it != it_end ){
					std::list< tcp_type::endpoint > temp_endpoint_list;
					// Check option uri regex grammer.
					if( !checkOptionURIregex( *it, check_result ) ){
						break;
					}
	
					boost::array<char,MAX_OPTION_SIZE> ary_buf;
					strcpy( ary_buf.begin(), it->c_str() );

					ptn_eplist_map_it_end = ptn_eplist_map_for_check.end();
					ptn_eplist_map_it = ptn_eplist_map_for_check.find( ary_buf );
					
					if( ptn_eplist_map_it != ptn_eplist_map_it_end ){
						check_result.flag = false;
						std::ostringstream ostr;						
						ostr << "Pattern '" << *it << "' is already set.";
						check_result.message = ostr.str();
						putLogError( 9999, check_result.message, __FILE__, __LINE__);			
						break;
					}

					if( ++it != it_end ){
						bool chkrs = true;
						if( ( *it == "-RS" || *it == "--end-point" ) ){
							++it;
							chkrs = setRealserverEndpoint< boost::asio::ip::tcp >( it, it_end, temp_endpoint_list, check_result );
						}else{							
							check_result.flag = false;
							std::ostringstream ostr;						
							ostr << "You have to set option '-RS/--end-point' after '-P/--pattern-match'.";
							check_result.message = ostr.str();
							putLogError(9999, check_result.message, __FILE__, __LINE__);			
							break;
						}
						if( !chkrs ){							
							break;	
						}else{
							pattern_checked = true;
						}
                                        } else {											
						check_result.flag = false;
						std::ostringstream ostr;						
						ostr << "You have to set option '-RS/--end-point' after '-P/--pattern-match'.";
						check_result.message = ostr.str();
						putLogError(9999, check_result.message, __FILE__, __LINE__);			
						break;
      					}

					ptn_eplist_map_for_check.insert(
						std::map<
							boost::array< char, MAX_OPTION_SIZE >,
							std::list< tcp_type::endpoint >
						>::value_type( ary_buf, temp_endpoint_list )
					);

				}
	                        //next item not exist
                                else {
                                        //set check result flag false
                                        check_result.flag = false;
					std::ostringstream ostr;
					ostr << "You have to set option value -P/pattern-match'.";
                                        //set check result message
                                        check_result.message = ostr.str();
                                        putLogError(9999, check_result.message, __FILE__,
                                                    __LINE__);

                                	break;
				}
			}
                        //other option string
                        else {
                                //set check result flag false
                                check_result.flag = false;
                                //set check result message
                                check_result.message = "Option error.";
                                putLogError(100005, check_result.message, __FILE__, __LINE__);
                                //loop break
                                break;
                        }

			if( it == it_end ){
				break;
			}else{
				++it;
			}
	        }
		if ( check_result.flag && !pattern_checked ){
			// set check result flag false
			check_result.flag = false;
			check_result.message = "You have to set option '-P/--pattern-match' and '-RS/--end-point'.";
			putLogError( 9999, check_result.message, __FILE__, __LINE__ );
		}
        } catch (const std::exception &ex) {
                check_result.flag = false;
                std::cerr << "protocol_module_url::check_parameter() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::check_message_result "
                                        "protocol_module_url::check_parameter() exception : "
                                        "error = %s.");
                formatter % ex.what();
                putLogError(100006, formatter.str(), __FILE__, __LINE__);
        } catch (...) {
                check_result.flag = false;
                std::cerr << "protocol_module_url::check_parameter() : Unknown exception." << std::endl;
                putLogError(100007, "function : protocol_module_base::check_message_result "
                            "protocol_module_url::check_parameter() : "
                            "Unknown exception.", __FILE__, __LINE__);
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::check_message_result "
                                        "protocol_module_url::check_parameter("
                                        "const std::vector<std::string>& args) : return_value = ("
                                        "check_message_result.flag = %d, check_message_result.message = %s).");
                formatter % check_result.flag % check_result.message;
                putLogDebug(100016, formatter.str(), __FILE__, __LINE__);
        }
        /*-------- DEBUG LOG END--------*/

        return check_result;	

}

//! parameter set
//! @param[in] module parameter string list
//! @return    result.flag true is parameter is no problem.
//! @return result.flag false is parameter is problem.
protocol_module_base::check_message_result protocol_module_url::set_parameter(const std::vector <
                std::string > & args)
{

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::check_message_result "
                                        "protocol_module_url::set_parameter("
                                        "const std::vector<std::string>& args) : args = %s.");
                std::string argsdump;
                for (std::vector<std::string>::const_iterator it = args.begin(); it != args.end(); ++it) {
                        argsdump += *it;
                        argsdump += " ";
                }
                formatter % argsdump;
                putLogDebug(100015, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        using namespace boost::xpressive;
	using namespace boost::lambda;
        //set check result flag true
        check_message_result check_result;
        check_result.flag = true;
        bool forward_checked = false;
        bool sorryuri_checked = false;
	bool pattern_checked = false;
	bool status_checked = false;

        // cf RFC 2396 (A. Collected BNF for URI)
        sregex    sorry_uri_regex
        =    +('/' >>
               *(
                 alpha | digit |
                 (set = '-', '_', '.', '!', '~', '*', '\'', '(', ')') |
                 '%' >> repeat<2>(xdigit) |
                 (set = ':', '@', '&', '=', '+', '$', ',')
               )
               >>
               *(';' >>
                 *(
                   alpha | digit |
                   (set = '-', '_', '.', '!', '~', '*', '\'', '(', ')') | // mark
                   '%' >> repeat<2>(xdigit) | // escaped
                   (set = ':', '@', '&', '=', '+', '$', ',')
                 ) // pchar
               ) // param
             ) // segment
             >>
             !('?' >>
               *(
                 (set = ';', '/', '?', ':', '@', '&', '=', '+', '$', ',') | //reserved
                 alpha | digit |
                 (set = '-', '_', '.', '!', '~', '*', '\'', '(', ')') | // mark
                 '%' >> repeat<2>(xdigit) // escaped
               ) // uric
             ) // query
             >>
             !('#' >>
               *(
                 (set = ';', '/', '?', ':', '@', '&', '=', '+', '$', ',') | //reserved
                 alpha | digit |
                 (set = '-', '_', '.', '!', '~', '*', '\'', '(', ')') | // mark
                 '%' >> repeat<2>(xdigit) // escaped
               ) // uric
             ); // fragment

	typedef std::vector<std::string>::const_iterator vec_str_it;
	typedef boost::asio::ip::tcp tcp_type;	

        //set forwarded flag true
        forwarded_for = FORWARDED_FOR_ON;

	try {
	        vec_str_it it = args.begin();
	        vec_str_it it_end = args.end();
		
		pattern_endpointlist_map_it ptn_eplist_map_it;
		pattern_endpointlist_map_it ptn_eplist_map_it_end;

	        //loop option strings
		while( it != it_end ){

                        //option string = "-F"
                        if (*it == "-F" || *it == "--forwarded-for") {
                                //set forwarded flag ON
                                forward_checked = true;
                        }
                        //option string  = "-S"
                        else if (*it == "-S" || *it == "--sorry-uri") {
                                //sorryURI flag = OFF
                                if (!sorryuri_checked) {
                                        //next item exist
                                        if (++it != it_end) {
                                                if (!it->empty() && (it->substr(0, 1) == "-" || it->substr(0, 2) == "--")) {
                                                        //set check result flag false
                                                        check_result.flag = false;
                                                        //set check result message
                                                        check_result.message = "You have to set option value '-S/--sorry-uri'.";
                                                        putLogError(100008, check_result.message, __FILE__,
                                                                    __LINE__);
                                                        //loop break
                                                        break;
                                                }
                                                //next option string's length > 127
                                                if (it->size() > MAX_OPTION_SIZE - 1) {
                                                        std::ostringstream ostr;
                                                        ostr << "'-S/--sorry-uri' option value '" << *it << "' is too long.";

                                                        //set check result flag false
                                                        check_result.flag = false;
                                                        //set check result message
                                                        check_result.message = ostr.str();
                                                        putLogError(100009, check_result.message, __FILE__,
                                                                    __LINE__);
                                                        //loop break
                                                        break;
                                                }
                                                //next option string's length <= 127
                                                else {
                                                        //regex check
                                                        //check OK
                                                        if (regex_match(*it, sorry_uri_regex)) {
                                                                sorryuri_checked = true;
                                                                strcpy(sorry_uri.data(), it->c_str());
                                                        }
                                                        //check NG
                                                        else {
                                                                std::ostringstream ostr;
                                                                ostr << "'-S/--sorry-uri' option value '" << *it << "' is not a valid URI.";

                                                                //set check result flag false
                                                                check_result.flag = false;
                                                                //set check result message
                                                                check_result.message = ostr.str();
                                                                putLogError(100010, check_result.message, __FILE__,
                                                                            __LINE__);

                                                                break;
                                                        }
                                                }
                                        }
                                        //next item not exist
                                        else {
                                                //set check result flag false
                                                check_result.flag = false;
                                                //set check result message
                                                check_result.message = "You have to set option value '-S/--sorry-uri'.";
                                                putLogError(100011, check_result.message, __FILE__,
                                                            __LINE__);

                                                break;
                                        }
                                }
                                //sorryURI flag = ON
                                else {
                                        //set check result flag false
                                        check_result.flag = false;
                                        //set check result message
                                        check_result.message = "Cannot set multiple option '-S/--sorry-uri'.";
                                        putLogError(100012, check_result.message, __FILE__,
                                                    __LINE__);

                                        break;
                                }
                        }
                        //option string = "-c/--statistic"
                        else if (*it == "-c" || *it == "--statistic") {
                                //statistic flag is OFF
                                if (!status_checked) {
                                        //next item exist
                                        if(++it != it_end) {
                                            //collect statistic flag must be 0 or 1
                                            if(*it == "0" || *it == "1"){
                                                        //check OK
                                                        //set statistic flag ON
                                                        status_checked = true;

                                                        //set collect statistic flag
                                                        statistic = boost::lexical_cast<int>(*it);                                          }
                                            else {
                                                        std::ostringstream ostr;
                                                        ostr << "'-c/--statistic' option value '" << *it << "' is not a valid value.";

                                                        //set check result flag false
                                                        check_result.flag = false;
                                                        //set check result message
                                                        check_result.message = ostr.str();
                                                        putLogError(100131, check_result.message, __FILE__, __LINE__);
                                                        //loop break
                                                        break;
                                                }
                                        }
                                        //next item is not exist
                                        else {
                                                //set check flag false
                                                check_result.flag = false;
                                                //set check result message
                                                check_result.message = "You have to set option value '-c/--statistic'.";
                                                putLogError(100132, check_result.message, __FILE__,
                                                            __LINE__);
                                                //loop break
                                                break;
                                        }
                                }
                                //statistic flag is ON
                                else {
                                        //set check result flag false
                                        check_result.flag = false;
                                        //set check result message
                                        check_result.message = "Cannot set multiple option '-c/--statistic'.";
                                        putLogError(100133, check_result.message, __FILE__,
                                                    __LINE__);
                                        //loop break
                                        break;
                                }
                        }
			// URL pattern set 
			else if ( *it == "-P" || *it == "--pattern-match" ) {
				if( ++it != it_end ){

					std::list< tcp_type::endpoint > temp_endpoint_list;
					// Check option uri regex grammer.
					if( !checkOptionURIregex( *it, check_result ) ){
						break;
					}

					boost::array<char,MAX_OPTION_SIZE> ary_buf;
					strcpy( ary_buf.begin(), it->c_str() );
					
					ptn_eplist_map_it = ptn_eplist_map.find( ary_buf );
					ptn_eplist_map_it_end = ptn_eplist_map.end();
					
					if( ptn_eplist_map_it != ptn_eplist_map_it_end ){
						check_result.flag = false;
						std::ostringstream ostr;						
						ostr << "Pattern '" << *it << "' is already set.";
						check_result.message = ostr.str();
						putLogError( 9999, check_result.message, __FILE__, __LINE__);			
						break;
					}

					if( ++it != it_end ){
						bool chkrs = true;
						if( ( *it == "-RS" || *it == "--end-point" ) ){
							++it;
							chkrs = setRealserverEndpoint< boost::asio::ip::tcp >( it, it_end, temp_endpoint_list, check_result );
						}else{							
							check_result.flag = false;
							std::ostringstream ostr;						
							ostr << "You have to set option '-RS/--end-point' after '-P/--pattern-match'.";
							check_result.message = ostr.str();
							putLogError(9999, check_result.message, __FILE__, __LINE__);			
							break;
						}
						if( !chkrs ){							
							break;	
						}else{
							pattern_checked = true;
						}
                                        } else {											
						check_result.flag = false;
						std::ostringstream ostr;						
						ostr << "You have to set option '-RS/--end-point' after '-P/--pattern-match'.";
						check_result.message = ostr.str();
						putLogError(9999, check_result.message, __FILE__, __LINE__);			
						break;
      					}

					ptn_eplist_map.insert(
						std::map<
							boost::array< char, MAX_OPTION_SIZE >,
							std::list< tcp_type::endpoint >
						>::value_type( ary_buf, temp_endpoint_list )
					);
					
					ptnsrgx_converter_pairlist.push_back(
						std::pair<
							boost::array< char, MAX_OPTION_SIZE >,
							sregex
						>( ary_buf, sregex::compile( ary_buf.data() ) )
					);

				}
	                        //next item not exist
                                else {
                                        //set check result flag false
                                        check_result.flag = false;
                                        //set check result message
                                        check_result.message = "You have to set option value '-P/pattern-match'.";
                                        putLogError(9999, check_result.message, __FILE__,
                                                    __LINE__);

                                	break;
				}
			}
                        //others
                        else {
                                //set check result flag false
                                check_result.flag = false;
                                //set check result message
                                check_result.message = "Option error.";

                                putLogError(100013, check_result.message, __FILE__,
                                            __LINE__);

                                break;
                        }
			
			if( it == it_end ){
				break;
			}else{
				++it;
			}
	        }
		if ( check_result.flag ){
			if( !pattern_checked ){
				// set check result flag false
				check_result.flag = false;
				check_result.message = "You have to set option '-P/--pattern-match' and '-RS/--end-point'.";
				putLogError( 9999, check_result.message, __FILE__, __LINE__ );
			}else{
	                        //forward flag = OFF
	                        if (!forward_checked) {
	                                forwarded_for = FORWARDED_FOR_OFF;
	                        }
	
	                        //collect statistic flag = OFF
	                        if (!status_checked) {
	                                statistic = COLLECT_STATS_OFF;
	                        }
			}
		}

        } catch (const std::exception &ex) {
                check_result.flag = false;
                std::cerr << "protocol_module_url::set_parameter() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::check_message_result "
                                        "protocol_module_url::set_parameter() : exception : "
                                        "error = %s.");
                formatter % ex.what();
                putLogError(100014, formatter.str(), __FILE__, __LINE__);
        } catch (...) {
                check_result.flag = false;
                std::cerr << "protocol_module_url::set_parameter() : Unknown exception." << std::endl;
                putLogError(100015, "function : protocol_module_base::check_message_result "
                            "protocol_module_url::set_parameter() : "
                            "Unknown exception.", __FILE__, __LINE__);
	}

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::check_message_result "
                                        "protocol_module_url::set_parameter("
                                        "const std::vector<std::string>& args) : return_value = ("
                                        "check_message_result.flag = %d, check_message_result.message = %s).");
                formatter % check_result.flag % check_result.message;
                putLogDebug(100016, formatter.str(), __FILE__, __LINE__);
        }

        return check_result;	

}

//! parameter add
//! @param[in] module parameter string list
//! @return    result.flag true is parameter is no problem.
//! @return result.flag false is parameter is problem.
protocol_module_base::check_message_result protocol_module_url::add_parameter(const std::vector <
                std::string > & args)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::check_message_result "
                                        "protocol_module_url::add_parameter("
                                        "const std::vector<std::string>& args) : args = %s.");
                std::string argsdump;
                for (std::vector<std::string>::const_iterator it = args.begin(); it != args.end(); ++it) {
                        argsdump += *it;
                        argsdump += " ";
                }
                formatter % argsdump;
                putLogDebug(100017, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        check_message_result check_result;
        //set check result flag true
        check_result.flag = true;

        //param list is not empty
        if (!args.empty()) {
                //set check result flag false
                check_result.flag = false;
                //set check result message
                check_result.message = "Cannot add option.";
                putLogError(100016, check_result.message, __FILE__, __LINE__);
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::check_message_result "
                                        "protocol_module_url::add_parameter("
                                        "const std::vector<std::string>& args) : return_value = ("
                                        "check_message_result.flag = %d, check_message_result.message = %s).");
                formatter % check_result.flag % check_result.message;
                putLogDebug(100018, formatter.str(), __FILE__, __LINE__);
        }
        /*-------- DEBUG LOG --------*/

        return check_result;
}

//! get option info
//! @param[out] module parameter string
void protocol_module_url::get_option_info(std::string &option)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100019, "in_function : void protocol_module_url::get_option_info("
                            "std::string& option).", __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        boost::format option_formatter("%s--sorry-uri '%s' --statistic %d");
        option_formatter % (forwarded_for ? "--forwarded-for " : "") % sorry_uri.c_array()
                         % statistic;
        option.assign(option_formatter.str());

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : void protocol_module_url::get_option_info("
                                        "std::string& option) : option = %s.");
                formatter % option;
                putLogDebug(100020, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
}

//! TCP/IP scheduled function registration.
//! @param[in] schedule module TCP/IP scheduled function object type
void protocol_module_url::register_schedule(tcp_schedule_func_type inschedule)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100021, "in_function : void protocol_module_url::register_schedule("
                            "tcp_schedule_func_type inschedule).", __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        schedule_tcp = inschedule;
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100022, "out_function : void protocol_module_url::register_schedule("
                            "tcp_schedule_func_type inschedule).", __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
}

//! UDP scheduled function registration
//! @param[in] schedule module UDP scheduled function object type
void protocol_module_url::register_schedule(udp_schedule_func_type inschedule)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                putLogDebug(100023,
                            "in/out_function : void protocol_module_url::register_schedule(udp_schedule_func_type inschedule).",
                            __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
}

//! called from session initialize use in upstream_thread
//! @param[in]    upstream thread id.
//! @param[in]    downstream thread id
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_session_initialize(
        const boost::thread::id up_thread_id, const boost::thread::id down_thread_id,
        const boost::asio::ip::tcp::endpoint &client_endpoint_tcp,
        const boost::asio::ip::udp::endpoint &client_endpoint_udp)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_session_initialize(const boost::thread::id up_thread_id, "
                                        "const boost::thread::id down_thread_id, const boost::asio::ip::tcp::endpoint& client_endpoint_tcp, "
                                        "const boost::asio::ip::udp::endpoint& client_endpoint_udp) : "
                                        "up_thread_id = %d, down_thread_id = %d, client_endpoint_tcp = [%s]:%d.");
                formatter % up_thread_id % down_thread_id % client_endpoint_tcp.address().to_string() % client_endpoint_tcp.port() ;
                putLogDebug(100024, formatter.str(), __FILE__, __LINE__);
        }

        EVENT_TAG status = FINALIZE;

        //session thread initialization
        try {
                thread_data_ptr p_up(new session_thread_data_url);
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("new : address = &(%d), size = %lu.");
                        formatter % static_cast<void *>(p_up.get()) % sizeof(session_thread_data_url);
                        putLogDebug(100025, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                p_up->thread_id = up_thread_id;
                p_up->thread_division = THREAD_DIVISION_UP_STREAM;
                p_up->pair_thread_id = down_thread_id;
                p_up->accept_end_flag = ACCEPT_END_FLAG_OFF;
                p_up->end_flag = END_FLAG_OFF;
                p_up->sorry_flag = SORRY_FLAG_OFF;
                p_up->sorryserver_switch_flag = SORRYSERVER_SWITCH_FLAG_OFF;
                p_up->realserver_switch_flag = REALSERVER_SWITCH_FLAG_OFF;
                p_up->last_status = INITIALIZE;
                p_up->client_endpoint_tcp = client_endpoint_tcp;

                receive_data recv_data;
                p_up->receive_data_map[client_endpoint_tcp] = recv_data;

                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        // data dump
                        boost::format
                        formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                  "handle_session_initialize() : session_thread_data_url(up_thread_id) : "
                                  "thread_id = %d, thread_division = %d, "
                                  "pair_thread_id = %d, accept_end_flag = %d, end_flag = %d, "
                                  "sorry_flag = %d, sorryserver_switch_flag = %d, realserver_switch_flag = %d, last_status = %d, client_endpoint_tcp = [%s]:%d.");
                        formatter % p_up->thread_id % p_up->thread_division % p_up->pair_thread_id % p_up->accept_end_flag
                        % p_up->end_flag % p_up->sorry_flag % p_up->sorryserver_switch_flag % p_up->realserver_switch_flag
                        % p_up->last_status % client_endpoint_tcp.address().to_string() % client_endpoint_tcp.port();
                        putLogDebug(100026, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/

                thread_data_ptr p_down(new session_thread_data_url);
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("new : address = &(%d), size = %lu.");
                        formatter % static_cast<void *>(p_down.get()) % sizeof(session_thread_data_url);
                        putLogDebug(100027, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                p_down->thread_id = down_thread_id;
                p_down->thread_division = THREAD_DIVISION_DOWN_STREAM;
                p_down->pair_thread_id = up_thread_id;
                p_down->accept_end_flag = ACCEPT_END_FLAG_OFF;
                p_down->end_flag = END_FLAG_OFF;
                p_down->sorry_flag = SORRY_FLAG_OFF;
                p_down->sorryserver_switch_flag = SORRYSERVER_SWITCH_FLAG_OFF;
                p_down->realserver_switch_flag = REALSERVER_SWITCH_FLAG_OFF;
                p_down->last_status = INITIALIZE;
                p_down->client_endpoint_tcp = client_endpoint_tcp;
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        // data dump
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_session_initialize() : session_thread_data_url(down_thread_id) : "
                                                "thread_id = %d, thread_division = %d, pair_thread_id = %d, accept_end_flag = %d, end_flag = %d, "
                                                "sorry_flag = %d, sorryserver_switch_flag = %d, realserver_switch_flag = %d, "
                                                "last_status = %d, client_endpoint_tcp = [%s]:%d.");
                        formatter % p_down->thread_id % p_down->thread_division % p_down->pair_thread_id % p_down->accept_end_flag
                        % p_down->end_flag % p_down->sorry_flag % p_down->sorryserver_switch_flag
                        % p_down->realserver_switch_flag % p_down->last_status % client_endpoint_tcp.address().to_string() % client_endpoint_tcp.port();
                        putLogDebug(100028, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/

                boost::mutex::scoped_lock slock(session_thread_data_map_mutex);

                session_thread_data_map[up_thread_id] = p_up;
                session_thread_data_map[down_thread_id] = p_down;

                status = ACCEPT;
        } catch (const std::bad_alloc &) {
                std::cerr << "protocol_module_url::handle_session_initialize() : exception : Could not allocate memory." << std::endl;
                boost::format formatter("Could not allocate memory. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100017, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_session_initialize() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_session_initialize() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100018, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_session_initialize() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::check_message_result "
                                        "protocol_module_url::handle_session_initialize() : "
                                        "Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100019, formatter.str(), __FILE__, __LINE__);

                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_session_initialize(const boost::thread::id up_thread_id, "
                                        "const boost::thread::id down_thread_id, const boost::asio::ip::tcp::endpoint& client_endpoint_tcp, "
                                        "const boost::asio::ip::udp::endpoint& client_endpoint_udp) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100029, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}
//! called from session finalize use in upstream thread.
//! @param[in]    upstream thread id.
//! @param[in]    downstream thread id
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_session_finalize(
        const boost::thread::id up_thread_id, const boost::thread::id down_thread_id)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_session_finalize(const boost::thread::id up_thread_id, "
                                        "const boost::thread::id down_thread_id) : "
                                        "up_thread_id = %d, down_thread_id = %d.");
                formatter % up_thread_id % down_thread_id;
                putLogDebug(100030, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        EVENT_TAG status = STOP;

        //session thread free
        try {
                boost::mutex::scoped_lock slock(session_thread_data_map_mutex);

                session_thread_data_map_it	session_thread_data_it = session_thread_data_map.find(up_thread_id);
                if (session_thread_data_it != session_thread_data_map.end()) {
                        thread_data_ptr p_up = session_thread_data_it->second;
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                boost::format formatter("delete : address = &(%d).");
                                formatter % static_cast<void *>(p_up.get());
                                putLogDebug(100031, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        session_thread_data_map.erase(up_thread_id);
                }

                session_thread_data_it = session_thread_data_map.find(down_thread_id);
                if (session_thread_data_it != session_thread_data_map.end()) {
			thread_data_ptr p_down = session_thread_data_it->second;
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                boost::format formatter("delete : address = &(%d).");
                                formatter % static_cast<void *>(p_down.get());
                                putLogDebug(100032, formatter.str(), __FILE__,
                                            __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        session_thread_data_map.erase(down_thread_id);
                }

                status = STOP;

        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_session_finalize() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_session_finalize() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100020, formatter.str(), __FILE__, __LINE__);
                status = STOP;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_session_finalize() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_session_finalize() : "
                                        "Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100021, formatter.str(), __FILE__, __LINE__);
                status = STOP;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_session_finalize(const boost::thread::id up_thread_id, "
                                        "const boost::thread::id down_thread_id) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100033, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! called from after session accept.in client socket use in upstream thread.
//! @param[in]    upstream thread id.
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_accept(const boost::thread::id thread_id)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_accept(const boost::thread::id thread_id) : thread_id = %d.");
                formatter % thread_id;
                putLogDebug(100034, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        EVENT_TAG status = FINALIZE;
        thread_data_ptr session_data;
        session_thread_data_map_it session_thread_it;

        try {
                boost::mutex::scoped_lock slock(session_thread_data_map_mutex);

                session_thread_it = session_thread_data_map.find(thread_id);
                if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                        boost::format formatter("Invalid thread id. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100022, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                session_data = session_thread_it->second;

                //set accept end flag ON
                session_data->accept_end_flag = ACCEPT_END_FLAG_ON;

                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_accept(const boost::thread::id thread_id) : ACCEPT_END_FLAG_ON. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogDebug(100035, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/

		//sorry flag on 
                if (session_data->sorry_flag == SORRY_FLAG_ON) { 
                        //set return status 
                        status = SORRYSERVER_SELECT; 
                } 
                //sorry flag off 
                else { 
			status = CLIENT_RECV;
                } 

        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_accept() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100036, formatter.str(), __FILE__, __LINE__);
                }
                status = FINALIZE;
                /*------DEBUG LOG END------*/
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_accept() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_accept() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100023, formatter.str(), __FILE__, __LINE__);

                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_accept() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG "
                                        "protocol_module_url::handle_accept() : "
                                        "Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100024, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_accept(const boost::thread::id thread_id) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100037, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! called from after session recv in client socket. use in upstream thread.
//! @param[in]    upstream thread id
//! @param[in]    receive buffer reference.
//! @param[in]    receive length
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_client_recv(const boost::thread::id thread_id,
                const boost::array<char, MAX_BUFFER_SIZE>& recvbuffer, const size_t recvlen)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                size_t buffer_size = recvbuffer.size() < recvlen ? recvbuffer.size() : recvlen;
                std::string buffer;
                dump_memory(recvbuffer.data(), buffer_size, buffer);
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_recv(const boost::thread::id thread_id, "
                                        "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                        "const size_t recvlen) : thread_id = %d, recvbuffer = %s, recvlen = %d.");
                formatter % thread_id % buffer % recvlen;
                putLogDebug(100038, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        EVENT_TAG status = FINALIZE;
        size_t data_remain_start = 0;
        size_t data_remain_size = 0;
        size_t request_data_remain_size = 0;
        size_t header_offset = 0;
        size_t header_offset_len = 0;
        size_t content_length_header_offset = 0;
        size_t content_length_header_len = 0;
        size_t content_len_value = 0;
        size_t pos = 0;
        size_t buffer_size = 0;
        const size_t cr_lf_cr_lf_len = strlen("\r\n\r\n");
        const size_t cr_lf_len = strlen("\r\n");
        std::string str_value;
        const std::string http_header = "";
        const std::string content_header = "Content-Length";
        thread_data_ptr session_data;
        char *buffer1 = NULL;
        char *buffer2 = NULL;
        bool bret = false;
        CHECK_RESULT_TAG check_result;
        session_thread_data_map_it session_thread_it;
        receive_data_map_it receive_data_it;

        //parameter check
        if (recvlen > recvbuffer.size()) {
                std::cerr << "protocol_module_url::handle_client_recv() : Data size bigger than buffer size." << std::endl;
                boost::format formatter("Data size bigger than buffer size. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100025, formatter.str(), __FILE__, __LINE__);
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_client_recv(const boost::thread::id thread_id, "
                                                "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                                "const size_t recvlen) : return_value = %d. thread id : %d.");
                        formatter % FINALIZE % boost::this_thread::get_id();
                        putLogDebug(100039, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                return FINALIZE;
        }

        try {
                {
                        boost::mutex::scoped_lock slock(session_thread_data_map_mutex);

                        session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100026, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        session_data = session_thread_it->second;
                }

                //end flag on
                if (session_data->end_flag == END_FLAG_ON) {
                        status = CLIENT_RECV;
                }
                //end flag off
                else {
                        receive_data_it = session_data->receive_data_map.find(session_data->client_endpoint_tcp);
                        if (unlikely(receive_data_it == session_data->receive_data_map.end())) {
                                boost::format formatter("Invalid endpoint. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100027, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        receive_data &recv_data = receive_data_it->second;

                        send_status_it it = recv_data.send_status_list.begin();
                        send_status_it it_end = recv_data.send_status_list.end();

                        //status list check
                        it = std::find_if(it, it_end, data_send_ok());
                        if (unlikely(it != it_end)) {
                                boost::format formatter("Sending data is not correct. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100028, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        //status list check
                        it = recv_data.send_status_list.begin();
                        it = std::adjacent_find(it, it_end, data_send_repeated());
                        if (unlikely(it != it_end)) {
                                boost::format formatter("Sending data is not correct. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100029, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                std::string datadump;
                                boost::format formatter("\nitem[%d] : status = %d, send_end_size = %d, "
                                                        "send_rest_size = %d, send_possible_size = %d, "
                                                        "send_offset = %d, unsend_size = %d, edit_division = %d.");
                                int i = 0;
                                for (it = recv_data.send_status_list.begin();
                                     it != recv_data.send_status_list.end();
                                     ++it, ++i) {
                                        formatter % i % it->status % it->send_end_size
                                        % it->send_rest_size % it->send_possible_size
                                        % it->send_offset % it->unsend_size % it->edit_division;
                                        datadump += formatter.str();
                                }

                                formatter.parse("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_client_recv() : send status list dump : send status list size = %d.%s");

                                formatter % recv_data.send_status_list.size() % datadump;
                                putLogDebug(100040, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/

                        it = recv_data.send_status_list.begin();
                        //get original status info
                        while (it != it_end) {
                                //item status is SEND_END
                                if (it->status == SEND_END) {
                                        //erase from list
                                        recv_data.send_status_list.erase(it++);
                                        continue;
                                }
                                //item status is SEND_CONTINUE
                                else if (it->status == SEND_CONTINUE) {
                                        it->send_offset += it->send_end_size;
                                        data_remain_start = it->send_offset;
                                        break;
                                }
                                //item status is SEND_NG
                                else {
                                        data_remain_start = it->send_offset;
                                        data_remain_size = it->unsend_size;
                                        break;
                                }

                                ++it;
                        }
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                std::string datadump;
                                boost::format formatter("\nitem[%d] : status = %d, send_end_size = %d, "
                                                        "send_rest_size = %d, send_possible_size = %d, "
                                                        "send_offset = %d, unsend_size = %d, edit_division = %d.");
                                int i = 0;
                                for (it = recv_data.send_status_list.begin();
                                     it != recv_data.send_status_list.end();
                                     ++it, ++i) {
                                        formatter % i % it->status % it->send_end_size
                                        % it->send_rest_size % it->send_possible_size
                                        % it->send_offset % it->unsend_size % it->edit_division;
                                        datadump += formatter.str();
                                }

                                formatter.parse("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_client_recv() : send status list dump : send status list size = %d.%s");

                                formatter % recv_data.send_status_list.size() % datadump;
                                putLogDebug(100041, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/

                        //receive buffer process
                        //buffer rest size < request size
                        if (recv_data.receive_buffer_rest_size < recvlen) {
                                //buffer max size < remain size + request size
                                //buffer is need reallocate
                                if (recv_data.receive_buffer_max_size < data_remain_size + recvlen) {
                                        //the buffer's size that will be allocated is exceed the upper limit value
                                        if (MAX_URL_MODULE_BUFFER_SIZE < data_remain_size + recvlen) {
                                                std::cerr << "protocol_module_url::handle_client_recv() : the buffer's size that will be allocated is exceed the upper limit value." << std::endl;
                                                boost::format formatter("The buffer's size that will be allocated is exceed the upper limit value. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogError(100030, formatter.str(), __FILE__, __LINE__);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_client_recv(const boost::thread::id thread_id, "
                                                                                "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                                                                "const size_t recvlen) : return_value = %d. thread id : %d.");
                                                        formatter % FINALIZE % boost::this_thread::get_id();
                                                        putLogDebug(100042, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                return FINALIZE;
                                        }

                                        buffer_size = (data_remain_size + recvlen) > MAX_BUFFER_SIZE ? (data_remain_size + recvlen) : MAX_BUFFER_SIZE;
                                        //receive_buffer1's memory allocate and initialization
                                        buffer1 = new char[buffer_size];
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("new : address = &(%d), size = %lu.");
                                                formatter % static_cast<void *>(buffer1) % buffer_size;
                                                putLogDebug(100043, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        memset(buffer1, 0, buffer_size);
                                        //receive_buffer2's memory allocate and initialization
                                        buffer2 = new char[buffer_size];
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("new : address = &(%d), size = %lu.");
                                                formatter % static_cast<void *>(buffer2) % buffer_size;
                                                putLogDebug(100044, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        memset(buffer2, 0, buffer_size);

                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer + data_remain_start, data_remain_size, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_client_recv() : before memcpy (data dump) : "
                                                        "data begin = %d, data_size = %d, data = %s");
                                                formatter % data_remain_start % data_remain_size % datadump;
                                                putLogDebug(100045, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //copy data from old buffer to new buffer
                                        memcpy(buffer1, recv_data.receive_buffer + data_remain_start, data_remain_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(buffer1, data_remain_size, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_client_recv() : after memcpy (data dump) : "
                                                        "data begin = 0, data_size = %d, data = %s");
                                                formatter % data_remain_size % datadump;
                                                putLogDebug(100046, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/

                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recvbuffer.data(), recvlen, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_client_recv() : before memcpy (data dump) : "
                                                        "data begin = 0, data_size = %d, data = %s");
                                                formatter % recvlen % datadump;
                                                putLogDebug(100047, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        memcpy(buffer1 + data_remain_size, recvbuffer.data(), recvlen);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(buffer1 + data_remain_size, recvlen, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_client_recv() : after memcpy (data dump) : "
                                                        "data begin = %d, data_size = %d, data = %s");
                                                formatter % data_remain_size % recvlen % datadump;
                                                putLogDebug(100048, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //free old buffer1 and old buffer2
                                        if (recv_data.receive_buffer1 != NULL) {
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("delete : address = &(%d).");
                                                        formatter % static_cast<void *>(recv_data.receive_buffer1);
                                                        putLogDebug(100049, formatter.str(), __FILE__,
                                                                    __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                delete[] recv_data.receive_buffer1;
                                                recv_data.receive_buffer1 = NULL;
                                        }

                                        if (recv_data.receive_buffer2 != NULL) {
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("delete : address = &(%d).");
                                                        formatter % static_cast<void *>(recv_data.receive_buffer2);
                                                        putLogDebug(100050, formatter.str(), __FILE__,
                                                                    __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                delete[] recv_data.receive_buffer2;
                                                recv_data.receive_buffer2 = NULL;
                                        }

                                        //set new buffer pointer
                                        recv_data.receive_buffer1 = buffer1;
                                        recv_data.receive_buffer2 = buffer2;
                                        recv_data.receive_buffer = recv_data.receive_buffer1;
                                        //set new buffer's max size
                                        recv_data.receive_buffer_max_size = buffer_size;
                                }
                                //buffer's max size >= remain data size + request size
                                //buffer isn't need reallocate, but switch
                                else {
                                        //pointer valid check
                                        if (unlikely(recv_data.receive_buffer1 == NULL || recv_data.receive_buffer2 == NULL)) {
                                                boost::format formatter("Invalid pointer. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogError(100031, formatter.str(), __FILE__, __LINE__);
                                                throw - 1;
                                        }
                                        //using buffer is buffer1
                                        if (recv_data.receive_buffer == recv_data.receive_buffer1) {
                                                //buffer2 initialization
                                                memset(recv_data.receive_buffer2, 0, recv_data.receive_buffer_max_size);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(recv_data.receive_buffer + data_remain_start, data_remain_size, datadump);
                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_client_recv() : before memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                                        formatter % data_remain_start % data_remain_size  % datadump;
                                                        putLogDebug(100051, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                //copy data from buffer1 to buffer2
                                                memcpy(recv_data.receive_buffer2, recv_data.receive_buffer + data_remain_start, data_remain_size);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(recv_data.receive_buffer2, recvlen, datadump);
                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_client_recv() : after memcpy (data dump) : "
                                                                "data begin = 0, data_size = %d, data = %s");
                                                        formatter % recvlen % datadump;
                                                        putLogDebug(100052, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(recvbuffer.data(), recvlen, datadump);
                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_client_recv() : before memcpy (data dump) : "
                                                                "data begin = 0, data_size = %d, data = %s");
                                                        formatter % recvlen % datadump;
                                                        putLogDebug(100053, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                memcpy(recv_data.receive_buffer2 + data_remain_size, recvbuffer.data(), recvlen);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(recv_data.receive_buffer2 + data_remain_size, recvlen, datadump);
                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_client_recv() : after memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                                        formatter % data_remain_size % recvlen % datadump;
                                                        putLogDebug(100054, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                //set buffer2 as using buffer
                                                recv_data.receive_buffer = recv_data.receive_buffer2;
                                        }
                                        //using buffer is buffer2
                                        else {
                                                //buffer1 initialization
                                                memset(recv_data.receive_buffer1, 0, recv_data.receive_buffer_max_size);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(recv_data.receive_buffer + data_remain_start, data_remain_size, datadump);
                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_client_recv() : before memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                                        formatter % data_remain_start % data_remain_size % datadump;
                                                        putLogDebug(100055, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                //copy data from buffer2 to buffer1
                                                memcpy(recv_data.receive_buffer1, recv_data.receive_buffer + data_remain_start, data_remain_size);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(recv_data.receive_buffer1, data_remain_size, datadump);
                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_client_recv() : after memcpy (data dump) : "
                                                                "data begin = 0, data_size = %d, data = %s");
                                                        formatter % data_remain_size % datadump;
                                                        putLogDebug(100056, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(recvbuffer.data(), recvlen, datadump);
                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_client_recv() : before memcpy (data dump) : "
                                                                "data begin = 0, data_size = %d, data = %s");
                                                        formatter % recvlen % datadump;
                                                        putLogDebug(100057, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                memcpy(recv_data.receive_buffer1 + data_remain_size, recvbuffer.data(), recvlen);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(recv_data.receive_buffer1 + data_remain_size, recvlen, datadump);
                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_client_recv() : after memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                                        formatter % data_remain_size % recvlen % datadump;
                                                        putLogDebug(100058, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                //set buffer1 as using buffer
                                                recv_data.receive_buffer = recv_data.receive_buffer1;
                                        }
                                }

                                //set buffer's rest size
                                recv_data.receive_buffer_rest_size = recv_data.receive_buffer_max_size - data_remain_size - recvlen;

                                //remain_size recalc
                                data_remain_size += recvlen;

                                send_status_it it_begin = recv_data.send_status_list.begin();
                                send_status_it it_end = recv_data.send_status_list.end();

                                //offset recalc
                                for (; it_begin != it_end; ++it_begin) {
                                        it_begin->send_offset -= data_remain_start;
                                }
                        }
                        //buffer's rest size >= request size
                        else {
                                //pointer valid check
                                if (unlikely(recv_data.receive_buffer == NULL)) {
                                        boost::format formatter("Invalid pointer. thread id : %d.");
                                        formatter % boost::this_thread::get_id();
                                        putLogError(100032, formatter.str(), __FILE__, __LINE__);
                                        throw - 1;
                                }
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(recvbuffer.data(), recvlen, datadump);
                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_client_recv() : before memcpy (data dump) : "
                                                                "data begin = 0, data_size = %d, data = %s");
                                        formatter % recvlen % datadump;
                                        putLogDebug(100059, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //copy data from parameter to using buffer
                                memcpy(recv_data.receive_buffer + recv_data.receive_buffer_max_size - recv_data.receive_buffer_rest_size,
                                       recvbuffer.data(), recvlen);
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(recv_data.receive_buffer + recv_data.receive_buffer_max_size - recv_data.receive_buffer_rest_size,
                                                    recvlen, datadump);
                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_client_recv() : after memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                        formatter % (recv_data.receive_buffer_max_size - recv_data.receive_buffer_rest_size)
                                        % recvlen % datadump;
                                        putLogDebug(100060, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //buffer's rest size recalc
                                recv_data.receive_buffer_rest_size -= recvlen;
                                //remain data size recalc
                                data_remain_size += recvlen;
                        }

                        it = recv_data.send_status_list.begin();
                        it_end = recv_data.send_status_list.end();

                        //set request rest size
                        request_data_remain_size = recvlen;

                        //original status process
                        for (; it != it_end; ++it) {
                                //status is SEND_CONTINUE
                                if (it->status == SEND_CONTINUE) {
                                        //send rest size > request size
                                        if (it->send_rest_size > request_data_remain_size) {
                                                //send possible size recalc
                                                it->send_possible_size = request_data_remain_size;
                                                //send rest size recalc
                                                it->send_rest_size -= request_data_remain_size;
                                                //send end size recalc
                                                it->send_end_size = 0;
                                                //request size recalc
                                                request_data_remain_size = 0;
                                        }
                                        //send rest size <= request size
                                        else {
                                                //send possible size recalc
                                                it->send_possible_size = it->send_rest_size;
                                                //send rest size recalc
                                                request_data_remain_size -= it->send_rest_size;
                                                //send end size recalc
                                                it->send_end_size = 0;
                                                //request size recalc
                                                it->send_rest_size = 0;
                                        }
                                        //set edit_division flag off
                                        it->edit_division = EDIT_DIVISION_NO_EDIT;
                                        //set status SEND_OK
                                        it->status = SEND_OK;
                                }
                                //status is SEND_NG
                                else if (it->status == SEND_NG) {
                                        if (statistic == COLLECT_STATS_ON || forwarded_for == FORWARDED_FOR_ON) {
                                                //check http method
                                                check_result = check_http_method(recv_data.receive_buffer + it->send_offset, data_remain_size);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_client_recv() : call check_http_method : "
                                                                                "return_value = %d. thread id : %d.");
                                                        formatter % check_result % boost::this_thread::get_id();
                                                        putLogDebug(100061, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                //check http method result is CHECK_OK
                                                if (check_result == CHECK_OK) {
                                                        //check http version
                                                        check_result = check_http_version(recv_data.receive_buffer + it->send_offset, data_remain_size);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_client_recv() : call check_http_version : "
                                                                                        "return_value = %d. thread id : %d.");
                                                                formatter % check_result % boost::this_thread::get_id();
                                                                putLogDebug(100062, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                }
                                                //check method and version result is CHECK_OK
                                                if (check_result == CHECK_OK) {
                                                        //search http header
                                                        bret = find_http_header(recv_data.receive_buffer + it->send_offset, data_remain_size, http_header,
                                                                                header_offset, header_offset_len);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_client_recv() : call find_http_header : "
                                                                                        "return_value = %d. thread id : %d.");
                                                                formatter % static_cast<int>(bret) % boost::this_thread::get_id();
                                                                putLogDebug(100063, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        //search http header result is OK
                                                        if (bret) {
                                                                //search Content_Length header
                                                                bret = find_http_header(recv_data.receive_buffer + it->send_offset, data_remain_size,
                                                                                        content_header, content_length_header_offset, content_length_header_len);
                                                                /*-------- DEBUG LOG --------*/
                                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                                "handle_client_recv() : call find_http_header : "
                                                                                                "return_value = %d. thread id : %d.");
                                                                        formatter % static_cast<int>(bret) % boost::this_thread::get_id();
                                                                        putLogDebug(100064, formatter.str(), __FILE__, __LINE__);
                                                                }
                                                                /*------DEBUG LOG END------*/
                                                                //search Content_Length result is OK
                                                                if (bret) {
                                                                        //Get Content_Length header's numeric value
                                                                        for (pos = 0; recv_data.receive_buffer[it->send_offset + content_length_header_offset + pos] != ':' && pos
                                                                             < content_length_header_len; ++pos)
                                                                                ;
                                                                        if (pos == content_length_header_len) {
                                                                                throw std::string("Content_Length field's value is invalid.");
                                                                        }

                                                                        ++pos;

                                                                        str_value.assign(recv_data.receive_buffer + it->send_offset + content_length_header_offset + pos,
                                                                                         content_length_header_len - pos);

                                                                        size_t pos_end = str_value.find_last_of('\r');
                                                                        if (pos_end != std::string::npos) {
                                                                                str_value = str_value.erase(pos_end);
                                                                        }

                                                                        for (pos = 0; !isgraph(str_value[pos]) && str_value[pos] != '\0'; ++pos);

                                                                        str_value = str_value.substr(pos);

                                                                        try {
                                                                                content_len_value = boost::lexical_cast<size_t>(str_value.c_str());
                                                                        } catch (const boost::bad_lexical_cast &ex) {
                                                                                throw std::string("Content_Length field's value is invalid.");
                                                                        }

                                                                        //send_rest_size recalc
                                                                        //set whole http header's length + Content_Length's value
                                                                        it->send_rest_size = header_offset + header_offset_len + cr_lf_cr_lf_len + content_len_value;
                                                                }
                                                                //search Content_Length result is NG
                                                                else {
                                                                        //send_rest_size recalc
                                                                        //set whole http header's length
                                                                        if (header_offset_len == 0) {
                                                                                it->send_rest_size = header_offset + header_offset_len + cr_lf_len;
                                                                        } else {
                                                                                it->send_rest_size = header_offset + header_offset_len + cr_lf_cr_lf_len;
                                                                        }
                                                                }

                                                                //increment http statistics
                                                                increment_stats(recv_data.receive_buffer + it->send_offset);
                                                                /*-------- DEBUG LOG --------*/
                                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                                "handle_client_recv() : call increment_stats : thread id : %d.");
                                                                        formatter % boost::this_thread::get_id();
                                                                        putLogDebug(100263, formatter.str(), __FILE__, __LINE__);
                                                                }
                                                                /*------DEBUG LOG END------*/

                                                                //set edit_division flag on
                                                                it->edit_division = EDIT_DIVISION_EDIT;
                                                        }
                                                        //search http header result is NG
                                                        else {
                                                                //unsend_size recalc
                                                                it->unsend_size += request_data_remain_size;
                                                                //request data rest size recalc
                                                                request_data_remain_size = 0;
                                                                break;
                                                        }
                                                }
                                                //check method and version result is CHECK_NG
                                                else if (check_result == CHECK_NG) {
                                                        //set edit_division flag off
                                                        it->edit_division = EDIT_DIVISION_NO_EDIT;
                                                        //send_rest_size recalc
                                                        it->send_rest_size = it->unsend_size + request_data_remain_size;
                                                }
                                                //check method and version result is CHECK_IMPOSSIBLE
                                                else {
                                                        //unsend_size recalc
                                                        it->unsend_size += request_data_remain_size;
                                                        //request data rest size recalc
                                                        request_data_remain_size = 0;
                                                        break;
                                                }
                                        } else {
                                                //set edit_division flag off
                                                it->edit_division = EDIT_DIVISION_NO_EDIT;
                                                //send_rest_size recalc
                                                it->send_rest_size = it->unsend_size + request_data_remain_size;
                                        }

                                        //recalc fields value according to send_rest_size and request rest size
                                        if (it->send_rest_size > it->unsend_size + request_data_remain_size) {
                                                it->send_possible_size = it->unsend_size + request_data_remain_size;
                                                it->send_rest_size -= (it->unsend_size + request_data_remain_size);
                                                it->send_end_size = 0;
                                                it->unsend_size = 0;
                                                request_data_remain_size = 0;
                                        } else {
                                                it->send_possible_size = it->send_rest_size;
                                                request_data_remain_size = it->unsend_size + request_data_remain_size - it->send_rest_size;
                                                it->send_end_size = 0;
                                                it->unsend_size = 0;
                                                it->send_rest_size = 0;
                                        }

                                        //change status from SEND_NG to SEND_OK
                                        it->status = SEND_OK;
                                }
                                //no request rest data to process
                                if (request_data_remain_size <= 0) {
                                        break;
                                }
                        }
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                std::string datadump;
                                boost::format formatter("\nitem[%d] : status = %d, send_end_size = %d, "
                                                        "send_rest_size = %d, send_possible_size = %d, "
                                                        "send_offset = %d, unsend_size = %d, edit_division = %d.");
                                int i = 0;
                                for (it = recv_data.send_status_list.begin();
                                     it != recv_data.send_status_list.end();
                                     ++it, ++i) {
                                        formatter % i % it->status % it->send_end_size
                                        % it->send_rest_size % it->send_possible_size
                                        % it->send_offset % it->unsend_size % it->edit_division;
                                        datadump += formatter.str();
                                }

                                formatter.parse("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_client_recv() : send status list dump : send status list size = %d.%s");

                                formatter % recv_data.send_status_list.size() % datadump;
                                putLogDebug(100065, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        //there are still rest data need to process
                        //new status created and add to status list
                        while (request_data_remain_size > 0) {
                                //new status created
                                send_status new_send_state;

                                new_send_state.edit_division = EDIT_DIVISION_NO_EDIT;
                                new_send_state.send_end_size = 0;
                                new_send_state.send_offset = 0;
                                new_send_state.send_possible_size = 0;
                                new_send_state.unsend_size = 0;
                                new_send_state.send_rest_size = 0;
                                //status initialize to SEND_NG
                                new_send_state.status = SEND_NG;
                                //add new status to status_list
                                recv_data.send_status_list.push_back(new_send_state);
                                std::list<send_status>::reverse_iterator new_send_it = recv_data.send_status_list.rbegin();
                                //calc offset
                                new_send_it->send_offset = recv_data.receive_buffer_max_size - recv_data.receive_buffer_rest_size
                                                           - request_data_remain_size;

                                if (statistic == COLLECT_STATS_ON || forwarded_for == FORWARDED_FOR_ON || session_data->sorry_flag == SORRY_FLAG_ON) {
                                        //check http method
                                        check_result = check_http_method(recv_data.receive_buffer + new_send_it->send_offset,
                                                                         request_data_remain_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_client_recv() : call check_http_method : "
                                                                        "return_value = %d. thread id : %d.");
                                                formatter % check_result % boost::this_thread::get_id();
                                                putLogDebug(100066, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //check http method result is CHECK_OK
                                        if (check_result == CHECK_OK) {
                                                //check http version
                                                check_result = check_http_version(recv_data.receive_buffer + new_send_it->send_offset,
                                                                                  request_data_remain_size);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_client_recv() : call check_http_version : "
                                                                                "return_value = %d. thread id : %d.");
                                                        formatter % check_result % boost::this_thread::get_id();
                                                        putLogDebug(100067, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                        }
                                        //check http method and version result is CHECK_OK
                                        if (check_result == CHECK_OK) {
                                                //search whole http header, get whole http header's offset and length
                                                bret = find_http_header(recv_data.receive_buffer + new_send_it->send_offset, request_data_remain_size,
                                                                        http_header, header_offset, header_offset_len);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_client_recv() : call find_http_header : "
                                                                                "return_value = %d. thread id : %d.");
                                                        formatter % check_result % boost::this_thread::get_id();
                                                        putLogDebug(100068, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                //searched whole http header
                                                if (bret) {
                                                        //search ContentLength http header, get ContentLength header's offset and length
                                                        bret = find_http_header(recv_data.receive_buffer + new_send_it->send_offset,
                                                                                request_data_remain_size, content_header, content_length_header_offset, content_length_header_len);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_client_recv() : call find_http_header : "
                                                                                        "return_value = %d. thread id : %d.");
                                                                formatter % static_cast<int>(bret) % boost::this_thread::get_id();
                                                                putLogDebug(100069, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/

                                                        //searched ContentLength http header
                                                        if (bret) {
                                                                //Get Content_Length header's numeric value
                                                                for (pos = 0;
                                                                     recv_data.receive_buffer[new_send_it->send_offset + content_length_header_offset + pos] != ':'
                                                                     && pos < content_length_header_len;
                                                                     ++pos);
                                                                if (pos == content_length_header_len) {
                                                                        throw std::string("Content_Length field's value is invalid.");
                                                                }

                                                                ++pos;

                                                                str_value.assign(recv_data.receive_buffer + new_send_it->send_offset + content_length_header_offset + pos,
                                                                                 content_length_header_len - pos);

                                                                size_t pos_end = str_value.find_last_of('\r');
                                                                if (pos_end != std::string::npos) {
                                                                        str_value = str_value.erase(pos_end);
                                                                }

                                                                for (pos = 0; !isgraph(str_value[pos]) && str_value[pos] != '\0'; ++pos);

                                                                str_value = str_value.substr(pos);
                                                                try {
                                                                        content_len_value = boost::lexical_cast<size_t>(str_value.c_str());
                                                                } catch (const boost::bad_lexical_cast &ex) {
                                                                        throw std::string("Content_Length field's value is invalid.");
                                                                }
                                                                //send_rest_size recalc
                                                                //set whole http header's  + whole http header's length + Content_Length's value
                                                                new_send_it->send_rest_size = header_offset + header_offset_len + cr_lf_cr_lf_len + content_len_value;
                                                        }
                                                        //not searched ContentLength http header
                                                        else {
                                                                //send_rest_size recalc
                                                                //set whole http header's  + whole http header's length
                                                                if (header_offset_len == 0) {
                                                                        new_send_it->send_rest_size = header_offset + header_offset_len + cr_lf_len;
                                                                } else {
                                                                        new_send_it->send_rest_size = header_offset + header_offset_len + cr_lf_cr_lf_len;
                                                                }

                                                        }

                                                        //increment http statistics
                                                        increment_stats(recv_data.receive_buffer + new_send_it->send_offset);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_client_recv() : call increment_stats : thread id : %d.");
                                                                formatter % boost::this_thread::get_id();
                                                                putLogDebug(100264, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/

                                                        //set edit_division flag on
                                                        new_send_it->edit_division = EDIT_DIVISION_EDIT;
                                                }
                                                //not searched whole http header
                                                else {
                                                        new_send_it->unsend_size = request_data_remain_size;
                                                        request_data_remain_size = 0;
                                                        break;
                                                }
                                        }
                                        //check http method or version result is CHECK_NG
                                        else if (check_result == CHECK_NG) {
                                                new_send_it->edit_division = EDIT_DIVISION_NO_EDIT;
                                                new_send_it->send_rest_size = request_data_remain_size;
                                        }

                                        //check http method or version result is CHECK_IMPOSSIBLE
                                        else {
                                                new_send_it->unsend_size = request_data_remain_size;
                                                request_data_remain_size = 0;
                                                break;
                                        }
                                } else {
                                        new_send_it->edit_division = EDIT_DIVISION_NO_EDIT;
                                        new_send_it->send_rest_size = request_data_remain_size;
                                }

                                //recalc fields value according to send_rest_size and request rest size
                                if (new_send_it->send_rest_size > request_data_remain_size) {
                                        new_send_it->send_possible_size = request_data_remain_size;
                                        new_send_it->send_rest_size -= request_data_remain_size;
                                        new_send_it->send_end_size = 0;
                                        request_data_remain_size = 0;
                                } else {
                                        new_send_it->send_possible_size = new_send_it->send_rest_size;
                                        request_data_remain_size -= new_send_it->send_rest_size;
                                        new_send_it->send_end_size = 0;
                                        new_send_it->send_rest_size = 0;
                                }

                                //change status from SEND_NG to SEND_OK
                                new_send_it->status = SEND_OK;
                        }

                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                std::string datadump;
                                boost::format formatter("\nitem[%d] : status = %d, send_end_size = %d, "
                                                        "send_rest_size = %d, send_possible_size = %d, "
                                                        "send_offset = %d, unsend_size = %d, edit_division = %d.");
                                int i = 0;
                                for (it = recv_data.send_status_list.begin();
                                     it != recv_data.send_status_list.end();
                                     ++it, ++i) {
                                        formatter % i % it->status % it->send_end_size
                                        % it->send_rest_size % it->send_possible_size
                                        % it->send_offset % it->unsend_size % it->edit_division;
                                        datadump += formatter.str();
                                }

                                formatter.parse("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_client_recv() : send status list dump : send status list size = %d.%s");

                                formatter % recv_data.send_status_list.size() % datadump;
                                putLogDebug(100070, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        //search for send_possible item in status list
                        send_status_it it_find = find_if(recv_data.send_status_list.begin(), recv_data.send_status_list.end(),
                                                         data_send_possible());
                        //the data that can be sent possible is exist
                        if (it_find != recv_data.send_status_list.end()) {
                                //sorry flag is on
                                if (session_data->sorry_flag == SORRY_FLAG_ON) {
                                        status = SORRYSERVER_CONNECT;
                                }
                                //sorry flag is off
                                else {
                                        status = REALSERVER_SELECT;
                                }
                        }
                        //the data that can be sent possible is not exist
                        else {
                                status = CLIENT_RECV;
                        }
                }
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_client_recv() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100071, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                status = FINALIZE;
        } catch (const std::string &ex) {
                std::cerr << "protocol_module_url::handle_client_recv() : exception : " << ex << std::endl;
                boost::format formatter("protocol_module_url::handle_client_recv() : exception :  %s. thread id : %d.");
                formatter % ex.c_str() % boost::this_thread::get_id();
                putLogError(100033, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (const std::bad_alloc &) {
                std::cerr << "protocol_module_url::handle_client_recv() : exception : Could not allocate memory." << std::endl;
                boost::format formatter("Could not allocate memory. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100034, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_client_recv() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_recv() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100035, formatter.str(), __FILE__, __LINE__);

                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_client_recv() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG "
                                        "protocol_module_url::handle_client_recv() : "
                                        "Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100036, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_recv(const boost::thread::id thread_id, "
                                        "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                        "const size_t recvlen) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100072, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! called from after realserver select.use in upstream thread.
//! @param[in]    upstream thread id
//! @param[out]    realserver TCP endpoint
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_realserver_select(
        const boost::thread::id thread_id, boost::asio::ip::tcp::endpoint &rs_endpoint)
{ 
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_select(const boost::thread::id thread_id, "
                                        "boost::asio::ip::tcp::endpoint & rs_endpoint) : "
                                        "thread_id = %d, rs_endpoint = [%s]:%d.");
                formatter % thread_id % rs_endpoint.address().to_string() % rs_endpoint.port();
                putLogDebug(100073, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

	using namespace boost::xpressive;
	
        EVENT_TAG status = FINALIZE;
        boost::asio::ip::tcp::endpoint tmp_endpoint;
        thread_data_ptr session_data;
        session_thread_data_map_it session_thread_it;
        session_thread_data_map_it session_thread_it_end;
        receive_data_map_it receive_data_it;
	bool ret = false;
	size_t url_offset = 0;
	size_t url_offset_len = 0;
	realserverlist_type::iterator    rs_list_itr;

        if (schedule_tcp.empty()) {
                std::cerr << "protocol_module_url::handle_realserver_select() : Schedule_tcp function is empty." << std::endl;
                boost::format formatter("Schedule_tcp function is empty. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100037, formatter.str(), __FILE__, __LINE__);
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_select(const boost::thread::id thread_id, "
                                                "boost::asio::ip::tcp::endpoint & rs_endpoint)"
                                                " : return_value = %d. thread id : %d.");
                        formatter % FINALIZE % boost::this_thread::get_id();
                        putLogDebug(100074, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                return FINALIZE;
        }

        try {
                {
                        boost::mutex::scoped_lock slock(session_thread_data_map_mutex);

                        session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100038, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        session_data = session_thread_it->second;
                }

		//endpoint check
		receive_data_it = session_data->receive_data_map.find( session_data->client_endpoint_tcp );
		if (unlikely(receive_data_it == session_data->receive_data_map.end())) {
			boost::format formatter("Invalid endpoint. thread id : %d.");
			formatter % boost::this_thread::get_id();
			putLogError(100043, formatter.str(), __FILE__, __LINE__);
			throw - 1;
		}

		receive_data &recv_data = receive_data_it->second;

		send_status_it it = recv_data.send_status_list.begin();
		send_status_it it_end = recv_data.send_status_list.end();

		//status list check
		it = std::find_if( it, it_end, data_send_ok() );
		if (unlikely(it == it_end)) {
		        boost::format formatter("Sending possible data is not existed. thread id : %d.");
		        formatter % boost::this_thread::get_id();
		        putLogError(100055, formatter.str(), __FILE__, __LINE__);
		        throw - 1;
		}

		// endpoint decide
		ret = find_uri( recv_data.receive_buffer + it->send_offset, it->send_possible_size, url_offset, url_offset_len );
		if( ret ){
			std::string tempstr = recv_data.receive_buffer; 
			
			pattern_sregex_converter_pairlist_it ptnsrgx_cvt_pairlist_it = ptnsrgx_converter_pairlist.begin();			
			pattern_sregex_converter_pairlist_it ptnsrgx_cvt_pairlist_it_end = ptnsrgx_converter_pairlist.end();

			sregex reg;
			// indicate client sends URL is matched 
			bool is_regex_match = false;
			// indicate Real Server found
			bool is_realserver_match = false;

			// search URL pattern
			while( ptnsrgx_cvt_pairlist_it != ptnsrgx_cvt_pairlist_it_end ){

				reg = ptnsrgx_cvt_pairlist_it->second;
				is_regex_match = regex_match( tempstr.substr( url_offset, url_offset_len ), reg );

				if( is_regex_match ){
					std::list< boost::asio::ip::tcp::endpoint >::iterator ep_list_it
						 = ( ptn_eplist_map.find( ptnsrgx_cvt_pairlist_it->first ) )->second.begin();
					std::list< boost::asio::ip::tcp::endpoint >::iterator ep_list_it_end
						 = ( ptn_eplist_map.find( ptnsrgx_cvt_pairlist_it->first ) )->second.end();
					// find the rs_endpoint in the rs_list
					{
						rs_list_scoped_lock scoped_lock(rs_list_lock, rs_list_unlock);

						// while( ep_list_it !=  ptn_ep_pairlist_it->second.end() ){
						while( ep_list_it != ep_list_it_end ){
							// check endpoint ( set by '-RS/--end-point' option and exist Real Server )
							for( rs_list_itr = rs_list_begin(); rs_list_itr != rs_list_end(); rs_list_itr = rs_list_next( rs_list_itr ) ){
								if ( *ep_list_it == rs_list_itr->tcp_endpoint && rs_list_itr->weight != 0 ) {
                                                                        rw_scoped_lock inc_lock(rslist_it_list_ref_count_inc_mutex);
									rslist_it_list.push_back( rs_list_itr );
									is_realserver_match = true;
									break;
								}
							}
							++ep_list_it;
						}
						if( is_realserver_match ){
                                                        rw_scoped_lock inc_lock(rslist_it_list_ref_count_inc_mutex);
							// "rs_list_end()" is rslist_it_list's sentinel
							rslist_it_list.push_back( rs_list_end() );
							// call Schedule Module in this function.
							useScheduleTCP( thread_id, rslist_it_list, rs_endpoint );
						}
					}
                                        rw_scoped_lock inc_lock(rslist_it_list_ref_count_inc_mutex);
					rslist_it_list.clear();
				}

				if( is_realserver_match ){
					break;
				}

				++ptnsrgx_cvt_pairlist_it;
			}

			if( is_regex_match ){
				if( is_realserver_match && rs_endpoint != tmp_endpoint ){
					// get the endpoint successfullyi
					session_data->target_endpoint = rs_endpoint;
					status = REALSERVER_CONNECT;					
				}else{
					// endpoint not found
		                        // set sorry flag on
		                        session_data->sorry_flag = SORRY_FLAG_ON;

        				CHECK_RESULT_TAG check_result;
                                        // check http method
                                        check_result = check_http_method( recv_data.receive_buffer + it->send_offset, it->send_possible_size );
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_select() : call check_http_method : "
                                                                        "return_value = %d. thread id : %d.");
                                                formatter % check_result % boost::this_thread::get_id();
                                                putLogDebug(100061, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/

                                        // check http method result is CHECK_OK
                                        if (check_result == CHECK_OK) {
                                                //check http version
                                                check_result = check_http_version( recv_data.receive_buffer + it->send_offset, it->send_possible_size );
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_realserver_select() : call check_http_version : "
                                                                                "return_value = %d. thread id : %d.");
                                                        formatter % check_result % boost::this_thread::get_id();
                                                        putLogDebug(100062, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                        }

					if( check_result == CHECK_OK ){
						it->edit_division = EDIT_DIVISION_EDIT;
					}

		                        /*-------- DEBUG LOG --------*/
		                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
		                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
		                                                        "handle_realserver_select() : SORRY_FLAG_ON. thread id : %d.");
		                                formatter % boost::this_thread::get_id();
		                                putLogDebug(100076, formatter.str(), __FILE__, __LINE__);
		                        }
		                        /*------DEBUG LOG END------*/

					putLogError(9999, "There is no realserver on list.", __FILE__, __LINE__);

		                        status = SORRYSERVER_SELECT;
				}
			}else{
				// client sends URL is no matched
				// set end flag on
				session_data->end_flag = END_FLAG_ON;
				/*-------- DEBUG LOG --------*/
				if (unlikely(LOG_LV_DEBUG == getloglevel())) {
					boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
				                                "handle_realserver_select() : END_FLAG_ON. thread id : %d.");
				        formatter % boost::this_thread::get_id();
				        putLogDebug(100229, formatter.str(), __FILE__, __LINE__);
				}
				/*------DEBUG LOG END------*/
				status = FINALIZE;				
			}

		}
		// URL Not found
		else{
			// set end flag on
			session_data->end_flag = END_FLAG_ON;
			/*-------- DEBUG LOG --------*/
			if (unlikely(LOG_LV_DEBUG == getloglevel())) {
				boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
			                                "handle_realserver_select() : END_FLAG_ON. thread id : %d.");
			        formatter % boost::this_thread::get_id();
			        putLogDebug(100229, formatter.str(), __FILE__, __LINE__);
			}
			/*------DEBUG LOG END------*/
			status = FINALIZE;				
		}

        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_select() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100077, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                status = FINALIZE;
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_realserver_select() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_select() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100040, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_realserver_select() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG "
                                        "protocol_module_url::handle_realserver_select() : "
                                        "Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100041, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_select(const boost::thread::id thread_id, "
                                        "boost::asio::ip::tcp::endpoint & rs_endpoint)"
                                        " : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100078, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}


//! called from after realserver select
//! @param[in]    upstream thread id
//! @param[out]    realserver UDP endpoint
//! @param[out]    sendbuffer reference
//! @param[out]    send data length
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_realserver_select(
        const boost::thread::id thread_id, boost::asio::ip::udp::endpoint &rs_endpoint, boost::array < char,
        MAX_BUFFER_SIZE > & sendbuffer, size_t &datalen)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in/out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_select(const boost::thread::id thread_id, "
                                        "boost::asio::ip::udp::endpoint& rs_endpoint, boost::array<char,MAX_BUFFER_SIZE>& sendbuffer, "
                                        "size_t& datalen) : "
                                        "return_value = %d. thread id : %d.");
                formatter % STOP % boost::this_thread::get_id();
                putLogDebug(100079, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        return STOP;
}
//! called from after realserver connect
//! @param[in]    upstream thread id
//! @param[out]    sendbuffer reference
//! @param[out]    send data length
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_realserver_connect(
        const boost::thread::id thread_id, boost::array<char, MAX_BUFFER_SIZE>& sendbuffer, size_t &datalen)
{

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_connect(const boost::thread::id thread_id, "
                                        "boost::array<char, MAX_BUFFER_SIZE>& sendbuffer, size_t& datalen) : "
                                        "thread_id = %d.");
                formatter % thread_id;
                putLogDebug(100080, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        EVENT_TAG status = FINALIZE;
        bool ret = false;
        size_t header_offset = 0;
        size_t header_offset_len = 0;
        size_t send_buffer_remian_size = 0;
        size_t copy_size = 0;
        const int send_buffer_end_size = sendbuffer.max_size();
        const std::string http_header = "";
        const std::string str_forword_for = "X-Forwarded-For";
        thread_data_ptr session_data;

        try {
			session_thread_data_map_mutex.lock();

			//thread id check
			session_thread_data_map_it	session_thread_it = session_thread_data_map.find(thread_id);
			if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
				boost::format formatter("Invalid thread id. thread id : %d.");
				formatter % boost::this_thread::get_id();
				putLogError(100042, formatter.str(), __FILE__, __LINE__);
				session_thread_data_map_mutex.unlock();
				throw - 1;
			}

			session_data = session_thread_it->second;

			//endpoint check
			receive_data_map_it	receive_data_it = session_data->receive_data_map.find(session_data->client_endpoint_tcp);
			if (unlikely(receive_data_it == session_data->receive_data_map.end())) {
				boost::format formatter("Invalid endpoint. thread id : %d.");
				formatter % boost::this_thread::get_id();
				putLogError(100043, formatter.str(), __FILE__, __LINE__);
				session_thread_data_map_mutex.unlock();
				throw - 1;
			}
	
			//receive_buffer pointer check
			 receive_data &recv_data = receive_data_it->second;
			if (unlikely(recv_data.receive_buffer == NULL)) {
				session_thread_data_map_mutex.unlock();
				return CLIENT_RECV;
			}

			//send list check
			send_status_it it = recv_data.send_status_list.begin();
			send_status_it it_end = recv_data.send_status_list.end();
			it = find_if(it, it_end, data_send_possible());
			if (unlikely(it == it_end)) {
				boost::format formatter("Sending possible data is not existed. thread id : %d.");
				formatter % boost::this_thread::get_id();
				putLogError(100045, formatter.str(), __FILE__, __LINE__);
				session_thread_data_map_mutex.unlock();
				throw - 1;
			}
			session_thread_data_map_mutex.unlock();

                //send buffer rest size initialization
                send_buffer_remian_size = send_buffer_end_size;

                //edit_division flag on
                if (it->edit_division == EDIT_DIVISION_EDIT  && forwarded_for == FORWARDED_FOR_ON) {
                        //edit list is empty
                        if (it->edit_data_list.empty()) {
                                //edit data create
                                edit_data edata;
                                edata.data_size = 0;
                                edata.insert_posission = 0;
                                edata.replace_size = 0;
                                //search X-Forwarded-For header
                                ret = find_http_header(recv_data.receive_buffer + it->send_offset, it->send_possible_size,
                                                       str_forword_for, header_offset, header_offset_len);
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_realserver_connect() : call find_http_header : "
                                                                "return_value = %d. thread id : %d.");
                                        formatter % static_cast<int>(ret) % boost::this_thread::get_id();
                                        putLogDebug(100081, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //search http header result is OK
                                if (ret) {
                                        //edit X-Forwarded-For header, set it to edata.data
                                        edata.data.assign(recv_data.receive_buffer + it->send_offset + header_offset, header_offset_len);
                                        edata.data += ", ";
                                        edata.data += session_data->client_endpoint_tcp.address().to_string();
                                        //save new X-Forwarded-For header offset
                                        edata.insert_posission = header_offset;
                                        //save new X-Forwarded-For header length
                                        edata.data_size = edata.data.size();
                                        //save old X-Forwarded-For header length
                                        edata.replace_size = header_offset_len;
                                }
                                //search http header result is NG
                                else {
                                        //search whole http header, get whole http header's offset and length
                                        ret = find_http_header(recv_data.receive_buffer + it->send_offset, it->send_possible_size, "",
                                                               header_offset, header_offset_len);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_connect() : call find_http_header : "
                                                                        "return_value = %d. thread id : %d.");
                                                formatter % static_cast<int>(ret) % boost::this_thread::get_id();
                                                putLogDebug(100082, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        if (!ret) {
                                                boost::format formatter("find_http_header() function failure. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogError(100046, formatter.str(), __FILE__, __LINE__);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_realserver_connect(const boost::thread::id thread_id, "
                                                                                "boost::array<char,MAX_BUFFER_SIZE>& sendbuffer, "
                                                                                "size_t& datalen) : return_value = %d. thread id : %d.");
                                                        formatter % FINALIZE % boost::this_thread::get_id();
                                                        putLogDebug(100083, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                return FINALIZE;
                                        }
                                        //create X-Forwarded-For header, put it to edata.data
                                        edata.data = str_forword_for;
                                        edata.data += ": ";
                                        edata.data += session_data->client_endpoint_tcp.address().to_string();
                                        edata.data += "\r\n";
                                        //save new X-Forwarded-For header offset
                                        edata.insert_posission = header_offset;
                                        //save new X-Forwarded-For header length
                                        edata.data_size = edata.data.size();
                                        //save old X-Forwarded-For header length
                                        edata.replace_size = 0;
                                }

                                //add to edit_data_list
                                it->edit_data_list.push_back(edata);
                        }

                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_connect() : Copy data loop start. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogDebug(100084, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        while (true) {
                                //edit_data_list is empty
                                if (it->edit_data_list.empty()) {
                                        //set edit_division flag on
                                        it->edit_division = EDIT_DIVISION_NO_EDIT;

                                        if (send_buffer_remian_size > 0 && it->send_possible_size > 0) {
                                                //send_buffer_remain_size is larger
                                                if (send_buffer_remian_size >= it->send_possible_size) {
                                                        copy_size = it->send_possible_size;
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                std::string datadump;
                                                                dump_memory(recv_data.receive_buffer + it->send_offset + it->send_end_size,
                                                                            it->send_possible_size, datadump);
                                                                boost::format formatter(
                                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_connect() : before memcpy (data dump) : "
                                                                        "data begin = %d, data_size = %d, data = %s");
                                                                formatter % (it->send_offset + it->send_end_size)
                                                                % copy_size % datadump;
                                                                putLogDebug(100085, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        //copy data from receive_buffer to sendbuffer by sending_possible size
                                                        memcpy(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                               recv_data.receive_buffer + it->send_offset + it->send_end_size,
                                                               it->send_possible_size);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                std::string datadump;
                                                                dump_memory(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                                            it->send_possible_size, datadump);

                                                                boost::format formatter(
                                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_connect() : after memcpy (data dump) : "
                                                                        "data begin = %d, data_size = %d, data = %s");
                                                                formatter % (send_buffer_end_size - send_buffer_remian_size)
                                                                % copy_size % datadump;
                                                                putLogDebug(100086, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/

                                                        it->send_end_size += copy_size;
                                                        it->send_possible_size = 0;
                                                        send_buffer_remian_size -= copy_size;
                                                }
                                                //send_possible_size is larger
                                                else {
                                                        copy_size = send_buffer_remian_size;
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                std::string datadump;
                                                                dump_memory(recv_data.receive_buffer + it->send_offset + it->send_end_size,
                                                                            copy_size, datadump);

                                                                boost::format formatter(
                                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_connect() : before memcpy (data dump) : "
                                                                        "data begin = %d, data_size = %d, data = %s");
                                                                formatter % (it->send_offset + it->send_end_size)
                                                                % copy_size % datadump;
                                                                putLogDebug(100087, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        //copy data from receive_buffer to sendbuffer by send buffer rest size
                                                        memcpy(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                               recv_data.receive_buffer + it->send_offset + it->send_end_size, copy_size);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                std::string datadump;
                                                                dump_memory(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                                            copy_size, datadump);

                                                                boost::format formatter(
                                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_connect() : after memcpy (data dump) : "
                                                                        "data begin = %d, data_size = %d, data = %s");
                                                                formatter % (send_buffer_end_size - send_buffer_remian_size)
                                                                % copy_size % datadump;
                                                                putLogDebug(100088, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        it->send_end_size += copy_size;
                                                        it->send_possible_size -= copy_size;
                                                        send_buffer_remian_size = 0;
                                                }
                                        }
                                        break;
                                }
                                //edit_data_list is not empty
                                else {
                                        //find the item in the list which has minimum insert_position
                                        std::list<edit_data>::iterator edit_min = std::min_element(it->edit_data_list.begin(),
                                                        it->edit_data_list.end());
                                        //send_buffer_remain_size is larger than data that before X-Forwarded-For
                                        if (send_buffer_remian_size >= edit_min->insert_posission - it->send_end_size) {
                                                //copy data before X-Forwarded-For
                                                copy_size = edit_min->insert_posission - it->send_end_size;
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(recv_data.receive_buffer + it->send_offset + it->send_end_size,
                                                                    copy_size, datadump);

                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_realserver_connect() : before memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                                        formatter % (it->send_offset + it->send_end_size)
                                                        % copy_size % datadump;
                                                        putLogDebug(100089, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                memcpy(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                       recv_data.receive_buffer + it->send_offset + it->send_end_size, copy_size);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                                    copy_size, datadump);
                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_realserver_connect() : after memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                                        formatter % (send_buffer_end_size - send_buffer_remian_size)
                                                        % copy_size % datadump;
                                                        putLogDebug(100090, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                it->send_end_size += copy_size;
                                                it->send_possible_size -= copy_size;
                                                send_buffer_remian_size -= copy_size;

                                                //there is remain buffer for copy X-Forwarded-For
                                                if (send_buffer_remian_size >= edit_min->data_size) {
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                std::string datadump;
                                                                dump_memory(edit_min->data.c_str(),
                                                                            edit_min->data_size, datadump);

                                                                boost::format formatter(
                                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_connect() : before memcpy (data dump) : "
                                                                        "data begin = 0, data_size = %d, data = %s");
                                                                formatter % edit_min->data_size % datadump;
                                                                putLogDebug(100091, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        //copy  X-Forwarded-For
                                                        memcpy(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                               edit_min->data.c_str(), edit_min->data_size);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                std::string datadump;
                                                                dump_memory(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                                            edit_min->data_size, datadump);
                                                                boost::format formatter(
                                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_connect() : after memcpy (data dump) : "
                                                                        "data begin = %d, data_size = %d, data = %s");
                                                                formatter % (send_buffer_end_size - send_buffer_remian_size)
                                                                % edit_min->data_size % datadump;
                                                                putLogDebug(100092, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        it->send_end_size += edit_min->replace_size;
                                                        it->send_possible_size -= edit_min->replace_size;
                                                        send_buffer_remian_size -= edit_min->data_size;
                                                        it->edit_data_list.erase(edit_min);
                                                }
                                                //
                                                else {
                                                        break;
                                                }
                                        }
                                        //data that before X-Forwarded-For is larger than send_buffer_remain_size
                                        else {
                                                copy_size = send_buffer_remian_size;
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(recv_data.receive_buffer + it->send_offset + it->send_end_size,
                                                                    copy_size, datadump);

                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_realserver_connect() : before memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                                        formatter % (it->send_offset + it->send_end_size)
                                                        % copy_size % datadump;
                                                        putLogDebug(100093, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                //copy data as large as possible
                                                memcpy(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                       recv_data.receive_buffer + it->send_offset + it->send_end_size, copy_size);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                                    copy_size, datadump);

                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_realserver_connect() : after memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                                        formatter % (send_buffer_end_size - send_buffer_remian_size)
                                                        % copy_size % datadump;
                                                        putLogDebug(100094, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/

                                                it->send_end_size += copy_size;
                                                it->send_possible_size -= copy_size;
                                                send_buffer_remian_size -= copy_size;
                                                break;
                                        }
                                }
                        }
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_connect() : Copy data loop end. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogDebug(100095, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                }
                //edit_division flag is off
                else {
                        //copy data as large as possible
                        //send_possible_size is larger
                        if (send_buffer_remian_size >= it->send_possible_size) {
                                copy_size = it->send_possible_size;
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(recv_data.receive_buffer + it->send_offset,
                                                    copy_size, datadump);
                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_connect() : before memcpy (data dump) : "
                                                "data begin = %d, data_size = %d, data = %s");
                                        formatter % it->send_offset
                                        % copy_size % datadump;
                                        putLogDebug(100096, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //copy data by send_possible size
                                memcpy(sendbuffer.data(), recv_data.receive_buffer + it->send_offset, copy_size);
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(sendbuffer.data(), copy_size, datadump);
                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_connect() : after memcpy (data dump) : "
                                                "data begin = 0, data_size = %d, data = %s");
                                        formatter % copy_size % datadump;
                                        putLogDebug(100097, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                it->send_end_size = copy_size;
                                it->send_possible_size = 0;
                                send_buffer_remian_size -= copy_size;
                        }
                        //buffer rest size is larger
                        else {
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(recv_data.receive_buffer + it->send_offset, send_buffer_remian_size, datadump);

                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_connect() : before memcpy (data dump) : "
                                                "data begin = %d, data_size = %d, data = %s");
                                        formatter % it->send_offset
                                        % send_buffer_remian_size % datadump;
                                        putLogDebug(100098, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //copy data by buffer rest size
                                memcpy(sendbuffer.data(), recv_data.receive_buffer + it->send_offset, send_buffer_remian_size);
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(sendbuffer.data(), send_buffer_remian_size, datadump);
                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_connect() : after memcpy (data dump) : "
                                                "data begin = 0, data_size = %d, data = %s");
                                        formatter % send_buffer_remian_size % datadump;
                                        putLogDebug(100099, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                it->send_end_size = send_buffer_remian_size;
                                it->send_possible_size -= send_buffer_remian_size;
                                send_buffer_remian_size = 0;
                        }
                }

                //set copied data length
                datalen = send_buffer_end_size - send_buffer_remian_size;
                status = REALSERVER_SEND;
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_connect() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100100, formatter.str(), __FILE__, __LINE__);
                }
                status = FINALIZE;
                /*------DEBUG LOG END------*/
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_realserver_connect() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_connect() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100047, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_realserver_connect() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_connect() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100048, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_connect(const boost::thread::id thread_id, "
                                        "boost::array<char,MAX_BUFFER_SIZE>& sendbuffer, "
                                        "size_t& datalen) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100101, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! called from after realserver connection fail
//! @param[in]    upstream thread id
//! @param[in]    fail realserver endpoint reference
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_realserver_connection_fail(
        const boost::thread::id thread_id, const boost::asio::ip::tcp::endpoint &rs_endpoint)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_connection_fail(const boost::thread::id thread_id, "
                                        "const boost::asio::ip::tcp::endpoint & rs_endpoint) : "
                                        "thread_id = %d, rs_endpoint = [%s]:%d.");
                formatter % thread_id % rs_endpoint.address().to_string() % rs_endpoint.port();
                putLogDebug(100102, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        EVENT_TAG status = FINALIZE;
        thread_data_ptr session_data;
        session_thread_data_map_it session_thread_it;

        try {
                boost::mutex::scoped_lock slock(session_thread_data_map_mutex);

                session_thread_it = session_thread_data_map.find(thread_id);
                if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                        boost::format formatter("Invalid thread id. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100049, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                session_data = session_thread_it->second;

                //set end flag ON
                session_data->end_flag = END_FLAG_ON;
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_connection_fail() : END_FLAG_ON. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogDebug(100103, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                status = CLIENT_DISCONNECT;
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_connection_fail() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100104, formatter.str(), __FILE__, __LINE__);
                }
                status = FINALIZE;
                /*------DEBUG LOG END------*/
        } catch (std::exception &ex) {
                std::cerr << "protocol_module_url::handle_realserver_connection_fail() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_connection_fail() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100050, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_realserver_connection_fail() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_connection_fail() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100051, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_connection_fail(const boost::thread::id thread_id, "
                                        "const boost::asio::ip::tcp::endpoint & rs_endpoint) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100105, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;

}
//! called from after realserver send.
//! @param[in]    upstream thread id
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_realserver_send(
        const boost::thread::id thread_id)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_send(const boost::thread::id thread_id) : "
                                        "thread_id = %d.");
                formatter % thread_id;
                putLogDebug(100106, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        EVENT_TAG status = FINALIZE;
        thread_data_ptr session_data;
        session_thread_data_map_it session_thread_it;
        receive_data_map_it receive_data_it;

        try {
                {
                        boost::mutex::scoped_lock sclock(session_thread_data_map_mutex);

                        //thread_id check
                        session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100052, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        session_data = session_thread_it->second;
                }

                //endpoint check
                receive_data_it = session_data->receive_data_map.find(session_data->client_endpoint_tcp);
                if (unlikely(receive_data_it == session_data->receive_data_map.end())) {
                        boost::format formatter("Invalid endpoint. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100053, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                receive_data &recv_data = receive_data_it->second;

                send_status_it it = recv_data.send_status_list.begin();
                send_status_it it_end = recv_data.send_status_list.end();

                //status list check
                it = std::adjacent_find(it, it_end, data_send_list_incorrect());
                if (unlikely(it != it_end)) {
                        boost::format formatter("Sending possible data is invalid. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100054, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                //status list check
                it = recv_data.send_status_list.begin();
                it = std::find_if(it, it_end, data_send_ok());
                if (unlikely(it == it_end)) {
                        boost::format formatter("Sending possible data is not existed. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100055, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                //sending possible data is exist
                if (it->send_possible_size > 0) {
                        //status remain SEND_OK
                        it->status = SEND_OK;
                        //offset recalc
                        it->send_offset += it->send_end_size;

                        //insert_position recalc
                        for (std::list<edit_data>::iterator list_it = it->edit_data_list.begin(); list_it
                             != it->edit_data_list.end(); ++list_it) {
                                list_it->insert_posission -= it->send_end_size;
                        }

                        //send_end_size recalc
                        it->send_end_size = 0;
                }
                //sending possible data is not exist
                else {
                        //can receive from client continue
                        if (it->send_rest_size > 0) {
                                //change status from SEND_OK to SEND_CONTINUE
                                it->status = SEND_CONTINUE;
                        }
                        //can not receive from client continue
                        else {
                                //change status from SEND_OK to SEND_END
                                it->status = SEND_END;
                        }
                }

                it = recv_data.send_status_list.begin();
                it = find_if(it, it_end, data_send_ok());
                //send_ok item is exist
                if (it != it_end) {
                        status = REALSERVER_CONNECT;
                }
                //send_ok item is exist
                else {
                        status = CLIENT_RECV;
                }
       
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_send() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100107, formatter.str(), __FILE__, __LINE__);
                }
                status = FINALIZE;
                /*------DEBUG LOG END------*/
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_realserver_send() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_send() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100056, formatter.str(), __FILE__, __LINE__);

                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_realserver_send() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_send() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100057, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_send(const boost::thread::id thread_id) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100108, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! called from after sorryserver select
//! @param[in]    upstream thread id
//! @param[in]    sorryserver endpoint reference
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_sorryserver_select(
        const boost::thread::id thread_id, boost::asio::ip::tcp::endpoint &sorry_endpoint)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_select(const boost::thread::id thread_id, "
                                        "boost::asio::ip::tcp::endpoint& sorry_endpoint) : "
                                        "thread_id = %d, sorry_endpoint = [%s]:%d.");
                formatter % thread_id % sorry_endpoint.address().to_string() % sorry_endpoint.port();
                putLogDebug(100109, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        EVENT_TAG status = FINALIZE;
        boost::asio::ip::tcp::endpoint client_endpoint;

        thread_data_ptr session_data;
        session_thread_data_map_it session_thread_it;
        receive_data_map_it receive_data_it;

        try {
                boost::mutex::scoped_lock sclock(session_thread_data_map_mutex);

                session_thread_it = session_thread_data_map.find(thread_id);
                if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                        boost::format formatter("Invalid thread id. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100058, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                session_data = session_thread_it->second;
                //set sorry_endpoint
                session_data->target_endpoint = sorry_endpoint;

                //endpoint check
                receive_data_it = session_data->receive_data_map.find(session_data->client_endpoint_tcp);
                if (unlikely(receive_data_it == session_data->receive_data_map.end())) {
                        boost::format formatter("Invalid endpoint. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100059, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                status = SORRYSERVER_CONNECT;
       
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_select() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100110, formatter.str(), __FILE__, __LINE__);
                }
                status = FINALIZE;
                /*------DEBUG LOG END------*/
        } catch (const std::bad_alloc &ex) {
                std::cerr << "protocol_module_url::handle_sorryserver_select() : exception : Could not allocate memory." << std::endl;
                boost::format formatter("Could not allocate memory. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100060, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_sorryserver_select() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_select() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100061, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_sorryserver_select() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_select() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100062, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format
                formatter(
                        "out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                        "handle_sorryserver_select(const boost::thread::id thread_id, boost::asio::ip::tcp::endpoint& sorry_endpoint) : return_value = %d."
                        " thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100111, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! called from after sorryserver connect
//!    @param[in]    upstream thread id
//! @param[out]    send buffer reference.
//! @param[out]    send length
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_sorryserver_connect(
        const boost::thread::id thread_id, boost::array<char, MAX_BUFFER_SIZE>& sendbuffer, size_t &datalen)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_connect(const boost::thread::id thread_id, "
                                        "boost::array<char, MAX_BUFFER_SIZE>& sendbuffer, size_t& datalen) : "
                                        "thread_id = %d.");
                formatter % thread_id;
                putLogDebug(100112, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        EVENT_TAG status = FINALIZE;
        bool ret = false;
        size_t header_offset = 0;
        size_t header_offset_len = 0;
        size_t url_offset = 0;
        size_t url_offset_len = 0;
        size_t send_buffer_remian_size = 0;
        size_t copy_size = 0;
        const int send_buffer_end_size = sendbuffer.max_size();
        const std::string http_header = "";
        const std::string str_forword_for = "X-Forwarded-For";
        thread_data_ptr session_data;
        session_thread_data_map_it session_thread_it;
        receive_data_map_it receive_data_it;

        try {
                {
                        boost::mutex::scoped_lock sclock(session_thread_data_map_mutex);

                        //thread id check
                        session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100063, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        session_data = session_thread_it->second;
                }

                //endpoint check
                receive_data_it = session_data->receive_data_map.find(session_data->client_endpoint_tcp);
                if (unlikely(receive_data_it
                             == session_data->receive_data_map.end())) {
                        boost::format formatter("Invalid endpoint. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100064, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                //receive_buffer pointer check
                receive_data &recv_data = receive_data_it->second;
                if (unlikely(recv_data.receive_buffer == NULL)) {
                        status = CLIENT_RECV;
                        goto handle_sorryserver_connect_out;
/*
                        boost::format formatter("Invalid pointer. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100065, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
*/
                }

                //send list check
                send_status_it it = recv_data.send_status_list.begin();
                send_status_it it_end = recv_data.send_status_list.end();

                it = find_if(it, it_end, data_send_possible());
                if (unlikely(it == it_end)) {
                        boost::format formatter("Sending possible data is not existed. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100066, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                //send buffer rest size initialization
                send_buffer_remian_size = send_buffer_end_size;

                //edit_division flag on
                if (it->edit_division == EDIT_DIVISION_EDIT) {
                        //edit list is empty
                        if (it->edit_data_list.empty()) {
                                //edit data create
                                edit_data edata;
                                edata.data_size = 0;
                                edata.insert_posission = 0;
                                edata.replace_size = 0;
                                //search uri
                                if (strlen(sorry_uri.data()) > 0) {

                                        ret = find_uri(recv_data.receive_buffer + it->send_offset, it->send_possible_size, url_offset,
                                                       url_offset_len);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_connect() : call find_uri : "
                                                                        "return_value = %d. thread id : %d.");
                                                formatter % static_cast<int>(ret) % boost::this_thread::get_id();
                                                putLogDebug(100113, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //search http header result is OK
                                        if (ret) {
                                                //edit sorry_uri, put it to edata.data
                                                edata.data = sorry_uri.data();
                                                //save new uri offset
                                                edata.insert_posission = url_offset;
                                                //save new uri length
                                                edata.data_size = edata.data.size();
                                                //save old uri length
                                                edata.replace_size = url_offset_len;
                                                //add to edit_data_list
                                                it->edit_data_list.push_back(edata);
                                        }
                                }

                                if (forwarded_for == FORWARDED_FOR_ON) {
                                        //search X-Forwarded-For header
                                        ret = find_http_header(recv_data.receive_buffer + it->send_offset, it->send_possible_size,
                                                               str_forword_for.c_str(), header_offset, header_offset_len);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_connect() : call find_http_header : "
                                                                        "return_value = %d. thread id : %d.");
                                                formatter % static_cast<int>(ret) % boost::this_thread::get_id();
                                                putLogDebug(100114, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/

                                        //search http header result is OK
                                        if (ret) {
                                                //edit X-Forwarded-For header, put it to edata.data
                                                edata.data.assign(recv_data.receive_buffer + it->send_offset + header_offset, header_offset_len);
                                                edata.data += ", ";
                                                edata.data += session_data->client_endpoint_tcp.address().to_string();
                                                //save new X-Forwarded-For header offset
                                                edata.insert_posission = header_offset;
                                                //save new X-Forwarded-For header length
                                                edata.data_size = edata.data.size();
                                                //save old X-Forwarded-For header length
                                                edata.replace_size = header_offset_len;
                                        }
                                        //search http header result is NG
                                        else {
                                                //search whole http header, get whole http header's offset and length
                                                ret = find_http_header(recv_data.receive_buffer + it->send_offset, it->send_possible_size, "",
                                                                       header_offset, header_offset_len);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_sorryserver_connect() : call find_http_header : "
                                                                                "return_value = %d. thread id : %d.");
                                                        formatter % static_cast<int>(ret) % boost::this_thread::get_id();
                                                        putLogDebug(100115, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                if (!ret) {
                                                        boost::format formatter("find_http_header() function failure. thread id : %d.");
                                                        formatter % boost::this_thread::get_id();
                                                        putLogError(100067, formatter.str(), __FILE__, __LINE__);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_sorryserver_connect(const boost::thread::id thread_id, "
                                                                                        "boost::array<char,MAX_BUFFER_SIZE>& sendbuffer, "
                                                                                        "size_t& datalen) : return_value = %d. thread id : %d.");
                                                                formatter % FINALIZE % boost::this_thread::get_id();
                                                                putLogDebug(100116, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        return FINALIZE;
                                                }
                                                //create X-Forwarded-For header, set it to edata.data
                                                edata.data = str_forword_for;
                                                edata.data += ": ";
                                                edata.data += session_data->client_endpoint_tcp.address().to_string();
                                                edata.data += "\r\n";
                                                //save new X-Forwarded-For header offset
                                                edata.insert_posission = header_offset;
                                                //save new X-Forwarded-For header length
                                                edata.data_size = edata.data.size();
                                                //save old X-Forwarded-For header length
                                                edata.replace_size = 0;
                                        }

                                        //add to edit_data_list
                                        it->edit_data_list.push_back(edata);
                                }
                        }

                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_sorryserver_connect() : Copy data loop start. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogDebug(100117, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        while (true) {
                                //edit_data_list is empty
                                if (it->edit_data_list.empty()) {
                                        //set edit_division flag off
                                        it->edit_division = EDIT_DIVISION_NO_EDIT;

                                        if (send_buffer_remian_size > 0 && it->send_possible_size > 0) {
                                                //send_buffer_remain_size is larger
                                                if (send_buffer_remian_size > it->send_possible_size) {
                                                        //copy data from receive_buffer to sendbuffer by sending_possible size
                                                        copy_size = it->send_possible_size;
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                std::string datadump;
                                                                dump_memory(recv_data.receive_buffer + it->send_offset + it->send_end_size,
                                                                            copy_size, datadump);
                                                                boost::format formatter(
                                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_connect() : before memcpy (data dump) : "
                                                                        "data begin = %d, data_size = %d, data = %s");
                                                                formatter % (it->send_offset + it->send_end_size)
                                                                % copy_size % datadump;
                                                                putLogDebug(100118, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        memcpy(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                               recv_data.receive_buffer + it->send_offset + it->send_end_size, copy_size);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                std::string datadump;
                                                                dump_memory(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                                            copy_size, datadump);
                                                                boost::format formatter(
                                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_connect() : after memcpy (data dump) : "
                                                                        "data begin = %d, data_size = %d, data = %s");
                                                                formatter % (send_buffer_end_size - send_buffer_remian_size)
                                                                % copy_size % datadump;
                                                                putLogDebug(100119, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        it->send_end_size += copy_size;
                                                        it->send_possible_size = 0;
                                                        send_buffer_remian_size -= copy_size;
                                                }
                                                //send_possible_size is larger
                                                else {
                                                        copy_size = send_buffer_remian_size;
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                std::string datadump;
                                                                dump_memory(recv_data.receive_buffer + it->send_offset + it->send_end_size,
                                                                            copy_size, datadump);
                                                                boost::format formatter(
                                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_connect() : before memcpy (data dump) : "
                                                                        "data begin = %d, data_size = %d, data = %s");
                                                                formatter % (it->send_offset + it->send_end_size)
                                                                % copy_size % datadump;
                                                                putLogDebug(100120, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        //copy data from receive_buffer to sendbuffer by send buffer rest size
                                                        memcpy(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                               recv_data.receive_buffer + it->send_offset + it->send_end_size,
                                                               copy_size);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                std::string datadump;
                                                                dump_memory(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                                            copy_size, datadump);
                                                                boost::format formatter(
                                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_connect() : after memcpy (data dump) : "
                                                                        "data begin = %d, data_size = %d, data = %s");
                                                                formatter % (send_buffer_end_size - send_buffer_remian_size)
                                                                % copy_size % datadump;
                                                                putLogDebug(100121, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        it->send_end_size += copy_size;
                                                        it->send_possible_size -= copy_size;
                                                        send_buffer_remian_size = 0;
                                                }
                                        }

                                        break;
                                }
                                //edit_data_list is not empty
                                else {
                                        //search item which insert_position is minimum
                                        std::list<edit_data>::iterator edit_min = std::min_element(it->edit_data_list.begin(),
                                                        it->edit_data_list.end());
                                        //send_buffer_remain_size is larger than data that before X-Forwarded-For/uri
                                        if (send_buffer_remian_size >= edit_min->insert_posission - it->send_end_size) {
                                                //copy data before X-Forwarded-For/url
                                                copy_size = edit_min->insert_posission - it->send_end_size;
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(recv_data.receive_buffer + it->send_offset + it->send_end_size,
                                                                    copy_size, datadump);

                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_sorryserver_connect() : before memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                                        formatter % (it->send_offset + it->send_end_size)
                                                        % copy_size % datadump;
                                                        putLogDebug(100122, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                memcpy(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                       recv_data.receive_buffer + it->send_offset + it->send_end_size, copy_size);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                                    copy_size, datadump);
                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_sorryserver_connect() : after memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                                        formatter % (send_buffer_end_size - send_buffer_remian_size)
                                                        % copy_size % datadump;
                                                        putLogDebug(100123, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                it->send_end_size += copy_size;
                                                it->send_possible_size -= copy_size;
                                                send_buffer_remian_size -= copy_size;

                                                //there is remain buffer for copy X-Forwarded-For/url
                                                if (send_buffer_remian_size >= edit_min->data_size) {
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                std::string datadump;
                                                                dump_memory(edit_min->data.c_str(),
                                                                            edit_min->data_size, datadump);
                                                                boost::format formatter(
                                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_connect() : before memcpy (data dump) : "
                                                                        "data begin = 0, data_size = %d, data = %s");
                                                                formatter % edit_min->data_size % datadump;
                                                                putLogDebug(100124, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        //copy X-Forwarded-For/uri
                                                        memcpy(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                               edit_min->data.c_str(), edit_min->data_size);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                std::string datadump;
                                                                dump_memory(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                                            edit_min->data_size, datadump);

                                                                boost::format formatter(
                                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_connect() : after memcpy (data dump) : "
                                                                        "data begin = %d, data_size = %d, data = %s");
                                                                formatter % (send_buffer_end_size - send_buffer_remian_size)
                                                                % edit_min->data_size % datadump;
                                                                putLogDebug(100125, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        it->send_end_size += edit_min->replace_size;
                                                        it->send_possible_size -= edit_min->replace_size;
                                                        send_buffer_remian_size -= edit_min->data_size;
                                                        it->edit_data_list.erase(edit_min);
                                                }
                                                //
                                                else {
                                                        break;
                                                }
                                        }
                                        //data that before X-Forwarded-For/uri is larger than send_buffer_remain_size
                                        else {
                                                copy_size = send_buffer_remian_size;
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                                    copy_size, datadump);

                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_sorryserver_connect() : before memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                                        formatter % (it->send_offset + it->send_end_size)
                                                        % copy_size % datadump;
                                                        putLogDebug(100126, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                //copy data as large as possible
                                                memcpy(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                       recv_data.receive_buffer + it->send_offset + it->send_end_size, send_buffer_remian_size);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        std::string datadump;
                                                        dump_memory(sendbuffer.data() + send_buffer_end_size - send_buffer_remian_size,
                                                                    copy_size, datadump);

                                                        boost::format formatter(
                                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_sorryserver_connect() : after memcpy (data dump) : "
                                                                "data begin = %d, data_size = %d, data = %s");
                                                        formatter % (send_buffer_end_size - send_buffer_remian_size)
                                                        % copy_size % datadump;
                                                        putLogDebug(100127, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                it->send_end_size += copy_size;
                                                it->send_possible_size -= copy_size;
                                                send_buffer_remian_size -= copy_size;
                                                break;
                                        }
                                }
                        }
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_sorryserver_connect() : Copy data loop end. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogDebug(100128, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                }
                //edit_division flag is off
                else {
                        //copy data as large as possible
                        //send_possible_size is larger
                        if (send_buffer_remian_size >= it->send_possible_size) {
                                copy_size = it->send_possible_size;
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(recv_data.receive_buffer + it->send_offset, copy_size, datadump);

                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_connect() : before memcpy (data dump) : "
                                                "data begin = %d, data_size = %d, data = %s");
                                        formatter % it->send_offset
                                        % copy_size % datadump;
                                        putLogDebug(100129, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //copy data by send_possible size
                                memcpy(sendbuffer.data(), recv_data.receive_buffer
                                       + it->send_offset, copy_size);
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(sendbuffer.data(), copy_size, datadump);

                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_connect() : after memcpy (data dump) : "
                                                "data begin = 0, data_size = %d, data = %s");
                                        formatter % copy_size % datadump;
                                        putLogDebug(100130, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                it->send_end_size = it->send_possible_size;
                                it->send_possible_size = 0;
                                send_buffer_remian_size -= copy_size;
                        }
                        //buffer rest size is larger
                        else {
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(recv_data.receive_buffer + it->send_offset, send_buffer_remian_size, datadump);

                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_connect() : before memcpy (data dump) : "
                                                "data begin = %d, data_size = %d, data = %s");
                                        formatter % it->send_offset
                                        % send_buffer_remian_size % datadump;
                                        putLogDebug(100131, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //copy data by buffer rest size
                                memcpy(sendbuffer.data(), recv_data.receive_buffer
                                       + it->send_offset, send_buffer_remian_size);
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(sendbuffer.data(), send_buffer_remian_size, datadump);

                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_connect() : after memcpy (data dump) : "
                                                "data begin = 0, data_size = %d, data = %s");
                                        formatter % send_buffer_remian_size % datadump;
                                        putLogDebug(100132, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                it->send_end_size = send_buffer_remian_size;
                                it->send_possible_size -= send_buffer_remian_size;
                                send_buffer_remian_size = 0;
                        }
                }

                //set copied data length
                datalen = send_buffer_end_size - send_buffer_remian_size;

                status = SORRYSERVER_SEND;

        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_connect() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100133, formatter.str(), __FILE__, __LINE__);
                }
                status = FINALIZE;
                /*------DEBUG LOG END------*/
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_sorryserver_connect() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_connect() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100068, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_sorryserver_connect() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_connect() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100069, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

handle_sorryserver_connect_out:
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_connect(const boost::thread::id thread_id, "
                                        "boost::array<char,MAX_BUFFER_SIZE>& sendbuffer, "
                                        "size_t& datalen) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100134, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! called from after sorryserver connection fail
//! @param[in]    upstream thread id
//! @param[in]    sorryserver endpoint reference.
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_sorryserver_connection_fail(
        const boost::thread::id thread_id, const boost::asio::ip::tcp::endpoint &sorry_endpoint)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_connection_fail(const boost::thread::id thread_id, "
                                        "const boost::asio::ip::tcp::endpoint & sorry_endpoint) : "
                                        "thread_id = %d, sorry_endpoint = [%s]:%d.");
                formatter % thread_id % sorry_endpoint.address().to_string() % sorry_endpoint.port();
                putLogDebug(100135, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        EVENT_TAG status = FINALIZE;
        thread_data_ptr session_data;
        session_thread_data_map_it session_thread_it;

        try {
                boost::mutex::scoped_lock slock(session_thread_data_map_mutex);

                session_thread_it = session_thread_data_map.find(thread_id);
                if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                        boost::format formatter("Invalid thread id. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100070, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                session_data = session_thread_it->second;

                //set end flag on
                session_data->end_flag = END_FLAG_ON;
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_connection_fail() : END_FLAG_ON. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogDebug(100136, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/

                status = FINALIZE;
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_connection_fail() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100137, formatter.str(), __FILE__, __LINE__);
                }
                status = FINALIZE;
                /*------DEBUG LOG END------*/
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_sorryserver_connection_fail() : exception : error=" << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_connection_fail() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100071, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_sorryserver_connection_fail() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_connection_fail() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100072, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format
                formatter(
                        "out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                        "handle_sorryserver_connection_fail( const boost::thread::id thread_id, const boost::asio::ip::tcp::endpoint & sorry_endpoint) : return_value = %d."
                        " thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100138, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;

}

//! called from after sorryserver send
//! @param[in]    upstream thread id
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_sorryserver_send(
        const boost::thread::id thread_id)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_send(const boost::thread::id thread_id) : "
                                        "thread_id = %d.");
                formatter % thread_id;
                putLogDebug(100139, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        EVENT_TAG status = FINALIZE;
        thread_data_ptr session_data;
        session_thread_data_map_it session_thread_it;
        receive_data_map_it receive_data_it;

        try {
                {
                        boost::mutex::scoped_lock slock(session_thread_data_map_mutex);

                        //thread_id check
                        session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100073, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        session_data = session_thread_it->second;
                }

                //endpoint check
                receive_data_it = session_data->receive_data_map.find(session_data->client_endpoint_tcp);
                if (unlikely(receive_data_it
                             == session_data->receive_data_map.end())) {
                        boost::format formatter("Invalid endpoint. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100074, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                receive_data &recv_data = receive_data_it->second;

                send_status_it it = recv_data.send_status_list.begin();
                send_status_it it_end = recv_data.send_status_list.end();
                //status list check
                it = std::adjacent_find(it, it_end, data_send_list_incorrect());
                if (unlikely(it != it_end)) {
                        boost::format formatter("Sending possible data is invalid. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100075, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                //status list check
                it = recv_data.send_status_list.begin();
                it = find_if(it, it_end, data_send_ok());
                if (unlikely(it == it_end)) {
                        boost::format formatter("Sending possible data is not existed. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100076, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                //sending possible data is exist
                if (it->send_possible_size > 0) {
                        //status remain SEND_OK
                        it->status = SEND_OK;
                        //offset recalc
                        it->send_offset += it->send_end_size;

                        //insert_position recalc
                        for (std::list<edit_data>::iterator list_it = it->edit_data_list.begin(); list_it
                             != it->edit_data_list.end(); ++list_it) {
                                list_it->insert_posission -= it->send_end_size;
                        }

                        //send_end_size recalc
                        it->send_end_size = 0;
                }
                //sending possible data is not exist
                else {
                        //can receive from client continue
                        if (it->send_rest_size > 0) {
                                //change status from SEND_OK to SEND_CONTINUE
                                it->status = SEND_CONTINUE;
                        }
                        //can not receive from client continue
                        else {
                                //change status from SEND_OK to SEND_END
                                it->status = SEND_END;
                        }
                }

                it = recv_data.send_status_list.begin();
                it = find_if(it, it_end, data_send_ok());
                //send_ok item is exist
                if (it != it_end) {
                        status = SORRYSERVER_CONNECT;
                }
                //send_ok item is exist
                else {
                        status = CLIENT_RECV;
                }

        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_send() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100140, formatter.str(), __FILE__, __LINE__);
                }
                status = FINALIZE;
                /*------DEBUG LOG END------*/
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_sorryserver_send() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_send() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100077, formatter.str(), __FILE__, __LINE__);

                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_sorryserver_send() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_send() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100078, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_send(const boost::thread::id thread_id) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100141, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;

}

//! called from after realserver receive.for UDP
//! @param[in]    downstream thread id
//! @param[in]    realserver UDP endpoint reference
//! @param[in]    receive from realserver buffer reference
//! @param[in]    recv data length
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_realserver_recv(
        const boost::thread::id thread_id, const boost::asio::ip::udp::endpoint &rs_endpoint, const boost::array < char,
        MAX_BUFFER_SIZE > & recvbuffer, const size_t recvlen)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in/out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_recv(const boost::thread::id thread_id, "
                                        "const boost::asio::ip::udp::endpoint& rs_endpoint, "
                                        "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                        "const size_t recvlen) : "
                                        "return_value = %d. thread id : %d.");
                formatter % STOP % boost::this_thread::get_id();
                putLogDebug(100142, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        return STOP;
}

//! called from after realserver receive for TCP/IP
//! @param[in]    downstream thread id
//! @param[in]    realserver TCP/IP endpoint reference
//! @param[in]    realserver receive buffer reference.
//! @param[in]    recv data length
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_realserver_recv(
        const boost::thread::id thread_id, const boost::asio::ip::tcp::endpoint &rs_endpoint, const boost::array < char,
        MAX_BUFFER_SIZE > & recvbuffer, const size_t recvlen)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                size_t buffer_size = recvbuffer.size() < recvlen ? recvbuffer.size() : recvlen;
                std::string buffer;
                dump_memory(recvbuffer.data(), buffer_size, buffer);
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_recv(const boost::thread::id thread_id, "
                                        "const boost::asio::ip::tcp::endpoint& rs_endpoint, "
                                        "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                        "const size_t recvlen) : thread_id = %d, rs_endpoint = [%s]:%d, recvbuffer = %s, recvlen = %d.");
                formatter % thread_id % rs_endpoint.address().to_string() % rs_endpoint.port()
                % buffer % recvlen;
                putLogDebug(100143, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        EVENT_TAG status = FINALIZE;
        size_t data_remain_start = 0;
        size_t data_remain_size = 0;
        size_t request_data_remain_size = 0;
        size_t header_offset = 0;
        size_t header_offset_len = 0;
        size_t content_length_header_offset = 0;
        size_t content_length_header_len = 0;
        size_t content_len_value = 0;
        size_t pos = 0;
        size_t buffer_size = 0;
        const size_t cr_lf_cr_lf_len = strlen("\r\n\r\n");
        const size_t cr_lf_len = strlen("\r\n");
        thread_data_ptr session_data;
        char *buffer1 = NULL;
        char *buffer2 = NULL;
        std::string str_value;
        const std::string http_header = "";
        const std::string content_header = "Content-Length";
        bool bret = false;
        CHECK_RESULT_TAG check_result;
        session_thread_data_map_it session_thread_it;
        receive_data_map_it receive_data_it;

        //parameter check
        if (recvlen > recvbuffer.size()) {
                std::cerr << "protocol_module_url::handle_realserver_recv() : Data size bigger than buffer size." << std::endl;
                boost::format formatter("Data size bigger than buffer size. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100079, formatter.str(), __FILE__, __LINE__);
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_recv(const boost::thread::id thread_id, "
                                                "const boost::asio::ip::tcp::endpoint& rs_endpoint, "
                                                "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                                "const size_t recvlen) : return_value = %d. thread id : %d.");
                        formatter % FINALIZE % boost::this_thread::get_id();
                        putLogDebug(100144, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                return FINALIZE;
        }

        try {
                {
                        boost::mutex::scoped_lock slock(session_thread_data_map_mutex);

                        session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end()
                                     || session_thread_it->second == NULL)) {
#ifdef	DEBUG
								// debug code insert 
								std::stringstream	buff;
								buff << "Invalid Thread ID. Thread ID[" << boost::this_thread::get_id() << "] map has thread datas = \n";
								for( session_thread_data_map_it itr = session_thread_data_map.begin(); itr != session_thread_data_map.end(); itr++ ){
									buff << session_thread_data_to_string( itr->second ) << "\n";
								}
								putLogError(100080, buff.str(), __FILE__, __LINE__ );
#else
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100080, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
#endif
                        }

                        session_data = session_thread_it->second;
                }

                receive_data_it = session_data->receive_data_map.find(rs_endpoint);
                if (receive_data_it == session_data->receive_data_map.end()) {
                        receive_data recv_data;
                        session_data->receive_data_map[rs_endpoint] = recv_data;
                }

                session_data->target_endpoint = rs_endpoint;

                receive_data &recv_data = session_data->receive_data_map[rs_endpoint];

                send_status_it it = recv_data.send_status_list.begin();
                send_status_it it_end = recv_data.send_status_list.end();

                //status list check
                it = std::find_if(it, it_end, data_send_ok());
                if (unlikely(it != it_end)) {
                        boost::format formatter("Sending data is not correct. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100081, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                //status list check
                it = recv_data.send_status_list.begin();
                it = std::adjacent_find(it, it_end, data_send_repeated());
                if (unlikely(it != it_end)) {
                        boost::format formatter("Sending data is not correct. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100082, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        std::string datadump;
                        boost::format formatter("\nitem[%d] : status = %d, send_end_size = %d, "
                                                "send_rest_size = %d, send_possible_size = %d, "
                                                "send_offset = %d, unsend_size = %d, edit_division = %d.");
                        int i = 0;
                        for (it = recv_data.send_status_list.begin();
                             it != recv_data.send_status_list.end();
                             ++it, ++i) {
                                formatter % i % it->status % it->send_end_size
                                % it->send_rest_size % it->send_possible_size
                                % it->send_offset % it->unsend_size % it->edit_division;
                                datadump += formatter.str();
                        }

                        formatter.parse("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_recv() : send status list dump : send status list size = %d.%s");

                        formatter % recv_data.send_status_list.size() % datadump;
                        putLogDebug(100145, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                it = recv_data.send_status_list.begin();
                //get original status info
                while (it != it_end) {
                        //item status is SEND_END
                        if (it->status == SEND_END) {
                                //erase from list
                                recv_data.send_status_list.erase(it++);
                                continue;
                        }
                        //item status is SEND_CONTINUE
                        else if (it->status == SEND_CONTINUE) {
                                it->send_offset += it->send_end_size;
                                data_remain_start = it->send_offset;
                                break;
                        }
                        //item status is SEND_NG
                        else {
                                data_remain_start = it->send_offset;
                                data_remain_size = it->unsend_size;
                                break;
                        }

                        ++it;
                }
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        std::string datadump;
                        boost::format formatter("\nitem[%d] : status = %d, send_end_size = %d, "
                                                "send_rest_size = %d, send_possible_size = %d, "
                                                "send_offset = %d, unsend_size = %d, edit_division = %d.");
                        int i = 0;
                        for (it = recv_data.send_status_list.begin();
                             it != recv_data.send_status_list.end();
                             ++it, ++i) {
                                formatter % i % it->status % it->send_end_size
                                % it->send_rest_size % it->send_possible_size
                                % it->send_offset % it->unsend_size % it->edit_division;
                                datadump += formatter.str();
                        }

                        formatter.parse("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_recv() : send status list dump : send status list size = %d.%s");

                        formatter % recv_data.send_status_list.size() % datadump;
                        putLogDebug(100146, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                //receive buffer process
                //buffer rest size < request size
                if (recv_data.receive_buffer_rest_size < recvlen) {
                        //buffer max size < remain size + request size
                        //buffer is need reallocate
                        if (recv_data.receive_buffer_max_size < data_remain_size + recvlen) {
                                //the buffer's size that will be allocated is exceed the upper limit value
                                if (MAX_URL_MODULE_BUFFER_SIZE < data_remain_size + recvlen) {
                                        std::cerr << "protocol_module_url::handle_realserver_recv() : the buffer's size that will be allocated is exceed the upper limit value." << std::endl;
                                        boost::format formatter("The buffer's size that will be allocated is exceed the upper limit value. thread id : %d.");
                                        formatter % boost::this_thread::get_id();
                                        putLogError(100083, formatter.str(), __FILE__, __LINE__);

                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_recv(const boost::thread::id thread_id, "
                                                                        "const boost::asio::ip::tcp::endpoint& rs_endpoint, "
                                                                        "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                                                        "const size_t recvlen) : return_value = %d. thread id : %d.");
                                                formatter % FINALIZE % boost::this_thread::get_id();
                                                putLogDebug(100147, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/

                                        return FINALIZE;
                                }
                                //receive_buffer1's memory allocate and initialization
                                buffer_size = (data_remain_size + recvlen) > MAX_BUFFER_SIZE ? (data_remain_size + recvlen) : MAX_BUFFER_SIZE;
                                buffer1 = new char[buffer_size];
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        boost::format formatter("new : address = &(%d), size = %lu.");
                                        formatter % static_cast<void *>(buffer1) % buffer_size;
                                        putLogDebug(100148, formatter.str(), __FILE__, __LINE__);
                                }
                                /*-------- DEBUG LOG --------*/
                                memset(buffer1, 0, buffer_size);
                                //receive_buffer2's memory allocate and initialization
                                buffer2 = new char[buffer_size];
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        boost::format formatter("new : address = &(%d), size = %lu.");
                                        formatter % static_cast<void *>(buffer2) % buffer_size;
                                        putLogDebug(100149, formatter.str(), __FILE__, __LINE__);
                                }
                                /*-------- DEBUG LOG END--------*/
                                memset(buffer2, 0, buffer_size);

                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(recv_data.receive_buffer + data_remain_start, data_remain_size, datadump);

                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_recv() : before memcpy (data dump) : "
                                                "data begin = %d, data_size = %d, data = %s");
                                        formatter % data_remain_start % data_remain_size % datadump;
                                        putLogDebug(100150, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //copy data from old buffer to new buffer
                                memcpy(buffer1, recv_data.receive_buffer + data_remain_start, data_remain_size);
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(buffer1, data_remain_size, datadump);

                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_recv() : after memcpy (data dump) : "
                                                "data begin = 0, data_size = %d, data = %s");
                                        formatter % data_remain_size % datadump;
                                        putLogDebug(100151, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/

                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(recvbuffer.data(), recvlen, datadump);
                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_recv() : before memcpy (data dump) : "
                                                "data begin = 0, data_size = %d, data = %s");
                                        formatter % recvlen % datadump;
                                        putLogDebug(100152, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                memcpy(buffer1 + data_remain_size, recvbuffer.data(), recvlen);
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(buffer1 + data_remain_size, recvlen, datadump);
                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_recv() : after memcpy (data dump) : "
                                                "data begin = %d, data_size = %d, data = %s");
                                        formatter % data_remain_size % recvlen % datadump;
                                        putLogDebug(100153, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //free old buffer1 and old buffer2
                                if (recv_data.receive_buffer1 != NULL) {
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("delete : address = &(%d).");
                                                formatter % static_cast<void *>(recv_data.receive_buffer1);
                                                putLogDebug(100154, formatter.str(), __FILE__,
                                                            __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        delete[] recv_data.receive_buffer1;
                                        recv_data.receive_buffer1 = NULL;
                                }

                                if (recv_data.receive_buffer2 != NULL) {
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("delete : address = &(%d).");
                                                formatter % static_cast<void *>(recv_data.receive_buffer2);
                                                putLogDebug(100155, formatter.str(), __FILE__,
                                                            __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        delete[] recv_data.receive_buffer2;
                                        recv_data.receive_buffer2 = NULL;
                                }

                                //set new buffer pointer
                                recv_data.receive_buffer1 = buffer1;
                                recv_data.receive_buffer2 = buffer2;
                                recv_data.receive_buffer = recv_data.receive_buffer1;
                                //set new buffer's max size
                                recv_data.receive_buffer_max_size = buffer_size;
                        }
                        //buffer's max size >= remain data size + request size
                        //buffer isn't need reallocate, but switch
                        else {
                                //pointer valid check
                                if (unlikely(recv_data.receive_buffer1 == NULL || recv_data.receive_buffer2 == NULL)) {
                                        boost::format formatter("Invalid pointer. thread id : %d.");
                                        formatter % boost::this_thread::get_id();
                                        putLogError(100084, formatter.str(), __FILE__, __LINE__);
                                        throw - 1;
                                }
                                //using buffer is buffer1
                                if (recv_data.receive_buffer == recv_data.receive_buffer1) {
                                        //buffer2 initialization
                                        memset(recv_data.receive_buffer2, 0, recv_data.receive_buffer_max_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer + data_remain_start, data_remain_size, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_recv() : before memcpy (data dump) : "
                                                        "data begin = %d, data_size = %d, data = %s");
                                                formatter % data_remain_start % data_remain_size % datadump;
                                                putLogDebug(100156, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //copy data from buffer1 to buffer2
                                        memcpy(recv_data.receive_buffer2, recv_data.receive_buffer + data_remain_start, data_remain_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer2, data_remain_size, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_recv() : after memcpy (data dump) : "
                                                        "data begin = 0, data_size = %d, data = %s");
                                                formatter % data_remain_size % datadump;
                                                putLogDebug(100157, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recvbuffer.data(), recvlen, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_recv() : before memcpy (data dump) : "
                                                        "data begin = 0, data_size = %d, data = %s");
                                                formatter % data_remain_size % datadump;
                                                putLogDebug(100158, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        memcpy(recv_data.receive_buffer2 + data_remain_size, recvbuffer.data(), recvlen);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer2 + data_remain_size, recvlen, datadump);

                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_recv() : after memcpy (data dump) : "
                                                        "data begin = %d, data_size = %d, data = %s");
                                                formatter % data_remain_size % recvlen % datadump;
                                                putLogDebug(100159, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //set buffer2 as using buffer
                                        recv_data.receive_buffer = recv_data.receive_buffer2;
                                }
                                //using buffer is buffer2
                                else {
                                        //buffer1 initialization
                                        memset(recv_data.receive_buffer1, 0, recv_data.receive_buffer_max_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer + data_remain_start, data_remain_size, datadump);

                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_recv() : before memcpy (data dump) : "
                                                        "data begin = %d, data_size = %d, data = %s");
                                                formatter % data_remain_start % data_remain_size % datadump;
                                                putLogDebug(100160, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //copy data from buffer2 to buffer1
                                        memcpy(recv_data.receive_buffer1, recv_data.receive_buffer + data_remain_start, data_remain_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer1, data_remain_size, datadump);

                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_recv() : after memcpy (data dump) : "
                                                        "data begin = 0, data_size = %d, data = %s");
                                                formatter % data_remain_size % datadump;
                                                putLogDebug(100161, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recvbuffer.data(), recvlen, datadump);

                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_recv() : before memcpy (data dump) : "
                                                        "data begin = 0, data_size = %d, data = %s");
                                                formatter % recvlen % datadump;
                                                putLogDebug(100162, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        memcpy(recv_data.receive_buffer1 + data_remain_size, recvbuffer.data(), recvlen);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer1 + data_remain_size, recvlen, datadump);

                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_recv() : after memcpy (data dump) : "
                                                        "data begin = %d, data_size = %d, data = %s");
                                                formatter % data_remain_size % recvlen % datadump;
                                                putLogDebug(100163, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //set buffer1 as using buffer
                                        recv_data.receive_buffer = recv_data.receive_buffer1;
                                }
                        }

                        //set buffer's rest size
                        recv_data.receive_buffer_rest_size = recv_data.receive_buffer_max_size - data_remain_size - recvlen;

                        //remain_size recalc
                        data_remain_size += recvlen;

                        send_status_it it_begin = recv_data.send_status_list.begin();
                        send_status_it it_end = recv_data.send_status_list.end();
                        //offset recalc
                        for (; it_begin != it_end; ++it_begin) {
                                it_begin->send_offset -= data_remain_start;
                        }
                }
                //buffer's rest size >= request size
                //copy directly
                else {
                        //pointer valid check
                        if (unlikely(recv_data.receive_buffer == NULL)) {
                                boost::format formatter("Invalid pointer. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100085, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                std::string datadump;
                                dump_memory(recvbuffer.data(), recvlen, datadump);
                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_recv() : before memcpy (data dump) : "
                                                        "data begin = 0, data_size = %d, data = %s");
                                formatter % recvlen % datadump;
                                putLogDebug(100164, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/

                        //copy data from parameter to using buffer
                        memcpy(recv_data.receive_buffer + recv_data.receive_buffer_max_size - recv_data.receive_buffer_rest_size,
                               recvbuffer.data(), recvlen);
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                std::string datadump;
                                dump_memory(recv_data.receive_buffer + recv_data.receive_buffer_max_size - recv_data.receive_buffer_rest_size,
                                            recvlen, datadump);
                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_recv() : before memcpy (data dump) : "
                                                        "data begin = %d, data_size = %d, data = %s");
                                formatter % (recv_data.receive_buffer_max_size - recv_data.receive_buffer_rest_size)
                                % recvlen % datadump;
                                putLogDebug(100165, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        //buffer's rest size recalc
                        recv_data.receive_buffer_rest_size -= recvlen;
                        //remain data size recalc
                        data_remain_size += recvlen;
                }

                it = recv_data.send_status_list.begin();
                it_end = recv_data.send_status_list.end();
                //request rest size initialization
                request_data_remain_size = recvlen;
                //original status process
                for (; it != it_end; ++it) {
                        //status is SEND_CONTINUE
                        if (it->status == SEND_CONTINUE) {
                                //send rest size > request size
                                if (it->send_rest_size > request_data_remain_size) {
                                        //send possible size recalc
                                        it->send_possible_size = request_data_remain_size;
                                        //send rest size recalc
                                        it->send_rest_size -= request_data_remain_size;
                                        //send end size recalc
                                        it->send_end_size = 0;
                                        //request size recalc
                                        request_data_remain_size = 0;
                                }
                                //send rest size <= request size
                                else {
                                        //send possible size recalc
                                        it->send_possible_size = it->send_rest_size;
                                        //send rest size recalc
                                        request_data_remain_size -= it->send_rest_size;
                                        //send end size recalc
                                        it->send_end_size = 0;
                                        //request size recalc
                                        it->send_rest_size = 0;
                                }
                                //change status from SEND_CONTINUE to SEND_OK
                                it->status = SEND_OK;
                        }
                        //status is SEND_NG
                        else if (it->status == SEND_NG) {
                                if (forwarded_for == FORWARDED_FOR_ON) {
                                        //check http method
                                        check_result = check_status_code(recv_data.receive_buffer + it->send_offset, data_remain_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_recv() : call check_http_method : "
                                                                        "return_value = %d. thread id : %d.");
                                                formatter % check_result % boost::this_thread::get_id();
                                                putLogDebug(100166, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //check http method result is CHECK_OK
                                        if (check_result == CHECK_OK) {
                                                //check http version
                                                check_result = check_http_version(recv_data.receive_buffer + it->send_offset, data_remain_size);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_realserver_recv() : call check_http_version : "
                                                                                "return_value = %d. thread id : %d.");
                                                        formatter % check_result % boost::this_thread::get_id();
                                                        putLogDebug(100167, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                        }
                                        //check method and version result is CHECK_OK
                                        if (check_result == CHECK_OK) {
                                                //search http header
                                                bret = find_http_header(recv_data.receive_buffer + it->send_offset, data_remain_size, http_header,
                                                                        header_offset, header_offset_len);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_realserver_recv() : call find_http_header : "
                                                                                "return_value = %d. thread id : %d.");
                                                        formatter % static_cast<int>(bret) % boost::this_thread::get_id();
                                                        putLogDebug(100168, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                //search http header result is OK
                                                if (bret) {
                                                        //search Content_Length header
                                                        bret = find_http_header(recv_data.receive_buffer + it->send_offset, data_remain_size,
                                                                                content_header, content_length_header_offset, content_length_header_len);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_realserver_recv() : call find_http_header : "
                                                                                        "return_value = %d. thread id : %d.");
                                                                formatter % static_cast<int>(bret) % boost::this_thread::get_id();
                                                                putLogDebug(100169, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        //search Content_Length result is OK
                                                        if (bret) {
                                                                //Get Content_Length header's numeric value
                                                                for (pos = 0; recv_data.receive_buffer[it->send_offset + content_length_header_offset + pos] != ':' && pos
                                                                     < content_length_header_len; ++pos)
                                                                        ;
                                                                if (pos == content_length_header_len) {
                                                                        throw std::string("Content_Length field's value is invalid.");
                                                                }

                                                                ++pos;

                                                                str_value.assign(recv_data.receive_buffer + it->send_offset + content_length_header_offset + pos,
                                                                                 content_length_header_len - pos);

                                                                size_t pos_end = str_value.find_last_of('\r');
                                                                if (pos_end != std::string::npos) {
                                                                        str_value = str_value.erase(pos_end);
                                                                }

                                                                for (pos = 0; !isgraph(str_value[pos]) && str_value[pos] != '\0'; ++pos)
                                                                        ;

                                                                str_value = str_value.substr(pos);

                                                                try {
                                                                        content_len_value = boost::lexical_cast<size_t>(str_value.c_str());
                                                                } catch (const boost::bad_lexical_cast &ex) {
                                                                        throw std::string("Content_Length field's value is invalid.");
                                                                }
                                                                //send_rest_size recalc
                                                                //set whole http header's length + Content_Length's value
                                                                it->send_rest_size = header_offset + header_offset_len + cr_lf_cr_lf_len + content_len_value;
                                                        }
                                                        //search Content_Length result is NG
                                                        else {
                                                                //send_rest_size recalc
                                                                //set whole http header's length
                                                                if (header_offset_len == 0) {
                                                                        it->send_rest_size = header_offset + header_offset_len + cr_lf_len;
                                                                } else {
                                                                        it->send_rest_size = header_offset + header_offset_len + cr_lf_cr_lf_len;
                                                                }
                                                        }
                                                }
                                                //search http header result is CHECK_NG
                                                else {
                                                        it->unsend_size += request_data_remain_size;
                                                        request_data_remain_size = 0;
                                                        break;
                                                }
                                        }
                                        //check method and version result is CHECK_NG
                                        else if (check_result == CHECK_NG) {
                                                //send_rest_size recalc
                                                it->send_rest_size = it->unsend_size + request_data_remain_size;
                                        }
                                        //check method and version result is CHECK_IMPOSSIBLE
                                        else {
                                                it->unsend_size += request_data_remain_size;
                                                request_data_remain_size = 0;
                                                break;
                                        }
                                } else {
                                        //send_rest_size recalc
                                        it->send_rest_size = it->unsend_size + request_data_remain_size;
                                }

                                //recalc fields value according to send_rest_size and request rest size
                                if (it->send_rest_size > it->unsend_size + request_data_remain_size) {
                                        it->send_possible_size = it->unsend_size + request_data_remain_size;
                                        it->send_rest_size -= (it->unsend_size + request_data_remain_size);
                                        it->send_end_size = 0;
                                        it->unsend_size = 0;
                                        request_data_remain_size = 0;
                                } else {
                                        it->send_possible_size = it->send_rest_size;
                                        request_data_remain_size = it->unsend_size + request_data_remain_size - it->send_rest_size;
                                        it->send_end_size = 0;
                                        it->unsend_size = 0;
                                        it->send_rest_size = 0;
                                }

                                //change status from SEND_NG to SEND_OK
                                it->status = SEND_OK;
                        }
                        //no request rest data to process
                        if (request_data_remain_size <= 0) {
                                break;
                        }
                }

                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        std::string datadump;
                        boost::format formatter("\nitem[%d] : status = %d, send_end_size = %d, "
                                                "send_rest_size = %d, send_possible_size = %d, "
                                                "send_offset = %d, unsend_size = %d, edit_division = %d.");
                        int i = 0;
                        for (it = recv_data.send_status_list.begin();
                             it != recv_data.send_status_list.end();
                             ++it, ++i) {
                                formatter % i % it->status % it->send_end_size
                                % it->send_rest_size % it->send_possible_size
                                % it->send_offset % it->unsend_size % it->edit_division;
                                datadump += formatter.str();
                        }

                        formatter.parse("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_recv() : send status list dump : send status list size = %d.%s");

                        formatter % recv_data.send_status_list.size() % datadump;
                        putLogDebug(100170, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                //there are still rest data need to process
                //new status created and add to status list
                while (request_data_remain_size > 0) {
                        //new status created
                        send_status new_send_state;
                        new_send_state.edit_division = EDIT_DIVISION_NO_EDIT;
                        new_send_state.send_end_size = 0;
                        new_send_state.send_offset = 0;
                        new_send_state.send_possible_size = 0;
                        new_send_state.unsend_size = 0;
                        new_send_state.send_rest_size = 0;
                        //status initialize to SEND_NG
                        new_send_state.status = SEND_NG;
                        //add new status to status_list
                        recv_data.send_status_list.push_back(new_send_state);
                        std::list<send_status>::reverse_iterator new_send_it = recv_data.send_status_list.rbegin();
                        //calc offset
                        new_send_it->send_offset = recv_data.receive_buffer_max_size - recv_data.receive_buffer_rest_size
                                                   - request_data_remain_size;

                        if (forwarded_for == FORWARDED_FOR_ON) {
                                //check http method
                                check_result = check_status_code(recv_data.receive_buffer + new_send_it->send_offset,
                                                                 request_data_remain_size);
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_realserver_recv() : call check_http_method : "
                                                                "return_value = %d. thread id : %d.");
                                        formatter % check_result % boost::this_thread::get_id();
                                        putLogDebug(100171, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //check http method result is CHECK_OK
                                if (check_result == CHECK_OK) {
                                        //check http version
                                        check_result = check_http_version(recv_data.receive_buffer + new_send_it->send_offset,
                                                                          request_data_remain_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_recv() : call check_http_version : "
                                                                        "return_value = %d. thread id : %d.");
                                                formatter % check_result % boost::this_thread::get_id();
                                                putLogDebug(100172, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                }
                                //check http method and version result is CHECK_OK
                                if (check_result == CHECK_OK) {
                                        //search whole http header, get whole http header's offset and length
                                        bret = find_http_header(recv_data.receive_buffer + new_send_it->send_offset, request_data_remain_size,
                                                                http_header, header_offset, header_offset_len);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_recv() : call find_http_header : "
                                                                        "return_value = %d. thread id : %d.");
                                                formatter % static_cast<int>(bret) % boost::this_thread::get_id();
                                                putLogDebug(100173, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //searched whole http header
                                        if (bret) {
                                                //search ContentLength http header, get ContentLength header's offset and length
                                                bret = find_http_header(recv_data.receive_buffer + new_send_it->send_offset,
                                                                        request_data_remain_size, content_header, content_length_header_offset, content_length_header_len);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_realserver_recv() : call find_http_header : "
                                                                                "return_value = %d. thread id : %d.");
                                                        formatter % static_cast<int>(bret) % boost::this_thread::get_id();
                                                        putLogDebug(100174, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/

                                                //searched ContentLength http header
                                                if (bret) {
                                                        //Get Content_Length header's numeric value
                                                        for (pos = 0; recv_data.receive_buffer[new_send_it->send_offset + content_length_header_offset + pos] != ':'
                                                             && pos < content_length_header_len; ++pos)
                                                                ;
                                                        if (pos == content_length_header_len) {
                                                                throw std::string("Content_Length field's value is invalid.");
                                                        }

                                                        ++pos;

                                                        str_value.assign(recv_data.receive_buffer + new_send_it->send_offset + content_length_header_offset + pos,
                                                                         content_length_header_len - pos);

                                                        size_t pos_end = str_value.find_last_of('\r');
                                                        if (pos_end != std::string::npos) {
                                                                str_value = str_value.erase(pos_end);
                                                        }

                                                        for (pos = 0; !isgraph(str_value[pos]) && str_value[pos] != '\0'; ++pos)
                                                                ;

                                                        str_value = str_value.substr(pos);
                                                        try {
                                                                content_len_value = boost::lexical_cast<size_t>(str_value.c_str());
                                                        } catch (const boost::bad_lexical_cast &ex) {
                                                                throw std::string("Content_Length field's value is invalid.");
                                                        }
                                                        //send_rest_size recalc
                                                        //set whole http header's  + whole http header's length + Content_Length's value
                                                        new_send_it->send_rest_size = header_offset + header_offset_len + cr_lf_cr_lf_len + content_len_value;
                                                }
                                                //not searched ContentLength http header
                                                else {
                                                        //send_rest_size recalc
                                                        //set whole http header's  + whole http header's length
                                                        if (header_offset_len == 0) {
                                                                new_send_it->send_rest_size = header_offset + header_offset_len + cr_lf_len;
                                                        } else {
                                                                new_send_it->send_rest_size = header_offset + header_offset_len + cr_lf_cr_lf_len;
                                                        }

                                                }
                                        }
                                        //not searched whole http header
                                        else {
                                                new_send_it->unsend_size = request_data_remain_size;
                                                request_data_remain_size = 0;
                                                break;
                                        }
                                }
                                //check http method or version result is CHECK_NG
                                else if (check_result == CHECK_NG) {
                                        new_send_it->send_rest_size = request_data_remain_size;
                                }
                                //check http method or version result is CHECK_IMPOSSIBLE
                                else {
                                        new_send_it->unsend_size = request_data_remain_size;
                                        request_data_remain_size = 0;
                                        break;
                                }
                        } else {
                                new_send_it->send_rest_size = request_data_remain_size;
                        }

                        //recalc fields value according to send_rest_size and request rest size
                        if (new_send_it->send_rest_size > request_data_remain_size) {
                                new_send_it->send_possible_size = request_data_remain_size;
                                new_send_it->send_rest_size -= request_data_remain_size;
                                new_send_it->send_end_size = 0;
                                new_send_it->send_end_size = 0;
                                request_data_remain_size = 0;
                        } else {
                                new_send_it->send_possible_size = new_send_it->send_rest_size;
                                request_data_remain_size -= new_send_it->send_rest_size;
                                new_send_it->send_end_size = 0;
                                new_send_it->send_rest_size = 0;
                        }

                        //change status from SEND_NG to SEND_OK
                        new_send_it->status = SEND_OK;
                }
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        std::string datadump;
                        boost::format formatter("\nitem[%d] : status = %d, send_end_size = %d, "
                                                "send_rest_size = %d, send_possible_size = %d, "
                                                "send_offset = %d, unsend_size = %d, edit_division = %d.");
                        int i = 0;
                        for (it = recv_data.send_status_list.begin();
                             it != recv_data.send_status_list.end();
                             ++it, ++i) {
                                formatter % i % it->status % it->send_end_size
                                % it->send_rest_size % it->send_possible_size
                                % it->send_offset % it->unsend_size % it->edit_division;
                                datadump += formatter.str();
                        }

                        formatter.parse("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_recv() : send status list dump : send status list size = %d.%s");

                        formatter % recv_data.send_status_list.size() % datadump;
                        putLogDebug(100175, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/

                //search for send_possible item in status list
                send_status_it it_find = find_if(recv_data.send_status_list.begin(), recv_data.send_status_list.end(),
                                                 data_send_possible());
                //the data that can be sent possible is exist
                if (it_find != recv_data.send_status_list.end()) {
                        status = CLIENT_CONNECTION_CHECK;
                }
                //the data that can be sent possible is not exist
                else {
                        status = REALSERVER_RECV;
                }
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_recv() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100176, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                status = FINALIZE;
        } catch (const std::string &ex) {
                std::cerr << "protocol_module_url::handle_realserver_recv() : exception : " << ex << std::endl;
                boost::format formatter("protocol_module_url::handle_realserver_recv() : exception : %s. thread id : %d.");
                formatter % ex.c_str() % boost::this_thread::get_id();
                putLogError(100086, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (const std::bad_alloc &) {
                std::cerr << "protocol_module_url::handle_realserver_recv() : exception : Could not allocate memory." << std::endl;
                boost::format formatter("Could not allocate memory. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100087, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_realserver_recv() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_recv() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100088, formatter.str(), __FILE__, __LINE__);

                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_realserver_recv() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG "
                                        "protocol_module_url::handle_realserver_recv() : "
                                        "Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100089, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_recv(const boost::thread::id thread_id, "
                                        "const boost::asio::ip::tcp::endpoint& rs_endpoint, "
                                        "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                        "const size_t recvlen) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100177, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}



//! called from after sorryserver receive
//! @param[in]    downstream thread id
//! @param[in]    sorryserver endpoint reference
//! @param[in]    receive from realserver buffer reference.
//! @param[in]    recv data length
//! @return     session use EVENT mode
protocol_module_base::EVENT_TAG protocol_module_url::handle_sorryserver_recv(
        const boost::thread::id thread_id, const boost::asio::ip::tcp::endpoint &sorry_endpoint, const boost::array <
        char, MAX_BUFFER_SIZE > & recvbuffer, const size_t recvlen)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                size_t buffer_size = recvbuffer.size() < recvlen ? recvbuffer.size() : recvlen;
                std::string buffer;
                dump_memory(recvbuffer.data(), buffer_size, buffer);
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_recv(const boost::thread::id thread_id, "
                                        "const boost::asio::ip::tcp::endpoint& sorry_endpoint, "
                                        "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                        "const size_t recvlen) : thread_id = %d, sorry_endpoint = [%s]:%d, recvbuffer = %s, recvlen = %d.");
                formatter % thread_id % sorry_endpoint.address().to_string() % sorry_endpoint.port()
                % buffer % recvlen;
                putLogDebug(100178, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        EVENT_TAG status = FINALIZE;
        size_t data_remain_start = 0;
        size_t data_remain_size = 0;
        size_t request_data_remain_size = 0;
        size_t header_offset = 0;
        size_t header_offset_len = 0;
        size_t content_length_header_offset = 0;
        size_t content_length_header_len = 0;
        size_t content_len_value = 0;
        size_t pos = 0;
        size_t buffer_size = 0;
        const size_t cr_lf_cr_lf_len = strlen("\r\n\r\n");
        const size_t cr_lf_len = strlen("\r\n");
        std::string str_value;
        const std::string http_header = "";
        const std::string content_header = "Content-Length";
        thread_data_ptr session_data;
        char *buffer1 = NULL;
        char *buffer2 = NULL;
        bool bret = false;
        CHECK_RESULT_TAG check_result;
        session_thread_data_map_it session_thread_it;
        receive_data_map_it receive_data_it;

        //parameter check
        if (recvlen > recvbuffer.size()) {
                std::cerr << "protocol_module_url::handle_sorryserver_recv() : Data size bigger than buffer size." << std::endl;
                boost::format formatter("Data size bigger than buffer size. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100090, formatter.str(), __FILE__,
                            __LINE__);
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_recv(const boost::thread::id thread_id, "
                                                "const boost::asio::ip::tcp::endpoint& sorry_endpoint, "
                                                "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                                "const size_t recvlen) : return_value = %d. thread id : %d.");
                        formatter % FINALIZE % boost::this_thread::get_id();
                        putLogDebug(100179, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                return FINALIZE;
        }

        try {
                {
                        boost::mutex::scoped_lock slock(session_thread_data_map_mutex);

                        session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100091, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        session_data = session_thread_it->second;
                }

                receive_data_it = session_data->receive_data_map.find(sorry_endpoint);
                if (unlikely(receive_data_it == session_data->receive_data_map.end())) {
                        receive_data recv_data;
                        session_data->receive_data_map[sorry_endpoint] = recv_data;
                }

                session_data->target_endpoint = sorry_endpoint;

                receive_data &recv_data = session_data->receive_data_map[sorry_endpoint];

                //status list check
                send_status_it it = recv_data.send_status_list.begin();
                send_status_it it_end = recv_data.send_status_list.end();
                it = std::find_if(it, it_end, data_send_ok());
                if (unlikely(it != it_end)) {
                        boost::format formatter("Sending data is invalid. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100092, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                //status list check
                it = recv_data.send_status_list.begin();
                it = std::adjacent_find(it, it_end, data_send_repeated());
                if (unlikely(it != it_end)) {
                        boost::format formatter("Sending data is invalid. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100093, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        std::string datadump;
                        boost::format formatter("\nitem[%d] : status = %d, send_end_size = %d, "
                                                "send_rest_size = %d, send_possible_size = %d, "
                                                "send_offset = %d, unsend_size = %d, edit_division = %d.");
                        int i = 0;
                        for (it = recv_data.send_status_list.begin();
                             it != recv_data.send_status_list.end();
                             ++it, ++i) {
                                formatter % i % it->status % it->send_end_size
                                % it->send_rest_size % it->send_possible_size
                                % it->send_offset % it->unsend_size % it->edit_division;
                                datadump += formatter.str();
                        }

                        formatter.parse("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_recv() : send status list dump : send status list size = %d.%s");

                        formatter % recv_data.send_status_list.size() % datadump;
                        putLogDebug(100180, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/

                it = recv_data.send_status_list.begin();
                //get original status info
                while (it != it_end) {
                        //item status is SEND_END
                        if (it->status == SEND_END) {
                                //erase from list
                                recv_data.send_status_list.erase(it++);
                                continue;
                        }
                        //item status is SEND_CONTINUE
                        else if (it->status == SEND_CONTINUE) {
                                it->send_offset += it->send_end_size;
                                data_remain_start = it->send_offset;
                                break;
                        }
                        //item status is SEND_NG
                        else {
                                data_remain_start = it->send_offset;
                                data_remain_size = it->unsend_size;
                                break;
                        }

                        ++it;
                }
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        std::string datadump;
                        boost::format formatter("\nitem[%d] : status = %d, send_end_size = %d, "
                                                "send_rest_size = %d, send_possible_size = %d, "
                                                "send_offset = %d, unsend_size = %d, edit_division = %d.");
                        int i = 0;
                        for (it = recv_data.send_status_list.begin();
                             it != recv_data.send_status_list.end();
                             ++it, ++i) {
                                formatter % i % it->status % it->send_end_size
                                % it->send_rest_size % it->send_possible_size
                                % it->send_offset % it->unsend_size % it->edit_division;
                                datadump += formatter.str();
                        }

                        formatter.parse("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_recv() : send status list dump : send status list size = %d.%s");

                        formatter % recv_data.send_status_list.size() % datadump;
                        putLogDebug(100181, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                //receive buffer process
                //buffer rest size < request size
                if (recv_data.receive_buffer_rest_size < recvlen) {
                        //buffer max size < remain size + request size
                        //buffer is need reallocate
                        if (recv_data.receive_buffer_max_size < data_remain_size + recvlen) {
                                //the buffer's size that will be allocated is exceed the upper limit value
                                if (MAX_URL_MODULE_BUFFER_SIZE < data_remain_size + recvlen) {
                                        std::cerr << "protocol_module_url::handle_sorryserver_recv() : the buffer's size that will be allocated is exceed the upper limit value." << std::endl;
                                        boost::format formatter("The buffer's size that will be allocated is exceed the upper limit value. thread id : %d.");
                                        formatter % boost::this_thread::get_id();
                                        putLogError(100094, formatter.str(), __FILE__, __LINE__);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_recv(const boost::thread::id thread_id, "
                                                                        "const boost::asio::ip::tcp::endpoint& sorry_endpoint, "
                                                                        "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                                                        "const size_t recvlen) : return_value = %d. thread id : %d.");
                                                formatter % FINALIZE % boost::this_thread::get_id();
                                                putLogDebug(100182, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        return FINALIZE;
                                }
                                //receive_buffer1's memory allocate and initialization
                                buffer_size = (data_remain_size + recvlen) > MAX_BUFFER_SIZE ? (data_remain_size + recvlen) : MAX_BUFFER_SIZE;
                                buffer1 = new char[buffer_size];
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        boost::format formatter("new : address = &(%d), size = %lu.");
                                        formatter % static_cast<void *>(buffer1) % buffer_size;
                                        putLogDebug(100183, formatter.str(), __FILE__, __LINE__);
                                }
                                /*-------- DEBUG LOG END--------*/
                                memset(buffer1, 0, buffer_size);
                                //receive_buffer2's memory allocate and initialization
                                buffer2 = new char[buffer_size];
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        boost::format formatter("new : address = &(%d), size = %lu.");
                                        formatter % static_cast<void *>(buffer2) % buffer_size;
                                        putLogDebug(100184, formatter.str(), __FILE__, __LINE__);
                                }
                                /*-------- DEBUG LOG END--------*/
                                memset(buffer2, 0, buffer_size);

                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(recv_data.receive_buffer + data_remain_start, data_remain_size, datadump);
                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_recv() : before memcpy (data dump) : "
                                                "data begin = %d, data_size = %d, data = %s");
                                        formatter % data_remain_start % data_remain_size % datadump;
                                        putLogDebug(100185, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //copy data from old buffer to new buffer
                                memcpy(buffer1, recv_data.receive_buffer + data_remain_start, data_remain_size);
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(buffer1, data_remain_size, datadump);
                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_recv() : after memcpy (data dump) : "
                                                "data begin = 0, data_size = %d, data = %s");
                                        formatter % data_remain_size % datadump;
                                        putLogDebug(100186, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/

                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(recvbuffer.data(), recvlen, datadump);
                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_recv() : before memcpy (data dump) : "
                                                "data begin = 0, data_size = %d, data = %s");
                                        formatter % recvlen % datadump;
                                        putLogDebug(100187, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                memcpy(buffer1 + data_remain_size, recvbuffer.data(), recvlen);
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        std::string datadump;
                                        dump_memory(buffer1 + data_remain_size, recvlen, datadump);
                                        boost::format formatter(
                                                "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_recv() : after memcpy (data dump) : "
                                                "data begin = %d, data_size = %d, data = %s");
                                        formatter % data_remain_size % recvlen % datadump;
                                        putLogDebug(100188, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //free old buffer1 and old buffer2
                                if (recv_data.receive_buffer1 != NULL) {
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("delete : address = &(%d).");
                                                formatter % static_cast<void *>(recv_data.receive_buffer1);
                                                putLogDebug(100189, formatter.str(), __FILE__,
                                                            __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        delete[] recv_data.receive_buffer1;
                                        recv_data.receive_buffer1 = NULL;
                                }

                                if (recv_data.receive_buffer2 != NULL) {
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("delete : address = &(%d).");
                                                formatter % static_cast<void *>(recv_data.receive_buffer2);
                                                putLogDebug(100190, formatter.str(), __FILE__,
                                                            __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        delete[] recv_data.receive_buffer2;
                                        recv_data.receive_buffer2 = NULL;
                                }

                                //set new buffer pointer
                                recv_data.receive_buffer1 = buffer1;
                                recv_data.receive_buffer2 = buffer2;
                                recv_data.receive_buffer = recv_data.receive_buffer1;
                                //set new buffer's max size
                                recv_data.receive_buffer_max_size = buffer_size;
                        }
                        //buffer's max size >= remain data size + request size
                        //buffer isn't need reallocate, but switch
                        else {
                                //pointer valid check
                                if (unlikely(recv_data.receive_buffer1 == NULL || recv_data.receive_buffer2 == NULL)) {
                                        boost::format formatter("Invalid pointer. thread id : %d.");
                                        formatter % boost::this_thread::get_id();
                                        putLogError(100095, formatter.str(), __FILE__, __LINE__);
                                        throw - 1;
                                }
                                //using buffer is buffer1
                                if (recv_data.receive_buffer == recv_data.receive_buffer1) {
                                        //buffer2 initialization
                                        memset(recv_data.receive_buffer2, 0, recv_data.receive_buffer_max_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer + data_remain_start, data_remain_size, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_sorryserver_recv() : before memcpy (data dump) : "
                                                        "data begin = %d, data_size = %d, data = %s");
                                                formatter % data_remain_start % data_remain_size % datadump;
                                                putLogDebug(100191, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //copy data from buffer1 to buffer2
                                        memcpy(recv_data.receive_buffer2, recv_data.receive_buffer + data_remain_start, data_remain_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer2, data_remain_size, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_sorryserver_recv() : after memcpy (data dump) : "
                                                        "data begin = 0, data_size = %d, data = %s");
                                                formatter % data_remain_size % datadump;
                                                putLogDebug(100192, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recvbuffer.data(), recvlen, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_sorryserver_recv() : before memcpy (data dump) : "
                                                        "data begin = 0, data_size = %d, data = %s");
                                                formatter % recvlen % datadump;
                                                putLogDebug(100193, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        memcpy(recv_data.receive_buffer2 + data_remain_size, recvbuffer.data(), recvlen);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer2 + data_remain_size, recvlen, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_sorryserver_recv() : after memcpy (data dump) : "
                                                        "data begin = %d, data_size = %d, data = %s");
                                                formatter % data_remain_size % recvlen % datadump;
                                                putLogDebug(100194, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //set buffer2 as using buffer
                                        recv_data.receive_buffer = recv_data.receive_buffer2;
                                }
                                //using buffer is buffer2
                                else {
                                        //buffer1 initialization
                                        memset(recv_data.receive_buffer1, 0, recv_data.receive_buffer_max_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer + data_remain_start, data_remain_size, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_sorryserver_recv() : before memcpy (data dump) : "
                                                        "data begin = %d, data_size = %d, data = %s");
                                                formatter % data_remain_start % data_remain_size % datadump;
                                                putLogDebug(100195, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //copy data from buffer2 to buffer1
                                        memcpy(recv_data.receive_buffer1, recv_data.receive_buffer + data_remain_start, data_remain_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer1, data_remain_size, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_sorryserver_recv() : after memcpy (data dump) : "
                                                        "data begin = 0, data_size = %d, data = %s");
                                                formatter % data_remain_size % datadump;
                                                putLogDebug(100196, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/

                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recvbuffer.data(), recvlen, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_sorryserver_recv() : before memcpy (data dump) : "
                                                        "data begin = 0, data_size = %d, data = %s");
                                                formatter % recvlen % datadump;
                                                putLogDebug(100197, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        memcpy(recv_data.receive_buffer1 + data_remain_size, recvbuffer.data(), recvlen);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                std::string datadump;
                                                dump_memory(recv_data.receive_buffer1 + data_remain_size, recvlen, datadump);
                                                boost::format formatter(
                                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_sorryserver_recv() : after memcpy (data dump) : "
                                                        "data begin = %d, data_size = %d, data = %s");
                                                formatter % data_remain_size % recvlen % datadump;
                                                putLogDebug(100198, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //set buffer1 as using buffer
                                        recv_data.receive_buffer = recv_data.receive_buffer1;
                                }
                        }

                        //set buffer's rest size
                        recv_data.receive_buffer_rest_size = recv_data.receive_buffer_max_size - data_remain_size - recvlen;

                        //remain_size recalc
                        data_remain_size += recvlen;

                        send_status_it it_begin = recv_data.send_status_list.begin();
                        send_status_it it_end = recv_data.send_status_list.end();
                        //offset recalc
                        for (; it_begin != it_end; ++it_begin) {
                                it_begin->send_offset -= data_remain_start;
                        }
                }
                //buffer's rest size >= request size
                //copy directly
                else {
                        //pointer valid check
                        if (unlikely(recv_data.receive_buffer == NULL)) {
                                boost::format formatter("Invalid pointer. thread id : %d");
                                formatter % boost::this_thread::get_id();
                                putLogError(100096, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                std::string datadump;
                                dump_memory(recvbuffer.data(), recvlen, datadump);
                                boost::format formatter(
                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_recv() : before memcpy (data dump) : "
                                        "data begin = 0, data_size = %d, data = %s");
                                formatter % recvlen % datadump;
                                putLogDebug(100199, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/

                        //copy data from parameter to using buffer
                        memcpy(recv_data.receive_buffer + recv_data.receive_buffer_max_size - recv_data.receive_buffer_rest_size,
                               recvbuffer.data(), recvlen);
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                std::string datadump;
                                dump_memory(recv_data.receive_buffer + recv_data.receive_buffer_max_size - recv_data.receive_buffer_rest_size, recvlen, datadump);
                                boost::format formatter(
                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_recv() : after memcpy (data dump) : "
                                        "data begin = %d, data_size = %d, data = %s");
                                formatter % (recv_data.receive_buffer_max_size - recv_data.receive_buffer_rest_size) % recvlen % datadump;
                                putLogDebug(100200, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        //buffer's rest size recalc
                        recv_data.receive_buffer_rest_size -= recvlen;
                        //remain data size recalc
                        data_remain_size += recvlen;
                }

                it = recv_data.send_status_list.begin();
                it_end = recv_data.send_status_list.end();
                //request rest size initialization
                request_data_remain_size = recvlen;
                //original status process
                for (; it != it_end; ++it) {
                        //status is SEND_CONTINUE
                        if (it->status == SEND_CONTINUE) {
                                //
                                if (it->send_rest_size > request_data_remain_size) {
                                        it->send_possible_size = request_data_remain_size;
                                        it->send_rest_size -= request_data_remain_size;
                                        it->send_end_size = 0;
                                        request_data_remain_size = 0;
                                } else {
                                        it->send_possible_size = it->send_rest_size;
                                        request_data_remain_size -= it->send_rest_size;
                                        it->send_end_size = 0;
                                        it->send_rest_size = 0;
                                }

                                //change status from SEND_CONTINUE to SEND_OK
                                it->status = SEND_OK;
                        }
                        //status is SEND_NG
                        else if (it->status == SEND_NG) {
                                if (forwarded_for == FORWARDED_FOR_ON) {
                                        //check http method
                                        check_result = check_status_code(recv_data.receive_buffer + it->send_offset, data_remain_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_recv() : call check_status_code : "
                                                                        "return_value = %d. thread id : %d.");
                                                formatter % check_result % boost::this_thread::get_id();
                                                putLogDebug(100201, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //check http method result is CHECK_OK
                                        if (check_result == CHECK_OK) {
                                                //check http version
                                                check_result = check_http_version(recv_data.receive_buffer + it->send_offset, data_remain_size);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_sorryserver_recv() : call check_http_version : "
                                                                                "return_value = %d. thread id : %d.");
                                                        formatter % check_result % boost::this_thread::get_id();
                                                        putLogDebug(100202, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                        }
                                        //check method and version result is CHECK_OK
                                        if (check_result == CHECK_OK) {
                                                //search http header
                                                bret = find_http_header(recv_data.receive_buffer + it->send_offset, data_remain_size, http_header,
                                                                        header_offset, header_offset_len);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_sorryserver_recv() : call find_http_header : "
                                                                                "return_value = %d. thread id : %d.");
                                                        formatter % static_cast<int>(bret) % boost::this_thread::get_id();
                                                        putLogDebug(100203, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                //search http header result is OK
                                                if (bret) {
                                                        //search Content_Length header
                                                        bret = find_http_header(recv_data.receive_buffer + it->send_offset, data_remain_size,
                                                                                content_header, content_length_header_offset, content_length_header_len);
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_sorryserver_recv() : call find_http_header : "
                                                                                        "return_value = %d. thread id : %d.");
                                                                formatter % static_cast<int>(bret) % boost::this_thread::get_id();
                                                                putLogDebug(100204, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        //search Content_Length result is OK
                                                        if (bret) {
                                                                //Get Content_Length header's numeric value
                                                                for (pos = 0; recv_data.receive_buffer[it->send_offset + content_length_header_offset + pos] != ':' && pos
                                                                     < content_length_header_len; ++pos)
                                                                        ;
                                                                if (pos == content_length_header_len) {
                                                                        throw std::string("Content_Length field's value is invalid.");
                                                                }

                                                                ++pos;

                                                                str_value.assign(recv_data.receive_buffer + it->send_offset + content_length_header_offset + pos,
                                                                                 content_length_header_len - pos);

                                                                size_t pos_end = str_value.find_last_of('\r');
                                                                if (pos_end != std::string::npos) {
                                                                        str_value = str_value.erase(pos_end);
                                                                }

                                                                for (pos = 0; !isgraph(str_value[pos]) && str_value[pos] != '\0'; ++pos)
                                                                        ;

                                                                str_value = str_value.substr(pos);

                                                                try {
                                                                        content_len_value = boost::lexical_cast<size_t>(str_value.c_str());
                                                                } catch (const boost::bad_lexical_cast &ex) {
                                                                        throw std::string("Content_Length field's value is invalid.");
                                                                }
                                                                //send_rest_size recalc
                                                                //set whole http header's length + Content_Length's value
                                                                it->send_rest_size = header_offset + header_offset_len + cr_lf_cr_lf_len + content_len_value;
                                                        }
                                                        //search Content_Length result is NG
                                                        else {
                                                                //send_rest_size recalc
                                                                //set whole http header's length
                                                                if (header_offset_len == 0) {
                                                                        it->send_rest_size = header_offset + header_offset_len + cr_lf_len;
                                                                } else {
                                                                        it->send_rest_size = header_offset + header_offset_len + cr_lf_cr_lf_len;
                                                                }
                                                        }
                                                }
                                                //search http header result is CHECK_NG
                                                else {
                                                        it->unsend_size += request_data_remain_size;
                                                        request_data_remain_size = 0;
                                                        break;
                                                }
                                        }
                                        //check method or version result is CHECK_NG
                                        else if (check_result == CHECK_NG) {
                                                it->send_rest_size = it->unsend_size + request_data_remain_size;
                                        }
                                        //check method and version result is CHECK_IMPOSSIBLE
                                        else {
                                                it->unsend_size += request_data_remain_size;
                                                request_data_remain_size = 0;
                                                break;
                                        }
                                } else {
                                        it->send_rest_size = it->unsend_size + request_data_remain_size;
                                }

                                //recalc fields value according to send_rest_size and request rest size
                                if (it->send_rest_size > it->unsend_size + request_data_remain_size) {
                                        it->send_possible_size = it->unsend_size + request_data_remain_size;
                                        it->send_rest_size -= (it->unsend_size + request_data_remain_size);
                                        it->send_end_size = 0;
                                        it->unsend_size = 0;
                                        request_data_remain_size = 0;
                                } else {
                                        it->send_possible_size = it->send_rest_size;
                                        request_data_remain_size = it->unsend_size + request_data_remain_size - it->send_rest_size;
                                        it->send_end_size = 0;
                                        it->unsend_size = 0;
                                        it->send_rest_size = 0;
                                }

                                //change status from SEND_NG to SEND_OK
                                it->status = SEND_OK;
                        }
                        //no request rest data to process
                        if (request_data_remain_size <= 0) {
                                break;
                        }
                }
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        std::string datadump;
                        boost::format formatter("\nitem[%d] : status = %d, send_end_size = %d, "
                                                "send_rest_size = %d, send_possible_size = %d, "
                                                "send_offset = %d, unsend_size = %d, edit_division = %d.");
                        int i = 0;
                        for (it = recv_data.send_status_list.begin();
                             it != recv_data.send_status_list.end();
                             ++it, ++i) {
                                formatter % i % it->status % it->send_end_size
                                % it->send_rest_size % it->send_possible_size
                                % it->send_offset % it->unsend_size % it->edit_division;
                                datadump += formatter.str();
                        }

                        formatter.parse("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_recv() : send status list dump : send status list size = %d.%s");

                        formatter % recv_data.send_status_list.size() % datadump;
                        putLogDebug(100205, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/

                //there are still rest data need to process
                //new status created and add to status list
                while (request_data_remain_size > 0) {
                        //new status created
                        send_status new_send_state;
                        new_send_state.edit_division = EDIT_DIVISION_NO_EDIT;
                        new_send_state.send_end_size = 0;
                        new_send_state.send_offset = 0;
                        new_send_state.send_possible_size = 0;
                        new_send_state.unsend_size = 0;
                        new_send_state.send_rest_size = 0;
                        //status initialize to SEND_NG
                        new_send_state.status = SEND_NG;
                        //add new status to status_list
                        recv_data.send_status_list.push_back(new_send_state);
                        std::list<send_status>::reverse_iterator new_send_it = recv_data.send_status_list.rbegin();
                        //calc offset
                        new_send_it->send_offset = recv_data.receive_buffer_max_size - recv_data.receive_buffer_rest_size
                                                   - request_data_remain_size;

                        if (forwarded_for == FORWARDED_FOR_ON) {
                                //check http method
                                check_result = check_status_code(recv_data.receive_buffer + new_send_it->send_offset,
                                                                 request_data_remain_size);
                                /*-------- DEBUG LOG --------*/
                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                "handle_sorryserver_recv() : call check_status_code : "
                                                                "return_value = %d. thread id : %d.");
                                        formatter % check_result % boost::this_thread::get_id();
                                        putLogDebug(100206, formatter.str(), __FILE__, __LINE__);
                                }
                                /*------DEBUG LOG END------*/
                                //check http method result is CHECK_OK
                                if (check_result == CHECK_OK) {
                                        //check http version
                                        check_result = check_http_version(recv_data.receive_buffer + new_send_it->send_offset,
                                                                          request_data_remain_size);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_recv() : call check_http_version : "
                                                                        "return_value = %d. thread id : %d.");
                                                formatter % check_result % boost::this_thread::get_id();
                                                putLogDebug(100207, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                }
                                //check http method and version result is CHECK_OK
                                if (check_result == CHECK_OK) {
                                        //search whole http header, get whole http header's offset and length
                                        bret = find_http_header(recv_data.receive_buffer + new_send_it->send_offset, request_data_remain_size,
                                                                http_header, header_offset, header_offset_len);
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_recv() : call find_http_header : "
                                                                        "return_value = %d. thread id : %d.");
                                                formatter % static_cast<int>(bret) % boost::this_thread::get_id();
                                                putLogDebug(100208, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //searched whole http header
                                        if (bret) {
                                                //search ContentLength http header, get ContentLength header's offset and length
                                                bret = find_http_header(recv_data.receive_buffer + new_send_it->send_offset,
                                                                        request_data_remain_size, content_header, content_length_header_offset, content_length_header_len);
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_sorryserver_recv() : call find_http_header : "
                                                                                "return_value = %d. thread id : %d.");
                                                        formatter % static_cast<int>(bret) % boost::this_thread::get_id();
                                                        putLogDebug(100209, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/

                                                //searched ContentLength http header
                                                if (bret) {
                                                        //Get Content_Length header's numeric value
                                                        for (pos = 0; recv_data.receive_buffer[new_send_it->send_offset + content_length_header_offset + pos] != ':'
                                                             && pos < content_length_header_len; ++pos)
                                                                ;
                                                        if (pos == content_length_header_len) {
                                                                throw std::string("Content_Length field's value is invalid.");
                                                        }
                                                        ++pos;

                                                        str_value.assign(recv_data.receive_buffer + new_send_it->send_offset + content_length_header_offset + pos,
                                                                         content_length_header_len - pos);

                                                        size_t pos_end = str_value.find_last_of('\r');
                                                        if (pos_end != std::string::npos) {
                                                                str_value = str_value.erase(pos_end);
                                                        }

                                                        for (pos = 0; !isgraph(str_value[pos]) && str_value[pos] != '\0'; ++pos)
                                                                ;

                                                        str_value = str_value.substr(pos);
                                                        try {
                                                                content_len_value = boost::lexical_cast<size_t>(str_value.c_str());
                                                        } catch (const boost::bad_lexical_cast &ex) {
                                                                throw std::string("Content_Length field's value is invalid.");
                                                        }
                                                        //send_rest_size recalc
                                                        //set whole http header's  + whole http header's length + Content_Length's value
                                                        new_send_it->send_rest_size = header_offset + header_offset_len + cr_lf_cr_lf_len + content_len_value;
                                                }
                                                //not searched ContentLength http header
                                                else {
                                                        //send_rest_size recalc
                                                        //set whole http header's  + whole http header's length
                                                        if (header_offset_len == 0) {
                                                                new_send_it->send_rest_size = header_offset + header_offset_len + cr_lf_len;
                                                        } else {
                                                                new_send_it->send_rest_size = header_offset + header_offset_len + cr_lf_cr_lf_len;
                                                        }


                                                }
                                        }
                                        //not searched whole http header
                                        else {
                                                new_send_it->unsend_size = request_data_remain_size;
                                                request_data_remain_size = 0;
                                                break;
                                        }
                                }
                                //check http method or version result is CHECK_NG
                                else if (check_result == CHECK_NG) {
                                        new_send_it->send_rest_size = request_data_remain_size;
                                }
                                //check http method or version result is CHECK_IMPOSSIBLE
                                else {
                                        new_send_it->unsend_size = request_data_remain_size;
                                        request_data_remain_size = 0;
                                        break;
                                }
                        } else {
                                new_send_it->send_rest_size = request_data_remain_size;
                        }

                        //recalc fields value according to send_rest_size and request rest size
                        if (new_send_it->send_rest_size > request_data_remain_size) {
                                new_send_it->send_possible_size = request_data_remain_size;
                                new_send_it->send_rest_size -= request_data_remain_size;
                                new_send_it->send_end_size = 0;
                                request_data_remain_size = 0;
                        } else {
                                new_send_it->send_possible_size = new_send_it->send_rest_size;
                                request_data_remain_size -= new_send_it->send_rest_size;
                                new_send_it->send_end_size = 0;
                                new_send_it->send_rest_size = 0;
                        }

                        //change status from SEND_NG to SEND_OK
                        new_send_it->status = SEND_OK;
                }
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        std::string datadump;
                        boost::format formatter("\nitem[%d] : status = %d, send_end_size = %d, "
                                                "send_rest_size = %d, send_possible_size = %d, "
                                                "send_offset = %d, unsend_size = %d, edit_division = %d.");
                        int i = 0;
                        for (it = recv_data.send_status_list.begin();
                             it != recv_data.send_status_list.end();
                             ++it, ++i) {
                                formatter % i % it->status % it->send_end_size
                                % it->send_rest_size % it->send_possible_size
                                % it->send_offset % it->unsend_size % it->edit_division;
                                datadump += formatter.str();
                        }

                        formatter.parse("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_recv() : send status list dump : send status list size = %d.%s");

                        formatter % recv_data.send_status_list.size() % datadump;
                        putLogDebug(100210, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/

                //search for send_possible item in status list
                send_status_it it_find = find_if(recv_data.send_status_list.begin(), recv_data.send_status_list.end(),
                                                 data_send_possible());
                //the data that can be sent possible is exist
                if (it_find != recv_data.send_status_list.end()) {
                        status = CLIENT_CONNECTION_CHECK;
                }
                //the data that can be sent possible is not exist
                else {
                        status = SORRYSERVER_RECV;
                }
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_recv() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100211, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                status = FINALIZE;
        } catch (const std::string &ex) {
                std::cerr << "protocol_module_url::handle_sorryserver_recv() : exception : " << ex << std::endl;
                boost::format formatter("protocol_module_url::handle_sorryserver_recv() : exception : %s. thread id : %d.");
                formatter % ex.c_str() % boost::this_thread::get_id();
                putLogError(100097, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (const std::bad_alloc &) {
                std::cerr << "protocol_module_url::handle_sorryserver_recv() : exception : Could not allocate memory." << std::endl;
                boost::format formatter("Could not allocate memory. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100098, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_sorryserver_recv() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_recv() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100099, formatter.str(), __FILE__, __LINE__);

                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_sorryserver_recv() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG "
                                        "protocol_module_url::handle_sorryserver_recv() : "
                                        "Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100100, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_recv(const boost::thread::id thread_id, "
                                        "const boost::asio::ip::tcp::endpoint& sorry_endpoint, "
                                        "const boost::array<char,MAX_BUFFER_SIZE>& recvbuffer, "
                                        "const size_t recvlen) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100212, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! called from UPSTREAM thread. make module original message.
//! @param[in]    downstream thread id.
//! @return     session use EVENT mode
protocol_module_base::EVENT_TAG protocol_module_url::handle_response_send_inform(
        const boost::thread::id thread_id)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in/out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_response_send_inform(const boost::thread::id thread_id) : "
                                        "return_value = %d. thread id : %d.");
                formatter % STOP % boost::this_thread::get_id();
                putLogDebug(100213, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        return STOP;
}

//! called from after client connection check. use TCP/IP only. create client send message.
//! @param[in]    downstream thread id
//! @param[out]    send buffer reference
//! @param[out]    send data length
//! @return     session use EVENT mode
protocol_module_base::EVENT_TAG protocol_module_url::handle_client_connection_check(
        const boost::thread::id thread_id, boost::array<char, MAX_BUFFER_SIZE>& sendbuffer, size_t &datalen)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_connection_check(const boost::thread::id thread_id, "
                                        "boost::array<char, MAX_BUFFER_SIZE>& sendbuffer, size_t& datalen) : "
                                        "thread_id = %d.");
                formatter % thread_id;
                putLogDebug(100214, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        EVENT_TAG status = FINALIZE;
        size_t send_buffer_size = sendbuffer.max_size();
        thread_data_ptr session_data;
        session_thread_data_map_it session_thread_it;
        receive_data_map_it receive_data_it;

        try {
                {
                        boost::mutex::scoped_lock sclock(session_thread_data_map_mutex);

                        session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100101, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        session_data = session_thread_it->second;
                }

                receive_data_it = session_data->receive_data_map.find(session_data->target_endpoint);
                if (unlikely(receive_data_it == session_data->receive_data_map.end())) {
                        boost::format formatter("Invalid endpoint. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100102, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                receive_data &recv_data = receive_data_it->second;

                //get the data that can be sent possible
                send_status_it it = find_if(recv_data.send_status_list.begin(), recv_data.send_status_list.end(),
                                            data_send_possible());
                if (unlikely(it == recv_data.send_status_list.end())) {
                        boost::format formatter("Sending possible data is not existed. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100103, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                //buffer size >= sending_possible size
                if (send_buffer_size > it->send_possible_size) {
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                std::string datadump;
                                dump_memory(recv_data.receive_buffer + it->send_offset, it->send_possible_size, datadump);

                                boost::format formatter(
                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_connection_check() : before memcpy (data dump) : "
                                        "data begin = %d, data_size = %d, data = %s");
                                formatter % it->send_offset % (it->send_possible_size) % datadump;
                                putLogDebug(100215, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        //copy data from receive_buffer to sendbuffer by sending_possible size
                        memcpy(sendbuffer.data(), recv_data.receive_buffer + it->send_offset, it->send_possible_size);
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                std::string datadump;
                                dump_memory(sendbuffer.data(), it->send_possible_size, datadump);

                                boost::format formatter(
                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_connection_check() : after memcpy (data dump) : "
                                        "data begin = 0, data_size = %d, data = %s");
                                formatter % (it->send_possible_size) % datadump;
                                putLogDebug(100216, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        //send_end_size recalc
                        it->send_end_size = it->send_possible_size;
                        //set copied data length
                        datalen = it->send_possible_size;
                        //sending_possible size recalc
                        it->send_possible_size = 0;
                }
                //buffer size < sending_possible size
                else {
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                std::string datadump;
                                dump_memory(recv_data.receive_buffer + it->send_offset, send_buffer_size, datadump);

                                boost::format formatter(
                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_connection_check() : before memcpy (data dump) : "
                                        "data begin = %d, data_size = %d, data = %s");
                                formatter % it->send_offset % send_buffer_size % datadump;
                                putLogDebug(100217, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        //copy data from receive_buffer to sendbuffer by buffer size
                        memcpy(sendbuffer.data(), recv_data.receive_buffer + it->send_offset, send_buffer_size);
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                std::string datadump;
                                dump_memory(sendbuffer.data(), send_buffer_size, datadump);

                                boost::format formatter(
                                        "function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_connection_check() : after memcpy (data dump) : "
                                        "data begin = 0, data_size = %d, data = %s");
                                formatter % send_buffer_size % datadump;
                                putLogDebug(100218, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        //send_end_size recalc
                        it->send_end_size = send_buffer_size;
                        //sending_possible size recalc
                        it->send_possible_size -= send_buffer_size;
                        //set copied data length
                        datalen = send_buffer_size;
                }

                status = CLIENT_SEND;
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_client_connection_check() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100219, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                status = FINALIZE;
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_client_connection_check() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_connection_check() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100104, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_client_connection_check() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_connection_check() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100105, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_connection_check(const boost::thread::id thread_id, "
                                        "boost::array<char, MAX_BUFFER_SIZE>& sendbuffer, size_t& datalen)"
                                        " : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100220, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;

}

//! called from after client select. use UDP only
//! @param[in]    downstream thread id
//!    @param[in]    client udp endpoint
//! @param[out]    send buffer reference
//! @param[out]    send data length
//! @return     session use EVENT mode
protocol_module_base::EVENT_TAG protocol_module_url::handle_client_select(
        const boost::thread::id thread_id, boost::asio::ip::udp::endpoint &cl_endpoint, boost::array < char,
        MAX_BUFFER_SIZE > & sendbuffer, size_t &datalen)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in/out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_select(const boost::thread::id thread_id, "
                                        "boost::asio::ip::udp::endpoint& cl_endpoint, "
                                        "boost::array<char,MAX_BUFFER_SIZE>& sendbuffer, "
                                        "size_t& datalen) : "
                                        "return_value = %d. thread id : %d.");
                formatter % STOP % boost::this_thread::get_id();
                putLogDebug(100221, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        return STOP;
}

//!    called from after client send
//!    @param[in]    downstream thread id
//! @return     session use EVENT mode
protocol_module_base::EVENT_TAG protocol_module_url::handle_client_send(
        const boost::thread::id thread_id)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_send(const boost::thread::id thread_id) : "
                                        "thread_id = %d.");
                formatter % thread_id;
                putLogDebug(100222, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        EVENT_TAG status = FINALIZE;
        thread_data_ptr session_data;
        session_thread_data_map_it session_thread_it;
        receive_data_map_it receive_data_it;

        try {
                {
                        boost::mutex::scoped_lock sclock(session_thread_data_map_mutex);
                        //thread_id check
                        session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100106, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }
                        session_data = session_thread_it->second;
                }
                //endpoint check
                receive_data_it = session_data->receive_data_map.find(session_data->target_endpoint);
                if (unlikely(receive_data_it == session_data->receive_data_map.end())) {
                        boost::format formatter("Invalid endpoint. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100107, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                receive_data &recv_data = receive_data_it->second;

                send_status_it it = recv_data.send_status_list.begin();
                send_status_it it_end = recv_data.send_status_list.end();

                //check status list
                it = std::adjacent_find(it, it_end, data_send_list_incorrect());
                if (unlikely(it != it_end)) {
                        boost::format formatter("Sending possible data is invalid. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100108, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }
                //status list check
                it = recv_data.send_status_list.begin();
                it = find_if(it, it_end, data_send_ok());
                if (unlikely(it == it_end)) {
                        boost::format formatter("Sending possible data is not existed. thread id : %d.");
                        formatter % boost::this_thread::get_id();
                        putLogError(100109, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                //sending possible data is exist
                if (it->send_possible_size > 0) {
                        //status remain SEND_OK
                        it->status = SEND_OK;
                        //offset recalc
                        it->send_offset += it->send_end_size;
                        //send_end_size recalc
                        it->send_end_size = 0;
                }
                //sending possible data is not exist
                else {
                        //can receive from client continue
                        if (it->send_rest_size > 0) {
                                //change status from SEND_OK to SEND_CONTINUE
                                it->status = SEND_CONTINUE;
                        }
                        //can not receive from client continue
                        else {
                                //change status from SEND_OK to SEND_END
                                it->status = SEND_END;
                        }
                }

                it = recv_data.send_status_list.begin();
                it = find_if(it, it_end, data_send_ok());
                //send_ok item is exist
                if (it != it_end) {
                        status = CLIENT_CONNECTION_CHECK;
                }
                //send_ok item is not exist
                else {
                        //end flag is on
                        if (session_data->end_flag == END_FLAG_ON) {
                                status = CLIENT_DISCONNECT;
                        }
                        //end flag is off
                        else {
                                //sorry flag is on
                                if (session_data->sorry_flag == SORRY_FLAG_ON) {
                                        status = SORRYSERVER_RECV;
                                }
                                //sorry flag is off
                                else {
                                        status = REALSERVER_RECV;
                                }
                        }
                }
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_client_send() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100223, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                status = FINALIZE;
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_client_send() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_send() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100110, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_client_send() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_send() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100111, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_send(const boost::thread::id thread_id) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100224, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! call from client disconnect event. use upstream thread and downstream thread.
//! @param[in]    upstream and downstream thread id( check! one thread one event! )
//! @return     session use EVENT mode
protocol_module_base::EVENT_TAG protocol_module_url::handle_client_disconnect(
        const boost::thread::id thread_id)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in/out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_client_disconnect(const boost::thread::id thread_id) : return_value = %d. thread id : %d.");
                formatter % FINALIZE % boost::this_thread::get_id();
                putLogDebug(100225, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        return FINALIZE;
}

//! call from sorry mode event. use upstream thread and downstream thread
//! @param[in]    upstream and downstream thread id( check! one thread one event and first time call pattern )
//! @return     session use EVENT mode
protocol_module_base::EVENT_TAG protocol_module_url::handle_sorry_enable(
        const boost::thread::id thread_id)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorry_enable(const boost::thread::id thread_id) : "
                                        "thread_id = %d.");
                formatter % thread_id;
                putLogDebug(100226, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        EVENT_TAG status = FINALIZE;
        boost::asio::ip::tcp::endpoint endpoint;
        bool send_possible = false;
        bool send_continue = false;
        bool send_disable = false;
        thread_data_ptr session_data;
        session_thread_data_map_it session_thread_it;
        receive_data_map_it receive_data_it;

        try {
                {
                        boost::mutex::scoped_lock slock(session_thread_data_map_mutex);
                        //check thread_id
                        session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100112, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        session_data = session_thread_it->second;
                }
                //check endpoint
                endpoint = session_data->thread_division == THREAD_DIVISION_UP_STREAM ? session_data->client_endpoint_tcp
                           : session_data->target_endpoint;
                receive_data_it = session_data->receive_data_map.find(endpoint);
                if (unlikely(receive_data_it == session_data->receive_data_map.end())) {
                        //must be down thread
                        if (unlikely(session_data->thread_division == THREAD_DIVISION_UP_STREAM)) {
                                boost::format formatter("Invalid endpoint. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100113, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }
                        session_data->sorry_flag = SORRY_FLAG_ON;
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_sorry_enable() : SORRY_FLAG_ON. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogDebug(100227, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        status = SORRYSERVER_RECV;
                } else {
                        receive_data &recv_data = receive_data_it->second;

                        //get this thread sending possible data
                        send_status_it it = find_if(recv_data.send_status_list.begin(), recv_data.send_status_list.end(),
                                                    data_send_possible());
                        if (it != recv_data.send_status_list.end()) {
                                send_possible = true;
                        }

                        it = find_if(recv_data.send_status_list.begin(), recv_data.send_status_list.end(), data_send_continue());
                        if (it != recv_data.send_status_list.end()) {
                                send_continue = true;
                        }

                        it = find_if(recv_data.send_status_list.begin(), recv_data.send_status_list.end(), data_send_disable());
                        if (it != recv_data.send_status_list.end()) {
                                send_disable = true;
                        }

                        //up thread
                        if (session_data->thread_division == THREAD_DIVISION_UP_STREAM) {
                                //accept_end_flag is off
                                if (session_data->accept_end_flag == ACCEPT_END_FLAG_OFF) {
                                        //set sorry flag on
                                        session_data->sorry_flag = SORRY_FLAG_ON;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorry_enable() : SORRY_FLAG_ON. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100228, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        status = ACCEPT;
                                }
                                //accept_end_flag is on
                                else {
                                        //set sorry flag on
                                        if (session_data->sorry_flag == SORRY_FLAG_ON) {
                                                if (send_possible) {
                                                        status = SORRYSERVER_CONNECT;
                                                } else {
                                                        status = SORRYSERVER_SELECT;
                                                }

                                        }
                                        //set sorry flag off
                                        else {
                                                //the data that can be sent continue is exist
                                                if (send_continue) {
                                                        //set end flag on
                                                        session_data->end_flag = END_FLAG_ON;
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_sorry_enable() : END_FLAG_ON. thread id : %d.");
                                                                formatter % boost::this_thread::get_id();
                                                                putLogDebug(100229, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        status = REALSERVER_DISCONNECT;
                                                }
                                                //the data that can be sent continue is not exist
                                                else {
                                                        //set sorryserver_switch_flag on
                                                        session_data->sorryserver_switch_flag = SORRYSERVER_SWITCH_FLAG_ON;
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_sorry_enable() : SORRYSERVER_SWITCH_FLAG_ON. thread id : %d.");
                                                                formatter % boost::this_thread::get_id();
                                                                putLogDebug(100230, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        //set sorry_flag on
                                                        session_data->sorry_flag = SORRY_FLAG_ON;
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_sorry_enable() : SORRY_FLAG_ON. thread id : %d.");
                                                                formatter % boost::this_thread::get_id();
                                                                putLogDebug(100231, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        status = REALSERVER_DISCONNECT;
                                                }
                                        }
                                }
                        }
                        //down thread
                        else {
                                //sorry_flag is on
                                if (session_data->sorry_flag == SORRY_FLAG_ON) {
                                        //sending possible data is exist
                                        if (send_possible) {
                                                status = CLIENT_CONNECTION_CHECK;
                                        }
                                        //sending possible data is not exist
                                        else {
                                                status = SORRYSERVER_RECV;
                                        }
                                }
                                //sorry_flag is off
                                else {
                                        //set sorry_flag on
                                        session_data->sorry_flag = SORRY_FLAG_ON;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorry_enable() : SORRY_FLAG_ON. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100232, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        session_data->sorryserver_switch_flag = SORRYSERVER_SWITCH_FLAG_ON;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorry_enable() : SORRYSERVER_SWITCH_FLAG_ON. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100233, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/

                                        //sending NG data is exist or send_rest_size > 0
                                        if (send_disable) {
                                                //set end flag on
                                                session_data->end_flag = END_FLAG_ON;
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_sorry_enable() : END_FLAG_ON. thread id : %d.");
                                                        formatter % boost::this_thread::get_id();
                                                        putLogDebug(100234, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                status = REALSERVER_DISCONNECT;
                                        }
                                        //
                                        else {
                                                //sending possible data is exist
                                                if (send_possible) {
                                                        status = CLIENT_CONNECTION_CHECK;
                                                }
                                                //sending possible data is not exist
                                                else {
                                                        status = SORRYSERVER_RECV;
                                                }
                                        }
                                }
                        }
                }
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorry_enable() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100235, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                status = FINALIZE;
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_sorry_enable() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorry_enable() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100114, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_sorry_enable() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorry_enable() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100115, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorry_enable(const boost::thread::id thread_id) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100236, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! call from sorry mode disable. use upstream thread and downstream thread.
//! @param[in]    upstream and downstream thread id( check! one thread one event )
//! @return     session use EVENT mode
protocol_module_base::EVENT_TAG protocol_module_url::handle_sorry_disable(
        const boost::thread::id thread_id)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorry_disable(const boost::thread::id thread_id) : "
                                        "thread_id = %d.");
                formatter % thread_id;
                putLogDebug(100237, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        EVENT_TAG status = FINALIZE;
        boost::asio::ip::tcp::endpoint endpoint;
        bool send_possible = false;
        bool send_disable = false;
        bool send_continue = false;
        thread_data_ptr session_data;

        try {
                {
                        boost::mutex::scoped_lock sclock(session_thread_data_map_mutex);
                        //check thread_id
                        session_thread_data_map_it session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100116, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }
                        //check pointer
                        session_data = session_thread_it->second;
                }
                //check endpoint
                endpoint = session_data->thread_division == THREAD_DIVISION_UP_STREAM ? session_data->client_endpoint_tcp
                           : session_data->target_endpoint;
                receive_data_map_it receive_data_it = session_data->receive_data_map.find(endpoint);
                if (unlikely(receive_data_it == session_data->receive_data_map.end())) {
                        //must be down thread
                        if (unlikely(session_data->thread_division == THREAD_DIVISION_UP_STREAM)) {
                                boost::format formatter("Invalid endpoint. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100117, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        session_data->sorry_flag = SORRY_FLAG_OFF;
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_sorry_disable() : SORRY_FLAG_OFF. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogDebug(100238, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/
                        status = REALSERVER_RECV;
                } else {
                        receive_data &recv_data = receive_data_it->second;

                        //get this thread sending possible data
                        send_status_it it = find_if(recv_data.send_status_list.begin(), recv_data.send_status_list.end(),
                                                    data_send_possible());
                        if (it != recv_data.send_status_list.end()) {
                                send_possible = true;
                        }

                        //sending NG data is exist or send_rest_size > 0
                        it = find_if(recv_data.send_status_list.begin(), recv_data.send_status_list.end(), data_send_disable());
                        if (it != recv_data.send_status_list.end()) {
                                send_disable = true;
                        }

                        //the data that can be sent continue is exist
                        it = find_if(recv_data.send_status_list.begin(), recv_data.send_status_list.end(), data_send_continue());
                        if (it != recv_data.send_status_list.end()) {
                                send_continue = true;
                        }

                        //up thread
                        if (session_data->thread_division == THREAD_DIVISION_UP_STREAM) {
                                //accept_end_flag is off
                                if (session_data->accept_end_flag == ACCEPT_END_FLAG_OFF) {
                                        //set sorry flag off
                                        session_data->sorry_flag = SORRY_FLAG_OFF;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorry_disable() : SORRY_FLAG_OFF. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100239, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/

                                        status = ACCEPT;
                                }
                                //accept_end_flag is on
                                else {
                                        //sorry flag is on
                                        if (session_data->sorry_flag == SORRY_FLAG_ON) {
                                                //the data that can be sent continue is exist
                                                if (send_continue) {
                                                        //set end flag on
                                                        session_data->end_flag = END_FLAG_ON;
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_sorry_disable() : END_FLAG_ON. thread id : %d.");
                                                                formatter % boost::this_thread::get_id();
                                                                putLogDebug(100240, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        status = SORRYSERVER_DISCONNECT;
                                                }
                                                //the data that can be sent continue is not exist
                                                else {
                                                        //set realserver_switch_flag on
                                                        session_data->realserver_switch_flag = REALSERVER_SWITCH_FLAG_ON;
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_sorry_disable() : REALSERVER_SWITCH_FLAG_ON. thread id : %d.");
                                                                formatter % boost::this_thread::get_id();
                                                                putLogDebug(100241, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        //set sorry_flag off
                                                        session_data->sorry_flag = SORRY_FLAG_OFF;
                                                        /*-------- DEBUG LOG --------*/
                                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                        "handle_sorry_disable() : SORRY_FLAG_OFF. thread id : %d.");
                                                                formatter % boost::this_thread::get_id();
                                                                putLogDebug(100242, formatter.str(), __FILE__, __LINE__);
                                                        }
                                                        /*------DEBUG LOG END------*/
                                                        status = SORRYSERVER_DISCONNECT;
                                                }
                                        }
                                        //sorry flag is off
                                        else {
                                                if (send_possible) {
                                                        status = REALSERVER_CONNECT;
                                                } else {
                                                        status = REALSERVER_SELECT;
                                                }
                                        }
                                }
                        }
                        //down thread
                        else {
                                //sorry_flag is on
                                if (session_data->sorry_flag == SORRY_FLAG_ON) {
                                        //set sorry_flag off
                                        session_data->sorry_flag = SORRY_FLAG_OFF;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorry_disable() : SORRY_FLAG_OFF. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100243, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        session_data->realserver_switch_flag = REALSERVER_SWITCH_FLAG_ON;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorry_disable() : REALSERVER_SWITCH_FLAG_ON. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100244, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        //sending NG data is exist or send_rest_size > 0
                                        if (send_disable) {
                                                //set end flag on
                                                session_data->end_flag = END_FLAG_ON;
                                                /*-------- DEBUG LOG --------*/
                                                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                                "handle_sorry_disable() : END_FLAG_ON. thread id : %d.");
                                                        formatter % boost::this_thread::get_id();
                                                        putLogDebug(100245, formatter.str(), __FILE__, __LINE__);
                                                }
                                                /*------DEBUG LOG END------*/
                                                status = SORRYSERVER_DISCONNECT;
                                        }
                                        //
                                        else {
                                                //sending possible data is exist
                                                if (send_possible) {
                                                        status = CLIENT_CONNECTION_CHECK;
                                                }
                                                //sending possible data is not exist
                                                else {
                                                        status = REALSERVER_RECV;
                                                }
                                        }
                                }
                                //sorry_flag is off
                                else {
                                        //sending possible data is exist
                                        if (send_possible) {
                                                status = CLIENT_CONNECTION_CHECK;
                                        }
                                        //sending possible data is not exist
                                        else {
                                                status = REALSERVER_RECV;
                                        }
                                }
                        }
                }


        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorry_disable() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100246, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                status = FINALIZE;
        } catch (std::exception &ex) {
                std::cerr << "protocol_module_url::handle_sorry_disable() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorry_disable() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100118, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_sorry_disable() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorry_disable() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100119, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorry_disable(const boost::thread::id thread_id) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100247, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! call from realserver disconnect. use upstream thread and downstream thread
//! @param[in]    upstream and downstream thread id( check! one thread one event )
//! @param[in]    disconnected realserver endpoint.
//! @return     session use EVENT mode
protocol_module_base::EVENT_TAG protocol_module_url::handle_realserver_disconnect(
        const boost::thread::id thread_id, const boost::asio::ip::tcp::endpoint &rs_endpoint)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_disconnect(const boost::thread::id thread_id, const boost::asio::ip::tcp::endpoint & rs_endpoint) : "
                                        "thread_id = %d, rs_endpoint = [%s]:%d.");
                formatter % thread_id % rs_endpoint.address().to_string() % rs_endpoint.port();
                putLogDebug(100248, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        EVENT_TAG status = FINALIZE;
        bool possible_flag = false;
        thread_data_ptr session_data;
        boost::asio::ip::tcp::endpoint endpoint;

        try {
                {
                        boost::mutex::scoped_lock sclock(session_thread_data_map_mutex);

                        session_thread_data_map_it session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100120, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        session_data = session_thread_it->second;
                }

                endpoint = session_data->thread_division == THREAD_DIVISION_UP_STREAM ? session_data->client_endpoint_tcp
                           : session_data->target_endpoint;
                receive_data_map_it receive_data_it = session_data->receive_data_map.find(endpoint);
                if (unlikely(receive_data_it == session_data->receive_data_map.end())) {
                        /*-------- DEBUG LOG --------*/
                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                        "handle_realserver_disconnect(const boost::thread::id thread_id, "
                                                        "const boost::asio::ip::tcp::endpoint & rs_endpoint) : return_value = %d. thread id : %d.");
                                formatter % FINALIZE % boost::this_thread::get_id();
                                putLogDebug(100263, formatter.str(), __FILE__, __LINE__);
                        }
                        /*------DEBUG LOG END------*/

                        return FINALIZE;
                }

                receive_data &recv_data = receive_data_it->second;

                //the data that can be sent possible is exist
                send_status_it it = find_if(recv_data.send_status_list.begin(), recv_data.send_status_list.end(),
                                            data_send_possible());
                if (it != recv_data.send_status_list.end()) {
                        possible_flag = true;
                }

                //up thread
                if (session_data->thread_division == THREAD_DIVISION_UP_STREAM) {
                        //end flag is on
                        if (session_data->end_flag == END_FLAG_ON) {
                                status = CLIENT_RECV;
                        }
                        //end flag is off
                        else {
                                //sorryserver_switch_flag is on
                                if (session_data->sorryserver_switch_flag == SORRYSERVER_SWITCH_FLAG_ON) {
                                        session_data->sorryserver_switch_flag = SORRYSERVER_SWITCH_FLAG_OFF;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_disconnect() : SORRYSERVER_SWITCH_FLAG_OFF. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100249, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/

                                        status = SORRYSERVER_SELECT;

                                }
                                //sorryserver_switch_flag is off
                                else {
                                        //set end flag on
                                        session_data->end_flag = END_FLAG_ON;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_disconnect() : END_FLAG_ON. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100250, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        status = CLIENT_RECV;
                                }
                        }
                }
                //down thread
                else {
                        if (session_data->end_flag == END_FLAG_ON) {
                                status = CLIENT_DISCONNECT;
                        } else {
                                if (session_data->sorryserver_switch_flag == SORRYSERVER_SWITCH_FLAG_ON) {
                                        session_data->sorryserver_switch_flag = SORRYSERVER_SWITCH_FLAG_OFF;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_disconnect() : SORRYSERVER_SWITCH_FLAG_OFF. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100251, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        status = SORRYSERVER_RECV;
                                } else {
                                        //set end flag on
                                        session_data->end_flag = END_FLAG_ON;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_realserver_disconnect() : END_FLAG_ON. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100252, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        status = CLIENT_DISCONNECT;
                                }
                        }

                        //the data that can be sent possible is exist
                        if (possible_flag) {
                                status = CLIENT_CONNECTION_CHECK;
                        }
                }
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_realserver_disconnect() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100253, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                status = FINALIZE;
        } catch (std::exception &ex) {
                std::cerr << "protocol_module_url::handle_realserver_disconnect() : exception: error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_disconnect() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100122, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_realserver_disconnect() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_disconnect() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100123, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_disconnect(const boost::thread::id thread_id, "
                                        "const boost::asio::ip::tcp::endpoint & rs_endpoint) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100254, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! call from sorry server disconnect. use upstream thread and downstream thread
//! @param[in]    upstream and downstream thread id( check! one thread one event )
//! @param[in]    disconnect sorryserver endpoint
//! @return        session use EVENT mode
//! @return        session use EVENT mode
protocol_module_base::EVENT_TAG protocol_module_url::handle_sorryserver_disconnect(
        const boost::thread::id thread_id, const boost::asio::ip::tcp::endpoint &sorry_endpoint)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format
                formatter(
                        "in_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                        "handle_sorryserver_disconnect(const boost::thread::id thread_id, "
                        "const boost::asio::ip::tcp::endpoint & sorry_endpoint) : "
                        "thread_id = %d, sorry_endpoint = [%s]:%d.");
                formatter % thread_id % sorry_endpoint.address().to_string() % sorry_endpoint.port() ;
                putLogDebug(100255, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        EVENT_TAG status = FINALIZE;
        bool possible_flag = false;
        thread_data_ptr session_data;
        boost::asio::ip::tcp::endpoint endpoint;

        try {
                {
                        boost::mutex::scoped_lock sclock(session_thread_data_map_mutex);

                        session_thread_data_map_it session_thread_it = session_thread_data_map.find(thread_id);
                        if (unlikely(session_thread_it == session_thread_data_map.end() || session_thread_it->second == NULL)) {
                                boost::format formatter("Invalid thread id. thread id : %d.");
                                formatter % boost::this_thread::get_id();
                                putLogError(100124, formatter.str(), __FILE__, __LINE__);
                                throw - 1;
                        }

                        session_data = session_thread_it->second;
                }

                endpoint = session_data->thread_division == THREAD_DIVISION_UP_STREAM ? session_data->client_endpoint_tcp
                           : session_data->target_endpoint;
                receive_data_map_it receive_data_it = session_data->receive_data_map.find(endpoint);
                if (unlikely(receive_data_it == session_data->receive_data_map.end())) {
                        boost::format formatter("Invalid endpoint(%s). thread id: %d.");
                        formatter % endpoint % boost::this_thread::get_id();
                        putLogError(100125, formatter.str(), __FILE__, __LINE__);
                        throw - 1;
                }

                receive_data &recv_data = receive_data_it->second;

                //the data that can be sent possible is exist
                send_status_it it = find_if(recv_data.send_status_list.begin(), recv_data.send_status_list.end(),
                                            data_send_possible());
                if (it != recv_data.send_status_list.end()) {
                        possible_flag = true;
                }

                //up thread
                if (session_data->thread_division == THREAD_DIVISION_UP_STREAM) {
                        //end flag is on
                        if (session_data->end_flag == END_FLAG_ON) {
                                status = CLIENT_RECV;
                        }
                        //end flag is off
                        else {
                                //realserver_switch_flag is on
                                if (session_data->realserver_switch_flag == REALSERVER_SWITCH_FLAG_ON) {
                                        session_data->realserver_switch_flag = REALSERVER_SWITCH_FLAG_OFF;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_disconnect() : REALSERVER_SWITCH_FLAG_OFF. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100256, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        status = REALSERVER_SELECT;
                                }
                                //realserver_switch_flag is off
                                else {
                                        //set end flag on
                                        session_data->end_flag = END_FLAG_ON;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_disconnect() : END_FLAG_ON. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100257, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        status = CLIENT_RECV;
                                }
                        }
                }
                //down thread
                else {
                        if (session_data->end_flag == END_FLAG_ON) {
                                status = CLIENT_DISCONNECT;
                        } else {
                                if (session_data->realserver_switch_flag == REALSERVER_SWITCH_FLAG_ON) {
                                        session_data->realserver_switch_flag = REALSERVER_SWITCH_FLAG_OFF;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_disconnect() : REALSERVER_SWITCH_FLAG_OFF. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100258, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        status = REALSERVER_RECV;
                                } else {
                                        session_data->end_flag = END_FLAG_ON;
                                        /*-------- DEBUG LOG --------*/
                                        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                                                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                                        "handle_sorryserver_disconnect() : END_FLAG_ON. thread id : %d.");
                                                formatter % boost::this_thread::get_id();
                                                putLogDebug(100259, formatter.str(), __FILE__, __LINE__);
                                        }
                                        /*------DEBUG LOG END------*/
                                        status = CLIENT_DISCONNECT;
                                }
                        }

                        //the data that can be sent possible is exist
                        if (possible_flag) {
                                status = CLIENT_CONNECTION_CHECK;
                        }
                }
        } catch (int e) {
                /*-------- DEBUG LOG --------*/
                if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                        boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                                "handle_sorryserver_disconnect() : catch exception e = %d. thread id : %d.");
                        formatter % e % boost::this_thread::get_id();
                        putLogDebug(100260, formatter.str(), __FILE__, __LINE__);
                }
                /*------DEBUG LOG END------*/
                status = FINALIZE;
        } catch (const std::exception &ex) {
                std::cerr << "protocol_module_url::handle_sorryserver_disconnect() : exception : error = " << ex.what() << "." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_disconnect() : exception : error = %s. thread id : %d.");
                formatter % ex.what() % boost::this_thread::get_id();
                putLogError(100126, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        } catch (...) {
                std::cerr << "protocol_module_url::handle_sorryserver_disconnect() : Unknown exception." << std::endl;
                boost::format formatter("function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_sorryserver_disconnect() : Unknown exception. thread id : %d.");
                formatter % boost::this_thread::get_id();
                putLogError(100127, formatter.str(), __FILE__, __LINE__);
                status = FINALIZE;
        }

        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format
                formatter(
                        "out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                        "handle_sorryserver_disconnect(const boost::thread::id thread_id, "
                        "const boost::asio::ip::tcp::endpoint& sorry_endpoint) : return_value = %d. thread id : %d.");
                formatter % status % boost::this_thread::get_id();
                putLogDebug(100261, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/

        return status;
}

//! call from realserver disconnect. use upstream thread and downstream thread.
//! @param[in]    upstream and downstream thread id( check! one thread one event )
//! @param[in]    disconnect realserver endpoint
//! @return        session use EVENT mode.
protocol_module_base::EVENT_TAG protocol_module_url::handle_realserver_close(
        const boost::thread::id thread_id, const boost::asio::ip::udp::endpoint &rs_endpoint)
{
        /*-------- DEBUG LOG --------*/
        if (unlikely(LOG_LV_DEBUG == getloglevel())) {
                boost::format formatter("in/out_function : protocol_module_base::EVENT_TAG protocol_module_url::"
                                        "handle_realserver_close(const boost::thread::id thread_id, "
                                        "const boost::asio::ip::udp::endpoint & rs_endpoint) : "
                                        "return_value = %d. thread id : %d.");
                formatter % STOP % boost::this_thread::get_id();
                putLogDebug(100262, formatter.str(), __FILE__, __LINE__);
        }
        /*------DEBUG LOG END------*/
        return STOP;
}

// local function --------------------------------------------------------------

// check optstr. length & regex grammer.
// @param optstr[in] regex pattern
// @param check_result[out] 
bool protocol_module_url::checkOptionURIregex(
	 const std::string &optstr, check_message_result &check_result
 )
{
	using namespace boost::xpressive;
	bool result_flag = true;

	//next option string's length > 127
	if (optstr.size() > MAX_OPTION_SIZE - 1) {
			std::ostringstream ostr;
			ostr << "'-P/--pattern-match' option value '" << optstr << "' is too long.";

			//set check result flag false
			check_result.flag = false;
			result_flag = false;
			//set check result message
			check_result.message = ostr.str();
			putLogError(100009, check_result.message, __FILE__,
						__LINE__);
	}
	//next option string's length <= 127
	else {
		//regex check
		//check OK
		try{
			sregex::compile( optstr.c_str() );
		}catch( std::exception& e ){
				//check NG		
				std::ostringstream ostr;
				ostr << "'-P/--pattern-match' option value '" << optstr << "' is not a valid URI,So regex compile error.";
				//set check result flag false
				check_result.flag = false;
				result_flag = false;
				//set check result message
				check_result.message = ostr.str();
				putLogError(100010, check_result.message, __FILE__,
						__LINE__);
	
		}	
	}

	return result_flag;
}

// set Real Server endpoint.
// @param: vsit[out] '-RS/--end-point' option string list iterator
// @param: vsit_end[in]
// @param: ep_list[out] Real Server endpoint list
// @param: check_result[out]
template <class T>
bool protocol_module_url::setRealserverEndpoint(
	std::vector<std::string>::const_iterator &vsit,
	const std::vector<std::string>::const_iterator &vsit_end,
	std::list< typename T::endpoint > &ep_list,
	check_message_result &check_result
)
{

	typename T::endpoint temp_endpoint;

	// check option string ( set another option string after '-RS/--end-point' or not )
	if( vsit == vsit_end || vsit->substr( 0, 1 ) == "-" || vsit->substr( 0, 2 ) == "--" ){
		check_result.flag = false;
		std::ostringstream ostr;						
		ostr << "You have to set option value '-RS/--end-point'.";
		check_result.message = ostr.str();
		putLogError(100009, check_result.message, __FILE__, __LINE__);
		return false;
	}

	bool result_flag = true;
	typename std::list< typename T::endpoint >::iterator findpos_it;

	while( vsit != vsit_end ){

		if( vsit->substr( 0, 1 ) != "-" && vsit->substr( 0, 2 ) != "--" ){
			error_code err;
			temp_endpoint = stringtoEndpoint< T >( (*vsit), err );
			if( err ){
				check_result.flag = false;
				std::ostringstream ostr;						
				ostr << "target endpoint parse error(--tcp-service).";
				check_result.message = ostr.str();
				putLogError(100009, check_result.message, __FILE__, __LINE__);
				result_flag = false;
				break;
			}else{
				error_code ec;
				checkEndpoint< T >( temp_endpoint, true, ec );
				if ( ec ) {
					check_result.flag = false;
					std::ostringstream ostr;
					// TODO:This is not 'parse error'
					ostr << "target endpoint is incorrect(--tcp-service).";
					check_result.message = ostr.str();
					putLogError(100009, check_result.message, __FILE__, __LINE__);
					result_flag = false;
					break;
				}
			}
			findpos_it = std::find_if(
				ep_list.begin(), ep_list.end(),
				std::bind1st( std::equal_to<typename T::endpoint>(), temp_endpoint ) );

			if( findpos_it == ep_list.end() ){
				ep_list.push_back( temp_endpoint );
			}else{
				check_result.flag = false;
				std::ostringstream ostr;						
				ostr << "target endpoint '" << (*vsit) << "' is already set.";
				check_result.message = ostr.str();
				putLogError(100009, check_result.message, __FILE__, __LINE__);
				result_flag = false;
				break;
			}
			++vsit;

		}else{
			// iterator decriment because another option start
			--vsit;
			break;
		}

	}

	return result_flag;
}

//! copy "l7vsadm.h"'s function.( but this function use 'xpressive'. )
//! endpoint string parse function
//! @param[in]  endpoint string
//! @param[out] error_code
//! @return     endpoint
template <class T>
typename T::endpoint protocol_module_url::stringtoEndpoint(const std::string &str, error_code &err)
{
        using namespace boost::xpressive;

        std::string::size_type pos = str.rfind(":");
        std::string hostname = str.substr(0, pos);
        std::string portname = str.substr(pos + 1, str.length());
        if (0 == hostname.length()) {
                err.setter(1, "hostname is not specified:");
                return typename T::endpoint();        
	}
        
	sregex re = +_d;

        if ( regex_match( portname, re ) ) {
                try {
                        boost::lexical_cast<unsigned short>( portname.c_str() );
                } catch (boost::bad_lexical_cast &) {
                        err.setter(1, "invalid port number:");
                        return typename T::endpoint();
                }
        }
        //remove "[","]"
        boost::algorithm::erase_first(hostname, "[");
        boost::algorithm::erase_last(hostname, "]");

        boost::asio::io_service        io_service;
        typename T::resolver           resolver(io_service);
        typename T::resolver::query    query(hostname, portname);
        typename T::resolver::iterator end;
        boost::system::error_code      ec;
        typename T::resolver::iterator itr = resolver.resolve(query, ec);
        if (ec) {
                std::stringstream buf;
                buf << "invalid endpoint:" << ec.message() << ":";
                err.setter(1, buf.str());
                return typename T::endpoint();
        }
        if (itr == end) {
                return typename T::endpoint();
        }

        return *itr;
}


//! copy "l7vsadm.h"'s function.
//! check endpoint function
//! @param[in]  endpoint
//! @param[in]  allow any_address or not
//! @param[out] error_code
template <class T>
void protocol_module_url::checkEndpoint(typename T::endpoint &ep, bool allow_any_address, error_code &err)
{
        if (!allow_any_address) {
                if ((ep.address().is_v4() &&
                     ep.address().to_v4() == boost::asio::ip::address_v4::any()) ||
                    (ep.address().is_v6() &&
                     ep.address().to_v6() == boost::asio::ip::address_v6::any())) {
                        err.setter(1, "invalid address (INADDR_ANY):");
                        return;
                }
        }
        if (ep.port() == 0) {
                err.setter(1, "invalid port number (0):");
                return;
        }
}

// VSに登録されているRSリストのイテレータを操作する関数群
std::list< realserver >::iterator protocol_module_url::begin_rslist_it()
{
        return *( rslist_it_list.begin() );
}  

std::list< realserver >::iterator protocol_module_url::end_rslist_it()
{
        return *( rslist_it_list.rbegin() );
}  

std::list< realserver >::iterator protocol_module_url::next_rslist_it( std::list< realserver >::iterator in_itr )
{
	// @takeda:【注意】引数in_itrはschedule_tcpを呼び出すためのダミーである
	++rslist_it_list_it;
        return *rslist_it_list_it;
}  

//! @takeda:schedule_tcpを無理やり使うための関数...
//! @param[in] thread_id
//! @param[in] in_rslist_it_list : registered Real Servers ( This parameter is substantially 'const' in this function. )
//! @param[out] rs_endpoint[out] : addressee Real Server
void protocol_module_url::useScheduleTCP(  
		const boost::thread::id thread_id,
		std::list< std::list<realserver>::iterator >& in_rslist_it_list,
		boost::asio::ip::tcp::endpoint &rs_endpoint
	)
{
	protocol_module_url::rslist_it_list_it = in_rslist_it_list.begin();

	//! realserver list iterator begin function object
        protocol_module_base::rs_list_itr_func_type begin = boost::bind( &protocol_module_url::begin_rslist_it, this );

	//! realserver list iterator end function object
        protocol_module_base::rs_list_itr_func_type end = boost::bind( &protocol_module_url::end_rslist_it, this );

	//! realserver list iterator next function object
        protocol_module_base::rs_list_itr_next_func_type next = boost::bind( &protocol_module_url::next_rslist_it, this, _1 );

	schedule_tcp( thread_id, begin, end, next, rs_endpoint );
}

//! Debug function. 
void protocol_module_url::dumpKeyary_endpointlist( int num )
{

	pattern_endpointlist_map_it it = ptn_eplist_map.begin();
	pattern_endpointlist_map_it it_end = ptn_eplist_map.end();

	typedef std::list<boost::asio::ip::tcp::endpoint> eplist;
	
	boost::format formatter01("---------dumpKeyary_endpointlist_map-----------");
        	putLogFatal(9999, formatter01.str(), __FILE__, __LINE__);

	int n = 29292;
	for( ;it != it_end; ++it ){
		boost::format formatter("%s:");
		formatter % it->first.data(); 
		std::string str_endpoint;
		eplist &el = it->second;
		for( eplist::iterator elit = el.begin(); elit != el.end(); ++elit ){
			str_endpoint +=  ( "[" + elit->address().to_string() + ":" + boost::lexical_cast< std::string >( elit->port() ) + "]" );
		}
        	putLogFatal( n, formatter.str()  + str_endpoint, __FILE__, __LINE__);       
		n++;
		str_endpoint.clear();
		formatter.clear();
	}

	boost::format formatter02("----------------------------------------------");
        	putLogFatal(9999, formatter02.str(), __FILE__, __LINE__);

}

void protocol_module_url::dumpRslist_it_list( void )
{

        putLogFatal(9999, "-------------dumpRslist_it_list---------------", __FILE__, __LINE__);
	
	putLogFatal(9999, "rslist_it_list size : " + boost::lexical_cast< std::string >( rslist_it_list.size() ) + "( end elem is Sentinel )", __FILE__, __LINE__ );

	std::list< realserver >::iterator end = end_rslist_it();
	std::list< std::list< realserver>::iterator >::iterator it = rslist_it_list.begin();
	std::string str_endpoint;

	while( *it != end ){
		str_endpoint = ( "[" + (*((*it))).tcp_endpoint.address().to_string() + ":" + boost::lexical_cast< std::string >( (*((*it))).tcp_endpoint.port()  )  + "]" );
		putLogFatal( 29290, str_endpoint, __FILE__, __LINE__ );
		++it;		
	}

        putLogFatal(9999, "----------------------------------------------", __FILE__, __LINE__);
}

void protocol_module_url::dumpOptionString( const std::vector<std::string>& args )
{
	using namespace std;

	static bool checked = false;

	if( checked == true ){
		return ;
	}

	vector< string >::const_iterator it = args.begin();

	cout<< "--------------dumpOptionString----------------" <<endl;
	
	while( it != args.end() ){
		cout<< *it << ", ";
		++it;
	}
	cout<<endl;

	cout<< "----------------------------------------------" <<endl;

	checked = true;

}

void protocol_module_url::dumpPtnsrgx_converter_pairlist( void )
{
//	putLogFatal(9999, "-------dumpPtnsrgx_converter_pairlist---------", __FILE__, __LINE__);
//
//	std::string str;
//
//	std::map< boost::array<char,MAX_OPTION_SIZE>, boost::xpressive::sregex >::iterator it = keyary_sregex_converter_map.begin();
//	std::map< boost::array<char,MAX_OPTION_SIZE>, boost::xpressive::sregex >::iterator end = keyary_sregex_converter_map.end();
//
//	while( it != end ){
//		str = it->first.data();
//		putLogFatal( 29290, "[ " + str + " ]", __FILE__, __LINE__ );
//		++it;		
//	}
//
//        putLogFatal(9999, "----------------------------------------------", __FILE__, __LINE__);
}

void protocol_module_url::testVolume( void )
{
	using namespace std;
	using namespace boost::lambda;

	boost::lambda::placeholder1_type X;

	list< pair< int , string > > lp;

	pair<int, string> pis = pair<int, string>( 50, "ggg" );

	lp.push_back( pair<int, string>( 10, "aaa" ) );
	lp.push_back( pair<int, string>( 30, "ccc" ) );
	lp.push_back( pair<int, string>( 50, "eee" ) );
	lp.push_back( pair<int, string>( 40, "ddd" ) );
	lp.push_back( pair<int, string>( 10, "aaa" ) );

	list< pair< int , string > >::iterator lp_it;
	list< pair< int , string > >::iterator pos_it;

	lp_it = lp.begin();

	pos_it = std::find_if( lp.begin(), lp.end(), ( &X->*( &pair<int,string>::first ) == pis.first ) );

}


}

extern "C" l7vs::protocol_module_base*
create_module()
{
        return dynamic_cast<l7vs::protocol_module_base *>(new l7vs::protocol_module_url());
}

extern "C" void
destroy_module(l7vs::protocol_module_base *in)
{
        delete in;
}
