/*
 * @file  config.c
 * @brief the configulation interface component
 * @brief it receives configulation command from l7vsadm,
 * @brief execute coresponding action and return result
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <stdlib.h>
#include <glib.h>

/* Added headers for sending messages to l7vsagent */
#include <stdio.h>
#include <errno.h>
#include <sys/msg.h>
#include <string.h>
#include <netdb.h>

#include "l7vs_config.h"
#include "l7vs_iomuxlist.h"
#include "l7vs_service.h"
#include "l7vs_sched.h"



static int	l7vs_config_callback( struct l7vs_iomux* );
static void	l7vs_config_dispatch( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_list_vs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_list_rs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_add_vs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_del_vs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_edit_vs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_add_rs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_del_rs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_edit_rs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_flush_vs( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_get_replication_info( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_get_log_info( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_get_snmp_log_info( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_get_snmp_connect_status( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_replication( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_log( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_snmp( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_snmp_all( union l7vs_config_req*, struct sockaddr_un* );
static void	l7vs_config_parameter( union l7vs_config_req*, struct sockaddr_un* );
static int	l7vs_config_send_response( struct sockaddr_un*, struct iovec*, int );

char	servicebuf[NI_MAXHOST + NI_MAXSERV + 2];
const char l7vs_config_sockname[] = L7VS_CONFIG_SOCKNAME;

static struct l7vs_iomux *l7vs_config_iomux;

/*!
 * Operate config init
 * @return      int     result OK=0, NG=<0
 */

int
l7vs_config_init(void)
{
        int s;
        int ret;
        int opt;
        mode_t mode;
        struct sockaddr_un addr;
	struct sockaddr_in *in_addr;
        socklen_t len;

        if(logger_get_log_level(LOG_CAT_L7VSD_ENVIRONMENT) == LOG_LV_DEBUG){
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_ENVIRONMENT,1,
                "in_function: int l7vs_config_init return int");
        }

	if (l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,1,"config is already initialized.");
        	if(logger_get_log_level(LOG_CAT_L7VSD_ENVIRONMENT) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_ENVIRONMENT,2,
                	"out_function: int l7vs_config_init return -1");
        	}
		return -1;
	}

        if (sizeof(addr.sun_path) < sizeof(l7vs_config_sockname)) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,1,"Internal error The config socket name is too large.");
        	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,1,
                	"out_function: int l7vs_config_init return -1");
        	}
                return -1;
        }

        s = socket(PF_LOCAL, SOCK_DGRAM, 0);
        if (s < 0) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,1,"socket: %s (Socket Type is SOCK_DGRAM)", strerror(errno));
        	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,2,
                	"out_function: int l7vs_config_init return < 0");
        	}
                return s;
        }

        opt = 1;
        ret = setsockopt(s, SOL_SOCKET, SO_PASSCRED, &opt, sizeof(opt));
        if (ret < 0) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,2,"setsockopt SO_PASSCRED: %s", strerror(errno));
                close(s);
        	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,3,
                	"out_function: int l7vs_config_init return < 0");
        	}
                return ret;
        }

        memset(&addr, 0, sizeof(addr));
        addr.sun_family = AF_LOCAL;
        strcpy(addr.sun_path, l7vs_config_sockname);
        len = SUN_LEN(&addr);

        mode = umask(0077);
        if (mode < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM,1,"umask: %s", strerror(errno));
                close(s);
        	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM,1,
                	"out_function: int l7vs_config_init return -1");
        	}
                return -1;
        }

        ret = bind(s, (struct sockaddr *)&addr, len);
        if (ret < 0) {
		in_addr = (struct sockaddr_in*)&addr;
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,3,"cannot create socket file \"%s\": %s", 
				addr.sun_path, strerror(errno));
                if (errno == EADDRINUSE) {
                        LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,4, "Config socket already exists. "
                                           "You could remove it if" " the l7vs daemon is not running");
                }
                close(s);
                umask(mode);
        	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,1,
                	"out_function: int l7vs_config_init return < 0");
        	}
                return ret;
        }
        umask(mode);

	l7vs_config_iomux = l7vs_iomux_get_from_avail_list();
	if (!l7vs_config_iomux) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,2,"can not get config_iomux");
        	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,4,
                	"out_function: int l7vs_config_init return < 0");
        	}
		return -1;
	}
	
        l7vs_config_iomux->fd = s;
        l7vs_config_iomux->data = NULL;
        l7vs_config_iomux->callback = l7vs_config_callback;
        l7vs_config_iomux->status = iomux_config_command_waiting;
	l7vs_iomux_add( l7vs_config_iomux, iom_read );

        if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
                char iomux_str[DEBUG_STR_LEN] = {0};
		l7vs_iomux_c_str(iomux_str,l7vs_config_iomux);
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,23,
                                      "creat: l7vs_config_iomux=%s",
                                       iomux_str);
        }                                  

        return 0;
}

/*!
 * Operate config finish
 * @return      void
 */
void
l7vs_config_fini(void)
{
        if(logger_get_log_level(LOG_CAT_L7VSD_ENVIRONMENT) == LOG_LV_DEBUG){
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_ENVIRONMENT,3,
                "in_function: void l7vs_config_fini return void");
        }
	if (l7vs_config_iomux) {
		l7vs_config_iomux->status = iomux_config_destroyed;
		l7vs_iomux_remove(l7vs_config_iomux);
		if (-1 != l7vs_config_iomux->fd) {
		        close(l7vs_config_iomux->fd);
		}
		l7vs_iomux_put_to_avail_list(l7vs_config_iomux);
		l7vs_config_iomux = NULL;
	}
        unlink(l7vs_config_sockname);
        if(logger_get_log_level(LOG_CAT_L7VSD_ENVIRONMENT) == LOG_LV_DEBUG){
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_ENVIRONMENT,4,
                "out_function: void l7vs_config_fini return void");
        }
}

/*!             
 * config call back
 * @param[in]           struct l7vs_iomux      l7vs_iomux pointer
 * @return              int             number of VirtualService, NG<0
 */     
static int
l7vs_config_callback(struct l7vs_iomux *iom )
{
        struct msghdr msg;
        struct sockaddr_un addr;
        struct iovec iov;
        struct ucred *cred;
        struct cmsghdr *cmsg;
        unsigned char cbuf[CMSG_LEN(sizeof(struct ucred))];
        union l7vs_config_req req;
        int ret;

	if (!iom) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,2,"error / iom is null");
        	if(logger_get_log_level(LOG_CAT_L7VSD_ENVIRONMENT) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_ENVIRONMENT,5,
                	"out_function: int l7vs_config_callback (struct l7vs_iomux *iom) return -1");
        	}
		return -1;
	}
	if (iomux_config_command_waiting != iom->status) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,2,"error / invalid status(%d)", iom->status);
        	if(logger_get_log_level(LOG_CAT_L7VSD_ENVIRONMENT) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_ENVIRONMENT,6,
                	"out_function: int l7vs_config_callback (struct l7vs_iomux *iom) return -1");
        	}
		return -1;
	}

        if(logger_get_log_level(LOG_CAT_L7VSD_ENVIRONMENT) == LOG_LV_DEBUG){
                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_ENVIRONMENT,7,
                "in_function: int l7vs_config_callback (struct l7vs_iomux *iom) iom = %p",iom);
        }

        cmsg = (struct cmsghdr *)cbuf;
        cmsg->cmsg_len = sizeof(cbuf);

        memset(&addr, 0, sizeof(addr));
        memset(&msg, 0, sizeof(msg));
        msg.msg_name = &addr;
        msg.msg_namelen = sizeof(addr);
        msg.msg_iov = &iov;
        msg.msg_iovlen = 1;
        msg.msg_control = cmsg;
        msg.msg_controllen = sizeof(cbuf);

        iov.iov_base = &req;
        iov.iov_len = sizeof(req);

        ret = recvmsg(iom->fd, &msg, 0);
        if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,3,"recvmsg on config socket: %s", strerror(errno));
        	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,5,
                	"out_function: int l7vs_config_callback (struct l7vs_iomux *iom) return -1");
        	}
                return -1;
        }

        if (msg.msg_flags & (MSG_TRUNC|MSG_CTRUNC)) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,4,"msg_flags=%x", msg.msg_flags);
        	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,6,
                	"out_function: int l7vs_config_callback (struct l7vs_iomux *iom) return -1");
        	}
                return -1;
        }

        if (! (cmsg->cmsg_level == SOL_SOCKET &&
               cmsg->cmsg_type == SCM_CREDENTIALS)) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,5,"Could not receive a remote credential");
        	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,7,
                	"out_function: int l7vs_config_callback (struct l7vs_iomux *iom) return -1");
        	}
                return -1;
        }

        /*
         * actually credential check is unnecessary,
         * at least for Linux 2.4.20.
         */
        cred = (struct ucred *)CMSG_DATA(cmsg);
        if (cred->uid != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_NETWORK,6,"Request from unprivileged user");
        	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,8,
                	"out_function: int l7vs_config_callback (struct l7vs_iomux *iom) return -1");
        	}
                return -1;
        }

        l7vs_config_dispatch(&req, &addr);

	l7vs_config_iomux->status = iomux_config_command_waiting;
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,9,
               	"out_function: int l7vs_config_callback (struct l7vs_iomux *iom) return 0");
       	}

        return 0;
}

struct l7vs_config_table {
        enum l7vs_config_command cmd;
        void (*func)(union l7vs_config_req *req, struct sockaddr_un *addr);
};

struct l7vs_config_table l7vs_config_table[] = {
        {L7VS_CONFIG_LIST_VS,   l7vs_config_list_vs},
        {L7VS_CONFIG_LIST_RS,   l7vs_config_list_rs},
        {L7VS_CONFIG_ADD_VS,    l7vs_config_add_vs},
        {L7VS_CONFIG_DEL_VS,    l7vs_config_del_vs},
        {L7VS_CONFIG_EDIT_VS,   l7vs_config_edit_vs},
        {L7VS_CONFIG_ADD_RS,    l7vs_config_add_rs},
        {L7VS_CONFIG_DEL_RS,    l7vs_config_del_rs},
        {L7VS_CONFIG_EDIT_RS,   l7vs_config_edit_rs},
        {L7VS_CONFIG_FLUSH_VS,  l7vs_config_flush_vs},
	{L7VS_CONFIG_GET_REPLICATION_INFO,l7vs_config_get_replication_info},
	{L7VS_CONFIG_REPLICATION,         l7vs_config_replication},
	{L7VS_CONFIG_GET_LOG_INFO,        l7vs_config_get_log_info},
	{L7VS_CONFIG_SET_LOG_LEVEL,       l7vs_config_log},
	{L7VS_CONFIG_SET_LOG_LEVEL_ALL,   l7vs_config_log},
	{L7VS_CONFIG_GET_SNMP_LOG_INFO,   l7vs_config_get_snmp_log_info},
	{L7VS_CONFIG_GET_SNMP_CONNECT_STATUS, l7vs_config_get_snmp_connect_status},
	{L7VS_CONFIG_SET_SNMP_LOG_LEVEL,  l7vs_config_snmp},
	{L7VS_CONFIG_SET_SNMP_LOG_LEVEL_ALL,l7vs_config_snmp_all},
	{L7VS_CONFIG_PARAMETER,           l7vs_config_parameter},
        {L7VS_CONFIG_NONE, NULL}
};

/*!             
 * config event dispatch
 * @param[in]           union l7vs_config_req request messege from l7vsadm
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_dispatch(union l7vs_config_req *req, struct sockaddr_un *addr)
{
        struct l7vs_config_table *t;
        struct iovec iov;
        struct l7vs_config_rsp_unknown_cmd rsp;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,10,
               	"in_function: void l7vs_config_dispatch (union l7vs_config_req *req, struct sockaddr_un *addr)"
		"req= %p: addr= %p",req,addr);
       	}
        for (t = l7vs_config_table; t->cmd != L7VS_CONFIG_NONE; t++) {
                if (t->cmd == req->cmd) {
                        break;
                }
        }

        if (t->cmd == L7VS_CONFIG_NONE) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,3,"Unknown command from config socket");
                rsp.cmd = req->cmd;
                rsp.code = L7VS_CONFIG_ERR_INVALID_COMMAND;
                iov.iov_base = &rsp;
                iov.iov_len = sizeof(rsp);
                l7vs_config_send_response(addr, &iov, 1);
                return;
        }

        t->func(req, addr);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,11,
               	"out_function: void l7vs_config_dispatch return void");
       	}
}

/*!             
 * config event list virtual service
 * @param[in]           union l7vs_config_req request messege from l7vsadm
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_list_vs(union l7vs_config_req *arg,
                    struct sockaddr_un *addr)
{
        struct l7vs_config_req_list_vs *req = &arg->list_vs;
        struct l7vs_config_rsp_list_vs rsp;
        struct l7vs_service_arg_multi *sas;
        struct iovec iov[2];
        int iovlen;
        int len;


       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,12,
               	"in_function: void l7vs_config_list_vs (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,3,"error / iom is null");
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_list_vs_received;

        sas = NULL;
        rsp.cmd = req->cmd;
        rsp.code = 0;
        rsp.num = 0;
        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);

        len = l7vs_service_list_service_arg(&sas, &rsp.num);
        if (len < 0) {
		l7vs_config_iomux->status = iomux_config_command_nomem_error;
                rsp.code = L7VS_CONFIG_ERR_NOMEM;
                l7vs_config_send_response(addr, iov, 1);
                return;
        }

        if (rsp.num == 0) {
                iovlen = 1;
        } else {
                iov[1].iov_base = sas;
                iov[1].iov_len = len;
                iovlen = 2;
        }

	l7vs_config_iomux->status = iomux_config_command_list_vs_succeeded;

        if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
                char config_str[DEBUG_STR_LEN] = {0};

		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: "
			"num=%d: ",
			rsp.cmd,rsp.code,rsp.num);

                LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,24,
                                      "list VirtualService: send data=%s",
                                       config_str);
        }

        l7vs_config_send_response(addr, iov, iovlen);
        if (sas != NULL) {
                free(sas);
        }
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,13,
               	"out_function: void l7vs_config_list_vs return void");
       	}
        return;
}

/*!             
 * config event list real server
 * @param[in]           union l7vs_config_req request messege from l7vsadm
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_list_rs(union l7vs_config_req *arg,
                    struct sockaddr_un *addr)
{
        struct l7vs_config_req_list_rs *req = &arg->list_rs;
        struct l7vs_config_rsp_list_rs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service  *srv=NULL;
        struct l7vs_dest_arg *das=NULL;
        struct iovec iov[2];
        int iovlen;
        int ret;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,14,
               	"in_function: void l7vs_config_list_rs (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,4,"error / iom is null");
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_list_rs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;
        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);

        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,5,"error / service not found");

		l7vs_config_iomux->status = iomux_config_command_novs_error;
                rsp.code = L7VS_CONFIG_ERR_NOVS;

        	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
                	char config_str[DEBUG_STR_LEN] = {0};
			snprintf(config_str,DEBUG_STR_LEN,
				"cmd=%d: "
				"code=%d: ",
				rsp.cmd,rsp.code);
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,2,
                                      "list RealServer: send data=%s",
                                       config_str);
        	}
                l7vs_config_send_response(addr, iov, 1);
                return;
        }

	l7vs_config_iomux->status = iomux_config_command_list_rs_vs_found;

        ret = l7vs_service_list_dest_arg(srv, &das);
        if (ret < 0) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,6,"error / dest not found");
		l7vs_config_iomux->status = iomux_config_command_nomem_error;
                rsp.code = L7VS_CONFIG_ERR_NOMEM;
        	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
                	char config_str[DEBUG_STR_LEN] = {0};
			snprintf(config_str,DEBUG_STR_LEN,
				"cmd=%d: "
				"code=%d: ",
				rsp.cmd,rsp.code);
                	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,3,
                                      "list RealServer: send data=%s",
                                       config_str);
        	}
                l7vs_config_send_response(addr, iov, 1);
                return;
        }

        rsp.num = ret;
        if (ret == 0) {
                iovlen = 1;
        } else {
                iov[1].iov_base = das;
                iov[1].iov_len = ret * sizeof(*das);
                iovlen = 2;
        }

	l7vs_config_iomux->status = iomux_config_command_list_rs_succeeded;
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,4,
                                      "list RealServer: send data=%s",
                                     config_str);
       	}

        l7vs_config_send_response(addr, iov, iovlen);
        if (das != NULL) {
                free(das);
		das = NULL;
        }

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,15,
               	"out_function: void l7vs_config_list_rs return void");
       	}
}

/*!             
 * config event add virtual service
 * @param[in]           union l7vs_config_req request messege from l7vsadm
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_add_vs(union l7vs_config_req *arg,
                   struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_vs *req = &arg->operate_vs;
        struct l7vs_config_rsp_operate_vs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service *srv;
        struct iovec iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,16,
               	"in_function: void l7vs_config_add_vs (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,7,"error / iom is null");
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_add_vs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv != NULL) {
		if(logger_get_log_level(LOG_CAT_L7VSD_VIRTUAL_SERVICE) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_VIRTUAL_SERVICE,1,"ADD_VS: service already exists");
		}
		l7vs_config_iomux->status = iomux_config_command_vs_exist_error;
                rsp.code = L7VS_CONFIG_ERR_VS_EXISTS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_add_vs_vs_not_found;

        srv = l7vs_service_create(sarg, &rsp.code);
        if (srv == NULL) {
		if(logger_get_log_level(LOG_CAT_L7VSD_VIRTUAL_SERVICE) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_VIRTUAL_SERVICE,2,"ADD_VS: service already exists");
		}
		l7vs_config_iomux->status = iomux_config_command_vs_create_error;
		goto out;
        } else {
		l7vs_config_iomux->status = iomux_config_command_add_vs_vs_created;
		LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_VIRTUAL_SERVICE,1,"ADD_VS: IFRM001: created a service ");
	}

	l7vs_config_iomux->status = iomux_config_command_add_vs_succeeded;

out:
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,5,
                                     "add VertualService: send data=%s",
                                     config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,17,
               	"out_function: void l7vs_config_add_vs return void");
       	}
}

/*!             
 * config event delete virtual service
 * @param[in]           union l7vs_config_req request messege from l7vsadm
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_del_vs(union l7vs_config_req *arg,
                   struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_vs *req = &arg->operate_vs;
        struct l7vs_config_rsp_operate_vs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service *srv;
	struct l7vs_dest *dest;
        struct iovec iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,18,
               	"in_function: void l7vs_config_del_vs (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,8,"error / iom is null");
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_del_vs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
		if(logger_get_log_level(LOG_CAT_L7VSD_VIRTUAL_SERVICE) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_VIRTUAL_SERVICE,3,"DEL_VS: service nonexistent");
		}
		l7vs_config_iomux->status = iomux_config_command_novs_error;
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_del_vs_vs_found;

	while (g_list_length(srv->dest_list) > 0) {
		dest = (struct l7vs_dest *)((g_list_first(srv->dest_list))->data);
		srv->dest_list = g_list_remove(srv->dest_list, dest);
		l7vs_dest_destroy(dest);
	}

	l7vs_config_iomux->status = iomux_config_command_del_vs_dest_destroyed;

        l7vs_service_destroy(srv);

	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_VIRTUAL_SERVICE,3,"DEL_VS: IFRM003: released virtual service");
	l7vs_config_iomux->status = iomux_config_command_del_vs_succeeded;

out:
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,6,
                                    "delete VertualService: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,19,
               	"out_function: void l7vs_config_del_vs retrurn void");
       	}
}

/*!             
 * config event edit virtual service
 * @param[in]           union l7vs_config_req request messege from l7vsadm
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_edit_vs(union l7vs_config_req *arg,
                    struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_vs *req = &arg->operate_vs;
        struct l7vs_config_rsp_operate_vs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service *srv;
        struct l7vs_scheduler *sched, *sched_old;
        struct iovec iov;
	int	retval;

	struct l7vs_service_arg_multi tmp_sarg;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,20,
               	"in_function: void l7vs_config_edit_vs (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}

	memset(&tmp_sarg,0x00,sizeof(struct l7vs_service_arg_multi));

	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,9,"error / iom is null");
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_edit_vs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
		if(logger_get_log_level(LOG_CAT_L7VSD_VIRTUAL_SERVICE) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_VIRTUAL_SERVICE,4,"EDIT_VS: service nonexistent");
		}
		l7vs_config_iomux->status = iomux_config_command_novs_error;
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_edit_vs_vs_found;


	if(sarg->srv_arg.sorry_cc != -1) {
		tmp_sarg.srv_arg.sorry_cc = sarg->srv_arg.sorry_cc;
	} else{
		tmp_sarg.srv_arg.sorry_cc = srv->sorry_cc;
	}

	memset(&(tmp_sarg.srv_arg.sorry_addr),0xff,sizeof(struct sockaddr_storage));
	if(memcmp(&(tmp_sarg.srv_arg.sorry_addr),&(sarg->srv_arg.sorry_addr),
	sizeof(struct sockaddr_storage))) {
		memcpy(&(tmp_sarg.srv_arg.sorry_addr),
			&(sarg->srv_arg.sorry_addr),
			sizeof(struct sockaddr_storage));
	} else{
		memcpy(&(tmp_sarg.srv_arg.sorry_addr),
			&(srv->sorry_dest->addr),
			sizeof(struct sockaddr_in));
	}

	if(sarg->srv_arg.sorry_flag != -1) {
		tmp_sarg.srv_arg.sorry_flag = sarg->srv_arg.sorry_flag;
	} else{
		tmp_sarg.srv_arg.sorry_flag = srv->sorry_flag;
	}

	if(sarg->srv_arg.qos_threshold_up != ULLONG_MAX) {
		tmp_sarg.srv_arg.qos_threshold_up = sarg->srv_arg.qos_threshold_up;
	} else{
		tmp_sarg.srv_arg.qos_threshold_up = srv->qos_threshold_up;
	}

	if(sarg->srv_arg.qos_threshold_down != ULLONG_MAX) {
		tmp_sarg.srv_arg.qos_threshold_down = sarg->srv_arg.qos_threshold_down;
	} else{
		tmp_sarg.srv_arg.qos_threshold_down = srv->qos_threshold_down;
	}

	retval = l7vs_service_set_QoS_Threshold( srv, &tmp_sarg );
	if( 0 > retval ){
		if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK_QOS) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK_QOS,1,"EDIT_VS: QoS set failure");
		}
		l7vs_config_iomux->status = iomux_config_command_set_qos_error;
                rsp.code = L7VS_CONFIG_ERR_QOSSET;
                goto out;
	}
	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK_QOS) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK_QOS,2,"EDIT_VS: QoS set success");
	}

	retval = l7vs_service_set_SorryServer_Values( srv, &tmp_sarg );
	if( 0 > retval ){
		if(logger_get_log_level(LOG_CAT_L7VSD_SORRY_SERVER) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SORRY_SERVER,1,"EDIT_VS: SorryServer set failure");
		}
		l7vs_config_iomux->status = iomux_config_command_set_sorry_server_error;
                rsp.code = L7VS_CONFIG_ERR_SORRYSET;
                goto out;
	}
	if(logger_get_log_level(LOG_CAT_L7VSD_SORRY_SERVER) == LOG_LV_DEBUG){
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SORRY_SERVER,2,"EDIT_VS: SorryServer set success");
	}

	memset(&(tmp_sarg.srv_arg.schedmod),0x00,sizeof(tmp_sarg.srv_arg.schedmod));
	if(memcmp(&(tmp_sarg.srv_arg.schedmod),&(sarg->srv_arg.schedmod),
	sizeof(tmp_sarg.srv_arg.schedmod))) {

        	sched = (struct l7vs_scheduler* )l7vs_sched_get(sarg->srv_arg.schedmod);
        	if (sched == NULL) {
			if(logger_get_log_level(LOG_CAT_L7VSD_SCHEDULE) == LOG_LV_DEBUG){
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,1,"EDIT_VS: unknown scheduler");
			}
			l7vs_config_iomux->status = iomux_config_command_nosched_error;
                	rsp.code = L7VS_CONFIG_ERR_NOSCHED;
                	goto out;
        	} else {
			if(logger_get_log_level(LOG_CAT_L7VSD_SCHEDULE) == LOG_LV_DEBUG){
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SCHEDULE,2,"EDIT_VS: scheduler set success");
			}
		}
	
		l7vs_config_iomux->status = iomux_config_command_edit_vs_sched_got;
	
        	sched_old = srv->scheduler;
        	if (sched != sched_old) {
                	l7vs_sched_unbind(sched_old, srv);
                	l7vs_sched_put(sched_old);
                	l7vs_sched_bind(sched, srv);
        	}
	}

	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSD_VIRTUAL_SERVICE,2,"EDIT_VS: IFRM002: changed virtual service");
	l7vs_config_iomux->status = iomux_config_command_edit_vs_succeeded;

out:
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,7,
                                    "edit VertualService: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,21,
               	"out_function: void l7vs_config_edit_vs return void");
       	}
}

/*!             
 * config event add real server
 * @param[in]           union l7vs_config_req request messege from l7vsadm
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_add_rs(union l7vs_config_req *arg,
                   struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_rs *req = &arg->operate_rs;
        struct l7vs_config_rsp_operate_rs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service *srv;
        struct iovec iov;
        int ret;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,22,
               	"in_function: void l7vs_config_add_rs (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,10,"error / iom is null");
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_add_rs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
		if(logger_get_log_level(LOG_CAT_L7VSD_REAL_SERVER) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REAL_SERVER,1,"ADD_RS: service nonexistent");
		}
		l7vs_config_iomux->status = iomux_config_command_novs_error;
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_add_rs_vs_found;

        if (l7vs_service_lookup_dest(srv, &req->darg.addr) != NULL) {
		if(logger_get_log_level(LOG_CAT_L7VSD_REAL_SERVER) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REAL_SERVER,2,"ADD_RS: real server already exists");
		}
		l7vs_config_iomux->status = iomux_config_command_rs_exist_error;
                rsp.code = L7VS_CONFIG_ERR_RS_EXISTS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_add_rs_rs_not_found;

        ret = l7vs_service_add_dest(srv, &req->darg);
        if (ret < 0) {
		if(logger_get_log_level(LOG_CAT_L7VSD_REAL_SERVER) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REAL_SERVER,3,"ADD_RS: failed to add real server");
		}
		l7vs_config_iomux->status = iomux_config_command_nomem_error;
                rsp.code = L7VS_CONFIG_ERR_NOMEM;
		goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_add_rs_succeeded;

out:
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,8,
                                    "add RealServer: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,23,
               	"out_function: void l7vs_config_add_rs return void");
       	}
}

/*!             
 * config event delete real server
 * @param[in]           union l7vs_config_req request messege from l7vsadm
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_del_rs(union l7vs_config_req *arg,
                   struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_rs *req = &arg->operate_rs;
        struct l7vs_config_rsp_operate_rs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service *srv;
        struct l7vs_dest *dest;
        struct iovec iov;
        int ret;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,24,
               	"in_function: void l7vs_config_del_rs (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,11,"error / iom is null");
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_del_rs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
		if(logger_get_log_level(LOG_CAT_L7VSD_REAL_SERVER) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REAL_SERVER,4,"DEL_RS: service nonexistent");
		}
		l7vs_config_iomux->status = iomux_config_command_novs_error;
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_del_rs_vs_found;

        dest = l7vs_service_lookup_dest(srv, &req->darg.addr);
        if (dest == NULL) {
		if(logger_get_log_level(LOG_CAT_L7VSD_REAL_SERVER) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REAL_SERVER,5,"DEL_RS: destination nonexistent");
		}
		l7vs_config_iomux->status = iomux_config_command_nors_error;
                rsp.code = L7VS_CONFIG_ERR_NORS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_del_rs_rs_found;

        ret = l7vs_service_remove_dest(srv, &req->darg);
        if (ret < 0) {
		if(logger_get_log_level(LOG_CAT_L7VSD_REAL_SERVER) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REAL_SERVER,6,"DEL_RS: failed to remove real server");
		}
                /* XXX actually impossible */
                rsp.code = L7VS_CONFIG_ERR_NORS;
		goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_del_rs_succeeded;

out:
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,9,
                                    "delete RealServer: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,25,
               	"out_function: void l7vs_config_del_rs return void");
       	}
}

/*!             
 * config event edit real server
 * @param[in]           union l7vs_config_req request messege from l7vsadm
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_edit_rs(union l7vs_config_req *arg,
                    struct sockaddr_un *addr)
{
        struct l7vs_config_req_operate_rs *req = &arg->operate_rs;
        struct l7vs_config_rsp_operate_rs rsp;
        struct l7vs_service_arg_multi *sarg;
        struct l7vs_service *srv;
        struct l7vs_dest *dest;
        struct iovec iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,26,
               	"in_function: void l7vs_config_edit_rs (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,12,"error / iom is null");
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_edit_rs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        sarg = (struct l7vs_service_arg_multi *)(req + 1);
        srv = l7vs_service_lookup(sarg);
        if (srv == NULL) {
		if(logger_get_log_level(LOG_CAT_L7VSD_REAL_SERVER) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REAL_SERVER,7,"EDIT_RS: service nonexistent");
		}
                rsp.code = L7VS_CONFIG_ERR_NOVS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_edit_rs_vs_found;

        dest = l7vs_service_lookup_dest(srv, &req->darg.addr);
        if (dest == NULL) {
		if(logger_get_log_level(LOG_CAT_L7VSD_REAL_SERVER) == LOG_LV_DEBUG){
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_REAL_SERVER,8,"EDIT_RS: destination nonexistent");
		}
                rsp.code = L7VS_CONFIG_ERR_NORS;
                goto out;
        }

	l7vs_config_iomux->status = iomux_config_command_edit_rs_rs_found;

	if(req->darg.weight != -1){
        	dest->weight = req->darg.weight;
	}

	l7vs_config_iomux->status = iomux_config_command_edit_rs_succeeded;

out:
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,10,
                                    "edit RealServer: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,27,
               	"out_function: void l7vs_config_edit_rs return void");
       	}
}

/*!             
 * config event delete all viertual service
 * @param[in]           union l7vs_config_req request messege from l7vsadm
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_flush_vs(union l7vs_config_req *arg, struct sockaddr_un *addr)
{
        struct l7vs_config_req_flush_vs *req = &arg->flush_vs;
        struct l7vs_config_rsp_flush_vs rsp;
        struct iovec iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,28,
               	"in_function: void l7vs_config_flush_vs (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,13,"error / iom is null");
		return;
	}
	l7vs_config_iomux->status = iomux_config_command_flush_vs_received;

        rsp.cmd = req->cmd;
        rsp.code = 0;

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        l7vs_service_flush_all();

	l7vs_config_iomux->status = iomux_config_command_flush_vs_succeeded;

       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,11,
                                    "flush VertualService: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,29,
               	"in_function: void l7vs_config_flush_vs return void");
       	}
}

/*!             
 * config send response
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @param[in]           struct iovec sending messege 
 * @param[in]           int sending messege size
 * @return              int   sendmsg NG<0 OK >=0
 */     
static int
l7vs_config_send_response(struct sockaddr_un *addr,
                          struct iovec *iov,
                          int iovlen)
{
        struct  msghdr msg;
        int ret;
	struct l7vs_config_rsp_unknown_cmd rsp;
	struct iovec err_iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,30,
               	"in_function: int l7vs_config_send_response (struct sockaddr_un *addr,struct iovec *iov,int iovlen)"
		"addr= %p : iov= %p : iovlen= %d",addr,iov,iovlen);
       	}

	if (!l7vs_config_iomux)	{
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,14,"error / iom is null");
		return -1;
	}

        memset(&msg, 0, sizeof(msg));
        msg.msg_name = addr;
        msg.msg_namelen = sizeof(*addr);
        msg.msg_iov = iov;
        msg.msg_iovlen = iovlen;

        ret = sendmsg(l7vs_config_iomux->fd, &msg, 0);
        if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,5,"sendmsg on config socket: %s", strerror(errno));
		// Try sending error response to l7vsadm.
		rsp.cmd = ((struct l7vs_config_rsp_unknown_cmd *)(&iov->iov_base))->cmd;
		rsp.code = L7VS_CONFIG_ERR_SENDMSG;
		err_iov.iov_base = &rsp;
		err_iov.iov_len = sizeof(rsp);
		memset(&msg, 0, sizeof(msg));
		msg.msg_name = addr;
		msg.msg_namelen = sizeof(*addr);
		msg.msg_iov = &err_iov;
		msg.msg_iovlen = 1;
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,6,"Try sending error response to l7vsadm.");
		if (sendmsg(l7vs_config_iomux->fd, &msg, 0) < 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_SYSTEM_SOCKET,7,"retry error sendmsg on config socket: %s", strerror(errno));
		}
        }
	l7vs_config_iomux->status = iomux_config_command_response_sent;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,31,
               	"out_function: int l7vs_config_send_response return ret = %d",ret);
       	}
        return ret;
}

/*!             
 * config event get replication information
 * @param[in]           union l7vs_config_req messege pointer
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_get_replication_info (union l7vs_config_req *arg, struct sockaddr_un *addr)
{
        struct l7vs_config_msg_replication *req = &arg->msg_replication;
        struct l7vs_config_msg_replication rsp;
        struct iovec iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,32,
               	"in_function: void l7vs_config_get_replication_info (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}

	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,15,"error / iom is null");
		return;
	}

	memset(&rsp,0x00,sizeof(struct l7vs_config_msg_replication));
        rsp.cmd = req->cmd;
        rsp.code = 0;
        iov.iov_base = &rsp;
        iov.iov_len = sizeof(struct l7vs_config_msg_replication);
        rsp.replication_mode = l7vs_replication_get_status();
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,12,
                                    "get replication info: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,33,
               	"out_function: void l7vs_config_get_replication_info return void");
       	}
}

/*!             
 * config event get snmp information
 * @param[in]           union l7vs_config_req messege pointer
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_get_snmp_log_info (union l7vs_config_req *arg, struct sockaddr_un *addr)
{
        struct l7vs_config_msg_snmp *req = &arg->msg_snmp;
        struct l7vs_config_msg_snmp rsp;
        struct iovec iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,34,
               	"in_function: void l7vs_config_get_snmp_log_info (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}

	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,16,"error / iom is null");
		return;
	}

	memset(&rsp,0x00,sizeof(struct l7vs_config_msg_snmp));
	memcpy(&rsp,req,sizeof(struct l7vs_config_msg_snmp));
        rsp.code = 0;
        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);
	rsp.snmp_log_level = l7vs_snmpbridge_get_loglevel(rsp.kind);
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,13,
                                    "get snmp log level: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,35,
               	"out_function: void l7vs_config_get_snmp_log_info return void");
       	}
}

/*!             
 * config event get snmp connect status
 * @param[in]           union l7vs_config_req messege pointer
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_get_snmp_connect_status (union l7vs_config_req *arg, struct sockaddr_un *addr)
{
        struct l7vs_config_msg_snmp *req = &arg->msg_snmp;
        struct l7vs_config_msg_snmp rsp;
        struct iovec iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,36,
               	"in_function: void l7vs_config_get_snmp_connect_status (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}

	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,17,"error / iom is null");
		return;
	}

	memset(&rsp,0x00,sizeof(struct l7vs_config_msg_snmp));
	memcpy(&rsp,req,sizeof(struct l7vs_config_msg_snmp));
        rsp.code = 0;
        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);
	rsp.connection_status = l7vs_snmpbridge_get_connectionstate();
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_PROGRAM,25,
                                    "get snmp connect status: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,37,
               	"out_function: void l7vs_config_get_snmp_connect_status return void");
       	}
}

/*!             
 * config event get log level information
 * @param[in]           union l7vs_config_req messege pointer
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_get_log_info (union l7vs_config_req *arg, struct sockaddr_un *addr)
{
        struct l7vs_config_msg_log *req = &arg->msg_log;
        struct l7vs_config_msg_log rsp;
        struct iovec iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,38,
               	"in_function: void l7vs_config_get_log_info (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,18,"error / iom is null");
		return;
	}

	memset(&rsp,0x00,sizeof(struct l7vs_config_msg_log));
	memcpy(&rsp,req,sizeof(struct l7vs_config_msg_log));
        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);
	rsp.vsd_log_level = logger_get_log_level(req->kind);
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,14,
                                    "get snmp connect status: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,39,
               	"out_function: void l7vs_config_get_log_info return void");
       	}
}

/*!             
 * config event action replication
 * @param[in]           union l7vs_config_req messege pointer
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_replication (union l7vs_config_req *arg, struct sockaddr_un *addr)
{
        struct l7vs_config_msg_replication *req = &arg->msg_replication;
        struct l7vs_config_msg_replication rsp;
        struct iovec iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,40,
               	"in_function: void l7vs_config_replication (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,19,"error / iom is null");
		return;
	}
	memset(&rsp,0x00,sizeof(struct l7vs_config_msg_replication));
	memcpy(&rsp,req,sizeof(struct l7vs_config_msg_replication));
        rsp.code = 0;
        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);
	switch(rsp.kind){
		case L7VS_CONFIG_REPLICATION_DUMP:
			l7vs_replication_dump_memory();
			break;
		case L7VS_CONFIG_REPLICATION_START:
			l7vs_replication_start();
			break;
		case L7VS_CONFIG_REPLICATION_STOP:
			l7vs_replication_stop();
			break;
		case L7VS_CONFIG_REPLICATION_COMPULSORILY:
			l7vs_replication_compulsorily();
			break;
	}
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,15,
                                    "config replication: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,41,
               	"out_function: void l7vs_config_replication return void");
       	}
}

/*!             
 * config event change snmp bridge apointed category log level
 * @param[in]           union l7vs_config_req messege pointer
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_snmp (union l7vs_config_req *arg, struct sockaddr_un *addr)
{
        struct l7vs_config_msg_snmp *req = &arg->msg_snmp;
        struct l7vs_config_msg_snmp rsp;
        struct iovec iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,42,
               	"in_function: void l7vs_config_snmp (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,20,"error / iom is null");
		return;
	}
	memset(&rsp,0x00,sizeof(struct l7vs_config_msg_snmp));
	memcpy(&rsp,req,sizeof(struct l7vs_config_msg_snmp));
        rsp.code = 0;
        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);
        l7vs_snmpbridge_change_loglevel(req->kind,req->snmp_log_level);
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,16,
                                    "config snmp: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,43,
               	"out_function: void l7vs_config_snmp return void");
       	}
}

/*!             
 * config event change snmp bridge all category log level
 * @param[in]           union l7vs_config_req messege pointer
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_snmp_all (union l7vs_config_req *arg, struct sockaddr_un *addr)
{
        struct l7vs_config_msg_snmp *req = &arg->msg_snmp;
        struct l7vs_config_msg_snmp rsp;
        struct iovec iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,44,
               	"in_function: void l7vs_config_snmp_all (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,21,"error / iom is null");
		return;
	}
	memset(&rsp,0x00,sizeof(struct l7vs_config_msg_snmp));
	memcpy(&rsp,req,sizeof(struct l7vs_config_msg_snmp));
        rsp.code = 0;
        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);
        l7vs_snmpbridge_change_loglevel_allcategory(req->snmp_log_level);
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,17,
                                    "config snmp: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,45,
               	"out_function: void l7vs_config_snmp_all return void");
       	}
}

/*!             
 * config event change l7vsd apointed category log level
 * @param[in]           union l7vs_config_req messege pointer
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_log (union l7vs_config_req *arg, struct sockaddr_un *addr)
{
        struct l7vs_config_msg_log *req = &arg->msg_log;
        struct l7vs_config_msg_log rsp;
        struct iovec iov;
	LOG_CATEGORY_TAG log_start,log_stop;
	int  l_start,l_stop,i;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,46,
               	"in_function: void l7vs_config_log (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}
	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,22,"error / iom is null");
		return;
	}
	memset(&rsp,0x00,sizeof(struct l7vs_config_msg_log));
	memcpy(&rsp,req,sizeof(struct l7vs_config_msg_log));
        rsp.code = 0;
        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);
	if(rsp.cmd == L7VS_CONFIG_SET_LOG_LEVEL_ALL){
		log_start = logger_get_category_range_start(LOG_MOD_L7VSD);
		log_stop  = logger_get_category_range_end(LOG_MOD_L7VSD);
		l_start =(int)log_start;
		l_stop = (int)log_stop;
		for(i = log_start;i <= l_stop; i++){
        		logger_set_log_level((LOG_CATEGORY_TAG)i,req->vsd_log_level);
		}

	}
	else{
        	logger_set_log_level(req->kind,req->vsd_log_level);
	}

       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,18,
                                    "config log level: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,47,
               	"out_function: void l7vs_config_log return void");
       	}
}

/*!             
 * config event relead parameter file 
 * @param[in]           union l7vs_config_req messege pointer
 * @param[in]           struct sockaddr_un l7vsd socket pointer
 * @return              void
 */     
static void
l7vs_config_parameter (union l7vs_config_req *arg, struct sockaddr_un *addr)
{
        struct l7vs_config_msg_parameter *req = &arg->msg_parameter;
        struct l7vs_config_msg_parameter rsp;
        struct iovec iov;

       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,48,
               	"in_function: void l7vs_config_parameter (union l7vs_config_req *arg, struct sockaddr_un *addr)"
		"arg= %p: addr= %p",arg,addr);
       	}

	if (!l7vs_config_iomux) {
                LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_ENVIRONMENT,23,"error / iom is null");
		return;
	}
	memset(&rsp,0x00,sizeof(struct l7vs_config_msg_parameter));
	memcpy(&rsp,req,sizeof(struct l7vs_config_msg_parameter));
        rsp.code = 0;
        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);
        parameter_reread_file(req->category);
       	if(logger_get_log_level(LOG_CAT_L7VSD_SYSTEM_SOCKET) == LOG_LV_DEBUG){
               	char config_str[DEBUG_STR_LEN] = {0};
		snprintf(config_str,DEBUG_STR_LEN,
			"cmd=%d: "
			"code=%d: ",
			rsp.cmd,rsp.code);
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_SYSTEM_SOCKET,19,
                                    "config parameter: send data=%s",
                                    config_str);
       	}
        l7vs_config_send_response(addr, &iov, 1);
       	if(logger_get_log_level(LOG_CAT_L7VSD_NETWORK) == LOG_LV_DEBUG){
               	LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_NETWORK,49,
               	"in_function: void l7vs_config_parameter return void");
       	}
}
