/*!
 * @file 	iomux.c
 * @brief	It is a code in which sets of iomux are managed. 
 * @brief	An actual set is operated only by this code. 
 * @brief	Moreover, epoll is operated by this code. 
 * @brief	L7VSD: Linux Virtual Server for Layer7 Load Balancing
 *
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/time.h>
#include <sys/epoll.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include <sys/time.h>
#include <glib.h>
#include "parameter_wrapper.h"
#include "logger_wrapper.h"
#include "l7vs.h"
#include "l7vs_conn.h"
#include "l7vs_iomuxlist.h"

/*
#ifndef  MAXEVENTS
#define	 MAXEVENTS	1024		//! epoll max event size
#endif
*/
static int maxevents = 1024;
static struct l7vs_iomux*	iomux_array = NULL;		//! iomux array
static struct epoll_event*	event_array = NULL;		//! event list

static GHashTable*		iomux_hash = NULL;			//! fd hashtable
static GList*			iomux_avail_list = NULL;	//! iomux available list
static unsigned int		iomux_avail_list_length;	//! number of iomux available list
static int			eventpoll = -1;					//! epoll fd

//inner functions
static gboolean	removeall( gpointer, gpointer, gpointer );

/*!
 * initialize iomux_hash
 *
 * @param[in]	void
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
int
l7vs_iomux_init()
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,1,
			"in_function: int l7vs_iomux_init() ");
	}
	/*------ DEBUG LOG END ------*/

	int ret = parameter_is_int_exist( PARAM_COMP_IOMUX, "max_events" );
    if (ret) {
        int event =  parameter_get_int_value( PARAM_COMP_IOMUX, "max_events" );
        if (event > 0)
            maxevents = event;
        else {
       		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_EVENT,1,
       			"l7vsd uses default max_events = %d ,because your value is invalid ",maxevents);
        }
    }
    else {
   		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSD_EVENT,2,
   			"max_events is not specified. l7vsd uses default max_events = %d ",maxevents);
    }

	iomux_avail_list_length = maxevents;
	if (iomux_avail_list == NULL) {
	    iomux_array = (struct l7vs_iomux*) calloc(sizeof(struct l7vs_iomux), maxevents);
	}
	if (!iomux_array) {	
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,26, "error / calloc failed");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,59,
				"in_function: int l7vs_iomux_init() "
				"return_value: -1");
		}
		/*------ DEBUG LOG END ------*/
		return -1;
	}

	if (iomux_avail_list == NULL) {
		event_array = (struct epoll_event*) calloc(sizeof(struct epoll_event), maxevents);
	}
	if (!event_array) {	
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,27, "error / calloc failed");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,60,
				"in_function: int l7vs_iomux_init() "
				"return_value: -1");
		}
		/*------ DEBUG LOG END ------*/
		return -1;
	}

	int i = 0;
	struct l7vs_iomux temp_iom = {-1, NULL, iomux_create, NULL};

	if (-1 !=  eventpoll) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,4, "error / epoll already exist");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,2,
				"in_function: int l7vs_iomux_init() "
				"return_value: -1");
		}
		/*------ DEBUG LOG END ------*/
		return -1;
	}

	//create epoll
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,3,
			"epoll_create : size=%d",
			maxevents);
	}
	/*------ DEBUG LOG END ------*/
	eventpoll = epoll_create(maxevents);
	if (0 > eventpoll) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,5, "error / epoll create error");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,4,
				"in_function: int l7vs_iomux_init() "
				"return_value: %d", eventpoll);
		}
		/*------ DEBUG LOG END ------*/
		return eventpoll;
	}

	//new hashtable
	if (!iomux_hash) {
		iomux_hash = g_hash_table_new(&g_int_hash, &g_int_equal);
	}

	//init iomux_array and avail_list
	if (!iomux_avail_list) {
		for (i = 0; i < maxevents; ++i) {
			memcpy(iomux_array + i, &temp_iom, sizeof(struct l7vs_iomux));
			iomux_avail_list = g_list_append(iomux_avail_list, &iomux_array[i]);
		}
	}

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,5,
			"in_function: int l7vs_iomux_init() "
			"return_value: 1");
	}
	/*------ DEBUG LOG END ------*/
	return 1;
}

/*!
 * finalize iomux_hash
 *
 * @param[in]	void
 * @return	void
 */
void
l7vs_iomux_fini()
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,6,
			"in_function: void l7vs_iomux_fini() ");
	}
	/*------ DEBUG LOG END ------*/

	if (-1 == eventpoll) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,6, "error / epoll is not initialized");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,7,
				"out_function: void l7vs_iomux_fini() "
				"return_value: void");
		}
		/*------ DEBUG LOG END ------*/
		return;
	}

	//all fd close and remove
	l7vs_iomux_removeall();

	//g_hashtable destroy
	if (iomux_hash) {
		g_hash_table_destroy(iomux_hash);
		iomux_hash = NULL;
	}

	//g_list free
	if (iomux_avail_list) {
		g_list_free(iomux_avail_list);
		iomux_avail_list = NULL;
	}

	//epoll close
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,8,
			"close: eventpoll=%d",
			eventpoll);
	}
	/*------ DEBUG LOG END ------*/
	close(eventpoll);
	eventpoll = -1;
	
	if (iomux_array) {
		free(iomux_array);
		iomux_array = NULL;
	}
	
	if (event_array) {
		free(event_array);
		event_array = NULL;
	}
	
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,9,
			"out_function: void l7vs_iomux_fini() "
			"return_value: void");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * get available iom from avail_list
 *
 * @param[in]	void
 * @return	available iom
 */
struct l7vs_iomux*
l7vs_iomux_get_from_avail_list()
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,10,
			"in_function: struct l7vs_iomux* l7vs_iomux_get_from_avail_list() ");
	}
	/*------ DEBUG LOG END ------*/

	if (0 == iomux_avail_list_length) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,7, "warning / no iomux available");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,11,
				"out_function: struct l7vs_iomux* l7vs_iomux_get_from_avail_list() "
				"return_value: iom=NULL");
		}
		/*------ DEBUG LOG END ------*/
		return NULL;
	}

	struct l7vs_iomux *iom = (struct l7vs_iomux *)(g_list_first(iomux_avail_list))->data;
	iomux_avail_list = g_list_remove(iomux_avail_list, (gpointer)iom);
	--iomux_avail_list_length;
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,12,
			"iomux_avail_list_length: %d",
			iomux_avail_list_length);
	}
	/*------ DEBUG LOG END ------*/

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		char ret_iomux_str[DEBUG_STR_LEN] = {0};
		l7vs_iomux_c_str(ret_iomux_str, iom);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,13,
			"out_function: struct l7vs_iomux* l7vs_iomux_get_from_avail_list() "
			"return_value: iom=%s",
			ret_iomux_str);
	}
	/*------ DEBUG LOG END ------*/
	return iom;
}

/*!
 * put disuse iom to avail_list
 *
 * @param[in]	iom	return iom
 * @return	void
 */
void
l7vs_iomux_put_to_avail_list(struct l7vs_iomux *iom)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		char iomux_str[DEBUG_STR_LEN] = {0};
		l7vs_iomux_c_str(iomux_str, iom);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,14,
			"in_function: void l7vs_iomux_put_to_avail_list(struct l7vs_iomux *iom) "
			"iom=%s",
			iomux_str);
	}
	/*------ DEBUG LOG END ------*/

	if (!iom) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,8, "error / iom is null");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,15,
				"out_function: void l7vs_iomux_put_to_avail_list(struct l7vs_iomux *iom) "
				"return_value: void");
		}
		/*------ DEBUG LOG END ------*/
		return;
	}

	iom->fd = -1;
	iom->callback = NULL;
	iom->status = iomux_create;
	iom->data = NULL;

	iomux_avail_list = g_list_append(iomux_avail_list, (gpointer)iom);
	++iomux_avail_list_length;
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,16,
			"iomux_avail_list_length: %d",
			iomux_avail_list_length);
	}
	/*------ DEBUG LOG END ------*/

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,17,
			"out_function: void l7vs_iomux_put_to_avail_list(struct l7vs_iomux *iom) "
			"return_value: void");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * iomux is added to hashtbale
 *
 * @param[in]	*iom	iompointer
 * @param[in]	action	iom epoll action
 * @return	void
 */
void 
l7vs_iomux_add(struct l7vs_iomux* iom, enum iomaction action)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		char iomux_str[DEBUG_STR_LEN] = {0};
		l7vs_iomux_c_str(iomux_str, iom);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,18,
			"in_function: void l7vs_iomux_add(struct l7vs_iomux* iom, enum iomaction action) "
			"iom=%s: "
			"iomaction=%d",
			iomux_str, action);
	}
	/*------ DEBUG LOG END ------*/

	int ret = 0;
	struct epoll_event event;

	if (!iom) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,9, "error / iom is null");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,19,
				"out_function: void l7vs_iomux_add(struct l7vs_iomux* iom, enum iomaction action) "
				"return_value: void");
		}
		/*------ DEBUG LOG END ------*/
		return;
	}
	if (-1 == iom->fd) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,10, "error / fd is not specified");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,20,
				"out_function: void l7vs_iomux_add(struct l7vs_iomux* iom, enum iomaction action) "
				"return_value: void");
		}
		/*------ DEBUG LOG END ------*/
		return;
	}

	event.data.ptr = iom;
	switch( action ){
		case iom_read:
			event.events = EPOLLET | EPOLLIN | EPOLLHUP;
			break;
		case iom_write:
			event.events = EPOLLET | EPOLLOUT | EPOLLHUP;
			break;
		default:
			//error !
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,11, "error / invalid action");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,21,
					"out_function: void l7vs_iomux_add(struct l7vs_iomux* iom, enum iomaction action) "
					"return_value: void");
			}
			/*------ DEBUG LOG END ------*/
			return;
	}

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		char iomux_str[DEBUG_STR_LEN] = {0};
		l7vs_iomux_c_str(iomux_str, iom);	
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,22,
			"epoll_ctl : epfd=%d: op=EPOLL_CTL_ADD: fd=%d: event.events=%d: event.data=%s",
			eventpoll, iom->fd, event.events, iomux_str);
	}
	/*------ DEBUG LOG END ------*/
	ret = epoll_ctl( eventpoll, EPOLL_CTL_ADD, iom->fd, &event );
	g_hash_table_insert(iomux_hash, &(iom->fd), iom );

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,23,
			"out_function: void l7vs_iomux_add(struct l7vs_iomux* iom, enum iomaction action) "
			"return_value: void");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * iomux is remove from hashtable
 *
 * @param[in]	*iom	iompointer
 * @return	void
 */
void
l7vs_iomux_remove(struct l7vs_iomux* iom)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		char iomux_str[DEBUG_STR_LEN] = {0};
		l7vs_iomux_c_str(iomux_str, iom);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,24,
			"in_function: void l7vs_iomux_remove(struct l7vs_iomux* iom) "
			"iom=%s",
			iomux_str);
	}
	/*------ DEBUG LOG END ------*/

	int ret = 0;

	if (!iom) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,12, "error / iom is null");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,25,
				"out_function: void l7vs_iomux_remove(struct l7vs_iomux* iom) "
				"return_value: void");
		}
		/*------ DEBUG LOG END ------*/
		return;
	}
	if (-1 == iom->fd) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,13, "error / fd is not specified");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,26,
				"out_function: void l7vs_iomux_remove(struct l7vs_iomux* iom) "
				"return_value: void");
		}
		/*------ DEBUG LOG END ------*/
		return;
	}
	
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,27,
			"epoll_ctl : epfd=%d: op=EPOLL_CTL_DEL: fd=%d",
			eventpoll, iom->fd);
	}
	/*------ DEBUG LOG END ------*/
	ret = epoll_ctl(eventpoll, EPOLL_CTL_DEL, iom->fd, NULL);
	g_hash_table_remove(iomux_hash, &(iom->fd));

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,28,
			"out_function: void l7vs_iomux_remove(struct l7vs_iomux* iom) "
			"return_value: void");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * iomux modify epoll events
 *
 * @param[in]	*iom	iom pointer
 * @param[in]	action	iom action mode
 * @return	void
 */
void
l7vs_iomux_mod(struct l7vs_iomux* iom, enum iomaction action)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		char iomux_str[DEBUG_STR_LEN] = {0};
		l7vs_iomux_c_str(iomux_str, iom);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,29,
			"in_function: void l7vs_iomux_mod(struct l7vs_iomux* iom, enum iomaction action) "
			"iom=%s: "
			"iomaction=%d",
			iomux_str, action);
	}
	/*------ DEBUG LOG END ------*/

	int ret = 0;
	struct epoll_event	event;

	if (!iom) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,14, "error / iom is null");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,30,
				"out_function: void l7vs_iomux_mod(struct l7vs_iomux* iom, enum iomaction action) "
				"return_value: void");
		}
		/*------ DEBUG LOG END ------*/
		return;
	}
	if (-1 == iom->fd) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,15, "error / fd is not specified");
		/*-------- DEBUG LOG --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,31,
				"out_function: void l7vs_iomux_mod(struct l7vs_iomux* iom, enum iomaction action) "
				"return_value: void");
		}
		/*------ DEBUG LOG END ------*/
		return;
	}

	event.data.ptr = iom;
	switch (action) {
		case	iom_read:
			event.events = EPOLLET | EPOLLIN | EPOLLHUP;
			break;
		case	iom_write:
			event.events = EPOLLET | EPOLLOUT | EPOLLHUP;
			break;
		case	iom_hup:
			event.events = EPOLLET | EPOLLHUP;
			break;
		default:
			//error !
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,16, "error / invalid action");
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,32,
					"out_function: void l7vs_iomux_mod(struct l7vs_iomux* iom, enum iomaction action) "
					"return_value: void");
			}
			/*------ DEBUG LOG END ------*/
			return;
	}
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		char iomux_str[DEBUG_STR_LEN] = {0};
		l7vs_iomux_c_str(iomux_str, iom);	
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,33,
			"epoll_ctl : epfd=%d: op=EPOLL_CTL_MOD: fd=%d: event.events=%d: event.data=%s",
			eventpoll, iom->fd, event.events, iomux_str);
	}
	/*------ DEBUG LOG END ------*/
	ret = epoll_ctl(eventpoll, EPOLL_CTL_MOD, iom->fd, &event);

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,34,
			"out_function: void l7vs_iomux_mod(struct l7vs_iomux* iom, enum iomaction action) "
			"return_value: void");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * close all filedescriptor
 *
 * removeall() function is called l7vs_iomux_delall()
 * GHashTable is don't all contenior;
 * but function callback set all contenor;
 */
static gboolean
removeall(gpointer key, gpointer value, gpointer userdata)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,35,
			"in_function: static gboolean removeall(gpointer key, gpointer value, gpointer userdata) "
			"key=%p: "
			"value=%p: "
			"userdata=%p",
			key, value, userdata);
	}
	/*------ DEBUG LOG END ------*/

	struct l7vs_iomux *iom = (struct l7vs_iomux *)value;
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,36,
			"epoll_ctl : epfd=%d: op=EPOLL_CTL_DEL: fd=%d",
			eventpoll, iom->fd);
	}
	/*------ DEBUG LOG END ------*/
	epoll_ctl(eventpoll, EPOLL_CTL_DEL, iom->fd, NULL);
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,37,
			"out_function: static gboolean removeall(gpointer key, gpointer value, gpointer userdata) "
			"return_value: TRUE");
	}
	/*------ DEBUG LOG END ------*/
	return TRUE;
}

/*!
 * all contnior delete hash_table
 *
 * @return	void*
 */ 
void
l7vs_iomux_removeall()
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,38,
			"in_function: void l7vs_iomux_removeall()");
	}
	/*------ DEBUG LOG END ------*/
	g_hash_table_foreach_remove(iomux_hash, removeall, NULL);
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,39,
			"out_function: void l7vs_iomux_removeall() "
			"return_value: TRUE");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * polling function for epoll.
 *
 * @param[in]	*timo	waittime;
 * @return	error	
 */
int
l7vs_iomux_poll(struct timeval *timo, int blocking)
{
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		if (!timo) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,40,
				"in_function: int l7vs_iomux_poll(struct timeval *timo, int blocking) "
				"timo=NULL, blocking = %d", blocking);
		}
		else {
			char time_str[DEBUG_STR_LEN] = {0};
			snprintf(time_str, DEBUG_STR_LEN,
				"timeval="
				"{tv_sec=%ld: "
				"tv_usec=%ld}"
				, timo->tv_sec
				, timo->tv_usec);
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,41,
				"in_function: int l7vs_iomux_poll(struct timeval *timo) "
				"timo=%s, blocking=%d",
				time_str, blocking);
		}
	}
	/*------ DEBUG LOG END ------*/

	int	fdnum;
	int	i;
	int	ret;
	struct l7vs_iomux	*iom;
	struct l7vs_conn	*conn;

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,42,
			"epoll_wait param: epfd=%d: maxevents=%d: timeout=0",
			eventpoll, maxevents);
	}
	/*------ DEBUG LOG END ------*/
	fdnum = epoll_wait(eventpoll, event_array, maxevents, blocking);
	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,43,
			"epoll_wait result: fdnum=%d",
			fdnum);
	}
	/*------ DEBUG LOG END ------*/
	for (i = 0; i < fdnum; ++i) {
		if (event_array[i].data.ptr) {
			iom = (struct l7vs_iomux *)event_array[i].data.ptr;
			/*-------- DEBUG LOG --------*/
			if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
				char iomux_str[DEBUG_STR_LEN] = {0};
				l7vs_iomux_c_str(iomux_str, iom);
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,44,
					"event_array[%d].data.ptr: %s",
					i, iomux_str);
			}
			/*------ DEBUG LOG END ------*/
			if (iom->status != iomux_create) {
				if (event_array[i].events & (EPOLLHUP | EPOLLERR)) {
					// disconnect when sending
					conn = (struct l7vs_conn *)iom->data;
					/*-------- DEBUG LOG --------*/
					if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
						char conn_str[DEBUG_STR_LEN] = {0};
						l7vs_conn_c_str(conn_str, conn);
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,45,
							"pointer_assign: conn=%s",
							conn_str);
					}
					/*------ DEBUG LOG END ------*/

					// destroy conn
					l7vs_conn_destroy(conn);

				} else if (event_array[i].events & (EPOLLIN | EPOLLOUT)) {
					if (iom->status != iomux_disabled) {
						if (iom->callback) {
							// if there is mismatch between status and events,
							// read event will regist after, (when send was finished) 
							// therefore, through callback
							if (!(event_array[i].events & EPOLLIN &&
							      (iomux_conn_sending == iom->status ||
							       iomux_conn_sending_busy == iom->status))) {
								// execute callback
								/*-------- DEBUG LOG --------*/
								if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
									char ex_iomux_str[DEBUG_STR_LEN] = {0};
									l7vs_iomux_c_str(ex_iomux_str, iom);
									LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,46,
										"iom->callback: %p: iom=%s",
										iom->callback, ex_iomux_str);
								}
								/*------ DEBUG LOG END ------*/
								ret = iom->callback(iom);
								/*-------- DEBUG LOG --------*/
								if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
									LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,47,
										"iom->callback result: ret=%d",
										ret);
								}
								/*------ DEBUG LOG END ------*/
	
							}
						} else {
							LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,17, "error / callback not bind");
						}
					}
				} else {
					LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,18, "error / unkown event:%x", event_array[i].events);
				}
			} else {
				if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,48, "debug / invalid status:iomux_create");
				}
			}
		} else {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSD_EVENT,19, "error / epolldata.data.ptr is null");
		}
	}

	/*-------- DEBUG LOG --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_L7VSD_EVENT)) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSD_EVENT,49,
			"out_function: int l7vs_iomux_poll(struct timeval *timo) "
			"return_value: 1");
	}
	/*------ DEBUG LOG END ------*/
	return 1;

}
