/*
famicom ROM cartridge utility - unagi
flash memory driver

Copyright (C) 2008  sato_tiff

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

flashmemory.c ηٹ
Υɤ򻲹͡žѤƥʤɤפʤȡ
Ƚ LGPL ŬѤ졢սΥɬפ롣
*/
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <windows.h>
#include "type.h"
#include "flashmemory.h"
/*
driver for Winbond W29C020, W49F002
*/
/*
JEDEC flash memory command
http://www.sst.com/downloads/software_driver/SST49LF002A.txt
*/
struct flash_task{
	long address, data;
};
enum{
	ADDRESS_0000 = 0,
	ADDRESS_2AAA = 0x2aaa,
	ADDRESS_5555 = 0x5555,
	FLASH_COMMAND_END
};
static const struct flash_task PRODUCTID_ENTRY[] = {
	{ADDRESS_5555, 0xaa},
	{ADDRESS_2AAA, 0x55},
	{ADDRESS_5555, 0x90},
	{FLASH_COMMAND_END, 0}
};
static const struct flash_task PRODUCTID_EXIT[] = {
	{ADDRESS_5555, 0xaa},
	{ADDRESS_2AAA, 0x55},
	{ADDRESS_5555, 0xf0},
	{FLASH_COMMAND_END, 0}
};
static const struct flash_task PROTECT_DISABLE[] = {
	{ADDRESS_5555, 0xaa},
	{ADDRESS_2AAA, 0x55},
	{ADDRESS_5555, 0xa0},
	{FLASH_COMMAND_END, 0}
};
static const struct flash_task PROTECT_ENABLE[] = {
	{ADDRESS_5555, 0xaa},
	{ADDRESS_2AAA, 0x55},
	{ADDRESS_5555, 0x80},
	{ADDRESS_5555, 0xaa},
	{ADDRESS_2AAA, 0x55},
	{ADDRESS_5555, 0x20},
	{FLASH_COMMAND_END, 0}
};
//boot lock lockout enable 򤤤ʤȥХڤؤ򤹤
//Ƭ0x100byte餤񤭴?
static const struct flash_task BOOTBLOCK_FIRST[] = {
	{ADDRESS_5555, 0xaa},
	{ADDRESS_2AAA, 0x55},
	{ADDRESS_5555, 0x80},
	{ADDRESS_5555, 0xaa},
	{ADDRESS_2AAA, 0x55},
	{ADDRESS_5555, 0x40},
	{ADDRESS_0000, 0},
	{FLASH_COMMAND_END, 0}
};
static const struct flash_task ERASE[] = {
	{ADDRESS_5555, 0xaa},
	{ADDRESS_2AAA, 0x55},
	{ADDRESS_5555, 0x80},
	{ADDRESS_5555, 0xaa},
	{ADDRESS_2AAA, 0x55},
	{ADDRESS_5555, 0x10},
	{FLASH_COMMAND_END, 0}
};

static void command_set(const struct flash_order *d, const struct flash_task *t)
{
	while(t->address != FLASH_COMMAND_END){
		long logical_address = 0;
		switch(t->address){
		case ADDRESS_0000: //bank ˤäƤǤʤ?
			logical_address = d->command_0000;
			break;
		case ADDRESS_2AAA:
			logical_address = d->command_2aaa;
			break;
		case ADDRESS_5555:
			logical_address = d->command_5555;
			break;
		default:
			assert(0); //unknown task address
		}
		d->flash_write(logical_address, t->data);
		t++;
	}
}

/*
---- product ID check ----
*/
static int productid_check(const struct flash_order *d, const struct flash_driver *f)
{
	u8 data[3];
	command_set(d, PRODUCTID_ENTRY);
	d->read(d->command_0000, 3, data);
	command_set(d, PRODUCTID_EXIT);
	if(f->id_manufacurer != data[0]){
		return NG;
	}
	if(f->id_device != data[1]){
		return NG;
	}
	return OK;
}

/*
---- toggle check ----
databit6
*/
const int CHECK_RETRY_MAX = 0x10000;
static int toggle_check(const struct flash_order *d, long address)
{
	u8 predata;
	int retry = 0;
	d->read(address, 1, &predata); //read DQ6
	predata &= 0x40;
	while(retry < CHECK_RETRY_MAX){
		u8 data;
		d->read(address, 1, &data); //read DQ6 again
		data &= 0x40;
		if(predata == data){
			return OK;
		}
		predata = data;
		retry++;
	}
	return NG;
}

/*
---- polling check ----
databit7
*/
static int polling_check(const struct flash_order *d, long address, u8 truedata)
{
	int retry = 0;
	
	truedata &= 0x80;
	while(retry < CHECK_RETRY_MAX){
		u8 data;
		d->read(address, 1, &data);
		data &= 0x80;
		if(truedata == data){
			return OK;
		}
		retry++;
	}
	return NG;
}

/*
---- erase ----
*/
static void flash_erase(const struct flash_order *d)
{
	command_set(d, ERASE);
	toggle_check(d, d->command_2aaa);
	Sleep(200); //Tec 0.2 sec
}

/*
---- program ----
*/
static int program_byte(const struct flash_order *d, long address, const u8 *data, long length)
{
	while(length != 0){
		if(*data != 0xff){
			command_set(d, PROTECT_DISABLE);
			d->flash_write(address, *data);
			if(toggle_check(d, address) == NG){
				if(DEBUG == 1){
					printf("%s NG\n", __FUNCTION__);
				}
				return NG;
			}
			if(0){
			u8 putdata;
			d->read(address, 1, &putdata);
			if(putdata != *data){
				printf("%s %06x retry\n", __FUNCTION__, (int) address);
				continue;
			}
			}
		}
		if((DEBUG == 1) && (address & 0x1f) == 0){
			printf("%s %06x\n", __FUNCTION__, (int) address);
			fflush(stdout);
		}
		address++;
		data++;
		length--;
	}
	return OK;
}

static int program_pagewrite(const struct flash_order *d, long address, const u8 *data, long length)
{
	const long toggle_address = address;
	command_set(d, PROTECT_DISABLE);
	while(length != 0){
		d->flash_write(address, *data);
		address++;
		data++;
		length--;
	}
	Sleep(15);
	int ret = toggle_check(d, toggle_address);
	if(0){
		data--;
		address -= 1;
		polling_check(d, address - 1, *data);
	}

	//command_set(d, PROTECT_ENABLE);
	//Sleep(15);
	return ret;
}

/*
---- block compare ----
*/
static void compare(const struct flash_order *d, long address, const u8 *data, long length)
{
	u8 *romdata, *r;
	int count = 0;
	romdata = malloc(length);
	d->read(address, length, romdata);
	r = romdata;
	while(length != 0){
		if(*r != *data){
			char safix = ' ';
			if((count & 7) == 7){
				safix = '\n';
			}
			count++;
			printf("%06x%c", (int)address, safix);
		}
		r++;
		data++;
		address++;
		length--;
	}
	free(romdata);
}

/*
ͭǥХɥ饤
*/
static void w49f002_init(const struct flash_order *d)
{
/*
byte program mode Ǥ 1->0 ˤ 0->1  erase Τߡ
äƽˤ erase ¹Ԥ
*/
	flash_erase(d);
}

static void w49f002_write(const struct flash_order *d)
{
	program_byte(d, d->address, d->data, d->length);
	compare(d, d->address, d->data, d->length);
}

static void w29c020_init(const struct flash_order *d)
{
/*
page write mode ǤϤȤˤʤ
*/
}

static void w29c020_write(const struct flash_order *d)
{
	const long pagesize = 0x80;
	int retry = 0;
	{
		long a = d->address;
		long i = d->length;
		const u8 *dd;
		u8 *cmp;

		dd = d->data;
		cmp = malloc(pagesize);
		while(i != 0){
			int result = program_pagewrite(d, a, dd, pagesize);
			if(result == NG){
				printf("%s: write error\n", __FUNCTION__);
				free(cmp);
				return;
			}
			d->read(a, pagesize, cmp);
			if(memcmp(cmp, dd, pagesize) == 0){
				a += pagesize;
				dd += pagesize;
				i -= pagesize;
			}else{
				retry++;
			}
		}
		free(cmp);
	}

	printf("write ok. retry %d\n", retry);
	compare(d, d->address, d->data, d->length);
	command_set(d, BOOTBLOCK_FIRST);
	Sleep(10);
}

/*
ǥХꥹ
*/
static const struct flash_driver DRIVER_W29C020 = {
	.name = "W29C020",
	.capacity = 0x40000,
	.id_manufacurer = 0xda,
	.id_device = 0x45,
	.productid_check = productid_check,
#if DEBUG==1
	.erase = flash_erase,
#endif
	.init = w29c020_init,
	.write = w29c020_write
};

static const struct flash_driver DRIVER_W29C040 = {
	.name = "W29C040",
	.capacity = 0x80000,
	.id_manufacurer = 0xda,
	.id_device = 0x46,
	.productid_check = productid_check,
#if DEBUG==1
	.erase = flash_erase,
#endif
	.init = w29c020_init,
	.write = w29c020_write
};

static const struct flash_driver DRIVER_W49F002 = {
	.name = "W49F002",
	.capacity = 0x40000,
	.id_manufacurer = 0xda,
	.id_device = 0xae,
	.productid_check = productid_check,
#if DEBUG==1
	.erase = flash_erase,
#endif
	.init = w49f002_init,
	.write = w49f002_write
};

static const struct flash_driver *DRIVER_LIST[] = {
	&DRIVER_W29C020, &DRIVER_W29C040, &DRIVER_W49F002,
	NULL
};

const struct flash_driver *flash_driver_get(const char *name)
{
	const struct flash_driver **d;
	d = DRIVER_LIST;
	while(*d != NULL){
		if(strcmp(name, (*d)->name) == 0){
			return *d;
		}
		d++;
	}
	return NULL;
}
