' table/KeyboardOperation.vb
'
' Copyright (c) 2008-2011, SystemBase Co.,Ltd.
' All rights reserved.
'
' Redistribution and use in source and binary forms, with or without
' modification, are permitted provided that the following conditions are met:
'
'    1. Redistributions of source code must retain the above copyright
'       notice, this list of conditions and the following disclaimer.
'    2. Redistributions in binary form must reproduce the above copyright
'       notice, this list of conditions and the following disclaimer in the
'       documentation and/or other materials provided with the distribution.
'
' THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
' IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
' ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
' LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
' CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
' SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
' INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
' CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
' ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
' POSSIBILITY OF SUCH DAMAGE.

Public Class CKeyboardOperation    

    Public Enum EBehavior
        NONE
        NEXT_FIELD
        NEXT_EDITABLE_FIELD
        NEXT_RECORD
        NEXT_CONTROL
    End Enum

    Public EnterBehavior As EBehavior = EBehavior.NEXT_RECORD
    Public TabBehavior As EBehavior = EBehavior.NEXT_FIELD
    Public StartEditKey As Keys = Keys.F2
    Public LeaveFromLastField As Boolean = True
    Public ClearByDelete As Boolean = True
    Public ClipboardPaste As Boolean = True
    Public ClipboardPasteAddRecord As Boolean = False
    Public ClipboardCopy As Boolean = True
    Public ClipboardCut As Boolean = True
    Public Undo As Boolean = True
    Public Redo As Boolean = True

    Public WriteOnly Property ClipboardEnabled() As Boolean
        Set(ByVal value As Boolean)
            Me.ClipboardCopy = value
            Me.ClipboardCut = value
            Me.ClipboardPaste = value
        End Set
    End Property

    Public WriteOnly Property UndoRedoEnabled() As Boolean
        Set(ByVal value As Boolean)
            Me.Undo = value
            Me.Redo = value
        End Set
    End Property

    Public Overridable Function IsInputKey(ByVal key As Keys, ByVal table As UTable, ByVal defaultValue As Boolean) As Boolean        
        Select Case key
            Case Keys.Shift Or Keys.Up, _
                 Keys.Shift Or Keys.Down, _
                 Keys.Shift Or Keys.Left, _
                 Keys.Shift Or Keys.Right
                If table.SelectRange IsNot Nothing Then
                    Return False
                End If
        End Select
        Return defaultValue
    End Function

    Public Overridable Function ProcessDialogKey(ByVal key As Keys, ByVal table As UTable) As Boolean
        If Me.processFieldSelected(key, table) Then
            Return True
        End If
        If Me.processStartEdit(key, table) Then
            Return True
        End If
        Return Me.processKey(key, table)
    End Function

    Public Overridable Sub LeaveEdit(ByVal direction As String, ByVal table As UTable)
        Select Case direction
            Case "UP"
                Me.prevRow(table)
            Case "DOWN"
                Me.nextRow(table)
            Case "LEFT"
                Me.prevCol(table)
            Case "RIGHT"
                Me.nextCol(table)
            Case "ENTER"
                Me.tabEnter(table, Me.EnterBehavior)
            Case "ENTER_PREV"
                Me.tabEnterPrev(table, Me.EnterBehavior)
            Case "TAB"
                Me.tabEnter(table, Me.TabBehavior)
            Case "TAB_PREV"
                Me.tabEnterPrev(table, Me.TabBehavior)
        End Select
    End Sub

    Protected Overridable Function processFieldSelected(ByVal key As Keys, ByVal table As UTable) As Boolean
        Dim handled As Boolean = False
        If key.Equals(Keys.Enter) And table.FocusField IsNot Nothing Then
            table.RaiseFieldSelected(table.FocusField, handled)
        End If
        Return handled
    End Function

    Protected Overridable Function processStartEdit(ByVal key As Keys, ByVal table As UTable) As Boolean
        If key.Equals(Me.StartEditKey) Then
            If table.StartEdit() IsNot Nothing Then
                Return True
            End If
        End If
        Return False
    End Function

    Protected Overridable Function processKey(ByVal key As Keys, ByVal table As UTable) As Boolean
        Select Case key
            Case Keys.Up, Keys.Down, Keys.Left, Keys.Right
                If table.FocusField Is Nothing Then
                    table.FocusField = table.Content.DefaultField
                Else
                    Select Case key
                        Case Keys.Up
                            Me.prevRow(table)
                        Case Keys.Down
                            Me.nextRow(table)
                        Case Keys.Left
                            Me.prevCol(table)
                        Case Keys.Right
                            Me.nextCol(table)
                    End Select
                End If
                Return True
            Case Keys.Home, Keys.End
                If table.FocusField IsNot Nothing Then
                    Select Case key
                        Case Keys.Home
                            Me.startCol(table)
                        Case Keys.End
                            Me.endCol(table)
                    End Select
                End If
                Return True
            Case Keys.PageDown
                Me.pageDown(table)
                Return True
            Case Keys.PageUp
                Me.pageUp(table)
                Return True
            Case Keys.Control Or Keys.Home
                Me.startContent(table)
                Return True
            Case Keys.Control Or Keys.End
                Me.endContent(table)
                Return True
            Case Keys.Enter
                Me.tabEnter(table, Me.EnterBehavior)
                Return True
            Case Keys.Enter Or Keys.Shift
                Me.tabEnterPrev(table, Me.EnterBehavior)
                Return True
            Case Keys.Tab
                Me.tabEnter(table, Me.TabBehavior)
                Return True
            Case Keys.Tab Or Keys.Shift
                Me.tabEnterPrev(table, Me.TabBehavior)
                Return True
            Case Keys.Delete
                If Me.ClearByDelete Then
                    If table.SelectRange IsNot Nothing Then
                        table.SelectRange.FieldsClear()
                    ElseIf table.FocusField IsNot Nothing Then
                        If table.FocusField.Editable.Equals(UTable.EAllow.ALLOW) Then
                            table.FocusField.Clear()
                        End If
                    End If
                    Return True
                End If
            Case Keys.Shift Or Keys.Up, _
                 Keys.Shift Or Keys.Down, _
                 Keys.Shift Or Keys.Left, _
                 Keys.Shift Or Keys.Right, _
                 Keys.Shift Or Keys.Home, _
                 Keys.Shift Or Keys.End, _
                 Keys.Shift Or Keys.PageUp, _
                 Keys.Shift Or Keys.PageDown, _
                 Keys.Control Or Keys.A, _
                 Keys.Control Or Keys.Shift Or Keys.Home, _
                 Keys.Control Or Keys.Shift Or Keys.End
                If table.FocusField IsNot Nothing AndAlso _
                   table.Setting.UserRangeSelectable Then
                    Select Case key
                        Case Keys.Shift Or Keys.Up
                            Me.rangeUp(table)
                            Return True
                        Case Keys.Shift Or Keys.Down
                            Me.rangeDown(table)
                            Return True
                        Case Keys.Shift Or Keys.Left
                            Me.rangeLeft(table)
                            Return True
                        Case Keys.Shift Or Keys.Right
                            Me.rangeRight(table)
                            Return True
                        Case Keys.Shift Or Keys.Home
                            Me.rangeStartCol(table)
                            Return True
                        Case Keys.Shift Or Keys.End
                            Me.rangeEndCol(table)
                            Return True
                        Case Keys.Shift Or Keys.PageUp
                            Me.rangePageUp(table)
                            Return True
                        Case Keys.Shift Or Keys.PageDown
                            Me.rangePageDown(table)
                            Return True
                        Case Keys.Control Or Keys.A
                            Me.rangeAll(table)
                            Return True
                    End Select
                End If
            Case Keys.Control Or Keys.V
                If Me.ClipboardPaste Then
                    table.ClipboardPaste(Me.ClipboardPasteAddRecord)
                    Return True
                End If
            Case Keys.Control Or Keys.C
                If Me.ClipboardCopy Then
                    table.ClipboardCopy()
                    Return True
                End If
            Case Keys.Control Or Keys.X
                If Me.ClipboardCut Then
                    table.ClipboardCopy()
                    If table.SelectRange IsNot Nothing Then
                        table.SelectRange.FieldsClear()
                    ElseIf table.FocusField IsNot Nothing Then
                        If table.FocusField.Editable = UTable.EAllow.ALLOW Then
                            table.FocusField.Clear()
                        End If
                    End If
                    Return True
                End If
            Case Keys.Control Or Keys.Z
                If Me.Undo Then
                    table.Undo()
                    Return True
                End If
            Case Keys.Control Or Keys.Y
                If Me.Redo Then
                    table.Redo()
                    Return True
                End If
        End Select
        Return False
    End Function

    Protected Overridable Sub nextRow(ByVal table As UTable)
        table.FocusGrid.Row = table.FocusField.Desc.Layout.BottomRow
        Dim field As UTable.CField = Me.nextRowField(table.FocusGrid, table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub prevRow(ByVal table As UTable)
        table.FocusGrid.Row = table.FocusField.Desc.Layout.Row
        Dim field As UTable.CField = Me.prevRowField(table.FocusGrid, table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub nextCol(ByVal table As UTable)
        table.FocusGrid.Col = table.FocusField.Desc.Layout.RightCol
        Dim field As UTable.CField = Me.nextColField(table.FocusGrid, table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub prevCol(ByVal table As UTable)
        table.FocusGrid.Col = table.FocusField.Desc.Layout.Col
        Dim field As UTable.CField = Me.prevColField(table.FocusGrid, table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub startCol(ByVal table As UTable)
        table.FocusGrid.Col = table.FocusField.Desc.Layout.Col
        Dim field As UTable.CField = Me.nextColField(New UTable.CGrid.CPoint(table.FocusGrid.Row, -1), table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub endCol(ByVal table As UTable)
        table.FocusGrid.Col = table.FocusField.Desc.Layout.Col
        Dim field As UTable.CField = Me.prevColField(New UTable.CGrid.CPoint(table.FocusGrid.Row, table.Cols.Count), table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub pageDown(ByVal table As UTable)
        If table.FocusField IsNot Nothing Then
            If table.FocusField.TopLevelContent Is table.Content Then
                table.SetVScrollValue(table.VScrollBar.Value + table.VScrollBar.LargeChange)
            End If
        End If
    End Sub

    Protected Overridable Sub pageUp(ByVal table As UTable)
        If table.FocusField IsNot Nothing Then
            If table.FocusField.TopLevelContent Is table.Content Then
                table.SetVScrollValue(table.VScrollBar.Value - table.VScrollBar.LargeChange)
            End If
        End If
    End Sub

    Protected Overridable Sub startContent(ByVal table As UTable)
        Dim r As UTable.CRecord = table.FocusField.TopLevelContent.LayoutCache.TopRecord
        If r IsNot Nothing Then
            Dim f As UTable.CField = r.DefaultField
            If f IsNot Nothing Then
                table.FocusField = f
            End If
        End If
    End Sub

    Protected Overridable Sub endContent(ByVal table As UTable)
        Dim r As UTable.CRecord = table.FocusField.TopLevelContent.LayoutCache.LastRecord
        If r IsNot Nothing Then
            Dim f As UTable.CField = r.LastField
            If f IsNot Nothing Then
                table.FocusField = f
            End If
        End If
    End Sub

    Protected Sub tabEnter(ByVal table As UTable, ByVal behavior As EBehavior)
        If table.FocusField Is Nothing Then
            table.FocusField = table.Content.DefaultField
        End If
        If table.FocusField Is Nothing Then
            Me.doNextControl(table)
        Else
            Select Case behavior
                Case EBehavior.NEXT_FIELD
                    Me.doNextField(table, False)
                Case EBehavior.NEXT_EDITABLE_FIELD
                    Me.doNextField(table, True)
                Case EBehavior.NEXT_RECORD
                    Me.nextRecord(table)
                Case EBehavior.NEXT_CONTROL
                    Me.doNextControl(table)
            End Select
        End If
    End Sub

    Protected Sub tabEnterPrev(ByVal table As UTable, ByVal behavior As EBehavior)
        If table.FocusField Is Nothing Then
            table.FocusField = table.Content.DefaultField
        End If
        If table.FocusField Is Nothing Then
            Me.doPrevControl(table)
        Else
            Select Case behavior
                Case EBehavior.NEXT_FIELD
                    Me.doPrevField(table, False)
                Case EBehavior.NEXT_EDITABLE_FIELD
                    Me.doPrevField(table, True)
                Case EBehavior.NEXT_RECORD
                    Me.prevRecord(table)
                Case EBehavior.NEXT_CONTROL
                    Me.doPrevControl(table)
            End Select
        End If
    End Sub

    Protected Sub doNextField(ByVal table As UTable, ByVal editableOnly As Boolean)
        Dim handled As Boolean = False
        table.raiseFocusNextField(table.FocusField, True, handled)
        If Not handled Then
            Me.[nextField](table, editableOnly)
        End If
    End Sub

    Protected Sub doPrevField(ByVal table As UTable, ByVal editableOnly As Boolean)
        Dim handled As Boolean = False
        table.raiseFocusNextField(table.FocusField, False, handled)
        If Not handled Then
            Me.prevField(table, editableOnly)
        End If
    End Sub

    Protected Overridable Sub nextField(ByVal table As UTable, ByVal editableOnly As Boolean)
        Dim f As UTable.CField
        If editableOnly Then
            f = table.FocusField.NextEditableField
        Else
            f = table.FocusField.NextField
        End If
        If f Is Nothing Then
            If table.FocusField.TopLevelContent Is table.HeaderContent Then
                If table.Content.Enabled Then
                    f = table.Content.DefaultField
                End If
            ElseIf table.FocusField.TopLevelContent Is table.Content Then
                If table.FooterContent.Enabled Then
                    f = table.FooterContent.DefaultField
                End If
            End If
        End If
        If f IsNot Nothing Then
            table.FocusField = f
        ElseIf LeaveFromLastField Then
            Me.doNextControl(table)
        End If
    End Sub

    Protected Overridable Sub prevField(ByVal table As UTable, ByVal editableOnly As Boolean)
        Dim f As UTable.CField
        If editableOnly Then
            f = table.FocusField.PrevEditableField
        Else
            f = table.FocusField.PrevField
        End If
        If f Is Nothing Then
            If table.FocusField.TopLevelContent Is table.Content Then
                If table.HeaderContent.Enabled Then
                    f = table.HeaderContent.LastField
                End If
            ElseIf table.FocusField.TopLevelContent Is table.FooterContent Then
                If table.Content.Enabled Then
                    f = table.Content.LastField
                End If
            End If
        End If
        If f IsNot Nothing Then
            table.FocusField = f
        ElseIf LeaveFromLastField Then
            Me.doPrevControl(table)
        End If
    End Sub

    Protected Overridable Sub nextRecord(ByVal table As UTable)
        Dim r As UTable.CRecord = table.FocusRecord.LayoutCache.NextRecord
        If r IsNot Nothing Then
            If r.Fields.ContainsKey(table.FocusField.Key) Then
                table.FocusField = r.Fields(table.FocusField.Key)
            Else
                table.FocusField = r.DefaultField
            End If
        End If
    End Sub

    Protected Overridable Sub prevRecord(ByVal table As UTable)
        Dim r As UTable.CRecord = table.FocusRecord.LayoutCache.PrevRecord
        If r IsNot Nothing Then
            If r.Fields.ContainsKey(table.FocusField.Key) Then
                table.FocusField = r.Fields(table.FocusField.Key)
            Else
                table.FocusField = r.DefaultField
            End If
        End If
    End Sub

    Protected Sub doNextControl(ByVal table As UTable)
        Dim handled As Boolean = False
        If table.FocusField IsNot Nothing Then
            table.raiseFocusNextControl(table.FocusField, True, handled)
        End If
        If Not handled Then
            Me.nextControl(table)
        End If
    End Sub

    Protected Sub doPrevControl(ByVal table As UTable)
        Dim handled As Boolean = False
        If table.FocusField IsNot Nothing Then
            table.raiseFocusNextControl(table.FocusField, False, handled)
        End If
        If Not handled Then
            Me.prevControl(table)
        End If
    End Sub

    Protected Overridable Sub nextControl(ByVal table As UTable)
        If table.TopLevelControl IsNot Nothing Then
            table.TopLevelControl.SelectNextControl(table, True, True, True, True)
        End If
    End Sub

    Protected Overridable Sub prevControl(ByVal table As UTable)
        If table.TopLevelControl IsNot Nothing Then
            table.TopLevelControl.SelectNextControl(table, False, True, True, True)
        End If
    End Sub

    Protected Function findFocusableField(ByVal record As UTable.CRecord, ByVal point As UTable.CGrid.CPoint) As UTable.CField
        If (point.Row >= 0 And point.Row < record.Rows.Count And _
            point.Col >= 0 And point.Col < record.Table.Cols.Count) AndAlso _
           (record.LayoutCache.Visible And _
            record.Rows(point.Row).Visible And _
            record.Table.Cols(point.Col).Visible) Then
            Dim field As UTable.CField = record.FindField(point)
            If field IsNot Nothing AndAlso field.Focusable Then
                Return field
            End If
        End If
        Return Nothing
    End Function

    Protected Function nextColField(ByVal point As UTable.CGrid.CPoint, ByVal record As UTable.CRecord) As UTable.CField
        Dim p As New UTable.CGrid.CPoint(point.Row, point.Col + 1)
        Do While p.Col < record.Table.Cols.Count
            Dim field As UTable.CField = Me.findFocusableField(record, p)
            If field IsNot Nothing Then
                Return field
            End If
            p.Col += 1
        Loop
        Return Nothing
    End Function

    Protected Function prevColField(ByVal point As UTable.CGrid.CPoint, ByVal record As UTable.CRecord) As UTable.CField
        Dim p As New UTable.CGrid.CPoint(point.Row, point.Col - 1)
        Do While p.Col >= 0
            Dim field As UTable.CField = Me.findFocusableField(record, p)
            If field IsNot Nothing Then
                Return field
            End If
            p.Col -= 1
        Loop
        Return Nothing
    End Function

    Protected Function nextRowField(ByVal point As UTable.CGrid.CPoint, ByVal record As UTable.CRecord) As UTable.CField
        Dim p As New UTable.CGrid.CPoint(point.Row + 1, point.Col)
        Do While p.Row < record.Rows.Count
            Dim field As UTable.CField = Me.findFocusableField(record, p)
            If field IsNot Nothing Then
                Return field
            End If
            field = Me.nextColField(p, record)
            If field IsNot Nothing Then
                Return field
            End If
            field = Me.prevColField(p, record)
            If field IsNot Nothing Then
                Return field
            End If
            p.Row += 1
        Loop
        If record.LayoutCache.NextRecord IsNot Nothing Then
            Dim r As UTable.CRecord = record.LayoutCache.NextRecord
            Return Me.nextRowField(New UTable.CGrid.CPoint(-1, point.Col), r)
        Else
            Return Nothing
        End If
    End Function

    Protected Function prevRowField(ByVal point As UTable.CGrid.CPoint, ByVal record As UTable.CRecord) As UTable.CField
        Dim p As New UTable.CGrid.CPoint(point.Row - 1, point.Col)
        Do While p.Row >= 0
            Dim field As UTable.CField = Me.findFocusableField(record, p)
            If field IsNot Nothing Then
                Return field
            End If
            field = Me.nextColField(p, record)
            If field IsNot Nothing Then
                Return field
            End If
            field = Me.prevColField(p, record)
            If field IsNot Nothing Then
                Return field
            End If
            p.Row -= 1
        Loop
        If record.LayoutCache.PrevRecord IsNot Nothing Then
            Dim r As UTable.CRecord = record.LayoutCache.PrevRecord
            Return Me.prevRowField(New UTable.CGrid.CPoint(r.Rows.Count, point.Col), r)
        Else
            Return Nothing
        End If
    End Function

    Protected Overridable Sub rangeUp(ByVal table As UTable)
        Using table.RenderBlock
            If table.SelectRange Is Nothing Then
                table.SelectRange = table.CreateRange(table.FocusField)
            End If
            If table.SelectRange IsNot Nothing Then
                Dim r As Integer = table.SelectRange.EndRow
                If r > 0 Then
                    r -= 1
                End If
                With table.SelectRange
                    table.SelectRange = table.CreateRange(.TopContent, .StartRow, .StartCol, r, .EndCol)
                    table.SelectRange.ScrollToEnd()
                End With
            End If
        End Using
    End Sub

    Protected Overridable Sub rangeDown(ByVal table As UTable)
        Using table.RenderBlock
            If table.SelectRange Is Nothing Then
                table.SelectRange = table.CreateRange(table.FocusField)
            End If
            If table.SelectRange IsNot Nothing Then
                Dim r As Integer = table.SelectRange.EndRow
                If r < table.SelectRange.TopContent.LayoutCache.SelectableRows.Count - 1 Then
                    r += 1
                End If
                With table.SelectRange
                    table.SelectRange = table.CreateRange(.TopContent, .StartRow, .StartCol, r, .EndCol)
                    table.SelectRange.ScrollToEnd()
                End With
            End If
        End Using
    End Sub

    Protected Overridable Sub rangeLeft(ByVal table As UTable)
        Using table.RenderBlock
            If table.SelectRange Is Nothing Then
                table.SelectRange = table.CreateRange(table.FocusField)
            End If
            If table.SelectRange IsNot Nothing Then
                Dim c As Integer = table.SelectRange.EndCol
                If c > 0 Then
                    c -= 1
                End If
                With table.SelectRange
                    table.SelectRange = table.CreateRange(.TopContent, .StartRow, .StartCol, .EndRow, c)
                    table.SelectRange.ScrollToEnd()
                End With
            End If
        End Using
    End Sub

    Protected Overridable Sub rangeRight(ByVal table As UTable)
        Using table.RenderBlock
            If table.SelectRange Is Nothing Then
                table.SelectRange = table.CreateRange(table.FocusField)
            End If
            If table.SelectRange IsNot Nothing Then
                Dim c As Integer = table.SelectRange.EndCol
                If c < table.SelectRange.TopContent.Table.LayoutCache.SelectableCols.Count - 1 Then
                    c += 1
                End If
                With table.SelectRange
                    table.SelectRange = table.CreateRange(.TopContent, .StartRow, .StartCol, .EndRow, c)
                    table.SelectRange.ScrollToEnd()
                End With
            End If
        End Using
    End Sub

    Protected Overridable Sub rangeStartCol(ByVal table As UTable)
        Using table.RenderBlock
            If table.SelectRange Is Nothing Then
                table.SelectRange = table.CreateRange(table.FocusField)
            End If
            If table.SelectRange IsNot Nothing Then
                Dim c As Integer = 0
                With table.SelectRange
                    table.SelectRange = table.CreateRange(.TopContent, .StartRow, .StartCol, .EndRow, c)
                    table.SelectRange.ScrollToEnd()
                End With
            End If
        End Using
    End Sub

    Protected Overridable Sub rangeEndCol(ByVal table As UTable)
        Using table.RenderBlock
            If table.SelectRange Is Nothing Then
                table.SelectRange = table.CreateRange(table.FocusField)
            End If
            If table.SelectRange IsNot Nothing Then
                Dim c As Integer = table.SelectRange.TopContent.Table.LayoutCache.SelectableCols.Count - 1
                With table.SelectRange
                    table.SelectRange = table.CreateRange(.TopContent, .StartRow, .StartCol, .EndRow, c)
                    table.SelectRange.ScrollToEnd()
                End With
            End If
        End Using
    End Sub

    Protected Overridable Sub rangePageUp(ByVal table As UTable)
        Using table.RenderBlock
            If table.FocusField.TopLevelContent Is table.Content Then
                If table.SelectRange Is Nothing Then
                    table.SelectRange = table.CreateRange(table.FocusField)
                End If
                If table.SelectRange IsNot Nothing Then
                    Dim y As Integer = table.SelectRange.EndRect.Top - table.VScrollBar.LargeChange
                    Dim t As Integer = table.SelectRange.TopRow
                    Do While t > 0 AndAlso y < table.SelectRange.TopContent.LayoutCache.SelectableRowsPos(t)
                        t -= 1
                    Loop
                    With table.SelectRange
                        table.SelectRange = table.CreateRange(.TopContent, .StartRow, .StartCol, t, .EndCol)
                        table.SelectRange.ScrollToEnd()
                    End With
                End If
            End If
        End Using
    End Sub

    Protected Overridable Sub rangePageDown(ByVal table As UTable)
        Using table.RenderBlock
            If table.FocusField.TopLevelContent Is table.Content Then
                If table.SelectRange Is Nothing Then
                    table.SelectRange = table.CreateRange(table.FocusField)
                End If
                If table.SelectRange IsNot Nothing Then
                    Dim y As Integer = table.SelectRange.EndRect.Bottom + table.VScrollBar.LargeChange
                    Dim t As Integer = table.SelectRange.TopRow
                    Do While t < table.SelectRange.TopContent.LayoutCache.SelectableRows.Count - 1 AndAlso _
                             y > table.SelectRange.TopContent.LayoutCache.SelectableRowsPos(t)
                        t += 1
                    Loop
                    With table.SelectRange
                        table.SelectRange = table.CreateRange(.TopContent, .StartRow, .StartCol, t, .EndCol)
                        table.SelectRange.ScrollToEnd()
                    End With
                End If
            End If
        End Using
    End Sub

    Protected Overridable Sub rangeAll(ByVal table As UTable)
        Using table.RenderBlock
            With table
                Dim topContent As UTable.CContent = table.FocusField.TopLevelContent
                .SelectRange = .CreateRange(topContent)
            End With
        End Using
    End Sub

End Class