package jp.sourceforge.masasa.xlsXmlTemplate;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import jp.sourceforge.masasa.xlsXmlTemplate.ExcelXMLConstants.Dom;
import jp.sourceforge.masasa.xlsXmlTemplate.ExcelXMLConstants.SpreadSheet;
import jp.sourceforge.masasa.xlsXmlTemplate.ExcelXMLConstants.Template;

import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

/**
 * スプレッド形式の {@code XML} の {@code DOM} に関するユーティリティクラスです.</br>
 *
 * @see org.w3c.dom.Document
 * @see org.w3c.dom.Element
 * @see org.w3c.dom.Node
 * @see org.w3c.dom.NodeList
 * @see org.w3c.dom.Text
 * @author masasa
 */
public final class ExcelDOMUtil {

    /**
     * デフォルトコンストラクタ.
     */
    private ExcelDOMUtil() {
        // do nothing
    }

    /**
     * {@link Element} からテキストを取得します.
     *
     * @param element
     *            対象エレメント.
     * @return エレメント内の有効テキスト.
     */
    public static String getText(Element element) {
        StringBuffer sb = new StringBuffer();
        NodeList list = element.getChildNodes();
        Node node;
        for (int i = 0; i < list.getLength(); i++) {
            node = list.item(i);
            if (node instanceof Text) sb.append(((Text) node).getNodeValue());
        }
        return sb.toString();
    }

    /**
     * {@link Element}から<code>elementName</code>に該当する{@link Element}を取得します.
     *
     * @param parent
     *            親エレメント.
     * @param elementName
     *            取得対象エレメント.
     * @return 子エレメント配列.
     */
    public static Element[] getElements(Element parent, String elementName) {
        NodeList list = parent.getElementsByTagName(elementName);
        Element[] elements = new Element[list.getLength()];
        for (int i = 0; i < list.getLength(); i++) {
            elements[i] = (Element) list.item(i);
        }
        return elements;
    }

    /**
     * {@link Element}から<code>elementName</code>に該当する{@link Element}を取得します.
     *
     * @param parent
     *            親エレメント.
     * @param elementName
     *            取得対象エレメント.
     * @return 子エレメント.
     */
    public static Element getElement(Element parent, String elementName) {
        return getElement(parent, new String[] { elementName });
    }

    /**
     * {@link Element}から<code>elementName</code>に該当する{@link Element}を取得します.
     *
     * @param parent
     *            親エレメント.
     * @param elementNames
     *            取得対象エレメント.
     * @return 子エレメント.
     */
    public static Element getElement(Element parent, String[] elementNames) {
        NodeList list;
        for (int i = 0; i < elementNames.length; i++) {
            list = parent.getElementsByTagName(elementNames[i]);
            if (list.getLength() != 0) parent = (Element) list.item(0);
            else return null;
        }
        return parent;
    }

    /**
     * {@link Element}に文字列をセットします.
     *
     * @param parent
     *            対象エレメント.
     * @param text
     *            文字列.
     */
    public static void setText(Element parent, String text) {
        Document doc = parent.getOwnerDocument();
        NodeList list = parent.getChildNodes();
        for (int i = 0; i < list.getLength(); i++) {
            parent.removeChild(list.item(i));
        }
        if (!StringUtils.isEmpty(text)) {
            String data = text;
            data = data.replaceAll("\r\n", "&#10;"); //$NON-NLS-1$ //$NON-NLS-2$
            data = data.replaceAll("\r", "&#10;"); //$NON-NLS-1$ //$NON-NLS-2$
            data = data.replaceAll("\n", "&#10;"); //$NON-NLS-1$ //$NON-NLS-2$
            parent.appendChild(doc.createTextNode(data));
        }
    }

    /**
     * {@link Document}を{@link OutputStream}に書き出します.
     *
     * @param doc
     *            対象ドキュメント.
     * @param out
     *            ストリーム.
     * @throws IOException
     *             ストリーム操作時の例外.
     * @throws TransformerException
     *             変換時の例外.
     */
    public static void writeXML(Document doc, OutputStream out) throws TransformerException, IOException {
        getByteXML(doc, out);
    }

    /**
     * {@link Document}を{@link OutputStream}に書き出し、バイト配列を返却します.
     *
     * @param doc
     *            対象ドキュメント.
     * @param out
     *            ストリーム.
     * @return 書き出しバイト配列.
     * @throws IOException
     *             ストリーム操作時の例外.
     * @throws TransformerException
     *             変換時の例外.
     */
    public static byte[] getByteXML(Document doc, OutputStream out) throws TransformerException, IOException {
        TransformerFactory tff = TransformerFactory.newInstance();
        Transformer tf = tff.newTransformer();
        DOMSource src = new DOMSource();
        src.setNode(doc);
        StreamResult target = new StreamResult();
        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        target.setOutputStream(byteOut);
        tf.transform(src, target);
        target.getOutputStream().close();
        String xml = new String(byteOut.toByteArray(), ExcelXMLConstants.XML_ENCODE);
        xml = xml.replaceAll("&amp;#10;", "&#10;"); //$NON-NLS-1$ //$NON-NLS-2$
        return xml.getBytes(ExcelXMLConstants.XML_ENCODE);
    }

    /**
     * 行を追加します.
     *
     * @param table
     *            テーブル.
     * @param row
     *            行.
     * @param count
     *            カウント.
     */
    public static void incrementRowCount(Element table, Element row, int count) {
        boolean flag = false;
        Element[] rows = getElements(table, Dom.ROW);
        Element[] aelement = rows;
        int i = 0;
        Element element;
        for (int j = aelement.length; i < j; i++) {
            element = aelement[i];
            if (element == row) {
                flag = true;
            } else if (flag && element.hasAttribute(SpreadSheet.INDEX)) {
                element.setAttribute(SpreadSheet.INDEX,
                        String.valueOf(Integer.parseInt(element.getAttribute(SpreadSheet.INDEX)) + count));
            }
        }
        table.setAttribute(SpreadSheet.EXPANDED_ROW_COUNT,
                String.valueOf(Integer.parseInt(table.getAttribute(SpreadSheet.EXPANDED_ROW_COUNT)) + count));
        Element[] aelement1 = getElements((Element) table.getParentNode(), Dom.DATA_VALIDATION);
        int k = 0;
        Element dataValidation;
        Element range;
        Element value;
        for (int l = aelement1.length; k < l; k++) {
            dataValidation = aelement1[k];
            range = getElements(dataValidation, Dom.RANGE)[0];
            processRange(range, getRowIndex(rows, row), count);
            value = getElements(dataValidation, Dom.VALUE)[0];
            processRangeValue(value, count);
        }

        aelement1 = getElements((Element) table.getParentNode(), Dom.NAMES);
        k = 0;
        Element names;
        Element[] aelement2;
        Element namedRange;
        for (int i1 = aelement1.length; k < i1; k++) {
            names = aelement1[k];
            aelement2 = getElements(names, Dom.NAMED_RANGE);
            int j1 = 0;
            for (int k1 = aelement2.length; j1 < k1; j1++) {
                namedRange = aelement2[j1];
                if ("Print_Area".equals(namedRange.getAttribute(SpreadSheet.NAME))) {
                    incrementPrintArea(namedRange, count, 0);
                }
            }
        }
    }

    /**
     * 行インデックスを返却します.
     *
     * @param rows
     *            走査行.
     * @param rowElement
     *            検索行.
     * @return インデックス.
     */
    private static int getRowIndex(Element[] rows, Element rowElement) {
        int index = 0;
        Element[] aelement = rows;
        int i = 0;
        Element row;
        for (int j = aelement.length; i < j; i++) {
            row = aelement[i];
            index++;
            if (row.hasAttribute(SpreadSheet.INDEX)) index = Integer.parseInt(row.getAttribute(SpreadSheet.INDEX));
            if (row == rowElement) break;
        }
        return index;
    }

    /**
     * @param namedRange
     * @param rowCount
     * @param columnCount
     */
    public static void incrementPrintArea(Element namedRange, int rowCount, int columnCount) {
        StringBuilder sb = new StringBuilder();
        String[] splitted = namedRange.getAttribute(SpreadSheet.REFERS_TO).split("!");
        sb.append(splitted[0]).append("!");
        String[] xyValues = splitted[1].split(":");
        String[] dim;
        String newValue;
        for (int i = 0; i < xyValues.length; i++) {
            if (i != 0) {
                sb.append(":");
                dim = xyValues[i].split("C");
                int row = Integer.parseInt(dim[0].substring(1)) + rowCount;
                int column = Integer.parseInt(dim[1]) + columnCount;
                newValue = (new StringBuilder("R")).append(row).append("C").append(column).toString();
                sb.append(newValue);
            } else {
                sb.append(xyValues[i]);
            }
        }

        namedRange.setAttribute(SpreadSheet.REFERS_TO, sb.toString());
    }

    /**
     * @param rangeElement
     * @param insertRow
     * @param count
     */
    private static void processRange(Element rangeElement, int insertRow, int count) {
        StringBuilder sb = new StringBuilder();
        String[] as = getText(rangeElement).split(",");
        int j = 0;
        String rangeValue;
        StringBuilder builder;
        String[] xyValues;
        String[] dim;
        for (int k = as.length; j < k; j++) {
            rangeValue = as[j];
            if (rangeValue.length() != 0) {
                if (sb.length() > 0) {
                    sb.append(",");
                }
                if (rangeValue.indexOf(':') == -1) {
                    builder = new StringBuilder();
                    builder.append(rangeValue);
                    builder.append(":");
                    builder.append(rangeValue);
                    rangeValue = builder.toString();
                    builder = null;
                }
                xyValues = rangeValue.split(":");
                for (int i = 0; i < xyValues.length; i++) {
                    if (i != 0) {
                        sb.append(":");
                    }
                    dim = xyValues[i].split("C");
                    int row = Integer.parseInt(dim[0].substring(1));
                    if (i == 0 && row > insertRow - count) {
                        row += count;
                    } else if (i == 1 && row >= insertRow - count) {
                        row += count;
                    }
                    sb.append(new StringBuilder("R").append(row).append("C").append(dim[1]).toString());
                }
            }
        }
        setText(rangeElement, sb.toString());
    }

    /**
     * @param rangeElement
     * @param count
     */
    private static void processRangeValue(Element rangeElement, int count) {
        StringBuilder sb = new StringBuilder();
        String[] as = getText(rangeElement).split(",");
        int j = 0;
        String rangeValue;
        String[] xyValues;
        String[] dim;
        for (int k = as.length; j < k; j++) {
            rangeValue = as[j];
            if (rangeValue.length() != 0) {
                if (sb.length() > 0) {
                    sb.append(",");
                }
                xyValues = rangeValue.split(":");
                for (int i = 0; i < xyValues.length; i++) {
                    if (i != 0) {
                        sb.append(":");
                    }
                    dim = xyValues[i].split("C");
                    if (!dim[0].substring(1).matches("[\\d]*")) {
                        sb.append(rangeValue);
                    } else {
                        int row = Integer.parseInt(dim[0].substring(1)) + count;
                        sb.append(new StringBuilder("R").append(row).append("C").append(dim[1]).toString());
                    }
                }
            }
        }
        setText(rangeElement, sb.toString());
    }

    /**
     * @param row
     */
    public static void clearRowData(Element row) {
        Element[] aelement = getElements(row, Dom.CELL);
        int i = 0;
        Element column;
        Element[] aelement1;
        Element data;
        String text;
        for (int j = aelement.length; i < j; i++) {
            column = aelement[i];
            aelement1 = getElements(column, Dom.DATA);
            int k = 0;
            for (int l = aelement1.length; k < l; k++) {
                data = aelement1[k];
                text = getText(data);
                if (!text.startsWith(Template.VLOOP)) {
                    setText(data, "");
                }
            }

        }

    }

    /**
     * @param rowDatas
     * @param target
     * @return
     */
    @SuppressWarnings("unchecked")
    public static int getVerticalLoopMaxLength(List rowDatas, String target) {
        int count = 0;
        Object tableData;
        Map map;
        for (Iterator iterator = rowDatas.iterator(); iterator.hasNext();) {
            tableData = iterator.next();
            map = (Map) ReflectionUtil.getProperty(tableData, target);
            if (map != null && count < map.size()) {
                count = map.size();
            }
        }
        return count;
    }

    /**
     * @param cell
     * @param text
     */
    public static void setTextToCell(Element cell, String text) {
        Element[] datas = getElements(cell, Dom.DATA);
        if (text.indexOf('\n') >= 0 || text.indexOf('\r') >= 0) {
            String styleId = cell.getAttribute(SpreadSheet.STYLE_ID);
            if (styleId != null && styleId.length() != 0) {
                Element workbook = (Element) cell.getParentNode().getParentNode().getParentNode().getParentNode();
                Element styles = getElement(workbook, Dom.STYLES);
                Element[] aelement = getElements(styles, Dom.STYLE);
                int i = 0;
                Element style;
                for (int j = aelement.length; i < j; i++) {
                    style = aelement[i];
                    if (!style.getAttribute(SpreadSheet.ID).equals(styleId)) {
                        continue;
                    }
                    getElement(style, Dom.ALIGNMENT).setAttribute(SpreadSheet.WRAP_TEXT, "1");
                    break;
                }
            }
        }
        text = text.trim();
        if (datas.length == 0) {
            Element dataNode = cell.getOwnerDocument().createElement(Dom.DATA);
            dataNode.setAttribute(SpreadSheet.TYPE, Dom.STRING);
            setText(dataNode, text);
            cell.appendChild(dataNode);
            // System.out.println("××× " + text + " : " + dataNode.getAttribute(SpreadSheet.TYPE));
        } else {
            if (StringUtils.isEmpty(text)) {
                datas[0].setAttribute(SpreadSheet.TYPE, Dom.STRING);
            } else if (!isNumber(text)) {
                datas[0].setAttribute(SpreadSheet.TYPE, Dom.STRING);
            }
            setText(datas[0], text);
            // System.out.println("○○○ " + text + " : " + datas[0].getAttribute(SpreadSheet.TYPE));
        }
    }

    /**
     * @param beginRow
     * @return
     */
    public static Element getHeaderRow(Element beginRow) {
        Element table = (Element) beginRow.getParentNode();
        Element[] rows = getElements(table, Dom.ROW);
        for (int i = 0; i < rows.length; i++) {
            if (rows[i] == beginRow) return rows[i - 1];
        }
        return null;
    }

    /**
     * @param text
     * @return
     */
    private static boolean isNumber(String text) {
        return text == null || text.trim().length() == 0 ? false : text.trim()
                .replaceAll(",", "").matches("^[-]?[0-9]*[.]?[0-9]*"); //$NON-NLS-1$
    }
}
