//
//  PowerManager.h
//  PowerManager
//
//  Created by nya on 05/06/21.
//  Copyright 2005 __MyCompanyName__. All rights reserved.
//


/**
	@file
	@brief
	自動起動，終了，スリープ，スリープ復帰<br>
	管理者パスワードはキーチェーンで保持
	
	@note
	依存関係：<br>
		Cocoa.framework<br>
		IOKit.framework<br>
		Security.framework<br>
		SystemConfigure.framework
*/


#import <Cocoa/Cocoa.h>


/**
	@name システムのスリープに関する通知
*/
//@{
///	システムがスリープする前に呼ばれる通知
#define	PMSystemWillSleepNotification		@"PMSystemWillSleepNotificationName"
///	システムがスリープから復帰した時に呼ばれる通知
#define	PMSystemDidAwakeNotification		@"PMSystemDidAwakeNotificationName"
//@}


/**
	@brief	パワーマネジメントイベントの種類
*/
typedef enum {
	/// 登録
	//@{
	PM_BOOT					= 0x01,			///< 起動
	PM_SHUTDOWN				= 0x02,			///< 終了
	PM_SLEEP				= 0x04,			///< スリープ
	PM_WAKE					= 0x08,			///< 復帰
	PM_BOOTWAKE				= 0x09,			///< 起動または復帰
	//@}
	
	/// キャンセル
	//@{
	PM_CANCEL_BOOT			= 0x81,			///< 起動
	PM_CANCEL_SHUTDOWN		= 0x82,			///< 終了
	PM_CANCEL_SLEEP			= 0x84,			///< スリープ
	PM_CANCEL_WAKE			= 0x88,			///< 復帰
	PM_CANCEL_BOOTWAKE		= 0x89,			///< 起動または復帰
	//@}
	
	PM_CANCEL_MASK			= 0x80
} PMKind;

/// エラー
typedef long		PMError;

/**
	@brief	エラー
*/
enum {
	PM_NOERROR					= 0,

	PM_PASSWORD_INVALIED_ERR	= 1,				///< 管理者パスワードが違う
	PM_ALREADY_ERR				= 0xFF02,			///< 既にある
	PM_NOTFOUND_ERR				= 0xFF03,			///< 見つからない
	PM_PASSWORD_YET_ERR			= 0xFF01,			///< まだない
	
	PM_PASS_NOTFOUND_ERR		= -25300,			///< キーチェーンにパスワードが保存されていない
	
	PM_PARAM_ERROR				= 0xFF00			///< パラメータエラー
};


/**
	@brief	パワーマネジメントを管理
*/
@interface PowerManager : NSObject {
	NSString				*password_, *username_;		///<	パスワードとユーザ名
	NSString				*appid_;					///<	アプリケーションID
	void					*sleepNotifier_;			///<	スリープ監視オブジェクト
}

/**
	@brief	アプリケーションIDを指定して初期化
	@param	appid	アプリケーションID． 指定しない場合は"com.pixela.framework.PowerManager"が設定される
	@return	初期化されたオブジェクト
*/
- (id) initWithApplicationIdentifier:(NSString *)appid;

/**
	@brief	管理者パスワードをキーチェーンから探して設定
	@return	キーチェーンに無い場合はPM_NOTFOUND_ERRが返るので，ユーザに入力してもらい，setPassword:で設定する
*/
- (PMError) setPasswordFromKeychain;

/**
	@brief	管理者パスワードを設定
	@param	pass	パスワード
	@param	b		YESのときキーチェーンに保存
	@return	エラー
*/
- (PMError) setPassword:(NSString *)pass saveToKeychain:(BOOL)b;

/**
	@param	現在のユーザ名を返す
	@return	現在のユーザ名
*/
- (NSString *) username;

/**
	@brief		パワーマネジメントイベントを全て取得
	@return		NSDictionaryのNSArray　中身は以下<br>
		key: CFSTR(kIOPMPowerEventTimeKey)		value: NSDate<br>
		key: CFSTR(kIOPMPowerEventAppNameKey)	value: Application ID<br>
		key: CFSTR(kIOPMPowerEventTypeKey)		value: 以下<br>
			CFSTR(kIOPMAutoWake)			自動スリープ解除<br>
			CFSTR(kIOPMAutoPowerOn)			自動起動<br>
			CFSTR(kIOPMAutoWakeOrPowerOn)	自動起動＆スリープ解除<br>
			CFSTR(kIOPMAutoSleep)			自動スリープ<br>
			CFSTR(kIOPMAutoShutdown)		自動終了
*/
- (NSArray *) allScheduledPowerEvents;

/**
	@brief	自分が登録したイベントを取得
	preturn	allScheduledPowerEventsと同じ
*/
- (NSArray *) scheduledEvents;

/**
	@brief	種類を指定して自分が登録したイベントを取得
	@param	kind	イベントの種類
	preturn	allScheduledPowerEventsと同じ
*/
- (NSArray *) scheduledEventsWithKind:(PMKind)kind;

/**
	@brief	パワーマネジメントイベントを登録
	@param	date	イベントの発生時間
	@param	kind	イベントの種類
	@return	エラー
	@note	管理者パスワードを設定している必要がある
*/
- (PMError) addEventWithDate:(NSDate *)date andKind:(PMKind)kind;

/**
	@brief	パワーマネジメントイベントを削除
	@param	date	イベントの発生時間
	@param	kind	イベントの種類
	@return	エラー
	@note	管理者パスワードを設定している必要がある
*/
- (PMError) removeEventWithDate:(NSDate *)date andKind:(PMKind)kind;

@end


