/*
    avicore
    copyright (c) 2000-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "avifilesel.h"


static void avi_file_selection_class_init (AviFileSelectionClass *klass);
static void avi_file_selection_init       (AviFileSelection      *filesel);


static GtkWindowClass *parent_class = NULL;


/******************************************************************************
*                                                                             *
******************************************************************************/
GtkType
avi_file_selection_get_type (void)
{
  static GType file_selection_type = 0;

  if (!file_selection_type)
    {
      static const GTypeInfo filesel_info =
      {
        sizeof (AviFileSelectionClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)avi_file_selection_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (AviFileSelection),
        0,              /* n_preallocs */
        (GInstanceInitFunc)avi_file_selection_init,
      };

      file_selection_type =
        g_type_register_static (GTK_TYPE_FILE_SELECTION, "AviFileSelection",
                                                            &filesel_info, 0);
    }

  return file_selection_type;
}


static void
avi_file_selection_class_init (AviFileSelectionClass *class)
{
  GObjectClass *gobject_class;
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  gobject_class = (GObjectClass*) class;
  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;

  parent_class = g_type_class_peek_parent (class);
}


static void
avi_file_selection_init (AviFileSelection *filesel)
{
  GtkWidget *hbox, *vbox;

  filesel->continual = FALSE;
  filesel->check = NULL;
  filesel->vbox = NULL;
  filesel->avi_edit = NULL;

  /* ja:フレームとボックス */
  hbox = gtk_widget_get_parent (GTK_FILE_SELECTION (filesel)->ok_button);
  gtk_widget_ref (hbox);
  gtk_container_remove
            (GTK_CONTAINER (GTK_FILE_SELECTION (filesel)->main_vbox), hbox);
  gtk_widget_ref (GTK_FILE_SELECTION (filesel)->main_vbox);
  gtk_container_remove (GTK_CONTAINER (filesel),
                                    GTK_FILE_SELECTION (filesel)->main_vbox);

  filesel->hbox = gtk_hbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (filesel->hbox),
                        GTK_FILE_SELECTION(filesel)->main_vbox, TRUE, TRUE, 0);
  gtk_widget_show (filesel->hbox);

  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), filesel->hbox, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (filesel), vbox);
  gtk_widget_show (vbox);
}


/******************************************************************************
*                                                                             *
* ja:ダイアログ関数(りんご)                                                   *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    title,タイトル
      RET,ウィジェット                                                      */
GtkWidget*
avi_file_selection_new (const gchar *title)
{
  AviFileSelection *filesel;

  filesel = g_object_new (AVI_TYPE_FILE_SELECTION, NULL);
  gtk_window_set_title (GTK_WINDOW (filesel), title);
  gtk_dialog_set_has_separator (GTK_DIALOG (filesel), FALSE);

  return GTK_WIDGET (filesel);
}


/*  ja:連続して読み込むチェックボタンを非表示にする
    filesel,ウィジェット                                                    */
void
avi_file_selection_hide_continual (AviFileSelection *filesel)
{
  if (filesel->check)
    {
      gtk_widget_destroy (filesel->vbox);
      filesel->check = NULL;
    }
}


/* 連続して読み込むチェックボタンが押されたとき */
static void
avi_file_selection_toggled (GtkWidget        *widget,
                            AviFileSelection *filesel)
{
  filesel->continual
                = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
}


/*  ja:連続して読み込むチェックボタンを表示にする
    filesel,ウィジェット                                                    */
void
avi_file_selection_show_continual (AviFileSelection *filesel)
{
  if (!filesel->check)
    {
      filesel->check = gtk_check_button_new_with_label (_("Continual"));
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (filesel->check),
                                    filesel->continual);
      g_signal_connect (G_OBJECT (filesel->check), "toggled",
                        G_CALLBACK (avi_file_selection_toggled), filesel);
      gtk_box_pack_start (GTK_BOX (GTK_FILE_SELECTION (filesel)->main_vbox),
                                            filesel->check, FALSE, FALSE, 0);
      gtk_widget_show (filesel->check);
    }
}


/*  ja:連続して読み込むチェックボタンを取得する
    filesel,ウィジェット
        RET,TRUE:連続して読み込む,FALSE:単独で読み込む                      */
/*gboolean
avi_file_selection_get_continual (AviFileSelection *filesel)
{
  return filesel->continual;
}*/


/*  ja:連続して読み込むチェックボタンを設定する
      filesel,ウィジェット
    continual,TRUE:連続して読み込む,FALSE:単独で読み込む                    */
void
avi_file_selection_set_continual (AviFileSelection *filesel,
                                  gboolean          continual)
{
  filesel->continual = continual;
  if (filesel->check)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (filesel->check),
                                  continual);
}


/*  ja:AVI編集ハンドルを取得する
    filesel,ウィジェット
        RET,AVI編集ハンドル,NULL:設定なし                                   */
/*AviEdit** avi_file_selection_get_handle (AviFileSelection *filesel)
{
  return filesel->avi_edit;
}*/


/******************************************************************************
*                                                                             *
*                                                                             *
*                                                                             *
******************************************************************************/
struct _AviFileSelectionStream
{
  AviEdit *avi_edit;      /* AVI編集構造体 */
  GtkWidget *button, *check, *combo;
  GtkWidget *label0, *label1, *label2, *spin0, *spin1, *spin2;
};
typedef struct _AviFileSelectionStream AviFileSelectionStream;


/* ja:ラジオボタン(AVI)が押された */
static void
avi_file_selection_toggled_avi (GtkWidget        *widget,
                                AviFileSelection *filesel)
{
  if (filesel->frame)
    gtk_widget_set_sensitive (GTK_WIDGET(filesel->frame), FALSE);
  if (filesel->notebook)
    {
      gtk_widget_set_sensitive (GTK_WIDGET(filesel->notebook), TRUE);
      if (GTK_IS_NOTEBOOK (filesel->notebook))
        {
          gint i;
          GList *glist;

          glist = gtk_container_children (GTK_CONTAINER (filesel->notebook));
          for (i = g_list_length (glist) - 1; i >= 0; i--)
            gtk_widget_set_sensitive (GTK_WIDGET (g_list_nth_data (glist, i)),
                                                                        TRUE);
          g_list_free (glist);
        }
    }
}


/* ja:ラジオボタン(ビットマップ)が押された */
static void
avi_file_selection_toggled_bitmap (GtkWidget        *widget,
                                   AviFileSelection *filesel)
{
  gtk_widget_set_sensitive (GTK_WIDGET(filesel->frame), TRUE);
  if (filesel->video > 1)
    {
      gint i;
      GList *glist;

      gtk_widget_set_sensitive (GTK_WIDGET(filesel->notebook), TRUE);
      glist = gtk_container_children (GTK_CONTAINER (filesel->notebook));
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        gtk_widget_set_sensitive (GTK_WIDGET (g_list_nth_data (glist, i)),
                                                                        FALSE);
      g_list_free (glist);
    }
  else if (filesel->notebook)
    {
      gtk_widget_set_sensitive (GTK_WIDGET(filesel->notebook), FALSE);
    }
}


/* ja:ラジオボタン(WAVE)が押された */
static void
avi_file_selection_toggled_wave (GtkWidget        *widget,
                                 AviFileSelection *filesel)
{
  if (filesel->frame)
    gtk_widget_set_sensitive (GTK_WIDGET (filesel->frame), FALSE);
  if (filesel->audio > 1)
    {
      gint i;
      GList *glist;

      gtk_widget_set_sensitive (GTK_WIDGET(filesel->notebook), TRUE);
      glist = gtk_container_children (GTK_CONTAINER (filesel->notebook));
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        gtk_widget_set_sensitive (GTK_WIDGET (g_list_nth_data (glist, i)),
                                                                        FALSE);
      g_list_free (glist);
    }
  else if (filesel->notebook)
    {
      gtk_widget_set_sensitive (GTK_WIDGET(filesel->notebook), FALSE);
    }
}


/* ja:ラジオボタン(シナリオオブジェクト)が押された */
static void
avi_file_selection_toggled_scenario (GtkWidget        *widget,
                                     AviFileSelection *filesel)
{
  if (filesel->frame)
    gtk_widget_set_sensitive (GTK_WIDGET (filesel->frame), FALSE);
  if (filesel->notebook)
    gtk_widget_set_sensitive (GTK_WIDGET(filesel->notebook), FALSE);
}


/* ja:ラジオボタン(ビデオストリーム)が押された */
static void
avi_file_selection_toggled_video (GtkWidget        *widget,
                                  AviFileSelection *filesel)
{
  gint max;

  max = (gint) g_object_get_data (G_OBJECT (widget), "user_data");
  gtk_spin_button_set_range (GTK_SPIN_BUTTON (filesel->spin0), 0, max);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (filesel->spin0), 0);
  gtk_spin_button_set_range (GTK_SPIN_BUTTON (filesel->spin1), 0, max);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (filesel->spin1), max);
}


/* ja:コンボボックスの内容が変わったとき */
static void
avi_file_selection_changed (GtkWidget              *widget,
                            AviFileSelectionStream *avi_stream)
{
  gchar *label;
  const gchar *text;
  gint i, length, quality, key_frame;
  GList *glist;
  IcmObject *icm_object = NULL;
  IcmInfo icm_info;

  text = gtk_entry_get_text (GTK_ENTRY (widget));
  glist = gtk_container_children
                        (GTK_CONTAINER (GTK_COMBO (avi_stream->combo)->list));
  length = g_list_length (glist);
  for (i = 0; i < length; i++)
    {
      gtk_label_get (GTK_LABEL (gtk_bin_get_child
                            (GTK_BIN (g_list_nth_data (glist, i)))), &label);
      if (g_strcmp (label, text) == 0)
        {
          icm_object = g_object_get_data
                        (G_OBJECT (g_list_nth_data (glist, i)), "user_data");
          break;
        }
    }
  g_list_free (glist);
  g_object_set_data (G_OBJECT (avi_stream->combo), "user_data", icm_object);

  quality = icm_object ? icm_compress_get_default_quality (icm_object) : -1;
  key_frame = icm_object ? icm_compress_get_default_key_frame (icm_object) : 0;
  /* ja:スピンボタン */
  if (avi_stream->spin0)
    {
      gtk_widget_set_sensitive (GTK_WIDGET(avi_stream->label0), quality >= 0);
      gtk_widget_set_sensitive (GTK_WIDGET(avi_stream->spin0), quality >= 0);
      if (quality >= 0)
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (avi_stream->spin0),
                                                                    quality);
    }
  if (avi_stream->spin1)
    {
      gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->label1),
                                                                key_frame > 0);
      gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->spin1), key_frame > 0);
      if (key_frame > 0)
        gtk_spin_button_set_value (GTK_SPIN_BUTTON (avi_stream->spin1),
                                                                    key_frame);
    }
  if (avi_stream->spin2)
    {
      if (icm_object && icm_get_info (icm_object, &icm_info)
                                    && (icm_info.flags & ICM_FLAG_CRUNCH) != 0)
        {
          gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->label2), TRUE);
          gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->spin2), TRUE);
        }
      else
        {
          gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->label2), FALSE);
          gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->spin2), FALSE);
        }
    }
  /* ja:チェックボタン */
  if (avi_stream->check)
    gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->check),
                                                        icm_object != NULL);
  /* ja:ボタン */
  if (avi_stream->button)
    gtk_widget_set_sensitive (GTK_WIDGET (avi_stream->button),
                        icm_object && icm_is_dialog_configure (icm_object));
}


/* ja:設定ボタンが押された */
static void
avi_file_selection_clicked_config (GtkWidget *widget,
                                   GtkWidget *combo)
{
  icm_dialog_configure (g_object_get_data (G_OBJECT (combo),"user_data"));
}


/* ja:ページの破棄 */
static void
avi_file_selection_destroy_vbox (GtkWidget              *widget,
                                 AviFileSelectionStream *avi_stream)
{
  g_free (avi_stream);
}


/* ja:コンボボックスの破棄 */
static void
avi_file_selection_destroy_list (GtkWidget *widget,
                                 gpointer   user_data)
{
  gint i;
  GList *glist;

  glist = gtk_container_children (GTK_CONTAINER (widget));
  for (i = g_list_length (glist) - 1; i >= 0; i--)
    icm_close (g_object_get_data (G_OBJECT (g_list_nth_data (glist, i)),
                                                                "user_data"));
  g_list_free (glist);
}


/*  ja:AVI編集ハンドルを設定する
     filesel,ウィジェット
    avi_edit,AVI編集ハンドル,NULL:設定なし                                  */
void
avi_file_selection_set_handle (AviFileSelection *filesel,
                               AviEdit          **avi_edit)
{
  gchar *text;
  gint i, j, leng, length, streams, video = 0, audio = 0;
  GList *glist;
  GList *glist_handler;     /* ja:登録されている4文字コード */
  GList *glist_object;      /* ja:IcmObjectのリスト */
  GList *glist_name;        /* ja:名前のリスト */
  GSList *gslist;           /* ja:ラジオボタン */
  GSList *gslist_v = NULL;  /* ja:ラジオボタン */
  GSList *gslist_a = NULL;  /* ja:ラジオボタン */
  GtkWidget *label0, *label1, *radio;
  GtkWidget *frame, *hbox0, *hbox1, *vbox;
  AviFileSelectionStream *avi_stream;
  IcmObject *icm_object;
  IcmInfo icm_info;

  if (filesel->vbox)
    {
      gtk_widget_destroy (filesel->vbox);
      filesel->vbox = NULL;
    }
  filesel->avi_edit = avi_edit;
  if (!avi_edit)
    return;
  for (streams = 0; avi_edit[streams] != NULL; streams++)
    switch (avi_type (avi_edit [streams]))
      {
        case AVI_STREAM_VIDEO: video++; break;
        case AVI_STREAM_AUDIO: audio++; break;
        default: return;
      }
  if (streams <= 0)
    return;
  filesel->video = video;
  filesel->audio = audio;
  /* en:AVI */
  if (filesel->video>0 || filesel->audio>1)
    {
      /* ja:ラジオボタン */
      filesel->radio0 = gtk_radio_button_new_with_mnemonic (NULL, _("_AVI"));
      g_signal_connect (G_OBJECT (filesel->radio0), "toggled",
                        G_CALLBACK (avi_file_selection_toggled_avi), filesel);
      gslist = gtk_radio_button_group (GTK_RADIO_BUTTON (filesel->radio0));
      if (streams > 1)
        {
          /* ja:ノートブック */
          filesel->notebook = gtk_notebook_new ();
          gtk_notebook_set_scrollable (GTK_NOTEBOOK (filesel->notebook), TRUE);
        }
      else
        {
          /* ja:フレーム */
          filesel->notebook = gtk_frame_new ("");
        }
      video = audio = 0;
      glist_handler = icm_get_handler_list ();
      length = g_list_length (glist_handler);
      for (i = 0; i < streams; i++)
        {
          avi_stream = g_malloc0 (sizeof (AviFileSelectionStream));
          avi_stream->avi_edit = avi_edit[i];
          vbox = gtk_vbox_new (FALSE, SPACING);
          gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
          g_object_set_data (G_OBJECT (vbox), "user_data", avi_stream);
          g_signal_connect (G_OBJECT (vbox), "destroy",
                    G_CALLBACK (avi_file_selection_destroy_vbox), avi_stream);
          if (avi_type (avi_edit[i]) == AVI_STREAM_VIDEO)
            {
              /* ja:未圧縮 */
              glist_object = g_list_append (NULL, NULL);
              glist_name = g_list_append (NULL, g_strdup (_("Uncompress")));
              /* ja:各4文字コードについて */
              for (j = 0; j < length; j++)
                {
                  icm_object = icm_open
                                ((guint32)g_list_nth_data (glist_handler, j),
                                ICM_MODE_COMPRESS);
                  if (!icm_object)
                    continue;
                  if (!icm_compress_query (icm_object, avi_edit[i]->bmih)
                                    || !icm_get_info (icm_object, &icm_info))
                    {
                      icm_close (icm_object);
                      continue;
                    }
                  /* ja:スピンボタン */
                  if (!avi_stream->spin0
                        && icm_compress_get_default_quality (icm_object) >= 0)
                        avi_stream->spin0 = gtk_spin_button_new (GTK_ADJUSTMENT
                                (gtk_adjustment_new (-1, -1, 10000, 1, 5, 0)),
                                 0, 0);
                  if (!avi_stream->spin1
                        && icm_compress_get_default_key_frame(icm_object) > 0)
                        avi_stream->spin1 = gtk_spin_button_new (GTK_ADJUSTMENT
                                (gtk_adjustment_new (0, 0, G_MAXINT, 1, 5, 0)),
                                 0, 0);
                  if (!avi_stream->spin2
                        && (icm_info.flags & ICM_FLAG_CRUNCH) != 0)
                        avi_stream->spin2 = gtk_spin_button_new (GTK_ADJUSTMENT
                                (gtk_adjustment_new (0, 0, G_MAXINT, 1, 5, 0)),
                                 0, 0);
                  /* ja:ボタン */
                  if (!avi_stream->button
                                    && icm_is_dialog_configure (icm_object))
                      avi_stream->button = gtk_button_new_with_mnemonic
                                                        (_("Con_figure..."));
                  glist_object = g_list_append (glist_object, icm_object);
                  glist_name = g_list_append (glist_name, icm_info.name);
                }
              if (g_list_length (glist_name) > 1)
                {
                  /* ja:チェックボタン */
                  avi_stream->check = gtk_check_button_new_with_mnemonic
                                                            (_("_Recompress"));
                  gtk_toggle_button_set_active
                                (GTK_TOGGLE_BUTTON (avi_stream->check), FALSE);
                  gtk_widget_set_sensitive (avi_stream->check, FALSE);
                  /* ja:コンボボックス */
                  avi_stream->combo = gtk_combo_new ();
                  gtk_combo_set_popdown_strings (GTK_COMBO (avi_stream->combo),
                                                                glist_name);
                  glist = gtk_container_children
                        (GTK_CONTAINER (GTK_COMBO (avi_stream->combo)->list));
                  leng = g_list_length (glist);
                  for (j = 0; j < leng; j++)
                    g_object_set_data (G_OBJECT (g_list_nth_data (glist, j)),
                                       "user_data",
                                       g_list_nth_data (glist_object, j));
                  gtk_combo_set_value_in_list (GTK_COMBO (avi_stream->combo),
                                                                TRUE, FALSE);
                  gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO
                                        (avi_stream->combo)->entry), FALSE);
                  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO
                            (avi_stream->combo)->entry),
                            (const gchar *)g_list_nth_data (glist_name, 0));
                  g_signal_connect (G_OBJECT (GTK_COMBO
                        (avi_stream->combo)->entry), "changed",
                        G_CALLBACK (avi_file_selection_changed), avi_stream);
                  g_signal_connect (G_OBJECT (GTK_COMBO
                        (avi_stream->combo)->list), "destroy",
                        G_CALLBACK (avi_file_selection_destroy_list), NULL);
                  /* ja:ラベル */
                  label0 = gtk_label_new_with_mnemonic (_("_Compressor"));
                  gtk_label_set_mnemonic_widget (GTK_LABEL (label0),
                                        GTK_COMBO (avi_stream->combo)->entry);
                }
              else
                {
                  label0 = gtk_label_new ((const gchar *)
                                              g_list_nth_data (glist_name, 0));
                  g_list_free (glist_name);
                }
              g_list_free (glist_object);
              if (avi_stream->spin0)
                {
                  gtk_spin_button_set_numeric
                                    (GTK_SPIN_BUTTON(avi_stream->spin0), TRUE);
/*                g_signal_connect (G_OBJECT (avi_stream->spin0), "activate",
                            G_CALLBACK (avi_file_selection_activate), filesel);
*/                avi_stream->label0 = gtk_label_new_with_mnemonic
                                                            (_("_Quality"));
                  gtk_label_set_mnemonic_widget
                        (GTK_LABEL (avi_stream->label0), avi_stream->spin0);
                  gtk_widget_set_sensitive (avi_stream->spin0, FALSE);
                  gtk_widget_set_sensitive (avi_stream->label0, FALSE);
                }
              if (avi_stream->spin1)
                {
                  gtk_spin_button_set_numeric
                                    (GTK_SPIN_BUTTON(avi_stream->spin1), TRUE);
/*                g_signal_connect (G_OBJECT (avi_stream->spin1), "activate",
                            G_CALLBACK (avi_file_selection_activate), filesel);
*/                avi_stream->label1 = gtk_label_new_with_mnemonic
                                                            (_("_Key Frame"));
                  gtk_label_set_mnemonic_widget
                        (GTK_LABEL (avi_stream->label1), avi_stream->spin1);
                  gtk_widget_set_sensitive (avi_stream->spin1, FALSE);
                  gtk_widget_set_sensitive (avi_stream->label1, FALSE);
                }
              if (avi_stream->spin2)
                {
                  gtk_spin_button_set_numeric
                                    (GTK_SPIN_BUTTON(avi_stream->spin2), TRUE);
/*                g_signal_connect (G_OBJECT (avi_stream->spin2), "activate",
                            G_CALLBACK (avi_file_selection_activate), filesel);
*/                avi_stream->label2 = gtk_label_new_with_mnemonic
                                                            (_("_Data Rate"));
                  gtk_label_set_mnemonic_widget
                        (GTK_LABEL (avi_stream->label2), avi_stream->spin2);
                  gtk_widget_set_sensitive (avi_stream->spin2, FALSE);
                  gtk_widget_set_sensitive (avi_stream->label2, FALSE);
                }
              if (avi_stream->button)
                {
                  g_signal_connect (G_OBJECT (avi_stream->button), "clicked",
                                G_CALLBACK (avi_file_selection_clicked_config),
                                                            avi_stream->combo);
                  GTK_WIDGET_SET_FLAGS (avi_stream->button, GTK_CAN_DEFAULT);
                  gtk_widget_set_sensitive (avi_stream->button, FALSE);
                }
              /* ja:フレームとボックス */
              if (avi_stream->combo)
                {
                  hbox0 = gtk_hbox_new (FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                            label0, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                            avi_stream->combo, TRUE, TRUE, 0);
                  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
                }
              else
                {
                  gtk_box_pack_start (GTK_BOX (vbox), label0, FALSE, FALSE, 0);
                }
              if (avi_stream->spin0)
                {
                  hbox0 = gtk_hbox_new (FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                        avi_stream->label0, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                        avi_stream->spin0, TRUE, TRUE, 0);
                  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
                }
              if (avi_stream->spin1)
                {
                  hbox0 = gtk_hbox_new (FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                        avi_stream->label1, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                        avi_stream->spin1, TRUE, TRUE, 0);
                  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
                }
              if (avi_stream->spin2)
                {
                  hbox0 = gtk_hbox_new (FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                        avi_stream->label2, FALSE, FALSE, 0);
                  gtk_box_pack_start (GTK_BOX (hbox0),
                                        avi_stream->spin2, TRUE, TRUE, 0);
                  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
                }
              if (avi_stream->check || avi_stream->button)
                {
                  hbox0 = gtk_hbox_new (FALSE, SPACING);
                  if (avi_stream->check)
                    gtk_box_pack_start (GTK_BOX (hbox0),
                                        avi_stream->check, TRUE, TRUE, 0);
                  if (avi_stream->button)
                    gtk_box_pack_end(GTK_BOX(hbox0),
                                        avi_stream->button, FALSE, FALSE, 0);
                  gtk_box_pack_end (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
                }
              video++;
            }
          else
            {
              gtk_box_pack_start (GTK_BOX (vbox),
                                    gtk_label_new (_("PCM")), FALSE, FALSE, 0);
              audio++;
            }
          text = avi_get_name (avi_edit[i]) != NULL
                                ? g_strdup (avi_get_name (avi_edit[i]))
                                : avi_type (avi_edit[i]) == AVI_STREAM_VIDEO
                                        ? g_strdup_printf ("Video #%d", video)
                                        : g_strdup_printf ("Audio #%d", audio);
          if (GTK_IS_NOTEBOOK (filesel->notebook))
            {
              if (avi_type(avi_edit[i]) == AVI_STREAM_VIDEO
                                                        && filesel->video > 1)
                {
                  radio = gtk_radio_button_new_with_label (gslist_v, text);
                  g_object_set_data (G_OBJECT (radio), "user_data",
                                    (gpointer)(avi_length (avi_edit[i]) - 1));
                  g_signal_connect (G_OBJECT(radio), "toggled",
                                G_CALLBACK (avi_file_selection_toggled_video),
                                filesel);
                  gslist_v = gtk_radio_button_group (GTK_RADIO_BUTTON (radio));
                }
              else if (avi_type (avi_edit[i]) == AVI_STREAM_AUDIO
                                                        && filesel->audio > 1)
                {
                  radio = gtk_radio_button_new_with_label (gslist_a, text);
                  gslist_a = gtk_radio_button_group (GTK_RADIO_BUTTON (radio));
                }
              else
                {
                  radio = NULL;
                }
              gtk_notebook_append_page (GTK_NOTEBOOK (filesel->notebook), vbox,
                                radio == NULL ? gtk_label_new (text) : radio);
            }
          else
            {
              gtk_container_add (GTK_CONTAINER (filesel->notebook), vbox);
              gtk_frame_set_label (GTK_FRAME (filesel->notebook), text);
            }
          g_free (text);
        }
    }
  else
    {
      gslist = NULL;
    }
  /* ja:ビットマップ */
  if (filesel->video > 0)
    {
      /* ja:ラジオボタン */
      filesel->radio1 = gtk_radio_button_new_with_mnemonic
                                                        (gslist, _("_Bitmap"));
      g_signal_connect (G_OBJECT (filesel->radio1), "toggled",
                    G_CALLBACK (avi_file_selection_toggled_bitmap), filesel);
      gslist = gtk_radio_button_group (GTK_RADIO_BUTTON (filesel->radio1));
      /* ja:スピンボタン */
      for (i = 0; i < streams; i++)
        if (avi_type (avi_edit[i]) == AVI_STREAM_VIDEO)
          break;
      filesel->spin0 = gtk_spin_button_new (GTK_ADJUSTMENT
            (gtk_adjustment_new (0, 0, avi_length (avi_edit[i]) - 1, 1, 5, 0)),
             0, 0);
      filesel->spin1 = gtk_spin_button_new (GTK_ADJUSTMENT
            (gtk_adjustment_new (avi_length (avi_edit[i]) - 1,
                                    0, avi_length (avi_edit[i]) - 1, 1, 5, 0)),
             0, 0);
      gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (filesel->spin0), TRUE);
      gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (filesel->spin1), TRUE);
/*    g_signal_connect (G_OBJECT (filesel->spin0), "activate",
                            G_CALLBACK (avi_file_selection_activate), filesel);
      g_signal_connect (G_OBJECT (filesel->spin1), "activate",
                            G_CALLBACK (avi_file_selection_activate), filesel);
*/    /* ja:ラベル */
      label0 = gtk_label_new_with_mnemonic (_("_Start"));
      gtk_label_set_mnemonic_widget (GTK_LABEL (label0), filesel->spin0);
      label1 = gtk_label_new_with_mnemonic (_("_End"));
      gtk_label_set_mnemonic_widget (GTK_LABEL (label1), filesel->spin1);
      /* ja:フレームとボックス */
      hbox0 = gtk_hbox_new (FALSE, SPACING);
      gtk_container_set_border_width (GTK_CONTAINER (hbox0), SPACING);
      hbox1 = gtk_hbox_new (FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox1), label0, FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox1), filesel->spin0, TRUE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (hbox0), hbox1, FALSE, FALSE, 0);
      hbox1 = gtk_hbox_new (FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox1), label1, FALSE, FALSE, 0);
      gtk_box_pack_start (GTK_BOX (hbox1), filesel->spin1, TRUE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (hbox0), hbox1, FALSE, FALSE, 0);
      filesel->frame = gtk_frame_new (_("Range"));
      gtk_container_add (GTK_CONTAINER (filesel->frame), hbox0);
      gtk_widget_set_sensitive (filesel->frame, FALSE);
    }
  /* en:WAVE */
  if (filesel->audio > 0)
    {
      /* ja:ラジオボタン */
      filesel->radio2 = gtk_radio_button_new_with_mnemonic
                                                        (gslist, _("_WAVE"));
      g_signal_connect (G_OBJECT (filesel->radio2), "toggled",
                        G_CALLBACK (avi_file_selection_toggled_wave), filesel);
      gslist = gtk_radio_button_group (GTK_RADIO_BUTTON (filesel->radio2));
    }
  /* ja:シナリオオブジェクト */
  /* ja:ラジオボタン */
  filesel->radio3 = gtk_radio_button_new_with_mnemonic
                                            (gslist, _("Scenario _Object"));
  g_signal_connect (G_OBJECT (filesel->radio3), "toggled",
                    G_CALLBACK (avi_file_selection_toggled_scenario), filesel);

  /* ja:フレームとボックス */
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (hbox0), SPACING);
  if (filesel->radio0)
    gtk_box_pack_start (GTK_BOX (hbox0), filesel->radio0, FALSE, FALSE, 0);
  if (filesel->radio1)
    gtk_box_pack_start (GTK_BOX (hbox0), filesel->radio1, FALSE, FALSE, 0);
  if (filesel->radio2)
    gtk_box_pack_start (GTK_BOX (hbox0), filesel->radio2, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), filesel->radio3, FALSE, FALSE, 0);
  frame = gtk_frame_new (_("Format"));
  gtk_container_add (GTK_CONTAINER (frame), hbox0);

  filesel->vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (filesel->vbox), frame, FALSE, FALSE, 0);
  if (filesel->frame)
    gtk_box_pack_start (GTK_BOX (filesel->vbox),
                        filesel->frame, FALSE, FALSE, 0);
  if (filesel->notebook)
    gtk_box_pack_start (GTK_BOX (filesel->vbox),
                      filesel->notebook, FALSE, FALSE, 0);
  gtk_widget_show_all (filesel->vbox);
  gtk_box_pack_start (GTK_BOX (filesel->hbox), filesel->vbox, FALSE, FALSE, 0);
}


/*  ja:保存用のオプションを取得する
    filesel,ウィジェット
        RET,オプション,NULL:エラー                                          */
AviSave*
avi_file_selection_get_option (AviFileSelection *filesel)
{
  gint i, streams, value;
  GList *glist;
  GtkWidget *child, *radio;
  AviFileSelectionStream *avi_stream;
  AviSave *avi_save;
  IcmObject *icm_object;

  if (!filesel->vbox)
    return NULL;
  avi_save = g_malloc0 (sizeof (AviSave));
  if (filesel->radio0
        && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (filesel->radio0)))
    {
      avi_save->type = AVI_TYPE_AVI;
      if (GTK_IS_NOTEBOOK (filesel->notebook))
        {
          glist = gtk_container_children (GTK_CONTAINER (filesel->notebook));
          streams = g_list_length (glist);
          g_list_free (glist);
        }
      else
        {
          streams = 1;
        }
      avi_save->avi_opt = g_malloc0 ((streams + 1) * sizeof (AviOptions));
      for (i = 0; i < streams; i++)
        {
          if (GTK_IS_NOTEBOOK (filesel->notebook))
            {
              child = gtk_notebook_get_nth_page
                                        (GTK_NOTEBOOK (filesel->notebook), i);
            }
          else
            {
              child = gtk_bin_get_child (GTK_BIN (filesel->notebook));
            }
          avi_stream = g_object_get_data (G_OBJECT (child), "user_data");
          avi_save->avi_opt[i].type = avi_type (avi_stream->avi_edit);
          if (avi_stream->combo && (icm_object = g_object_get_data
                                (G_OBJECT (avi_stream->combo), "user_data")))
            {
              avi_save->avi_opt[i].handler = icm_get_handler(icm_object);
              avi_save->avi_opt[i].key_frame = avi_stream->spin0
                                && GTK_WIDGET_IS_SENSITIVE (avi_stream->spin0)
                                    ? gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (avi_stream->spin0)) : 0;
              avi_save->avi_opt[i].quality = avi_stream->spin1
                                && GTK_WIDGET_IS_SENSITIVE (avi_stream->spin1)
                                    ? gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (avi_stream->spin1)) : -1;
              avi_save->avi_opt[i].data_rate = avi_stream->spin2
                                && GTK_WIDGET_IS_SENSITIVE (avi_stream->spin2)
                                    ? gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (avi_stream->spin2)) : 0;
              avi_save->avi_opt[i].recompress = avi_stream->check
                                && GTK_WIDGET_IS_SENSITIVE (avi_stream->check)
                                && gtk_toggle_button_get_active
                                    (GTK_TOGGLE_BUTTON (avi_stream->check));
              avi_save->avi_opt[i].param_size
                                            = icm_get_state_size (icm_object);
              if (avi_save->avi_opt[i].param_size > 0)
                {
                  avi_save->avi_opt[i].param
                                = g_malloc (avi_save->avi_opt[i].param_size);
                  icm_get_state (icm_object, avi_save->avi_opt[i].param,
                                            avi_save->avi_opt[i].param_size);
                }
              avi_save->avi_opt[i].bmih_size = icm_compress_get_format_size
                                    (icm_object, avi_stream->avi_edit->bmih);
              avi_save->avi_opt[i].bmih
                                = g_malloc (avi_save->avi_opt[i].bmih_size);
              if (!icm_compress_get_format (icm_object,
                    avi_stream->avi_edit->bmih, avi_save->avi_opt[i].bmih))
                {
                  g_free (avi_save->avi_opt[i].bmih);
                  avi_save->avi_opt[i].bmih = NULL;
                  avi_save->avi_opt[i].bmih_size = 0;
                }
            }
        }
    }
  else if (filesel->radio1
        && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (filesel->radio1)))
    {
      avi_save->type = AVI_TYPE_BITMAP;
      avi_save->start = gtk_spin_button_get_value_as_int
                                            (GTK_SPIN_BUTTON (filesel->spin0));
      avi_save->end = gtk_spin_button_get_value_as_int
                                            (GTK_SPIN_BUTTON (filesel->spin1));
      if (avi_save->start > avi_save->end)
        {
          value = avi_save->start;
          avi_save->start = avi_save->end;
          avi_save->end = value;
        }
      if (GTK_IS_NOTEBOOK (filesel->notebook))
        {
          glist = gtk_container_children (GTK_CONTAINER (filesel->notebook));
          streams = g_list_length (glist);
          g_list_free (glist);
          for (i = 0; i < streams; i++)
            {
              child = gtk_notebook_get_nth_page
                                    (GTK_NOTEBOOK (filesel->notebook), i);
              radio = gtk_notebook_get_tab_label
                                    (GTK_NOTEBOOK (filesel->notebook), child);
              avi_stream = g_object_get_data (G_OBJECT (child), "user_data");
              if (avi_type (avi_stream->avi_edit) == AVI_STREAM_VIDEO
                && (!GTK_IS_RADIO_BUTTON (radio)
                || gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio))))
                break;
            }
          avi_save->stream = i;
        }
    }
  else if (filesel->radio2
        && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (filesel->radio2)))
    {
      avi_save->type = AVI_TYPE_WAVE;
      if (GTK_IS_NOTEBOOK (filesel->notebook))
        {
          glist = gtk_container_children (GTK_CONTAINER (filesel->notebook));
          streams = g_list_length (glist);
          g_list_free (glist);
          for (i = 0; i < streams; i++)
            {
              child = gtk_notebook_get_nth_page
                                    (GTK_NOTEBOOK (filesel->notebook), i);
              radio = gtk_notebook_get_tab_label
                                    (GTK_NOTEBOOK (filesel->notebook), child);
              avi_stream = g_object_get_data (G_OBJECT (child), "user_data");
              if (avi_type (avi_stream->avi_edit) == AVI_STREAM_AUDIO
                && (!GTK_IS_RADIO_BUTTON (radio)
                || gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio))))
                break;
            }
          avi_save->stream = i;
        }
    }
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(filesel->radio3)))
    {
      avi_save->type = AVI_TYPE_SCENARIO;
    }
  else
    {
      g_free(avi_save);
      avi_save = NULL;
    }
  return avi_save;
}
