/*
    Video maid
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "command.h"
#include "general.h"
#include "sigfile.h"
#include "thread.h"
#include "toolbar.h"
#include "misc/misc.h"
#include "misc/misc_mdi.h"


/******************************************************************************
*                                                                             *
* ja:一般関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:キャレットを消す
    vmaid,ウインドウ情報                                                    */
void
hide_caret (VmaidWindow *vmaid)
{
  gint sx;
  GdkRectangle rc;

  gtk_timeout_remove (timer_id);
  timer_id = 0;
  if (caret)
    {
      caret = FALSE;
      sx = MAX ((vmaid->drawing->allocation.width - 4) / vmaid->width - 2, 1);
      if (vmaid->top <= vmaid->cursor.frame
                                    && vmaid->cursor.frame <= vmaid->top + sx)
        {
          rc.x = (vmaid->cursor.frame - vmaid->top + 1) * vmaid->width + 1;
          rc.y = system_font_height * 2
                        + (vmaid->avi_edit[0] && vmaid->cursor.stream > 0
                                                        ? vmaid->height : 0);
          rc.width = 2;
          rc.height = vmaid->height;
          gtk_widget_draw (vmaid->drawing, &rc);
        }
    }
}


/*  ja:キャレットの描画
     vmaid,ウインドウ情報
    cursor,古いキャレット位置                                               */
void
draw_caret (VmaidWindow *vmaid,
            VmaidCursor *cursor)
{
  gchar *text;
  gint t, sx;
  guint context_id;
  GdkRectangle rc;

  sx = MAX ((vmaid->drawing->allocation.width - 4) / vmaid->width - 2, 1);
  if (cursor && timer_id != 0 && (vmaid->cursor.frame != cursor->frame
                                    || vmaid->cursor.stream != cursor->stream))
    {
      /* ja:タイマ再設定 */
      gtk_timeout_remove (timer_id);
      timer_id = 0;
      /* ja:古いキャレットを消去 */
      if (vmaid->top <= cursor->frame && cursor->frame <= vmaid->top + sx)
        {
          rc.x = (cursor->frame-vmaid->top + 1) * vmaid->width + 1;
          rc.y = system_font_height * 2
                                    + (vmaid->avi_edit[0] && cursor->stream > 0
                                                        ? vmaid->height : 0);
          rc.width = 2;
          rc.height = vmaid->height;
          gtk_widget_draw (vmaid->drawing, &rc);
        }
    }
  /* ja:新しいキャレットを描画 */
  if (timer_id == 0)
    caret = TRUE;
  if (vmaid->top <= vmaid->cursor.frame
                                    && vmaid->cursor.frame <= vmaid->top + sx)
    {
      rc.x = (vmaid->cursor.frame - vmaid->top + 1) * vmaid->width + 1;
      rc.y = system_font_height * 2
                            + (vmaid->avi_edit[0] && vmaid->cursor.stream > 0
                                                        ? vmaid->height : 0);
      rc.width = 2;
      rc.height = vmaid->height;
      gtk_widget_draw (vmaid->drawing, &rc);
    }
  if (timer_id == 0)
    timer_id = gtk_timeout_add (500, signal_timeout, NULL);
#ifdef G_HAVE_GINT64
  t = (gint64)vmaid->cursor.frame * vmaid->scale * 1000 / vmaid->rate;
#else /* not G_HAVE_GINT64 */
  t = vmaid->cursor.frame * vmaid->scale * 1000 / vmaid->rate;
#endif /* not G_HAVE_GINT64 */
  text = vmaid->select.stream < 0
        ? g_strdup_printf (_("Frame %d  Time %02d:%02d:%02d"),
                                vmaid->cursor.frame,
                                t / 60000 % 100, t / 1000 % 60, t / 10 % 100)
        : g_strdup_printf (_("Frame %d  Selected %d ... %d"),
            vmaid->cursor.frame,
            vmaid->select.frame
                        - (vmaid->cursor.frame < vmaid->select.frame ? 1 : 0),
            vmaid->cursor.frame
                        - (vmaid->cursor.frame > vmaid->select.frame ? 1 : 0));
  context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (status), "Status");
  gtk_statusbar_pop (GTK_STATUSBAR (status), context_id);
  gtk_statusbar_push (GTK_STATUSBAR (status), context_id, text);
  g_free (text);
}


/*  ja:メニューを設定する
    vmaid,ウインドウ情報                                                    */
void
set_menu_bar (VmaidWindow *vmaid)
{
  if (vmaid)
    {
      gtk_widget_set_sensitive (gtk_menu_get_attach_widget (GTK_MENU
                (gtk_item_factory_get_widget (ifactory_menu, "<main>/Edit"))),
                                                                        TRUE);
      gtk_widget_set_sensitive (gtk_menu_get_attach_widget (GTK_MENU
                (gtk_item_factory_get_widget (ifactory_menu, "<main>/View"))),
                                                                        TRUE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/File/Close"), TRUE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/File/Save As..."),
                                    vmaid->avi_edit[0] || vmaid->avi_edit[1]);
//    gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
//              "<main>/File/Preview..."),
//                                  vmaid->avi_edit[0] || vmaid->avi_edit[1]);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/File/Properties..."),
                                    vmaid->avi_edit[0] || vmaid->avi_edit[1]);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/Edit/Undo"), vmaid->undo != NULL);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/Edit/Redo"), vmaid->redo != NULL);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/Edit/Cut"), vmaid->select.stream >= 0);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/Edit/Copy"), vmaid->select.stream >= 0);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/Edit/Delete"), vmaid->select.stream >= 0);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/Edit/Jump..."),
                                    vmaid->avi_edit[0] || vmaid->avi_edit[1]);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/Edit/All"), vmaid->avi_edit[0] || vmaid->avi_edit[1]);
      gtk_widget_set_sensitive (toolbar_items[ 2].item,TRUE);
      gtk_widget_set_sensitive (toolbar_items[ 4].item,
                    vmaid->avi_edit[0] != NULL || vmaid->avi_edit[1] != NULL);
//    gtk_widget_set_sensitive (toolbar_items[ 6].item,
//                  vmaid->avi_edit[0] != NULL || vmaid->avi_edit[1] != NULL);
      gtk_widget_set_sensitive (toolbar_items[ 8].item,
                                vmaid->select.stream >= 0);
      gtk_widget_set_sensitive (toolbar_items[ 9].item,
                                vmaid->select.stream >= 0);
      gtk_widget_set_sensitive (toolbar_items[10].item, TRUE);
      gtk_widget_set_sensitive (toolbar_items[11].item,
                                vmaid->select.stream >= 0);
      gtk_widget_set_sensitive (toolbar_items[13].item, vmaid->undo != NULL);
      gtk_widget_set_sensitive (toolbar_items[14].item, vmaid->redo != NULL);
    }
  else
    {
      gtk_widget_set_sensitive (gtk_menu_get_attach_widget (GTK_MENU
                (gtk_item_factory_get_widget (ifactory_menu, "<main>/Edit"))),
                                                                        FALSE);
      gtk_widget_set_sensitive (gtk_menu_get_attach_widget (GTK_MENU
                (gtk_item_factory_get_widget (ifactory_menu, "<main>/View"))),
                                                                        FALSE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/File/Close"), FALSE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/File/Save As..."), FALSE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/File/Preview..."), FALSE);
      gtk_widget_set_sensitive (gtk_item_factory_get_widget (ifactory_menu,
                "<main>/File/Properties..."), FALSE);
      gtk_widget_set_sensitive (toolbar_items[ 2].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[ 4].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[ 6].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[ 8].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[ 9].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[10].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[11].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[13].item, FALSE);
      gtk_widget_set_sensitive (toolbar_items[14].item, FALSE);
    }
}


/*  ja:指定範囲を再描画する
     vmaid,ウインドウ情報
    cursor,古いキャレット位置
    select,古い選択開始位置
       top,古い左の座標                                                     */
void
clear_sel (VmaidWindow *vmaid,
           VmaidCursor *cursor,
           VmaidCursor *select,
           const gint   top)
{
  gint sx, frame, width, dx;
  gint left_old = -1, right_old = -1, left_new = -1, right_new = -1;
  GdkGC *gc;
  GdkRectangle rc, rc0, rc1;

  sx = MAX ((vmaid->drawing->allocation.width - 4) / vmaid->width - 2, 1);
  /* ja:スクロール */
  if (vmaid->top != top)
    {
      /* ja:移動量が多すぎるときには全体を再描画 */
      rc.x = vmaid->width + 2;
      rc.y = 0;
      rc.width = vmaid->width * sx;
      rc.height = system_font_height * 2 + ((vmaid->avi_edit[0] ? 1 : 0)
                            + (vmaid->avi_edit[1] ? 1 : 0)) * vmaid->height;
      if (ABS (top-vmaid->top) <= sx)
        {
          /* ja:移動量が小さいとき */
          dx = (top - vmaid->top) * vmaid->width;
          gc = gdk_gc_new (vmaid->drawing->window);
          gdk_gc_set_exposures (gc, TRUE);
          gdk_window_copy_area (vmaid->drawing->window, gc,
                        dx > 0 ? rc.x + dx : rc.x, 0,
                        vmaid->drawing->window, dx > 0 ? rc.x : rc.x - dx, 0,
                        rc.width - ABS (dx), rc.height);
          gdk_gc_destroy (gc);
          if (dx < 0)
            rc.x = rc.x + rc.width + dx;
          rc.width = ABS (dx);
        }
      if (vmaid->cursor.frame == vmaid->top
                                    || vmaid->cursor.frame == vmaid->top + sx
            || vmaid->cursor.frame == top || vmaid->cursor.frame == top + sx)
        {
          /* ja:端のキャレット */
          rc0.x = vmaid->cursor.frame == vmaid->top
                                                || vmaid->cursor.frame == top
                            ? vmaid->width + 1 : vmaid->width * (sx + 1) + 2;
          rc0.y = system_font_height * 2
                            + (vmaid->avi_edit[0] && vmaid->cursor.stream > 0
                                                        ? vmaid->height : 0);
          rc0.width = 1;
          rc0.height = vmaid->height;
          rc1 = rc;
          gdk_rectangle_union (&rc0, &rc1, &rc);
        }
      if (cursor->frame == vmaid->top || cursor->frame == vmaid->top + sx
                        || cursor->frame == top || cursor->frame == top + sx)
        {
          /* ja:端のキャレット */
          rc0.x = cursor->frame == vmaid->top || cursor->frame == top
                            ? vmaid->width + 1 : vmaid->width * (sx + 1) + 2;
          rc0.y = system_font_height * 2
                                    + (vmaid->avi_edit[0] && cursor->stream > 0
                                                        ? vmaid->height : 0);
          rc0.width = 1;
          rc0.height = vmaid->height;
          rc1 = rc;
          gdk_rectangle_union (&rc0, &rc1, &rc);
        }
    }
  else
    {
      rc.x = -1;
    }
  /* ja:以前の左右の選択について */
  if (select->stream >= 0)
    {
      frame = select->frame - (cursor->frame < select->frame ? 1 : 0);
      if (frame < top)
        left_old = frame;
      else if (top + sx <= frame)
        right_old = frame;
    }
  /* ja:現在の左右の選択について */
  if (vmaid->select.stream >= 0)
    {
      frame = vmaid->select.frame
                        - (vmaid->cursor.frame < vmaid->select.frame ? 1 : 0);
      if (frame < vmaid->top)
        left_new = frame;
      else if (vmaid->top + sx <= frame)
        right_new = frame;
    }
  if (left_old != left_new || right_old != right_new)
    {
      rc0.y = 0;
      rc0.width = vmaid->width;
      rc0.height
                = ((vmaid->avi_edit[0] ? 1 : 0) + (vmaid->avi_edit[1] ? 1 : 0))
                                    * vmaid->height + system_font_height * 2;
      /* ja:左側について */
      if (left_old != left_new)
        {
          rc0.x = 0;
          if (rc.x < 0)
            {
              rc = rc0;
            }
          else
            {
              rc1 = rc;
              gdk_rectangle_union (&rc0, &rc1, &rc);
            }
        }
      /* ja:右側について */
      if (right_old != right_new)
        {
          rc0.x = (sx + 1) * vmaid->width + 4;
          if (rc.x < 0)
            {
              rc = rc0;
            }
          else
            {
              rc1 = rc;
              gdk_rectangle_union (&rc0, &rc1, &rc);
            }
        }
    }
  if (vmaid->select.stream < 0 && select->stream >= 0)
    {
      /* ja:選択が解除された */
      width = MIN (MAX (cursor->frame, select->frame), vmaid->top + sx)
                        - MAX (MIN (cursor->frame, select->frame), vmaid->top);
      if (width > 0)
        {
          rc0.x
            = (MAX (MIN (cursor->frame, select->frame) - vmaid->top, 0) + 1)
                                                            * vmaid->width + 2;
          rc0.y = MIN (cursor->stream, select->stream) * vmaid->height
                                                    + system_font_height * 2;
          rc0.width = width * vmaid->width;
          rc0.height = (ABS (cursor->stream - select->stream) + 1)
                                                            * vmaid->height;
          if (!vmaid->avi_edit[0])
            rc0.y-=vmaid->height;
          gtk_widget_draw (vmaid->drawing, &rc);
          if (rc.x < 0)
            {
              rc = rc0;
            }
          else
            {
              rc1 = rc;
              gdk_rectangle_union (&rc0, &rc1, &rc);
            }
        }
    }
  else if (vmaid->select.stream >= 0)
    {
      if (vmaid->cursor.stream != cursor->stream)
        {
          /* ja:ストリームが変化した */
          width = MIN (MAX (cursor->frame, select->frame), vmaid->top + sx)
                        - MAX (MIN (cursor->frame, select->frame), vmaid->top);
          if (width > 0)
            {
              rc0.x = (MAX (MIN (cursor->frame, select->frame) - vmaid->top, 0)
                                                    + 1) * vmaid->width + 2;
              rc0.y = (vmaid->select.stream == 0 ? vmaid->height : 0)
                                                    + system_font_height * 2;
              rc0.width = width * vmaid->width;
              rc0.height = vmaid->height;
              if (rc.x < 0)
                {
                  rc = rc0;
                }
              else
                {
                  rc1 = rc;
                  gdk_rectangle_union (&rc0, &rc1, &rc);
                }
            }
        }
      if (vmaid->cursor.frame != cursor->frame)
        {
          /* ja:位置が変化した */
          width
            = MIN (MAX (vmaid->cursor.frame, cursor->frame), vmaid->top + sx)
                - MAX (MIN (vmaid->cursor.frame, cursor->frame), vmaid->top);
          if (width > 0)
            {
              rc0.x = (MAX (MIN (vmaid->cursor.frame, cursor->frame)
                                    - vmaid->top, 0) + 1) * vmaid->width + 2;
              rc0.y = MIN (vmaid->cursor.stream, vmaid->select.stream)
                                    * vmaid->height + system_font_height * 2;
              rc0.width = width * vmaid->width;
              rc0.height = (ABS (vmaid->cursor.stream - vmaid->select.stream)
                                                        + 1) * vmaid->height;
              if (!vmaid->avi_edit[0])
                rc0.y -= vmaid->height;
              if (rc.x < 0)
                {
                  rc = rc0;
                }
              else
                {
                  rc1 = rc;
                  gdk_rectangle_union (&rc0, &rc1, &rc);
                }
            }
        }
    }
  if (rc.x >= 0)
    gtk_widget_draw (vmaid->drawing, &rc);
}


/*  ja:リストから削除する
      d,リストの先頭
    RET,削除したリストの数                                                  */
gint
delete_list (VmaidHistory **d)
{
  gint i, count = 0;
  VmaidHistory *d0, *d1;

  for (d0 = *d; d0; d0 = d1)
    {
      for (i = 0; i < 2; i++)
        if (d0->avi_edit[i])
          avi_release (d0->avi_edit[i]);
      d1 = d0->next;
      g_free (d0);
      count++;
    }
  *d = NULL;
  return count;
}


/*  ja:編集履歴操作
    vmaid,ウインドウ情報
    kind,TRUE:redo,FALSE:undo                                               */
void
history_operation (VmaidWindow    *vmaid,
                   const gboolean  kind)
{
  gint i, sx, max;
  VmaidHistory *d, e;

  if (kind)
    {
      d = vmaid->redo;
      vmaid->redo = d->next;
    }
  else
    {
      d = vmaid->undo;
      vmaid->undo = d->next;
    }
  e.cursor = vmaid->cursor;
  e.select = vmaid->select;
  for (i = 0; i < 2; i++)
    e.avi_edit[i] = vmaid->avi_edit[i];
  vmaid->cursor = d->cursor;
  vmaid->select = d->select;
  for (i = 0; i < 2; i++)
    vmaid->avi_edit[i] = d->avi_edit[i];
  *d = e;
  if (kind)
    {
      d->next = vmaid->undo;
      vmaid->undo = d;
    }
  else
    {
      d->next = vmaid->redo;
      vmaid->redo = d;
    }
  max = get_max_frame (vmaid, -1) + 1;
  sx = MAX ((vmaid->drawing->allocation.width - 4) / vmaid->width - 2, 1);
  if (vmaid->cursor.frame > max)
    vmaid->cursor.frame = max;
  if (vmaid->top > vmaid->cursor.frame)
    vmaid->top = vmaid->cursor.frame;
  else if (vmaid->top < vmaid->cursor.frame - sx)
    vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
  set_menu_bar (vmaid);
  gtk_widget_draw (vmaid->drawing, NULL);
  draw_caret (vmaid, NULL);
}


/*  ja:ファイルの保存を問い合わせる
    vmaid,ウインドウ情報
      RET,TRUE:ファイルは閉じる,FALSE:ファイルを閉じない                    */
gboolean
prompt_close (VmaidWindow *vmaid)
{
  gint page;

  page = misc_mdi_get_page_from_data (MISC_MDI (mdi), vmaid);
  if (misc_mdi_get_edited (MISC_MDI (mdi), page))
    {
      gchar *text, *utf8str;

      utf8str = g_filename_to_utf8
            (misc_mdi_get_file (MISC_MDI (mdi), page), -1, NULL, NULL, NULL);
      text = g_strdup_printf (_("File %s was edited.\nSave?"), utf8str);
      g_free (utf8str);
      switch (misc_message_box ("Video maid", text, 0,
                                    _("_Yes"), _("_No"), _("Cancel"), NULL))
        {
          case 0:
            {
              gint page_num;

              page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
              if (page != page_num)
                gtk_notebook_set_current_page (GTK_NOTEBOOK (mdi), page);
              else
                page_num = -1;
              command_save (NULL, 0, NULL);
              if (page_num >= 0)
                gtk_notebook_set_current_page (GTK_NOTEBOOK (mdi), page_num);
              if (!misc_mdi_get_edited (MISC_MDI (mdi), page))
                break;
            }
          case 2:
          case -1:
            g_free (text);
            return FALSE;
        }
      g_free (text);
    }
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:高レベル関数群                                                           *
*                                                                             *
******************************************************************************/
/*  ja:最大フレーム数を求める
     vmaid,ウインドウ情報
    stream,有効なストリーム
       RET,最大フレーム数                                                   */
gint get_max_frame (VmaidWindow *vmaid,
                    const gint   stream)
{
  gint max0, max1;

  max0 = vmaid->avi_edit[0] ? avi_length (vmaid->avi_edit[0]) - 1 : 0;
#ifdef G_HAVE_GINT64
  max1 = vmaid->avi_edit[1] ? (gint64)avi_length_time (vmaid->avi_edit[1])
                        * vmaid->rate / ((gint64)vmaid->scale * 1000) - 1 : 0;
#else /* not G_HAVE_GINT64 */
  max1 = vmaid->avi_edit[1] ? avi_length_time (vmaid->avi_edit[1])
                                * vmaid->rate / (vmaid->scale * 1000) - 1 : 0;
#endif /* not G_HAVE_GINT64 */
  switch (stream)
    {
      case 0: return max0;
      case 1: return max1;
    }
  return MAX (max0, max1);
}
