/*
    avicore
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef __AVICORE_H__
#define __AVICORE_H__


#include "common.h"
#include "avifmt.h"
#include "bitmap.h"
#include "icm.h"
#include "wave.h"
#include "misc/fileio.h"


#ifdef __cplusplus
extern "C" {
#endif


/******************************************************************************
*                                                                             *
* ja:編集構造体                                                               *
*                                                                             *
******************************************************************************/
#define AVI_TYPE_UNKNOW     (-1)
#define AVI_TYPE_AVI        0
#define AVI_TYPE_BITMAP     1
#define AVI_TYPE_WAVE       2
#define AVI_TYPE_SCENARIO   3


#define AVI_EDIT_CACHE      20


typedef struct AviIndex_Tag
{
  goffset offset;
  guint32 size;
  guint32 flags;
} AviIndex;
typedef struct AviFile_Tag
{
  FileIO *fio;                  /* ja:ファイルポインタ */
  gpointer *data;               /* ja:メモリ */
  gchar *name;                  /* ja:ファイル名 */
  gint start,length;            /* ja:有効な範囲 */
  gint entries;                 /* ja:インデックスの数 */
  gint param;                   /* ja:ストリーム番号 */
  AviIndex *index;              /* ja:インデックス */
  /* ja:リスト */
  struct AviFile_Tag *prev,*next;
  /* ja:ビデオ固有 */
  guint32 handler;
  BitmapInfoHeader *bmih;       /* ja:オリジナルのフォーマット */
  /* ja:オーディオ固有 */
  WaveFormatEx *wfx;            /* オリジナルのフォーマット */
} AviFile;
typedef struct AviBuffer_Tag
{
  gpointer data;                /* ja:キャッシュ */
  /* ja:ビデオ固有 */
  gboolean raw;                 /* ja:TRUE:32ビットRAW,FALSE:ビットマップ */
  gint pos;                     /* ja:位置 */
  gint width,height;            /* ja:サイズ(RAW固有) */
} AviBuffer;
typedef struct AviEdit_Tag
{
  gint offset;
  AviFile *file;
  guint32 type;                 /* ja:streamtypeAUDIO/streamtypeVIDEO */
  gint length;                  /* ja:サンプル数 */
  guint32 rate;
  guint32 scale;                /* ja:rate÷scale=1秒間のフレーム数 */
  guint16 priority;             /* ja:優先度 */
  guint16 language;             /* ja:言語 */
  gchar *name;                  /* ja:名前 */
  AviBuffer buf[AVI_EDIT_CACHE];/* ja:キャッシュ */
  /* ja:ビデオ固有 */
  BitmapInfoHeader *bmih;       /* ja:展開後のフォーマット */
  /* ja:オーディオ固有 */
  WaveFormatEx *wfx;            /* ja:展開後のフォーマット */
} AviEdit;
typedef struct AviFrame_Tag
{
  guint32 handler;              /* ja:4文字コード */
  gint pos;                     /* ja:最後に取得したフレーム */
  AviEdit *avi_edit;
  IcmObject *icm_object;        /* ja:ICM展開 */
  BitmapInfoHeader *bmih;       /* ja:展開後のフォーマット */
  gsize bmih_size;              /* ja:展開後のフォーマットのサイズ */
} AviFrame;
typedef struct AviOptions_Tag
{
  guint32 type;                 /* ja:streamtypeAUDIO/streamtypeVIDEO */
  /* ja:ビデオ固有 */
  guint32 handler;              /* ja:4文字コード */
  gint key_frame;               /* ja:周期(0:すべてキーフレーム) */
  gint quality;                 /* ja:品質0～10000,-1:デフォルト */
  gsize data_rate;              /* ja:1秒間のサイズ(0:無効) */
  gboolean recompress;          /* ja:TRUE:再圧縮する,FALSE:再圧縮なし */
  gpointer param;               /* ja:オプションデータ */
  gsize param_size;             /* ja:オプションデータのサイズ */
  BitmapInfoHeader *bmih;       /* ja:圧縮後のフォーマット */
  gsize bmih_size;              /* ja:圧縮後のフォーマットのサイズ */
  /* ja:オーディオ固有 */
  WaveFormatEx *wfx;            /* ja:圧縮後のフォーマット */
} AviOptions;
typedef struct AviSave_Tag
{
  gint type;                    /* ja:保存するファイルの種類 */
  /* ja:AVI固有 */
  AviOptions *avi_opt;          /* ja:オプション */
  /* ja:ビットマップ固有 */
  gint stream;                  /* ja:対象ストリーム */
  gint start;                   /* ja:最初のフレーム */
  gint end;                     /* ja:最後のフレーム */
} AviSave;


/******************************************************************************
*                                                                             *
* ja:フォーマットマクロ                                                       *
*                                                                             *
******************************************************************************/
/*  ja:AVIのRateを取得する
    avi_edit,AVI編集ハンドル
         RET,Rate                                                           */
#define avi_get_rate(avi_edit) (((AviEdit *)(avi_edit))->rate)


/*  ja:AVIのScaleを取得する
    avi_edit,AVI編集ハンドル
         RET,Scale                                                          */
#define avi_get_scale(avi_edit) (((AviEdit *)(avi_edit))->scale)


/*  ja:AVIの優先度を取得する
    avi_edit,AVI編集ハンドル
         RET,優先度                                                         */
#define avi_get_priority(avi_edit) (((AviEdit *)(avi_edit))->priority)


/*  ja:AVIの言語を取得する
    avi_edit,AVI編集ハンドル
         RET,言語                                                           */
#define avi_get_language(avi_edit) (((AviEdit *)(avi_edit))->language)


/*  ja:AVIの名前を取得する
    avi_edit,AVI編集ハンドル
         RET,名前                                                           */
#define avi_get_name(avi_edit) (((AviEdit *)(avi_edit))->name)


/*  ja:AVIの幅を取得する
    avi_edit,AVI編集ハンドル
         RET,幅,0:エラー                                                    */
#define avi_get_width(avi_edit)                                             \
                    ((avi_edit)->bmih!=NULL?bmih_get_width((avi_edit)->bmih):0)


/*  ja:AVIの高さを取得する
    avi_edit,AVI編集ハンドル
         RET,高さ,0:エラー                                                  */
#define avi_get_height(avi_edit)                                            \
                ((avi_edit)->bmih!=NULL?bmih_get_height((avi_edit)->bmih):0)


/*  ja:AVIの色数を取得する
    avi_edit,AVI編集ハンドル
         RET,色数,0:エラー                                                  */
#define avi_get_bit_count(avi_edit)                                         \
                ((avi_edit)->bmih!=NULL?bmih_get_bit_count((avi_edit)->bmih):0)


/*  ja:AVIの圧縮形式を取得する
    avi_edit,AVI編集ハンドル
         RET,圧縮形式,-1:エラー                                             */
#define avi_get_compression(avi_edit)                                       \
    ((avi_edit)->bmih!=NULL?bmih_get_compression((avi_edit)->bmih):(guint32)-1)


/*  ja:AVIのX解像度を取得する
    avi_edit,AVI編集ハンドル
         RET,X解像度,-1:エラー                                              */
#define avi_get_x_pixels_per_meter(avi_edit)                                \
    ((avi_edit)->bmih!=NULL?bmih_get_x_pixels_per_meter((avi_edit)->bmih):-1)


/*  ja:AVIのY解像度を取得する
    avi_edit,AVI編集ハンドル
         RET,Y解像度,-1:エラー                                              */
#define avi_get_y_pixels_per_meter(avi_edit)                                \
    ((avi_edit)->bmih!=NULL?bmih_get_y_pixels_per_meter((avi_edit)->bmih):-1)


/*  ja:AVIのレートを取得する
    avi_edit,AVI編集ハンドル
         RET,レート,0:エラー                                                */
#define avi_get_samples_per_sec(avi_edit)                                   \
            ((avi_edit)->wfx!=NULL?wfx_get_samples_per_sec((avi_edit)->wfx):0)


/*  ja:AVIのチャンネルを取得する
    avi_edit,AVI編集ハンドル
         RET,チャンネル,0:エラー                                            */
#define avi_get_channels(avi_edit)                                          \
                    ((avi_edit)->wfx!=NULL?wfx_get_channels((avi_edit)->wfx):0)


/*  ja:AVIのビットを取得する
    avi_edit,AVI編集ハンドル
         RET,ビット,0:エラー                                                */
#define avi_get_bits_per_sample(avi_edit)                                   \
            ((avi_edit)->wfx!=NULL?wfx_get_bits_per_sample((avi_edit)->wfx):0)


/*  ja:AVIのRateを設定する
    avi_edit,AVI編集ハンドル
       value,Rate                                                           */
#define avi_set_rate(avi_edit,value) (((AviEdit *)(avi_edit))->rate=(value))


/*  ja:AVIのScaleを設定する
    avi_edit,AVI編集ハンドル
       value,Scale                                                          */
#define avi_set_scale(avi_edit,value) (((AviEdit *)(avi_edit))->scale=(value))


/*  ja:AVIの優先度を設定する
    avi_edit,AVI編集ハンドル
       value,優先度                                                         */
#define avi_set_priority(avi_edit,value)                                    \
                                    (((AviEdit *)(avi_edit))->priority=(value))


/*  ja:AVIの言語を設定する
    avi_edit,AVI編集ハンドル
       value,言語                                                           */
#define avi_set_language(avi_edit,value)                                    \
                                    (((AviEdit *)(avi_edit))->language=(value))


/*  ja:AVIの名前を設定する
    avi_edit,AVI編集ハンドル
       value,名前                                                           */
#define avi_set_name(avi_edit,value)                                        \
    (                                                                       \
        g_free(((AviEdit *)(avi_edit))->name),                              \
        ((AviEdit *)(avi_edit))->name=value!=NULL?g_strdup(value):NULL      \
    )


/*  ja:AVIの幅を設定する
    avi_edit,AVI編集ハンドル
       width,幅                                                             */
#define avi_set_width(avi_edit,width)                                       \
    ((avi_edit)->bmih!=NULL?(bmih_set_width((avi_edit)->bmih,width),        \
    bmih_set_size_image((avi_edit)->bmih,bm_image_bytes((avi_edit)->bmih))):0)


/*  ja:AVIの高さを設定する
    avi_edit,AVI編集ハンドル
      height,高さ                                                           */
#define avi_set_height(avi_edit,height)                                     \
    ((avi_edit)->bmih!=NULL?(bmih_set_height((avi_edit)->bmih,height),      \
    bmih_set_size_image((avi_edit)->bmih,bm_image_bytes((avi_edit)->bmih))):0)


/*  ja:AVIの色数を設定する
     avi_edit,AVI編集ハンドル
    bit_count,色数                                                          */
#define avi_set_bit_count(avi_edit,bit_count)                               \
    ((avi_edit)->bmih!=NULL                                                 \
        ?(bm_header_bytes((avi_edit)->bmih)                                 \
            <bx_header_bytes(   bmih_get_width((avi_edit)->bmih),           \
                                bmih_get_height((avi_edit)->bmih),          \
                                bit_count,                                  \
                                bmih_get_compression((avi_edit)->bmih),     \
                                bmih_get_color_used((avi_edit)->bmih))      \
            ?((avi_edit)->bmih=g_realloc((avi_edit)->bmih,                  \
                    bx_header_bytes(bmih_get_width((avi_edit)->bmih),       \
                                    bmih_get_height((avi_edit)->bmih),      \
                                    bit_count,                              \
                                    bmih_get_compression((avi_edit)->bmih), \
                                    bmih_get_color_used((avi_edit)->bmih))))\
            :NULL                                                           \
        ),bmih_set_bit_count((avi_edit)->bmih,bit_count)                    \
        :0)


/*  ja:AVIのX解像度を設定する
              avi_edit,AVI編集ハンドル
    x_pixels_per_meter,X解像度                                              */
#define avi_set_x_pixels_per_meter(avi_edit,x_pixels_per_meter)             \
        ((avi_edit)->bmih!=NULL                                             \
        ?bmih_set_x_pixels_per_meter((avi_edit)->bmih,x_pixels_per_meter):-1)


/*  ja:AVIのY解像度を設定する
              avi_edit,AVI編集ハンドル
    y_pixels_per_meter,Y解像度                                              */
#define avi_set_y_pixels_per_meter(avi_edit,y_pixels_per_meter)             \
        ((avi_edit)->bmih!=NULL                                             \
        ?bmih_set_y_pixels_per_meter((avi_edit)->bmih,y_pixels_per_meter):-1)


/*  ja:AVIのレートを設定する
           avi_edit,AVI編集ハンドル
    samples_per_sec,レート                                                  */
#define avi_set_samples_per_sec(avi_edit,samples_per_sec)                   \
                ((avi_edit)->wfx!=NULL                                      \
                ?wfx_set_samples_per_sec((avi_edit)->wfx,samples_per_sec):0)


/*  ja:AVIのチャンネルを設定する
    avi_edit,AVI編集ハンドル
    channels,チャンネル                                                     */
#define avi_set_channels(avi_edit,channels)                                 \
        ((avi_edit)->wfx!=NULL?wfx_set_channels((avi_edit)->wfx,channels):0)


/*  ja:AVIのビットを設定する
           avi_edit,AVI編集ハンドル
    bits_per_sample,ビット                                                  */
#define avi_set_bits_per_sample(avi_edit,bits_per_sample)                   \
                ((avi_edit)->wfx!=NULL                                      \
                ?wfx_set_bits_per_sample((avi_edit)->wfx,bits_per_sample):0)


/******************************************************************************
*                                                                             *
* ja:サンプル/時間マクロ                                                      *
*                                                                             *
******************************************************************************/
/*  ja:AVIのタイプを求める
    avi_edit,AVI編集ハンドル
         RET,streamtypeAUDIO/streamtypeVIDEO                                */
#define avi_type(avi_edit) (((AviEdit *)(avi_edit))->type)


/*  ja:AVIのサンプル番号から時間を求める
    avi_edit,AVI編集ハンドル
      sample,サンプル番号
         RET,時間(ミリ秒)                                                   */
#ifdef G_HAVE_GINT64
# define avi_sample_to_time(avi_edit,sample)                                \
                    (gint)((guint64)(sample)*avi_get_scale(avi_edit)*1000   \
                                                    /avi_get_rate(avi_edit))
#else /* not G_HAVE_GINT64 */
# define avi_sample_to_time(avi_edit,sample)                                \
                ((sample)*avi_get_scale(avi_edit)*1000/avi_get_rate(avi_edit))
#endif /* not G_HAVE_GINT64 */


/*  ja:AVIの時間からサンプル番号を求める
    avi_edit,AVI編集ハンドル
        time,時間(ミリ秒)
         RET,サンプル番号                                                   */
#ifdef G_HAVE_GINT64
# define avi_time_to_sample(avi_edit,time)                                  \
                            (gint)((guint64)(time)*avi_get_rate(avi_edit)   \
                                    /((guint64)avi_get_scale(avi_edit)*1000))
#else /* not G_HAVE_GINT64 */
# define avi_time_to_sample(avi_edit,time)                                  \
                ((time)*avi_get_rate(avi_edit)/(avi_get_scale(avi_edit)*1000))
#endif /* not G_HAVE_GINT64 */


/*  ja:AVIの長さを求める
    avi_edit,AVI編集ハンドル
         RET,サンプル数                                                     */
#define avi_length(avi_edit) (((AviEdit *)(avi_edit))->length)


/*  ja:AVIの長さを時間で求める
    avi_edit,AVI編集ハンドル
         RET,時間(ミリ秒)                                                   */
#define avi_length_time(avi_edit)                                           \
                            avi_sample_to_time(avi_edit,avi_length(avi_edit))


/******************************************************************************
*                                                                             *
* ja:初期化関数                                                               *
*                                                                             *
******************************************************************************/
/*  ja:AVIを初期化する                                                      */
gboolean
avi_init (void);


/*  ja:AVIを終了する                                                        */
gboolean
avi_exit (void);


/******************************************************************************
*                                                                             *
* ja:AVI入力関数(いちご)                                                      *
*                                                                             *
******************************************************************************/
/*  ja:ファイルを判別する
    file,ファイル名
     RET,0:AVI,1:ビットマップ,2:WAVEファイル,3:シナリオオブジェクト,-1:不明 */
gint
avi_file_type (const gchar *file);


/*  ja:AVIのストリームの数を取得する
    file,ファイル名
     RET,ストリームの数,0:エラー                                            */
guint32
avi_main_streams (const gchar *file);


/*  ja:AVIをファイルから開く
     file,ファイル名
    param,ストリーム番号
      RET,AVI編集ハンドル,NULL:エラー                                       */
AviEdit *
avi_open (const gchar *file,
          const gint   param);


/*  ja:AVIを閉じる
    avi_edit,AVI編集ハンドル
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_release (AviEdit *avi_edit);


/*  ja:AVIのデータのバイト数を取得する
    avi_edit,AVI編集ハンドル
       start,読み込みをはじめるサンプル番号
     samples,サンプル数
         RET,バイト数                                                       */
gint
avi_sample_size (AviEdit    *avi_edit,
                 const gint  start, 
                 const gint  samples);


/*  ja:AVIのデータを読み込む
    avi_edit,AVI編集ハンドル
       start,読み込みをはじめるサンプル番号
     samples,サンプル数
      buffer,データを格納するバッファ
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_read (AviEdit    *avi_edit,
          const gint  start,
          const gint  samples,
          gpointer    buffer);


/*  ja:AVIのフォーマットのサイズを取得する
    avi_edit,AVI編集ハンドル
         RET,フォーマットのバイト数,0:エラー                                */
gint
avi_format_size (AviEdit *avi_edit);


/*  ja:AVIのフォーマットを読み込む
    avi_edit,AVI編集ハンドル
      format,バッファへのポインタ
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_read_format (AviEdit  *avi_edit,
                 gpointer  format);


/******************************************************************************
*                                                                             *
* ja:フレーム関数(みんと)                                                     *
*                                                                             *
******************************************************************************/
/*  ja:DIBに展開されたフレームを開く
    avi_edit,AVI編集ハンドル
         RET,AVIフレーム構造体,NULL:エラー                                  */
AviFrame *
avi_get_frame_open (AviEdit *avi_edit);


/*  ja:DIBに展開されたフレームを閉じる
    avi_frame,AVIフレーム構造体
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_get_frame_close (AviFrame *avi_frame);


/*  ja:DIBに展開されたフレームを取得する
    avi_frame,AVIフレーム構造体
          pos,取得するフレーム(サンプル番号)
        width,幅
       height,高さ
    bit_count,色数
          RET,パックDIBへのポインタ,NULL:エラー                             */
BitmapInfoHeader *
avi_get_frame (AviFrame   *avi_frame,
               const gint  pos,
               const gint  width,
               const gint  height,
               const gint  bit_count);


/*  ja:RAWに展開されたフレームを取得する
    avi_frame,AVIフレーム構造体
          pos,取得するフレーム(サンプル番号)
        width,幅
       height,高さ
          RET,RAWへのポインタ,NULL:エラー                               */
guint8 *
avi_get_frame32 (AviFrame   *avi_frame,
                 const gint  pos,
                 const gint  width, 
                 const gint  height);


/*  ja:フレームがキーフレームか判定する
    avi_edit,AVI編集ハンドル
         pos,取得するフレーム(サンプル番号)
         RET,TRUE:キーフレーム,FALSE:非キーフレーム                         */
gboolean
avi_is_keyframe (AviEdit    *avi_edit,
                 const gint  pos);


/******************************************************************************
*                                                                             *
* ja:編集関数(れたす)                                                         *
*                                                                             *
******************************************************************************/
/*  ja:AVIを削除する
    avi_edit,AVI編集ハンドル
       start,削除をはじめるサンプル番号
     samples,サンプル数
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_delete (AviEdit    *avi_edit,
            const gint  start,
            const gint  samples);


/*  ja:AVIをコピーする
    avi_edit,AVI編集ハンドル
       start,コピーをはじめるサンプル番号
     samples,サンプル数
         RET,AVI編集ハンドル,NULL:エラー                                    */
AviEdit *
avi_copy (AviEdit    *avi_edit,
          const gint  start,
          const gint  samples);


/*  ja:AVIを貼り付ける
    avi_dst,貼り付けられるAVI編集ハンドル
        pos,貼り付けるサンプル番号
    avi_src,貼り付けるAVI編集ハンドル
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
avi_paste (AviEdit *avi_dst,
           const    gint pos,
           AviEdit *avi_src);


/*  ja:AVIを複製する
    avi_edit,AVI編集ハンドル
         RET,AVI編集ハンドル                                                */
#define avi_clone(avi_edit) avi_copy(avi_edit,0,avi_length(avi_edit))


/******************************************************************************
*                                                                             *
* ja:シナリオオブジェクト関数(ぷりん)                                         *
*                                                                             *
******************************************************************************/
/*  ja:AVIをシナリオオブジェクトに変換する
      avi_edit,AVI編集ハンドルへのポインタ
    interleave,TRUE:インターリーブする,FALSE:インターリーブしない
           RET,シナリオオブジェクト,NULL:エラー                             */
gchar *
avi_to_scenario (AviEdit        *avi_edit[],
                 const gboolean  interleave);


/*  ja:シナリオオブジェクトからAVIを作る
    scenario,シナリオオブジェクト
         RET,AVI編集ハンドルへのポインタ,NULL:エラー                        */
AviEdit **
avi_from_scenario (const gchar *scenario);


/******************************************************************************
*                                                                             *
* ja:保存関数(ざくろ)                                                         *
*                                                                             *
******************************************************************************/
/*  ja:保存用のオプションを解放する
    avi_save,オプション
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_save_free (AviSave *avi_save);


/*  ja:すべて保存されるかチェックする
    avi_edit,AVI編集ハンドル
    avi_save,オプション
         RET,TRUE:すべて保存される,FALSE:すべて保存されない                 */
#define avi_save_is_all(avi_edit,avi_save) ((AviEdit **)(avi_edit)!=NULL    \
        && ((AviEdit **)(avi_edit))[0]!=NULL && (AviSave *)(avi_save)!=NULL \
        && (((AviSave *)(avi_save))->type==AVI_TYPE_AVI                     \
            || ((AviSave *)(avi_save))->type==AVI_TYPE_SCENARIO             \
            || (((AviEdit **)(avi_edit))[1]==NULL                           \
                            && ((AviSave *)(avi_save))->type==AVI_TYPE_WAVE)))


/*  ja:ファイルに保存する
     avi_edit,AVI編集ハンドルへのポインタ
         file,ファイル名
      avi_opt,オプション
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_save_avi (AviEdit     *avi_edit[],
              const gchar *file,
              AviOptions  *avi_opt,
              gboolean   (*func)(gint, gpointer),
              gpointer     user_data);


/*  ja:ビットマップファイルに保存する
     avi_edit,AVI編集ハンドル
         file,ファイル名
        start,最初のフレーム
          end,最後のフレーム
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_save_bitmap (AviEdit      *avi_edit,
                 const gchar  *file,
                 const gint    start,
                 const gint    end,
                 gboolean    (*func)(gint, gpointer),
                 gpointer      user_data);


/*  ja:WAVEファイルに保存する
     avi_edit,AVI編集ハンドル
         file,ファイル名
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_save_wave (AviEdit   *avi_edit,
               const      gchar *file,
               gboolean (*func)(gint, gpointer),
               gpointer   user_data);


/*  ja:ファイルに保存する
     avi_edit,AVI編集ハンドルへのポインタ
         file,ファイル名
     avi_save,オプション
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_save_with_options (AviEdit      *avi_edit[],
                       const gchar  *file,
                       AviSave      *avi_save,
                       gboolean    (*func)(gint, gpointer),
                       gpointer      user_data);


/******************************************************************************
*                                                                             *
* ja:ダイアログ関数(りんご)                                                   *
*                                                                             *
******************************************************************************/
/*  ja:保存用のオプションを取得する
    avi_edit,AVI編集ハンドルへのポインタ
         RET,オプション,NULL:キャンセル:エラー                              */
AviSave *
avi_save_dialog (AviEdit *avi_edit[]);


/*  ja:保存用のオプションを取得する
    avi_edit,AVI編集ハンドルへのポインタ
        file,選択したファイル
        path,デフォルトのディレクトリ
         RET,オプション,NULL:エラー                                         */
AviSave *
avi_save_dialog_with_file (AviEdit      *avi_edit[],
                           gchar       **file,
                           const gchar  *path);


#ifdef __cplusplus
}
#endif


#endif
