/*
    Video maid
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "general.h"
#include "preview.h"
#include <gdk/gdkkeysyms.h>
#if ! GTK_CHECK_VERSION(2,6,0)
# include "icons.h"
#endif /* not GTK_CHECK_VERSION(2,6,0) */
#ifdef G_OS_WIN32
# include <windows.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
*                                                                             *
* ja:プレビュー関数群                                                         *
*                                                                             *
******************************************************************************/
#ifdef G_OS_WIN32
# define WAVBUF_MAX 8
#endif /* G_OS_WIN32 */


typedef struct _PreviewDialog
{
  gint frame, max, pos, start;
  guint timer_id;
  guint16 block_align;
  guint32 rate, scale;
  GTimer *gtimer;
  GtkWidget *drawing, *vscale, *status;
  GtkWidget *button0, *button1, *button2, *button3, *button4, *button5;
  AviEdit *avi_edit[2];
  AviFrame *avi_frame;
  AviPcm *avi_pcm;
#ifdef G_OS_WIN32
  HWND hWnd;
  HWAVEOUT hWaveOut;
  WAVEHDR whrOut[WAVBUF_MAX];
#endif /* G_OS_WIN32 */
} PreviewDialog;


/* ja:描画 */
static gboolean
preview_expose (GtkWidget      *widget,
                GdkEventExpose *event,
                PreviewDialog  *prevdlg)
{
  gint i, x, y, width, height;
  GdkGC *gc;
  GdkRectangle rc0, rc1;

  width = MIN (avi_get_width (prevdlg->avi_edit[0]) * widget->allocation.height
            / avi_get_height (prevdlg->avi_edit[0]), widget->allocation.width);
  height = MIN (avi_get_height (prevdlg->avi_edit[0])
                                                    * widget->allocation.width
            / avi_get_width (prevdlg->avi_edit[0]), widget->allocation.height);
  x = (widget->allocation.width - width) / 2;
  y = (widget->allocation.height - height) / 2;

  gc = gdk_gc_new (widget->window);
  for (i = 0; i < 4; i++)
    gdk_color_alloc (gdk_colormap_get_system (), system_color + i);
  gdk_gc_set_clip_rectangle (gc, &event->area);
  /* ja:背景 */
  gdk_gc_set_foreground (gc, system_color + 1);
  if (width < widget->allocation.width)
    {
      rc0.x = rc0.y = 0;
      rc0.width = x;
      rc0.height = widget->allocation.height;
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        gdk_draw_rectangle (widget->window, gc, TRUE, rc1.x, rc1.y,
                                                        rc1.width, rc1.height);
      rc0.x = x + width;
      rc0.width = widget->allocation.width - rc0.x;
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        gdk_draw_rectangle (widget->window, gc, TRUE, rc1.x, rc1.y,
                                                        rc1.width, rc1.height);
    }
  if (height < widget->allocation.height)
    {
      rc0.x = rc0.y = 0;
      rc0.width = widget->allocation.width;
      rc0.height = y;
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        gdk_draw_rectangle (widget->window, gc, TRUE, rc1.x, rc1.y,
                                                        rc1.width, rc1.height);
      rc0.y = y + height;
      rc0.height = widget->allocation.height - rc0.y;
      if (gdk_rectangle_intersect (&event->area, &rc0, &rc1))
        gdk_draw_rectangle (widget->window, gc, TRUE, rc1.x, rc1.y,
                                                        rc1.width, rc1.height);
    }
  /* ja:フレーム */
  if (prevdlg->avi_frame)
    {
      guchar *buf;

      buf = avi_get_frame32 (prevdlg->avi_frame,
                MIN (prevdlg->frame, avi_length (prevdlg->avi_edit[0]) - 1),
                                                                width, height);
      if (buf)
        gdk_draw_rgb_32_image (widget->window, gc, x, y, width, height,
                                        GDK_RGB_DITHER_NORMAL, buf, width * 4);
    }
  gdk_gc_destroy (gc);
  return TRUE;
}


/* ja:タイマ */
gboolean
preview_timeout (PreviewDialog *prevdlg)
{
  gint frame;

  frame = g_timer_elapsed (prevdlg->gtimer, NULL)
                            * prevdlg->rate / prevdlg->scale + prevdlg->start;
  if (frame > prevdlg->max)
    {
      frame = prevdlg->max;
      gtk_button_clicked (GTK_BUTTON (prevdlg->button1));
    }
  if (prevdlg->frame != frame)
    gtk_range_set_value (GTK_RANGE (prevdlg->vscale), frame);
  return TRUE;
}


#ifdef G_OS_WIN32
static LRESULT CALLBACK
WndProc (HWND   hWnd,
         UINT   uMsg,
         WPARAM wParam,
         LPARAM lParam)
{
  if (uMsg == WM_USER)
    {
      PreviewDialog *prevdlg;
      LPWAVEHDR lpwhrOut;

      prevdlg = GUINT_TO_POINTER (wParam);
      lpwhrOut = GUINT_TO_POINTER (lParam);
      waveOutUnprepareHeader (prevdlg->hWaveOut, lpwhrOut, sizeof (WAVEHDR));
      g_free (lpwhrOut->lpData);
      lpwhrOut->lpData = NULL;
      lpwhrOut->dwBufferLength = 0;
      if (prevdlg->pos >= 0)
        {
          gpointer out_buf;
          gint samples, out_samples;

          samples = MIN (avi_length (prevdlg->avi_edit[1]) - prevdlg->pos,
                                                                        1000);
          if (samples > 0 && avi_get_pcm (prevdlg->avi_pcm,
                                prevdlg->pos, samples, &out_buf, &out_samples))
            {
              lpwhrOut->lpData = out_buf;
              lpwhrOut->dwBufferLength = out_samples * prevdlg->block_align;
              if (waveOutPrepareHeader (prevdlg->hWaveOut,
                            lpwhrOut, sizeof (WAVEHDR)) == MMSYSERR_NOERROR)
                {
                  waveOutWrite (prevdlg->hWaveOut, lpwhrOut, sizeof (WAVEHDR));
                  prevdlg->pos += samples;
                }
              else
                {
                  g_free (lpwhrOut->lpData);
                  lpwhrOut->lpData = NULL;
                  lpwhrOut->dwBufferLength = 0;
                }
            }
        }
    }
  return DefWindowProc (hWnd, uMsg, wParam, lParam);
}


static VOID CALLBACK
waveOutProc (HWAVEOUT hWaveOut,
             UINT     uMsg,
             DWORD    dwInstance,
             DWORD    dwParam1,
             DWORD    dwParam2)
{
  if (uMsg == WOM_DONE)
    {
      PreviewDialog *prevdlg;

      prevdlg = GUINT_TO_POINTER (dwInstance);
      if (prevdlg->hWnd)
        PostMessage (prevdlg->hWnd, WM_USER, dwInstance, dwParam1);
    }
}
#endif /* G_OS_WIN32 */


/* ja:再生ボタンが押された */
static void
preview_clicked_button_play (GtkWidget     *widget,
                             PreviewDialog *prevdlg)
{
  gtk_widget_set_sensitive (prevdlg->button0, FALSE);
  gtk_widget_set_sensitive (prevdlg->button1, TRUE);
  gtk_widget_set_sensitive (prevdlg->button2, FALSE);
  gtk_widget_set_sensitive (prevdlg->button3, FALSE);
  gtk_widget_set_sensitive (prevdlg->button4, FALSE);
  gtk_widget_set_sensitive (prevdlg->button5, FALSE);
  gtk_widget_set_sensitive (prevdlg->vscale, FALSE);

  prevdlg->start = prevdlg->frame;
  prevdlg->timer_id = g_timeout_add_full (G_PRIORITY_LOW,
            (guint)MAX ((glonglong)prevdlg->scale * 1000 / prevdlg->rate, 1),
                                (GSourceFunc)preview_timeout, prevdlg, NULL);
#ifdef G_OS_WIN32
  if (prevdlg->hWaveOut)
    {
      gint i;

      prevdlg->pos = avi_time_to_sample (prevdlg->avi_edit[1],
                    (gint)((glonglong)prevdlg->frame * prevdlg->scale * 1000
                                                            / prevdlg->rate));
      for (i = 0; i < WAVBUF_MAX; i++)
        {
          gpointer out_buf;
          gint samples, out_samples;

          samples = MIN (avi_length (prevdlg->avi_edit[1]) - prevdlg->pos,
                                                                        1000);
          if (samples > 0 && avi_get_pcm (prevdlg->avi_pcm,
                                prevdlg->pos, samples, &out_buf, &out_samples))
            {
              prevdlg->whrOut[i].lpData = out_buf;
              prevdlg->whrOut[i].dwBufferLength = out_samples
                                                        * prevdlg->block_align;
              if (waveOutPrepareHeader (prevdlg->hWaveOut,
                                        &prevdlg->whrOut[i],
                                        sizeof (WAVEHDR)) == MMSYSERR_NOERROR)
                {
                  waveOutWrite (prevdlg->hWaveOut,
                                        &prevdlg->whrOut[i], sizeof (WAVEHDR));
                  prevdlg->pos += samples;
                }
              else
                {
                  g_free (prevdlg->whrOut[i].lpData);
                  prevdlg->whrOut[i].lpData = NULL;
                  prevdlg->whrOut[i].dwBufferLength = 0;
                }
            }
        }
    }
#endif /* G_OS_WIN32 */
  prevdlg->gtimer = g_timer_new ();
}


/* ja:停止ボタンが押された */
static void
preview_clicked_button_stop (GtkWidget     *widget,
                             PreviewDialog *prevdlg)
{
  if (prevdlg->timer_id != 0)
    {
      gtk_timeout_remove (prevdlg->timer_id);
      prevdlg->timer_id = 0;
    }
  if (prevdlg->gtimer)
    {
      g_timer_destroy (prevdlg->gtimer);
      prevdlg->gtimer = NULL;
    }
#ifdef G_OS_WIN32
  if (prevdlg->hWaveOut)
    {
      prevdlg->pos = -1;
      waveOutReset (prevdlg->hWaveOut);
    }
#endif /* G_OS_WIN32 */

  gtk_widget_set_sensitive (prevdlg->button0, prevdlg->frame < prevdlg->max);
  gtk_widget_set_sensitive (prevdlg->button1, FALSE);
  gtk_widget_set_sensitive (prevdlg->button2, prevdlg->frame > 0);
  gtk_widget_set_sensitive (prevdlg->button3, prevdlg->frame > 0);
  gtk_widget_set_sensitive (prevdlg->button4, prevdlg->frame < prevdlg->max);
  gtk_widget_set_sensitive (prevdlg->button5, prevdlg->frame < prevdlg->max);
  gtk_widget_set_sensitive (prevdlg->vscale, TRUE);
}


/* ja:最初に巻戻しボタンが押された */
static void
preview_clicked_button_previous (GtkWidget     *widget,
                                 PreviewDialog *prevdlg)
{
  gtk_range_set_value (GTK_RANGE (prevdlg->vscale), 0);
}


/* ja:巻戻しボタンが押された */
static void
preview_clicked_button_rewind (GtkWidget     *widget,
                               PreviewDialog *prevdlg)
{
  gtk_range_set_value (GTK_RANGE (prevdlg->vscale), --prevdlg->frame);
}


/* ja:早送りボタンが押された */
static void
preview_clicked_button_forward (GtkWidget     *widget,
                                PreviewDialog *prevdlg)
{
  gtk_range_set_value (GTK_RANGE (prevdlg->vscale), ++prevdlg->frame);
}


/* ja:最後に早送りボタンが押された */
static void
preview_clicked_button_next (GtkWidget     *widget,
                             PreviewDialog *prevdlg)
{
  gtk_range_set_value (GTK_RANGE (prevdlg->vscale), prevdlg->max);
}


/* ja:スケールの値が変更された */
static void
preview_value_changed (GtkWidget     *widget,
                       PreviewDialog *prevdlg)
{
  gchar *text;
  gint t;
  guint context_id;

  prevdlg->frame = gtk_range_get_value (GTK_RANGE (widget));
  if (!GTK_WIDGET_IS_SENSITIVE (prevdlg->button1))
    {
      gtk_widget_set_sensitive (prevdlg->button0,
                                                prevdlg->frame < prevdlg->max);
      gtk_widget_set_sensitive (prevdlg->button2, prevdlg->frame > 0);
      gtk_widget_set_sensitive (prevdlg->button3, prevdlg->frame > 0);
      gtk_widget_set_sensitive (prevdlg->button4,
                                                prevdlg->frame < prevdlg->max);
      gtk_widget_set_sensitive (prevdlg->button5,
                                                prevdlg->frame < prevdlg->max);
    }
  t = (glonglong)prevdlg->frame * prevdlg->scale * 1000 / prevdlg->rate;
  text = g_strdup_printf (_("Frame %d  Time %02d:%02d:%02d"), prevdlg->frame,
                                t / 60000 % 100, t / 1000 % 60, t / 10 % 100);
  context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (prevdlg->status),
                                                                    "Status");
  gtk_statusbar_pop (GTK_STATUSBAR (prevdlg->status), context_id);
  gtk_statusbar_push (GTK_STATUSBAR (prevdlg->status), context_id, text);
  g_free (text);

  if (prevdlg->drawing)
    {
      GdkRectangle rc;

      rc.width = MIN (avi_get_width (prevdlg->avi_edit[0])
                                    * prevdlg->drawing->allocation.height
                                    / avi_get_height (prevdlg->avi_edit[0]),
                                        prevdlg->drawing->allocation.width);
      rc.height = MIN (avi_get_height (prevdlg->avi_edit[0])
                                    * prevdlg->drawing->allocation.width
                                    / avi_get_width (prevdlg->avi_edit[0]),
                                        prevdlg->drawing->allocation.height);
      rc.x = (prevdlg->drawing->allocation.width - rc.width) / 2;
      rc.y = (prevdlg->drawing->allocation.height - rc.height) / 2;
      gtk_widget_draw (prevdlg->drawing, &rc);
    }
}


/* ja:ESCが押された */
static gboolean
preview_key_press (GtkWidget     *widget,
                   GdkEventKey   *event,
                   PreviewDialog *prevdlg)
{
  if (event->keyval == GDK_Return || event->keyval == GDK_KP_Enter
        || event->keyval == GDK_ISO_Enter || event->keyval == GDK_3270_Enter
                || event->keyval == GDK_space || event->keyval == GDK_KP_Space)
    gtk_button_clicked (GTK_BUTTON (GTK_WIDGET_IS_SENSITIVE (prevdlg->button0)
                                    ? prevdlg->button0 : prevdlg->button1));
  if (event->keyval == GDK_Home && GTK_WIDGET_IS_SENSITIVE (prevdlg->button2))
    gtk_button_clicked (GTK_BUTTON (prevdlg->button2));
  else if ((event->keyval == GDK_Left || ((event->state & GDK_MOD1_MASK)
                        && (event->keyval == GDK_H || event->keyval == GDK_h)))
                                && GTK_WIDGET_IS_SENSITIVE (prevdlg->button3))
    gtk_button_clicked (GTK_BUTTON (prevdlg->button3));
  else if ((event->keyval == GDK_Right || ((event->state & GDK_MOD1_MASK)
                        && (event->keyval == GDK_L || event->keyval == GDK_l)))
                                && GTK_WIDGET_IS_SENSITIVE (prevdlg->button4))
    gtk_button_clicked (GTK_BUTTON (prevdlg->button4));
  else if (event->keyval == GDK_End
                                && GTK_WIDGET_IS_SENSITIVE (prevdlg->button5))
    gtk_button_clicked (GTK_BUTTON (prevdlg->button5));
  else if (event->keyval == GDK_Escape)
    gtk_widget_destroy (widget);
  return TRUE;
}


/* ja:破棄 */
static void
preview_destroy (GtkWidget     *widget,
                 PreviewDialog *prevdlg)
{
  if (prevdlg->timer_id != 0)
    {
      gtk_timeout_remove (prevdlg->timer_id);
      prevdlg->timer_id = 0;
    }
  if (prevdlg->gtimer)
    {
      g_timer_destroy (prevdlg->gtimer);
      prevdlg->gtimer = NULL;
    }
  if (prevdlg->avi_frame)
    {
      avi_get_frame_close (prevdlg->avi_frame);
      prevdlg->avi_frame = NULL;
    }
  if (prevdlg->avi_pcm)
    {
      avi_get_pcm_close (prevdlg->avi_pcm);
      prevdlg->avi_pcm = NULL;
    }
#ifdef G_OS_WIN32
  if (prevdlg->hWaveOut)
    {
      prevdlg->pos = -1;
      waveOutReset (prevdlg->hWaveOut);
      waveOutClose (prevdlg->hWaveOut);
      prevdlg->hWaveOut = NULL;
    }
  if (prevdlg->hWnd)
    {
      DestroyWindow (prevdlg->hWnd);
      prevdlg->hWnd = NULL;
    }
#endif /* G_OS_WIN32 */
  gtk_main_quit ();
}


/*  ja:プレビュー
    vmaid,ウインドウ情報                                                    */
void
preview (VmaidWindow *vmaid)
{
  gchar *text;
  gint t;
  guint context_id;
  GtkRequisition req;
  GtkWidget *dialog, *hbox0, *hbox1, *vbox0, *vbox1;
  PreviewDialog prevdlg;
#if ! GTK_CHECK_VERSION(2,6,0)
  GdkPixbuf *pixbuf;
#endif /* not GTK_CHECK_VERSION(2,6,0) */
#ifdef G_OS_WIN32
  static gboolean register_class = FALSE;
#endif /* G_OS_WIN32 */

  prevdlg.max = get_max_frame (vmaid, -1);
  prevdlg.frame = MIN (vmaid->cursor.frame, prevdlg.max);
  prevdlg.rate = vmaid->rate;
  prevdlg.scale = vmaid->scale;
  prevdlg.timer_id = 0;
  prevdlg.gtimer = NULL;
  prevdlg.avi_edit[0] = vmaid->avi_edit[0];
  prevdlg.avi_edit[1] = vmaid->avi_edit[1];
  prevdlg.avi_frame = prevdlg.avi_edit[0]
                            ? avi_get_frame_open (prevdlg.avi_edit[0]) : NULL;
  prevdlg.avi_pcm = prevdlg.avi_edit[1]
                            ? avi_get_pcm_open (prevdlg.avi_edit[1]) : NULL;
  prevdlg.pos = -1;
#ifdef G_OS_WIN32
  if (prevdlg.avi_edit[1] && waveOutGetNumDevs() > 0)
    {
      WAVEFORMATEX wfxOut;

      if (!register_class)
        {
          WNDCLASSEX wcxOut;

          g_memset (&wcxOut, 0, sizeof (WNDCLASSEX));
          wcxOut.cbSize = sizeof (WNDCLASSEX);
          wcxOut.lpfnWndProc = WndProc;
          wcxOut.hInstance = GetModuleHandle (NULL);
          wcxOut.lpszClassName = _T("Video maid Preview Window");
          register_class = RegisterClassEx (&wcxOut) != 0;
        }
      prevdlg.hWnd = CreateWindow (_T("Video maid Preview Window"),
                                   NULL,
                                   WS_DISABLED | WS_POPUP,
                                   CW_USEDEFAULT,
                                   CW_USEDEFAULT,
                                   CW_USEDEFAULT,
                                   CW_USEDEFAULT,
                                   NULL,
                                   NULL,
                                   GetModuleHandle (NULL),
                                   NULL);
      wfxOut.wFormatTag = WAVE_FORMAT_PCM;
      wfxOut.nChannels = avi_get_channels (prevdlg.avi_edit[1]);
      wfxOut.nSamplesPerSec = avi_get_samples_per_sec (prevdlg.avi_edit[1]);
      wfxOut.wBitsPerSample = avi_get_bits_per_sample (prevdlg.avi_edit[1]);
      wfxOut.nBlockAlign = wfxOut.nChannels * wfxOut.wBitsPerSample / 8;
      wfxOut.nAvgBytesPerSec = wfxOut.nSamplesPerSec * wfxOut.nBlockAlign;
      wfxOut.cbSize = 0;
      if (waveOutOpen (&prevdlg.hWaveOut, WAVE_MAPPER, &wfxOut,
                                        (DWORD)GPOINTER_TO_UINT (waveOutProc),
                                        (DWORD)GPOINTER_TO_UINT (&prevdlg),
                                        CALLBACK_FUNCTION) != MMSYSERR_NOERROR)
        prevdlg.hWaveOut = NULL;
      prevdlg.block_align = wfxOut.nBlockAlign;
    }
  else
    {
      prevdlg.hWnd = NULL;
      prevdlg.hWaveOut = NULL;
      prevdlg.block_align = 0;
    }
  g_memset (prevdlg.whrOut, 0, WAVBUF_MAX * sizeof (WAVEHDR));
#endif /* G_OS_WIN32 */
  /* ja:ウインドウ */
  dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Preview"));
  g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (preview_key_press), &prevdlg);
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                    G_CALLBACK (preview_destroy), &prevdlg);
  /* ja:描画領域 */
  if (prevdlg.avi_edit[0])
    {
      prevdlg.drawing = gtk_drawing_area_new ();
      g_signal_connect (G_OBJECT (prevdlg.drawing), "expose-event",
                                        G_CALLBACK (preview_expose), &prevdlg);
      gtk_widget_set_size_request (prevdlg.drawing,
                                        avi_get_width (vmaid->avi_edit[0]),
                                        avi_get_height (vmaid->avi_edit[0]));
    }
  else
    {
      prevdlg.drawing = NULL;
    }
  /* ja:ボタン */
  prevdlg.button0 = gtk_button_new ();
  prevdlg.button1 = gtk_button_new ();
  prevdlg.button2 = gtk_button_new ();
  prevdlg.button3 = gtk_button_new ();
  prevdlg.button4 = gtk_button_new ();
  prevdlg.button5 = gtk_button_new ();
#if GTK_CHECK_VERSION(2,6,0)
  gtk_container_add (GTK_CONTAINER (prevdlg.button0), gtk_image_new_from_stock
                            (GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_BUTTON));
  gtk_container_add (GTK_CONTAINER (prevdlg.button1), gtk_image_new_from_stock
                            (GTK_STOCK_MEDIA_STOP, GTK_ICON_SIZE_BUTTON));
  gtk_container_add (GTK_CONTAINER (prevdlg.button2), gtk_image_new_from_stock
                            (GTK_STOCK_MEDIA_PREVIOUS, GTK_ICON_SIZE_BUTTON));
  gtk_container_add (GTK_CONTAINER (prevdlg.button3), gtk_image_new_from_stock
                            (GTK_STOCK_MEDIA_REWIND, GTK_ICON_SIZE_BUTTON));
  gtk_container_add (GTK_CONTAINER (prevdlg.button4), gtk_image_new_from_stock
                            (GTK_STOCK_MEDIA_FORWARD, GTK_ICON_SIZE_BUTTON));
  gtk_container_add (GTK_CONTAINER (prevdlg.button5), gtk_image_new_from_stock
                            (GTK_STOCK_MEDIA_NEXT, GTK_ICON_SIZE_BUTTON));
#else /* not GTK_CHECK_VERSION(2,6,0) */
  pixbuf = gdk_pixbuf_new_from_xpm_data (media_play16_xpm);
  gtk_container_add (GTK_CONTAINER (prevdlg.button0),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
  pixbuf = gdk_pixbuf_new_from_xpm_data (media_stop16_xpm);
  gtk_container_add (GTK_CONTAINER (prevdlg.button1),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
  pixbuf = gdk_pixbuf_new_from_xpm_data (media_previous16_xpm);
  gtk_container_add (GTK_CONTAINER (prevdlg.button2),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
  pixbuf = gdk_pixbuf_new_from_xpm_data (media_rewind16_xpm);
  gtk_container_add (GTK_CONTAINER (prevdlg.button3),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
  pixbuf = gdk_pixbuf_new_from_xpm_data (media_forward16_xpm);
  gtk_container_add (GTK_CONTAINER (prevdlg.button4),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
  pixbuf = gdk_pixbuf_new_from_xpm_data (media_next16_xpm);
  gtk_container_add (GTK_CONTAINER (prevdlg.button5),
                                        gtk_image_new_from_pixbuf (pixbuf));
  g_object_unref (pixbuf);
#endif /* not GTK_CHECK_VERSION(2,6,0) */
  g_signal_connect (G_OBJECT (prevdlg.button0), "clicked",
                    G_CALLBACK (preview_clicked_button_play), &prevdlg);
  g_signal_connect (G_OBJECT (prevdlg.button1), "clicked",
                    G_CALLBACK (preview_clicked_button_stop), &prevdlg);
  g_signal_connect (G_OBJECT (prevdlg.button2), "clicked",
                    G_CALLBACK (preview_clicked_button_previous), &prevdlg);
  g_signal_connect (G_OBJECT (prevdlg.button3), "clicked",
                    G_CALLBACK (preview_clicked_button_rewind), &prevdlg);
  g_signal_connect (G_OBJECT (prevdlg.button4), "clicked",
                    G_CALLBACK (preview_clicked_button_forward), &prevdlg);
  g_signal_connect (G_OBJECT (prevdlg.button5), "clicked",
                    G_CALLBACK (preview_clicked_button_next), &prevdlg);
  gtk_widget_set_sensitive (prevdlg.button0, prevdlg.frame < prevdlg.max);
  gtk_widget_set_sensitive (prevdlg.button1, FALSE);
  gtk_widget_set_sensitive (prevdlg.button2, prevdlg.frame > 0);
  gtk_widget_set_sensitive (prevdlg.button3, prevdlg.frame > 0);
  gtk_widget_set_sensitive (prevdlg.button4, prevdlg.frame < prevdlg.max);
  gtk_widget_set_sensitive (prevdlg.button5, prevdlg.frame < prevdlg.max);
  /* ja:スケール */
  prevdlg.vscale = gtk_hscale_new_with_range (0, prevdlg.max, 1);
  gtk_scale_set_draw_value (GTK_SCALE (prevdlg.vscale), FALSE);
  gtk_range_set_value (GTK_RANGE (prevdlg.vscale), prevdlg.frame);
  gtk_widget_size_request (prevdlg.vscale, &req);
  gtk_widget_set_size_request (prevdlg.vscale, req.width * 2, req.height);
  g_signal_connect (G_OBJECT (prevdlg.vscale), "value-changed",
                                G_CALLBACK (preview_value_changed), &prevdlg);
  /* ja:ステータスバー */
  prevdlg.status = gtk_statusbar_new ();
  t = (glonglong)prevdlg.frame * prevdlg.scale * 1000 / prevdlg.rate;
  text = g_strdup_printf (_("Frame %d  Time %02d:%02d:%02d"), prevdlg.frame,
                                t / 60000 % 100, t / 1000 % 60, t / 10 % 100);
  context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (prevdlg.status),
                                                                    "Status");
  gtk_statusbar_pop (GTK_STATUSBAR (prevdlg.status), context_id);
  gtk_statusbar_push (GTK_STATUSBAR (prevdlg.status), context_id, text);
  g_free (text);

  /* ja:フレームとボックス */
  vbox0 = gtk_vbox_new (FALSE, 0);
  vbox1 = gtk_vbox_new (FALSE, 0);
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prevdlg.button0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prevdlg.button1, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, FALSE, FALSE, 0);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prevdlg.button2, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prevdlg.button3, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prevdlg.button4, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prevdlg.button5, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (vbox1), hbox0, FALSE, FALSE, 0);
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (hbox0), SPACING);
  gtk_box_pack_start (GTK_BOX (hbox0), vbox1, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox0), prevdlg.vscale, TRUE, TRUE, 0);
  if (prevdlg.drawing)
    gtk_box_pack_start (GTK_BOX (vbox0), prevdlg.drawing, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX (vbox0), prevdlg.status, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (vbox0), hbox0, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (dialog), vbox0);

  /* ja:表示 */
  gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (window));
  gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
  gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER_ON_PARENT);
  gtk_window_set_skip_taskbar_hint (GTK_WINDOW (dialog), TRUE);
  gtk_grab_add (dialog);
  gtk_widget_show_all (dialog);
  if (prevdlg.drawing)
    gtk_widget_set_size_request (prevdlg.drawing, 1, 1);
  gtk_widget_set_size_request (prevdlg.vscale, req.width, req.height);
  gtk_main ();
}
