/*!
******************************************************************************

	@file	heap.cpp

	Copyright (C) 2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "heap.h"
#include "printf.h"

static u32 heap_entry_used;
static HEAP_ENTRY heap_entry[HEAP_ENTRY_MAX];
static HEAP_ENTRY *heap_alloc_list;
static HEAP_ENTRY *heap_free_list;
static volatile int heap_processing;

#define HEAP_SIZE			0x01000000				// 16MB
ALIGN(4096) static u8		_heap[HEAP_SIZE];
#define HEAP_START_ADDR		(_heap)

void heap_init( void )
{
	HEAP_ENTRY *entry = &heap_entry[0];

	entry->addr = HEAP_START_ADDR;
	entry->size = HEAP_SIZE;
	entry->next = NULL;
	entry->prev = NULL;

	for ( int i=1; i<HEAP_ENTRY_MAX; i++ ) {
		entry = &heap_entry[i];
		entry->addr = NULL;
		entry->size = 0;
		entry->next = NULL;
		entry->prev = NULL;
	}

	heap_alloc_list = NULL;
	heap_free_list  = &heap_entry[0];
	heap_entry_used = 1;
	heap_processing = 0;
}

void * heap_alloc( size_t size )
{
	u8 *ptr = NULL;

//	lock( &heap_processing );
	do {
		if ( heap_entry_used >= HEAP_ENTRY_MAX ) {
			vmm_printf( VMM_DEBUG, "heap_alloc() failed. (entry full)\n" );
			break;
		}

		HEAP_ENTRY *entry = heap_free_list;
		while ( entry != NULL )
		{	// リストから十分な空き容量がある領域を探す
			if ( entry->size >= size )
				break;
			entry = entry->next;
		}

		if ( entry == NULL ) {
			vmm_printf( VMM_DEBUG, "heap_alloc() failed. (out of memory)\n" );
			break;
		}

		ptr = entry->addr;
		entry->size -= size;
		if ( entry->size > 0 )
		{	// このエントリは再利用可能なので、アドレスを更新する
			entry->addr += size;
		}
		else
		{	// このエントリはもう使えないので、リストから外す
			if ( entry->next != NULL )
				entry->next->prev = entry->prev;
			if ( entry->prev != NULL )
				entry->prev->next = entry->next;
			else
				heap_free_list = entry->next;
			entry->addr = NULL;
			entry->next = NULL;
			entry->prev = NULL;
			heap_entry_used--;
		}

		// リストの先頭にエントリを追加する
		for ( int i=0; i<HEAP_ENTRY_MAX; i++ ) {
			entry = &heap_entry[i];
			if ( entry->addr == NULL )
				break;
		}
		if ( entry->addr != NULL ) {
			vmm_printf( VMM_DEBUG, "heap_alloc() failed. (internal error)\n" );
			abort();
		}
		entry->addr = ptr;
		entry->size = size;
		entry->next = heap_alloc_list;
		entry->prev = NULL;
		if ( heap_alloc_list != NULL )
			heap_alloc_list->prev = entry;
		heap_alloc_list = entry;
		heap_entry_used++;
	} while (0);
//	unlock( &heap_processing );

	return ptr;
}

int heap_free( void *ptr )
{
	int result = -1;

//	lock( &heap_processing );
	do {
		if ( ptr == NULL ) {
			vmm_printf( VMM_DEBUG, "heap_free() failed. (NULL pointer)\n" );
			break;
		}

		if ( (ptr < HEAP_START_ADDR) || (ptr >= HEAP_START_ADDR + HEAP_SIZE) ) {
			vmm_printf( VMM_DEBUG, "heap_free() failed. (out of range)\n" );
			break;
		}

		if ( heap_alloc_list == NULL ) {
			vmm_printf( VMM_DEBUG, "heap_free() failed. (heap_alloc_list == NULL)\n" );
			break;
		}

		// 引数に指定されたポインタを含むエントリを探す
		HEAP_ENTRY *entry = heap_alloc_list;
		while ( entry != NULL )
		{
			if ( (entry->addr <= ptr) && (ptr < entry->addr + entry->size) )
				break;
			entry = entry->next;
		}
		if ( entry == NULL ) {
			vmm_printf( VMM_DEBUG, "heap_free() failed. (not allocated)\n" );
			break;
		}

		if ( entry->addr != ptr )
		{	// 先頭アドレスが指定されていない場合も、範囲内なら領域を解放する
			vmm_printf( VMM_DEBUG, "heap_free(): [Warning] %08x is not start address. (start=%08x, size=%08x)\n",
						ptr, entry->addr, entry->size );
		}

		// 空き領域に戻す
		if ( entry->next != NULL )
			entry->next->prev = entry->prev;
		if ( entry->prev != NULL )
			entry->prev->next = entry->next;
		else
			heap_alloc_list = entry->next;
		entry->next = NULL;
		entry->prev = NULL;
		if ( heap_free_list != NULL ) {
			HEAP_ENTRY *free_entry = heap_free_list;
			while ( free_entry != NULL ) {
				if ( entry->addr + entry->size == free_entry->addr )
				{	// 空き領域を統合する
					// before: [entry][free_entry]
					//  after: [<---free_entry-->]
					free_entry->addr -= entry->size;
					free_entry->size += entry->size;
					heap_entry_used--;
					break;
				}
				else if ( entry->addr == free_entry->addr + free_entry->size )
				{	// 空き領域を統合する
					// before: [free_entry][entry]
					//  after: [<---free_entry-->]
					free_entry->size += entry->size;
					heap_entry_used--;
					break;
				}
				else if ( entry->addr < free_entry->addr )
				{	// 空き領域リストに挿入する
					entry->prev = free_entry->prev;
					entry->next = free_entry;
					if ( free_entry->prev != NULL )
						free_entry->prev->next = entry;
					else
						heap_free_list = entry;
					free_entry->prev = entry;
					break;
				}
				else if ( free_entry->next == NULL )
				{	// 空き領域リストの最後に追加する
					entry->prev = free_entry;
					free_entry->next = entry;
					break;
				}
				free_entry = free_entry->next;
			}
			if ( free_entry == NULL ) {
				vmm_printf( VMM_DEBUG, "heap_free() failed. (internal error)\n" );
				abort();
			}
			result = 0;
		}
		else {
			heap_free_list = entry;
			result = 0;
		}

	} while (0);
//	unlock( &heap_processing );

	return result;
}
