/*!
******************************************************************************

	@file	timer.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "printf.h"
#include "timer.h"
#include "irq.h"
#include "pic.h"
#include "io.h"
#include "rtc.h"
#include "task.h"
#include "disp.h"
#include "acpi.h"
#include "atomic.h"

static TIMER timer[TIMER_MAX];

#ifndef _VSUN86_PCSIM
static u32	  cpu_clock_khz			  = 0;
static u16	  timer_interval		  = TIMER_INTERVAL_10MS;
#endif	//!_VSN86_PCSIM
static time_t timer_system_start_time = 0;
static u64	  timer_start_tick		  = 0;

static bool timer_irq_handler( u8 irq, void *args );
#ifdef	_VSUN86_PCSIM
#define TIMER_INTERVAL	10
static SDL_TimerID sdl_timer;
static Uint32 sdl_timer_callback( Uint32, void * );
#endif	//_VSUN86_PCSIM

bool timer_init( void )
{
#ifndef _VSUN86_PCSIM
	// CPUクロックを計測する
	u64 tsc[2];
	u64 start, end;
	const u64 divisor = acpi_get_tick_freq() / 1000;
	start = acpi_get_tick_count() / divisor;
	tsc[0] = rdtsc();
	do {
		end = acpi_get_tick_count() / divisor;
	} while ( (end - start) < 100 );
	tsc[1] = rdtsc();
	cpu_clock_khz = (tsc[1] - tsc[0]) / 100;
	vmm_printf( VMM_DEBUG, "cpu_clock = %d.%03d MHz\n", cpu_clock_khz / 1000, cpu_clock_khz % 1000 );
#endif	//!_VSUN86_PCSIM

	// タイマを初期化する
#ifndef _VSUN86_PCSIM
	outb( IO_PIT_CTRL, 0x34 );
	outb( IO_PIT_CNT0, timer_interval & 0xFF );
	outb( IO_PIT_CNT0, timer_interval >> 8   );
#endif	//!_VSUN86_PCSIM
	for ( int i=0; i<TIMER_MAX; i++ )
		timer[i].used = false;

	// 割り込みハンドラを登録する
	if ( !irq_register( IRQ_TIMER, IRQ_TRIGGER_EDGE, timer_irq_handler, NULL ) )
		return false;
#ifdef	_VSUN86_PCSIM
	sdl_timer = SDL_AddTimer( TIMER_INTERVAL, sdl_timer_callback, NULL );
	if ( sdl_timer == NULL )
		return false;
#endif	//_VSUN86_PCSIM

	return true;
}

u64 timer_get_tick_count( void )
{
#ifndef _VSUN86_PCSIM
	const u64 divisor = acpi_get_tick_freq() / 1000;
	return (acpi_get_tick_count() - timer_start_tick) / divisor;
#else	//_VSUN86_PCSIM
	return SDL_GetTicks() - timer_start_tick;
#endif	//_VSUN86_PCSIM
}

bool timer_get_system_time( VSUN86_SYSTEM_TIME *tp )
{
	if ( tp == NULL )
		return false;

	const u64 msec = timer_system_start_time + timer_get_tick_count();
	time_t seconds = msec/ 1000;
	struct tm *t = gmtime( &seconds );
	if ( t == NULL )
		return false;

	tp->year = (u16)(t->tm_year + 1900);
	tp->mon  = (u8)t->tm_mon + 1;
	tp->mday = (u8)t->tm_mday;
	tp->wday = (u8)t->tm_wday;
	tp->hour = (u8)t->tm_hour;
	tp->min  = (u8)t->tm_min;
	tp->sec  = (u8)t->tm_sec;
	tp->msec = msec % 1000;

	return true;
}

bool timer_get_local_time( VSUN86_SYSTEM_TIME *tp )
{
	if ( tp == NULL )
		return false;

	const u64 msec = timer_system_start_time + timer_get_tick_count();
	time_t seconds = msec / 1000;
	struct tm *t = localtime( &seconds );
	if ( t == NULL )
		return false;

	tp->year = (u16)(t->tm_year + 1900);
	tp->mon  = (u8)t->tm_mon + 1;
	tp->mday = (u8)t->tm_mday;
	tp->wday = (u8)t->tm_wday;
	tp->hour = (u8)t->tm_hour;
	tp->min  = (u8)t->tm_min;
	tp->sec  = (u8)t->tm_sec;
	tp->msec = msec % 1000;

	return true;
}

void timer_rtc_to_system_time( bool rtc_is_local )
{
	struct tm t;
	time_t seconds;

	// RTCから現在時刻を取得してシステムタイムを更新する
	rtc_get_time( &t );
	seconds = mktime( &t );
	if ( rtc_is_local )
	{	// RTCの時刻＝ローカルタイムと解釈する
		seconds -= _timezone;
	}
	struct tm *tp = gmtime( &seconds );
	vmm_printf( VMM_DEBUG, "%4d/%02d/%02d [%3d] (%d) %02d:%02d:%02d (UTC)\n",
				tp->tm_year + 1900, tp->tm_mon + 1, tp->tm_mday, tp->tm_yday, tp->tm_wday,
				tp->tm_hour, tp->tm_min, tp->tm_sec );
	timer_system_start_time = mktime( &t ) * 1000;
#ifndef _VSUN86_PCSIM
	timer_start_tick = acpi_get_tick_count();
#else	//_VSUN86_PCSIM
	timer_start_tick = SDL_GetTicks();
#endif	//_VSUN86_PCSIM
}

void timer_task( void *args )
{
	(void)args;

	TASK_EVENT e;
	while ( task_wait_event( &e ) )
	{
		/*
		if ( e.from == TASK_ID_TIMER ) {
			switch ( e.msg )
			{
			case MSG_TIMER_TICK:
				{	// タイマが満了しているかチェックする
					vmm_printf( VMM_DEBUG, "MSG_TIMER_TICK\n" );
					const u64 tick = timer_get_tick_count();
					for ( u32 i=0; i<TIMER_MAX; i++ ) {
						if ( timer[i].proc == NULL )
							continue;
						if ( timer[i].realtime )
							continue;
						if ( tick >= timer[i].expire_tick )
						{	// タイマハンドラを実行する
							if ( timer[i].proc( timer[i].args ) )
							{	// タイマを削除する
								timer_del( i );
							}
							else
							{	// タイマ満了時間を再設定する
								timer[i].expire_tick = tick + timer[i].msec;
							}
						}
					}
				}
				break;
			}
		}
		*/
	}
}

u32 timer_add( u32 msec, TIMER_HANDLER proc, void *args, bool realtime, TASK_ID task_id )
{
	u32 i;
	static volatile int processing = 0;

	if ( realtime ) {
		if ( proc == NULL )
			return TIMER_ID_INVALID;
	}
	else {
		if ( task_id == TASK_ID_INVALID )
			return TIMER_ID_INVALID;	// 非リアルタイムの場合はタスクIDの指定が必要
	}

	lock( &processing );
	pic_disable_irq( IRQ_TIMER );
	{
		for ( i=0; i<TIMER_MAX; i++ )
		{	// 登録できる場所を探す
			if ( !timer[i].used ) {
				timer[i].used		 = true;
				timer[i].proc		 = proc;
				timer[i].args		 = args;
				timer[i].msec		 = msec;
				timer[i].expire_tick = timer_get_tick_count() + msec;
				timer[i].realtime	 = realtime;
				timer[i].task_id	 = task_id;
				/*
				if ( timer_interval > TIMER_INTERVAL_1MS * msec )
				{	// 次回の割り込みまでにタイマが満了してしまうので、インターバルを変更する
					timer_interval = TIMER_INTERVAL_1MS * msec;
					outb( IO_PIT_CTRL, 0x34 );
					outb( IO_PIT_CNT0, timer_interval & 0xFF );
					outb( IO_PIT_CNT0, timer_interval >> 8   );
				}
				*/
				break;
			}
		}
	}
	pic_enable_irq( IRQ_TIMER );
	unlock( &processing );

	if ( i >= TIMER_MAX )
		return TIMER_ID_INVALID;

	return i;
}

void timer_del( u32 id )
{
	if ( id < TIMER_MAX )
		timer[id].used = false;
}

static bool timer_irq_handler( u8 irq, void *args )
{
	(void)irq;
	(void)args;

	const u64 tick = timer_get_tick_count();

	u64 min_remain_time = 100;
	for ( u32 i=0; i<TIMER_MAX; i++ ) {
		if ( !timer[i].used )
			continue;
		if ( tick >= timer[i].expire_tick )
		{	// タイマハンドラを実行する
			if ( !timer[i].realtime ) {
				task_send_event( timer[i].task_id, MSG_TIMER_EXPIRED, i, 0 );
//				vmm_printf( VMM_DEBUG, "MSG_TIMER_EXPIRED : task_id=%08x, proc=%08x args=%08x\n",
//							timer[i].task_id, timer[i].proc, timer[i].args );
				timer_del( i );		// タイマハンドラの実行結果を待たずにタイマを削除する
				continue;
			}
			if ( timer[i].proc( timer[i].args ) )
			{	// タイマを削除する
				timer_del( i );
			}
			else
			{	// タイマ満了時間を再設定する
				timer[i].expire_tick = tick + timer[i].msec;
				if ( min_remain_time > timer[i].msec )
					min_remain_time = timer[i].msec;
			}
		}
		else {
			const u64 remain_time = timer[i].expire_tick - tick;
			if ( min_remain_time > remain_time )
				min_remain_time = remain_time;
		}
	}
	/*
	if ( min_remain_time != timer_interval / TIMER_INTERVAL_1MS )
	{	// 次に満了するタイマに合わせてインターバルを変更する
		timer_interval = TIMER_INTERVAL_1MS * min_remain_time;
		outb( IO_PIT_CTRL, 0x34 );
		outb( IO_PIT_CNT0, timer_interval & 0xFF );
		outb( IO_PIT_CNT0, timer_interval >> 8   );
	}
	*/

	return true;
}

void timer_usleep( u32 usec )
{
#ifndef _VSUN86_PCSIM
	const u64 thresh = (u64)usec * (acpi_get_tick_freq() / 1000000);
	u64 start = acpi_get_tick_count();
	u64 end;
	do {
		end = acpi_get_tick_count();
	} while ( (end - start) <= thresh );
#else	//_VSUN86_PCSIM
	(void)usec;
	SDL_Delay( 1 );
#endif	//_VSUN86_PCSIM
}

#ifdef	_VSUN86_PCSIM
static Uint32 sdl_timer_callback( Uint32 /*interval*/, void * /*param*/ )
{
	SDL_Event ev;

	ev.type			= SDL_VSUN86_VIRQ;
	ev.user.code	= IRQ_TIMER;
	ev.user.data1	= 0;
	ev.user.data2	= 0;

	if ( 0 != pic_set_irq( IRQ_TIMER ) )
		return TIMER_INTERVAL;

	if ( 0 != SDL_PushEvent( &ev ) ) {
		fprintf( stderr, "SDL_PushEvent(SDL_VSUN86_VIRQ) failed.\n" );
		return 0;
	}

	return TIMER_INTERVAL;
}
#endif	//_VSUN86_PCSIM
