/*!
******************************************************************************

	@file	usb.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "usb.h"
#include "atomic.h"
#include "printf.h"

#include "usb/host/uhci.h"
#include "usb/host/ohci.h"
#include "usb/host/ehci.h"

#include "usb/class/hub.h"
#include "usb/class/hid.h"
#include "usb/class/msc.h"

static u8			usb_device_num;
static USB_DEVICE	usb_device[USB_DEVICE_MAX];
static size_t		usb_devbuf_used;

static bool usb_alloc_queue( USB_DEVICE *dev, u8 ep, size_t buf_len );
//static void usb_print_descriptor( const USB_DESCRIPTOR *desc );

ALIGN(4096) u8 _usb_hcqhbuf[USB_HCQHBUF_SIZE];
ALIGN(4096) u8 _usb_hctdbuf[USB_HCTDBUF_SIZE];
ALIGN(4096) u8 _usb_devbuf[USB_DEVBUF_SIZE];

#define USB_STRING_MAX				512
#define USB_REQ_TIMEOUT_DEFAULT		5000	// [ms]

bool usb_init( void )
{
	usb_device_num = 0;
	memset( usb_device, 0, sizeof(usb_device) );
	usb_devbuf_used = 0;

	if ( !uhci_init() || !ohci_init() || !ehci_init() )
		return false;

	return true;
}

bool usb_start( void )
{
	vmm_printf( VMM_INFO, "\nInitializing USB Devices ...\n\n" );

	if ( !uhci_start() || !ohci_start() || !ehci_start() )
		return false;

	return true;
}

USB_DEVICE * usb_alloc_device( USB_HCD *hcd, void *hcd_priv, bool low_speed )
{
	if ( usb_device_num >= USB_DEVICE_MAX )
		return NULL;

	if ( (hcd == NULL) || (hcd_priv == NULL) )
		return NULL;

	const u8 device_id = usb_device_num;
	const u8 func_addr = device_id + 1;

	USB_DEVICE *dev = &usb_device[device_id];
	memset( dev, 0, sizeof(USB_DEVICE) );
	dev->hcd		  = hcd;
	dev->hcd_priv	  = hcd_priv;
	dev->low_speed	  = low_speed? 1:0;
	dev->max_pkt_size = low_speed? 8:64;

	USB_ENDPOINT *endp = &dev->ep[0];
	endp->valid		   = 1;
	endp->addr		   = 0;
	endp->attr		   = USB_ENDPT_CTRL;
	endp->interval	   = 0xFF;
	endp->max_pkt_size = low_speed? 8:64;
	endp->active	   = 0;

	dev->func_addr = 0;
	if ( !usb_alloc_queue( dev, 0, 1024 ) ) {
		vmm_printf( VMM_ERROR, "usb(%d): usb_alloc_queue() failed.\n", func_addr );
		return NULL;
	}

	// (デフォルトアドレスで)デバイスディスクリプタを取得する
	USB_DEVICE_DESC desc;
	if ( !usb_get_descriptor( dev, USB_DT_DEVICE, 0, &desc, sizeof(desc), USB_REQ_TIMEOUT_DEFAULT ) ) {
		vmm_printf( VMM_ERROR, "usb(%d): GET_DESCRIPTOR (Device, Default Address) failed.\n", func_addr );
		return NULL;
	}
	dev ->max_pkt_size = desc.bMaxPacketSize0;
	endp->max_pkt_size = desc.bMaxPacketSize0;

	// アドレスを設定する
	if ( !usb_set_address( dev, func_addr, USB_REQ_TIMEOUT_DEFAULT ) ) {
		vmm_printf( VMM_ERROR, "usb(%d): SET_ADDRESS failed.\n", func_addr );
		return NULL;
	}

	dev->func_addr = func_addr;
	if ( !usb_alloc_queue( dev, 0, 1024 ) ) {
		vmm_printf( VMM_ERROR, "usb(%d): usb_alloc_queue() failed.\n", func_addr );
		return NULL;
	}

	usb_device_num++;

	return dev;
}

bool usb_register( USB_DEVICE *dev, USB_HCD *child_hcd )
{
	USB_DEVICE_DESC dev_desc;
	USB_STRING_DESC str_desc_langid;
	USB_CONFIG_DESC cfg_desc;
	USB_DESCRIPTOR	desc;
	char product_name[256];

	// デバイスディスクリプタを取得する
	if ( !usb_get_descriptor( dev, USB_DT_DEVICE, 0, &dev_desc, sizeof(dev_desc), USB_REQ_TIMEOUT_DEFAULT ) ) {
		vmm_printf( VMM_ERROR, "usb(%d): GET_DESCRIPTOR (Device) failed.\n", dev->func_addr );
		return false;
	}

	// 製品名を取得する
	product_name[0] = '\0';
	if ( dev_desc.iProduct != 0 ) {
		if ( !usb_get_descriptor( dev, USB_DT_STRING, 0,
								  &str_desc_langid, sizeof(str_desc_langid), USB_REQ_TIMEOUT_DEFAULT ) ) {
			vmm_printf( VMM_ERROR, "usb(%d): GET_DESCRIPTOR (String) failed.\n", dev->func_addr );
			return false;
		}
		if ( !usb_get_descriptor( dev, USB_DT_STRING | dev_desc.iProduct, str_desc_langid.wLANGID[0],
								  &desc, sizeof(desc), USB_REQ_TIMEOUT_DEFAULT ) ) {
			vmm_printf( VMM_ERROR, "usb(%d): GET_DESCRIPTOR (String:LANGID[0]) failed.\n", dev->func_addr );
			return false;
		}
		if ( desc.str.bLength > 2 ) {
			const u8 cnt = (desc.str.bLength - 2) >> 1;
			for ( u32 n=0; n<cnt; n++ )
				product_name[n] = desc.str.bString[n<<1];
			product_name[cnt] = '\0';
		}
	}

	// コンフィギュレーションディスクリプタを取得する
	if ( !usb_get_descriptor( dev, USB_DT_CONFIG, 0, &cfg_desc, sizeof(cfg_desc), USB_REQ_TIMEOUT_DEFAULT ) ) {
		vmm_printf( VMM_ERROR, "usb(%d): GET_DESCRIPTOR (Configuration) failed.\n", dev->func_addr );
		return false;
	}
	const size_t cfg_desc_len = (cfg_desc.wTotalLength < sizeof(desc))? cfg_desc.wTotalLength : sizeof(desc);
	if ( cfg_desc_len == sizeof(desc) ) {
		vmm_printf( VMM_DEBUG, "usb(%d): DESCRIPTOR(Configuration).wTotalLength >= %04x(%04x)\n",
					dev->func_addr, cfg_desc_len, cfg_desc.wTotalLength );
	}
	if ( !usb_get_descriptor( dev, USB_DT_CONFIG, 0, &desc, cfg_desc_len, USB_REQ_TIMEOUT_DEFAULT ) ) {
		vmm_printf( VMM_ERROR, "usb(%d): GET_DESCRIPTOR (Configuration) failed.\n", dev->func_addr );
		return false;
	}
	dev->config_number = cfg_desc.bConfigurationValue;
	u32 off = 0;

	while ( off < cfg_desc_len ) {
		USB_DESCRIPTOR *p = (USB_DESCRIPTOR *)&desc.b[off];
		off += p->hdr.bLength;
		if ( (off > cfg_desc_len) || (p->hdr.bLength == 0) ) {
			vmm_printf( VMM_ERROR, "usb(%d): Descriptor parse error.\n", dev->func_addr );
			return false;
		}
//		vmm_printf( VMM_DEBUG, "usb(%d): bLength=%02x, bDescriptorType=%02x\n",
//					dev->func_addr, p->hdr.bLength, p->hdr.bDescriptorType );
		switch ( p->hdr.bDescriptorType << 8 )
		{
		case USB_DT_INTERFACE:
			{
				dev->iface_number	= p->iface.bInterfaceNumber;
				dev->iface_class	= p->iface.bInterfaceClass;
				dev->iface_subclass = p->iface.bInterfaceSubClass;
				dev->iface_protocol = p->iface.bInterfaceProtocol;
			}
			break;
		case USB_DT_ENDPOINT:
			{
				const u8 ep = p->ep.bEndpointAddress & 0x0F;
				dev->ep[ep].valid		 = 1;
				dev->ep[ep].addr		 = p->ep.bEndpointAddress;
				dev->ep[ep].attr		 = p->ep.bmAttributes;
				dev->ep[ep].interval	 = p->ep.bInterval;
				dev->ep[ep].max_pkt_size = p->ep.wMaxPacketSize;
				dev->ep[ep].active		 = 0;
				if ( !usb_alloc_queue( dev, ep, 8192 ) ) {
					vmm_printf( VMM_DEBUG, "usb(%d): usb_alloc_queue() failed.\n", dev->func_addr );
					return false;
				}
			}
			break;
		}
	}

	// コンフィギュレーションを選択する
	if ( !usb_set_configuration( dev, dev->config_number, USB_REQ_TIMEOUT_DEFAULT ) ) {
		vmm_printf( VMM_ERROR, "usb(%d): SET_CONFIGURATION failed.\n", dev->func_addr );
		return false;
	}

	// インターフェイスを選択する
	if ( !usb_set_interface( dev, dev->iface_number, USB_REQ_TIMEOUT_DEFAULT ) ) {
		vmm_printf( VMM_ERROR, "usb(%d): SET_INTERFACE failed.\n", dev->func_addr );
		return false;
	}

	vmm_printf( VMM_INFO, "%04X:%04X %s\n", dev_desc.idVendor, dev_desc.idProduct, product_name );

	switch ( dev->iface_class )
	{
	case USB_CLASS_HUB:
		if ( usb_hub_init( dev, child_hcd ) ) {
//			vmm_printf( VMM_NOTICE, "... Hub Class\n" );
			return true;
		}
		break;

	case USB_CLASS_HID:
		if ( usb_hid_init( dev ) ) {
			/*
			const char *subclass_name;
			switch ( dev->iface_protocol )
			{
			case USB_HID_KEYBOARD:	subclass_name = "Keyboard";		break;
			case USB_HID_MOUSE:		subclass_name = "Mouse";		break;
			default:				subclass_name = "Unknown";		break;
			}
			vmm_printf( VMM_NOTICE, "... HID Class (%s)\n", subclass_name );
			*/
			return true;
		}
		break;

	case USB_CLASS_STORAGE:
		if ( usb_msc_init( dev ) ) {
			/*
			const char *subclass_name;
			switch ( dev->iface_subclass )
			{
			case USB_MSC_SUBCLASS_RBC:			subclass_name = "RBC";					break;
			case USB_MSC_SUBCLASS_SFF_8020i:	subclass_name = "ATAPI(SFF-8020i)";		break;
			case USB_MSC_SUBCLASS_QIC_157:		subclass_name = "ATAPI(QIC-157)";		break;
			case USB_MSC_SUBCLASS_UFI:			subclass_name = "UFI";					break;
			case USB_MSC_SUBCLASS_SFF_8070i:	subclass_name = "ATAPI(SFF-8070i)";		break;
			case USB_MSC_SUBCLASS_SCSI:			subclass_name = "SCSI";					break;
			default:							subclass_name = "Unknown";				break;
			}
			vmm_printf( VMM_NOTICE, "... Mass Storage Class (Command Block:%s)\n", subclass_name );
			*/
			return true;
		}
		break;

	default:
		break;
	}

	vmm_printf( VMM_ERROR, "... initialization failed.\n" );
	return false;
}

static bool usb_alloc_queue( USB_DEVICE *dev, u8 ep, size_t buf_len )
{
	if ( dev->func_addr >= USB_DEVICE_MAX )
		return false;

	USB_ENDPOINT *endp = &dev->ep[ep];
	if ( !endp->valid )
		return false;
	if ( (endp->buf != NULL) && (endp->buf_len != buf_len) )
		return false;

	USB_HCD *hcd = (USB_HCD *)dev->hcd;
	if ( hcd == NULL )
		return false;
	if ( hcd->alloc_queue == NULL )
		return false;

	if ( usb_devbuf_used + buf_len > USB_DEVBUF_SIZE )
		return false;
	bool result = hcd->alloc_queue( dev, ep, buf_len );
	if ( !result )
		return false;
	endp->buf	  = &USB_DEVBUF[usb_devbuf_used];
	endp->buf_len = buf_len;
	usb_devbuf_used += buf_len;

	return true;
}

bool usb_ctrl_msg( USB_DEVICE *dev, u8 req_type, u8 request, u16 value, u16 index,
				   void *buf, size_t buf_len, int timeout )
{
	if ( dev->func_addr >= USB_DEVICE_MAX )
		return false;

	USB_ENDPOINT *endp = &dev->ep[0];
	if ( !endp->valid )
		return false;
	if ( endp->buf == NULL )
		return false;
	if ( sizeof(USB_REQUEST) + buf_len > endp->buf_len )
		return false;
	if ( USB_ENDPT_XFER_TYPE( endp->attr ) != USB_ENDPT_CTRL )
		return false;

	USB_HCD *hcd = (USB_HCD *)dev->hcd;
	if ( hcd == NULL )
		return false;
	if ( hcd->ctrl_msg == NULL )
		return false;

	if ( !try_lock( &endp->active ) )
		 return false;
	bool result = hcd->ctrl_msg( dev, req_type, request, value, index, buf, buf_len, timeout );
	unlock( &endp->active );

	return result;
}

bool usb_bulk_write( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout )
{
	if ( dev->func_addr >= USB_DEVICE_MAX )
		return false;
	if ( ep >= USB_ENDPOINT_MAX )
		return false;

	USB_ENDPOINT *endp = &dev->ep[ep];
	if ( !endp->valid )
		return false;
	if ( endp->buf == NULL )
		return false;
	if ( buf_len > endp->buf_len )
		return false;
	if ( USB_ENDPT_XFER_TYPE( endp->attr ) != USB_ENDPT_BULK )
		return false;
	if ( endp->addr & 0x80 )
		return false;

	USB_HCD *hcd = (USB_HCD *)dev->hcd;
	if ( hcd == NULL )
		return false;
	if ( hcd->bulk_xfer == NULL )
		return false;

	if ( !try_lock( &endp->active ) )
		 return false;
	bool result = hcd->bulk_xfer( dev, ep, buf, buf_len, timeout );
	unlock( &endp->active );

	return result;
}

bool usb_bulk_read( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout )
{
	if ( dev->func_addr >= USB_DEVICE_MAX )
		return false;
	if ( ep >= USB_ENDPOINT_MAX )
		return false;

	USB_ENDPOINT *endp = &dev->ep[ep];
	if ( !endp->valid )
		return false;
	if ( endp->buf == NULL )
		return false;
	if ( buf_len > endp->buf_len )
		return false;
	if ( USB_ENDPT_XFER_TYPE( endp->attr ) != USB_ENDPT_BULK )
		return false;
	if ( !(endp->addr & 0x80) )
		return false;

	USB_HCD *hcd = (USB_HCD *)dev->hcd;
	if ( hcd == NULL )
		return false;
	if ( hcd->bulk_xfer == NULL )
		return false;

	if ( !try_lock( &endp->active ) )
		 return false;
	bool result = hcd->bulk_xfer( dev, ep, buf, buf_len, timeout );
	unlock( &endp->active );

	return result;
}

bool usb_interrupt_write( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout )
{
	if ( dev->func_addr >= USB_DEVICE_MAX )
		return false;
	if ( ep >= USB_ENDPOINT_MAX )
		return false;

	USB_ENDPOINT *endp = &dev->ep[ep];
	if ( !endp->valid )
		return false;
	if ( endp->buf == NULL )
		return false;
	if ( buf_len > endp->buf_len )
		return false;
	if ( USB_ENDPT_XFER_TYPE( endp->attr ) != USB_ENDPT_INT )
		return false;
	if ( endp->addr & 0x80 )
		return false;

	USB_HCD *hcd = (USB_HCD *)dev->hcd;
	if ( hcd == NULL )
		return false;
	if ( hcd->interrupt_xfer == NULL )
		return false;

	if ( !try_lock( &endp->active ) )
		 return false;
	bool result = hcd->interrupt_xfer( dev, ep, buf, buf_len, timeout );
	unlock( &endp->active );

	return result;
}

bool usb_interrupt_read( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout )
{
	if ( dev->func_addr >= USB_DEVICE_MAX )
		return false;
	if ( ep >= USB_ENDPOINT_MAX )
		return false;

	USB_ENDPOINT *endp = &dev->ep[ep];
	if ( !endp->valid )
		return false;
	if ( endp->buf == NULL )
		return false;
	if ( buf_len > endp->buf_len )
		return false;
	if ( USB_ENDPT_XFER_TYPE( endp->attr ) != USB_ENDPT_INT )
		return false;
	if ( !(endp->addr & 0x80) )
		return false;

	USB_HCD *hcd = (USB_HCD *)dev->hcd;
	if ( hcd == NULL )
		return false;
	if ( hcd->interrupt_xfer == NULL )
		return false;

	if ( !try_lock( &endp->active ) )
		 return false;
	bool result = hcd->interrupt_xfer( dev, ep, buf, buf_len, timeout );
	unlock( &endp->active );

	return result;
}

bool usb_get_descriptor( USB_DEVICE *dev, u16 desc_type, u16 lang_id, void *desc, size_t size, int timeout )
{
	if ( !usb_ctrl_msg( dev, USB_REQ_DIR_IN | USB_REQ_TYPE_STD | USB_REQ_DEVICE,
						USB_GET_DESCRIPTOR, desc_type, lang_id, desc, size, timeout ) )
		return false;

	USB_DESCRIPTOR *p = (USB_DESCRIPTOR *)desc;
	if ( (p->hdr.bLength == 0) || (p->hdr.bDescriptorType == 0) )
		return false;

//	if ( dev->func_addr != 0 )
//		usb_print_descriptor( p );

	return true;
}

bool usb_set_address( USB_DEVICE *dev, u8 addr, int timeout )
{
	if ( !usb_ctrl_msg( dev, USB_REQ_DIR_OUT | USB_REQ_TYPE_STD | USB_REQ_DEVICE,
						USB_SET_ADDRESS, addr, 0, NULL, 0, timeout ) )
		return false;

	dev->func_addr = addr;
	return true;
}

bool usb_set_configuration( USB_DEVICE *dev, u8 config, int timeout )
{
	if ( !usb_ctrl_msg( dev, USB_REQ_DIR_OUT | USB_REQ_TYPE_STD | USB_REQ_DEVICE,
						USB_SET_CONFIGURATION, config, 0, NULL, 0, timeout ) )
		return false;

	dev->config_number = config;
	return true;
}

bool usb_set_interface( USB_DEVICE *dev, u8 iface, int timeout )
{
	if ( !usb_ctrl_msg( dev, USB_REQ_DIR_OUT | USB_REQ_TYPE_STD | USB_REQ_INTERFACE,
						USB_SET_INTERFACE, 0, iface, NULL, 0, timeout ) )
		return false;

	dev->iface_number = iface;
	return true;
}

/*
static void usb_print_descriptor( const USB_DESCRIPTOR *desc )
{
	switch ( desc->hdr.bDescriptorType << 8 )
	{
	case USB_DT_DEVICE:
		{
			vmm_printf( VMM_DEBUG, "[Device Desciptor]\n" );
			vmm_printf( VMM_DEBUG, "bLength ............... %02x\n", desc->dev.bLength );
			vmm_printf( VMM_DEBUG, "bDescriptorType ....... %02x\n", desc->dev.bDescriptorType );
			vmm_printf( VMM_DEBUG, "bcdUSB ................ %04x\n", desc->dev.bcdUSB );
			vmm_printf( VMM_DEBUG, "bDeviceClass .......... %02x\n", desc->dev.bDeviceClass );
			vmm_printf( VMM_DEBUG, "bDeviceSubClass ....... %02x\n", desc->dev.bDeviceSubClass );
			vmm_printf( VMM_DEBUG, "bDeviceProtocol ....... %02x\n", desc->dev.bDeviceProtocol );
			vmm_printf( VMM_DEBUG, "bMaxPacketSize0 ....... %02x\n", desc->dev.bMaxPacketSize0 );
			vmm_printf( VMM_DEBUG, "idVendor .............. %04x\n", desc->dev.idVendor );
			vmm_printf( VMM_DEBUG, "idProduct ............. %04x\n", desc->dev.idProduct );
			vmm_printf( VMM_DEBUG, "bcdDevice ............. %04x\n", desc->dev.bcdDevice );
			vmm_printf( VMM_DEBUG, "iManufacturer ......... %02x\n", desc->dev.iManufacturer );
			vmm_printf( VMM_DEBUG, "iProduct .............. %02x\n", desc->dev.iProduct );
			vmm_printf( VMM_DEBUG, "iSerialNumber ......... %02x\n", desc->dev.iSerialNumber );
			vmm_printf( VMM_DEBUG, "bNumConfigurations .... %02x\n", desc->dev.bNumConfigurations );
		}
		break;

	case USB_DT_CONFIG:
		{
			vmm_printf( VMM_DEBUG, "[Configuration Desciptor]\n" );
			vmm_printf( VMM_DEBUG, "bLength ............... %02x\n", desc->cfg.bLength );
			vmm_printf( VMM_DEBUG, "bDescriptorType ....... %02x\n", desc->cfg.bDescriptorType );
			vmm_printf( VMM_DEBUG, "wTotalLength .......... %04x\n", desc->cfg.wTotalLength );
			vmm_printf( VMM_DEBUG, "bNumInterfaces ........ %02x\n", desc->cfg.bNumInterfaces );
			vmm_printf( VMM_DEBUG, "bConfigurationValue ... %02x\n", desc->cfg.bConfigurationValue );
			vmm_printf( VMM_DEBUG, "iConfiguration ........ %02x\n", desc->cfg.iConfiguration );
			vmm_printf( VMM_DEBUG, "bmAttributes .......... %02x\n", desc->cfg.bmAttributes );
			vmm_printf( VMM_DEBUG, "bMaxPower ............. %02x\n", desc->cfg.bMaxPower );
		}
		break;

	case USB_DT_STRING:
		{
			vmm_printf( VMM_DEBUG, "[String Desciptor]\n" );
			vmm_printf( VMM_DEBUG, "bLength ............... %02x\n", desc->str.bLength );
			vmm_printf( VMM_DEBUG, "bDescriptorType ....... %02x\n", desc->str.bDescriptorType );
			for ( int i=0; i<(desc->cfg.bLength-2)/2; i++ )
				vmm_printf( VMM_DEBUG, "wLANGID[%d] ............ %04x\n", i, desc->str.wLANGID[i] );
		}
		break;

	case USB_DT_INTERFACE:
		{
			vmm_printf( VMM_DEBUG, "[Interface Desciptor]\n" );
			vmm_printf( VMM_DEBUG, "bLength ............... %02x\n", desc->iface.bLength );
			vmm_printf( VMM_DEBUG, "bDescriptorType ....... %02x\n", desc->iface.bDescriptorType );
			vmm_printf( VMM_DEBUG, "bInterfaceNumber ...... %02x\n", desc->iface.bInterfaceNumber );
			vmm_printf( VMM_DEBUG, "bAlternateSetting ..... %02x\n", desc->iface.bAlternateSetting );
			vmm_printf( VMM_DEBUG, "bNumEndpoints ......... %02x\n", desc->iface.bNumEndpoints );
			vmm_printf( VMM_DEBUG, "bInterfaceClass ....... %02x\n", desc->iface.bInterfaceClass );
			vmm_printf( VMM_DEBUG, "bInterfaceSubClass .... %02x\n", desc->iface.bInterfaceSubClass );
			vmm_printf( VMM_DEBUG, "bInterfaceProtocol .... %02x\n", desc->iface.bInterfaceProtocol );
			vmm_printf( VMM_DEBUG, "iInterface ............ %02x\n", desc->iface.iInterface );
		}
		break;

	case USB_DT_ENDPOINT:
		{
			vmm_printf( VMM_DEBUG, "[Endpoint Descriptor]\n" );
			vmm_printf( VMM_DEBUG, "bLength ............... %02x\n", desc->ep.bLength );
			vmm_printf( VMM_DEBUG, "bDescriptorType ....... %02x\n", desc->ep.bDescriptorType );
			vmm_printf( VMM_DEBUG, "bEndpointAddress ...... %02x\n", desc->ep.bEndpointAddress );
			vmm_printf( VMM_DEBUG, "bmAttributes .......... %02x\n", desc->ep.bmAttributes );
			vmm_printf( VMM_DEBUG, "wMaxPacketSize  ....... %04x\n", desc->ep.wMaxPacketSize );
			vmm_printf( VMM_DEBUG, "bInterval ............. %02x\n", desc->ep.bInterval );
		}
		break;

	default:
		return;
	}
	vmm_printf( VMM_DEBUG, "\n" );
}
*/
