/*!
******************************************************************************

	@file	usb.h

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#ifndef __USB_H__
#define __USB_H__

#pragma pack(1)

#define USB_DEVICE_MAX		128
#define USB_ENDPOINT_MAX	16

struct _USB_DEVICE;
typedef struct {
	bool ( *alloc_queue		)( struct _USB_DEVICE *, u8, size_t );
	bool ( *ctrl_msg		)( struct _USB_DEVICE *, u8, u8, u16, u16, void *, size_t, int );
	bool ( *bulk_xfer		)( struct _USB_DEVICE *, u8, void *, size_t, int );
	bool ( *interrupt_xfer	)( struct _USB_DEVICE *, u8, void *, size_t, int );
} USB_HCD;

typedef struct {
	union {
		u8	b[16];
		volatile struct {
			u32		flags;
			u32		tail;
			u32		head;
			u32		next;
		} ohci;
		volatile struct {
			u32		qh_link;
			u32		qe_link;
		} uhci;
	};
} USB_HCQH;

typedef struct {
	union {
		u8	b[64];
		union {
			volatile struct {
				u32		flags;
				u32		buf_ptr;
				u32		next;
				u32		buf_end;
			} gen;
			volatile struct {
				u32		flags;
				u32		buf_page0;
				u32		next;
				u32		buf_end;
				union {
					u16		offset[8];
					u16		status_word[8];
				};
			} iso;
		} ohci;
	};
} USB_HCTD;

#define USB_HCQH_ALIGN			16
#define USB_HCQH_MAX			USB_ENDPOINT_MAX

#define USB_HCTD_ALIGN			64
#define USB_HCTD_MAX			20
#define USB_HCTD_NULL			0
#define USB_HCTD_CTRL_SETUP		16
#define USB_HCTD_CTRL_DATA		17
#define USB_HCTD_CTRL_STATUS	18

#define USB_HCQHBUF			NON_CACHED_PTR( _usb_hcqhbuf, u8 )
#define USB_HCQHBUF_SIZE	0x00008000		// 32KB  (128*16*16B)
extern u8 _usb_hcqhbuf[USB_HCQHBUF_SIZE];

#define USB_HCTDBUF			NON_CACHED_PTR( _usb_hctdbuf, u8 )
#define USB_HCTDBUF_SIZE	0x00028000		// 160KB (128*20*64B)
extern u8 _usb_hctdbuf[USB_HCTDBUF_SIZE];

#define USB_DEVBUF			NON_CACHED_PTR( _usb_devbuf, u8 )
#define USB_DEVBUF_SIZE		0x00400000		// 4MB   (128*32KB)
extern u8 _usb_devbuf[USB_DEVBUF_SIZE];

typedef struct {
	u8				valid;
	u8				addr;
	u8				attr;
	u8				interval;
	u16				max_pkt_size;
	u16				rsvd;
	volatile int	active;
	void *			buf;
	size_t			buf_len;
} USB_ENDPOINT;

typedef struct _USB_DEVICE {
	void *			hcd;
	void *			hcd_priv;
	u8				func_addr;
	u8				low_speed;
	u8				max_pkt_size;
	u8				config_number;
	u8				iface_number;
	u8				iface_class;
	u8				iface_subclass;
	u8				iface_protocol;
	USB_ENDPOINT	ep[USB_ENDPOINT_MAX];
	union {
		struct {
			u8	int_in;
		} hid;
		struct {
			u8	bulk_in;
			u8	bulk_out;
			u8	rsvd[2];
			bool ( *bulk_xfer )( struct _USB_DEVICE *, void *, u32, void *, u32, bool );
		} msc;
	};
} USB_DEVICE;

typedef struct {
	u8		bmRequestType;
	u8		bRequest;
	u16		wValue;
	u16		wIndex;
	u16		wLength;
} USB_REQUEST;

// [bmRequestType]
// b7: Data transfer direction
#define USB_REQ_DIR_MASK		0x80
#define USB_REQ_DIR_IN			0x80	// Device-to-host
#define USB_REQ_DIR_OUT			0x00	// Host-to-device
// b6-5: Type
#define USB_REQ_TYPE_MASK		0x60
#define USB_REQ_TYPE_STD		0x00	// Standard
#define USB_REQ_TYPE_CLASS		0x20	// Class
#define USB_REQ_TYPE_VENDOR		0x40	// Vendor
// b4-0: Recipient
#define USB_REQ_RECP_MASK		0x1F
#define USB_REQ_DEVICE			0x00	// Device
#define USB_REQ_INTERFACE		0x01	// Interface
#define USB_REQ_ENDPOINT		0x02	// Endpoint
#define USB_REQ_OTHER			0x03	// Other

// [bRequest]
// Standard Request
#define USB_GET_STATUS			0x00
#define USB_CLEAR_FEATURE		0x01
#define USB_SET_FEATURE			0x03
#define USB_SET_ADDRESS			0x05
#define USB_GET_DESCRIPTOR		0x06
#define USB_SET_DESCRIPTOR		0x07
#define USB_GET_CONFIGURATION	0x08
#define USB_SET_CONFIGURATION	0x09
#define USB_GET_INTERFACE		0x0A
#define USB_SET_INTERFACE		0x0B
#define USB_SYNCH_FRAME			0x0C

// [wValue]
// Descriptor Type
#define USB_DT_DEVICE			0x0100
#define USB_DT_CONFIG			0x0200
#define USB_DT_STRING			0x0300
#define USB_DT_INTERFACE		0x0400
#define USB_DT_ENDPOINT			0x0500
#define USB_DT_DEV_QUALIFIER	0x0600
#define USB_DT_OTHER_SPD_CONF	0x0700
#define USB_DT_INTERFACE_PWR	0x0800
#define USB_DT_STD				0x0000
#define USB_DT_CLASS			0x0020
#define USB_DT_VENDOR			0x0040

typedef struct {
	u8		bLength;
	u8		bDescriptorType;
	u16		bcdUSB;
	u8		bDeviceClass;
	u8		bDeviceSubClass;
	u8		bDeviceProtocol;
	u8		bMaxPacketSize0;
	u16		idVendor;
	u16		idProduct;
	u16		bcdDevice;
	u8		iManufacturer;
	u8		iProduct;
	u8		iSerialNumber;
	u8		bNumConfigurations;
} USB_DEVICE_DESC;

typedef struct {
	u8		bLength;
	u8		bDescriptorType;
	u16		wTotalLength;
	u8		bNumInterfaces;
	u8		bConfigurationValue;
	u8		iConfiguration;
	u8		bmAttributes;
	u8		bMaxPower;
} USB_CONFIG_DESC;

typedef struct {
	u8		bLength;
	u8		bDescriptorType;
	u8		bInterfaceNumber;
	u8		bAlternateSetting;
	u8		bNumEndpoints;
	u8		bInterfaceClass;
	u8		bInterfaceSubClass;
	u8		bInterfaceProtocol;
	u8		iInterface;
} USB_INTERFACE_DESC;

typedef struct {
	u8		bLength;
	u8		bDescriptorType;
	u8		bEndpointAddress;
	u8		bmAttributes;
	u16		wMaxPacketSize;
	u8		bInterval;
} USB_ENDPOINT_DESC;

#define USB_ENDPT_XFER_TYPE(attr)	((attr) & 0x03)
#define USB_ENDPT_CTRL				0x00
#define USB_ENDPT_ISO				0x01
#define USB_ENDPT_BULK				0x02
#define USB_ENDPT_INT				0x03

#define USB_ENDPT_SYNC_TYPE(attr)	((attr) & 0x0C)
#define USB_ENDPT_NO_SYNC			0x00
#define USB_ENDPT_ASYNC				0x01
#define USB_ENDPT_ADAPTIVE			0x02
#define USB_ENDPT_SYNC				0x03

#define USB_ENDPT_USAGE_TYPE(attr)	((attr) & 0x30)
#define USB_ENDPT_DATA				0x00
#define USB_ENDPT_FEEDBACK			0x01
#define USB_ENDPT_IMPLICIT			0x02

typedef struct {
	u8		bLength;
	u8		bDescriptorType;
	union {
		u16	wLANGID[1];
		u16	wString[1];
		u8	bString[1];
	};
} USB_STRING_DESC;

typedef union {
	struct {
		u8	b[255];
		u8	padding;
	};
	struct {
		u8	bLength;
		u8	bDescriptorType;
	} hdr;
	USB_DEVICE_DESC		dev;
	USB_CONFIG_DESC		cfg;
	USB_INTERFACE_DESC	iface;
	USB_ENDPOINT_DESC	ep;
	USB_STRING_DESC		str;
} USB_DESCRIPTOR;

#define USB_CLASS_AUDIO			0x01
#define USB_CLASS_CDC_CTRL		0x02
#define USB_CLASS_HID			0x03
#define USB_CLASS_PHYSICAL		0x05
#define USB_CLASS_IMAGE			0x06
#define USB_CLASS_PRINTER		0x07
#define USB_CLASS_STORAGE		0x08
#define USB_CLASS_HUB			0x09
#define USB_CLASS_CDC_DATA		0x0A
#define USB_CLASS_SMART_CARD	0x0B
#define USB_CLASS_SECURITY		0x0D
#define USB_CLASS_VIDEO			0x0E
#define USB_CLASS_HEALTHCARE	0x0F
#define USB_CLASS_DIAG			0xDC
#define USB_CLASS_WIRELESS		0xE0
#define USB_CLASS_MISC			0xEF
#define USB_CLASS_APP_SPEC		0xFE
#define USB_CLASS_VENDOR_SPEC	0xFF

#pragma pack()

extern bool usb_init( void );
extern bool usb_start( void );
extern USB_DEVICE * usb_alloc_device( USB_HCD *hcd, void *hcd_priv, bool low_speed );
extern bool usb_register( USB_DEVICE *dev, USB_HCD *child_hcd );

extern bool usb_ctrl_msg( USB_DEVICE *dev, u8 req_type, u8 request, u16 value, u16 index,
						  void *buf, size_t buf_len, int timeout );
extern bool usb_bulk_write( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout );
extern bool usb_bulk_read( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout );
extern bool usb_interrupt_write( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout );
extern bool usb_interrupt_read( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout );

extern bool usb_get_descriptor( USB_DEVICE *dev, u16 desc_type, u16 lang_id, void *desc, size_t size, int timeout );
extern bool usb_set_address( USB_DEVICE *dev, u8 addr, int timeout );
extern bool usb_set_configuration( USB_DEVICE *dev, u8 config, int timeout );
extern bool usb_set_interface( USB_DEVICE *dev, u8 iface, int timeout );

#endif // !__USB_H__
